///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/scene/objects/geometry/MeshObject.h>
#include <core/scene/ObjectNode.h>

#include "MeshExportInterface.h"

namespace POVRay {

IMPLEMENT_PLUGIN_CLASS(MeshExportInterface, POVRayExportInterface)

/******************************************************************************
* Exports the given scene object in the POV-Ray format and returns true.
* Return false if the scene object type is not supported by this interface class.
* Throws an exception if an error occurred.
******************************************************************************/
bool MeshExportInterface::exportSceneObject(SceneObject* sceneObj, POVRayWriter& writer, ObjectNode* contextNode, const AffineTransformation& objToWorldTM)
{
	// Check if the generic scene object can be converted to a mesh.	
	if(!sceneObj->canConvertTo(PLUGINCLASSINFO(MeshObject)))
		return false;
		
	// Turn the scene object into a mesh object.
	MeshObject::SmartPtr meshObj = sceneObj->convertTo<MeshObject>(writer.time());
	OVITO_ASSERT(meshObj);
	if(!meshObj) return false;
	
	const TriMesh& mesh = meshObj->mesh();
	
	writer << "mesh2 {" << endl;
	
	// Write mesh vertices.
	writer << "vertex_vectors {" << endl;
	writer << mesh.vertexCount();
	for(int vindex = 0; vindex < mesh.vertexCount(); vindex++) {
		writer << "," << endl << mesh.vertex(vindex);
	}
	writer << endl << "}" << endl;
	
	// Write mesh faces.
	writer << "face_indices {" << endl;
	writer << mesh.faceCount() << "," << endl;
	for(int findex = 0; findex < mesh.faceCount(); findex++) {
		const TriMeshFace& face = mesh.face(findex);
		writer << "<" << face.vertex(0) << ", " << face.vertex(1) << ", " << face.vertex(2) << ">" << endl;		
	}
	writer << "}" << endl;

	// Write material
	writer << "material {" << endl;
	writer << "  texture { pigment { color " << contextNode->displayColor() << " } }" << endl;
	writer << "}" << endl;

	// Write transformation.
	writer << "matrix " << objToWorldTM << endl;
	
	writer << "}" << endl;
		
	return true;
}

};

