///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/RolloutContainer.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/plugins/branding/Branding.h>
#include <core/plugins/PluginManager.h>
#include <core/viewport/Window3DContainer.h>
#include <core/viewport/Window3D.h>
#include <core/viewport/ViewportManager.h>
#include "AtomVizBenchmarkUtility.h"

namespace AtomViz {

IMPLEMENT_PLUGIN_CLASS(AtomVizBenchmarkUtility, UtilityPlugin)

/// The global window instance.
QPointer<AtomVizOpenGLBenchmarkWindow> AtomVizOpenGLBenchmarkWindow::_singletonInstance(NULL);

/******************************************************************************
* Constructor.
******************************************************************************/
AtomVizOpenGLBenchmarkWindow::AtomVizOpenGLBenchmarkWindow(QWidget* parent)
	: QMainWindow(parent, Qt::Window)
{
	setWindowTitle(tr("OpenGL Test Utility"));
	setAttribute(Qt::WA_DeleteOnClose);

	_contentWidget = new QTextEdit(this);
	_contentWidget->setReadOnly(true);
	setCentralWidget(_contentWidget);

	QToolBar* toolBar = addToolBar(tr("OpenGL test utility"));
	toolBar->addAction(QIcon(":/core/main/file_save.png"), tr("Save report to PDF file"), this, SLOT(saveReport()));

	performOpenGLTest();

	resize(800, 600);
}

/******************************************************************************
* Opens the benchmark window.
******************************************************************************/
AtomVizOpenGLBenchmarkWindow* AtomVizOpenGLBenchmarkWindow::showBenchmarkWindow()
{
	if(_singletonInstance == NULL)
		_singletonInstance = new AtomVizOpenGLBenchmarkWindow(MAIN_FRAME);
	_singletonInstance->show();
	_singletonInstance->raise();
	_singletonInstance->activateWindow();
	return _singletonInstance;
}


/******************************************************************************
* Handles the "Run Benchmark" command event.
******************************************************************************/
void AtomVizOpenGLBenchmarkWindow::performOpenGLTest()
{
	QString report;

	report.append(tr("<h2>OVITO System Report and OpenGL Test</h2>"));
	report.append(tr("<p>Generated on:</i> %1</p>").arg(QDateTime::currentDateTime().toString()));

	report.append(tr("<h3>Application</h3><p>"));
	Branding::SmartPtr branding = BrandingManager::primaryBranding();
	report.append(tr("<i>Product name:</i> %1<br>").arg(branding->productName()));
	report.append(tr("<i>Product description:</i> %1<br>").arg(branding->productDescription()));
	report.append(tr("<i>Product version:</i> %1<br>").arg(branding->productVersion()));
	report.append(tr("<i>Installed plugins:</i><ul>"));
	Q_FOREACH(Plugin* plugin, PLUGIN_MANAGER.plugins())
		report.append(tr("<li>%1 %2</li>").arg(plugin->pluginId()).arg(plugin->pluginVersion()));
	report.append(tr("</ul>"));
	report.append(tr("<i>Command line:</i> %1").arg(QCoreApplication::arguments().join(" ")));
	report.append(tr("</p>"));

	report.append(tr("<h3>Operating System</h3><p>"));
#if defined(Q_WS_MAC)
	report.append(tr("<i>Platform:</i> Mac OS X<br>"));
#elif defined(Q_WS_X11)
	report.append(tr("<i>Platform:</i> Unix/Linux/X11<br>"));
#elif defined(Q_WS_WIN)
	report.append(tr("<i>Platform:</i> Windows<br>"));
#endif
#if defined(Q_OS_CYGWIN)
	report.append(tr("<i>System:</i> Cygwin<br>"));
#elif defined(Q_OS_DARWIN)
	report.append(tr("<i>System:</i> Darwin<br>"));
#elif defined(Q_OS_FREEBSD)
	report.append(tr("<i>System:</i> FreeBSD<br>"));
#elif defined(Q_OS_LINUX)
	report.append(tr("<i>System:</i> Linux<br>"));
#elif defined(Q_OS_SOLARIS)
	report.append(tr("<i>System:</i> Solaris<br>"));
#elif defined(Q_OS_UNIX)
	report.append(tr("<i>System:</i> Unix<br>"));
#elif defined(Q_OS_WIN32)
	report.append(tr("<i>System:</i> Windows<br>"));
#endif

	// Get 'uname' output.
	QProcess unameProcess;
	unameProcess.start("uname -m -i -o -r -v", QIODevice::ReadOnly);
	unameProcess.waitForFinished();
	QByteArray unameOutput = unameProcess.readAllStandardOutput();
	report.append(tr("<i>uname output:</i> %1<br>").arg(QString(unameOutput)));

	// Get 'lsb_release' output.
	QProcess lsbProcess;
	lsbProcess.start("lsb_release -s -i -d -r", QIODevice::ReadOnly);
	lsbProcess.waitForFinished();
	QByteArray lsbOutput = lsbProcess.readAllStandardOutput();
	report.append(tr("<i>LSB output:</i> %1").arg(QString(lsbOutput)));
	report.append(tr("</p>"));

	report.append(tr("<h3>OpenGL</h3><p>"));
	report.append(tr("<i>Vendor</i>: %1<br><i>Renderer</i>: %2")
		.arg((const char*)glGetString(GL_VENDOR))
		.arg((const char*)glGetString(GL_RENDERER)));
	QString ext1 = tr("<font color=green>supported</font>");
	QString ext0 = tr("<font color=red>not supported</font>");
	Window3D* win3D = VIEWPORT_MANAGER.activeViewport();
	OVITO_ASSERT(win3D);
	report.append(tr("<br>Direct rendering: %1").arg(win3D->format().directRendering() ? ext1 : ext0));
	report.append(tr("<br>Depth buffer bits: %1").arg(win3D->format().depthBufferSize()));
	report.append(tr("<br>Extensions:<ul>"));
	report.append(tr("<li>EXT_compiled_vertex_array: %1</li>").arg(win3D->hasCompiledVertexArraysExtension() ? ext1 : ext0));
	report.append(tr("<li>KTX_buffer_region: %1</li>").arg(win3D->hasBufferRegionsExtension() ? ext1 : ext0));
	report.append(tr("<li>WIN_swap_hint: %1</li>").arg(win3D->hasSwapHintExtension() ? ext1 : ext0));
	report.append(tr("<li>ARB_point_parameters: %1</li>").arg(win3D->hasPointParametersExtension() ? ext1 : ext0));
	report.append(tr("<li>ARB_vertex_buffer_object: %1</li>").arg(win3D->hasVertexBufferObjectsExtension() ? ext1 : ext0));
	report.append(tr("<li>EXT_framebuffer_object: %1</li>").arg(win3D->hasFrameBufferExtension() ? ext1 : ext0));
	report.append(tr("<li>ARB_shader_objects: %1</li>").arg(win3D->hasShaderObjectsExtension() ? ext1 : ext0));
	report.append(tr("<li>ARB_vertex_shader: %1</li>").arg(win3D->hasVertexShaderExtension() ? ext1 : ext0));
	report.append(tr("<li>ARB_fragment_shader: %1</li>").arg(win3D->hasFragmentShaderExtension() ? ext1 : ext0));
	report.append(tr("<li>EXT_fog_coord: %1</li>").arg(win3D->hasFogCoordExtension() ? ext1 : ext0));
	report.append(tr("</ul>"));
	report.append(tr("</p>"));

	QVector<QImage> testImages;
	report.append(tr("<h3>Compatibility Tests</h3><p>"));
	try {
		testImages = renderCompatibilityTests();
		report.append("<table>");
		report.append(tr("<tr><td><b>Test</b></td><td><b>Description</b></td><td><b>Reference image</b></td><td><b>Your system</b></td></tr>"));
		appendTestImage(report, 0, tr("Quad geometry with billboard texture maps<br>Uniform atom radius"), testImages[0], _contentWidget->document());
		appendTestImage(report, 1, tr("Scaled point sprites with billboard texture maps<br>Uniform atom radius"), testImages[1], _contentWidget->document());
		appendTestImage(report, 2, tr("Scaled point sprites with 2D hardware shader<br>Uniform atom radius"), testImages[2], _contentWidget->document());
		appendTestImage(report, 3, tr("Cube geometry with raytrace hardware shader<br>Uniform atom radius"), testImages[3], _contentWidget->document());
		appendTestImage(report, 4, tr("Quad geometry with billboard texture maps<br>Varying atom radius"), testImages[4], _contentWidget->document());
		appendTestImage(report, 5, tr("Scaled point sprites with billboard texture maps<br>Varying atom radius"), testImages[5], _contentWidget->document());
		appendTestImage(report, 6, tr("Scaled point sprites with 2D hardware shader<br>Varying atom radius"), testImages[6], _contentWidget->document());
		appendTestImage(report, 7, tr("Cube geometry with raytrace hardware shader<br>Varying atom radius"), testImages[7], _contentWidget->document());
		report.append("</table>");
	}
	catch(const Exception& ex) {
		report.append(tr("<p><font color=red>ERROR: Could not render compatibility test pictures on your system.<br>%1</font></p>").arg(ex.message()));
	}

	_contentWidget->setHtml(report);
}

void AtomVizOpenGLBenchmarkWindow::appendTestImage(QString& report, int testIndex, const QString& description, const QImage& image, QTextDocument* document)
{
	report.append(QString("<tr><td>%1.</td><td>%2</td><td><img src=\"refimage%3://\" /></td><td>").arg(testIndex+1).arg(description).arg(testIndex));
	if(image.isNull())
		report.append(tr("Rendering mode not supported by your system."));
	else
		report.append(QString("<img src=\"testimage%1://\" />").arg(testIndex));
	report.append("</td></tr>");
	document->addResource(QTextDocument::ImageResource, QUrl(QString("testimage%1://").arg(testIndex)), image);
	document->addResource(QTextDocument::ImageResource, QUrl(QString("refimage%1://").arg(testIndex)), QImage(QString(":/atomviz/opengl_test_images/refimage%1.png").arg(testIndex)));
	//image.save(QString("/home/stuko/projekte/ovito/src/atomviz/resources/opengl_test_images/refimage%1.png").arg(testIndex));
}

/******************************************************************************
* Render test pictures for OpenGL compatibility check.
******************************************************************************/
QVector<QImage> AtomVizOpenGLBenchmarkWindow::renderCompatibilityTests()
{
	Window3D* renderingContainer = VIEWPORT_MANAGER.activeViewport();
	if(!renderingContainer || !renderingContainer->hasFrameBufferExtension())
		throw Exception(tr("Your graphics card doesn't support the OpenGL GL_EXT_framebuffer_object extension."));

    // Create the offscreen framebuffer object - make sure to have a current
    // context before creating it.
	renderingContainer->makeCurrent();

	GLuint fbWidth = 180;
	GLuint fbHeight = 180;
	GLuint oglFrameBuffer;
	GLuint oglRenderBuffer;
	GLuint oglDepthRenderBuffer;

	// Create a frame-buffer object and a render-buffer object for color and depth channel.
	CHECK_OPENGL(renderingContainer->glGenFramebuffersEXT(1, &oglFrameBuffer));
	CHECK_OPENGL(renderingContainer->glGenRenderbuffersEXT(1, &oglRenderBuffer));
	CHECK_OPENGL(renderingContainer->glGenRenderbuffersEXT(1, &oglDepthRenderBuffer));

	CHECK_OPENGL(renderingContainer->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, oglDepthRenderBuffer));
	renderingContainer->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT24, fbWidth, fbHeight);
	GLenum errorCode = glGetError();
	if(errorCode == GL_NO_ERROR) {
		CHECK_OPENGL(renderingContainer->glBindRenderbufferEXT(GL_RENDERBUFFER_EXT, oglRenderBuffer));
		renderingContainer->glRenderbufferStorageEXT(GL_RENDERBUFFER_EXT, GL_RGBA8, fbWidth, fbHeight);
		errorCode = glGetError();
	}
	if(errorCode != GL_NO_ERROR) {
		// Clean up
		renderingContainer->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);
		throw Exception(tr("Failed to reserve an OpenGL offscreen buffer with the requested size. Please descrease the image resolution or the antialising level."));
	}

	// Activate the off-screen frame-buffer and associate the render-buffer objects.
	CHECK_OPENGL(renderingContainer->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, oglFrameBuffer));
	CHECK_OPENGL(renderingContainer->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_RENDERBUFFER_EXT, oglRenderBuffer));
	CHECK_OPENGL(renderingContainer->glFramebufferRenderbufferEXT(GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, oglDepthRenderBuffer));

	// Check for errors...
	GLenum status = renderingContainer->glCheckFramebufferStatusEXT(GL_FRAMEBUFFER_EXT);
	if(status != GL_NO_ERROR && status != GL_FRAMEBUFFER_COMPLETE_EXT) {
		// Clean up
		renderingContainer->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
		renderingContainer->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);
		switch(status) {
			case GL_FRAMEBUFFER_UNSUPPORTED_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Unsupported OpenGL framebuffer format.\nTry to reduce the image resolution or the antialising level to fix this problem.");
			case GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete attachment.");
			case GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing attachment.");
#ifdef GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT
			case GL_FRAMEBUFFER_INCOMPLETE_DUPLICATE_ATTACHMENT_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, duplicate attachment.");
#endif
			case GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, attached images must have same dimensions.");
			case GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, attached images must have same format.");
			case GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing draw buffer.");
			case GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER_EXT:
				throw Exception("Failed to initialize the OpenGL framebuffer. Framebuffer incomplete, missing read buffer.");
			default:
				throw Exception(tr("Failed to initialize the OpenGL framebuffer. An undefined error has occurred: status=%1").arg(status));
		}
	}

	CHECK_OPENGL(glReadBuffer(GL_COLOR_ATTACHMENT0_EXT));

	// Prepare OpenGL rendering state.
	glPushAttrib(GL_LIGHTING_BIT|GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_ENABLE_BIT|GL_VIEWPORT_BIT|GL_TRANSFORM_BIT);
	glDisable(GL_SCISSOR_TEST);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);
	glViewport(0, 0, fbWidth, fbHeight);
	glDisable(GL_ALPHA_TEST);
	glDisable(GL_LIGHTING);

	// Setup projection matrix.
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	Matrix4 modelMatrix(AffineTransformation::lookAt(Point3(0,-6,0), ORIGIN, Vector3(0,0,1)));
	glLoadMatrix(modelMatrix.constData());
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	Matrix4 projectionMatrix(Matrix4::perspective(FLOATTYPE_PI/2.8f, 1.0f, 0.1f, 16.0f));
	glLoadMatrix(projectionMatrix.constData());

	// Initialize buffer with background color.
	glClearColor(0.8, 0.8, 0.8, 1);

	AtomsRenderer::RenderingMethod testRenderMethods[] = {
			AtomsRenderer::QUAD_GEOMETRY_IMPOSTERS,
			AtomsRenderer::POINT_SPRITE_IMPOSTERS,
			AtomsRenderer::SHADED_POINT_SPRITE_IMPOSTERS,
			AtomsRenderer::SHADED_RAYTRACED_SPHERES
	};

	QVector<QImage> images;
	for(int varyingAtomSize = 0; varyingAtomSize <= 1; varyingAtomSize++) {
		for(int testNumber = 0; testNumber < 4; testNumber++) {
			glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
			AtomsRenderer renderer;
			if(renderer.prepare(renderingContainer, false, testRenderMethods[testNumber]) != testRenderMethods[testNumber]) {
				images.push_back(QImage());
				continue;
			}
			renderer.beginAtoms(2);
			renderer.specifyAtom(Point3(0.0f, 0.0f,-1.0f), 255, 255, 0, 2.0);
			renderer.specifyAtom(Point3(0.5f,-0.1f,+1.0f), 255, 0, 255, varyingAtomSize ? 1.0 : 2.0);
			renderer.endAtoms();
			renderer.renderOffscreen(true, projectionMatrix, QSize(fbWidth, fbHeight));
			glFlush();

			// Read out the OpenGL framebuffer.
			QImage::Format image_format = QImage::Format_ARGB32_Premultiplied;
			QImage img(fbWidth, fbHeight, image_format);
			CHECK_OPENGL(glReadPixels(0, 0, fbWidth, fbHeight, GL_RGBA, GL_UNSIGNED_BYTE, img.bits()));
			// OpenGL gives ABGR (i.e. RGBA backwards); Qt wants ARGB
			images.push_back(img.rgbSwapped().mirrored());
		}
	}

	// Restore old settings.
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
	glPopAttrib();

	// Make the normal window the target again.
	renderingContainer->glBindFramebufferEXT(GL_FRAMEBUFFER_EXT, 0);

	// Clean up
	renderingContainer->glDeleteFramebuffersEXT(1, &oglFrameBuffer);
	renderingContainer->glDeleteRenderbuffersEXT(1, &oglRenderBuffer);
	renderingContainer->glDeleteRenderbuffersEXT(1, &oglDepthRenderBuffer);

	return images;
}

/******************************************************************************
* This opens the file dialog and lets the user save the report to a PDF file.
******************************************************************************/
void AtomVizOpenGLBenchmarkWindow::saveReport()
{
	QString fileName = QFileDialog::getSaveFileName(this, tr("Save report"), QString(), "*.pdf");
	if(fileName.isEmpty() == false) {

		if(QFileInfo(fileName).suffix().isEmpty())
			fileName.append(".pdf");

		QPrinter printer(QPrinter::HighResolution);
		printer.setOutputFormat(QPrinter::PdfFormat);
		printer.setOutputFileName(fileName);
		_contentWidget->document()->print(&printer);
	}
}

};
