/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  A ViewSubject that represents a fixed set of solid surfaces (iActor)
//
#ifndef ISOLIDVIEWSUBJECT_H
#define ISOLIDVIEWSUBJECT_H


#include "iviewsubject.h"


#include "iarray.h"
#include "icolor.h"

class iActor;


class iSolidViewSubject : public iViewSubject
{
	
public:
	
	vtkTypeMacro(iSolidViewSubject,iViewSubject);

	static const iObjectType& Type();
	virtual const iObjectType& RealType();
	
	virtual void SetColor(int l, const iColor& c);
	const iColor& GetColor(int l) const;

	virtual void SetOpacity(int l, float o);
	float GetOpacity(int l) const;
	
	virtual void SetPaletteId(int l, int p);
	int GetPaletteId(int l) const;
		
	void UpdateLookupTables();

	//
	//  Inherited members
	//
	virtual void Reset();
	virtual void ShowClipPlane(bool s);
	virtual void ShowColorBars(bool s);
	virtual void UpdateMaterialProperties();
	virtual bool IsVisible() const;

protected:
	
	iSolidViewSubject(iViewModule *vm, const iDataType &type, const iString &name, int numActors, unsigned int flags, float maxOpacity = 1.0f);
	virtual ~iSolidViewSubject();

	virtual void ResetPipeline() = 0;

	virtual void SolidViewSubjectPackStateBody(iString &s) const = 0;
	virtual void SolidViewSubjectUnPackStateBody(const iString &s) = 0;

	virtual void ShowColorBarsBody(bool s) = 0;

	//
	//  It is more convenient to keep separate components in separate arrays
	//
	iArray<iActor*> mActors;
	iArray<float> mOpacities;
	iArray<int> mPalettes;
	iArray<iColor> mColors;

private:

	const bool mNoColor, mNoOpacity, mNoPalette;
	const bool mSameColor, mSameOpacity, mSamePalette;
	const float mMaxOpacity;

	virtual void ViewSubjectPackStateBody(iString &s) const;
	virtual void ViewSubjectUnPackStateBody(const iString &s);

	static const iObjectKey& KeyOpacity();
	static const iObjectKey& KeyPalette();
	static const iObjectKey& KeyColor();
};


inline float iSolidViewSubject::GetOpacity(int l) const
{ 
	return mOpacities[l];
}
	

inline int iSolidViewSubject::GetPaletteId(int l) const
{ 
	return mPalettes[l];
}
	

inline const iColor& iSolidViewSubject::GetColor(int l) const
{
	return mColors[l];
}


//
//  Re-define ViewSubject macros
//
#define ISOLIDVIEWSUBJECT_DECLARE_INHERITED_KEYS \
	IVIEWSUBJECT_DECLARE_INHERITED_KEYS; \
	static const iObjectKey& KeyOpacity(); \
	static const iObjectKey& KeyPalette(); \
	static const iObjectKey& KeyColor()


#define ISOLIDVIEWSUBJECT_DECLARE_INHERITED_MEMBERS \
	virtual bool CanBeShown() const; \
	protected: \
	virtual void ShowColorBarsBody(bool); \
	virtual void ResetPipeline(); \
	virtual void SolidViewSubjectPackStateBody(iString &s) const; \
	virtual void SolidViewSubjectUnPackStateBody(const iString &s); \
	virtual void ViewSubjectSyncWithData(const iDataSyncRequest &r); \
	virtual void ShowBody(bool s); \
	public: \
	virtual const iObjectType& RealType()


#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COLOR(_type_,_num_) \
	IOBJECT_DEFINE_INHERITED_KEY(iSolidViewSubject,_type_,Color,c,Color,_num_)

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_OPACITY(_type_,_num_) \
	IOBJECT_DEFINE_INHERITED_KEY(iSolidViewSubject,_type_,Opacity,o,Float,_num_)

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_PALETTE(_type_,_num_) \
	IOBJECT_DEFINE_INHERITED_KEY(iSolidViewSubject,_type_,Palette,p,Int,_num_)

#define IVIEWSUBJECT_DEFINE_INHERITED_KEYS_ALL(_type_,_num_) \
	IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(_type_); \
	IVIEWSUBJECT_DEFINE_INHERITED_KEYS_SHADING(_type_); \
	IVIEWSUBJECT_DEFINE_INHERITED_KEYS_EXTENDING(_type_); \
	IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COLOR(_type_,_num_); \
	IVIEWSUBJECT_DEFINE_INHERITED_KEYS_OPACITY(_type_,_num_); \
	IVIEWSUBJECT_DEFINE_INHERITED_KEYS_PALETTE(_type_,_num_)

#endif // ISOLIDVIEWSUBJECT_H

