package r

import (
	. "github.com/alecthomas/chroma" // nolint
	"github.com/alecthomas/chroma/lexers/internal"
)

// Ruby lexer.
var Ruby = internal.Register(MustNewLexer(
	&Config{
		Name:      "Ruby",
		Aliases:   []string{"rb", "ruby", "duby"},
		Filenames: []string{"*.rb", "*.rbw", "Rakefile", "*.rake", "*.gemspec", "*.rbx", "*.duby", "Gemfile"},
		MimeTypes: []string{"text/x-ruby", "application/x-ruby"},
		DotAll:    true,
	},
	Rules{
		"root": {
			{`\A#!.+?$`, CommentHashbang, nil},
			{`#.*?$`, CommentSingle, nil},
			{`=begin\s.*?\n=end.*?$`, CommentMultiline, nil},
			{Words(``, `\b`, `BEGIN`, `END`, `alias`, `begin`, `break`, `case`, `defined?`, `do`, `else`, `elsif`, `end`, `ensure`, `for`, `if`, `in`, `next`, `redo`, `rescue`, `raise`, `retry`, `return`, `super`, `then`, `undef`, `unless`, `until`, `when`, `while`, `yield`), Keyword, nil},
			{`(module)(\s+)([a-zA-Z_]\w*(?:::[a-zA-Z_]\w*)*)`, ByGroups(Keyword, Text, NameNamespace), nil},
			{`(def)(\s+)`, ByGroups(Keyword, Text), Push("funcname")},
			{"def(?=[*%&^`~+-/\\[<>=])", Keyword, Push("funcname")},
			{`(class)(\s+)`, ByGroups(Keyword, Text), Push("classname")},
			{Words(``, `\b`, `initialize`, `new`, `loop`, `include`, `extend`, `raise`, `attr_reader`, `attr_writer`, `attr_accessor`, `attr`, `catch`, `throw`, `private`, `module_function`, `public`, `protected`, `true`, `false`, `nil`), KeywordPseudo, nil},
			{`(not|and|or)\b`, OperatorWord, nil},
			{Words(``, `\?`, `autoload`, `block_given`, `const_defined`, `eql`, `equal`, `frozen`, `include`, `instance_of`, `is_a`, `iterator`, `kind_of`, `method_defined`, `nil`, `private_method_defined`, `protected_method_defined`, `public_method_defined`, `respond_to`, `tainted`), NameBuiltin, nil},
			{`(chomp|chop|exit|gsub|sub)!`, NameBuiltin, nil},
			{Words(`(?<!\.)`, `\b`, `Array`, `Float`, `Integer`, `String`, `__id__`, `__send__`, `abort`, `ancestors`, `at_exit`, `autoload`, `binding`, `callcc`, `caller`, `catch`, `chomp`, `chop`, `class_eval`, `class_variables`, `clone`, `const_defined?`, `const_get`, `const_missing`, `const_set`, `constants`, `display`, `dup`, `eval`, `exec`, `exit`, `extend`, `fail`, `fork`, `format`, `freeze`, `getc`, `gets`, `global_variables`, `gsub`, `hash`, `id`, `included_modules`, `inspect`, `instance_eval`, `instance_method`, `instance_methods`, `instance_variable_get`, `instance_variable_set`, `instance_variables`, `lambda`, `load`, `local_variables`, `loop`, `method`, `method_missing`, `methods`, `module_eval`, `name`, `object_id`, `open`, `p`, `print`, `printf`, `private_class_method`, `private_instance_methods`, `private_methods`, `proc`, `protected_instance_methods`, `protected_methods`, `public_class_method`, `public_instance_methods`, `public_methods`, `putc`, `puts`, `raise`, `rand`, `readline`, `readlines`, `require`, `scan`, `select`, `self`, `send`, `set_trace_func`, `singleton_methods`, `sleep`, `split`, `sprintf`, `srand`, `sub`, `syscall`, `system`, `taint`, `test`, `throw`, `to_a`, `to_s`, `trace_var`, `trap`, `untaint`, `untrace_var`, `warn`), NameBuiltin, nil},
			{`__(FILE|LINE)__\b`, NameBuiltinPseudo, nil},
			{"(?<!\\w)(<<-?)([\"`\\']?)([a-zA-Z_]\\w*)(\\2)(.*?\\n)", String, nil},
			{`(<<-?)("|\')()(\2)(.*?\n)`, String, nil},
			{`__END__`, CommentPreproc, Push("end-part")},
			{`(?:^|(?<=[=<>~!:])|(?<=(?:\s|;)when\s)|(?<=(?:\s|;)or\s)|(?<=(?:\s|;)and\s)|(?<=\.index\s)|(?<=\.scan\s)|(?<=\.sub\s)|(?<=\.sub!\s)|(?<=\.gsub\s)|(?<=\.gsub!\s)|(?<=\.match\s)|(?<=(?:\s|;)if\s)|(?<=(?:\s|;)elsif\s)|(?<=^when\s)|(?<=^index\s)|(?<=^scan\s)|(?<=^sub\s)|(?<=^gsub\s)|(?<=^sub!\s)|(?<=^gsub!\s)|(?<=^match\s)|(?<=^if\s)|(?<=^elsif\s))(\s*)(/)`, ByGroups(Text, LiteralStringRegex), Push("multiline-regex")},
			{`(?<=\(|,|\[)/`, LiteralStringRegex, Push("multiline-regex")},
			{`(\s+)(/)(?![\s=])`, ByGroups(Text, LiteralStringRegex), Push("multiline-regex")},
			{`(0_?[0-7]+(?:_[0-7]+)*)(\s*)([/?])?`, ByGroups(LiteralNumberOct, Text, Operator), nil},
			{`(0x[0-9A-Fa-f]+(?:_[0-9A-Fa-f]+)*)(\s*)([/?])?`, ByGroups(LiteralNumberHex, Text, Operator), nil},
			{`(0b[01]+(?:_[01]+)*)(\s*)([/?])?`, ByGroups(LiteralNumberBin, Text, Operator), nil},
			{`([\d]+(?:_\d+)*)(\s*)([/?])?`, ByGroups(LiteralNumberInteger, Text, Operator), nil},
			{`@@[a-zA-Z_]\w*`, NameVariableClass, nil},
			{`@[a-zA-Z_]\w*`, NameVariableInstance, nil},
			{`\$\w+`, NameVariableGlobal, nil},
			{"\\$[!@&`\\'+~=/\\\\,;.<>_*$?:\"^-]", NameVariableGlobal, nil},
			{`\$-[0adFiIlpvw]`, NameVariableGlobal, nil},
			{`::`, Operator, nil},
			Include("strings"),
			{`\?(\\[MC]-)*(\\([\\abefnrstv#"\']|x[a-fA-F0-9]{1,2}|[0-7]{1,3})|\S)(?!\w)`, LiteralStringChar, nil},
			{`[A-Z]\w+`, NameConstant, nil},
			{Words(`(\.|::)`, ``, `*`, `**`, `-`, `+`, `-@`, `+@`, `/`, `%`, `&`, `|`, `^`, "`", `~`, `[]`, `[]=`, `<<`, `>>`, `<`, `<>`, `<=>`, `>`, `>=`, `==`, `===`), ByGroups(Operator, NameOperator), nil},
			{"(\\.|::)([a-zA-Z_]\\w*[!?]?|[*%&^`~+\\-/\\[<>=])", ByGroups(Operator, Name), nil},
			{`[a-zA-Z_]\w*[!?]?`, Name, nil},
			{`(\[|\]|\*\*|<<?|>>?|>=|<=|<=>|=~|={3}|!~|&&?|\|\||\.{1,3})`, Operator, nil},
			{`[-+/*%=<>&!^|~]=?`, Operator, nil},
			{`[(){};,/?:\\]`, Punctuation, nil},
			{`\s+`, Text, nil},
		},
		"funcname": {
			{`\(`, Punctuation, Push("defexpr")},
			{"(?:([a-zA-Z_]\\w*)(\\.))?([a-zA-Z_]\\w*[!?]?|\\*\\*?|[-+]@?|[/%&|^`~]|\\[\\]=?|<<|>>|<=?>|>=?|===?)", ByGroups(NameClass, Operator, NameFunction), Pop(1)},
			Default(Pop(1)),
		},
		"classname": {
			{`\(`, Punctuation, Push("defexpr")},
			{`<<`, Operator, Pop(1)},
			{`[A-Z_]\w*`, NameClass, Pop(1)},
			Default(Pop(1)),
		},
		"defexpr": {
			{`(\))(\.|::)?`, ByGroups(Punctuation, Operator), Pop(1)},
			{`\(`, Operator, Push()},
			Include("root"),
		},
		"in-intp": {
			{`\{`, LiteralStringInterpol, Push()},
			{`\}`, LiteralStringInterpol, Pop(1)},
			Include("root"),
		},
		"string-intp": {
			{`#\{`, LiteralStringInterpol, Push("in-intp")},
			{`#@@?[a-zA-Z_]\w*`, LiteralStringInterpol, nil},
			{`#\$[a-zA-Z_]\w*`, LiteralStringInterpol, nil},
		},
		"string-intp-escaped": {
			Include("string-intp"),
			{`\\([\\abefnrstv#"\']|x[a-fA-F0-9]{1,2}|[0-7]{1,3})`, LiteralStringEscape, nil},
		},
		"interpolated-regex": {
			Include("string-intp"),
			{`[\\#]`, LiteralStringRegex, nil},
			{`[^\\#]+`, LiteralStringRegex, nil},
		},
		"interpolated-string": {
			Include("string-intp"),
			{`[\\#]`, LiteralStringOther, nil},
			{`[^\\#]+`, LiteralStringOther, nil},
		},
		"multiline-regex": {
			Include("string-intp"),
			{`\\\\`, LiteralStringRegex, nil},
			{`\\/`, LiteralStringRegex, nil},
			{`[\\#]`, LiteralStringRegex, nil},
			{`[^\\/#]+`, LiteralStringRegex, nil},
			{`/[mixounse]*`, LiteralStringRegex, Pop(1)},
		},
		"end-part": {
			{`.+`, CommentPreproc, Pop(1)},
		},
		"strings": {
			{`\:@{0,2}[a-zA-Z_]\w*[!?]?`, LiteralStringSymbol, nil},
			{Words(`\:@{0,2}`, ``, `*`, `**`, `-`, `+`, `-@`, `+@`, `/`, `%`, `&`, `|`, `^`, "`", `~`, `[]`, `[]=`, `<<`, `>>`, `<`, `<>`, `<=>`, `>`, `>=`, `==`, `===`), LiteralStringSymbol, nil},
			{`:'(\\\\|\\'|[^'])*'`, LiteralStringSymbol, nil},
			{`'(\\\\|\\'|[^'])*'`, LiteralStringSingle, nil},
			{`:"`, LiteralStringSymbol, Push("simple-sym")},
			{`([a-zA-Z_]\w*)(:)(?!:)`, ByGroups(LiteralStringSymbol, Punctuation), nil},
			{`"`, LiteralStringDouble, Push("simple-string")},
			{"(?<!\\.)`", LiteralStringBacktick, Push("simple-backtick")},
			{`%[QWx]?\{`, LiteralStringOther, Push("cb-intp-string")},
			{`%[qsw]\{`, LiteralStringOther, Push("cb-string")},
			{`%r\{`, LiteralStringRegex, Push("cb-regex")},
			{`%[QWx]?\[`, LiteralStringOther, Push("sb-intp-string")},
			{`%[qsw]\[`, LiteralStringOther, Push("sb-string")},
			{`%r\[`, LiteralStringRegex, Push("sb-regex")},
			{`%[QWx]?\(`, LiteralStringOther, Push("pa-intp-string")},
			{`%[qsw]\(`, LiteralStringOther, Push("pa-string")},
			{`%r\(`, LiteralStringRegex, Push("pa-regex")},
			{`%[QWx]?<`, LiteralStringOther, Push("ab-intp-string")},
			{`%[qsw]<`, LiteralStringOther, Push("ab-string")},
			{`%r<`, LiteralStringRegex, Push("ab-regex")},
			{`(%r([\W_]))((?:\\\2|(?!\2).)*)(\2[mixounse]*)`, String, nil},
			{`%[qsw]([\W_])((?:\\\1|(?!\1).)*)\1`, LiteralStringOther, nil},
			{`(%[QWx]([\W_]))((?:\\\2|(?!\2).)*)(\2)`, String, nil},
			{`(?<=[-+/*%=<>&!^|~,(])(\s*)(%([\t ])(?:(?:\\\3|(?!\3).)*)\3)`, ByGroups(Text, LiteralStringOther, None), nil},
			{`^(\s*)(%([\t ])(?:(?:\\\3|(?!\3).)*)\3)`, ByGroups(Text, LiteralStringOther, None), nil},
			{`(%([^a-zA-Z0-9\s]))((?:\\\2|(?!\2).)*)(\2)`, String, nil},
		},
		"simple-string": {
			Include("string-intp-escaped"),
			{`[^\\"#]+`, LiteralStringDouble, nil},
			{`[\\#]`, LiteralStringDouble, nil},
			{`"`, LiteralStringDouble, Pop(1)},
		},
		"simple-sym": {
			Include("string-intp-escaped"),
			{`[^\\"#]+`, LiteralStringSymbol, nil},
			{`[\\#]`, LiteralStringSymbol, nil},
			{`"`, LiteralStringSymbol, Pop(1)},
		},
		"simple-backtick": {
			Include("string-intp-escaped"),
			{"[^\\\\`#]+", LiteralStringBacktick, nil},
			{`[\\#]`, LiteralStringBacktick, nil},
			{"`", LiteralStringBacktick, Pop(1)},
		},
		"cb-intp-string": {
			{`\\[\\{}]`, LiteralStringOther, nil},
			{`\{`, LiteralStringOther, Push()},
			{`\}`, LiteralStringOther, Pop(1)},
			Include("string-intp-escaped"),
			{`[\\#{}]`, LiteralStringOther, nil},
			{`[^\\#{}]+`, LiteralStringOther, nil},
		},
		"cb-string": {
			{`\\[\\{}]`, LiteralStringOther, nil},
			{`\{`, LiteralStringOther, Push()},
			{`\}`, LiteralStringOther, Pop(1)},
			{`[\\#{}]`, LiteralStringOther, nil},
			{`[^\\#{}]+`, LiteralStringOther, nil},
		},
		"cb-regex": {
			{`\\[\\{}]`, LiteralStringRegex, nil},
			{`\{`, LiteralStringRegex, Push()},
			{`\}[mixounse]*`, LiteralStringRegex, Pop(1)},
			Include("string-intp"),
			{`[\\#{}]`, LiteralStringRegex, nil},
			{`[^\\#{}]+`, LiteralStringRegex, nil},
		},
		"sb-intp-string": {
			{`\\[\\\[\]]`, LiteralStringOther, nil},
			{`\[`, LiteralStringOther, Push()},
			{`\]`, LiteralStringOther, Pop(1)},
			Include("string-intp-escaped"),
			{`[\\#\[\]]`, LiteralStringOther, nil},
			{`[^\\#\[\]]+`, LiteralStringOther, nil},
		},
		"sb-string": {
			{`\\[\\\[\]]`, LiteralStringOther, nil},
			{`\[`, LiteralStringOther, Push()},
			{`\]`, LiteralStringOther, Pop(1)},
			{`[\\#\[\]]`, LiteralStringOther, nil},
			{`[^\\#\[\]]+`, LiteralStringOther, nil},
		},
		"sb-regex": {
			{`\\[\\\[\]]`, LiteralStringRegex, nil},
			{`\[`, LiteralStringRegex, Push()},
			{`\][mixounse]*`, LiteralStringRegex, Pop(1)},
			Include("string-intp"),
			{`[\\#\[\]]`, LiteralStringRegex, nil},
			{`[^\\#\[\]]+`, LiteralStringRegex, nil},
		},
		"pa-intp-string": {
			{`\\[\\()]`, LiteralStringOther, nil},
			{`\(`, LiteralStringOther, Push()},
			{`\)`, LiteralStringOther, Pop(1)},
			Include("string-intp-escaped"),
			{`[\\#()]`, LiteralStringOther, nil},
			{`[^\\#()]+`, LiteralStringOther, nil},
		},
		"pa-string": {
			{`\\[\\()]`, LiteralStringOther, nil},
			{`\(`, LiteralStringOther, Push()},
			{`\)`, LiteralStringOther, Pop(1)},
			{`[\\#()]`, LiteralStringOther, nil},
			{`[^\\#()]+`, LiteralStringOther, nil},
		},
		"pa-regex": {
			{`\\[\\()]`, LiteralStringRegex, nil},
			{`\(`, LiteralStringRegex, Push()},
			{`\)[mixounse]*`, LiteralStringRegex, Pop(1)},
			Include("string-intp"),
			{`[\\#()]`, LiteralStringRegex, nil},
			{`[^\\#()]+`, LiteralStringRegex, nil},
		},
		"ab-intp-string": {
			{`\\[\\<>]`, LiteralStringOther, nil},
			{`<`, LiteralStringOther, Push()},
			{`>`, LiteralStringOther, Pop(1)},
			Include("string-intp-escaped"),
			{`[\\#<>]`, LiteralStringOther, nil},
			{`[^\\#<>]+`, LiteralStringOther, nil},
		},
		"ab-string": {
			{`\\[\\<>]`, LiteralStringOther, nil},
			{`<`, LiteralStringOther, Push()},
			{`>`, LiteralStringOther, Pop(1)},
			{`[\\#<>]`, LiteralStringOther, nil},
			{`[^\\#<>]+`, LiteralStringOther, nil},
		},
		"ab-regex": {
			{`\\[\\<>]`, LiteralStringRegex, nil},
			{`<`, LiteralStringRegex, Push()},
			{`>[mixounse]*`, LiteralStringRegex, Pop(1)},
			Include("string-intp"),
			{`[\\#<>]`, LiteralStringRegex, nil},
			{`[^\\#<>]+`, LiteralStringRegex, nil},
		},
	},
))
