#                              -*- Mode: Perl -*- 
# Dependency.pm --- 
# Author           : Manoj Srivastava ( srivasta@tiamat.datasync.com ) 
# Created On       : Wed Jan 22 09:53:33 1997
# Created On Node  : tiamat.datasync.com
# Last Modified By : Manoj Srivastava
# Last Modified On : Sat May 15 05:34:10 1999
# Last Machine Used: glaurung.green-gryphon.com
# Update Count     : 348
# Status           : Unknown, Use with caution!
# HISTORY          : 
# Description      : 
# 
# 


require 5.001;

package Debian::Package::Dependency;
use strict;
use Carp;

my %Debug;

=head1 NAME

  Debian::Package::Dependency - Meta info for a single dependency.

=cut

=head1 SYNOPSIS

      use Debian::Package::Dependency;
      
      # initialize the dependency list object using the value of the
      # Pre-Depends, Dependes, Recommends, or Suggests field values.
      $dep_lst = 
        Debian::Package::Dependency->new('string' => "Value",
					 'Type' => "Field"});
    
      # Print it back out again
      $dep_lst->print();

=cut


=head1 DESCRIPTION


This module implements a Debian::Package::Dependency object. 

=cut

=head2 new

This is the constructor for the package.  It takes a named argument,
I<string>, whose value is the value of an element in a dependecy list,
and it takes another named argument I<Type>, which is the name of the
field.. This is a fancy way of saying we reord the name, relationship,
and version number of the package we have a relationship with, with
only the name being required. The relation ship is one of = << <= >>,
or >=.

=cut

sub new{
  my $this = shift;
  my %params = @_;
  my $class = ref($this) || $this;
  my $self = {};
  $Debug{'Routines'}=0;
  
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->new\n";
  }
  croak("Illegal arguments") unless defined $params{'string'};
   croak("Illegal arguments") unless defined $params{'Type'};
 
  bless $self, $class;
  $self->initialize('string' => $params{'string'},
		    'Type' => $params{'Type'});
  croak ("Could not initialize dependency $params{'string'}.") unless $self;
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->new done\n";
  }
  $self->{' _Debug'} = 0;
  return $self;
}


=head2

Internally, new uses the method B<initialize>, which does the hard work.
Specifically, the package name, and, if relevant, the relationship (as
in ==, =>, =<, <<, >>), and the version, of the package the current
  package depends on, are recorded.

=cut

sub initialize {
  my $self = shift;
  my %params = @_;
  my @packages = ();
  
  
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->initialize \n";
  }
  croak("Illegal arguments") unless defined $params{'string'};
  croak("Illegal arguments") unless defined $params{'Type'};
  $self->{'Type'} = $params{'Type'};
  
  if ($params{'string'} =~ /\(|\)/) {
    if ($params{'string'} =~ 
        /\s*(\S+)\s*\(\s*([\<\>\=]+)\s*([\w:\+\-\.]+)\s*\)/o) {
      my ($var, $rel, $ver);
      
      $var = $1;
      $rel = $2;
      $ver = $3;
      
      $self->{'Name'} = $var;
      $self->{'Rel'}  = $rel;
      $self->{'Ver'}  = $ver;
    }
  }
  else {
    if ($params{'string'} =~/\s*(\S+)\s*/) {
      my ($var);
      
      $var = $1;
      $self->{'Name'} = $var;
    }
  }
  croak ("Could not initialize dependency $params{'string'}.") 
    unless $self->{'Name'};
  croak ("Could not initialize dependency $params{'string'}.") 
    unless $self;
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->initialize done\n";
  }
}


=head2 print

This method takes the internal representation of a dependency and
prints it. (The format is such that one may reproduce the original
line in the Package file).

=cut

sub print {
  my $self = shift;
  
  print "$self->{'Name'}";
  print " ($self->{'Rel'} " if $self->{'Rel'};
  print "$self->{'Ver'})" if $self->{'Ver'};
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->print\n";
  }
}


=head2 as_string

This method takes the internal representation of a dependency and
converts it into a string (Instead of printing it).

=cut

sub as_string {
  my $self = shift;
  my $ret_val;
  
  $ret_val  = "$self->{'Name'}";
  $ret_val .= " ($self->{'Rel'} " if $self->{'Rel'};
  $ret_val .= "$self->{'Ver'})"   if $self->{'Ver'};
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->as_string\n";
  }
}


=head2 match

This routine compares two dependecy objects  and returns 'Yes' if
there is a match. It takes a dependency object as an argument.

=cut

sub match {
  my $self = shift;
  my $dependency = shift;
  my $ret = 'No';
  
  if ($self->{'Name'} eq $dependency->{'Name'}) {
    if ($self->{'Rel'}) {
      if ($self->{'Rel'} eq $dependency->{'Rel'} &&
	 $self->{'Ver'} eq $dependency->{'Ver'}) {
	$ret = 'Yes';
      }
    }
    elsif (!$dependency->{'Rel'}) {
      $ret = 'Yes';
    }
  }
  return $ret;
}

=head2 depend

This method takes Named package lists New and Installed, and check
self to see if the dependencies are satisfied. It first checks the
package list new, and then the package list installed. Internally, the
checking is done using the routine do_depend. This is the routine that
adds ordering information to the array if the current packageis not a
virtual package. In case there is a version mismatch within the new
list, this is a cause for concern, and if the parameter Consistent is
defined, it prints a warning on stderr.

=cut

sub depend {
  my $self = shift;
  my $params = shift;           # I know, this is unusual
  my $Results = 'Unknown';
  my $report;

  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->depend\n";
  }
  croak("Required parameter Package absent") 
    unless $params->{'Package'}->{'Package'};
  croak("Required parameter New package list absent") 
    unless $params->{'New'};
  
  
  $Results = $self->do_depend($params->{'New'}, $params->{'Package'});
  
  print STDERR "\tDEBUG: (dependency.pm):240 ",
  "dependency  $self->{'Name'} type $self->{'Type'}", 
    "\n\t\t Package: $params->{'Package'}->{'Package'}", 
    " ($params->{'Package'}->{'Version'})\n "
    if $self->{' _Debug'};
  print STDERR "\tDEBUG: (dependency.pm):241 $self->{'Name'}" 
    if $self->{' _Debug'};
  print STDERR "($self->{'Rel'} $self->{'Ver'})" 
    if $self->{' _Debug'} && $self->{'Ver'};
  print STDERR " Gave result=($Results)\n" if $self->{' _Debug'};

 SWITCH: for ($Results) {
    /Failed/ && do {
      if ($self->{'Type'} !~ /Conflict/o && $params->{'Consistent'}) {
        # Should warn since new list is inconsistent
        my $msg = 
          "$params->{'Package'}->{'Package'} " . 
            " ($params->{'Package'}->{'Version'}) " . 
              " in the new packages\n section is " .
		"inconsistent with dependency \n" . 
		  "$self->{'Name'}" ;
	$msg .= "($self->{'Rel'} $self->{'Ver'})" if $self->{'Ver'};
	$msg .= "\n";
	print STDERR "\tDEBUG: (dependency.pm):264 $self->{'Name'}",
	"\n\t/Failed, add 'Type'=$self->{'Type'}",
	"\n\t'Category' = Warn,'Report' = $msg\n"
	  if $self->{' _Debug'};

        $params->{'Package'}->{' _Results'}->add('Type' => $self->{'Type'},
						 'Category' => "Warn",
						 'Report' => $msg);
      }
    };
    
    /Conflict/ && do {
      if ($params->{'Consistent'}) {
        # Should warn since new list is inconsistent
        my $msg = 
          "$params->{'Package'}->{'Package'} " . 
            " ($params->{'Package'}->{'Version'}) " . 
              " in the new packages\n section is " . 
		"in conflict with \n" .
		  "$self->{'Name'}" ;
	$msg .= "($self->{'Rel'} $self->{'Ver'})" if $self->{'Ver'};
	$msg .= "\n";
	print STDERR "\tDEBUG: (dependency.pm):287 $self->{'Name'}",
	"\n\t/Conflict, add 'Type'=$self->{'Type'}",
	"\n\t'Category' = Warn,'Report' = $msg\n"
	  if $self->{' _Debug'};
        $params->{'Package'}->{' _Results'}->add('Type' => $self->{'Type'},
						 'Category' => "Warn",
						 'Report' => $msg);
      }
      $params->{'New'}->{' _Targets'}->add
	('Type' => $self->{'Type'},
	 'Target' => $self->{'Name'},
	 'Dependant' => $params->{'Package'}->{'Package'});
      last;
    };
    
    /Noconflict/ && do {
      last;			# no action...
    };

    /Success/ && do {
      $params->{'New'}->{' _Targets'}->add
	('Type' => $self->{'Type'},
	 'Target' => $self->{'Name'},
	 'Dependant' => $params->{'Package'}->{'Package'});
    };

    
    /Provided/ && do {
      my @Targets = ();
      
      if (defined $params->{'New'}->{' _Provided'}->{$self->{'Name'}})
      {
	@Targets = 
	  split(/,/, 
		$params->{'New'}->{' _Provided'}->{$self->{'Name'}});
      }
      else {
	push (@Targets, $self->{'Name'});
      }
      for (@Targets) {
	$params->{'New'}->{' _Targets'}->add
	  ('Type' => $self->{'Type'},
	   'Target' => $_,
	   'Dependant' => $params->{'Package'}->{'Package'});
      }
    };

    /Unknown|Failed/ && do {
      # Try the installed list
      if ($params->{'Installed'}) {
        $Results = 
	  $self->do_depend($params->{'Installed'}, $params->{'Package'});
	print STDERR "\tDEBUG: (dependency.pm):316 $self->{'Name'}" 
	  if $self->{' _Debug'};
	print STDERR "($self->{'Rel'} $self->{'Ver'})" 
	  if $self->{' _Debug'} && $self->{'Ver'};
	print STDERR " Gave results=($Results) \n" if $self->{' _Debug'};
	if ($Results =~ /Success|Conflict/og ) {
	  $params->{'Installed'}->{' _Targets'}->add
	    ('Type' => $self->{'Type'},
	     'Target' => $self->{'Name'},
	     'Dependant' => $params->{'Package'}->{'Package'});
	}
	if ($Results =~ /Provided/og ) {
	  my @Targets = ();
	  my $provider_disappeared = 1;
	  
	  if (defined $params->{'Installed'}->{' _Provided'}->{$self->{'Name'}})
	    {
	      @Targets = 
		split(/,/, 
		      $params->{'Installed'}->{' _Provided'}->{$self->{'Name'}});
	    }
	  else {
	    push (@Targets, $self->{'Name'});
	  }
	  
	  # If this is provided by one or more installed packages and
	  # all these providing packages are to be upgraded, this means that
	  # the new versions doesn't provide the needed package
	  # anymore! Otherwise we wouldn't have come here, the package
	  # would already have been provided by the new list.
	  for (@Targets) {
	    if (!exists $params->{'New'}->{$_}) {
	      $provider_disappeared = 0;
	      last;
	    }
	  }
	  
	  if ($provider_disappeared) {
	    $Results = "Unknown";
	  }
	  else {
	    for (@Targets) {
	      $params->{'Installed'}->{' _Targets'}->add
		('Type' => $self->{'Type'},
	         'Target' => $_,
	         'Dependant' => $params->{'Package'}->{'Package'});
	    }
	  }
	};
      }
      last;
    }; 
  } 
  
  $report = "$self->{'Name'}";
  $report .= "($self->{'Rel'} $self->{'Ver'})" if $self->{'Rel'};
  $report .= "\n";
  
 SWITCH: for ($Results) {
    my $category;

    if ("$self->{'Type'}" !~ /Conflict/) {
      /Unknown/ || do {
	$category = "Unknown";
	if ($params->{'Warn'}) {
	  $category = "Warn";
	}
	print STDERR "\tDEBUG: (dependency.pm):398 $self->{'Name'}",
	"\n\tNot conflict, not unknow, remove 'Type'=$self->{'Type'}",
	"\n\t'Category' = $category,'Report' = $report\n"
	  if $self->{' _Debug'};
	$params->{'Package'}->{' _Results'}->remove
	  ('Type' => $self->{'Type'},
	   'Category' => "$category",
	   'Report' => $report);
      };
      /Unknown/ && do {
	$category = "Unknown";
	if ($params->{'Warn'}) {
	  $category = "Warn";
	}
	print STDERR "\tDEBUG: (dependency.pm):411 $self->{'Name'}",
	"\n\tNot conflict, unknow, add 'Type'=$self->{'Type'}",
	"\n\t'Category' = $category,'Report' = $report\n"
	  if $self->{' _Debug'};
	$params->{'Package'}->{' _Results'}->add
	  ('Type' => $self->{'Type'},
	   'Category' => "$category",
	   'Report' => $report);
      };
    }

    /Success|Provided/ && do {
	print STDERR "\tDEBUG: (dependency.pm):423 $self->{'Name'}",
	"\n\tSuccess|Provided, add 'Type'=$self->{'Type'}",
	"\n\t'Category' = $category,'Report' = $report\n"
	  if $self->{' _Debug'};
      $params->{'Package'}->{' _Results'}->add('Type' => $self->{'Type'},
					       'Category' => "Found",
					       'Report' => $report);
      last;
    };

    /Conflict/ && do {
      $category = "Conflict";
      if ($params->{'Warn'}) {
	$category = "Warn";
      }
	print STDERR "\tDEBUG: (dependency.pm):438 $self->{'Name'}",
	"\n\tConflict, add 'Type'=$self->{'Type'}",
	"\n\t'Category' = $category,'Report' = $report\n"
	  if $self->{' _Debug'};
      $params->{'Package'}->{' _Results'}->add('Type' => $self->{'Type'},
					       'Category' => $category,
					       'Report' => $report);
      last;
    };
    /Failed/ && do {
      if ($self->{'Type'} ne "Conflict") {
        $category = "Failed";
        if ($params->{'Warn'}) {
           $category = "Warn";
        }
	print STDERR "\tDEBUG: (dependency.pm):453 $self->{'Name'}",
	"\n\t/Failed, add 'Type'=$self->{'Type'}",
	"\n\t'Category' = $category,'Report' = $report\n"
	  if $self->{' _Debug'};
        $params->{'Package'}->{' _Results'}->add('Type' => $self->{'Type'},
       					         'Category' => $category,
					         'Report' => $report);
      }
      last;
    };
  }
  return $Results;
}

=head2 do_depend

This routine  takes a package list, and checks if the current
dependency is satisfied by any package therein, either as a real
package or a provided one. 

=cut

sub do_depend {
  my $self     = shift;
  my $pkg_list = shift;
  my $pkg      = shift;
  my $Replaces = $pkg->{' _Replaces'};
  my $ret      = -1;
  my $Name     = $self->{'Name'};
  my $mark     = undef;
  
				  				    
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->do_depend\n";
  }

  $ret = 'Unknown'; 
  
  if (defined $pkg_list->{$Name}) { ## we have a match.
    # But is it a valid match?
     print STDERR "DEBUG: matched Package is $Name\n" if $Debug{'Routines'};

    $mark = 
      $pkg_list->{$Name}->test_mark("Mark" =>
				    'Deleted|Removed|deconfigured');
    # We essentially ignore the package if it is marked as we tested
    if (! (defined $mark && $mark)) { # Ok, consider this package   

      print STDERR "DEBUG: matched Package is $Name\n" if $Debug{'Routines'};
      print STDERR "DEBUG: No Mark is ($mark)\n" if $Debug{'Routines'};

      # error if this is a conflict.
      if ($self->{'Type'} =~ /Conflict/) {
	$ret = 'Conflict'; 
      }
      else {
	$ret = 'Success'; 
      }
      
      if ($pkg_list->{$Name}->{'Version'} && $self->{'Rel'}) { 
	# we do use version numbers.
	my $return_val = 0;
	
	print STDERR "\tDEBUG: compare-versions ",
	$pkg_list->{$Name}->{'Version'}, " ", $self->{'Rel'},
	" ", $self->{'Ver'}, "\n" if $self->{' _Debug'};

	$return_val =
	    $pkg_list->compare_versions($pkg_list->{$Name}->{'Version'},
					$self->{'Rel'}, $self->{'Ver'});
	print "\tDEBUG: returned ($return_val)\n" 
	  if $self->{' _Debug'};
	
	if (!$return_val) {        # No match
	  if ($self->{'Type'} =~ /Conflict/) {
	    $ret = 'Noconflict';             # no conflict
	  }
	  else {
	    print STDERR "\tDEBUG: type $self->{'Type'} Failed\n"
	      if $self->{' _Debug'};
	    $ret = 'Failed';      # Failed dependency! Error.
	  }
	}
      }
    }
     else {
       print STDERR "DEBUG: Failed: Package is $Name\n" if $Debug{'Routines'};
       print STDERR "DEBUG: Mark is ($mark)\n" if $Debug{'Routines'};
     }
   }
  
  # if we still don't know, check the Provided packages as well
  if ($ret eq 'Unknown' && !$self->{'Rel'} && $pkg_list->{' _Provided'}) {
    if (defined $pkg_list->{' _Provided'}->{$Name}) {
      if ($self->{'Type'} =~ /Conflict/) {
	# split the _Provided list into its parts and then look for an
	# exact match of the package name. Suggested by Roman Hodek
	# <Roman.Hodek@informatik.uni-erlangen.de> 
	if (!grep {$_ eq $pkg->{'Package'}}
	    split(/\s*,\s*/, $pkg_list->{' _Provided'}->{$Name})) {
	  # So, we do not provide this.
	  $ret = 'Conflict'; 
	}
	else {
	  # Test to see if anyone else provided this too
	  if ($pkg->{'Package'} eq $pkg_list->{' _Provided'}->{$Name}) {
	    #Well, this is clearly not a problem, right? Neither is
	    #this unknown, but, if we return provided, which means we
	    #both conflict with and provide this, it means that the
	    #caller routine, depends, shall not check the installed
	    #packages, and we may well miss a conflict. There fore, we
	    #return Unknown, rather than provided. I hate special
	    #cases.
	    print STDERR "\tDEBUG: (dependency.pm):543 $self->{'Name'}",
	    "We provide this, as well as conflict with this\n"
	      if $self->{' _Debug'};
	    $ret = 'Unknown'; 
	  }
	  else {
	    # Someone else provides this too!!
	    $ret = 'Conflict'; 
	  }
	}
      }
      else {
        $ret = 'Provided'; 
      }
    }
  }
  
  # A package does not conflict with itself

  if($ret =~ /Conflict/) {
    if ($Name eq $pkg->{'Package'}) {
      $ret = 'Noconflict'; 
    }
  }
  
  # Well, if it is a conflict, maybe we replace a Package as well
  
  if($ret =~ /Conflict/  && $Replaces) {
    my $ret_val = $Replaces->match($self);
    if ($ret_val  eq 'Yes') {
      $ret = 'Noconflict'; 
    }
    
  }
  if ($Debug{'Routines'}) {
    print STDERR "Routine: Debian::Package::Dependency->do_depend done\n";
  }
  
  return $ret;
}

=head2 order 

This method takes a named argument, New, which is a pointer to the new
packages list, and a named parameter Package, which is a pointer to
the package this dependency belongs to (the results are appended to
the package object), and, if the dependency type is conflict, it also
takes a named argument, Installed, which is a list of packages already
installed on the machine.

=cut

sub order {
  my $self = shift;
  my $params = shift;           # I know, this is unusual
  my @providers;
  my $pkg_name;			# the package whose dependency this is
  my $target;			# The target of the dependency
  
  croak("Required parameter Package absent") 
    unless $params->{'Package'}->{'Package'}; 
  croak("Required parameter New package list absent") 
    unless $params->{'New'};

  # Our Package's name is
  $pkg_name = $params->{'Package'}->{'Package'};
  $target = $self->{'Name'};
  
=pod  

 Conflicts: p (<< v) can also introduce implicit dependencies, if a
 version << v of p is installed, and is to be updated. Then the
 update must come first to avoid a dpkg error.


 The conditions to detect those cases are:
  - this is a Conflicts: with << or <= relation
  - the conflicted-with package is to be updated (we assume here
    that the new list is consistent and there is no conflict
    (anymore)) 
  - there is a conflict with the installed list

=cut 

=pod

 If there is a dependency from the conflicted-with package to
 us, this would create a cycle :-| For the common case, it
 seems more desirable to install the conflict-update first, so
 we have to remove the dependency. Also this package has to
 be installed with --force-depends then.

=cut

  if ($self->{'Type'} eq 'Conflicts') {
    # For conflicts, we also need to pass in the parameter installed
    croak("Required parameter Installed absent") 
      unless $params->{'Installed'};
    
    
    if ($self->{'Rel'} && ($self->{'Rel'} eq '<<' || 
			   $self->{'Rel'} eq '<='    ) 
	&& $params->{'New'}->{$target}->{'Package'}) {
      
      
      
      my $dep_type = $self->do_depend($params->{'Installed'}, 
				      $params->{'Package'});
      
      if ($dep_type eq 'Conflict'){
	if (exists($params->{'New'}->{$target}->{' _Pre-Depends'}->{$pkg_name}))
	  {
	    $params->{'New'}->{$target}->mark  (Mark=>'--pending-configure');
	    $params->{'New'}->{$target}->mark  (Mark=>'--ignore-errors');
	    $params->{'New'}->{$pkg_name}->mark(Mark=>'--auto-deconfigure');
	    $params->{'New'}->{$pkg_name}->mark(Mark=>'--ignore-errors');
	    # do not add an ordering edge for this conflict
	    return;
	  }
	if (exists($params->{'New'}->{$target}->{' _Depends'}->{$pkg_name}))
	  { # Warn the user about this circular dependency
	    warn "Circular Depends/Conflicts ordering between $pkg_name " . 
	      " and $target\n$target must be installed with --force-depends\n";
	    
	    # Remove the dependency on our package from the upgrade of a
	    # package whose earlier version we conflicted with; even if
	    # that package depends on us, the upgrade has to be installed
	    # first 
	    @{$params->{'New'}->{$target}->{' _Order'}} =
	      grep($_ ne "$pkg_name $target",
		   @{$params->{'New'}->{$target}->{' _Order'}} );
	    
	    $params->{'New'}->{$target}->mark('Mark' => '--force-depends');
	  }
	push(@{$params->{'Package'}->{' _Order'}}, "$target $pkg_name");
      }
    }
  }
  else {
    if ($params->{'New'}->{' _Provided'}->{$target}) {
      # Pesky multiple providers
      push (@providers, 
	    split(/,/, 
		  $params->{'New'}->{' _Provided'}->{$target}));
    }
    else {
      push (@providers, $target);
    }
    
    for (@providers) {
      if ($params->{'New'}->{$_}->{'Package'}) { # Aha! new package
        push(@{$params->{'Package'}->{' _Order'}}, 
	     "$_ $pkg_name");
      }
    }
  }
}

=head1 NOTES

This package is not really meant to be accesed directly, Dependency
objects are created and manipulated by the dependency list objects.

=cut

=head1 CAVEATS

This is very inchoate, at the moment, and needs testing.

=cut

=head1 BUGS

None Known so far.

=cut

=head1 AUTHOR

Manoj Srivastava <srivasta@debian.org>

=cut

#{ # Execute simple test if run as a script
#  package main; no strict;
#  eval join('',<main::DATA>) || die "$@ $main::DATA" unless caller();
#}


1;

__END__
# Test code. Execute this module 
