/*
 * vrwbuild.c - VRwave preprocessing
 * Copyright (c) 1996,97 IICM
 *
 * created: mpichler, 19961021
 * changed: mpichler, 19970109
 * changed: mpopetz, 19970128
 * changed: kwagen, 19970717
 *
 * $Id: vrwbuild.c,v 1.10 1997/08/01 13:59:32 kwagen Exp $
 */


#ifdef __cplusplus
extern "C" {
#endif
#include "Builder.h"
#ifdef __cplusplus
} // C++
#endif


#ifdef macintosh
# include <vectors.h>
# include <ge3d.h>
# define M_PI _PI
# define PACKAGE iicm_vrml_vrwave_
#else
# include <ge3d/vectors.h>
# include <ge3d/ge3d.h>
#endif

#include "jutils.h"

#include <math.h>  /* M_PI */
#ifndef M_PI
# include <values.h>
#endif
#include <stdio.h>  /* debugging */


/* conversion radians to degrees */
#define DEGREES(R)  ( (R) * (180 / M_PI) )



/* buildTransform */
/* build transformation matrix from transformation components */
/* see nodesRef.html#Transform of VRML 2.0 spec */
/* also compare with QvTransform::build */

void name2(PACKAGE,Builder_buildTransform) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfFloat* transh, HArrayOfFloat* roth, HArrayOfFloat* scaleh,
  HArrayOfFloat* srh, HArrayOfFloat* centerh, HArrayOfFloat* trfmath
)
{
  /* get a feeling, how readable native code embedded in Java may be ... */
  const vector3D* tran = (const vector3D*) unhand (transh)->body;
  const vector3D* center = (const vector3D*) unhand (centerh)->body;
  const vector3D* rot = (const vector3D*) unhand (roth)->body;
  const vector3D* scrot = (const vector3D*) unhand (srh)->body;
  float rotangle = (unhand (roth)->body) [3];  /* rad */
  float scrotangle = (unhand (srh)->body) [3];
  const float* scale = (const float*) unhand (scaleh)->body;
  /* TODO: check here whether trfmath is large enough to hold 16 floats */
  float (*mat)[4] = (float (*)[4]) unhand (trfmath)->body;
  /* matrix4D is float[4][4], what a pointer cannot be casted to */

  ge3dPushIdentity ();
  ge3d_translate (tran->x + center->x, tran->y + center->y, tran->z + center->z);
  ge3dRotate (rot, rotangle);
  ge3dRotate (scrot, scrotangle);  /* just rotates the scaling vector */
  ge3dScale (scale);
  ge3dRotate (scrot, - scrotangle);
  ge3d_translate (-center->x, -center->y, -center->z);  /* - center */
  ge3d_get_and_pop_matrix (mat);  /* out (by reference) */
} /* buildTransform */


/* buildTextureMatrix */
void name2(PACKAGE,Builder_buildTextureMatrix) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfFloat* centerh, float rotangle, HArrayOfFloat* scaleh, HArrayOfFloat* transh,
  HArrayOfFloat* texmath
)
{
  const vector2D* tran = (const vector2D*) unhand (transh)->body;
  const vector2D* center = (const vector2D*) unhand (centerh)->body;
  const float* scale = (const float*) unhand (scaleh)->body;
  float (*mat)[4] = (float (*)[4]) unhand (texmath)->body;

  ge3dPushIdentity ();
  ge3d_translate (-center->x, -center->y, 0.0);
  ge3d_scale (scale[0], scale[1], 1.0, 1.0);
  ge3d_rotate_axis ('z', rotangle * (180 / M_PI));
  ge3d_translate (center->x + tran->x, center->y + tran->y, 0.0);
  ge3d_get_and_pop_matrix (mat);  
} /* buildTextureMatrix */


/* getNumfaces */
/* count the number of faces in a vertex index array */

jn_int32 name2(PACKAGE,Builder_getNumfaces) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfInt* vindh, jn_int32 nv
)
{
  const jn_int32* cind = (const jn_int32*) unhand (vindh)->body;
  /* be aware: obj_length (vindh) may be greater than numvind (dynamic array) */
  jn_int32 numfaces = 0;

  /* printf ("%ld vertex indices, starting with %ld, %ld, %ld ... ", nv, cind[0], cind[1], cind[2]); */

  /* see QvIndexedFaceSet::build */
  if (nv && cind [nv-1] >= 0)  /* incomplete last face */
    numfaces = 1;
  while (nv--)
  { if (*cind++ < 0)
      numfaces++;
  }

  /* printf ("no. of faces: %ld\n", numfaces); */

  return numfaces;
} /* getNumfaces */


/* compute normal out of 3 points */
/* see vecutil.h */

static void computeNormal (const point3D* p, const point3D* q, const point3D* r, vector3D* n)
{
  vector3D a, b;
  sub3D (*r, *q, a);
  sub3D (*p, *q, b);
  crp3D (a, b, *n);  /* n = (r - q) x (p - q) */
}

/* revert normals if defined cw */

static void revertNormals (vector3D* fn, int numfaces)
{
  int i;
  for (i=0; i<numfaces; i++)
    neg3D (fn[i]);
}


/* buildFaceNormals */
/* generate face normals for flat shading */

void name2(PACKAGE,Builder_buildFacenormals) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfFloat* coordsh,               /* coordinates */
  HArrayOfInt* vindh,                   /* vertex indices */
  jn_int32 nv,                          /* number of vertex indices */
  HArrayOfFloat* fnormalsh,             /* out: face normals */
  jn_int32 ccw                          /* frontface orientation */
)
{
  const point3D* vertexlist = (const point3D*) unhand (coordsh)->body;
  const jn_int32* cind = (const jn_int32*) unhand (vindh)->body;
  vector3D* fn = (vector3D*) unhand (fnormalsh)->body;
  float norm2;
  int numfaces = 0;
  /* assert: fn allocated to hold numfaces float-triples */

  /* TODO: range check for coord indices into coords array */
  /* TODO: move this code into a reusable library */

  /* see QvIndexedFaceSet::build */
  while (nv)
  {
    /* here at the beginning of a new face */
    int v0 = cind [0];
    int v1 = cind [1];
    int v2 = cind [2];
    /* printf ("computing face normal from indices %d, %d, %d\n", v0, v1, v2); */
    if (v0 >= 0 && v1 >= 0 && v2 >= 0 && nv > 2)
    {
      if (nv == 3 || cind [3] < 0)  /* triangle: simple normal vector calculation of the 3 vertices */
        computeNormal (vertexlist + v0, vertexlist + v1, vertexlist + v2, fn);
      else  /* more than 3 vertices */
      { /* use Newell's method to calculate normal vector
         * first three vertices may be collinear and need not be a convex "ear"
         * see Graphics Gems, Vol. III, V.5, p. 231
         */
        const point3D* vert1, * vert2;
        init3D (*fn, 0, 0, 0);
        vert2 = vertexlist + v0;
        while (nv > 1 && cind [1] >= 0)
        {
          vert1 = vert2;
          vert2 = vertexlist + *++cind;
          nv--;
          fn->x += (vert1->y - vert2->y) * (vert1->z + vert2->z);
          fn->y += (vert1->z - vert2->z) * (vert1->x + vert2->x);
          fn->z += (vert1->x - vert2->x) * (vert1->y + vert2->y);
        }
        vert1 = vert2;
        vert2 = vertexlist + v0;
        fn->x += (vert1->y - vert2->y) * (vert1->z + vert2->z);
        fn->y += (vert1->z - vert2->z) * (vert1->x + vert2->x);
        fn->z += (vert1->x - vert2->x) * (vert1->y + vert2->y);
      }

      norm2 = dot3D (*fn, *fn);
      if (norm2 > 0.0)  /* normalize */
      { norm2 = 1 / sqrt (norm2);
        scl3D (*fn, norm2);
      }
      /* cerr << *fn << endl; */
    }
    else  /* face with less than 3 vertices */
      init3D (*fn, 0, 0, 0);

    numfaces++;

    /* goto next face */
    fn++;
    while (*cind >= 0 && nv)
      cind++, nv--;
    if (nv)  /* skip index -1 (face separator) */
      cind++, nv--;
  }

  fn = (vector3D*) unhand (fnormalsh)->body;
  if (!ccw)
    revertNormals (fn, numfaces);

/* fn = (vector3D*) unhand (fnormalsh)->body;
 * printf ("first two normal vectors: (%g, %g, %g), (%g, %g, %g) ...\n",
 *   fn[0].x, fn[0].y, fn[0].z, fn[1].x, fn[1].y, fn[1].z);
 */
} /* buildFaceNormals */


/* buildDefaultTexcoords */
void name2(PACKAGE,Builder_buildDefaultTexcoords) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfFloat* coordsh,               /* coordinates */
  jn_int32 ncoords,                     /* number coordinates */
  HArrayOfFloat* texcoordsh             /* out: default texture coordinates */
)
{
  const float* vertexlist = (const float*) unhand (coordsh)->body;
  float* tc = (float*) unhand (texcoordsh)->body;

  float size[3], min[3], max[3];
  int s = 0, t = 1, i, j, ntexc = 0;

  /* could be programmed much more readable by use of point3D*, but this code ports easier to Java */

  for (i=0; i<3; i++)
  {
    min[i] = vertexlist[i];
    max[i] = vertexlist[i];
  }
  for (i=0; i<ncoords; i++)
  {
    j = i % 3;
    if (vertexlist[i] < min[j])
      min[j] = vertexlist[i];
    if (vertexlist[i] > max[j])
      max[j] = vertexlist[i];
  }
  for (i=0; i<3; i++)
    size[i] = max[i] - min[i];

  if (size[t] > size[s])
  {
    s = 1;
    t = 0;
  }
  if (size[2] > size[t])
  {
    t = 2;
    if (size[t] > size[s])
    {
      t = s;
      s = 2;
    }
  }
  
  for (i=0; i<ncoords; i++)
  {
    j = i % 3;
    if (j == s)
      tc[ntexc] = (vertexlist[i] - min[s])/size[s];
    if (j == t)
      tc[ntexc+1] = (vertexlist[i] - min[t])/size[s];
    if (j == 2)
      ntexc += 2;
  }
} /* buildDefaultTexcoords */


/* compare two vectors */
static int equal (const vector3D* a, const vector3D* b)
{
  return (a->x == b->x && a->y == b->y && a->z == b->z);
}

/* normalize vector */
static void normalize (vector3D* normal)
{
  float norm = dot3D (*normal, *normal);
  if (norm > 0.0)
  {
    norm = 1/sqrt (norm);
    scl3D (*normal, norm);
  }
}

/* calculate transformation matrix for a right-handed rotation defined */
/* by a normalized rotation axis vector and the rotation angle and */
/* use this transformation on a vector */
static void rotate (vector3D* axis, float angle, vector3D* vec)
{
  float m[9];
  float a = axis->x * axis->x;
  float b = axis->y * axis->y;
  float c = axis->z * axis->z;
  float d = axis->x * axis->y;
  float e = axis->x * axis->z;
  float f = axis->y * axis->z;
  float g = sin (angle);
  float h = cos (angle);
  float x = vec->x;
  float y = vec->y;
  float z = vec->z;

  m[0] = a + (1 - a) * h;
  m[3] = d * (1 - h) - axis->z * g;
  m[6] = e * (1 - h) + axis->y * g;
  m[1] = d * (1 - h) + axis->z * g;
  m[4] = b + (1 - b) * h;
  m[7] = f * (1 - h) - axis->x * g;
  m[2] = e * (1 - h) - axis->y * g;
  m[5] = f * (1 - h) + axis->x * g;
  m[8] = c + (1 - c) * h;

  vec->x = x * m[0] + y * m[3] + z * m[6];
  vec->y = x * m[1] + y * m[4] + z * m[7];
  vec->z = x * m[2] + y * m[5] + z * m[8];
}

/* buildExtrusionData */
/* build data needed to draw the extrusion shape using GE3D.drawFaceSet. */
jn_int32 name2(PACKAGE,Builder_buildExtrusionData) (struct name3(H,PACKAGE,Builder)* handle,
  HArrayOfFloat* spineh,                /* spine points coordinates */
  jn_int32 numspine,                    /* number of spine points */
  HArrayOfFloat* crsectionh,            /* cross section coordinates */
  jn_int32 numcrs,                      /* number of cross section coordinates */
  HArrayOfFloat* scaleh,                /* scale values along the spine */
  jn_int32 numscale,                    /* number of scale values */
  HArrayOfFloat* rotationh,             /* cross section rotations along the spine */
  jn_int32 numrot,                      /* number of rotations */
  jn_int32 caps,                        /* bit 0/1 set => draw also begin/end cap */
  HArrayOfFloat* coordsh,               /* out: calculated coordinates */
  HArrayOfInt* coordindsh,              /* out: coordinate index */
  HArrayOfFloat* texcoordsh,            /* out: texture coordinates */
  HArrayOfInt* texcindsh                /* out: texture coordinate index */
)
{
  const vector3D* spine = (const vector3D*) unhand (spineh)->body;
  const point2D* crs = (const point2D*) unhand (crsectionh)->body;
  const point2D* scale = (const point2D*) unhand (scaleh)->body;
  const float* rotation = (const float*) unhand (rotationh)->body;
  point3D* coords = (point3D*) unhand (coordsh)->body;
  jn_int32* coordinds = (jn_int32*) unhand (coordindsh)->body;
  point2D* texcoords = (point2D*) unhand (texcoordsh)->body;
  jn_int32* texcinds = (jn_int32*) unhand (texcindsh)->body;

  int i, j, k, l;
  int closedsurf = 0;
  int firstz = 0;
  vector3D x, y, z, tmp1, tmp2;
  const point2D* csc = scale;
  const float* crot = rotation;
  point3D* ccoord = coords;
  const point2D* ccrs = crs;
  jn_int32* cind = coordinds;
  int closedspine = equal (spine, spine+numspine-1);
  int closedcrs = (crs->x == crs[numcrs-1].x && crs->y == crs[numcrs-1].y);
  vector3D* pz = (vector3D*) malloc (sizeof (vector3D) * numspine);
  float angle;
  float* spinelength = (float*) malloc (sizeof (float) * numspine);
  float* crslength = (float*) malloc (sizeof (float) * numcrs);
  int numcinds = 0;
  point2D* ctexcrd = texcoords;
  int texcoordoffs = 0;
  float xmin, xmax, zmin, zmax;

  /* check if cross section coordinates at first spine point are equal to those
     at the last spine point (=> smoothing !) */
  if (closedspine)
  {
    if (scale->x == scale[numscale-1].x && scale->y == scale[numscale-1].y)
    {
      const float* lastrot = rotation + (numrot*4 - 4);
      if (rotation[0] == lastrot[0] && rotation[1] == lastrot[1] &&
          rotation[2] == lastrot[2] && rotation[3] == lastrot[3])
        closedsurf = 1;
    }
  }

  if (caps & 1)  /* beginCap should be drawn, build coordindices */
  {
    if (closedcrs)  /* closed cross section, don't use last vertex again, it's equal to 1st vertex. */
    {
      for (i = 1; i < numcrs; i++)
        *(cind++) = numcrs-i-1;
      numcinds += numcrs;
    }
    else
    {
      for (i = 0; i < numcrs; i++)
        *(cind++) = numcrs-i-1;
      numcinds += numcrs+1;
    }
    *(cind++) = -1;
  }

  /* step 1: precalculate orientation (z axis in local coordinate system, pz)
     at each spine point */

  if (numspine > 2)  /* less than 3 spine points are treated specially */
  {
    for (i = 0; i < numspine; i++)
    {
      if (!i || i == numspine-1)
      {
        if (closedspine)
        {
          sub3D (spine[1], spine[0], tmp1);
          sub3D (spine[numspine-2], spine[0], tmp2);
        }
        else
        {
          if (!i)
          {
            sub3D (spine[2], spine[1], tmp1);
            sub3D (spine[0], spine[1], tmp2);
          }
          else
          {
            sub3D (spine[numspine-1], spine[numspine-2], tmp1);
            sub3D (spine[numspine-3], spine[numspine-2], tmp2);
          }
        }
      }
      else
      {
        sub3D (spine[i+1], spine[i], tmp1);
        sub3D (spine[i-1], spine[i], tmp2);
      }
      crp3D (tmp1, tmp2, z);
      if (norm3D (z) > 0)  /* valid z-axis vector (!= (0,0,0)) */
      {
        normalize (&z);
        if (i == firstz)  /* first valid z-axis vector, use it also for previous spine points */
          for (j = 0; j <= firstz; j++)
            init3D (pz[j], z.x, z.y, z.z);
        else
        {
          if (i && dot3D (z, *(pz+i-1)) < 0)
            neg3D (z);
          init3D (pz[i], z.x, z.y, z.z);
        }
      }
      else  /* z-axis vector == (0,0,0) */
      {
        if (i == firstz)  /* no z-axis vectors != (0,0,0) so far */
          firstz = i+1;
        else  /* use z-axis vector of previous spine point */
          init3D (pz[i], pz[i-1].x, pz[i-1].y, pz[i-1].z);
      }
    }
  }

  if (numspine < 3 || firstz == numspine)  /* less than 3 spine points or all spine points collinear */
  {
    for (i = 1; i < numspine; i++)
    {
      sub3D (spine[i], spine[0], y);
      if (norm3D (y) > 0)
      {
        normalize (&y);
        break;
      }
    }
    if (norm3D (y) > 0)  /* at least two spine curve coordinates differ */
    {
      if (y.y == 1)
        init3D (z, 0, 0, 1);
      else if (y.y == -1)
        init3D (z, 0, 0, -1);
      else
      {
        angle = acos (y.y);
        init3D (tmp1, y.z, 0, -y.x);
        init3D (z, 0, 0, 1);
        normalize (&tmp1);
        rotate (&tmp1, angle, &z);
      }
    }
    else  /* all spine curve coordinates specify the same point */
      init3D (z, 0, 0, 1);
    for (i = 0; i < numspine; i++)
      init3D (pz[i], z.x, z.y, z.z);
  }

  /* printf ("firstz = %d\n", firstz);
  for (i = 0; i < numspine; i++)
    printf ("%f %f %f\n", pz[i].x, pz[i].y, pz[i].z); */


  /* step 2: geometry data */

  for (i = 0; i < numspine; i++)
  {
    /* printf ("spinepoint %d\n", i); */
    if (!i || i == numspine-1)
    {
      if (closedspine)
        sub3D (spine[1], spine[numspine-2], y);
      else
      {
        if (!i)
          sub3D (spine[1], spine[0], y);
        else
          sub3D (spine[numspine-1], spine[numspine-2], y);
      }
    }
    else
      sub3D (spine[i+1], spine[i-1], y);
    if (norm3D (y) == 0)
      init3D (y, 0, 1, 0);      
    init3D (z, pz[i].x, pz[i].y, pz[i].z);
    crp3D (y, z, x);
    normalize (&x);
    normalize (&y);
    if (*(crot+3) != 0)
    {
      init3D (tmp1, *crot, *(crot+1), *(crot+2));
      angle = *(crot+3);
      rotate (&tmp1, angle, &x);
      rotate (&tmp1, angle, &y);
      rotate (&tmp1, angle, &z);
    }

    /* printf ("new coordinate system:\n");
       printf ("x-axis: %f %f %f\n", x.x, x.y, x.z);
       printf ("y-axis: %f %f %f\n", y.x, y.y, y.z);
       printf ("z-axis: %f %f %f\n", z.x, z.y, z.z);
       printf ("add coordinates for spine point %f %f %f\n", csp->x, csp->y, csp->z);
       printf ("scale %f %f\n", csc->x, csc->y); */

    ccrs = crs;

    k = i;
    if (closedsurf && i == numspine-2)
      k = -1;

    for (j = 0; j < numcrs; j++)
    {
      init3D (*ccoord, spine[i].x + ccrs->x * csc->x * x.x + ccrs->y * csc->y * z.x,
        spine[i].y + ccrs->x * csc->x * x.y + ccrs->y * csc->y * z.y,
        spine[i].z + ccrs->x * csc->x * x.z + ccrs->y * csc->y * z.z );
      ccoord++;
      ccrs++;
      if (i < numspine-1 && j < numcrs-1)
      {
        l = j;
        if (closedcrs && j == numcrs-2)
          l = -1;
        if ((i + j) & 1)
        {
          *(cind++) = i * numcrs + j;
          *(cind++) = i * numcrs + l + 1;
          *(cind++) = (k + 1) * numcrs + j;
          *(cind++) = -1;
          *(cind++) = (k + 1) * numcrs + j;
          *(cind++) = i * numcrs + l + 1;
          *(cind++) = (k + 1) * numcrs + l + 1;
          *(cind++) = -1;
        }
        else
        {
          *(cind++) = i * numcrs + j;
          *(cind++) = (k + 1) * numcrs + l + 1;
          *(cind++) = (k + 1) * numcrs + j;
          *(cind++) = -1;
          *(cind++) = i * numcrs + j;
          *(cind++) = i * numcrs + l + 1;
          *(cind++) = (k + 1) * numcrs + l + 1;
          *(cind++) = -1;
        }
        numcinds += 8;
      }
    }

    if (i < numscale-1)
      csc++;
    if (i < numrot-1)
      crot += 4;
  }

  if (caps & 2)  /* endCap should be drawn, build coordindices */
  {
    if (closedcrs)
    {
      for (i = 0; i < numcrs-1; i++)
        *(cind++) = (numspine - 1) * numcrs + i;
      numcinds += numcrs;
    }
    else
    {
      for (i = 0; i < numcrs; i++)
        *(cind++) = (numspine - 1) * numcrs + i;
      numcinds += numcrs+1;
    }
    *(cind++) = -1;
  }

  free (pz);


  /* step 3: texture coordinates */

  spinelength[0] = 0;
  for (i = 1; i < numspine; i++)
  {
    sub3D (spine[i], spine[i-1], tmp1);
    spinelength[i] = spinelength[i-1] + norm3D (tmp1);
  }
  /* printf ("spinelength = %f\n", spinelength[numspine-1]); */

  crslength[0] = 0;
  for (i = 1; i < numcrs; i++)
  {
    init3D (tmp1, crs[i].x - crs[i-1].x, crs[i].y - crs[i-1].y, 0);
    crslength[i] = crslength[i-1] + norm3D (tmp1);
  }
  /* printf ("crslength = %f\n", crslength[numcrs-1]); */

  cind = texcinds;

  if (caps)
  {
    xmin = xmax = crs[0].x;
    zmin = zmax = crs[0].y;
    for (i = 1; i < numcrs; i++)
    {
      if (crs[i].x < xmin)
        xmin = crs[i].x;
      if (crs[i].x > xmax)
        xmax = crs[i].x;
      if (crs[i].y < zmin)
        zmin = crs[i].y;
      if (crs[i].y > zmax)
        zmax = crs[i].y;
    }

    /* printf ("x: %f - %f, z: %f - %f\n", xmin, xmax, zmin, zmax); */

    for (i = 0; i < numcrs; i++)
    {
      ctexcrd->x = (crs[i].x - xmin) / (xmax - xmin);
      ctexcrd->y = 1 - (crs[i].y - zmin) / (zmax - zmin);
      /* printf ("%f, %f\n", ctexcrd->x, ctexcrd->y); */
      ctexcrd++;
    }
  }

  if (caps & 1)
  {
    if (closedcrs)
    {
      for (i = 1; i < numcrs; i++)
        *(cind++) = numcrs-i-1;
    }
    else
    {
      for (i = 0; i < numcrs; i++)
        *(cind++) = numcrs-i-1;
    }
    *(cind++) = -1;
    texcoordoffs += numcrs;
  }

  for (i = 0; i < numspine; i++)
  {
    for (j = 0; j < numcrs; j++)
    {
      ctexcrd->x = crslength[j] / crslength[numcrs-1];
      ctexcrd->y = spinelength[i] / spinelength[numspine-1];
      ctexcrd++;
      if (i < numspine-1 && j < numcrs-1)
      {
        if ((i + j) & 1)
        {
          *(cind++) = texcoordoffs + i * numcrs + j;
          *(cind++) = texcoordoffs + i * numcrs + j + 1;
          *(cind++) = texcoordoffs + (i + 1) * numcrs + j;
          *(cind++) = -1;
          *(cind++) = texcoordoffs + (i + 1) * numcrs + j;
          *(cind++) = texcoordoffs + i * numcrs + j + 1;
          *(cind++) = texcoordoffs + (i + 1) * numcrs + j + 1;
          *(cind++) = -1;
        }
        else
        {
          *(cind++) = texcoordoffs + i * numcrs + j;
          *(cind++) = texcoordoffs + (i + 1) * numcrs + j + 1;
          *(cind++) = texcoordoffs + (i + 1) * numcrs + j;
          *(cind++) = -1;
          *(cind++) = texcoordoffs + i * numcrs + j;
          *(cind++) = texcoordoffs + i * numcrs + j + 1;
          *(cind++) = texcoordoffs + (i + 1) * numcrs + j + 1;
          *(cind++) = -1;
        }
      }
    }
  }
      
  if (caps & 2)  /* endCap */
  {
    if (closedcrs)
    {
      for (i = 0; i < numcrs-1; i++)
        *(cind++) = i;
    }
    else
    {
      for (i = 0; i < numcrs; i++)
        *(cind++) = i;
    }
    *(cind++) = -1;
  }

  free (spinelength);
  free (crslength);

  return numcinds;
}  /* buildExtrusionData */
