/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.utils.params;

import java.io.*;
import java.util.Properties;

import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.utils.SubstanceConstants.ImageWatermarkKind;

/**
 * Reads the properties from a local file. This class is <b>for internal use
 * only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public class PropertiesFileParamReader implements ParamReader {
	/**
	 * Parser for the local file.
	 */
	protected Properties props;

	/**
	 * Loads property file.
	 * 
	 * @param configFile
	 *            Location of the configuration file.
	 * @return Property parser.
	 * @throws IOException
	 */
	private static Properties loadProperties(String configFile)
			throws IOException {
		InputStream in = null;
		try {
			in = new FileInputStream(configFile);
			Properties p = new Properties();
			p.load(in);
			return p;
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (Exception e) {
					// ignore
				}
			}
		}
	}

	/**
	 * Creates a new local file reader.
	 * 
	 * @param configFile
	 *            Location of the configuration file.
	 */
	public PropertiesFileParamReader(String configFile) {
		try {
			this.props = loadProperties(configFile);
		} catch (Exception exc) {
			this.props = null;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getButtonShaperProperty()
	 */
	public String getButtonShaperProperty() {
		if (this.props == null)
			return null;
		return this.props
				.getProperty(SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getGradientPainterProperty()
	 */
	public String getGradientPainterProperty() {
		if (this.props == null)
			return null;
		return this.props
				.getProperty(SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getTitlePainterProperty()
	 */
	public String getTitlePainterProperty() {
		if (this.props == null)
			return null;
		return this.props
				.getProperty(SubstanceLookAndFeel.TITLE_PAINTER_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getBorderPainterProperty()
	 */
	public String getBorderPainterProperty() {
		if (this.props == null)
			return null;
		return this.props
				.getProperty(SubstanceLookAndFeel.BORDER_PAINTER_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getThemeProperty()
	 */
	public String getThemeProperty() {
		if (this.props == null)
			return null;
		return this.props.getProperty(SubstanceLookAndFeel.THEME_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getTraceFileNameProperty()
	 */
	public String getTraceFileNameProperty() {
		if (this.props == null)
			return null;
		return this.props.getProperty(SubstanceLookAndFeel.TRACE_FILE);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getWatermarkThemeProperty()
	 */
	public String getWatermarkProperty() {
		if (this.props == null)
			return null;
		return this.props.getProperty(SubstanceLookAndFeel.WATERMARK_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getWatermarkImageProperty()
	 */
	public String getWatermarkImageProperty() {
		if (this.props == null)
			return null;
		return this.props
				.getProperty(SubstanceLookAndFeel.WATERMARK_IMAGE_PROPERTY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#toBleedWatermark()
	 */
	public boolean toBleedWatermark() {
		if (this.props == null)
			return false;
		String paramToBleedWatermark = this.props
				.getProperty(SubstanceLookAndFeel.WATERMARK_TO_BLEED);
		return (paramToBleedWatermark != null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#toShowExtraElementProperty()
	 */
	public boolean toShowExtraElementProperty() {
		if (this.props == null)
			return true;
		String paramNoExtraElements = this.props
				.getProperty(SubstanceLookAndFeel.NO_EXTRA_ELEMENTS);
		return (paramNoExtraElements == null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#isDebugUiMode()
	 */
	public boolean isDebugUiMode() {
		if (this.props == null)
			return true;
		String paramDebugUiMode = this.props
				.getProperty(SubstanceLookAndFeel.DEBUG_UI_MODE);
		return (paramDebugUiMode == null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#toShowHeapStatusPanelProperty()
	 */
	public String toShowHeapStatusPanelProperty() {
		if (this.props == null)
			return null;
		return this.props.getProperty(SubstanceLookAndFeel.HEAP_STATUS_PANEL);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#toEnableInvertedThemes()
	 */
	public boolean toEnableInvertedThemes() {
		if (this.props == null)
			return false;
		String paramEnableInvertedThemes = this.props
				.getProperty(SubstanceLookAndFeel.ENABLE_INVERTED_THEMES);
		return (paramEnableInvertedThemes != null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#toEnableNegatedThemes()
	 */
	public boolean toEnableNegatedThemes() {
		if (this.props == null)
			return false;
		String paramEnableNegatedThemes = this.props
				.getProperty(SubstanceLookAndFeel.ENABLE_NEGATED_THEMES);
		return (paramEnableNegatedThemes != null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getWatermarkImageKindProperty()
	 */
	public ImageWatermarkKind getWatermarkImageKindProperty() {
		if (this.props == null)
			return null;
		String paramWatermarkImageKind = this.props
				.getProperty(SubstanceLookAndFeel.WATERMARK_IMAGE_KIND);
		if (paramWatermarkImageKind == null)
			return null;
		return ImageWatermarkKind.valueOf(paramWatermarkImageKind);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.utils.params.ParamReader#getWatermarkImageOpacityProperty()
	 */
	public Float getWatermarkImageOpacityProperty() {
		if (this.props == null)
			return null;
		String paramWatermarkImageOpacity = this.props
				.getProperty(SubstanceLookAndFeel.WATERMARK_IMAGE_OPACITY);
		if (paramWatermarkImageOpacity == null)
			return null;
		try {
			return Float.parseFloat(paramWatermarkImageOpacity);
		} catch (NumberFormatException nfe) {
			return null;
		}
	}
}
