/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.utils;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;
import java.util.*;

import javax.swing.AbstractButton;

import org.jvnet.lafwidget.animation.*;
import org.jvnet.substance.SubstanceImageCreator;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.button.BaseButtonShaper;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.ComponentState.ColorSchemeKind;
import org.jvnet.substance.utils.SubstanceConstants.Side;

/**
 * Delegate class for painting backgrounds of buttons in <b>Substance </b> look
 * and feel. This class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public class PairwiseButtonBackgroundDelegate {
	/**
	 * Cache for background images for pairwise backgrounds. Each time
	 * {@link #getPairwiseBackground(AbstractButton, int, int, Side)} is called,
	 * it checks <code>this</code> map to see if it already contains such
	 * background. If so, the background from the map is returned.
	 */
	private static Map<String, BufferedImage> pairwiseBackgrounds = new HashMap<String, BufferedImage>();

	/**
	 * Contains information on a button background.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static class ButtonBackground {
		/**
		 * Indicates whether the button is painted in active visual state.
		 */
		public boolean isPaintedActive;

		/**
		 * The button background image.
		 */
		public BufferedImage backgroundImage;

		/**
		 * Button background info object.
		 * 
		 * @param isPaintedActive
		 *            Indicates whether the button is painted in active visual
		 *            state.
		 * @param backgroundImage
		 *            The button background image.
		 */
		public ButtonBackground(boolean isPaintedActive,
				BufferedImage backgroundImage) {
			this.isPaintedActive = isPaintedActive;
			this.backgroundImage = backgroundImage;
		}

	}

	/**
	 * Resets image maps (used when setting new theme).
	 * 
	 * @see SubstanceLookAndFeel#setCurrentTheme(String)
	 * @see SubstanceLookAndFeel#setCurrentTheme(SubstanceTheme)
	 */
	public static synchronized void reset() {
		pairwiseBackgrounds.clear();
	}

	/**
	 * Retrieves background image for the specified button in button pair (such
	 * as scrollbar arrows, for example).
	 * 
	 * @param button
	 *            Button.
	 * @param painter
	 *            Gradient painter.
	 * @param width
	 *            Button width.
	 * @param height
	 *            Button height.
	 * @param side
	 *            Button orientation.
	 * @param toIgnoreOpenSides
	 *            If <code>true</code>, the open side setting (controlled by
	 *            the {@link SubstanceLookAndFeel#BUTTON_OPEN_SIDE_PROPERTY} is
	 *            ignored.
	 * @return Button background image.
	 */
	public static BufferedImage getPairwiseBackground(AbstractButton button,
			SubstanceGradientPainter painter, int width, int height,
			SubstanceConstants.Side side, boolean toIgnoreOpenSides) {
		if (SubstanceCoreUtilities.isButtonNeverPainted(button))
			return null;

		BufferedImage resultNonFlat = null;
		ComponentState state = ComponentState.getState(button.getModel(),
				button);
		ComponentState prevState = SubstanceCoreUtilities
				.getPrevComponentState(button);

		float cyclePos = state.getCycleCount();
		ControlBackgroundComposite controlComposite = SubstanceCoreUtilities
				.getControlBackgroundComposite(button);

		SubstanceTheme theme = SubstanceThemeUtilities.getTheme(button, state);
		ColorScheme colorScheme = theme.getColorScheme();
		ColorScheme colorScheme2 = colorScheme;
		ColorScheme borderScheme = theme.getBorderTheme().getColorScheme();
		ColorScheme borderScheme2 = borderScheme;

		FadeTracker fadeTracker = FadeTracker.getInstance();
		FadeState fadeState = fadeTracker.getFadeState(button,
				FadeKind.ROLLOVER);
		if (fadeState != null) {
			Composite defaultComposite = controlComposite
					.getBackgroundComposite(button, button.getParent(), -1,
							false);
			Composite activeComposite = controlComposite
					.getBackgroundComposite(button, button.getParent(), -1,
							true);

			colorScheme = SubstanceThemeUtilities.getTheme(button, prevState)
					.getColorScheme();
			colorScheme2 = SubstanceThemeUtilities.getTheme(button, state)
					.getColorScheme();
			borderScheme = SubstanceThemeUtilities.getTheme(button, state)
					.getBorderTheme().getColorScheme();
			borderScheme2 = SubstanceThemeUtilities.getTheme(button, prevState)
					.getBorderTheme().getColorScheme();
			cyclePos = fadeState.getFadePosition();
			if (!fadeState.isFadingIn())
				cyclePos = 10 - cyclePos;

			// System.out.println("\t" + width + ":" + prevState.name() + ":"
			// + colorScheme.getClass().getSimpleName() + " -> "
			// + state.name() + ":"
			// + colorScheme2.getClass().getSimpleName() + " at "
			// + cyclePos);

			BufferedImage imageDefault = getPairwiseBackground(button, painter,
					width, height, side, cyclePos, colorScheme, colorScheme2,
					borderScheme, borderScheme2, defaultComposite,
					toIgnoreOpenSides);
			BufferedImage imageActive = getPairwiseBackground(button, painter,
					width, height, side, cyclePos, colorScheme, colorScheme2,
					borderScheme, borderScheme2, activeComposite,
					toIgnoreOpenSides);
			if (imageDefault == null)
				return null;
			Graphics2D graphics = imageDefault.createGraphics();
			graphics.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, cyclePos / 10.0f));
			// System.out.println("Painting [" + colorScheme + "-->"
			// + colorScheme2 + "]" + cyclePos / 10.0f);
			graphics.drawImage(imageActive, 0, 0, null);
			resultNonFlat = imageDefault;
		} else {
			Composite graphicsComposite = controlComposite
					.getBackgroundComposite(
							button,
							button.getParent(),
							0,
							(ComponentState.getState(button.getModel(), null)
									.getColorSchemeKind() == ColorSchemeKind.CURRENT));
			// System.out.println("\t" + width + ":" +
			// colorScheme.getClass().getSimpleName() + ":"
			// + colorScheme2.getClass().getSimpleName());
			resultNonFlat = getPairwiseBackground(button, painter, width,
					height, side, 0, colorScheme, colorScheme2, borderScheme,
					borderScheme2, graphicsComposite, toIgnoreOpenSides);
		}

		BufferedImage result = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D resultGr = result.createGraphics();
		if (SubstanceCoreUtilities.hasFlatAppearance(button.getParent(), false)) {
			// Special handling of flat buttons
			BufferedImage temp = SubstanceCoreUtilities.getBlankImage(
					resultNonFlat.getWidth(), resultNonFlat.getHeight());
			Graphics2D tempGr = temp.createGraphics();
			try {
				if (FadeTracker.getInstance().isTracked(button,
						FadeKind.ROLLOVER)
						&& !state.isKindActive(FadeKind.SELECTION)
						&& state.isKindActive(FadeKind.ENABLE)) {
					float fadeCoef = FadeTracker.getInstance().getFade10(
							button, FadeKind.ROLLOVER);
					tempGr.setComposite(AlphaComposite.getInstance(
							AlphaComposite.SRC_OVER, fadeCoef / 10.0f));
					tempGr.drawImage(resultNonFlat, 0, 0, null);
				} else {
					if ((state != ComponentState.DISABLED_UNSELECTED)
							&& (state != ComponentState.DEFAULT)) {
						tempGr.drawImage(resultNonFlat, 0, 0, null);
					}
				}
				resultGr.drawImage(temp, 0, 0, null);
			} finally {
				tempGr.dispose();
			}
		} else {
			resultGr.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, SubstanceThemeUtilities.getTheme(
							button).getThemeAlpha(button, state)));
			resultGr.drawImage(resultNonFlat, 0, 0, null);
		}
		return result;
	}

	/**
	 * Retrieves background image for the specified button in button pair (such
	 * as scrollbar arrows, for example).
	 * 
	 * @param button
	 *            Button.
	 * @param kind
	 *            Color scheme kind.
	 * @param painter
	 *            Gradient painter.
	 * @param width
	 *            Button width.
	 * @param height
	 *            Button height.
	 * @param side
	 *            Button orientation.
	 * @param cyclePos
	 *            Cycle position.
	 * @param colorScheme
	 *            The first color scheme.
	 * @param colorScheme2
	 *            The second color scheme.
	 * @param borderScheme
	 *            The first border color scheme.
	 * @param borderScheme2
	 *            The second border color scheme.
	 * @param graphicsComposite
	 *            Composite to apply before painting the button.
	 * @param toIgnoreOpenSides
	 *            If <code>true</code>, the open side setting (controlled by
	 *            the {@link SubstanceLookAndFeel#BUTTON_OPEN_SIDE_PROPERTY} is
	 *            ignored.
	 * @return Button background image.
	 */
	private synchronized static BufferedImage getPairwiseBackground(
			AbstractButton button, SubstanceGradientPainter painter, int width,
			int height, SubstanceConstants.Side side, float cyclePos,
			ColorScheme colorScheme, ColorScheme colorScheme2,
			ColorScheme borderScheme, ColorScheme borderScheme2,
			Composite graphicsComposite, boolean toIgnoreOpenSides) {
		if (SubstanceCoreUtilities.isButtonNeverPainted(button))
			return null;
		Set<Side> openSides = toIgnoreOpenSides ? new HashSet<Side>()
				: SubstanceCoreUtilities.getSides(button,
						SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY);
		String openKey = "";
		for (Side oSide : openSides) {
			openKey += oSide.name() + "-";
		}
		boolean noBorder = SubstanceCoreUtilities.isSpinnerButton(button)
				&& !button.getParent().isEnabled();
		String sideKey = (side == null) ? "null" : side.toString();
		String key = width + ":" + height + ":" + sideKey + ":" + cyclePos
				+ ":" + openKey + ":"
				+ SubstanceCoreUtilities.getSchemeId(colorScheme) + ":"
				+ SubstanceCoreUtilities.getSchemeId(colorScheme2) + ":"
				+ SubstanceCoreUtilities.getSchemeId(borderScheme) + ":"
				+ SubstanceCoreUtilities.getSchemeId(borderScheme2) + ":"
				+ button.getClass().getName() + ":" + painter.getDisplayName()
				+ ":" + noBorder;
		// System.out.println("\tKey " + key);
		if (!pairwiseBackgrounds.containsKey(key)) {
			// System.out.println("\tNot found");
			BufferedImage newBackground = null;

			// buttons will be rectangular to make two scrolls (horizontal
			// and vertical) touch the corners.
			int radius = 0;

			int deltaLeft = openSides.contains(Side.LEFT) ? 3 : 0;
			int deltaRight = openSides.contains(Side.RIGHT) ? 3 : 0;
			int deltaTop = openSides.contains(Side.TOP) ? 3 : 0;
			int deltaBottom = openSides.contains(Side.BOTTOM) ? 3 : 0;

			GeneralPath contour = null;

			// SubstanceGradientPainter painter = new StandardGradientPainter();

			SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
					.getBorderPainter(button);

			int borderDelta = (int) Math.floor(SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(button)) / 2.0);
			if (side != null) {
				switch (side) {
				case TOP:
				case BOTTOM:
					// arrow in vertical bar
					contour = BaseButtonShaper.getBaseOutline(height + deltaTop
							+ deltaBottom, width + deltaLeft + deltaRight,
							radius, null, borderDelta);

					newBackground = painter.getContourBackground(height
							+ deltaTop + deltaBottom, width + deltaLeft
							+ deltaRight, contour, false, colorScheme,
							colorScheme2, cyclePos, true,
							colorScheme != colorScheme2);
					if (!noBorder) {
						borderPainter.paintBorder(newBackground.getGraphics(),
								button, height + deltaTop + deltaBottom, width
										+ deltaLeft + deltaRight, contour,
								null, borderScheme, borderScheme2, cyclePos,
								borderScheme != borderScheme2);
					}
					newBackground = SubstanceImageCreator.getRotated(
							newBackground, 3);
					break;
				case RIGHT:
				case LEFT:
					// arrow in horizontal bar
					contour = BaseButtonShaper.getBaseOutline(width + deltaLeft
							+ deltaRight, height + deltaTop + deltaBottom,
							radius, null, borderDelta);

					newBackground = painter.getContourBackground(width
							+ deltaLeft + deltaRight, height + deltaTop
							+ deltaBottom, contour, false, colorScheme,
							colorScheme2, cyclePos, true,
							colorScheme != colorScheme2);
					if (!noBorder) {
						borderPainter.paintBorder(newBackground.getGraphics(),
								button, width + deltaLeft + deltaRight, height
										+ deltaTop + deltaBottom, contour,
								null, borderScheme, borderScheme2, cyclePos,
								borderScheme != borderScheme2);
					}
					break;
				}
			} else {
				contour = BaseButtonShaper.getBaseOutline(width + deltaLeft
						+ deltaRight, height + deltaTop + deltaBottom, radius,
						null, borderDelta);

				newBackground = painter.getContourBackground(width + deltaLeft
						+ deltaRight, height + deltaTop + deltaBottom, contour,
						false, colorScheme, colorScheme2, cyclePos, true,
						colorScheme != colorScheme2);
				if (!noBorder) {
					borderPainter.paintBorder(newBackground.getGraphics(),
							button, width + deltaLeft + deltaRight, height
									+ deltaTop + deltaBottom, contour, null,
							borderScheme, borderScheme2, cyclePos,
							borderScheme != borderScheme2);
				}
			}

			BufferedImage finalBackground = SubstanceCoreUtilities
					.getBlankImage(width, height);
			Graphics2D finalGraphics = (Graphics2D) finalBackground
					.getGraphics();
			finalGraphics.translate(-deltaLeft, -deltaTop);
			finalGraphics.drawImage(newBackground, 0, 0, null);
			// borderPainter.paintBorder(finalGraphics, width, height, contour,
			// colorScheme, colorScheme2, cyclePos,
			// colorScheme != colorScheme2);

			// System.out.println("\tCreated new background " + width + ":" +
			// height);
			pairwiseBackgrounds.put(key, finalBackground);
		}
		BufferedImage opaque = pairwiseBackgrounds.get(key);
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(opaque
				.getWidth(), opaque.getHeight());
		Graphics2D resultGr = result.createGraphics();
		resultGr.setComposite(graphicsComposite);
		resultGr.drawImage(opaque, 0, 0, null);
		resultGr.dispose();
		return result;
	}

	/**
	 * Simple constructor.
	 */
	public PairwiseButtonBackgroundDelegate() {
		super();
	}

	/**
	 * Updates background of the specified button.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param button
	 *            Button to update.
	 * @param side
	 *            Button side.
	 */
	public static void updateBackground(Graphics g, AbstractButton button,
			SubstanceConstants.Side side) {
		Graphics2D graphics = (Graphics2D) g.create();

		SubstanceGradientPainter painter = SubstanceCoreUtilities
				.getGradientPainter(button);
		int width = button.getWidth();
		int height = button.getHeight();

		painter = new SimplisticSoftBorderReverseGradientPainter();
		BufferedImage bgImage = getPairwiseBackground(button, painter, width,
				height, side, false);

		graphics.drawImage(bgImage, 0, 0, null);
		graphics.dispose();
	}
}
