/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.theme;

import java.awt.Color;
import java.awt.Component;
import java.util.*;

import javax.swing.JTabbedPane;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.substance.painter.SubstanceGradientPainter;
import org.jvnet.substance.painter.decoration.DecorationAreaType;
import org.jvnet.substance.theme.transform.ShiftTransform;
import org.jvnet.substance.theme.transform.ThemeTransform;
import org.jvnet.substance.utils.*;

/**
 * Complex theme. A complex theme is defined by four separate themes:
 * <ul>
 * <li>Active theme - used for the active (selected, pressed, rolled over)
 * controls.
 * <li>Default theme - used for inactive enabled controls.
 * <li>Disabled theme - used for disabled controls.
 * <li>Active title pane theme - used for title panes of active windows.
 * </ul>
 * This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceComplexTheme extends SubstanceTheme {
	/**
	 * Theme for controls in active visual state.
	 */
	protected SubstanceTheme activeTheme;

	/**
	 * Theme for controls in default visual state.
	 */
	protected SubstanceTheme defaultTheme;

	/**
	 * Theme for controls in disabled visual state.
	 */
	protected SubstanceTheme disabledTheme;

	/**
	 * Theme for title panes of active windows.
	 */
	protected SubstanceTheme activeTitlePaneTheme;

	/**
	 * Theme for title panes of inactive windows.
	 */
	protected SubstanceTheme defaultTitlePaneTheme;

	/**
	 * Theme for watermarks.
	 */
	protected SubstanceTheme watermarkTheme;

	/**
	 * Maps from component state to the matching themes. This map doesn't have
	 * to contain entries for all {@link ComponentState} instances.
	 */
	protected Map<ComponentState, SubstanceTheme> stateThemeMap;

	/**
	 * Maps from component state to the alpha channel applied on theme. This map
	 * doesn't have to contain entries for all {@link ComponentState} instances.
	 */
	protected Map<ComponentState, Float> stateThemeAlphaMap;

	/**
	 * Maps from component state to the matching themes for highlight themes.
	 * This map doesn't have to contain entries for all {@link ComponentState}
	 * instances.
	 */
	protected Map<ComponentState, SubstanceTheme> stateHighlightThemeMap;

	/**
	 * Maps from component state to the alpha channel applied on highlight
	 * theme. This map doesn't have to contain entries for all
	 * {@link ComponentState} instances.
	 */
	protected Map<ComponentState, Float> stateHighlightThemeAlphaMap;

	protected Map<DecorationAreaType, SubstanceTheme> decorationThemeMap;

	protected Set<DecorationAreaType> decoratedAreaSet;

	protected Set<DecorationAreaType> toUseDecorationOnActiveControls;

	/**
	 * Painter for painting the non-active controls.
	 */
	protected SubstanceGradientPainter nonActivePainter;

	/**
	 * The start of fade effect on the selected tabs in {@link JTabbedPane}s.
	 * 
	 * @see #selectedTabFadeEnd
	 */
	protected double selectedTabFadeStart;

	/**
	 * The end of fade effect on the selected tabs in {@link JTabbedPane}s.
	 * 
	 * @see #selectedTabFadeStart
	 */
	protected double selectedTabFadeEnd;

	protected boolean isPaintingToolbarDropShadows;

	/**
	 * If <code>true</code>, the light background color and the background
	 * color are the same.
	 * 
	 * @since version 4.0
	 */
	protected boolean useSameBackgroundColor;

	/**
	 * Creates a new complex theme.
	 * 
	 * @param displayName
	 *            Theme display name.
	 * @param themeKind
	 *            Theme kind.
	 * @param activeTheme
	 *            Active theme.
	 * @param defaultTheme
	 *            Default theme.
	 * @param disabledTheme
	 *            Disabled theme.
	 */
	public SubstanceComplexTheme(String displayName, ThemeKind themeKind,
			SubstanceTheme activeTheme, SubstanceTheme defaultTheme,
			SubstanceTheme disabledTheme) {
		this(displayName, themeKind, activeTheme, defaultTheme, disabledTheme,
				activeTheme, defaultTheme);
	}

	/**
	 * Creates a new complex theme.
	 * 
	 * @param displayName
	 *            Theme display name.
	 * @param themeKind
	 *            Theme kind.
	 * @param activeTheme
	 *            Active theme.
	 * @param defaultTheme
	 *            Default theme.
	 * @param disabledTheme
	 *            Disabled theme.
	 * @param activeTitlePaneTheme
	 *            Active title pane theme.
	 */
	public SubstanceComplexTheme(String displayName, ThemeKind themeKind,
			SubstanceTheme activeTheme, SubstanceTheme defaultTheme,
			SubstanceTheme disabledTheme, SubstanceTheme activeTitlePaneTheme) {
		this(displayName, themeKind, activeTheme, defaultTheme, disabledTheme,
				activeTitlePaneTheme, defaultTheme);
	}

	/**
	 * Creates a new complex theme.
	 * 
	 * @param displayName
	 *            Theme display name.
	 * @param themeKind
	 *            Theme kind.
	 * @param activeTheme
	 *            Active theme.
	 * @param defaultTheme
	 *            Default theme.
	 * @param disabledTheme
	 *            Disabled theme.
	 * @param activeTitlePaneTheme
	 *            Active title pane theme.
	 * @param watermarkTheme
	 *            Watermark theme.
	 */
	public SubstanceComplexTheme(String displayName, ThemeKind themeKind,
			SubstanceTheme activeTheme, SubstanceTheme defaultTheme,
			SubstanceTheme disabledTheme, SubstanceTheme activeTitlePaneTheme,
			SubstanceTheme watermarkTheme) {
		super(defaultTheme.getColorScheme(), displayName, themeKind);
		this.activeTheme = SubstanceThemeUtilities
				.getConstantTheme(activeTheme);
		this.defaultTheme = SubstanceThemeUtilities
				.getConstantTheme(defaultTheme);
		this.disabledTheme = SubstanceThemeUtilities
				.getConstantTheme(disabledTheme);
		this.activeTitlePaneTheme = SubstanceThemeUtilities
				.getConstantTheme(activeTitlePaneTheme);
		this.watermarkTheme = SubstanceThemeUtilities
				.getConstantTheme(watermarkTheme);
		this.selectedTabFadeStart = 1.0;
		this.selectedTabFadeEnd = 1.0;
		this.useSameBackgroundColor = false;

		this.stateThemeMap = new HashMap<ComponentState, SubstanceTheme>();
		this.stateThemeAlphaMap = new HashMap<ComponentState, Float>();
		this.stateHighlightThemeMap = new HashMap<ComponentState, SubstanceTheme>();
		this.stateHighlightThemeAlphaMap = new HashMap<ComponentState, Float>();
		this.decoratedAreaSet = new HashSet<DecorationAreaType>();
		this.decorationThemeMap = new HashMap<DecorationAreaType, SubstanceTheme>();
		this.toUseDecorationOnActiveControls = new HashSet<DecorationAreaType>();

		this.decorationThemeMap.put(DecorationAreaType.PRIMARY_TITLE_PANE,
				this.activeTitlePaneTheme);
		this.decorationThemeMap.put(DecorationAreaType.SECONDARY_TITLE_PANE,
				this.activeTitlePaneTheme);
		this.decoratedAreaSet.add(DecorationAreaType.PRIMARY_TITLE_PANE);
		this.decoratedAreaSet.add(DecorationAreaType.SECONDARY_TITLE_PANE);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getActiveTheme()
	 */
	@Override
	public SubstanceTheme getActiveTheme() {
		return this.activeTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDefaultTheme()
	 */
	@Override
	public SubstanceTheme getDefaultTheme() {
		return this.defaultTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDisabledTheme()
	 */
	@Override
	public SubstanceTheme getDisabledTheme() {
		return this.disabledTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getActiveTitlePaneTheme()
	 */
	@Override
	public SubstanceTheme getActiveTitlePaneTheme() {
		return this.activeTitlePaneTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDefaultTitlePaneTheme()
	 */
	@Override
	public SubstanceTheme getDefaultTitlePaneTheme() {
		if (this.defaultTitlePaneTheme == null)
			return this.getActiveTitlePaneTheme();
		return this.defaultTitlePaneTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getLightBackgroundColor()
	 */
	@Override
	public Color getLightBackgroundColor() {
		if (this.useSameBackgroundColor)
			return this.getBackgroundColor();
		return super.getLightBackgroundColor();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getFirstTheme()
	 */
	@Override
	public SubstanceTheme getFirstTheme() {
		SubstanceComplexTheme result = new SubstanceComplexTheme("First "
				+ this.getDisplayName(), this.getKind(), this.getActiveTheme()
				.getFirstTheme(), this.getDefaultTheme().getFirstTheme(), this
				.getDisabledTheme().getFirstTheme(), this
				.getActiveTitlePaneTheme().getFirstTheme());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getSecondTheme()
	 */
	@Override
	public SubstanceTheme getSecondTheme() {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Second "
				+ this.getDisplayName(), this.getKind(), this.getActiveTheme()
				.getSecondTheme(), this.getDefaultTheme().getSecondTheme(),
				this.getDisabledTheme().getSecondTheme(), this
						.getActiveTitlePaneTheme().getSecondTheme());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#saturate(double, boolean)
	 */
	@Override
	public SubstanceTheme saturate(double saturateFactor,
			boolean toSaturateEverything) {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Saturated ["
				+ toSaturateEverything + "] " + this.getDisplayName() + " "
				+ (int) (100 * saturateFactor) + "%", this.getKind(), this
				.getActiveTheme()
				.saturate(saturateFactor, toSaturateEverything),
				toSaturateEverything ? this.getDefaultTheme().saturate(
						saturateFactor, true) : this.getDefaultTheme(),
				toSaturateEverything ? this.getDisabledTheme().saturate(
						saturateFactor, true) : this.getDisabledTheme(),
				toSaturateEverything ? this.getActiveTitlePaneTheme().saturate(
						saturateFactor, true) : this.getActiveTitlePaneTheme(),
				toSaturateEverything ? this.getWatermarkTheme().saturate(
						saturateFactor, true) : this.getWatermarkTheme());
		this.copyTo(result, null);
		return result;
	}

	@Override
	public SubstanceTheme shift(Color backgroundShiftColor,
			double backgroundShiftFactor, Color foregroundShiftColor,
			double foregroundShiftFactor) {
		ThemeTransform shiftTransform = new ShiftTransform(
				backgroundShiftColor, backgroundShiftFactor,
				foregroundShiftColor, foregroundShiftFactor);
		ThemeTransform shiftDisabledTransform = new ShiftTransform(
				backgroundShiftColor, backgroundShiftFactor / 2.0,
				foregroundShiftColor, foregroundShiftFactor / 2.0);
		// ThemeTransform shiftDisabledTransform = new ShiftDisabledTransform(
		// backgroundShiftColor, backgroundShiftFactor,
		// foregroundShiftColor, foregroundShiftFactor);
		SubstanceComplexTheme result = new SubstanceComplexTheme("Shift "
				+ this.getDisplayName() + " to [" + backgroundShiftColor + "] "
				+ (int) (100 * backgroundShiftFactor) + "% * ["
				+ foregroundShiftColor + "]"
				+ (int) (100 * foregroundShiftFactor) + "%", this.getKind(),
				shiftTransform.transform(this.getActiveTheme()), shiftTransform
						.transform(this.getDefaultTheme()),
				shiftDisabledTransform.transform(this.getDisabledTheme()),
				shiftTransform.transform(this.getActiveTitlePaneTheme()),
				shiftTransform.transform(this.getWatermarkTheme()));
		this.copyTo(result, shiftTransform);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#shade(double)
	 */
	@Override
	public SubstanceTheme shade(double shadeFactor) {
		SubstanceComplexTheme result = new SubstanceComplexTheme(
				"Shade " + this.getDisplayName() + " "
						+ (int) (100 * shadeFactor) + "%", this.getKind(), this
						.getActiveTheme().shade(shadeFactor), this
						.getDefaultTheme().shade(shadeFactor), this
						.getDisabledTheme().shade(shadeFactor), this
						.getActiveTitlePaneTheme().shade(shadeFactor), this
						.getWatermarkTheme().shade(shadeFactor));
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tint(double)
	 */
	@Override
	public SubstanceTheme tint(double tintFactor) {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Tint "
				+ this.getDisplayName() + " " + (int) (100 * tintFactor) + "%",
				this.getKind(), this.getActiveTheme().tint(tintFactor), this
						.getDefaultTheme().tint(tintFactor), this
						.getDisabledTheme().tint(tintFactor), this
						.getActiveTitlePaneTheme().tint(tintFactor), this
						.getWatermarkTheme().tint(tintFactor));
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tone(double)
	 */
	@Override
	public SubstanceTheme tone(double toneFactor) {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Tone "
				+ this.getDisplayName() + " " + (int) (100 * toneFactor) + "%",
				this.getKind(), this.getActiveTheme().tone(toneFactor), this
						.getDefaultTheme().tone(toneFactor), this
						.getDisabledTheme().tone(toneFactor), this
						.getActiveTitlePaneTheme().tone(toneFactor), this
						.getWatermarkTheme().tone(toneFactor));
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#invert()
	 */
	@Override
	public SubstanceTheme invert() {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Inverted "
				+ this.getDisplayName(), this.getKind(), this.getActiveTheme()
				.invert(), this.getDefaultTheme().invert(), this
				.getDisabledTheme().invert(), this.getActiveTitlePaneTheme()
				.invert(), this.getWatermarkTheme().invert());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#negate()
	 */
	@Override
	public SubstanceTheme negate() {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Negated "
				+ this.getDisplayName(), this.getKind(), this.getActiveTheme()
				.negate(), this.getDefaultTheme().negate(), this
				.getDisabledTheme().negate(), this.getActiveTitlePaneTheme()
				.negate(), this.getWatermarkTheme().negate());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#hueShift(double)
	 */
	@Override
	public SubstanceTheme hueShift(double hueShiftFactor) {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Hue-shift "
				+ this.getDisplayName() + " " + (int) (100 * hueShiftFactor)
				+ "%", this.getKind(), this.getActiveTheme().hueShift(
				hueShiftFactor), this.getDefaultTheme()
				.hueShift(hueShiftFactor), this.getDisabledTheme().hueShift(
				hueShiftFactor), this.getActiveTitlePaneTheme().hueShift(
				hueShiftFactor), this.getWatermarkTheme().hueShift(
				hueShiftFactor));
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#protanopia()
	 */
	@Override
	public SubstanceTheme protanopia() {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Protanopia "
				+ this.getDisplayName(), this.getKind(), this.getActiveTheme()
				.protanopia(), this.getDefaultTheme().protanopia(), this
				.getDisabledTheme().protanopia(), this
				.getActiveTitlePaneTheme().protanopia(), this
				.getWatermarkTheme().protanopia());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#tritanopia()
	 */
	@Override
	public SubstanceTheme tritanopia() {
		SubstanceComplexTheme result = new SubstanceComplexTheme("Tritanopia "
				+ this.getDisplayName(), this.getKind(), this.getActiveTheme()
				.tritanopia(), this.getDefaultTheme().tritanopia(), this
				.getDisabledTheme().tritanopia(), this
				.getActiveTitlePaneTheme().tritanopia(), this
				.getWatermarkTheme().tritanopia());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#deuteranopia()
	 */
	@Override
	public SubstanceTheme deuteranopia() {
		SubstanceComplexTheme result = new SubstanceComplexTheme(
				"Deuteranopia " + this.getDisplayName(), this.getKind(), this
						.getActiveTheme().deuteranopia(), this
						.getDefaultTheme().deuteranopia(), this
						.getDisabledTheme().deuteranopia(), this
						.getActiveTitlePaneTheme().deuteranopia(), this
						.getWatermarkTheme().deuteranopia());
		this.copyTo(result, null);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getWatermarkTheme()
	 */
	@Override
	public SubstanceTheme getWatermarkTheme() {
		return this.watermarkTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getScrollbarTrackPainter()
	 */
	@Override
	public SubstanceGradientPainter getNonActivePainter() {
		if (this.nonActivePainter != null)
			return this.nonActivePainter;
		return super.getNonActivePainter();
	}

	/**
	 * Sets the painter for painting the non-active controls.
	 * 
	 * @param nonActivePainter
	 *            Painter for painting the non-active controls.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setNonActivePainter(SubstanceGradientPainter nonActivePainter) {
		this.nonActivePainter = nonActivePainter;
	}

	/**
	 * Sets the theme for title panes of inactive windows.
	 * 
	 * @param defaultTitlePaneTheme
	 *            Theme for title panes of inactive windows.
	 * @since version 3.3
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setDefaultTitlePaneTheme(SubstanceTheme defaultTitlePaneTheme) {
		this.defaultTitlePaneTheme = defaultTitlePaneTheme;
	}

	/**
	 * Sets the theme for the default visual state.
	 * 
	 * @param defaultTheme
	 *            The theme for the default visual state.
	 * @since version 4.0
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setDefaultTheme(SubstanceTheme defaultTheme) {
		this.defaultTheme = defaultTheme;
	}

	/**
	 * Sets the theme for the disabled visual state.
	 * 
	 * @param disabledTheme
	 *            The theme for the disabled visual state.
	 * @since version 4.0
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setDisabledTheme(SubstanceTheme disabledTheme) {
		this.disabledTheme = disabledTheme;
	}

	/**
	 * Sets the theme for the active visual state.
	 * 
	 * @param activeTheme
	 *            The theme for the active visual state.
	 * @since version 4.0
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setActiveTheme(SubstanceTheme activeTheme) {
		this.activeTheme = activeTheme;
	}

	/**
	 * Sets the theme for the title panes of active windows.
	 * 
	 * @param activeTitlePaneTheme
	 *            The theme for the title panes of active windows.
	 * @since version 4.0
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setActiveTitlePaneTheme(SubstanceTheme activeTitlePaneTheme) {
		this.activeTitlePaneTheme = activeTitlePaneTheme;

		this.decorationThemeMap.put(DecorationAreaType.PRIMARY_TITLE_PANE,
				activeTitlePaneTheme);
		this.decorationThemeMap.put(DecorationAreaType.SECONDARY_TITLE_PANE,
				activeTitlePaneTheme);
	}

	/**
	 * Sets the theme for the watermark.
	 * 
	 * @param watermarkTheme
	 *            The theme for the watermark.
	 * @since version 4.0
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setWatermarkTheme(SubstanceTheme watermarkTheme) {
		this.watermarkTheme = watermarkTheme;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getSelectedTabFadeStart()
	 */
	@Override
	public double getSelectedTabFadeStart() {
		return this.selectedTabFadeStart;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getSelectedTabFadeEnd()
	 */
	@Override
	public double getSelectedTabFadeEnd() {
		return this.selectedTabFadeEnd;
	}

	/**
	 * Sets the end of fade effect on the selected tabs in {@link JTabbedPane}s.
	 * The value should be in 0.0-1.0 range.
	 * 
	 * @param selectedTabFadeEnd
	 *            The end of fade effect on the selected tabs in
	 *            {@link JTabbedPane}s. Should be in 0.0-1.0 range.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setSelectedTabFadeEnd(double selectedTabFadeEnd) {
		this.selectedTabFadeEnd = selectedTabFadeEnd;
	}

	/**
	 * Sets the start of fade effect on the selected tabs in {@link JTabbedPane}s.
	 * The value should be in 0.0-1.0 range.
	 * 
	 * @param selectedTabFadeStart
	 *            The start of fade effect on the selected tabs in
	 *            {@link JTabbedPane}s. Should be in 0.0-1.0 range.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setSelectedTabFadeStart(double selectedTabFadeStart) {
		this.selectedTabFadeStart = selectedTabFadeStart;
	}

	/**
	 * Sets usage of background color for light background color.
	 * 
	 * @param useSameBackgroundColor
	 *            if <code>true</code>, the light background color and the
	 *            background color are the same.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setUseSameBackgroundColor(boolean useSameBackgroundColor) {
		this.useSameBackgroundColor = useSameBackgroundColor;
	}

	/**
	 * Registers a theme for the specific component state.
	 * 
	 * @param stateTheme
	 *            Theme for the specified component state.
	 * @param alpha
	 *            Alpha channel for the theme.
	 * @param useForHighlights
	 *            if <code>true</code>, the specified theme will be used as
	 *            highlight theme as well (for the same component state).
	 * @param states
	 *            Component states.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void registerComponentStateTheme(SubstanceTheme stateTheme,
			float alpha, boolean useForHighlights, ComponentState... states) {
		if (states != null) {
			for (ComponentState state : states) {
				this.stateThemeMap.put(state, stateTheme);
				this.stateThemeAlphaMap.put(state, alpha);
			}
		}
		if (useForHighlights) {
			this
					.registerComponentHighlightStateTheme(stateTheme, alpha,
							states);
		}
	}

	/**
	 * Registers a theme for the specific component state.
	 * 
	 * @param stateTheme
	 *            Theme for the specified component state.
	 * @param useForHighlights
	 *            if <code>true</code>, the specified theme will be used as
	 *            highlight theme as well (for the same component state).
	 * @param states
	 *            Component states.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void registerComponentStateTheme(SubstanceTheme stateTheme,
			boolean useForHighlights, ComponentState... states) {
		this.registerComponentStateTheme(stateTheme, 1.0f, useForHighlights,
				states);
	}

	/**
	 * Registers a highlight theme for the specific component state if the
	 * component state is not <code>null</code>, or a global highlight theme
	 * otherwise.
	 * 
	 * @param stateHighlightTheme
	 *            Highlight theme for the specified component state.
	 * @param states
	 *            Component states. If <code>null</code>, the specified theme
	 *            will be applied for all states left unspecified.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void registerComponentHighlightStateTheme(
			SubstanceTheme stateHighlightTheme, ComponentState... states) {
		if ((states == null) || (states.length == 0)) {
			for (ComponentState state : ComponentState.values()) {
				if (this.stateHighlightThemeMap.containsKey(state))
					continue;
				if (!state.isKindActive(FadeKind.ENABLE))
					continue;
				if (state == ComponentState.DEFAULT)
					continue;
				this.stateHighlightThemeMap.put(state, stateHighlightTheme);
			}
		} else {
			for (ComponentState state : states) {
				this.stateHighlightThemeMap.put(state, stateHighlightTheme);
			}
		}
	}

	/**
	 * Registers a highlight theme for the specific component state if the
	 * component state is not <code>null</code>, or a global highlight theme
	 * otherwise.
	 * 
	 * @param stateHighlightTheme
	 *            Highlight theme for the specified component state.
	 * @param alpha
	 *            Alpha channel for the highlight theme.
	 * @param states
	 *            Component states. If <code>null</code>, the specified theme
	 *            will be applied for all states left unspecified.
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void registerComponentHighlightStateTheme(
			SubstanceTheme stateHighlightTheme, float alpha,
			ComponentState... states) {
		if ((states == null) || (states.length == 0)) {
			for (ComponentState state : ComponentState.values()) {
				if (!state.isKindActive(FadeKind.ENABLE))
					continue;
				if (state == ComponentState.DEFAULT)
					continue;
				if (!this.stateHighlightThemeMap.containsKey(state))
					this.stateHighlightThemeMap.put(state, stateHighlightTheme);
				if (!this.stateHighlightThemeAlphaMap.containsKey(state))
					this.stateHighlightThemeAlphaMap.put(state, alpha);
			}
		} else {
			for (ComponentState state : states) {
				this.stateHighlightThemeMap.put(state, stateHighlightTheme);
				this.stateHighlightThemeAlphaMap.put(state, alpha);
			}
		}
	}

	@Override
	public SubstanceTheme getTheme(Component comp, ComponentState componentState) {
		return this.getTheme(comp, componentState, false);
	}

	@Override
	public SubstanceTheme getTheme(Component comp,
			ComponentState componentState, boolean toIgnoreHighlights) {
		if (!toIgnoreHighlights
				&& SubstanceHighlightManager.getInstance().toUseHighlightTheme(
						comp)) {
			SubstanceTheme registered = this.stateHighlightThemeMap
					.get(componentState);
			if (registered != null)
				return registered;
		} else {
			SubstanceTheme registered = this.stateThemeMap.get(componentState);
			if (registered != null)
				return registered;
		}
		return super.getTheme(comp, componentState, toIgnoreHighlights);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getHighlightTheme(java.awt.Component,
	 *      org.jvnet.substance.utils.ComponentState)
	 */
	@Override
	public SubstanceTheme getHighlightTheme(Component comp,
			ComponentState componentState) {
		SubstanceTheme registered = this.stateHighlightThemeMap
				.get(componentState);
		if (registered != null)
			return registered;
		return super.getHighlightTheme(comp, componentState);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getHighlightThemeAlpha(java.awt.Component,
	 *      org.jvnet.substance.utils.ComponentState)
	 */
	@Override
	public float getHighlightThemeAlpha(Component comp,
			ComponentState componentState) {
		Float registered = this.stateHighlightThemeAlphaMap.get(componentState);
		if (registered != null)
			return registered.floatValue();
		return super.getHighlightThemeAlpha(comp, componentState);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getThemeAlpha(java.awt.Component,
	 *      org.jvnet.substance.utils.ComponentState)
	 */
	@Override
	public float getThemeAlpha(Component comp, ComponentState componentState) {
		Float registered = this.stateThemeAlphaMap.get(componentState);
		if (registered != null)
			return registered.floatValue();
		return super.getThemeAlpha(comp, componentState);
	}

	/**
	 * Copies theme-agnostic settings of <code>this</code> theme to the
	 * specified theme.
	 * 
	 * @param dst
	 *            Theme to copy to.
	 * @param shiftTransform
	 *            Transformation to apply to the copied settings.
	 */
	protected void copyTo(SubstanceComplexTheme dst,
			ThemeTransform shiftTransform) {
		dst.setNonActivePainter(this.nonActivePainter);
		dst.setSelectedTabFadeStart(this.selectedTabFadeStart);
		dst.setSelectedTabFadeEnd(this.selectedTabFadeEnd);
		dst.setUseSameBackgroundColor(this.useSameBackgroundColor);
		for (Map.Entry<ComponentState, SubstanceTheme> entry : this.stateThemeMap
				.entrySet()) {
			float alpha = this.stateThemeAlphaMap.get(entry.getKey());
			if (shiftTransform == null)
				dst.registerComponentStateTheme(entry.getValue(), alpha, false,
						entry.getKey());
			else
				dst.registerComponentStateTheme(shiftTransform.transform(entry
						.getValue()), alpha, false, entry.getKey());
		}
		for (Map.Entry<ComponentState, SubstanceTheme> entry : this.stateHighlightThemeMap
				.entrySet()) {
			ComponentState state = entry.getKey();
			if (this.stateHighlightThemeAlphaMap.containsKey(state)) {
				float alpha = this.stateHighlightThemeAlphaMap.get(state);
				if (shiftTransform == null)
					dst.registerComponentHighlightStateTheme(entry.getValue(),
							alpha, entry.getKey());
				else
					dst
							.registerComponentHighlightStateTheme(
									shiftTransform.transform(entry.getValue()),
									alpha, entry.getKey());
			} else {
				if (shiftTransform == null)
					dst.registerComponentHighlightStateTheme(entry.getValue(),
							entry.getKey());
				else
					dst.registerComponentHighlightStateTheme(shiftTransform
							.transform(entry.getValue()), entry.getKey());
			}
		}
	}

	/**
	 * Registers the specified theme to be used on controls in decoration areas.
	 * 
	 * @param decorationTheme
	 *            The theme to use on controls in decoration areas.
	 * @param toUseOnActiveControls
	 *            If <code>false</code>, the parameter theme will only be
	 *            used on controls in default state. In this case, controls in
	 *            disabled or active states will use the global theme.
	 * @param areaTypes
	 *            Enumerates the area types that are affected by the parameter
	 *            theme.
	 * @since version 4.3
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void registerDecorationAreaTheme(SubstanceTheme decorationTheme,
			boolean toUseOnActiveControls, DecorationAreaType... areaTypes) {
		if (decorationTheme == null)
			return;

		SubstanceTheme theme = toUseOnActiveControls ? decorationTheme
				: SubstanceThemeUtilities.getConstantTheme(decorationTheme);
		for (DecorationAreaType areaType : areaTypes) {
			this.decoratedAreaSet.add(areaType);
			this.decorationThemeMap.put(areaType, theme);
			if (toUseOnActiveControls) {
				this.toUseDecorationOnActiveControls.add(areaType);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#getDecorationTheme(org.jvnet.substance.painter.decoration.DecorationAreaType)
	 */
	@Override
	public SubstanceTheme getDecorationTheme(DecorationAreaType decorationType) {
		SubstanceTheme result = this.decorationThemeMap.get(decorationType);
		if (result != null)
			return result;
		return this;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#toUseDecorationPainter(org.jvnet.substance.painter.decoration.DecorationAreaType)
	 */
	@Override
	public boolean toUseDecorationPainter(DecorationAreaType decorationType) {
		return this.decoratedAreaSet.contains(decorationType);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#toUseDecorationThemeOnActiveControls(org.jvnet.substance.painter.decoration.DecorationAreaType)
	 */
	@Override
	public boolean toUseDecorationThemeOnActiveControls(
			DecorationAreaType decorationType) {
		return this.toUseDecorationOnActiveControls.contains(decorationType);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.theme.SubstanceTheme#isPaintingToolbarDropShadows()
	 */
	@Override
	public boolean isPaintingToolbarDropShadows() {
		return this.isPaintingToolbarDropShadows;
	}

	/**
	 * Marks the toolbars to have drop shadows on the top few pixels.
	 * 
	 * @since version 4.3
	 * @deprecated This functionality will be moved to the base skin class in
	 *             version 5.0.
	 */
	public void setPaintingToolbarDropShadows() {
		this.isPaintingToolbarDropShadows = true;
	}
}
