/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.fonts;

import java.awt.Font;
import java.io.*;

import javax.swing.UIDefaults;

/**
 * The default font policy for KDE desktops.
 * 
 * @author <a href="mailto:paranoid.tiberiumlabs@gmail.com">Paranoid</a>
 */
public class DefaultKDEFontPolicy implements FontPolicy {

	/**
	 * method to check if current user session is KDE
	 * 
	 * @return {@Code true} if KDE session is currently running.
	 */
	public static boolean isKDERunning() {
		// KDE_FULL_SESSION=true
		return "true".equals(System.getenv("KDE_FULL_SESSION"));
	}

	private static final String SANS_SERIF = "sans";

	private static FontSet fontSet = null;

	public synchronized FontSet getFontSet(String lafName, UIDefaults table) {
		if (fontSet == null) {
			// size is the most important, then family and then style
			int commonSize = 10;
			int menuSize = 10;
			int titleSize = 10;
			int commonStyle = Font.PLAIN;
			int menuStyle = Font.PLAIN;
			int titleStyle = Font.BOLD;
			String commonFamily = SANS_SERIF;
			String menuFamily = SANS_SERIF;
			String titleFamily = SANS_SERIF;

			/*
			 * have some font params in this file.
			 * 
			 * [General] font menuFont [WM] activeFont
			 * 
			 * plain: null bold: Sans Serif,10,-1,5,75,0,0,0,0,0 italic: Sans
			 * Serif,10,-1,5,50,1,0,0,0,0 italic-bold: Sans
			 * Serif,10,-1,5,75,1,0,0,0,0
			 */
			// no use to do checks, they all will go back to default font, so
			// just catching exceptions...
			try {

				String kdeglobals = getFileContent(".kde/share/config/kdeglobals");

				// <editor-fold defaultstate="collapsed" desc=" reading common
				// font ">
				try {
					String font = getIniParam(kdeglobals, "[General]", "font");
					String[] params = font.split(",");
					commonSize = Integer.parseInt(params[1]);
					boolean bold = "75".equals(params[4]);
					boolean italic = "1".equals(params[5]);
					if (bold && italic) {
						commonStyle = Font.BOLD + Font.ITALIC;
					} else if (italic) {
						commonStyle = Font.ITALIC;
					} else if (bold) {
						commonStyle = Font.BOLD;
					} else {
						commonStyle = Font.PLAIN;
					}
				} catch (Exception fontReadingException) {
					commonStyle = Font.PLAIN;
					commonSize = 10;
					commonFamily = SANS_SERIF;
				}
				// </editor-fold>

				// <editor-fold defaultstate="collapsed" desc=" reading menu
				// font ">
				try {
					String font = getIniParam(kdeglobals, "[General]",
							"menuFont");
					String[] params = font.split(",");
					menuSize = Integer.parseInt(params[1]);
					boolean bold = "75".equals(params[4]);
					boolean italic = "1".equals(params[5]);
					if (bold && italic) {
						menuStyle = Font.BOLD + Font.ITALIC;
					} else if (italic) {
						menuStyle = Font.ITALIC;
					} else if (bold) {
						menuStyle = Font.BOLD;
					} else {
						menuStyle = Font.PLAIN;
					}
				} catch (Exception menuFontException) {
					menuSize = 10;
					menuStyle = Font.PLAIN;
					menuFamily = SANS_SERIF;
				}
				// </editor-fold>

				// <editor-fold defaultstate="collapsed" desc=" reading title
				// font ">
				try {
					String font = getIniParam(kdeglobals, "[WM]", "activeFont");
					String[] params = font.split(",");
					titleSize = Integer.parseInt(params[1]);
					boolean bold = "75".equals(params[4]);
					boolean italic = "1".equals(params[5]);
					if (bold && italic) {
						titleStyle = Font.BOLD + Font.ITALIC;
					} else if (italic) {
						// System.out.println("italic");
						titleStyle = Font.ITALIC;
					} else if (bold) {
						titleStyle = Font.BOLD;
					} else {
						titleStyle = Font.PLAIN;
					}
				} catch (Exception titleFontException) {
					titleSize = 10;
					titleStyle = Font.BOLD;
					titleFamily = SANS_SERIF;
				}
				// </editor-fold>

			} catch (Exception kdeglobalsReadingException) {
				commonSize = 10;
				menuSize = 10;
				titleSize = 10;

				commonStyle = Font.PLAIN;
				menuStyle = Font.PLAIN;
				titleStyle = Font.BOLD;

				commonFamily = SANS_SERIF;
				menuFamily = SANS_SERIF;
				titleFamily = SANS_SERIF;
			}

			double dcommonSize = commonSize;
			double dmenuSize = menuSize;
			double dtitleSize = titleSize;

			int dpi = 96;
			try {
				String dpiParam = getIniParam(
						getFileContent(".kde/share/config/kcmfonts"),
						"[General]", "forceFontDPI");
				dpi = Integer.parseInt(dpiParam);
			} catch (Exception ex) {
			}

			if (dpi < 0) {
				dpi = 96;
			}
			if (dpi < 50) {
				dpi = 50;
			}
			dcommonSize = ((commonSize * dpi) / 72.0);
			dmenuSize = ((menuSize * dpi) / 72.0);
			dtitleSize = ((titleSize * dpi) / 72.0);

			commonSize = (int) (dcommonSize + 0.5);
			if (commonSize < 1) {
				commonSize = 1;
			}

			menuSize = (int) (dmenuSize + 0.5);
			if (menuSize < 1) {
				menuSize = 1;
			}

			titleSize = (int) (dtitleSize + 0.5);
			if (titleSize < 1) {
				titleSize = 1;
			}

			Font commonFont = new Font(commonFamily, commonStyle, commonSize);
			Font menuFont = new Font(menuFamily, menuStyle, menuSize);
			Font titleFont = new Font(titleFamily, titleStyle, titleSize);

			fontSet = FontSets.createDefaultFontSet(commonFont, menuFont,
					commonFont, commonFont, commonFont, titleFont);
		}
		return fontSet;
	}

	private String getIniParam(String content, String category, String param)
			throws Exception {
		// checking if our param in our category - we don't need same params
		// from other categories
		int categoryIndex = content.indexOf(category);
		int nextCategoryIndex = content.indexOf('[', categoryIndex + 1);
		if (nextCategoryIndex < 0) {
			nextCategoryIndex = content.length();
		}
		int paramIndex = content.indexOf(param, categoryIndex);
		if (paramIndex >= 0 && paramIndex < nextCategoryIndex) {

			// now paramString contains full string of our parameter
			String paramString = content.substring(
					paramIndex,
					content.indexOf(System.getProperty("line.separator"),
							paramIndex)).trim();

			// getting just our value, we don't need other symbols
			int equalityIndex = paramString.indexOf('=');
			/*
			 * paramString.indexOf('#'); paramString.indexOf(';');
			 * 
			 * don't know do we need to remove trailing comments after this
			 * symbols? have nothing similar in my system...
			 */
			return paramString.substring(equalityIndex + 1);
		}
		throw new Exception("No such param in current category");
	}

	private String getFileContent(String filePath) throws IOException {
		String fileSeparator = System.getProperty("file.separator");
		if (!"/".equals(fileSeparator)) {
			filePath = filePath.replace("/", fileSeparator);
		}

		// creating file from user home, using system's file separator:
		BufferedReader in = new BufferedReader(new FileReader(System
				.getProperty("user.home")
				+ fileSeparator + filePath));
		StringBuilder sb = new StringBuilder();

		// size same as inside BufferedReader code
		char[] buffer = new char[8192];
		int read = 0;
		while ((read = in.read(buffer)) >= 0) {
			sb.append(buffer, 0, read);
		}
		in.close();

		return sb.toString();
	}

}
