/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.reflect.Constructor;
import java.util.*;

import javax.swing.*;
import javax.swing.plaf.UIResource;
import javax.swing.plaf.basic.BasicLookAndFeel;

import org.jvnet.lafplugin.*;
import org.jvnet.lafwidget.LafWidgetRepository;
import org.jvnet.lafwidget.animation.*;
import org.jvnet.lafwidget.utils.TrackableThread;
import org.jvnet.substance.border.*;
import org.jvnet.substance.button.*;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.combo.ComboPopupPrototypeCallback;
import org.jvnet.substance.fonts.*;
import org.jvnet.substance.grip.GripPainter;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.painter.decoration.*;
import org.jvnet.substance.painter.highlight.ClassicHighlightPainter;
import org.jvnet.substance.painter.highlight.SubstanceHighlightPainter;
import org.jvnet.substance.painter.text.DefaultTextPainter;
import org.jvnet.substance.painter.text.SubstanceTextPainter;
import org.jvnet.substance.plugin.*;
import org.jvnet.substance.skin.*;
import org.jvnet.substance.tabbed.BaseTabCloseListener;
import org.jvnet.substance.tabbed.TabCloseCallback;
import org.jvnet.substance.theme.*;
import org.jvnet.substance.theme.SubstanceTheme.ThemeKind;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.SubstanceConstants.ImageWatermarkKind;
import org.jvnet.substance.utils.SubstanceConstants.MenuGutterFillKind;
import org.jvnet.substance.utils.params.*;
import org.jvnet.substance.watermark.*;

import contrib.com.jgoodies.looks.common.ShadowPopupFactory;

/**
 * Main class for <b>Substance </b> look and feel. <b>All</b> static methods in
 * this class should be called when Substance is the currently set look and
 * feel.<br>
 * <br>
 * Since version 3.1, <b>Substance</b> no longer extends Metal. As before, this
 * is an implementation detail and not part of an official API.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceLookAndFeel extends BasicLookAndFeel {
	/**
	 * The name of plugin configuration XML resource name. This is used for the
	 * <a href="https://laf-plugin.dev.java.net">laf-plugin</a> support layer
	 * of third-party components.
	 */
	public static final String PLUGIN_XML = "META-INF/substance-plugin.xml";

	/**
	 * Plugin manager for component plugins.
	 */
	protected static ComponentPluginManager componentPlugins;

	/**
	 * Plugin manager for theme plugins.
	 */
	protected static PluginManager themePlugins;

	/**
	 * Plugin manager for skin plugins.
	 */
	protected static PluginManager skinPlugins;

	/**
	 * Plugin manager for watermark plugins.
	 */
	protected static PluginManager watermarkPlugins;

	/**
	 * Plugin manager for button shaper plugins.
	 */
	protected static PluginManager shaperPlugins;

	/**
	 * Plugin manager for gradient painter plugins.
	 */
	protected static PluginManager painterPlugins;

	/**
	 * Plugin manager for title painter plugins.
	 */
	protected static PluginManager titlePainterPlugins;

	/**
	 * Plugin manager for border painter plugins.
	 */
	protected static PluginManager borderPainterPlugins;

	/**
	 * List of all listeners on skin changes.
	 */
	protected static Set<SkinChangeListener> skinChangeListeners = new HashSet<SkinChangeListener>();

	/**
	 * List of all listeners on theme changes.
	 */
	protected static Set<ThemeChangeListener> themeChangeListeners;

	/**
	 * List of all listeners on watermark changes.
	 */
	protected static Set<WatermarkChangeListener> watermarkChangeListeners;

	/**
	 * List of all listeners on button shaper changes.
	 */
	protected static Set<ButtonShaperChangeListener> buttonShaperChangeListeners;

	/**
	 * List of all listeners on gradient painter changes.
	 */
	protected static Set<GradientPainterChangeListener> gradientPainterChangeListeners;

	/**
	 * List of all listeners on border painter changes.
	 */
	protected static Set<BorderPainterChangeListener> borderPainterChangeListeners;

	/**
	 * List of all listeners on closing tabs.
	 */
	protected static Set<LocaleChangeListener> localeChangeListeners;

	static {
		themeChangeListeners = new HashSet<ThemeChangeListener>();
		watermarkChangeListeners = new HashSet<WatermarkChangeListener>();
		buttonShaperChangeListeners = new HashSet<ButtonShaperChangeListener>();
		gradientPainterChangeListeners = new HashSet<GradientPainterChangeListener>();
		borderPainterChangeListeners = new HashSet<BorderPainterChangeListener>();
		localeChangeListeners = new HashSet<LocaleChangeListener>();

		themeChangeListeners.add(new ThemeChangeListener() {
			public void themeChanged() {
				for (Frame frame : Frame.getFrames())
					SubstanceCoreUtilities.resetMenuBars(frame);
			}
		});

		localeChangeListeners.add(new LocaleChangeListener() {
			public void localeChanged() {
				for (Frame frame : Frame.getFrames())
					SubstanceCoreUtilities.resetMenuBars(frame);
			}
		});

	}

	/**
	 * Contains a list of all control class names that should ignore the
	 * animations.
	 */
	protected static Set<String> ignoreAnimationsSet = new HashSet<String>();

	/**
	 * List of all globally registered mixed themes.
	 */
	protected static Map<String, SubstanceMixTheme> mixedThemes = new HashMap<String, SubstanceMixTheme>();

	/**
	 * Contains default background composites on per-class basis. This map is
	 * "consulted" if no {@link #BACKGROUND_COMPOSITE} property is set on the
	 * control.
	 * 
	 * @see #BACKGROUND_COMPOSITE
	 * @see #setBackgroundComposite(Class, ControlBackgroundComposite)
	 * @see #getBackgroundComposite(Component)
	 */
	protected static Map<Class<?>, ControlBackgroundComposite> backgroundComposites = new HashMap<Class<?>, ControlBackgroundComposite>();

	/**
	 * Indicates whether extra UI elements (such as menu items in system menu or
	 * menu search panel) should be shown.
	 * 
	 * @see #toShowExtraElements()
	 * @see #NO_EXTRA_ELEMENTS
	 */
	protected static boolean toShowExtraElements;

	/**
	 * Indicates whether option dialogs (error, question, warning, info) should
	 * use constant themes for icon coloring. Note that since version 4.0, the
	 * default setting is <code>true</code> (use constant theme). To use
	 * theme-consistent coloring, call
	 * {@link #setToUseConstantThemesOnDialogs(boolean)} and pass
	 * <code>true</code>.
	 * 
	 * @see #isToUseConstantThemesOnDialogs()
	 * @see #setToUseConstantThemesOnDialogs(boolean)
	 */
	protected static boolean toUseConstantThemesOnDialogs = true;

	/**
	 * Indicates whether inverted themes should be enabled.
	 * 
	 * @see #toEnableInvertedThemes()
	 * @see #ENABLE_INVERTED_THEMES
	 */
	protected static boolean toEnableInvertedThemes;

	/**
	 * Indicates whether inverted themes should be enabled.
	 * 
	 * @see #toEnableNegatedThemes()
	 * @see #ENABLE_NEGATED_THEMES
	 */
	protected static boolean toEnableNegatedThemes;

	/**
	 * Indicates whether the watermark should "bleed" through lists, tables and
	 * trees.
	 * 
	 * @see #toBleedWatermark()
	 * @see #WATERMARK_TO_BLEED
	 */
	protected static boolean toBleedWatermark;

	/**
	 * Indicates whether the application will be run in debug UI mode
	 * 
	 * @see #isDebugUiMode()
	 * @see #DEBUG_UI_MODE
	 */
	protected static boolean isDebugUiMode;

	/**
	 * Change listener on keyboard focus manager - fix for defect 208.
	 */
	protected PropertyChangeListener focusOwnerChangeListener;

	/**
	 * The current keyboard focus manager - fix for defect 208.
	 */
	protected KeyboardFocusManager currentKeyboardFocusManager;

	/**
	 * Fade kind for border animations. Enabled by default, use
	 * {@link FadeConfigurationManager#disallowFades(FadeKind)} to disable.
	 */
	public static final FadeKind BORDER_ANIMATION_KIND = new FadeKind(
			"substancelaf.borderAnimation");

	/**
	 * <p>
	 * Fade kind for tree decorations animation kind. Disabled by default, use
	 * {@link FadeConfigurationManager#allowFades(FadeKind)} to enable.
	 * </p>
	 * 
	 * <p>
	 * Tree decorations include collapse / expand icons and vertical /
	 * horizontal lines.
	 * </p>
	 * 
	 * @since version 3.3
	 */
	public static final FadeKind TREE_DECORATIONS_ANIMATION_KIND = new FadeKind(
			"substancelaf.treeDecorationsAnimation", false);

	/**
	 * Fade kind for smart tree scroll. Disabled by default, use
	 * {@link FadeConfigurationManager#allowFades(FadeKind)} to enable.
	 * </p>
	 * 
	 * <p>
	 * Smart tree scroll is relevant for scroll panes containing a tree. When
	 * enabled, it automatically scrolls the tree horizontally when the viewport
	 * shows mainly empty area (especially relevant for multi-level trees with
	 * narrow viewports).
	 * </p>
	 * 
	 * @since 4.0
	 */
	public final static FadeKind TREE_SMART_SCROLL_ANIMATION_KIND = new FadeKind(
			"substancelaf.treeSmartScrollAnimationKind", false);

	/**
	 * Property name for specifying the debug UI mode. Application running in
	 * debug UI mode will have additional UI controls for inspecting / changing
	 * the state of various controls. This property can be set as either VM flag
	 * (no value needed) or as a global setting on {@link UIManager}. In the
	 * latter case, the value should be either {@link Boolean#TRUE} or
	 * {@link Boolean#FALSE}.
	 * 
	 * <p>
	 * Example of setting this property via VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.debugUiMode
	 * </code>
	 * 
	 * <p>
	 * Example of setting this property on {@link UIManager}:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.DEBUG_UI_MODE, Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 3.1
	 */
	public static final String DEBUG_UI_MODE = "substancelaf.debugUiMode";

	/**
	 * Property name for setting theme. This property is used both as a JVM flag
	 * and as a client property that can be set on a specific component.
	 * 
	 * <p>
	 * If this property is used as a JVM flag, the value should be a
	 * fully-qualified name of the theme class. This class must have a default
	 * (no-argument) constructor. This means that you can not specify "generic"
	 * themes such as {@link SubstanceMixTheme} or {@link SubstanceToneTheme}.
	 * </p>
	 * 
	 * <p>
	 * If this property is used as a client property, the value can be one of:
	 * </p>
	 * <ul>
	 * <li>{@link String} - theme display name (only for primitive themes).</li>
	 * <li>{@link ThemeInfo} object (call {@link #getAllThemes()}).</li>
	 * <li>{@link SubstanceTheme} object itself.</li>
	 * </ul>
	 * 
	 * <p>
	 * Example of specifying a custom default theme as a VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.theme=org.jvnet.substance.theme.SubstanceBottleGreenTheme
	 * </code>
	 * 
	 * <p>
	 * Example of using a theme display name as client property value:
	 * </p>
	 * <code>
	 * JCheckBox cb = new JCheckBox("text");<br>
	 * cb.putClientProperty(SubstanceLookAndFeel.THEME_PROPERTY, "Bottle Green");
	 * </code>
	 * 
	 * <p>
	 * Example of using a {@link ThemeInfo} object as client property value:
	 * </p>
	 * <code>
	 * JCheckBox cb = new JCheckBox("text");<br>
	 * ThemeInfo ti = SubstanceLookAndFeel.getAllThemes().get("Bottle Green");<br>
	 * cb.putClientProperty(SubstanceLookAndFeel.THEME_PROPERTY, ti);
	 * </code>
	 * 
	 * <p>
	 * Example of using a {@link SubstanceTheme} object as client property
	 * value:
	 * </p>
	 * <code>
	 * JCheckBox cb = new JCheckBox("text");<br>
	 * cb.putClientProperty(SubstanceLookAndFeel.THEME_PROPERTY, <br>
	 * &nbsp;&nbsp;new SubstanceBottleGreenTheme());
	 * </code>
	 * 
	 * @since version 2.0
	 * @see #getCurrentThemeName()
	 * @see #getTheme()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 */
	public static final String THEME_PROPERTY = "substancelaf.theme";

	/**
	 * Property name for setting watermark. This is only for JVM flag. The value
	 * should be a fully-qualified name of the watermark class. This class must
	 * have a default (no-argument) constructor. This means that you can not
	 * specify "generic" watermark such as {@link SubstanceNoiseWatermark}.
	 * 
	 * <p>
	 * Example of specifying a custom default watermark as a VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.watermark=org.jvnet.substance.watermark.SubstanceKatakanaWatermark
	 * </code>
	 * 
	 * @since version 2.0
	 * @see #getCurrentWatermark()
	 * @see #getCurrentWatermarkName()
	 * @see #setCurrentWatermark(String)
	 * @see #setCurrentWatermark(SubstanceWatermark)
	 * @see #registerWatermarkChangeListener(WatermarkChangeListener)
	 * @see #unregisterWatermarkChangeListener(WatermarkChangeListener)
	 */
	public static final String WATERMARK_PROPERTY = "substancelaf.watermark";

	/**
	 * Property name for setting {@link SubstanceImageWatermark} watermark
	 * image. This is only for JVM flag. The value should be a location of
	 * watermark image as either filename or <i>http</i>-prefixed URL.
	 * 
	 * <p>
	 * Example of specifying a local image as watermark image:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.watermark.image=C:/images/myimage.jpg
	 * </code>
	 * 
	 * <p>
	 * Example of specifying a URL image as watermark image:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.watermark.image=http://images.com/myimage.jpg
	 * </code>
	 * 
	 * @since version 2.0
	 * @see #getImageWatermarkKind()
	 * @see #getImageWatermarkOpacity()
	 * @see #setImageWatermarkKind(ImageWatermarkKind)
	 * @see #setImageWatermarkOpacity(float)
	 * @see #WATERMARK_IMAGE_KIND
	 * @see #WATERMARK_IMAGE_OPACITY
	 */
	public static final String WATERMARK_IMAGE_PROPERTY = "substancelaf.watermark.image";

	/**
	 * Property name for requesting that watermark should bleed through lists,
	 * trees and tables. This is both a JVM flag and a client property on
	 * {@link UIManager} (global setting).
	 * 
	 * <p>
	 * When this property is used as a JVM flag, there is no need to specify any
	 * value:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.watermark.tobleed
	 * </code>
	 * 
	 * <p>
	 * When this property is set a client property on {@link UIManager}, the
	 * value should be one of {@link Boolean#TRUE} or {@link Boolean#FALSE}.
	 * For example,
	 * </p>
	 * <code>
	 * &nbsp;&nbsp;UIManager.put(SubstanceLookAndFeel.WATERMARK_TO_BLEED, Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 2.2
	 */
	public static final String WATERMARK_TO_BLEED = "substancelaf.watermark.tobleed";

	/**
	 * Client property name for requesting that watermark should not be painted
	 * on the component and its descendants. This property can be set either as
	 * client property on some component or as global property on
	 * {@link UIManager}. The value should be either {@link Boolean#TRUE} or
	 * {@link Boolean#FALSE}.
	 * 
	 * <p>
	 * In order to compute whether the current watermark should be painted on
	 * some component, the component's hierarchy is traversed bottom up. The
	 * first component that has this property set, defines the watermark
	 * visibility. Finally, if neither component not its ancestors define this
	 * property, the global setting on {@link UIManager} is checked. If there is
	 * no global setting, the watermark is <b>not</b> ignored (it is painted).
	 * Here is an example to illustrate the above:
	 * 
	 * <p>
	 * <code>
	 * JPanel topPanel = new JPanel();<br>
	 * topPanel.putClientProperty(SubstanceLookAndFeel.WATERMARK_IGNORE,<br>
	 * &nbsp;&nbsp;Boolean.TRUE);<br>
	 * JPanel panel1 = new JPanel();<br>
	 * JPanel panel2 = new JPanel();<br>
	 * panel2.putClientProperty(SubstanceLookAndFeel.WATERMARK_IGNORE, <br>
	 * &nbsp;&nbsp;Boolean.FALSE);<br>
	 * JTextField tf1 = new JTextField("field1");<br>
	 * tf1.putClientProperty(SubstanceLookAndFeel.WATERMARK_IGNORE, Boolean.TRUE);<br>
	 * JTextField tf2 = new JTextField("field2");<br>
	 * panel1.add(tf1);<br>
	 * panel1.add(tf2);<br>
	 * topPanel.add(panel1);<br>
	 * topPanel.add(panel2);
	 * </code>
	 * </p>
	 * 
	 * <p>
	 * In the code above, if {@link UIManager} has no value for this property,
	 * the watermark will be painted only on <b>panel2</b> and <b>tf1</b>. The
	 * <b>panel1</b> and <b>tf2</b> "inherit" the watermark-ignore setting
	 * from their <b>topPanel</b> ancestor.
	 * 
	 * @since version 2.2
	 */
	public static final String WATERMARK_IGNORE = "substancelaf.watermark.toignore";

	/**
	 * VM property for specifying the image watermark kind. Should be the name
	 * of one of the values in {@link SubstanceConstants.ImageWatermarkKind}
	 * enum. Is used only as a JVM flag. The default value (if no VM flag is
	 * specified) is
	 * {@link SubstanceConstants.ImageWatermarkKind#SCREEN_CENTER_SCALE}.
	 * Example of specifying custom default image watermark kind:
	 * 
	 * <p>
	 * <code>
	 * -Dsubstancelaf.watermark.image.kind=APP_ANCHOR
	 * </code>
	 * 
	 * @since version 2.3
	 * @see #getImageWatermarkKind()
	 * @see #getImageWatermarkOpacity()
	 * @see #setImageWatermarkKind(ImageWatermarkKind)
	 * @see #setImageWatermarkOpacity(float)
	 * @see #WATERMARK_IMAGE_OPACITY
	 * @see #WATERMARK_IMAGE_PROPERTY
	 */
	public static final String WATERMARK_IMAGE_KIND = "substancelaf.watermark.image.kind";

	/**
	 * VM property for specifying the image watermark opacity. Should be in
	 * 0.0-1.0 range. Is used only as a JVM flag. The default value (if no VM
	 * flag is specified) is 0.2. The default value is also taken if the VM flag
	 * value can not be parsed as a float number. If the VM flag value can be
	 * parsed as a float, but is outside of 0.0-1.0 range, an
	 * {@link IllegalArgumentException} will be thrown at the application
	 * startup. Example of setting custom default watermark image opacity: *
	 * 
	 * <p>
	 * <code>
	 * -Dsubstancelaf.watermark.image.opacity=0.6
	 * </code>
	 * 
	 * @since version 2.3
	 * @see #getImageWatermarkKind()
	 * @see #getImageWatermarkOpacity()
	 * @see #setImageWatermarkKind(ImageWatermarkKind)
	 * @see #setImageWatermarkOpacity(float)
	 * @see #WATERMARK_IMAGE_KIND
	 * @see #WATERMARK_IMAGE_PROPERTY
	 */
	public static final String WATERMARK_IMAGE_OPACITY = "substancelaf.watermark.image.opacity";

	/**
	 * Client property name for ignoring default (minimum) dimension for a
	 * single button. This property can be set either as a client property on a
	 * specific button or as a global setting on {@link UIManager}. The value
	 * should be either {@link Boolean#TRUE} or {@link Boolean#FALSE}.
	 * <p>
	 * Note that {@link SubstanceButtonShaper} implementations are not required
	 * to respect this property. The current implementations of the default
	 * {@link StandardButtonShaper} and {@link ClassicButtonShaper} respect this
	 * property.
	 * </p>
	 * 
	 * <p>
	 * Example of marking a button to ignore minimum dimension settings:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * <p>
	 * Example of marking all application buttons to ignore minimum dimension
	 * settings:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 2.1
	 */
	public static final String BUTTON_NO_MIN_SIZE_PROPERTY = "substancelaf.buttonnominsize";

	/**
	 * Client property name for specifying that a single control / all
	 * application controls should always be painted in active color (unless a
	 * control is disabled). This property can be set either as a client
	 * property on a specific control or as a global setting on
	 * {@link UIManager}. The value should be either {@link Boolean#TRUE} or
	 * {@link Boolean#FALSE}.
	 * 
	 * <p>
	 * Example of marking a button to be painted active:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * <p>
	 * Example of marking all application controls to be painted active:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 3.0
	 */
	public static final String PAINT_ACTIVE_PROPERTY = "substancelaf.paintactive";

	/**
	 * Client property name for specifying that a single button / all
	 * application buttons should never paint the background. This property can
	 * be set either as a client property on a specific button or as a global
	 * setting on {@link UIManager}. The value should be either
	 * {@link Boolean#TRUE} or {@link Boolean#FALSE}. Note that unlike the
	 * {@link #FLAT_PROPERTY}, a button marked with this property will <b>never</b>
	 * show the background (will always be painted flat).
	 * 
	 * <p>
	 * Example of marking a button to never paint background:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.BUTTON_PAINT_NEVER_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * <p>
	 * Example of marking all application buttons to never paint background:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.BUTTON_PAINT_NEVER_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 2.3
	 * @see #FLAT_PROPERTY
	 */
	public static final String BUTTON_PAINT_NEVER_PROPERTY = "substancelaf.buttonpaintnever";

	/**
	 * Client property name for specifying a straight side for a single button.
	 * This property must be set as a client property on a specific button. The
	 * value can be:
	 * 
	 * <p>
	 * <ul>
	 * <li>The name of one of the values in {@link SubstanceConstants.Side}
	 * enum.
	 * <li>Array of names of the values in {@link SubstanceConstants.Side}
	 * enum.
	 * <li>A value in {@link SubstanceConstants.Side} enum.
	 * <li>Array of values in {@link SubstanceConstants.Side} enum.
	 * </ul>
	 * 
	 * <p>
	 * Note that not all {@link SubstanceButtonShaper} implementations are
	 * required to respect this property. The default
	 * {@link StandardButtonShaper} and {@link ClassicButtonShaper} respect this
	 * property.
	 * </p>
	 * 
	 * <p>
	 * Example of marking a button to have straight north side:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.BUTTON_SIDE_PROPERTY,<br>
	 * &nbsp;&nbsp;SubstanceConstants.Side.RIGHT.name());
	 * </code>
	 * 
	 * @since version 2.1
	 * @see #BUTTON_OPEN_SIDE_PROPERTY
	 */
	public static final String BUTTON_SIDE_PROPERTY = "substancelaf.buttonside";

	/**
	 * Client property name for specifying an open side for a single button.
	 * This property must be set as a client property on a specific button. The
	 * value can be:
	 * 
	 * <p>
	 * <ul>
	 * <li>The name of one of the values in {@link SubstanceConstants.Side}
	 * enum.
	 * <li>Array of names of the values in {@link SubstanceConstants.Side}
	 * enum.
	 * <li>A value in {@link SubstanceConstants.Side} enum.
	 * <li>Array of values in {@link SubstanceConstants.Side} enum.
	 * </ul>
	 * </p>
	 * <p>
	 * Example of marking a button to have open north side:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * Set<Side> openSides = new HashSet<Side>();<br>
	 * openSides.put(Side.TOP);<br>
	 * button.putClientProperty(SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY, <br>
	 * &nbsp;&nbsp;openSides);
	 * </code>
	 * 
	 * @since version 3.1
	 * @see #BUTTON_SIDE_PROPERTY
	 */
	public static final String BUTTON_OPEN_SIDE_PROPERTY = "substancelaf.buttonopenSide";

	/**
	 * Property name for specifying button shaper. This property is used both as
	 * a JVM flag and as a client property that can be set on a specific button.
	 * 
	 * <p>
	 * If this property is used as a JVM flag, the value should be a
	 * fully-qualified name of the button shaper class. This class must have a
	 * default (no-argument) constructor.
	 * </p>
	 * 
	 * <p>
	 * If this property is used as a client property, the value can be one of:
	 * <ul>
	 * <li>{@link String} - fully-qualified name of the button shaper class.</li>
	 * <li>{@link SubstanceButtonShaper} object itself.</li>
	 * </ul>
	 * </p>
	 * 
	 * <p>
	 * Example of specifying a custom default button shaper as a VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.buttonShaper=org.jvnet.substance.button.ClassicButtonShaper
	 * </code>
	 * 
	 * <p>
	 * Example of using a shaper class name as client property value:
	 * </p>
	 * <code>
	 * JButton b = new JButton("text");<br>
	 * b.putClientProperty(SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY, <br>
	 * &nbsp;&nbsp;"org.jvnet.substance.button.ClassicButtonShaper");
	 * </code>
	 * 
	 * <p>
	 * Example of using a {@link SubstanceButtonShaper} object as client
	 * property value:
	 * </p>
	 * <code>
	 * JButton b = new JButton("text");<br>
	 * b.putClientProperty(SubstanceLookAndFeel.BUTTON_SHAPER_PROPERTY, <br>
	 * &nbsp;&nbsp;new ClassicButtonShaper());
	 * </code>
	 * 
	 * @since version 2.1
	 * @see #getCurrentButtonShaper()
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(String)
	 * @see #setCurrentButtonShaper(SubstanceButtonShaper)
	 * @see #registerButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #unregisterButtonShaperChangeListener(ButtonShaperChangeListener)
	 */
	public static final String BUTTON_SHAPER_PROPERTY = "substancelaf.buttonShaper";

	/**
	 * Client property name for specifying the corner radius for buttons.
	 * Currently, this property is respected only on toolbar buttons. This
	 * property can be set either as a client property on a specific toolbar
	 * button, a client property on a specific toolbar (will hold for all
	 * buttons in the toolbar) or as a global setting on {@link UIManager}. The
	 * value should be a {@link Float} instance.
	 * 
	 * <p>
	 * Example of specifying a (toolbar) button to have corner radius of 5
	 * pixels:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.CORNER_RADIUS, <br>
	 * &nbsp;&nbsp;Float.valueOf(5.0f));
	 * </code>
	 * 
	 * <p>
	 * Example of specifying all buttons of a toolbar to have corner radius of 3
	 * pixels:
	 * </p>
	 * <code>
	 * JToolBar toolbar = new JToolBar("toolbar");<br>
	 * toolbar.putClientProperty(SubstanceLookAndFeel.CORNER_RADIUS, <br>
	 * &nbsp;&nbsp;Float.valueOf(3.0f));
	 * </code>
	 * 
	 * <p>
	 * Example of specifying all toolbar buttons to have corner radius of 0
	 * pixels:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.CORNER_RADIUS, Float.valueOf(0.0f));
	 * </code>
	 * 
	 * @since version 3.0
	 */
	public static final String CORNER_RADIUS = "substancelaf.cornerRadius";

	/**
	 * Property name for specifying that the component should be painted flat
	 * (no background / border) when it's inactive. This property should be
	 * specified on a specific component or its parent and must have either
	 * {@link Boolean#TRUE} or {@link Boolean#FALSE} value. For example, this is
	 * how to mark a button to appear flat:
	 * <p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.FLAT_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 3.0
	 * @see #BUTTON_PAINT_NEVER_PROPERTY
	 */
	public static final String FLAT_PROPERTY = "substancelaf.componentFlat";

	/**
	 * Client property name for specifying that a single control / all
	 * application controls have overlay functionality. This property can be set
	 * either as a client property on a specific control or as a global setting
	 * on {@link UIManager}. The value should be either {@link Boolean#TRUE} or
	 * {@link Boolean#FALSE}.
	 * 
	 * <p>
	 * Example of marking a scroll pane to have overlays on the scroll bars:
	 * </p>
	 * <code>
	 * JPanel myPanel = new JPanel();<br>
	 * JScrollPane jsp = new JScrollPane(myPanel);<br>
	 * myPanel.putClientProperty(SubstanceLookAndFeel.OVERLAY_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * <p>
	 * Example of marking all application controls to have overlays:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.OVERLAY_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 3.2
	 */
	public static final String OVERLAY_PROPERTY = "substancelaf.overlay";

	/**
	 * Property name for specifying custom grip handles. This applies to scroll
	 * bars, split pane dividers and other relevant controls. The value should
	 * be an instance of {@link GripPainter} interface.
	 * 
	 * <p>
	 * Example of marking a scroll pane to have custom grip handles on its
	 * scroll bars:
	 * </p>
	 * <code>
	 * JPanel myPanel = new JPanel();<br>
	 * JScrollPane jsp = new JScrollPane(myPanel);<br>
	 * myPanel.putClientProperty(SubstanceLookAndFeel.GRIP_PAINTER, <br>
	 * &nbsp;&nbsp;new DragBumpsGripPainter());
	 * </code>
	 * 
	 * @since version 3.2
	 * @deprecated Will be removed in version 5.0
	 */
	public static final String GRIP_PAINTER = "substancelaf.gripPainter";

	/**
	 * Property name for specifying gradient painter. This property is used both
	 * as a JVM flag and as a client property that can be set on a specific
	 * button.
	 * 
	 * <p>
	 * For the JVM flag, the value should be fully-qualified name of the
	 * gradient painter class. This class must have a default (no-argument)
	 * constructor. For the client property, the value can be either a
	 * fully-qualified class name as above or an instance of
	 * {@link SubstanceGradientPainter}.
	 * </p>
	 * 
	 * <p>
	 * Example of specifying a custom default gradient painter as a VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.gradientPainter=org.jvnet.substance.painter.SpecularGradientPainter
	 * </code>
	 * 
	 * <p>
	 * Example of using a painter class name as client property value:
	 * </p>
	 * <code>
	 * JButton b = new JButton("text");<br>
	 * b.putClientProperty(SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY, <br>
	 * &nbsp;&nbsp;"org.jvnet.substance.painter.SpecularGradientPainter");
	 * </code>
	 * 
	 * <p>
	 * Example of using a painter instance as client property value:
	 * </p>
	 * <code>
	 * JButton b = new JButton("text");<br>
	 * b.putClientProperty(SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY,<br>
	 * &nbsp;&nbsp;new SpecularGradientPainter());
	 * </code>
	 * 
	 * @since version 2.1
	 * @see #getCurrentGradientPainter()
	 * @see #getCurrentGradientPainterName()
	 * @see #setCurrentGradientPainter(String)
	 * @see #setCurrentGradientPainter(SubstanceGradientPainter)
	 * @see #registerGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #unregisterGradientPainterChangeListener(GradientPainterChangeListener)
	 */
	public static final String GRADIENT_PAINTER_PROPERTY = "substancelaf.gradientPainter";

	/**
	 * Property name for specifying title painter. This property is used both as
	 * a JVM flag and as a client property that can be set on a specific root
	 * pane / internal frame.
	 * 
	 * <p>
	 * For JVM flag, the value should be a fully-qualified name of the title
	 * painter class. This class must have a default (no-argument) constructor.
	 * For client property, the value should be either a fully-qualified class
	 * name as above or an instance of {@link SubstanceTitlePainter}.
	 * </p>
	 * 
	 * <p>
	 * Example of specifying a custom default title painter as a VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.titlePainter=org.jvnet.substance.title.ClassicTitlePainter
	 * </code>
	 * 
	 * <p>
	 * Example of using a painter class name as client property value:
	 * </p>
	 * <code>
	 * JFrame fr = new JFrame("text");<br>
	 * fr.getRootPane().putClientProperty(SubstanceLookAndFeel.TITLE_PAINTER_PROPERTY,<br>
	 * &nbsp;&nbsp;"org.jvnet.substance.title.ClassicTitlePainter");
	 * </code>
	 * 
	 * @since version 3.0
	 * @see #getCurrentTitlePainter()
	 * @see #getCurrentTitlePainterName()
	 * @see #setCurrentTitlePainter(String)
	 * @see #setCurrentTitlePainter(SubstanceTitlePainter)
	 * @see #registerTitlePainterChangeListener(TitlePainterChangeListener)
	 * @see #unregisterTitlePainterChangeListener(TitlePainterChangeListener)
	 */
	public static final String TITLE_PAINTER_PROPERTY = "substancelaf.titlePainter";

	/**
	 * Property name for specifying border painter. This property is used both
	 * as a JVM flag and as a client property that can be set on a specific
	 * component.
	 * 
	 * <p>
	 * For JVM flag, the value should be a fully-qualified name of the border
	 * painter class. This class must have a default (no-argument) constructor.
	 * For client property, the value should be either a fully-qualified class
	 * name as above or an instance of {@link SubstanceBorderPainter}.
	 * </p>
	 * 
	 * <p>
	 * Example of specifying a custom default border painter as a VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.borderPainter=org.jvnet.substance.border.ClassicBorderPainter
	 * </code>
	 * 
	 * <p>
	 * Example of using a painter class name as client property value:
	 * </p>
	 * <code>
	 * JButton button = new JButton("text");<br>
	 * button.putClientProperty(SubstanceLookAndFeel.BORDER_PAINTER_PROPERTY,<br>
	 * &nbsp;&nbsp;"org.jvnet.substance.border.ClassicBorderPainter");
	 * </code>
	 * 
	 * @since version 4.0
	 * @see #getCurrentBorderPainter()
	 * @see #getCurrentBorderPainterName()
	 * @see #setCurrentBorderPainter(String)
	 * @see #setCurrentBorderPainter(SubstanceBorderPainter)
	 * @see #registerBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #unregisterBorderPainterChangeListener(BorderPainterChangeListener)
	 */
	public static final String BORDER_PAINTER_PROPERTY = "substancelaf.borderPainter";

	/**
	 * Property name for specifying that custom frame title panes should show
	 * the heap status panel. Can be used as a VM flag.
	 * 
	 * <p>
	 * When this VM flag is set (no need to specify a value), the system menu on
	 * decorated title panes will have a check box menu item that controls the
	 * visibility of the heap status panel. The
	 * {@link #permanentlyShowHeapStatusPanel(JRootPane)} (can be called <b>only</b>
	 * when the VM flag is set) removes the above check box menu item and adds
	 * the heap status panel to the title pane (if it wasn't shown prior to
	 * calling that method). The
	 * {@link #permanentlyHideHeapStatusPanel(JRootPane)} (can be called <b>only</b>
	 * when the VM flag is set) removes the above check box menu item and
	 * removes the heap status panel to the title pane (if it was shown prior to
	 * calling that method).
	 * </p>
	 * 
	 * <code>
	 * -Dsubstancelaf.heapStatusPanel
	 * </code>
	 * 
	 * @since version 2.1
	 * @see #permanentlyHideHeapStatusPanel(JRootPane)
	 * @see #permanentlyShowHeapStatusPanel(JRootPane)
	 */
	public static final String HEAP_STATUS_PANEL = "substancelaf.heapStatusPanel";

	/**
	 * Client property name for specifying that contents of some frame, dialog,
	 * internal frame, desktop icon or tab have been modified and not saved. The
	 * property can be set on:
	 * <p>
	 * <ul>
	 * <li>{@link JInternalFrame} - the <b>close</b> button of the specific
	 * internal frame will be animated (in case that the internal frame has
	 * decorated title pane). In addition, once such internal frame is iconified
	 * (to a {@link JInternalFrame.JDesktopIcon}), the close button of the
	 * matching desktop icon is animated as well.</li>
	 * <li>{@link JRootPane} - the <b>close</b> button of the title pane of
	 * the matching frame / dialog will be animated (in case that the frame /
	 * dialog have decorated title pane).</li>
	 * <li>{@link JComponent} in a {@link JTabbedPane}. Based on the
	 * {@link #TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION} property presence,
	 * either the entire tab or its close button area is animated. In this case,
	 * this property must be set on the tab component itself, <b>not</b> on one
	 * of its child components.</li>
	 * </ul>
	 * </p>
	 * <p>
	 * The animation cycles between red, orange and yellow themes. In most cases
	 * (all but tabs not marked with
	 * {@link #TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION} property), the
	 * animation will be visible only when the mouse hovers over the close
	 * button of the matching container (frame, dialog, internal frame, desktop
	 * icon, tab). The tooltip of the close button is changed as well to reflect
	 * that the container contents are marked as modified.
	 * </p>
	 * 
	 * <p>
	 * Here is a sample text editing application that illustrates the use of
	 * this property. Once the contents of the text pane are changed, the frame
	 * is marked as modified. The <b>Save</b> button marks the frame as
	 * not-modified. In the real application, the listener on this button will
	 * need to persist the changes as well.
	 * </p>
	 * 
	 * <code>
	 * public class Changer extends JFrame {<br>
	 * &nbsp;&nbsp;public Changer() {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;super("Changer");<br>
	 * <br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;this.setLayout(new BorderLayout());<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;JTextPane textArea = new JTextPane();<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;this.add(textArea, BorderLayout.CENTER);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;textArea.getDocument().addDocumentListener(new
	 * DocumentListener() {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;private void handleChange() {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;getRootPane().putClientProperty(<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;SubstanceLookAndFeel.WINDOW_MODIFIED,
	 * Boolean.TRUE);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;public void
	 * changedUpdate(DocumentEvent e) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;handleChange();<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;public void
	 * insertUpdate(DocumentEvent e) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;handleChange();<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;public void
	 * removeUpdate(DocumentEvent e) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;handleChange();<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;});<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;JPanel buttons = new JPanel(new
	 * FlowLayout(FlowLayout.RIGHT));<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;JButton saveButton = new JButton("Save");<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;saveButton.addActionListener(new ActionListener() {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;public void
	 * actionPerformed(ActionEvent e) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;getRootPane().putClientProperty(<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;SubstanceLookAndFeel.WINDOW_MODIFIED,
	 * Boolean.FALSE);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;});<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;buttons.add(saveButton);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;this.add(buttons, BorderLayout.SOUTH);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);<br>
	 * &nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;public static void main(String ... args) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;try {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;UIManager.setLookAndFeel(new
	 * SubstanceLookAndFeel());<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;catch (Exception exc) {}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;JFrame.setDefaultLookAndFeelDecorated(true);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;Changer ch = new Changer();<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;ch.setPreferredSize(new Dimension(200, 200));<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;ch.setSize(ch.getPreferredSize());<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;ch.setLocationRelativeTo(null);<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;ch.setVisible(true);<br>
	 * &nbsp;&nbsp;}<br> }
	 * </code>
	 * 
	 * @since version 2.1
	 */
	public final static String WINDOW_MODIFIED = "windowModified";

	/**
	 * Client property name for specifying the background composite for various
	 * translucency effects. This property can be set either as a client
	 * property on a specific control or as a global setting on
	 * {@link UIManager}. The value should be an instance of
	 * {@link ControlBackgroundComposite}. Available core implementations
	 * include {@link AlphaControlBackgroundComposite},
	 * {@link DefaultControlBackgroundComposite} and
	 * {@link DecayControlBackgroundComposite}.
	 * </p>
	 * 
	 * <p>
	 * Note that setting this property on a container (such as
	 * {@link JScrollPane}) will affect all its children (recursively) unless
	 * they set this property.
	 * </p>
	 * 
	 * <p>
	 * Example of setting a custom background composite on a scroll pane (will
	 * affect the painting of the scroll bars and all its children):
	 * </p>
	 * <code>
	 * JPanel myPanel = new JPanel();<br>
	 * JScrollPane jsp = new JScrollPane(myPanel);<br>
	 * myPanel.putClientProperty(SubstanceLookAndFeel.BACKGROUND_COMPOSITE,<br>
	 * &nbsp;&nbsp;new AlphaControlBackgroundComposite(0.4f, 0.7f));
	 * </code>
	 * 
	 * <p>
	 * Example of marking all application controls with the same background
	 * composite:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.BACKGROUND_COMPOSITE, <br>
	 * &nbsp;&nbsp;new AlphaControlBackgroundComposite(0.6f));
	 * </code>
	 * 
	 * @since version 3.1
	 */
	public final static String BACKGROUND_COMPOSITE = "substancelaf.backgroundComposite";

	/**
	 * Client property name for specifying that a single tab / all tabs of a
	 * single tabbed pane / all tabs of all tabbed panes should have close
	 * buttons. This property can be specified on a single tab component, on a
	 * {@link JTabbedPane} itself (will hold for all tab components that don't
	 * define this property) or on {@link UIManager}. The value should be
	 * either {@link Boolean#TRUE} or {@link Boolean#FALSE}. By default, the
	 * close buttons are not displayed.
	 * 
	 * <p>
	 * Example of setting that all tabs in the application will have close
	 * buttons:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * <p>
	 * A more complex example:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);<br>
	 * JTabbedPane jtpMain = new JTabbedPane();<br>
	 * JTabbedPane jtpSecondary = new JTabbedPane();<br>
	 * jtpSecondary.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.FALSE);<br>
	 * JPanel panelSecondary = new JPanel();<br>
	 * jtpMain.addTab(jtpSecondary);<br>
	 * jtpMain.addTab(panelSecondary);<br>
	 * JPanel tab1 = new JPanel();<br>
	 * JPanel tab2 = new JPanel();<br>
	 * tab2.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_PROPERTY, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);<br>
	 * jtpSecondary.addTab(tab1);<br>
	 * jtpSecondary.addTab(tab2);
	 * </code>
	 * 
	 * <p>
	 * In the example above, the first first-level child (<b>jtpSecondary</b>)
	 * doesn't have the close button (since it overrides the global setting).
	 * The second first-level child tab (<b>panelSecondary</b>) has close
	 * button (from the global <b>UIManager</b> setting). The first
	 * second-level tab doesn't have the close button (setting inherited from
	 * the parent <b>jtpSecondary</b> tab that overrides the global setting).
	 * The second second-level tab has the close button (since its setting
	 * overrides the parent setting).
	 * </p>
	 * 
	 * @since version 2.1
	 * @see #TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION
	 * @see #TABBED_PANE_CLOSE_CALLBACK
	 */
	public final static String TABBED_PANE_CLOSE_BUTTONS_PROPERTY = "substancelaf.tabbedpanehasclosebuttons";

	/**
	 * Client property name for specifying that a single tabbed pane / all
	 * tabbed panes should have vertical layout (rotated tabs) on
	 * {@link SwingConstants#LEFT} and {@link SwingConstants#RIGHT} tab
	 * placements. This property can be specified either on a single
	 * {@link JTabbedPane} or on {@link UIManager}. The value should be either
	 * {@link Boolean#TRUE} or {@link Boolean#FALSE}. By default, the tab
	 * orientation on all placements is horizontal.
	 * 
	 * <p>
	 * Example of setting that all tabbed panes in the application will be
	 * vertically oriented:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * <p>
	 * Example of specifying that the specific tabbed pane will be vertically
	 * oriented:
	 * </p>
	 * <code>
	 * JTabbedPane jtpMain = new JTabbedPane();<br>
	 * jtpMain.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 2.2
	 * @see #TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS
	 */
	public final static String TABBED_PANE_VERTICAL_ORIENTATION = "substancelaf.tabbedpaneverticalOrientation";

	/**
	 * Client property name for specifying that when a single tabbed pane / all
	 * tabbed panes is layed-out vertically, the tab icons remain unrotated.This
	 * property can be specified on a single tab component, on a
	 * {@link JTabbedPane} itself (will hold for all tab components that don't
	 * define this property) or on {@link UIManager}. The value should be
	 * either {@link Boolean#TRUE} or {@link Boolean#FALSE}. By default, the
	 * vertically oriented tabs will have the rotated icons. In case the icon
	 * internal layout has horizontal / vertical meaning, using this property
	 * with value {@link Boolean#TRUE} will preserve the icon orientation. Note
	 * that this setting is only relevant for tabs marked with
	 * {@link #TABBED_PANE_VERTICAL_ORIENTATION} property.
	 * 
	 * <p>
	 * Example of setting that all vertically-oriented tabs in the application
	 * will have unrotated icons:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * <p>
	 * An example of tabbed pane with two different kinds of icon orientation:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);<br>
	 * JTabbedPane jtp = new JTabbedPane();<br>
	 * jtp.setTabPlacement(SwingConstants.LEFT);<br>
	 * JPanel tab1 = new JPanel();<br>
	 * JPanel tab2 = new JPanel();<br>
	 * tab2.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS, <br>
	 * &nbsp;&nbsp;Boolean.FALSE);<br>
	 * jtp.addTab(tab1);<br>
	 * jtp.addTab(tab2);
	 * </code>
	 * 
	 * <p>
	 * In the example above, the first tab (<b>tab1</b>) has the
	 * horizontally-oriented icon (the same as with {@link SwingConstants#TOP}
	 * or {@link SwingConstants#BOTTOM}) , while the second tab (<b>tab2</b>)
	 * has the rotated icon (oriented along the tab text).
	 * </p>
	 * 
	 * @since version 2.2
	 * @see #TABBED_PANE_VERTICAL_ORIENTATION
	 */
	public final static String TABBED_PANE_VERTICAL_ORIENTATION_ROTATE_ICONS = "substancelaf.tabbedpaneverticalOrientationRotateIcons";

	/**
	 * Client property name for specifying that only the close button of a
	 * marked-as-modified tab component should pulsate. This property can be
	 * specified on a single tab component, on a {@link JTabbedPane} itself
	 * (will hold for all tab components that don't define this property) or on
	 * {@link UIManager}. The value should be either {@link Boolean#TRUE} or
	 * {@link Boolean#FALSE}. By default, the animation on modified tabs is on
	 * the entire tab rectangle. Note that this setting is only relevant for
	 * tabs marked with {@link #WINDOW_MODIFIED} property.
	 * 
	 * <p>
	 * Example of setting that all tabs in the application will have modified
	 * animation on close button:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * <p>
	 * A more complex example:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);<br>
	 * JTabbedPane jtpMain = new JTabbedPane();<br>
	 * JTabbedPane jtpSecondary = new JTabbedPane();<br>
	 * jtpSecondary.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION, <br>
	 * &nbsp;&nbsp;Boolean.FALSE);<br>
	 * JPanel panelSecondary = new JPanel();<br>
	 * jtpMain.addTab(jtpSecondary);<br>
	 * jtpMain.addTab(panelSecondary);<br>
	 * JPanel tab1 = new JPanel();<br>
	 * JPanel tab2 = new JPanel();<br>
	 * tab2.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);<br>
	 * jtpSecondary.addTab(tab1);<br>
	 * jtpSecondary.addTab(tab2);
	 * </code>
	 * 
	 * <p>
	 * In the example above, the first first-level child (<b>jtpSecondary</b>)
	 * has the animation on the entire tab (since it overrides the global
	 * setting). The second first-level child tab (<b>panelSecondary</b>) has
	 * animation on the close button (from the global <b>UIManager</b>
	 * setting). The first second-level tab has the animation on the entire tab
	 * (setting inherited from the parent <b>jtpSecondary</b> tab that
	 * overrides the global setting). The second second-level tab has animation
	 * on the close button (since its setting overrides the parent setting).
	 * </p>
	 * 
	 * @since version 2.2
	 * @see #TABBED_PANE_CLOSE_BUTTONS_PROPERTY
	 * @see #TABBED_PANE_CLOSE_CALLBACK
	 */
	public final static String TABBED_PANE_CLOSE_BUTTONS_MODIFIED_ANIMATION = "substancelaf.tabbedpaneclosebuttonsmodifiedanimation";

	/**
	 * Client property name for specifying the callback for deciding on the tab
	 * close type. This property can be specified on a single tab component, on
	 * a {@link JTabbedPane} itself (will hold for all tab components that don't
	 * define this property) or on {@link UIManager}. The value should be an
	 * instance of {@link TabCloseCallback}. Note that this setting is only
	 * relevant for tabs marked with {@link #TABBED_PANE_CLOSE_BUTTONS_PROPERTY}
	 * property.
	 * 
	 * <p>
	 * Example of custom tab close callback set on a tabbed pane:
	 * </p>
	 * <code>
	 * TabCloseCallback closeCallback = new TabCloseCallback() {<br>
	 * &nbsp;&nbsp;public TabCloseKind onAreaClick(JTabbedPane tabbedPane,<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;int tabIndex, MouseEvent mouseEvent) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;if (mouseEvent.getButton() != MouseEvent.BUTTON3)<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;return TabCloseKind.NONE;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;if (mouseEvent.isShiftDown()) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;return TabCloseKind.ALL;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;return TabCloseKind.THIS;<br>
	 * &nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;public TabCloseKind onCloseButtonClick(JTabbedPane tabbedPane,<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;int tabIndex, MouseEvent mouseEvent) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;if (mouseEvent.isAltDown()) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;return TabCloseKind.ALL_BUT_THIS;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;if (mouseEvent.isShiftDown()) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;return TabCloseKind.ALL;<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;return TabCloseKind.THIS;<br>
	 * &nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;public String getAreaTooltip(JTabbedPane tabbedPane, int tabIndex) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;return null;<br>
	 * &nbsp;&nbsp;}<br>
	 * <br>
	 * &nbsp;&nbsp;public String getCloseButtonTooltip(JTabbedPane tabbedPane,<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;int tabIndex) {<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;StringBuffer result = new StringBuffer();<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;result.append("&lt;html&gt;&lt;body&gt;");<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;result.append("Mouse click closes &lt;b&gt;"<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;+ tabbedPane.getTitleAt(tabIndex) + "&lt;/b&gt; tab");<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;result.append("&lt;br&gt;&lt;b&gt;Alt&lt;/b&gt;-Mouse click closes all tabs but &lt;b&gt;"<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;+ tabbedPane.getTitleAt(tabIndex) + "&lt;/b&gt; tab");<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;result.append("&lt;br&gt;&lt;b&gt;Shift&lt;/b&gt;-Mouse click closes all tabs");<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;result.append("&lt;/body&gt;&lt;/html&gt;");<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;return result.toString();<br>
	 * &nbsp;&nbsp;}<br>
	 * };<br>
	 *
	 * JTabbedPane jtp = new JTabbedPane();<br>
	 * jtp.putClientProperty(<br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;SubstanceLookAndFeel.TABBED_PANE_CLOSE_CALLBACK, <br>
	 * &nbsp;&nbsp;&nbsp;&nbsp;closeCallback);
	 * </code>
	 * 
	 * @since version 2.3
	 */
	public final static String TABBED_PANE_CLOSE_CALLBACK = "substancelaf.tabbedpanecloseCallback";

	/**
	 * Client property name for specifying the text alignment kind for on
	 * {@link SwingConstants#LEFT} and {@link SwingConstants#RIGHT} tab
	 * placements. This property can be specified either on a single
	 * {@link JTabbedPane} or on {@link UIManager}. The value should be one of
	 * {@link SubstanceConstants.TabTextAlignmentKind} enum. By default, the
	 * text alignment is {@link SubstanceConstants.TabTextAlignmentKind#DEFAULT}
	 * (centered).
	 * 
	 * <p>
	 * Example of setting that all tabbed panes in the application will follow
	 * placement for the text alignment:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_TEXT_ALIGNMENT_KIND, <br>
	 * &nbsp;&nbsp;TabTextAlignmentKind.FOLLOW_PLACEMENT);
	 * </code>
	 * 
	 * <p>
	 * Example of specifying that the specific tabbed pane will follow placement
	 * for the text alignment:
	 * </p>
	 * <code>
	 * JTabbedPane jtpMain = new JTabbedPane();<br>
	 * jtpMain.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_TEXT_ALIGNMENT_KIND, <br>
	 * &nbsp;&nbsp;TabTextAlignmentKind.FOLLOW_PLACEMENT);
	 * </code>
	 * 
	 * @since version 2.3
	 */
	public final static String TABBED_PANE_TEXT_ALIGNMENT_KIND = "substancelaf.tabbedpanetextAlignmentKind";

	/**
	 * Client property name for specifying the content pane border kind. This
	 * property can be specified either on a single {@link JTabbedPane} or on
	 * {@link UIManager}. The value should be one of
	 * {@link SubstanceConstants.TabContentPaneBorderKind} enum. By default, the
	 * border kind is
	 * {@link SubstanceConstants.TabContentPaneBorderKind#DOUBLE_FULL}.
	 * 
	 * <p>
	 * Example of setting that all tabbed panes in the application have single
	 * full border (default setting prior to version 4.1):
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.TABBED_PANE_CONTENT_BORDER_KIND, <br>
	 * &nbsp;&nbsp;TabContentPaneBorderKind.SINGLE_FULL);
	 * </code>
	 * 
	 * <p>
	 * Example of specifying that the specific tabbed pane has single full
	 * border (default setting prior to version 4.1):
	 * </p>
	 * <code>
	 * JTabbedPane jtpMain = new JTabbedPane();<br>
	 * jtpMain.putClientProperty(SubstanceLookAndFeel.TABBED_PANE_CONTENT_BORDER_KIND, <br>
	 * &nbsp;&nbsp;TabContentPaneBorderKind.SINGLE_FULL);
	 * </code>
	 * 
	 * @since version 4.1
	 */
	public final static String TABBED_PANE_CONTENT_BORDER_KIND = "substancelaf.tabbedPaneContentBorderKind";

	/**
	 * Client property name for specifying combo popup flyout orientation. This
	 * property can be set on either a specific {@link JComboBox} or globally on
	 * {@link UIManager}. The value should be one of the {@link Integer}s
	 * below:
	 * 
	 * <p>
	 * <ul>
	 * <li>The default {@link SwingConstants#SOUTH} - the popup is displayed
	 * directly below the combo aligned to the left.
	 * <li>{@link SwingConstants#NORTH} - the popup is displayed directly above
	 * the combo aligned to the left.
	 * <li>{@link SwingConstants#EAST} - the popup is displayed to the left of
	 * the combo aligned to the top.
	 * <li>{@link SwingConstants#WEST} - the popup is displayed to the right of
	 * the combo aligned to the top.
	 * <li>{@link SwingConstants#CENTER} - the popup is displayed centered
	 * vertically over the combo aligned to the left.
	 * </ul>
	 * </p>
	 * 
	 * <p>
	 * Note that the combo arrow changes in accordance with the combo popup
	 * flyout orientation. Example of setting a combobox with a custom flyout
	 * orientation:
	 * </p>
	 * <code>
	 * JComboBox cb = new JComboBox(<br>
	 * &nbsp;&nbsp;new Object[] { "Ester", "Jordi", "Jordina", "Jorge", "Sergi" });<br>
	 * cb.putClientProperty(SubstanceLookAndFeel.COMBO_BOX_POPUP_FLYOUT_ORIENTATION, <br>
	 * &nbsp;&nbsp;SwingConstants.CENTER);
	 * </code>
	 * 
	 * @since version 2.3
	 * @see #COMBO_POPUP_PROTOTYPE
	 */
	public final static String COMBO_BOX_POPUP_FLYOUT_ORIENTATION = "substancelaf.comboboxpopupFlyoutOrientation";

	/**
	 * Client property name for specifying scroll pane button policy. This
	 * property can be set on either a specific {@link JScrollPane} or globally
	 * on {@link UIManager}. The value should be one of the
	 * {@link SubstanceConstants.ScrollPaneButtonPolicyKind} enum. Example of
	 * setting a scroll pane with a custom button policy:
	 * 
	 * <p>
	 * <code>
	 * JScrollPane jsp = new JScrollPane(new JPanel());<br>
	 * jsp.putClientProperty(SubstanceLookAndFeel.SCROLL_PANE_BUTTONS_POLICY,<br>
	 * &nbsp;&nbsp;ScrollPaneButtonPolicyKind.MULTIPLE);
	 * </code>
	 * 
	 * @since version 3.1
	 */
	public final static String SCROLL_PANE_BUTTONS_POLICY = "substancelaf.scrollPaneButtonsPolicy";

	/**
	 * Property name for specifying that no extra UI elements (such as menu
	 * items in system menu or menu search panel) should be shown. This property
	 * can be set as either VM flag (no value needed) or as a global setting on
	 * {@link UIManager}. In the latter case, the value should be either
	 * {@link Boolean#TRUE} or {@link Boolean#FALSE}. Partial list of UI
	 * elements that will not be present when this property is specified:
	 * 
	 * <p>
	 * <ul>
	 * <li>System menu items
	 * <li>Menu search panel
	 * <li>Lock icon on non-editable text components
	 * </ul>
	 * </p>
	 * 
	 * <p>
	 * Example of setting this property via VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.noExtraElements
	 * </code>
	 * 
	 * <p>
	 * Example of setting this property on {@link UIManager}:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.NO_EXTRA_ELEMENTS, Boolean.TRUE);
	 * SwingUtilities.updateComponentTree(myFrame);
	 * </code>
	 * 
	 * @see #toShowExtraElements
	 * @since version 2.1
	 */
	public final static String NO_EXTRA_ELEMENTS = "substancelaf.noExtraElements";

	/**
	 * Property name for specifying menu gutter fill kind. Menu gutter is the
	 * part of the menu where checkmarks and icons are painted. The value should
	 * be one of {@link MenuGutterFillKind} enum. This property can be set
	 * globally on the {@link UIManager}. The default value is
	 * {@link MenuGutterFillKind#HARD}.
	 * 
	 * <p>
	 * Example of setting soft fill kind:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.MENU_GUTTER_FILL_KIND, MenuGutterFillKind.SOFT);
	 * </code>
	 * 
	 * @since version 3.2
	 */
	public final static String MENU_GUTTER_FILL_KIND = "substancelaf.menuGutterFillKind";

	/**
	 * Client property name for specifying the kind of focus indication on
	 * buttons, check boxes and radio buttons. The value should be one of
	 * {@link SubstanceConstants.FocusKind} enum. This property can be set
	 * either as client property on some component or as global property on
	 * {@link UIManager}.
	 * 
	 * <p>
	 * In order to compute the kind of focus indication for some component, the
	 * component's hierarchy is traversed bottom up. The first component that
	 * has this property set, defines the focus indication kind. Finally, if
	 * neither component not its ancestors define this property, the global
	 * setting on {@link UIManager} is checked. If there is no global setting,
	 * the default {@link SubstanceConstants.FocusKind#ALL_INNER} is used. Here
	 * is an example to illustrate the above:
	 * </p>
	 * 
	 * <code>
	 * &nbsp;&nbsp;JPanel topPanel = new JPanel();<br>
	 * &nbsp;&nbsp;topPanel.putClientProperty(SubstanceLookAndFeel.FOCUS_KIND, FocusKind.UNDERLINE);<br>
	 * &nbsp;&nbsp;JPanel panel1 = new JPanel();<br>
	 * &nbsp;&nbsp;JButton b1 = new JButton("button1");<br>
	 * &nbsp;&nbsp;b1.putClientProperty(SubstanceLookAndFeel.FOCUS_KIND, FocusKind.TEXT);<br>
	 * &nbsp;&nbsp;JButton b2 = new JButton("button2");<br>
	 * &nbsp;&nbsp;JButton b3 = new JButton("button3");<br>
	 * &nbsp;&nbsp;b3.putClientProperty(SubstanceLookAndFeel.FOCUS_KIND, FocusKind.ALL_INNER);<br>
	 * &nbsp;&nbsp;panel1.add(b1);<br>
	 * &nbsp;&nbsp;panel1.add(b2);<br>
	 * &nbsp;&nbsp;topPanel.add(panel1);<br>
	 * &nbsp;&nbsp;topPanel.add(b3);<br>
	 * </code>
	 * 
	 * <p>
	 * In the code above:
	 * </p>
	 * <ul>
	 * <li>Button <b>b1</b> will have
	 * {@link SubstanceConstants.FocusKind#NONE} focus kind which is set
	 * directly on the button.
	 * <li>Button <b>b2</b> will have
	 * {@link SubstanceConstants.FocusKind#UNDERLINE} focus kind which is
	 * inherited from its <b>topPanel</b> parent.
	 * <li>Button <b>b3</b> will have
	 * {@link SubstanceConstants.FocusKind#ALL_INNER} focus kind which is set
	 * directly on the button.
	 * </ul>
	 * 
	 * @since 2.2
	 * @see SubstanceConstants.FocusKind
	 */
	public final static String FOCUS_KIND = "substancelaf.focusKind";

	/**
	 * Property name for specifying the combobox popup prototype display value
	 * which is used to compute the width of the popup at runtime. The property
	 * value should be one of:
	 * 
	 * <p>
	 * <ul>
	 * <li>{@link ComboPopupPrototypeCallback} - will provide
	 * application-specific logic at runtime.
	 * <li>{@link Object} - will point to the prototype entry itself.
	 * </ul>
	 * </p>
	 * 
	 * <p>
	 * This property can be set either on a specific {@link JComboBox} or
	 * globally on {@link UIManager}.
	 * </p>
	 * 
	 * <p>
	 * Here is an example of combo popup prototype set to a model element:
	 * </p>
	 * <code>
	 * 	JComboBox comboProto1 = new JComboBox(new Object[] { "aa", "aaaaa",<br>
	 *	&nbsp;&nbsp;"aaaaaaaaaa", "this one is the one", "aaaaaaaaaaaaaaaaaaaaa" });<br>
	 *	comboProto1.setPrototypeDisplayValue("aaaaa");<br>
	 *	comboProto1.putClientProperty(SubstanceLookAndFeel.COMBO_POPUP_PROTOTYPE,<br>
	 *	&nbsp;&nbsp;"this one is the one");
	 * </code>
	 * 
	 * <p>
	 * Here is an example of combo popup prototype set to a dynamic callback.
	 * This callback always returns the last model element:
	 * </p>
	 * <code>
	 *  JComboBox comboProto3 = new JComboBox(new Object[] { "aa", "aaaaa",<br>
	 *	&nbsp;&nbsp;"this is not", "this one is not it",<br>
	 *	&nbsp;&nbsp;"this one is it that is for the popup" });<br>
	 *	comboProto3.setPrototypeDisplayValue("aaaaa");<br>
	 *	comboProto3.putClientProperty(SubstanceLookAndFeel.COMBO_POPUP_PROTOTYPE,<br>
	 *	&nbsp;&nbsp;new ComboPopupPrototypeCallback() {<br>
	 *	&nbsp;&nbsp;&nbsp;&nbsp;public Object getPopupPrototypeDisplayValue(JComboBox jc) {<br>
	 *	&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;return jc.getModel().getElementAt(<br>
	 *	&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;jc.getModel().getSize() - 1);<br>
	 *	&nbsp;&nbsp;&nbsp;&nbsp;}<br>
	 *	&nbsp;&nbsp;});
	 *  </code>
	 * 
	 * @since version 3.0
	 * @see #COMBO_BOX_POPUP_FLYOUT_ORIENTATION
	 */
	public final static String COMBO_POPUP_PROTOTYPE = "substancelaf.comboPopupPrototype";

	/**
	 * VM property name for specifying the trace file. The trace file will
	 * contain output of the memory analyser which can be used to pinpoint the
	 * memory leaks. The property value is used as a filename for tracing the
	 * memory allocations. Example for specifying the trace file name:
	 * 
	 * <p>
	 * <code>
	 * -Dsubstancelaf.traceFile=C:/temp/myApp.substance.log
	 * </code>
	 * </p>
	 * 
	 * @see MemoryAnalyzer
	 * @since version 2.0
	 */
	public final static String TRACE_FILE = "substancelaf.traceFile";

	/**
	 * VM property name for the configuration file. If this property is
	 * specified in the VM flags, all other global VM Substance-specific
	 * settings are ignored and are read instead from the (local) configuration
	 * file. The property value should point to a file in the regular
	 * {@link Properties} format. Example of specifying configuration file:
	 * 
	 * <p>
	 * <code>
	 * -Dsubstancelaf.configFile=C:/myConfigSubstance.properties
	 * </code>
	 * </p>
	 * 
	 * @since version 2.2
	 */
	public final static String CONFIG_FILE = "substancelaf.configFile";

	/**
	 * Property name for enabling inverted themes. This property can be set as
	 * either VM flag (no value needed) or as a global setting on
	 * {@link UIManager}. In the latter case, the value should be either
	 * {@link Boolean#TRUE} or {@link Boolean#FALSE}. When this property is
	 * set, the {@link #getAllThemes()} method will return entries for all
	 * {@link SubstanceTheme.ThemeKind#BRIGHT} and
	 * {@link SubstanceTheme.ThemeKind#COLD} themes. All these entries will be
	 * marked with {@link SubstanceTheme.ThemeKind#INVERTED} kind.
	 * 
	 * <p>
	 * Example of setting this property via VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.enableInvertedThemes
	 * </code>
	 * 
	 * <p>
	 * Example of setting this property on {@link UIManager}:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.ENABLE_INVERTED_THEMES, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 2.3
	 * @see #ENABLE_NEGATED_THEMES
	 */
	public final static String ENABLE_INVERTED_THEMES = "substancelaf.enableInvertedThemes";

	/**
	 * Property name for enabling negated themes. This property can be set as
	 * either VM flag (no value needed) or as a global setting on
	 * {@link UIManager}. In the latter case, the value should be either
	 * {@link Boolean#TRUE} or {@link Boolean#FALSE}. When this property is
	 * set, the {@link #getAllThemes()} method will return entries for all
	 * {@link SubstanceTheme.ThemeKind#BRIGHT} and
	 * {@link SubstanceTheme.ThemeKind#COLD} themes. All these entries will be
	 * marked with {@link SubstanceTheme.ThemeKind#NEGATED} kind.
	 * 
	 * <p>
	 * Example of setting this property via VM flag:
	 * </p>
	 * <code>
	 * -Dsubstancelaf.enableNegatedThemes
	 * </code>
	 * 
	 * <p>
	 * Example of setting this property on {@link UIManager}:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.ENABLE_NEGATED_THEMES, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 3.1
	 * @see #ENABLE_INVERTED_THEMES
	 */
	public final static String ENABLE_NEGATED_THEMES = "substancelaf.enableNegatedThemes";

	/**
	 * Client property name for specifying the number of echo characters for
	 * each password character. The value should be an instance of
	 * {@link Integer}, otherwise will be ignored. This property can be set
	 * either on a specific {@link JPasswordField} or globally on
	 * {@link UIManager}.
	 * 
	 * <p>
	 * Example of having all password fields echo 3 characters per each typed
	 * user character:
	 * </p>
	 * <code>
	 * UIManager.put(SubstanceLookAndFeel.PASSWORD_ECHO_PER_CHAR, <br>
	 * &nbsp;&nbsp;new Integer(3));
	 * </code>
	 * 
	 * <p>
	 * Example of having a specific password field echo 2 characters per each
	 * typed user character:
	 * </p>
	 * <code>
	 * JPasswordField jpf = new JPasswordField();<br>
	 * jpf.putClientProperty(SubstanceLookAndFeel.PASSWORD_ECHO_PER_CHAR, <br>
	 * &nbsp;&nbsp;new Integer(2));
	 * </code>
	 * 
	 * @since version 2.2
	 */
	public final static String PASSWORD_ECHO_PER_CHAR = "substancelaf.passwordEchoPerChar";

	// /**
	// * Client property name for specifying that the toolbar buttons should be
	// * flat (no background when non-selected and non-rollover). The value
	// should
	// * be an instance of {@link Boolean}. The default value is
	// * <code>true</code>. This property can be specified on a single button,
	// * on a single {@link JToolBar} or globally on {@link UIManager}.
	// *
	// * <p>
	// * Example of specifying that a specific toolbar button should not appear
	// * flat:
	// * </p>
	// * <code>
	// * JToolBar tb = new JToolBar("Toolbar");<br>
	// * JButton b = new JButton("Button");<br>
	// * b.putClientProperty(SubstanceLookAndFeel.TOOLBAR_BUTTON_FLAT, <br>
	// * &nbsp;&nbsp;Boolean.FALSE);<br>
	// * tb.add(b);
	// * </code>
	// *
	// * <p>
	// * Example of specifying that all toolbar buttons of the specific toolbar
	// * (except those that specify their own property) should not appear flat:
	// * </p>
	// * <code>
	// * JToolBar tb = new JToolBar("Toolbar");<br>
	// * tb.putClientProperty(SubstanceLookAndFeel.TOOLBAR_BUTTON_FLAT, <br>
	// * &nbsp;&nbsp;Boolean.FALSE);<br>
	// * </code>
	// *
	// * @since version 2.3
	// * @see #BUTTON_PAINT_NEVER_PROPERTY
	// * @see #FLAT_PROPERTY
	// */
	// public final static String TOOLBAR_BUTTON_FLAT =
	// "substancelaf.toolbarButtonFlat";

	/**
	 * <p>
	 * Client property name for specifying that icons on buttons and toggle
	 * buttons should match the color of the current theme when they are in
	 * default state. The button is in default state when it's not pressed, not
	 * selected, not armed and not rolled over. The value should be an instance
	 * of {@link Boolean}. By default, all buttons show regular (full-color
	 * original) icons. The value can be set on the specific button or globally
	 * on {@link UIManager}.
	 * </p>
	 * 
	 * <p>
	 * Example of marking a button to have theme-matching icon in default state:
	 * </p>
	 * <code>
	 * JButton jb = new JButton("sample", myIcon);<br>
	 * jb.putClientProperty(SubstanceLookAndFeel.USE_THEMED_DEFAULT_ICONS, <br>
	 * &nbsp;&nbsp;Boolean.TRUE);
	 * </code>
	 * 
	 * @since version 3.3
	 */
	public final static String USE_THEMED_DEFAULT_ICONS = "substancelaf.useThemedDefaultIcons";

	/**
	 * <p>
	 * Client property name for specifying that the colorization amount applied
	 * to the background and foreground of the current theme and the specific
	 * control. By default, when the application does not use any custom colors,
	 * all the controls are painted with the colors of the current theme / skin.
	 * The colors coming from the look-and-feel implement the marker
	 * {@link UIResource} interface which allows the UI delegates to
	 * differentiate between application-specific colors which are not changed,
	 * and the LAF-provide colors that are changed on LAF switch.
	 * </p>
	 * 
	 * <p>
	 * Up until version 4.2, the support for custom background and foreground
	 * colors in Substance was not very consistent. On some controls, such as
	 * buttons, the background color was ignored completely (in favour of the
	 * current theme), while on others it was used to color the entire bounds of
	 * the control (such as check box). This new client property installs the
	 * "smart colorization" mode which uses the colors of the current theme
	 * <b>and</b> the custom background and foreground colors (where installed
	 * by application) to colorize the relevant portions of the control. For
	 * example, on checkbox the custom background color will be used to colorize
	 * the check box itself, while the custom foreground color will be applied
	 * to the check box text and the check mark.
	 * </p>
	 * 
	 * <p>
	 * The value of this property specifies the actual colorization amount.
	 * Value of 0.0 results in no colorization at all. Values closer to 1.0
	 * result in controls being colorized in almost full amount of the custom
	 * colors. Note that in order to maintain the gradients (fill, border, etc),
	 * even value of 1.0 does not result in full custom color being applied to
	 * the relevant visuals of the control.
	 * </p>
	 * 
	 * <p>
	 * This property can be specified globally on {@link UIManager}, applying
	 * on all controls, or on the specific component / container. In the later
	 * case, the value will be applied to the component / container itself and
	 * all its children that do not specify a custom value for this property.
	 * </p>
	 * 
	 * <p>
	 * The value should be an instance of {@link Double} in 0.0-1.0 range.
	 * </p>
	 * 
	 * <p>
	 * Example of marking a button to have custom background color and
	 * colorizing it with 50%:
	 * </p>
	 * <code>
	 * JButton jb = new JButton("sample", myIcon);<br>
	 * jb.setBackground(Color.red);<br>
	 * jb.putClientProperty(SubstanceLookAndFeel.COLORIZATION_FACTOR, <br>
	 * &nbsp;&nbsp;new Double(0.5));
	 * </code>
	 * 
	 * @since version 4.2
	 * @see Component#setBackground(Color)
	 * @see Component#setForeground(Color)
	 */
	public final static String COLORIZATION_FACTOR = "substancelaf.colorizationFactor";

	/**
	 * Internal client property name for storing application-specific font
	 * policy.
	 * 
	 * @since version 3.3
	 * @see #setFontPolicy(FontPolicy)
	 * @see #getFontPolicy()
	 */
	protected final static String SUBSTANCE_FONT_POLICY_KEY = "substancelaf.fontPolicyKey";

	/**
	 * The parameter reader.
	 */
	protected static ParamReader paramReader;

	/**
	 * Resource bundle for <b>Substance</b> labels.
	 */
	private static ResourceBundle LABEL_BUNDLE = null;

	/**
	 * Class loader for {@link #LABEL_BUNDLE}.
	 */
	private static ClassLoader labelBundleClassLoader;

	/**
	 * Retrieves the current label bundle.
	 * 
	 * @return The current label bundle.
	 * @see #resetLabelBundle()
	 */
	public static synchronized ResourceBundle getLabelBundle() {
		if (SubstanceLookAndFeel.LABEL_BUNDLE == null) {
			// fix for RFE 157 (allowing custom class loader for
			// resource bundles which can remove server calls
			// in applets)
			if (SubstanceLookAndFeel.labelBundleClassLoader == null) {
				SubstanceLookAndFeel.LABEL_BUNDLE = ResourceBundle.getBundle(
						"org.jvnet.substance.resources.Labels", Locale
								.getDefault());
			} else {
				SubstanceLookAndFeel.LABEL_BUNDLE = ResourceBundle.getBundle(
						"org.jvnet.substance.resources.Labels", Locale
								.getDefault(),
						SubstanceLookAndFeel.labelBundleClassLoader);
			}
			for (LocaleChangeListener lcl : SubstanceLookAndFeel.localeChangeListeners)
				lcl.localeChanged();
		}
		return SubstanceLookAndFeel.LABEL_BUNDLE;
	}

	/**
	 * Retrieves the label bundle for the specified locale.
	 * 
	 * @param locale
	 *            Locale.
	 * @return The label bundle for the specified locale.
	 */
	public static synchronized ResourceBundle getLabelBundle(Locale locale) {
		// fix for RFE 157 (allowing custom class loader for
		// resource bundles which can remove server calls
		// in applets)
		if (SubstanceLookAndFeel.labelBundleClassLoader == null) {
			return ResourceBundle.getBundle(
					"org.jvnet.substance.resources.Labels", locale);
		} else {
			return ResourceBundle.getBundle(
					"org.jvnet.substance.resources.Labels", locale,
					SubstanceLookAndFeel.labelBundleClassLoader);
		}
	}

	/**
	 * Resets the current label bundle. Useful when the application changes
	 * Locale at runtime.
	 * 
	 * @see #getLabelBundle()
	 */
	public static synchronized void resetLabelBundle() {
		SubstanceLookAndFeel.LABEL_BUNDLE = null;
		LafWidgetRepository.resetLabelBundle();
	}

	/**
	 * Creates new instance of <b>Substance</b> look and feel.
	 */
	public SubstanceLookAndFeel() {
		SubstanceLookAndFeel.componentPlugins = new ComponentPluginManager(
				SubstanceLookAndFeel.PLUGIN_XML);

		SubstanceLookAndFeel.themePlugins = new PluginManager(
				SubstanceLookAndFeel.PLUGIN_XML, LafPlugin.TAG_MAIN,
				SubstanceThemePlugin.TAG_THEME_PLUGIN_CLASS);

		SubstanceLookAndFeel.skinPlugins = new PluginManager(
				SubstanceLookAndFeel.PLUGIN_XML, LafPlugin.TAG_MAIN,
				SubstanceSkinPlugin.TAG_SKIN_PLUGIN_CLASS);

		SubstanceLookAndFeel.watermarkPlugins = new PluginManager(
				SubstanceLookAndFeel.PLUGIN_XML, LafPlugin.TAG_MAIN,
				SubstanceWatermarkPlugin.TAG_WATERMARK_PLUGIN_CLASS);

		SubstanceLookAndFeel.shaperPlugins = new PluginManager(
				SubstanceLookAndFeel.PLUGIN_XML, LafPlugin.TAG_MAIN,
				SubstanceButtonShaperPlugin.TAG_BUTTON_SHAPER_PLUGIN_CLASS);

		SubstanceLookAndFeel.painterPlugins = new PluginManager(
				SubstanceLookAndFeel.PLUGIN_XML,
				LafPlugin.TAG_MAIN,
				SubstanceGradientPainterPlugin.TAG_GRADIENT_PAINTER_PLUGIN_CLASS);

		SubstanceLookAndFeel.borderPainterPlugins = new PluginManager(
				SubstanceLookAndFeel.PLUGIN_XML, LafPlugin.TAG_MAIN,
				SubstanceBorderPainterPlugin.TAG_BORDER_PAINTER_PLUGIN_CLASS);

		// Bootstrap reader - for reading (possible) configuration file location
		JvmFlagsParamReader bootstrapReader = new JvmFlagsParamReader();

		String configFileName = bootstrapReader.getConfigFileName();

		// Choose actual reader (continue working with bootstrap VM reader
		// or switch to file-based reader)
		SubstanceLookAndFeel.paramReader = bootstrapReader;
		if (configFileName != null)
			SubstanceLookAndFeel.paramReader = new PropertiesFileParamReader(
					configFileName);
	}

	/**
	 * The current theme.
	 */
	private static SubstanceTheme currentTheme = null;

	/**
	 * Returns the current theme.
	 * 
	 * @return The current theme.
	 * @see #getCurrentThemeName()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static SubstanceTheme getTheme() {
		if (currentTheme == null) {
			return new SubstanceAquaTheme();
		}
		return currentTheme;
	}

	/**
	 * Returns the current color scheme for components in active visual state.
	 * 
	 * @return Current color scheme for components in active visual state.
	 * @see #getDefaultColorScheme()
	 * @see #getDisabledColorScheme()
	 */
	public static ColorScheme getActiveColorScheme() {
		return SubstanceThemeUtilities.getTheme(null, ComponentState.ACTIVE)
				.getColorScheme();
	}

	/**
	 * Returns the current color scheme for components in default visual state.
	 * 
	 * @return Current color scheme for components in default visual state.
	 * @see #getActiveColorScheme()
	 * @see #getDisabledColorScheme()
	 */
	public static ColorScheme getDefaultColorScheme() {
		return SubstanceThemeUtilities.getTheme(null, ComponentState.DEFAULT)
				.getColorScheme();
	}

	/**
	 * Returns the current color scheme for components in disabled visual state.
	 * 
	 * @return Current color scheme for components in disabled visual state.
	 * @see #getActiveColorScheme()
	 * @see #getDefaultColorScheme()
	 */
	public static ColorScheme getDisabledColorScheme() {
		return SubstanceThemeUtilities.getTheme(null,
				ComponentState.DISABLED_UNSELECTED).getColorScheme();
	}

	/**
	 * Sets new theme. In order for the theme to be correctly set, you need to
	 * call {@link SwingUtilities#updateComponentTreeUI(java.awt.Component)} on
	 * all open frames (call {@link Frame#getFrames()} to get a list of all open
	 * frames).
	 * 
	 * @param themeClassName
	 *            Theme class name (full qualified).
	 * @return The status of the theme change.
	 * @see #getCurrentThemeName()
	 * @see #getTheme()
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static boolean setCurrentTheme(String themeClassName) {
		return setCurrentTheme(themeClassName, true);
	}

	/**
	 * Sets new theme. In order for the theme to be correctly set, you need to
	 * call {@link SwingUtilities#updateComponentTreeUI(java.awt.Component)} on
	 * all open frames (call {@link Frame#getFrames()} to get a list of all open
	 * frames).
	 * 
	 * @param themeClassName
	 *            Theme class name (full qualified).
	 * @param toUpdateUiTable
	 *            Indicates whether the UI table needs to be updated.
	 * @return The status of the theme change.
	 */
	private static boolean setCurrentTheme(String themeClassName,
			boolean toUpdateUiTable) {
		try {
			Class<?> themeClass = Class.forName(themeClassName);
			if (themeClass == null) {
				return false;
			}
			Object obj = themeClass.newInstance();
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceTheme)) {
				return false;
			}
			return SubstanceLookAndFeel.setCurrentTheme((SubstanceTheme) obj,
					toUpdateUiTable);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new theme. In order for the theme to be correctly set, you need to
	 * call {@link SwingUtilities#updateComponentTreeUI(java.awt.Component)} on
	 * all open frames (call {@link Frame#getFrames()} to get a list of all open
	 * frames).
	 * 
	 * @param themeInfo
	 *            Theme information.
	 * @return The status of the theme change.
	 * @see #getCurrentThemeName()
	 * @see #getTheme()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static boolean setCurrentTheme(ThemeInfo themeInfo) {
		return setCurrentTheme(themeInfo, true);
	}

	/**
	 * Sets new theme. In order for the theme to be correctly set, you need to
	 * call {@link SwingUtilities#updateComponentTreeUI(java.awt.Component)} on
	 * all open frames (call {@link Frame#getFrames()} to get a list of all open
	 * frames).
	 * 
	 * @param themeInfo
	 *            Theme information.
	 * @param toUpdateUiTable
	 *            Indicates whether the UI table needs to be updated.
	 * @return The status of the theme change.
	 */
	private static boolean setCurrentTheme(ThemeInfo themeInfo,
			boolean toUpdateUiTable) {
		try {
			SubstanceTheme theme = SubstanceTheme.createInstance(themeInfo);
			if (theme == null)
				return false;
			return SubstanceLookAndFeel.setCurrentTheme(theme, toUpdateUiTable);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new theme. In order for the theme to be correctly set, you need to
	 * call {@link SwingUtilities#updateComponentTreeUI(java.awt.Component)} on
	 * all open frames (call {@link Frame#getFrames()} to get a list of all open
	 * frames).
	 * 
	 * @param theme
	 *            Theme object.
	 * @return The status of the theme change.
	 * @see #getCurrentThemeName()
	 * @see #getTheme()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static boolean setCurrentTheme(SubstanceTheme theme) {
		return SubstanceLookAndFeel.setCurrentTheme(theme, true);
	}

	/**
	 * Sets new theme.
	 * 
	 * @param theme
	 *            Theme object.
	 * @param toUpdateUiTable
	 *            Indicates whether the UI table needs to be updated.
	 * @return The status of the theme change.
	 */
	private static boolean setCurrentTheme(SubstanceTheme theme,
			boolean toUpdateUiTable) {
		SubstanceLookAndFeel.currentTheme = theme;
		if (toUpdateUiTable) {
			UIDefaults defaults = UIManager.getLookAndFeelDefaults();
			// The table will be null when the skin is set using a custom
			// LAF
			if (defaults != null) {
				SubstanceLookAndFeel.currentTheme
						.addCustomEntriesToTable(UIManager
								.getLookAndFeelDefaults());
				SubstanceLookAndFeel.componentPlugins
						.processAllDefaultsEntries(UIManager
								.getLookAndFeelDefaults(),
								SubstanceLookAndFeel.currentTheme);
			}
		}

		SubstanceCoreUtilities.resetCaches();

		if (SubstanceLookAndFeel.currentWatermark != null) {
			if (SubstanceLookAndFeel.currentWatermark.isDependingOnTheme())
				SubstanceLookAndFeel.currentWatermark.updateWatermarkImage();
		}

		for (ThemeChangeListener themeChangeListener : SubstanceLookAndFeel.themeChangeListeners)
			themeChangeListener.themeChanged();

		return true;
	}

	/**
	 * Returns the current color theme name.
	 * 
	 * @return Current color theme name.
	 * @see #getTheme()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static String getCurrentThemeName() {
		if (SubstanceLookAndFeel.getTheme() == null)
			return null;
		return SubstanceLookAndFeel.getTheme().getDisplayName();
	}

	/**
	 * Returns all available themes.
	 * 
	 * @return All available color themes. Key - theme name, value - theme
	 *         information.
	 */
	public static Map<String, ThemeInfo> getAllThemes() {
		Map<String, ThemeInfo> result = new TreeMap<String, ThemeInfo>();
		for (Object themePlugin : SubstanceLookAndFeel.themePlugins
				.getAvailablePlugins(true)) {
			for (ThemeInfo themeInfo : ((SubstanceThemePlugin) themePlugin)
					.getThemes()) {
				result.put(themeInfo.getDisplayName(), themeInfo);
				if (SubstanceLookAndFeel.toEnableInvertedThemes()) {
					if ((themeInfo.getThemeKind() == ThemeKind.BRIGHT)
							|| (themeInfo.getThemeKind() == ThemeKind.COLD)) {
						ThemeInfo invertedInfo = new ThemeInfo("Inverted "
								+ themeInfo.getDisplayName(), themeInfo
								.getClassName(), ThemeKind.INVERTED);
						result.put(invertedInfo.getDisplayName(), invertedInfo);
					}
				}
				if (SubstanceLookAndFeel.toEnableNegatedThemes()) {
					if ((themeInfo.getThemeKind() == ThemeKind.BRIGHT)
							|| (themeInfo.getThemeKind() == ThemeKind.COLD)) {
						ThemeInfo negatedInfo = new ThemeInfo("Negated "
								+ themeInfo.getDisplayName(), themeInfo
								.getClassName(), ThemeKind.NEGATED);
						result.put(negatedInfo.getDisplayName(), negatedInfo);
					}
				}
			}
		}
		for (SubstanceMixTheme mixedTheme : SubstanceLookAndFeel.mixedThemes
				.values()) {
			SubstanceTheme[] origThemes = (mixedTheme).getOriginalThemes();
			String[] origThemeClassNames = new String[origThemes.length];
			for (int i = 0; i < origThemes.length; i++)
				origThemeClassNames[i] = origThemes[i].getClass().getName();

			ThemeInfo mixedInfo = new MixedThemeInfo(mixedTheme
					.getDisplayName(), origThemeClassNames);
			result.put(mixedInfo.getDisplayName(), mixedInfo);
		}

		return result;
	}

	/**
	 * The current watermark.
	 */
	private static SubstanceWatermark currentWatermark = null;

	/**
	 * Returns the current watermark name.
	 * 
	 * @return Current watermark name.
	 * @see #getCurrentWatermark()
	 * @see #setCurrentWatermark(String)
	 * @see #setCurrentWatermark(SubstanceWatermark)
	 * @see #registerWatermarkChangeListener(WatermarkChangeListener)
	 * @see #unregisterWatermarkChangeListener(WatermarkChangeListener)
	 * @see #WATERMARK_PROPERTY
	 */
	public static String getCurrentWatermarkName() {
		if (SubstanceLookAndFeel.currentWatermark == null)
			return null;
		return SubstanceLookAndFeel.currentWatermark.getDisplayName();
	}

	/**
	 * Returns all available watermarks.
	 * 
	 * @return All available watermarks. Key - watermark name, value - watermark
	 *         information.
	 */
	public static Map<String, WatermarkInfo> getAllWatermarks() {
		Map<String, WatermarkInfo> result = new TreeMap<String, WatermarkInfo>();
		for (Object watermarkPlugin : SubstanceLookAndFeel.watermarkPlugins
				.getAvailablePlugins(true)) {
			for (WatermarkInfo watermarkInfo : ((SubstanceWatermarkPlugin) watermarkPlugin)
					.getWatermarks()) {
				result.put(watermarkInfo.getDisplayName(), watermarkInfo);
			}
		}
		return result;
	}

	/**
	 * Returns the current watermark.
	 * 
	 * @return The current watermark.
	 * @see #getCurrentWatermarkName()
	 * @see #setCurrentWatermark(String)
	 * @see #setCurrentWatermark(SubstanceWatermark)
	 * @see #registerWatermarkChangeListener(WatermarkChangeListener)
	 * @see #unregisterWatermarkChangeListener(WatermarkChangeListener)
	 * @see #WATERMARK_PROPERTY
	 */
	public static SubstanceWatermark getCurrentWatermark() {
		return SubstanceLookAndFeel.currentWatermark;
	}

	/**
	 * Sets new watermark.
	 * 
	 * @param watermarkClassName
	 *            Watermark class name (full qualified).
	 * @return The status of the watermark change.
	 * @see #getCurrentWatermark()
	 * @see #getCurrentWatermarkName()
	 * @see #setCurrentWatermark(SubstanceWatermark)
	 * @see #registerWatermarkChangeListener(WatermarkChangeListener)
	 * @see #unregisterWatermarkChangeListener(WatermarkChangeListener)
	 * @see #WATERMARK_PROPERTY
	 */
	public static boolean setCurrentWatermark(String watermarkClassName) {
		try {
			Class<?> watermarkClass = Class.forName(watermarkClassName);
			if (watermarkClass == null) {
				return false;
			}
			Object obj = null;
			// see if has constructor that gets a ParamReader class
			Constructor<?> paramCtr = null;

			try {
				paramCtr = watermarkClass
						.getConstructor(new Class[] { ParamReader.class });
			} catch (NoSuchMethodException nsme) {
				// no biggie - doesn't exist
				paramCtr = null;
			}
			if (paramCtr != null) {
				obj = paramCtr.newInstance(new Object[] { null });
			} else {
				obj = watermarkClass.newInstance();
			}
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceWatermark)) {
				return false;
			}
			return SubstanceLookAndFeel
					.setCurrentWatermark((SubstanceWatermark) obj);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new watermark.
	 * 
	 * @param watermarkClassName
	 *            Watermark class name (full qualified).
	 * @param paramReader
	 *            Parameter reader. Is here for the
	 *            {@link SubstanceImageWatermark}.
	 * @return The status of the watermark change.
	 */
	private static boolean setCurrentWatermark(String watermarkClassName,
			ParamReader paramReader) {
		try {
			Class<?> watermarkClass = Class.forName(watermarkClassName);
			if (watermarkClass == null) {
				return false;
			}
			Object obj = null;
			// see if has constructor that gets a ParamReader class
			Constructor<?> paramCtr = null;
			try {
				paramCtr = watermarkClass
						.getConstructor(new Class[] { ParamReader.class });
			} catch (NoSuchMethodException nsme) {
				paramCtr = null;
			}
			if (paramCtr != null) {
				obj = paramCtr.newInstance(new Object[] { paramReader });
			} else {
				obj = watermarkClass.newInstance();
			}
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceWatermark)) {
				return false;
			}
			return SubstanceLookAndFeel
					.setCurrentWatermark((SubstanceWatermark) obj);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new watermark.
	 * 
	 * @param currentWatermark
	 *            Watermark object.
	 * @return The status of the watermark change.
	 * @see #getCurrentWatermark()
	 * @see #getCurrentWatermarkName()
	 * @see #setCurrentWatermark(String)
	 * @see #registerWatermarkChangeListener(WatermarkChangeListener)
	 * @see #unregisterWatermarkChangeListener(WatermarkChangeListener)
	 * @see #WATERMARK_PROPERTY
	 */
	public static boolean setCurrentWatermark(
			SubstanceWatermark currentWatermark) {
		// fix for defect 109 - memory leak on watermark switch
		if (SubstanceLookAndFeel.currentWatermark != null)
			SubstanceLookAndFeel.currentWatermark.dispose();
		SubstanceLookAndFeel.currentWatermark = currentWatermark;
		boolean status = currentWatermark.updateWatermarkImage();
		if (status == true) {
			for (WatermarkChangeListener watermarkChangeListener : SubstanceLookAndFeel.watermarkChangeListeners)
				watermarkChangeListener.watermarkChanged();
		}
		return status;
	}

	/**
	 * The current button shaper.
	 */
	private static SubstanceButtonShaper currentButtonShaper = null;

	/**
	 * Returns the current button shaper name.
	 * 
	 * @return Current button shaper name.
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(String)
	 * @see #setCurrentButtonShaper(SubstanceButtonShaper)
	 * @see #registerButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #unregisterButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #BUTTON_SHAPER_PROPERTY
	 */
	public static String getCurrentButtonShaperName() {
		if (SubstanceLookAndFeel.currentButtonShaper == null)
			return null;
		return SubstanceLookAndFeel.currentButtonShaper.getDisplayName();
	}

	/**
	 * Returns all available button shapers.
	 * 
	 * @return All available button shapers. Key - button shaper name, value -
	 *         button shaper information.
	 */
	public static Map<String, ButtonShaperInfo> getAllButtonShapers() {
		Map<String, ButtonShaperInfo> result = new TreeMap<String, ButtonShaperInfo>();
		for (Object buttonShaperPlugin : SubstanceLookAndFeel.shaperPlugins
				.getAvailablePlugins(true)) {
			for (ButtonShaperInfo buttonShaperInfo : ((SubstanceButtonShaperPlugin) buttonShaperPlugin)
					.getButtonShapers()) {
				result.put(buttonShaperInfo.getDisplayName(), buttonShaperInfo);
			}
		}
		return result;
	}

	/**
	 * Returns the current button shaper.
	 * 
	 * @return The current button shaper.
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(String)
	 * @see #setCurrentButtonShaper(SubstanceButtonShaper)
	 * @see #registerButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #unregisterButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #BUTTON_SHAPER_PROPERTY
	 */
	public static SubstanceButtonShaper getCurrentButtonShaper() {
		return SubstanceLookAndFeel.currentButtonShaper;
	}

	/**
	 * Sets new button shaper.
	 * 
	 * @param buttonShaperClassName
	 *            Button shaper class name (full qualified).
	 * @return The status of the button shaper change.
	 * @see #getCurrentButtonShaper()
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(SubstanceButtonShaper)
	 * @see #registerButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #unregisterButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #BUTTON_SHAPER_PROPERTY
	 */
	public static boolean setCurrentButtonShaper(String buttonShaperClassName) {
		return SubstanceLookAndFeel.setCurrentButtonShaper(
				buttonShaperClassName, true);
	}

	/**
	 * Sets new button shaper.
	 * 
	 * @param buttonShaperClassName
	 *            Button shaper class name (full qualified).
	 * @param toEnsureAccess
	 *            If <code>true</code>, an exception will be thrown when this
	 *            function is called and the current LAF (as returned by
	 *            {@link UIManager#getLookAndFeel()}) is not <b>Substance</b>.
	 * @return The status of the button shaper change.
	 */
	private static boolean setCurrentButtonShaper(String buttonShaperClassName,
			boolean toEnsureAccess) {
		try {
			Class<?> buttonShaperClass = Class.forName(buttonShaperClassName);
			if (buttonShaperClass == null) {
				return false;
			}
			Object obj = buttonShaperClass.newInstance();
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceButtonShaper)) {
				return false;
			}
			return SubstanceLookAndFeel
					.setCurrentButtonShaper((SubstanceButtonShaper) obj);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new button shaper.
	 * 
	 * @param currentButtonShaper
	 *            Button shaper object.
	 * @return The status of the button shaper change.
	 * @see #getCurrentButtonShaper()
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(String)
	 * @see #registerButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #unregisterButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #BUTTON_SHAPER_PROPERTY
	 */
	public static boolean setCurrentButtonShaper(
			SubstanceButtonShaper currentButtonShaper) {
		return SubstanceLookAndFeel.setCurrentButtonShaper(currentButtonShaper,
				true);
	}

	/**
	 * Sets new button shaper.
	 * 
	 * @param currentButtonShaper
	 *            Button shaper object.
	 * @param toReset
	 *            If <code>true</code>, relevant caches will be reset.
	 * @return The status of the button shaper change.
	 */
	private static boolean setCurrentButtonShaper(
			SubstanceButtonShaper currentButtonShaper, boolean toReset) {
		SubstanceLookAndFeel.currentButtonShaper = currentButtonShaper;
		if (toReset) {
			SubstanceCoreUtilities.resetCaches();
		}

		for (ButtonShaperChangeListener buttonShaperChangeListener : SubstanceLookAndFeel.buttonShaperChangeListeners)
			buttonShaperChangeListener.buttonShaperChanged();

		return true;
	}

	/**
	 * The current gradient painter.
	 */
	private static SubstanceGradientPainter currentGradientPainter = null;

	/**
	 * Returns the current gradient painter name.
	 * 
	 * @return Current gradient painter name.
	 * @see #getCurrentGradientPainter()
	 * @see #setCurrentGradientPainter(String)
	 * @see #setCurrentGradientPainter(SubstanceGradientPainter)
	 * @see #registerGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #unregisterGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #GRADIENT_PAINTER_PROPERTY
	 */
	public static String getCurrentGradientPainterName() {
		if (SubstanceLookAndFeel.currentGradientPainter == null)
			return null;
		return SubstanceLookAndFeel.currentGradientPainter.getDisplayName();
	}

	/**
	 * Returns all available gradient painters.
	 * 
	 * @return All available gradient painters. Key - gradient painter name,
	 *         value - gradient painter information.
	 */
	public static Map<String, GradientPainterInfo> getAllGradientPainters() {
		Map<String, GradientPainterInfo> result = new TreeMap<String, GradientPainterInfo>();
		for (Object gradientPainterPlugin : SubstanceLookAndFeel.painterPlugins
				.getAvailablePlugins(true)) {
			for (GradientPainterInfo gradientPainterInfo : ((SubstanceGradientPainterPlugin) gradientPainterPlugin)
					.getGradientPainters()) {
				result.put(gradientPainterInfo.getDisplayName(),
						gradientPainterInfo);
			}
		}
		return result;
	}

	/**
	 * Returns the current gradient painter.
	 * 
	 * @return The current gradient painter.
	 * @see #getCurrentGradientPainterName()
	 * @see #setCurrentGradientPainter(String)
	 * @see #setCurrentGradientPainter(SubstanceGradientPainter)
	 * @see #registerGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #unregisterGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #GRADIENT_PAINTER_PROPERTY
	 */
	public static SubstanceGradientPainter getCurrentGradientPainter() {
		return SubstanceLookAndFeel.currentGradientPainter;
	}

	/**
	 * Sets new gradient painter.
	 * 
	 * @param gradientPainterClassName
	 *            Gradient painter class name (full qualified).
	 * @return The status of the gradient painter change.
	 * @see #getCurrentGradientPainter()
	 * @see #getCurrentGradientPainterName()
	 * @see #setCurrentGradientPainter(SubstanceGradientPainter)
	 * @see #registerGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #unregisterGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #GRADIENT_PAINTER_PROPERTY
	 */
	public static boolean setCurrentGradientPainter(
			String gradientPainterClassName) {
		try {
			Class<?> gradientPainterClass = Class
					.forName(gradientPainterClassName);
			if (gradientPainterClass == null) {
				return false;
			}
			Object obj = gradientPainterClass.newInstance();
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceGradientPainter)) {
				return false;
			}
			return SubstanceLookAndFeel.setCurrentGradientPainter(
					(SubstanceGradientPainter) obj, true);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new gradient painter.
	 * 
	 * @param currentGradientPainter
	 *            Gradient painter object.
	 * @return The status of the gradient painter change.
	 * @see #getCurrentGradientPainter()
	 * @see #getCurrentGradientPainterName()
	 * @see #setCurrentGradientPainter(String)
	 * @see #registerGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #unregisterGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #GRADIENT_PAINTER_PROPERTY
	 */
	public static boolean setCurrentGradientPainter(
			SubstanceGradientPainter currentGradientPainter) {
		return SubstanceLookAndFeel.setCurrentGradientPainter(
				currentGradientPainter, true);
	}

	/**
	 * Sets new gradient painter.
	 * 
	 * @param currentGradientPainter
	 *            Gradient painter object.
	 * @param toReset
	 *            If <code>true</code>, internal image caches are reset.
	 * @return The status of the gradient painter change.
	 */
	private static boolean setCurrentGradientPainter(
			SubstanceGradientPainter currentGradientPainter, boolean toReset) {
		SubstanceLookAndFeel.currentGradientPainter = currentGradientPainter;
		if (toReset) {
			ButtonBackgroundDelegate.reset();
			SubstanceCheckBoxUI.reset();
			SubstanceRadioButtonUI.reset();
		}

		for (GradientPainterChangeListener gradientPainterChangeListener : SubstanceLookAndFeel.gradientPainterChangeListeners)
			gradientPainterChangeListener.gradientPainterChanged();

		return true;
	}

	/**
	 * The current border painter.
	 */
	private static SubstanceBorderPainter currentBorderPainter = null;

	/**
	 * Returns the current border painter name.
	 * 
	 * @return Current border painter name.
	 * @since version 4.0
	 * @see #getCurrentBorderPainter()
	 * @see #setCurrentBorderPainter(String)
	 * @see #setCurrentBorderPainter(SubstanceBorderPainter)
	 * @see #registerBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #unregisterBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #BORDER_PAINTER_PROPERTY
	 */
	public static String getCurrentBorderPainterName() {
		if (SubstanceLookAndFeel.currentBorderPainter == null)
			return null;
		return SubstanceLookAndFeel.currentBorderPainter.getDisplayName();
	}

	/**
	 * Returns all available border painters.
	 * 
	 * @return All available border painters. Key - border painter name, value -
	 *         border painter information.
	 * @since version 4.0
	 */
	public static Map<String, BorderPainterInfo> getAllBorderPainters() {
		Map<String, BorderPainterInfo> result = new TreeMap<String, BorderPainterInfo>();
		for (Object BorderPainterPlugin : SubstanceLookAndFeel.borderPainterPlugins
				.getAvailablePlugins(true)) {
			for (BorderPainterInfo BorderPainterInfo : ((SubstanceBorderPainterPlugin) BorderPainterPlugin)
					.getBorderPainters()) {
				result.put(BorderPainterInfo.getDisplayName(),
						BorderPainterInfo);
			}
		}
		return result;
	}

	/**
	 * Returns the current Border painter.
	 * 
	 * @return The current Border painter.
	 * @since version 4.0
	 * @see #getCurrentBorderPainterName()
	 * @see #setCurrentBorderPainter(String)
	 * @see #setCurrentBorderPainter(SubstanceBorderPainter)
	 * @see #registerBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #unregisterBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #BORDER_PAINTER_PROPERTY
	 */
	public static SubstanceBorderPainter getCurrentBorderPainter() {
		return SubstanceLookAndFeel.currentBorderPainter;
	}

	/**
	 * Sets new border painter.
	 * 
	 * @param borderPainterClassName
	 *            Border painter class name (full qualified).
	 * @return The status of the border painter change.
	 * @since version 4.0
	 * @see #getCurrentBorderPainter()
	 * @see #getCurrentBorderPainterName()
	 * @see #setCurrentBorderPainter(SubstanceBorderPainter)
	 * @see #registerBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #unregisterBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #BORDER_PAINTER_PROPERTY
	 */
	public static boolean setCurrentBorderPainter(String borderPainterClassName) {
		try {
			Class<?> BorderPainterClass = Class.forName(borderPainterClassName);
			if (BorderPainterClass == null) {
				return false;
			}
			Object obj = BorderPainterClass.newInstance();
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceBorderPainter)) {
				return false;
			}
			return SubstanceLookAndFeel.setCurrentBorderPainter(
					(SubstanceBorderPainter) obj, true);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Sets new border painter.
	 * 
	 * @param currentBorderPainter
	 *            Border painter object.
	 * @return The status of the border painter change.
	 * @since version 4.0
	 * @see #getCurrentBorderPainter()
	 * @see #getCurrentBorderPainterName()
	 * @see #setCurrentBorderPainter(String)
	 * @see #registerBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #unregisterBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #BORDER_PAINTER_PROPERTY
	 */
	public static boolean setCurrentBorderPainter(
			SubstanceBorderPainter currentBorderPainter) {
		return SubstanceLookAndFeel.setCurrentBorderPainter(
				currentBorderPainter, true);
	}

	/**
	 * Sets new border painter.
	 * 
	 * @param newBorderPainter
	 *            Border painter object.
	 * @param toReset
	 *            If <code>true</code>, internal image caches are reset.
	 * @return The status of the border painter change.
	 */
	private static boolean setCurrentBorderPainter(
			SubstanceBorderPainter newBorderPainter, boolean toReset) {
		SubstanceLookAndFeel.currentBorderPainter = newBorderPainter;

		if (toReset) {
			ButtonBackgroundDelegate.reset();
			SubstanceCheckBoxUI.reset();
			SubstanceRadioButtonUI.reset();
		}

		for (BorderPainterChangeListener borderPainterChangeListener : SubstanceLookAndFeel.borderPainterChangeListeners)
			borderPainterChangeListener.borderPainterChanged();

		return true;
	}

	/**
	 * The current text painter.
	 */
	private static SubstanceTextPainter currentTextPainter = new DefaultTextPainter();

	public static SubstanceTextPainter getCurrentTextPainter() {
		return SubstanceLookAndFeel.currentTextPainter;
	}

	public static boolean setCurrentTextPainter(
			SubstanceTextPainter newTextPainter) {
		SubstanceLookAndFeel.currentTextPainter = newTextPainter;
		return true;
	}

	private static SubstanceHighlightPainter currentHighlightPainter = new ClassicHighlightPainter();

	public static SubstanceHighlightPainter getCurrentHighlightPainter() {
		return SubstanceLookAndFeel.currentHighlightPainter;
	}

	public static boolean setCurrentHighlightPainter(
			SubstanceHighlightPainter newHighlightPainter) {
		SubstanceLookAndFeel.currentHighlightPainter = newHighlightPainter;
		return true;
	}

	private static SubstanceDecorationPainter currentDecorationPainter = new Glass3DDecorationPainter();

	public static SubstanceDecorationPainter getCurrentDecorationPainter() {
		return SubstanceLookAndFeel.currentDecorationPainter;
	}

	public static boolean setCurrentDecorationPainter(
			SubstanceDecorationPainter newDecorationPainter) {
		SubstanceLookAndFeel.currentDecorationPainter = newDecorationPainter;
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.LookAndFeel#getDescription()
	 */
	@Override
	public String getDescription() {
		return "Substance Look and Feel by Kirill Grouchnikov";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.LookAndFeel#getID()
	 */
	@Override
	public String getID() {
		return "Substance";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.LookAndFeel#getName()
	 */
	@Override
	public String getName() {
		return "Substance";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.LookAndFeel#isNativeLookAndFeel()
	 */
	@Override
	public boolean isNativeLookAndFeel() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.LookAndFeel#isSupportedLookAndFeel()
	 */
	@Override
	public boolean isSupportedLookAndFeel() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLookAndFeel#initClassDefaults(javax.swing.UIDefaults)
	 */
	@Override
	protected void initClassDefaults(UIDefaults table) {
		super.initClassDefaults(table);

		String UI_CLASSNAME_PREFIX = "org.jvnet.substance.Substance";
		Object[] uiDefaults = {

		"ButtonUI", UI_CLASSNAME_PREFIX + "ButtonUI",

		"CheckBoxUI", UI_CLASSNAME_PREFIX + "CheckBoxUI",

		"ComboBoxUI", UI_CLASSNAME_PREFIX + "ComboBoxUI",

		"CheckBoxMenuItemUI", UI_CLASSNAME_PREFIX + "CheckBoxMenuItemUI",

		"DesktopIconUI", UI_CLASSNAME_PREFIX + "DesktopIconUI",

		"DesktopPaneUI", UI_CLASSNAME_PREFIX + "DesktopPaneUI",

		"EditorPaneUI",
				UI_CLASSNAME_PREFIX + "EditorPaneUI",

				"FileChooserUI",
				UI_CLASSNAME_PREFIX + "FileChooserUI",

				// "FileChooserUI", "javax.swing.plaf.metal.MetalFileChooserUI",

				"FormattedTextFieldUI",
				UI_CLASSNAME_PREFIX + "FormattedTextFieldUI",

				"InternalFrameUI", UI_CLASSNAME_PREFIX + "InternalFrameUI",

				"LabelUI", UI_CLASSNAME_PREFIX + "LabelUI",

				"ListUI", UI_CLASSNAME_PREFIX + "ListUI",

				"MenuUI", UI_CLASSNAME_PREFIX + "MenuUI",

				"MenuBarUI", UI_CLASSNAME_PREFIX + "MenuBarUI",

				"MenuItemUI", UI_CLASSNAME_PREFIX + "MenuItemUI",

				"OptionPaneUI", UI_CLASSNAME_PREFIX + "OptionPaneUI",

				"PanelUI", UI_CLASSNAME_PREFIX + "PanelUI",

				"PasswordFieldUI", UI_CLASSNAME_PREFIX + "PasswordFieldUI",

				"PopupMenuUI", UI_CLASSNAME_PREFIX + "PopupMenuUI",

				"PopupMenuSeparatorUI",
				UI_CLASSNAME_PREFIX + "PopupMenuSeparatorUI",

				"ProgressBarUI", UI_CLASSNAME_PREFIX + "ProgressBarUI",

				"RadioButtonUI", UI_CLASSNAME_PREFIX + "RadioButtonUI",

				"RadioButtonMenuItemUI",
				UI_CLASSNAME_PREFIX + "RadioButtonMenuItemUI",

				"RootPaneUI", UI_CLASSNAME_PREFIX + "RootPaneUI",

				"ScrollBarUI", UI_CLASSNAME_PREFIX + "ScrollBarUI",

				"ScrollPaneUI", UI_CLASSNAME_PREFIX + "ScrollPaneUI",

				"SeparatorUI", UI_CLASSNAME_PREFIX + "SeparatorUI",

				"SliderUI", UI_CLASSNAME_PREFIX + "SliderUI",

				"SpinnerUI", UI_CLASSNAME_PREFIX + "SpinnerUI",

				"SplitPaneUI", UI_CLASSNAME_PREFIX + "SplitPaneUI",

				"TabbedPaneUI", UI_CLASSNAME_PREFIX + "TabbedPaneUI",

				"TableUI", UI_CLASSNAME_PREFIX + "TableUI",

				"TableHeaderUI", UI_CLASSNAME_PREFIX + "TableHeaderUI",

				"TextAreaUI", UI_CLASSNAME_PREFIX + "TextAreaUI",

				"TextFieldUI", UI_CLASSNAME_PREFIX + "TextFieldUI",

				"TextPaneUI", UI_CLASSNAME_PREFIX + "TextPaneUI",

				"ToggleButtonUI", UI_CLASSNAME_PREFIX + "ToggleButtonUI",

				"ToolBarUI", UI_CLASSNAME_PREFIX + "ToolBarUI",

				"ToolBarSeparatorUI",
				UI_CLASSNAME_PREFIX + "ToolBarSeparatorUI",

				"ToolTipUI", UI_CLASSNAME_PREFIX + "ToolTipUI",

				"TreeUI", UI_CLASSNAME_PREFIX + "TreeUI",

		};
		table.putDefaults(uiDefaults);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLookAndFeel#initComponentDefaults(javax.swing.UIDefaults)
	 */
	@Override
	protected void initComponentDefaults(UIDefaults table) {
		super.initComponentDefaults(table);

		// if (LookUtils.IS_OS_WINDOWS) {
		initFontDefaults(table);
		// } else {
		// String fontFamilyName = SubstanceLookAndFeel.getFontFamilyName();
		// int regKind = SubstanceLookAndFeel.isBasicFontBold() ? Font.BOLD
		// : Font.PLAIN;
		// int baseFontSize = SubstanceLookAndFeel.getBasicFontSize();
		// Font fontPlainBase = new FontUIResource(new Font(fontFamilyName,
		// regKind, baseFontSize));
		// Font fontBoldBase = new FontUIResource(new Font(fontFamilyName,
		// Font.BOLD, baseFontSize));
		// Font fontBoldBaseP1 = new FontUIResource(new Font(fontFamilyName,
		// Font.BOLD, baseFontSize + 1));
		// Object[] fontDefaults = {
		//
		// "Button.font", fontPlainBase,
		//
		// "CheckBox.font", fontPlainBase,
		//
		// "CheckBoxMenuItem.font", fontPlainBase,
		//
		// "CheckBoxMenuItem.acceleratorFont", fontPlainBase,
		//
		// "ComboBox.font", fontPlainBase,
		//
		// "DesktopIcon.font", fontBoldBaseP1,
		//
		// "EditorPane.font", fontPlainBase,
		//
		// "FormattedTextField.font", fontPlainBase,
		//
		// "InternalFrame.titleFont", fontBoldBaseP1,
		//
		// "Label.font", fontPlainBase,
		//
		// "List.font", fontPlainBase,
		//
		// "Menu.font", fontPlainBase,
		//
		// "Menu.acceleratorFont", fontPlainBase,
		//
		// "MenuBar.font", fontPlainBase,
		//
		// "MenuItem.font", fontPlainBase,
		//
		// "MenuItem.acceleratorFont", fontPlainBase,
		//
		// "OptionPane.font", fontPlainBase,
		//
		// "OptionPane.messageFont", fontPlainBase,
		//
		// "OptionPane.buttonFont", fontPlainBase,
		//
		// "Panel.font", fontPlainBase,
		//
		// "PasswordField.font", fontPlainBase,
		//
		// "PopupMenu.font", fontPlainBase,
		//
		// "ProgressBar.font", fontPlainBase,
		//
		// "RadioButton.font", fontPlainBase,
		//
		// "RadioButtonMenuItem.font", fontPlainBase,
		//
		// "RadioButtonMenuItem.acceleratorFont", fontPlainBase,
		//
		// "ScrollPane.font", fontPlainBase,
		//
		// "Spinner.font", fontPlainBase,
		//
		// "TabbedPane.font", fontPlainBase,
		//
		// "Table.font", fontPlainBase,
		//
		// "TableHeader.font", fontPlainBase,
		//
		// "TextField.font", fontPlainBase,
		//
		// "TextPane.font", fontPlainBase,
		//
		// "ToolBar.font", fontPlainBase,
		//
		// "ToggleButton.font", fontPlainBase,
		//
		// "Tree.font", fontPlainBase,
		//
		// "Viewport.font", fontPlainBase,
		//
		// "Spinner.font", fontPlainBase,
		//
		// "TextArea.font", fontPlainBase,
		//
		// "TitledBorder.font", fontBoldBase,
		//
		// "ToolTip.font", fontPlainBase,
		//
		// };
		// table.putDefaults(fontDefaults);
		// }
		SubstanceLookAndFeel.getTheme().addCustomEntriesToTable(table);
	}

	/**
	 * Sets the {@link FontPolicy} to be used with Substance family. If the
	 * specified policy is <code>null</code>, the default will be reset. Note
	 * that after calling this method you need to call
	 * {@link UIManager#setLookAndFeel(LookAndFeel)} with Substance and
	 * {@link SwingUtilities#updateComponentTreeUI(Component)} to have the new
	 * font policy applied to your windows.
	 * 
	 * @param fontPolicy
	 *            The {@link FontPolicy} to be used with Substance family, or
	 *            <code>null</code> to reset to the default
	 * 
	 * @see #getFontPolicy()
	 * @see SubstanceLookAndFeel#SUBSTANCE_FONT_POLICY_KEY
	 */
	public static void setFontPolicy(FontPolicy fontPolicy) {
		UIManager.put(SUBSTANCE_FONT_POLICY_KEY, fontPolicy);
		SubstanceSizeUtils.setControlFontSize(-1);
		SubstanceSizeUtils.resetPointsToPixelsRatio(fontPolicy);
	}

	/**
	 * Looks up and retrieves the {@link FontPolicy} used by the Substance
	 * family. If a {@link FontPolicy} has been set, it'll be returned.
	 * Otherwise, this method checks if a {@link FontPolicy} or {@link FontSet}
	 * is defined in the system properties or UIDefaults. If so, it is returned.
	 * If no {@link FontPolicy} has been set for this look, in the system
	 * properties or {@link UIDefaults}, the default Substance font policy will
	 * be returned.
	 * 
	 * @return the {@link FontPolicy} set for this Look&amp;feel - if any, the
	 *         {@link FontPolicy} specified in the system properties or
	 *         {@link UIDefaults} - if any, or the default Substance font
	 *         policy.
	 * 
	 * @see #setFontPolicy
	 * @see FontPolicies
	 * @see FontPolicies#customSettingsPolicy(FontPolicy)
	 */
	public static FontPolicy getFontPolicy() {
		FontPolicy policy = (FontPolicy) UIManager
				.get(SUBSTANCE_FONT_POLICY_KEY);
		if (policy != null)
			return policy;

		// return default policy
		return SubstanceFontUtilities.getDefaultFontPolicy();
	}

	/**
	 * Looks up the correct control font and sets it for all controls.
	 * 
	 * @param table
	 *            The UI defaults table.
	 */
	private void initFontDefaults(UIDefaults table) {
		FontSet substanceFontSet = getFontPolicy()
				.getFontSet("Substance", null);
		initFontDefaults(table, substanceFontSet);
	}

	// public static FontSet getFontSet() {
	// FontPolicy fontChoicePolicy = getFontPolicy();
	// FontSet fontSet = fontChoicePolicy.getFontSet("Substance", null);
	// FontSet substanceFontSet = new SubstanceFontSet(fontSet);
	// return substanceFontSet;
	// }
	//
	/**
	 * Sets Fonts in the given FontSet as defaults for all known component types
	 * in the given UIDefaults table.
	 * 
	 * @param table
	 *            the UIDefaults table used to set fonts
	 * @param fontSet
	 *            describes the set of Fonts to be installed
	 */
	private static void initFontDefaults(UIDefaults table, FontSet fontSet) {
		Font controlFont = fontSet.getControlFont();
		Font menuFont = fontSet.getMenuFont();
		Font messageFont = fontSet.getMessageFont();
		Font toolTipFont = fontSet.getSmallFont();
		Font titleFont = fontSet.getTitleFont();
		Font windowFont = fontSet.getWindowTitleFont();

		// System.out.println("Control: " + fontSet.getControlFont());
		// System.out.println("Menu: " + fontSet.getMenuFont());
		// System.out.println("Message: " + fontSet.getMessageFont());
		// System.out.println("Small: " + fontSet.getSmallFont());
		// System.out.println("Title: " + fontSet.getTitleFont());
		// System.out.println("Window title: " + fontSet.getWindowTitleFont());

		Object[] defaults = {

		"Button.font", controlFont,

		"CheckBox.font", controlFont,

		"ColorChooser.font", controlFont,

		"ComboBox.font", controlFont,

		"EditorPane.font", controlFont,

		"FormattedTextField.font", controlFont,

		"Label.font", controlFont,

		"List.font", controlFont,

		"Panel.font", controlFont,

		"PasswordField.font", controlFont,

		"ProgressBar.font", controlFont,

		"RadioButton.font", controlFont,

		"ScrollPane.font", controlFont,

		"Spinner.font", controlFont,

		"TabbedPane.font", controlFont,

		"Table.font", controlFont,

		"TableHeader.font", controlFont,

		"TextArea.font", controlFont,

		"TextField.font", controlFont,

		"TextPane.font", controlFont,

		"ToolBar.font", controlFont,

		"ToggleButton.font", controlFont,

		"Tree.font", controlFont,

		"Viewport.font", controlFont,

		"InternalFrame.titleFont", windowFont,

		"DesktopIcon.titleFont", windowFont,

		"OptionPane.font", messageFont,

		"OptionPane.messageFont", messageFont,

		"OptionPane.buttonFont", messageFont,

		"TitledBorder.font", titleFont,

		"ToolTip.font", toolTipFont,

		"CheckBoxMenuItem.font", menuFont,

		"CheckBoxMenuItem.acceleratorFont", menuFont,

		"Menu.font", menuFont,

		"Menu.acceleratorFont", menuFont,

		"MenuBar.font", menuFont,

		"MenuItem.font", menuFont,

		"MenuItem.acceleratorFont", menuFont,

		"PopupMenu.font", menuFont,

		"RadioButtonMenuItem.font", menuFont,

		"RadioButtonMenuItem.acceleratorFont", menuFont,
		// ?
		};
		table.putDefaults(defaults);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLookAndFeel#getDefaults()
	 */
	@Override
	public UIDefaults getDefaults() {
		UIDefaults table = super.getDefaults();

		SubstanceLookAndFeel.componentPlugins.processAllDefaultsEntries(table,
				SubstanceLookAndFeel.currentTheme);
		return table;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLookAndFeel#initialize()
	 */
	@Override
	public void initialize() {
		super.initialize();
		ShadowPopupFactory.install();
		if (SubstanceLookAndFeel.currentTheme == null) {
			// set theme
			String paramTheme = SubstanceLookAndFeel.paramReader
					.getThemeProperty();
			boolean isSetTheme = false;
			if (paramTheme != null) {
				isSetTheme = SubstanceLookAndFeel.setCurrentTheme(paramTheme,
						false);
			}
			if (!isSetTheme) {
				try {
					for (Object themePlugin : SubstanceLookAndFeel.themePlugins
							.getAvailablePlugins(true)) {
						String defaultThemeClassName = ((SubstanceThemePlugin) themePlugin)
								.getDefaultThemeClassName();
						if (defaultThemeClassName == null)
							continue;
						isSetTheme = SubstanceLookAndFeel.setCurrentTheme(
								defaultThemeClassName, false);
						if (isSetTheme)
							break;
					}
					if (!isSetTheme)
						SubstanceLookAndFeel.setCurrentTheme(
								new SubstanceAquaTheme(), false);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		} else {
			SubstanceLookAndFeel.setCurrentTheme(
					SubstanceLookAndFeel.currentTheme, false);
		}

		ImageWatermarkKind imageWatermarkKind = SubstanceLookAndFeel.paramReader
				.getWatermarkImageKindProperty();
		if (imageWatermarkKind != null)
			SubstanceLookAndFeel.setImageWatermarkKind(imageWatermarkKind);

		Float imageWatermarkOpacity = SubstanceLookAndFeel.paramReader
				.getWatermarkImageOpacityProperty();
		if (imageWatermarkOpacity != null)
			SubstanceLookAndFeel
					.setImageWatermarkOpacity(imageWatermarkOpacity);

		if (SubstanceLookAndFeel.currentWatermark == null) {
			// set watermark
			String paramWatermark = SubstanceLookAndFeel.paramReader
					.getWatermarkProperty();
			boolean isSetWatermark = false;
			if (paramWatermark != null) {
				isSetWatermark = SubstanceLookAndFeel.setCurrentWatermark(
						paramWatermark, SubstanceLookAndFeel.paramReader);
			}
			if (!isSetWatermark) {
				try {
					for (Object watermarkPlugin : SubstanceLookAndFeel.watermarkPlugins
							.getAvailablePlugins(true)) {
						String defaultWatermarkClassName = ((SubstanceWatermarkPlugin) watermarkPlugin)
								.getDefaultWatermarkClassName();
						if (defaultWatermarkClassName == null)
							continue;
						isSetWatermark = SubstanceLookAndFeel
								.setCurrentWatermark(defaultWatermarkClassName,
										SubstanceLookAndFeel.paramReader);
						if (isSetWatermark)
							break;
					}
					if (!isSetWatermark)
						SubstanceLookAndFeel
								.setCurrentWatermark(new SubstanceStripeWatermark());
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		}

		if (SubstanceLookAndFeel.currentButtonShaper == null) {
			// set button shaper
			String paramButtonShaper = SubstanceLookAndFeel.paramReader
					.getButtonShaperProperty();
			boolean isSetButtonShaper = false;
			if (paramButtonShaper != null) {
				isSetButtonShaper = SubstanceLookAndFeel
						.setCurrentButtonShaper(paramButtonShaper, false);
			}
			if (!isSetButtonShaper) {
				try {
					for (Object buttonShaperPlugin : SubstanceLookAndFeel.shaperPlugins
							.getAvailablePlugins(true)) {
						for (ButtonShaperInfo buttonShaperInfo : ((SubstanceButtonShaperPlugin) buttonShaperPlugin)
								.getButtonShapers()) {
							if (buttonShaperInfo.isDefault()) {
								isSetButtonShaper = SubstanceLookAndFeel
										.setCurrentButtonShaper(
												buttonShaperInfo.getClassName(),
												false);
							}
						}
					}
					if (!isSetButtonShaper)
						SubstanceLookAndFeel.setCurrentButtonShaper(
								new StandardButtonShaper(), false);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		}

		if (SubstanceLookAndFeel.currentGradientPainter == null) {
			// set gradient painter
			String paramGradientPainter = SubstanceLookAndFeel.paramReader
					.getGradientPainterProperty();
			boolean isSetGradientPainter = false;
			if (paramGradientPainter != null) {
				isSetGradientPainter = SubstanceLookAndFeel
						.setCurrentGradientPainter(paramGradientPainter);
			}
			if (!isSetGradientPainter) {
				try {
					for (Object gradientPainterPlugin : SubstanceLookAndFeel.painterPlugins
							.getAvailablePlugins(true)) {
						for (GradientPainterInfo gradientPainterInfo : ((SubstanceGradientPainterPlugin) gradientPainterPlugin)
								.getGradientPainters()) {
							if (gradientPainterInfo.isDefault()) {
								isSetGradientPainter = SubstanceLookAndFeel
										.setCurrentGradientPainter(gradientPainterInfo
												.getClassName());
							}
						}
					}
					if (!isSetGradientPainter)
						SubstanceLookAndFeel.setCurrentGradientPainter(
								new StandardGradientPainter(), false);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		}

		if (SubstanceLookAndFeel.currentBorderPainter == null) {
			// set border painter
			String paramBorderPainter = SubstanceLookAndFeel.paramReader
					.getBorderPainterProperty();
			boolean isSetBorderPainter = false;
			if (paramBorderPainter != null) {
				isSetBorderPainter = SubstanceLookAndFeel
						.setCurrentBorderPainter(paramBorderPainter);
			}
			if (!isSetBorderPainter) {
				try {
					for (Object borderPainterPlugin : SubstanceLookAndFeel.borderPainterPlugins
							.getAvailablePlugins(true)) {
						for (BorderPainterInfo borderPainterInfo : ((SubstanceBorderPainterPlugin) borderPainterPlugin)
								.getBorderPainters()) {
							if (borderPainterInfo.isDefault()) {
								isSetBorderPainter = SubstanceLookAndFeel
										.setCurrentBorderPainter(borderPainterInfo
												.getClassName());
							}
						}
					}
					if (!isSetBorderPainter)
						SubstanceLookAndFeel.setCurrentBorderPainter(
								new StandardBorderPainter(), false);
				} catch (Exception exc) {
					exc.printStackTrace();
				}
			}
		}

		// tracer for memory analysis
		String paramTraceFile = SubstanceLookAndFeel.paramReader
				.getTraceFileNameProperty();
		if (paramTraceFile != null) {
			MemoryAnalyzer.commence(1000, paramTraceFile);
			for (Object plugin : SubstanceLookAndFeel.componentPlugins
					.getAvailablePlugins(true))
				MemoryAnalyzer.enqueueUsage("Has plugin '"
						+ plugin.getClass().getName() + "'");
			for (Object plugin : SubstanceLookAndFeel.themePlugins
					.getAvailablePlugins(true))
				MemoryAnalyzer.enqueueUsage("Has theme plugin '"
						+ plugin.getClass().getName() + "'");
			for (Object plugin : SubstanceLookAndFeel.watermarkPlugins
					.getAvailablePlugins(true))
				MemoryAnalyzer.enqueueUsage("Has watermark plugin '"
						+ plugin.getClass().getName() + "'");
			for (Object plugin : SubstanceLookAndFeel.shaperPlugins
					.getAvailablePlugins(true))
				MemoryAnalyzer.enqueueUsage("Has button shaper plugin '"
						+ plugin.getClass().getName() + "'");

		}

		// to show heap status panel in title pane?
		String heapStatusPanelParam = SubstanceLookAndFeel.paramReader
				.toShowHeapStatusPanelProperty();
		SubstanceTitlePane
				.setCanHaveHeapStatusPanel(heapStatusPanelParam != null);
		SubstanceTitlePane.setHeapStatusLogfileName(heapStatusPanelParam);

		// to show extra UI elements?
		SubstanceLookAndFeel.toShowExtraElements = SubstanceLookAndFeel.paramReader
				.toShowExtraElementProperty();

		// to enable inverted themes?
		SubstanceLookAndFeel.toEnableInvertedThemes = SubstanceLookAndFeel.paramReader
				.toEnableInvertedThemes();

		// to enable inverted themes?
		SubstanceLookAndFeel.toEnableNegatedThemes = SubstanceLookAndFeel.paramReader
				.toEnableNegatedThemes();

		// to bleed watermark?
		SubstanceLookAndFeel.toBleedWatermark = SubstanceLookAndFeel.paramReader
				.toBleedWatermark();

		// is debug UI mode?
		SubstanceLookAndFeel.isDebugUiMode = SubstanceLookAndFeel.paramReader
				.isDebugUiMode();

		try {
			Class.forName("org.jvnet.lafwidget.menu.MenuSearchWidget");
			SubstanceMenuBarUI.showSearchPanels();
		} catch (ClassNotFoundException cnfe) {
		}

		// initialize component plugins
		SubstanceLookAndFeel.componentPlugins.initializeAll();

		// initialize widget support
		LafWidgetRepository.getRepository().setLafSupport(
				new SubstanceWidgetSupport());

		// fix for defect 208 - tracking changes to focus owner
		// and repainting the default button
		this.focusOwnerChangeListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if ("focusOwner".equals(evt.getPropertyName())) {
					Component newFocusOwner = (Component) evt.getNewValue();
					if (newFocusOwner != null) {
						JRootPane rootPane = SwingUtilities
								.getRootPane(newFocusOwner);
						if (rootPane == null)
							return;
						JButton defaultButton = rootPane.getDefaultButton();
						if (defaultButton == null)
							return;
						defaultButton.repaint();
					}
				}
				if ("managingFocus".equals(evt.getPropertyName())) {
					if (Boolean.FALSE.equals(evt.getNewValue())) {
						// new keyboard focus manager has been installed
						currentKeyboardFocusManager
								.removePropertyChangeListener(focusOwnerChangeListener);
						currentKeyboardFocusManager = KeyboardFocusManager
								.getCurrentKeyboardFocusManager();
						currentKeyboardFocusManager
								.addPropertyChangeListener(focusOwnerChangeListener);
					}
				}
			}
		};
		this.currentKeyboardFocusManager = KeyboardFocusManager
				.getCurrentKeyboardFocusManager();
		this.currentKeyboardFocusManager
				.addPropertyChangeListener(this.focusOwnerChangeListener);

		// TitleButtonManager.getManager().configure();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLookAndFeel#uninitialize()
	 */
	@Override
	public void uninitialize() {
		super.uninitialize();
		ShadowPopupFactory.uninstall();

		SubstanceLookAndFeel.stopThreads();
		// if (MemoryAnalyzer.isRunning())
		// MemoryAnalyzer.requestStop();

		try {
			Class.forName("org.jvnet.lafwidget.menu.MenuSearchWidget");
			SubstanceMenuBarUI.hideSearchPanels(false);
		} catch (ClassNotFoundException cnfe) {
		}

		// fix for defect 109 - memory leak on watermarks
		SubstanceLookAndFeel.currentWatermark.dispose();

		// uninitialize component plugins
		SubstanceLookAndFeel.componentPlugins.uninitializeAll();

		// reset widget support
		LafWidgetRepository.getRepository().unsetLafSupport();

		// dispose the current text painter
		SubstanceLookAndFeel.currentTextPainter.dispose();

		this.currentKeyboardFocusManager
				.removePropertyChangeListener(this.focusOwnerChangeListener);
		this.focusOwnerChangeListener = null;
		this.currentKeyboardFocusManager = null;

		// TitleButtonManager.getManager().clean();
	}

	/**
	 * Stops all Substance threads. Use with extreme caution - improper use may
	 * result in UI artifacts and runtime exceptions. In general, this method is
	 * for internal use only.
	 */
	public static void stopThreads() {
		PulseTracker.stopAllTimers();
		// TabPulseTracker.stopAllTimers();
		// TabAnimationTracker.stopAllTimers();
		FadeTracker.getInstance().stopAllTimers();
		TrackableThread.requestStopAllThreads();
	}

	/**
	 * Returns all available plugins. Each entry is {@link LafPlugin}.
	 * 
	 * @return All available plugins.
	 */
	@SuppressWarnings("unchecked")
	public static Set getPlugins() {
		return Collections
				.unmodifiableSet(SubstanceLookAndFeel.componentPlugins
						.getAvailablePlugins(true));
	}

	/**
	 * Registers a new listener on skin change.
	 * 
	 * @param skinChangeListener
	 *            New listener on skin change.
	 * @see #setSkin(String)
	 * @see #setSkin(SubstanceSkin)
	 * @see #unregisterSkinChangeListener(SkinChangeListener)
	 */
	public static void registerSkinChangeListener(
			SkinChangeListener skinChangeListener) {
		SubstanceLookAndFeel.skinChangeListeners.add(skinChangeListener);
	}

	/**
	 * Unregisters a listener on skin change.
	 * 
	 * @param skinChangeListener
	 *            The listener to unregister.
	 * @see #setSkin(String)
	 * @see #setSkin(SubstanceSkin)
	 * @see #registerSkinChangeListener(SkinChangeListener)
	 */
	public static void unregisterSkinChangeListener(
			SkinChangeListener skinChangeListener) {
		SubstanceLookAndFeel.skinChangeListeners.remove(skinChangeListener);
	}

	/**
	 * Registers a new listener on theme change.
	 * 
	 * @param themeChangeListener
	 *            New listener on theme change.
	 * @see #getCurrentThemeName()
	 * @see #getTheme()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #unregisterThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static void registerThemeChangeListener(
			ThemeChangeListener themeChangeListener) {
		SubstanceLookAndFeel.themeChangeListeners.add(themeChangeListener);
	}

	/**
	 * Unregisters a listener on theme change.
	 * 
	 * @param themeChangeListener
	 *            The listener to unregister.
	 * @see #getCurrentThemeName()
	 * @see #getTheme()
	 * @see #setCurrentTheme(String)
	 * @see #setCurrentTheme(SubstanceTheme)
	 * @see #setCurrentTheme(ThemeInfo)
	 * @see #registerThemeChangeListener(ThemeChangeListener)
	 * @see #THEME_PROPERTY
	 */
	public static void unregisterThemeChangeListener(
			ThemeChangeListener themeChangeListener) {
		SubstanceLookAndFeel.themeChangeListeners.remove(themeChangeListener);
	}

	/**
	 * Registers a new listener on watermark change.
	 * 
	 * @param watermarkChangeListener
	 *            New listener on watermark change.
	 * @see #getCurrentWatermark()
	 * @see #getCurrentWatermarkName()
	 * @see #setCurrentWatermark(String)
	 * @see #setCurrentWatermark(SubstanceWatermark)
	 * @see #unregisterWatermarkChangeListener(WatermarkChangeListener)
	 * @see #WATERMARK_PROPERTY
	 */
	public static void registerWatermarkChangeListener(
			WatermarkChangeListener watermarkChangeListener) {
		SubstanceLookAndFeel.watermarkChangeListeners
				.add(watermarkChangeListener);
	}

	/**
	 * Unregisters a listener on watermark change.
	 * 
	 * @param watermarkChangeListener
	 *            The listener to unregister.
	 * @see #getCurrentWatermark()
	 * @see #getCurrentWatermarkName()
	 * @see #setCurrentWatermark(String)
	 * @see #setCurrentWatermark(SubstanceWatermark)
	 * @see #registerWatermarkChangeListener(WatermarkChangeListener)
	 * @see #WATERMARK_PROPERTY
	 */
	public static void unregisterWatermarkChangeListener(
			WatermarkChangeListener watermarkChangeListener) {
		SubstanceLookAndFeel.watermarkChangeListeners
				.remove(watermarkChangeListener);
	}

	/**
	 * Registers a new listener on button shaper change.
	 * 
	 * @param buttonShaperChangeListener
	 *            New listener on button shaper change.
	 * @see #getCurrentButtonShaper()
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(String)
	 * @see #setCurrentButtonShaper(SubstanceButtonShaper)
	 * @see #unregisterButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #BUTTON_SHAPER_PROPERTY
	 */
	public static void registerButtonShaperChangeListener(
			ButtonShaperChangeListener buttonShaperChangeListener) {
		SubstanceLookAndFeel.buttonShaperChangeListeners
				.add(buttonShaperChangeListener);
	}

	/**
	 * Unregisters a listener on button shaper change.
	 * 
	 * @param buttonShaperChangeListener
	 *            The listener to unregister.
	 * @see #getCurrentButtonShaper()
	 * @see #getCurrentButtonShaperName()
	 * @see #setCurrentButtonShaper(String)
	 * @see #setCurrentButtonShaper(SubstanceButtonShaper)
	 * @see #registerButtonShaperChangeListener(ButtonShaperChangeListener)
	 * @see #BUTTON_SHAPER_PROPERTY
	 */
	public static void unregisterButtonShaperChangeListener(
			ButtonShaperChangeListener buttonShaperChangeListener) {
		SubstanceLookAndFeel.buttonShaperChangeListeners
				.remove(buttonShaperChangeListener);
	}

	/**
	 * Registers a new listener on gradient painter change.
	 * 
	 * @param gradientPainterChangeListener
	 *            New listener on gradient painter change.
	 * @see #getCurrentGradientPainter()
	 * @see #getCurrentGradientPainterName()
	 * @see #setCurrentGradientPainter(String)
	 * @see #setCurrentGradientPainter(SubstanceGradientPainter)
	 * @see #unregisterGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #GRADIENT_PAINTER_PROPERTY
	 */
	public static void registerGradientPainterChangeListener(
			GradientPainterChangeListener gradientPainterChangeListener) {
		SubstanceLookAndFeel.gradientPainterChangeListeners
				.add(gradientPainterChangeListener);
	}

	/**
	 * Unregisters a listener on gradient painter change.
	 * 
	 * @param gradientPainterChangeListener
	 *            The listener to unregister.
	 * @see #getCurrentGradientPainter()
	 * @see #getCurrentGradientPainterName()
	 * @see #setCurrentGradientPainter(String)
	 * @see #setCurrentGradientPainter(SubstanceGradientPainter)
	 * @see #registerGradientPainterChangeListener(GradientPainterChangeListener)
	 * @see #GRADIENT_PAINTER_PROPERTY
	 */
	public static void unregisterGradientPainterChangeListener(
			GradientPainterChangeListener gradientPainterChangeListener) {
		SubstanceLookAndFeel.gradientPainterChangeListeners
				.remove(gradientPainterChangeListener);
	}

	/**
	 * Registers a new listener on border painter change.
	 * 
	 * @param borderPainterChangeListener
	 *            New listener on border painter change.
	 * @see #getCurrentBorderPainter()
	 * @see #getCurrentBorderPainterName()
	 * @see #setCurrentBorderPainter(String)
	 * @see #setCurrentBorderPainter(SubstanceBorderPainter)
	 * @see #unregisterBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #BORDER_PAINTER_PROPERTY
	 */
	public static void registerBorderPainterChangeListener(
			BorderPainterChangeListener borderPainterChangeListener) {
		SubstanceLookAndFeel.borderPainterChangeListeners
				.add(borderPainterChangeListener);
	}

	/**
	 * Unregisters a listener on border painter change.
	 * 
	 * @param borderPainterChangeListener
	 *            The listener to unregister.
	 * @see #getCurrentBorderPainter()
	 * @see #getCurrentBorderPainterName()
	 * @see #setCurrentBorderPainter(String)
	 * @see #setCurrentBorderPainter(SubstanceBorderPainter)
	 * @see #registerBorderPainterChangeListener(BorderPainterChangeListener)
	 * @see #BORDER_PAINTER_PROPERTY
	 */
	public static void unregisterBorderPainterChangeListener(
			BorderPainterChangeListener borderPainterChangeListener) {
		SubstanceLookAndFeel.borderPainterChangeListeners
				.remove(borderPainterChangeListener);
	}

	/**
	 * Registers the specified listener on tab-close events on <b>all</b>
	 * tabbed panes.
	 * 
	 * @param tabCloseListener
	 *            Listener to register.
	 * @see #registerTabCloseChangeListener(JTabbedPane, BaseTabCloseListener)
	 * @see #unregisterTabCloseChangeListener(BaseTabCloseListener)
	 * @see #unregisterTabCloseChangeListener(JTabbedPane, BaseTabCloseListener)
	 */
	public static void registerTabCloseChangeListener(
			BaseTabCloseListener tabCloseListener) {
		TabCloseListenerManager.getInstance()
				.registerListener(tabCloseListener);
	}

	/**
	 * Registers the specified listener on tab-close events on <b>the specified</b>
	 * tabbed pane.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane. If <code>null</code>, the tab close listener
	 *            is registered globally (for all tabbed panes).
	 * @param tabCloseListener
	 *            Listener to register.
	 * @see #registerTabCloseChangeListener(BaseTabCloseListener)
	 * @see #unregisterTabCloseChangeListener(BaseTabCloseListener)
	 * @see #unregisterTabCloseChangeListener(JTabbedPane, BaseTabCloseListener)
	 */
	public static void registerTabCloseChangeListener(JTabbedPane tabbedPane,
			BaseTabCloseListener tabCloseListener) {
		TabCloseListenerManager.getInstance().registerListener(tabbedPane,
				tabCloseListener);
	}

	/**
	 * Unregisters the specified listener on tab-close events on <b>all</b>
	 * tabbed panes.
	 * 
	 * @param tabCloseListener
	 *            Listener to unregister.
	 * @see #registerTabCloseChangeListener(BaseTabCloseListener)
	 * @see #registerTabCloseChangeListener(JTabbedPane, BaseTabCloseListener)
	 * @see #unregisterTabCloseChangeListener(JTabbedPane, BaseTabCloseListener)
	 */
	public static void unregisterTabCloseChangeListener(
			BaseTabCloseListener tabCloseListener) {
		TabCloseListenerManager.getInstance().unregisterListener(
				tabCloseListener);
	}

	/**
	 * Unregisters the specified listener on tab-close events on <b>the
	 * specified</b> tabbed pane.
	 * 
	 * @param tabbedPane
	 *            Tabbed pane. If <code>null</code>, the tab close listener
	 *            is unregistered globally (for all tabbed panes).
	 * @param tabCloseListener
	 *            Listener to unregister.
	 * @see #registerTabCloseChangeListener(BaseTabCloseListener)
	 * @see #registerTabCloseChangeListener(JTabbedPane, BaseTabCloseListener)
	 * @see #unregisterTabCloseChangeListener(BaseTabCloseListener)
	 */
	public static void unregisterTabCloseChangeListener(JTabbedPane tabbedPane,
			BaseTabCloseListener tabCloseListener) {
		TabCloseListenerManager.getInstance().unregisterListener(tabbedPane,
				tabCloseListener);
	}

	/**
	 * Returns the set of all listeners registered on tab-close events on <b>all</b>
	 * tabbed panes.
	 * 
	 * @return Set of all listeners registered on tab-close events on <b>all</b>
	 *         tabbed panes.
	 */
	public static Set<BaseTabCloseListener> getAllTabCloseListeners() {
		return TabCloseListenerManager.getInstance().getListeners();
	}

	/**
	 * Returns all listeners registered on tab closing of the specified tabbed
	 * pane.
	 * 
	 * @param tabbedPane
	 *            A tabbed pane. If <code>null</code>, all globally
	 *            registered tab close listeners are returned.
	 * @return All listeners registered on tab closing of the specified tabbed
	 *         pane.
	 */
	public static Set<BaseTabCloseListener> getAllTabCloseListeners(
			JTabbedPane tabbedPane) {
		return TabCloseListenerManager.getInstance().getListeners(tabbedPane);
	}

	/**
	 * Registers a new listener on locale change.
	 * 
	 * @param localeListener
	 *            New listener on locale change.
	 */
	public static void registerLocaleChangeListener(
			LocaleChangeListener localeListener) {
		SubstanceLookAndFeel.localeChangeListeners.add(localeListener);
	}

	/**
	 * Unregisters a listener on locale change.
	 * 
	 * @param localeListener
	 *            The listener to unregister.
	 */
	public static void unregisterLocaleChangeListener(
			LocaleChangeListener localeListener) {
		SubstanceLookAndFeel.localeChangeListeners.remove(localeListener);
	}

	/**
	 * Returns all listeners registered on locale change.
	 * 
	 * @return All listeners registered on locale change.
	 */
	public static Set<LocaleChangeListener> getLocaleListeners() {
		return Collections
				.unmodifiableSet(SubstanceLookAndFeel.localeChangeListeners);
	}

	/**
	 * Returns indication whether extra UI elements (such as menu items in
	 * system menu or menu search panel) should be shown.
	 * 
	 * @return <code>true</code> if extra UI elements (such as menu items in
	 *         system menu or menu search panel) should be shown,
	 *         <code>false</code> otherwise.
	 */
	public static boolean toShowExtraElements() {
		if (!SubstanceLookAndFeel.toShowExtraElements)
			return false;
		return !Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.NO_EXTRA_ELEMENTS));
	}

	/**
	 * Returns indication whether inverted themes should be enabled.
	 * 
	 * @return <code>true</code> if inverted themes should be enabled,
	 *         <code>false</code> otherwise.
	 */
	public static boolean toEnableInvertedThemes() {
		if (SubstanceLookAndFeel.toEnableInvertedThemes)
			return true;
		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.ENABLE_INVERTED_THEMES));
	}

	/**
	 * Returns indication whether negated themes should be enabled.
	 * 
	 * @return <code>true</code> if negated themes should be enabled,
	 *         <code>false</code> otherwise.
	 */
	public static boolean toEnableNegatedThemes() {
		if (SubstanceLookAndFeel.toEnableNegatedThemes)
			return true;
		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.ENABLE_NEGATED_THEMES));
	}

	/**
	 * Returns indication whether the watermark should "bleed" through list,
	 * table and tree renderers (where possible).
	 * 
	 * @return <code>true</code> if the watermark should "bleed" through list,
	 *         table and tree renderers (where possible), <code>false</code>
	 *         otherwise.
	 */
	public static boolean toBleedWatermark() {
		if (SubstanceLookAndFeel.toBleedWatermark)
			return true;
		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.WATERMARK_TO_BLEED));
	}

	/**
	 * Returns indication whether the application is in debug UI mode.
	 * 
	 * @return <code>true</code> if the application is in debug UI mode,
	 *         <code>false</code> otherwise.
	 * @since version 3.1
	 */
	public static boolean isDebugUiMode() {
		if (SubstanceLookAndFeel.isDebugUiMode)
			return true;
		return Boolean.TRUE.equals(UIManager
				.get(SubstanceLookAndFeel.DEBUG_UI_MODE));
	}

	/**
	 * Returns the parameter reader.
	 * 
	 * @return Parameter reader.
	 */
	public static ParamReader getParamReader() {
		return SubstanceLookAndFeel.paramReader;
	}

	/**
	 * Hides menu search panels on all open frames.
	 * 
	 * @see #showMenuSearchPanels()
	 */
	public static void hideMenuSearchPanels() {
		SubstanceMenuBarUI.hideSearchPanels(true);
	}

	/**
	 * Shows menu search panels on all open frames. For each panel, the menu
	 * search panel will be shown only if the corresponding settings allow it.
	 * 
	 * @see #hideMenuSearchPanels()
	 */
	public static void showMenuSearchPanels() {
		SubstanceMenuBarUI.showSearchPanels();
	}

	/**
	 * Requests that animations be ignored on all instances of the specified
	 * class.
	 * 
	 * @param componentClazz
	 *            Component class.
	 */
	public static void ignoreAnimationsOnClass(Class<?> componentClazz) {
		SubstanceLookAndFeel.ignoreAnimationsSet.add(componentClazz.getName());
	}

	/**
	 * Requests that animations be ignored on all instances of the specified
	 * class.
	 * 
	 * @param componentClazzName
	 *            Component class name.
	 */
	public static void ignoreAnimationsOnClass(String componentClazzName) {
		SubstanceLookAndFeel.ignoreAnimationsSet.add(componentClazzName);
	}

	/**
	 * Checks whether animations should be ignored on instances of the specified
	 * class.
	 * 
	 * @param componentClazz
	 *            Component class.
	 * @return <code>true</code> if animations should be ignored on instances
	 *         of the specified class, <code>false</code> otherwise.
	 */
	public static boolean toIgnoreAnimation(Class<?> componentClazz) {
		return SubstanceLookAndFeel.ignoreAnimationsSet.contains(componentClazz
				.getName());
	}

	/**
	 * Makes the heap status panels appear permanently on the specified title
	 * pane and removes the corresponding check box menu items from the system
	 * menu. Note that heap status panel can be shown only on custom decorated
	 * frames / dialogs. Use
	 * {@link JFrame#setDefaultLookAndFeelDecorated(boolean)} with
	 * <code>true</code> or -Dsubstancelaf.useDecorations VM flag to run your
	 * app in custom decoration mode.
	 * 
	 * @param pane
	 *            The root pane.
	 * @throws IllegalArgumentException
	 *             if the heap status panel functionality is not available (was
	 *             not enabled).
	 * @see #permanentlyHideHeapStatusPanel(JRootPane)
	 * @see #HEAP_STATUS_PANEL
	 */
	public static void permanentlyShowHeapStatusPanel(JRootPane pane) {
		if (!SubstanceTitlePane.getCanHaveHeapStatusPanel())
			throw new IllegalArgumentException(
					"This function can only be called when heap status panel functionality is enabled");
		((SubstanceRootPaneUI) pane.getUI())
				.setHeapStatusPanePermanentVisibility(true);
	}

	/**
	 * Makes the heap status panels disappear permanently on the specified title
	 * pane and removes the corresponding check box menu items from the system
	 * menu.
	 * 
	 * @param pane
	 *            The root pane.
	 * @throws IllegalArgumentException
	 *             if the heap status panel functionality is not available (was
	 *             not enabled).
	 * @see #permanentlyShowHeapStatusPanel(JRootPane)
	 * @see #HEAP_STATUS_PANEL
	 */
	public static void permanentlyHideHeapStatusPanel(JRootPane pane) {
		if (!SubstanceTitlePane.getCanHaveHeapStatusPanel())
			throw new IllegalArgumentException(
					"This function can only be called when heap status panel functionality is enabled");
		((SubstanceRootPaneUI) pane.getUI())
				.setHeapStatusPanePermanentVisibility(false);
	}

	/**
	 * Sets the global kind for image-based watermarks.
	 * 
	 * @param kind
	 *            New value for the global kind for image-based watermarks.
	 * @see #getImageWatermarkKind()
	 * @see #getImageWatermarkOpacity()
	 * @see #setImageWatermarkOpacity(float)
	 * @see #WATERMARK_IMAGE_KIND
	 * @see #WATERMARK_IMAGE_OPACITY
	 * @see #WATERMARK_IMAGE_PROPERTY
	 */
	public static void setImageWatermarkKind(ImageWatermarkKind kind) {
		SubstanceImageWatermark.setKind(kind);
		if (SubstanceLookAndFeel.currentWatermark instanceof SubstanceImageWatermark)
			SubstanceLookAndFeel.currentWatermark.updateWatermarkImage();
	}

	/**
	 * Returns the global kind for image-based watermarks.
	 * 
	 * @return The global kind for image-based watermarks.
	 * @see #getImageWatermarkOpacity()
	 * @see #setImageWatermarkKind(ImageWatermarkKind)
	 * @see #setImageWatermarkOpacity(float)
	 * @see #WATERMARK_IMAGE_KIND
	 * @see #WATERMARK_IMAGE_OPACITY
	 * @see #WATERMARK_IMAGE_PROPERTY
	 */
	public static ImageWatermarkKind getImageWatermarkKind() {
		return SubstanceImageWatermark.getKind();
	}

	/**
	 * Sets the global opacity for image-based watermarks.
	 * 
	 * @param opacity
	 *            New value for the global opacity for image-based watermarks.
	 *            Should be in 0.0-1.0 range.
	 * @throws IllegalArgumentException
	 *             If the value is not in 0.0-1.0 range.
	 * @see #getImageWatermarkKind()
	 * @see #getImageWatermarkOpacity()
	 * @see #setImageWatermarkKind(ImageWatermarkKind)
	 * @see #WATERMARK_IMAGE_KIND
	 * @see #WATERMARK_IMAGE_OPACITY
	 * @see #WATERMARK_IMAGE_PROPERTY
	 */
	public static void setImageWatermarkOpacity(float opacity) {
		SubstanceImageWatermark.setOpacity(opacity);
		if (SubstanceLookAndFeel.currentWatermark instanceof SubstanceImageWatermark)
			SubstanceLookAndFeel.currentWatermark.updateWatermarkImage();
	}

	/**
	 * Returns the global opacity for image-based watermarks.
	 * 
	 * @return The global opacity for image-based watermarks. The value is in
	 *         0.0-1.0 range.
	 * @see #getImageWatermarkKind()
	 * @see #setImageWatermarkKind(ImageWatermarkKind)
	 * @see #setImageWatermarkOpacity(float)
	 * @see #WATERMARK_IMAGE_KIND
	 * @see #WATERMARK_IMAGE_OPACITY
	 * @see #WATERMARK_IMAGE_PROPERTY
	 */
	public static float getImageWatermarkOpacity() {
		return SubstanceImageWatermark.getOpacity();
	}

	/**
	 * Globally registers a new <code>mixed</code> theme.
	 * 
	 * @param mixedTheme
	 *            A <code>mixed</code> theme to register.
	 * @see #addMixedTheme(SubstanceTheme...)
	 * @see #addMixedThemeBy(SubstanceTheme)
	 * @see #hasMixedThemes()
	 * @see #removeMixedTheme(SubstanceMixTheme)
	 * @see #removeMixedThemeBy(SubstanceTheme)
	 */
	public static void addMixedTheme(SubstanceMixTheme mixedTheme) {
		if (SubstanceLookAndFeel.mixedThemes.containsKey(mixedTheme
				.getDisplayName()))
			return;
		SubstanceLookAndFeel.mixedThemes.put(mixedTheme.getDisplayName(),
				mixedTheme);
	}

	/**
	 * Globally registers a new <code>mixed</code> theme.
	 * 
	 * @param themes
	 *            The base themes for the new <code>mixed</code> theme.
	 * @see #addMixedTheme(SubstanceMixTheme)
	 * @see #addMixedThemeBy(SubstanceTheme)
	 * @see #hasMixedThemes()
	 * @see #removeMixedTheme(SubstanceMixTheme)
	 * @see #removeMixedThemeBy(SubstanceTheme)
	 */
	public static void addMixedTheme(SubstanceTheme... themes) {
		SubstanceLookAndFeel.addMixedTheme(new SubstanceMixTheme(themes));
	}

	/**
	 * Globally registers a collection of <code>mixed</code> themes such as:
	 * <ul>
	 * <li>The specified theme is the main theme for a new <code>mixed</code>
	 * theme.</li>
	 * <li>The secondary theme is of the same kind.</li>
	 * </ul>
	 * 
	 * @param mainTheme
	 *            The main theme for the new <code>mixed</code> themes.
	 * @see #addMixedTheme(SubstanceMixTheme)
	 * @see #addMixedTheme(SubstanceTheme...)
	 * @see #hasMixedThemes()
	 * @see #removeMixedTheme(SubstanceMixTheme)
	 * @see #removeMixedThemeBy(SubstanceTheme)
	 */
	public static void addMixedThemeBy(SubstanceTheme mainTheme) {
		ThemeKind mainThemeKind = mainTheme.getKind();
		if ((mainThemeKind == ThemeKind.BRIGHT)
				|| (mainThemeKind == ThemeKind.COLD)) {
			for (ThemeInfo themeInfo : SubstanceLookAndFeel.getAllThemes()
					.values()) {
				ThemeKind themeKind = themeInfo.getThemeKind();
				if (themeKind != mainTheme.getKind())
					continue;
				if (themeInfo.getDisplayName().equals(
						mainTheme.getDisplayName()))
					continue;

				try {
					Class<?> themeClass = Class.forName(themeInfo
							.getClassName());
					if (themeClass == null) {
						continue;
					}
					Object obj = themeClass.newInstance();
					if (obj == null) {
						continue;
					}
					if (!(obj instanceof SubstanceTheme)) {
						continue;
					}
					SubstanceLookAndFeel.addMixedTheme(new SubstanceMixTheme(
							mainTheme, (SubstanceTheme) obj));
				} catch (Exception exc) {
				}
			}
		}
	}

	/**
	 * Globally unregisters the specified <code>mixed</code> theme.
	 * 
	 * @param mixedTheme
	 *            The <code>mixed</code> theme to unregister.
	 * @see #addMixedTheme(SubstanceMixTheme)
	 * @see #addMixedTheme(SubstanceTheme...)
	 * @see #addMixedThemeBy(SubstanceTheme)
	 * @see #hasMixedThemes()
	 * @see #removeMixedThemeBy(SubstanceTheme)
	 */
	public static void removeMixedTheme(SubstanceMixTheme mixedTheme) {
		SubstanceLookAndFeel.mixedThemes.remove(mixedTheme.getDisplayName());
	}

	/**
	 * Globally unregisters a collection of <code>mixed</code> themes such
	 * that the specified theme is the main theme.
	 * 
	 * @param mainTheme
	 *            The main theme for the <code>mixed</code> themes to be
	 *            unregistered.
	 * @see #addMixedTheme(SubstanceMixTheme)
	 * @see #addMixedTheme(SubstanceTheme...)
	 * @see #addMixedThemeBy(SubstanceTheme)
	 * @see #hasMixedThemes()
	 * @see #removeMixedTheme(SubstanceMixTheme)
	 */
	public static void removeMixedThemeBy(SubstanceTheme mainTheme) {
		for (Iterator<Map.Entry<String, SubstanceMixTheme>> it = SubstanceLookAndFeel.mixedThemes
				.entrySet().iterator(); it.hasNext();) {
			Map.Entry<String, SubstanceMixTheme> entry = it.next();
			if (entry.getValue().getOriginalThemes()[0].getDisplayName()
					.equals(mainTheme.getDisplayName()))
				it.remove();
		}
	}

	/**
	 * Checks whether there are any <code>mixed</code> themes registered
	 * globally.
	 * 
	 * @return <code>true</code> if there is at least one <code>mixed</code>
	 *         theme registered globally, <code>false</code> otherwise.
	 * @see #addMixedTheme(SubstanceMixTheme)
	 * @see #addMixedTheme(SubstanceTheme...)
	 * @see #addMixedThemeBy(SubstanceTheme)
	 * @see #removeMixedTheme(SubstanceMixTheme)
	 * @see #removeMixedThemeBy(SubstanceTheme)
	 */
	public static boolean hasMixedThemes() {
		return SubstanceLookAndFeel.mixedThemes.size() > 0;
	}

	/**
	 * Checks whether the <code>JOptionPane</code>s created with predefined
	 * message types should use constant themes for the icons.
	 * 
	 * @return <code>true</code> if the <code>JOptionPane</code>s created
	 *         with predefined message types should use constant themes for the
	 *         icons, <code>false</code> otherwise.
	 * @see #setToUseConstantThemesOnDialogs(boolean)
	 */
	public static boolean isToUseConstantThemesOnDialogs() {
		return SubstanceLookAndFeel.toUseConstantThemesOnDialogs;
	}

	/**
	 * Sets the new setting for the icons of the <code>JOptionPane</code>s
	 * created with predefined message types.
	 * 
	 * @param toUseConstantThemesOnDialogs
	 *            if <code>true</code>, the <code>JOptionPane</code>s
	 *            created with predefined message types should use constant
	 *            themes for the icons.
	 * @see #isToUseConstantThemesOnDialogs()
	 */
	public static void setToUseConstantThemesOnDialogs(
			boolean toUseConstantThemesOnDialogs) {
		SubstanceLookAndFeel.toUseConstantThemesOnDialogs = toUseConstantThemesOnDialogs;
	}

	/**
	 * Sets the specified skin.
	 * 
	 * @param skin
	 *            Skin to set.
	 * @return <code>true</code> if the specified skin has been set
	 *         successfully, <code>false</code> otherwise.
	 * @since version 3.1
	 * @see #setSkin(SubstanceSkin)
	 * @see #registerSkinChangeListener(SkinChangeListener)
	 * @see #unregisterSkinChangeListener(SkinChangeListener)
	 */
	public static boolean setSkin(SubstanceSkin skin) {
		try {
			if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel)) {
				UIManager.setLookAndFeel(new SubstanceLookAndFeel());
			}
			if (!skin.set())
				return false;
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					for (Frame frame : Frame.getFrames()) {
						SwingUtilities.updateComponentTreeUI(frame);
						for (Window window : frame.getOwnedWindows()) {
							SwingUtilities.updateComponentTreeUI(window);
						}
					}

					for (SkinChangeListener skinChangeListener : SubstanceLookAndFeel.skinChangeListeners)
						skinChangeListener.skinChanged();
				}
			});
			return true;
		} catch (NoClassDefFoundError ncdfe) {
			// this may happen when a skin references some class
			// that can't be found in the classpath.
			return false;
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * Sets the specified skin.
	 * 
	 * @param skinClassName
	 *            Skin to set.
	 * @return <code>true</code> if the specified skin has been set
	 *         successfully, <code>false</code> otherwise.
	 * @since version 3.1
	 * @see #setSkin(SubstanceSkin)
	 * @see #registerSkinChangeListener(SkinChangeListener)
	 * @see #unregisterSkinChangeListener(SkinChangeListener)
	 */
	public static boolean setSkin(String skinClassName) {
		try {
			Class<?> skinClass = Class.forName(skinClassName);
			if (skinClass == null) {
				return false;
			}
			Object obj = skinClass.newInstance();
			if (obj == null) {
				return false;
			}
			if (!(obj instanceof SubstanceSkin)) {
				return false;
			}
			return SubstanceLookAndFeel.setSkin((SubstanceSkin) obj);
		} catch (Exception exc) {
			return false;
		}
	}

	/**
	 * Returns all available skins.
	 * 
	 * @return All available skins. Key - skin display name, value - skin
	 *         information.
	 */
	public static Map<String, SkinInfo> getAllSkins() {
		Map<String, SkinInfo> result = new TreeMap<String, SkinInfo>();
		for (Object skinPlugin : SubstanceLookAndFeel.skinPlugins
				.getAvailablePlugins(true)) {
			for (SkinInfo skinInfo : ((SubstanceSkinPlugin) skinPlugin)
					.getSkins()) {
				result.put(skinInfo.getDisplayName(), skinInfo);
			}
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.LookAndFeel#getSupportsWindowDecorations()
	 */
	@Override
	public boolean getSupportsWindowDecorations() {
		return true;
	}

	/**
	 * Sets the class loader for {@link #LABEL_BUNDLE}.
	 * 
	 * @param labelBundleClassLoader
	 *            Class loader for {@link #LABEL_BUNDLE}.
	 * @since version 3.1
	 */
	public static void setLabelBundleClassLoader(
			ClassLoader labelBundleClassLoader) {
		SubstanceLookAndFeel.labelBundleClassLoader = labelBundleClassLoader;
		LafWidgetRepository.setLabelBundleClassLoader(labelBundleClassLoader);
	}

	/**
	 * Returns the title pane of the specified top-level window.
	 * 
	 * @param window
	 *            Top-level window.
	 * @return If the parameter is either {@link JFrame} or {@link JDialog} and
	 *         has custom decorations, the result is the title pane,
	 *         <code>null</code> otherwise.
	 * @since version 3.1
	 */
	public static JComponent getTitlePaneComponent(Window window) {
		JRootPane rootPane = null;
		if (window instanceof JFrame) {
			JFrame f = (JFrame) window;
			rootPane = f.getRootPane();
		}
		if (window instanceof JDialog) {
			JDialog d = (JDialog) window;
			rootPane = d.getRootPane();
		}
		if (rootPane != null) {
			SubstanceRootPaneUI ui = (SubstanceRootPaneUI) rootPane.getUI();
			return ui.getTitlePane();
		}
		return null;
	}

	/**
	 * Specifies that a control of some class should use the specified
	 * background composite when it (the control) doesn't define the
	 * {@link #BACKGROUND_COMPOSITE} client property.
	 * 
	 * @param clazz
	 *            Component class.
	 * @param composite
	 *            Default background composite for controls of the specified
	 *            class.
	 */
	public static synchronized void setBackgroundComposite(Class<?> clazz,
			ControlBackgroundComposite composite) {
		backgroundComposites.put(clazz, composite);
	}

	/**
	 * Returns the default background composite for the specified component.
	 * 
	 * @param component
	 *            Component.
	 * @return The default background composite for the specified component.
	 */
	public static synchronized ControlBackgroundComposite getBackgroundComposite(
			Component component) {
		ControlBackgroundComposite result = backgroundComposites.get(component
				.getClass());
		if (result == null)
			result = new DefaultControlBackgroundComposite();
		return result;
	}

	/**
	 * Sets the decoration type of the specified component and all its children.
	 * 
	 * @param comp
	 *            Component.
	 * @param type
	 *            Decoration type of the component and all its children.
	 */
	public static void setDecorationType(JComponent comp,
			DecorationAreaType type) {
		SubstanceDecorationUtilities.setDecorationType(comp, type);
	}
}
