/*  -*- c -*-  */
/* -------------------------------------------------------------------- *
**  copyright (c) 1995 ipvr stuttgart and thomas harrer
** -------------------------------------------------------------------- *
**
**  libhelp
**
**  a comprehensive hypertext help system for OSF/Motif(tm) applications. 
**  based on libhtmlw from NCSA Mosaic(tm) version 2.4
**
**  written by thomas harrer
**  e-mail: Thomas.Harrer@rus.uni-stuttgart.de
**  
** -------------------------------------------------------------------- *
*h  $Id: bcache.c,v 1.7 1995/06/28 12:59:30 thomas Exp $
** -------------------------------------------------------------------- *
**
*h  module:		bcache.c
**
**  contents:		buffer cache implementation
**
**  interface:		functions
**			* bcache_find
**			* bcache_insert
**			* bcache_free
**			* bcache_current
**			* bcache_flush
**
** -------------------------------------------------------------------- *
**  license and copying issues:
**
**  this software is free; you can redistribute it and/or modify it 
**  under terms similar to the gnu general public license (version 1 
**  or any later version published by the free software foundation). 
**  see the file Licence for more details.
**
**  this program is distributed in the hope that it will be useful,
**  but without any warranty; without even the implied warranty of
**  merchantability or fitness for a particular purpose.  
** -------------------------------------------------------------------- */
/* -------------------------------------------------------------------- *
*g  include files
** -------------------------------------------------------------------- */
#include "helpp.h"
#include "buffer.h"
#include "bcache.h"
#include "util.h"
#include "path.h"

/* -------------------------------------------------------------------- *
*g  module identification
** -------------------------------------------------------------------- */
#ifdef RCSID
static char rcsid [] =
    "$Id: bcache.c,v 1.7 1995/06/28 12:59:30 thomas Exp $";
#endif /* RCSID */

/* -------------------------------------------------------------------- *
*g  global type:	bcache_t
**			buffer cache.
** -------------------------------------------------------------------- */
typedef struct bcache_s
{
    char*		filename;      /* file from reference		*/
    char*		path;	       /* the preferred path.           */
    buffer_t*		buf;	       /* the buffer.			*/
    struct bcache_s*	next;
    
} bcache_t;

/* -------------------------------------------------------------------- *
*p  private prototypes
** -------------------------------------------------------------------- */
static void bcache_free (bcache_t*);

/* -------------------------------------------------------------------- *
*g  global variables
** -------------------------------------------------------------------- */
static bcache_t* the_bcache = NULL;    
static int	bcache_len = 0;	      	

/* -------------------------------------------------------------------- *
*p  procedure-name:	bcache_current
**
**  purpose:		returns the current path and filename or NULL.
**			the returned buffer must be freed by the caller.
** -------------------------------------------------------------------- *
**  returns:	        if we have a file and a pathname, we concatenate
**			them. if we have either a file or a path, we return 
**			it. else we return NULL.
** -------------------------------------------------------------------- */
char*
bcache_current (void)
{
    char* current_path = NULL;	/* scope: return value. */

    execute ("bcache_current");
    
    /* if we have the bcache we return the value, else NULL.  */
    if (the_bcache) {

	char* current = the_bcache->filename;

	if (current) {
	    /* absolute names don't have path's added.  */
	    if ((current[0] == '.') || (current[0] == '/')) {

		checked_strdup (current_path, current);

	    } else  {

		/* relative names are appended to the path.  */
		char* path = the_bcache->path;
	    
		if (path) {
		    int len = c_strlen (current) + c_strlen (path) + 10;
		    checked_malloc (current_path, len, char);
		    c_strcpy (current_path, path);
		    c_strcat (current_path, current);
		} else  {
		    checked_strdup (current_path, current);
		}
	    }

	} else  { 
	    
	    if (the_bcache->path) {
		checked_strdup (current_path, the_bcache->path);
	    }
	}
    }
    
    return current_path;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bcache_find
**
**  purpose:		returns a cached buffer or NULL
** -------------------------------------------------------------------- *
**  args:		filename
**  return type:	buffer_t*	
**  side effect:	the current path is added to the help path.
**  error handling.:	returns NULL.
** -------------------------------------------------------------------- */
buffer_t*	
bcache_find (/* i  */ char* ref)
{
    /* local data  */
    buffer_t* return_buffer = NULL; /* scope: return value */
    char* current_path = NULL;	/* pointer to the current path. */
    
    execute ("bcache_find");
    
    if (the_bcache) {

	if (ref) {

	    /* local data  */
	    /* scope: local  */
	    char* filename = get_file_and_strip_path (ref);
	    /* scope: local  */
	    char* pathname = get_path (ref);
	    
	    bcache_trace ("ref: %s", ref);
	    bcache_trace ("\tfile: %s", filename);
	    bcache_trace ("\tpath: %s", pathname);

	    if ((!filename) || (filename[0] != '\0')) {

		bcache_t*	bc = the_bcache;
		bcache_t*	pre = NULL;
		
		/*
		 *  we traverse the cache and compare file and
		 *  path names.
		 */
		while (bc) {

		    /*
		     *  logic: if we have no filename we just compare 
		     *  with the path. else we compare file and path.
		     */
		    
		    bcache_trace ("* filename: %s\n", filename);
		    bcache_trace ("* bc->filename: %s\n", bc->filename);
		    bcache_trace ("* pathname: %s\n", pathname);
		    bcache_trace ("* bc->path: %s\n", bc->path);
		    
		    if ((!filename) && (!bc->filename)) {
			if (((pathname) && (bc->path))
			    && (0 == c_strcmp (bc->path, pathname))) {
			    break;
			}			
		    }
		    if (((filename) && (bc->filename))
			&& (0 == c_strcmp (bc->filename, filename))) {
			if (((pathname) && (bc->path))
			    && (0 == c_strcmp (bc->path, pathname))) {
			    break;
			} else  {
			    break;
			}
		    }
		    pre = bc;
		    bc = bc->next;
		}

		/* bc is NULL if nothing was found.  */
		if (bc) {

		    return_buffer = bc->buf; /* the desired buffer.  */
		    current_path = bc->path;
		    
		    bcache_trace ("found %s", bc->filename);
		    bcache_trace ("path %s\n", bc->path);

		    /* we move the element to the first position.  */
		    if (pre) {

			pre->next = bc->next;
			bc->next = the_bcache;
			the_bcache = bc;

			bcache_trace ("moved %s on top\n", bc->filename);
		    }
		}

	    } else {

		/* reference to the current buffer.  */
		if (ref[0] == '#') {
		    return_buffer = the_bcache->buf;
		    current_path = the_bcache->path;
		    bcache_trace ("current_doc: %s ", the_bcache->filename);
		    bcache_trace ("path: %s\n", the_bcache->path);
		}
	    }

	    /* these strings are allocated.  */
	    if (pathname) checked_free (pathname);
	    if (filename) checked_free (filename);
	}
    }
    path_add_current (current_path);
    return return_buffer;
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bcache_insert
**
**  purpose:		inserts a buffer into the cache.
** -------------------------------------------------------------------- *
**  args:		current_path is a pointer to an allocated string.
**			bcache_free is responsible to free it.
**  side effect:	the current path is added to the help path.
** -------------------------------------------------------------------- */
void
bcache_insert (/* i  */ char* 		ref,
	       /* i  */ char*		current_path,
	       /* i  */ buffer_t* 	buf)
{
    execute ("bcache_insert");

    /* if the cache is full, we remove the last element.  */
    if (bcache_len >= BCACHE_SIZE) {
	
	/* we unline the last element.  */
	bcache_t* bc = the_bcache;      /* != NULL here. */
	bcache_t* pre = NULL;
	
	while (bc->next) {
	    pre = bc;
	    bc = bc->next;
	}
	
	if (pre) { 
	    pre->next = NULL;
	} else {
	    the_bcache = NULL;	       /* possible ? */
	}
	bcache_trace ("freed %s\n", bc->filename);
	bcache_free (bc);
	bcache_len--;
    } 

    /* begin  */ {

	/* local data  */
	bcache_t* 	new;
	checked_calloc (new, 1, bcache_t); /* scoppe -> bcache_free */

	if (ref) {
	    /* scope -> bcache_free  */
	    new->filename = get_file_and_strip_path (ref);
	} else  {
	    new->filename = NULL;
	}
	
	new->buf = buf;		/* is valid and not NULL */
	new->path = current_path; /* was allocated by read_html_file */
	new->next = the_bcache;
	the_bcache = new;
	bcache_len++;	
	bcache_trace ("added %s ", new->filename);
	bcache_trace ("path %s ", new->path);
	bcache_trace ("(cache = %d)\n", bcache_len);

	/* side effect: we add the current path to the help path.  */
	/* but we make a copy.  */
	path_add_current (current_path);
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bcache_free
**
**  purpose:		free's the requested cache element.
** -------------------------------------------------------------------- */
static void
bcache_free (bcache_t* bc)
{
    execute ("bcache_free");
    
    if (bc) {
	if (bc->filename) checked_free (bc->filename);
	if (bc->path) checked_free (bc->path);
	if (bc->buf) bf_free (bc->buf);
	checked_free (bc);
    } else  {
	fprintf (stderr, "Error: called bcache_free with no arg\n");
    }
}

/* -------------------------------------------------------------------- *
*p  procedure-name:	bcache_flush
**
**  purpose:		releases all cached memory.
** -------------------------------------------------------------------- */
void
bcache_flush (void)
{
    bcache_t* bc = the_bcache;
    
    execute ("bcache_flush");

    while (bc) {

	bcache_t* bcn = bc->next;
	bcache_free (bc);
	bcache_len--;
	bc = bcn;
    }
    if (bcache_len != 0) {
	fprintf (stderr, "inconsistency in buffer cache (flush)\n");
	exit (EXIT_FAILURE);
    }
    the_bcache = NULL;
}

/* -------------------------------------------------------------------- *
*l  emacs:
**  local variables:
**  mode:		c
**  outline-regexp:	"\*[HGPLT]"
**  comment-column:	32
**  eval:		(outline-minor-mode t)
**  end:
** -------------------------------------------------------------------- */
