// ----------------------------------------------------------------------------
//
//  Copyright (C) 2013-2014 Fons Adriaensen <fons@linuxaudio.org>
//    
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// ----------------------------------------------------------------------------


#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "netdata.h"
#include "jackrx.h"
#include "netrx.h"


Jackrx::Jackrx (const char *jname, const char*jserv, int nchan, const int *clist) :
    _client (0),
    _nchan (nchan),     
    _state (INIT),
    _freew (false)
{
    init (jname, jserv, clist);
}


Jackrx::~Jackrx (void)
{
    fini ();
}


void Jackrx::init (const char *jname, const char *jserv, const int *clist)
{
    int                 i, opts, spol, flags;
    char                s [16];
    jack_status_t       stat;
    struct sched_param  spar;

    opts = JackNoStartServer;
    if (jserv) opts |= JackServerName;
    _client = jack_client_open (jname, (jack_options_t) opts, &stat, jserv);
    if (_client == 0)
    {
        fprintf (stderr, "Can't connect to Jack, is the server running ?\n");
        exit (1);
    }
    jack_set_process_callback (_client, jack_static_process, (void *) this);
    jack_set_latency_callback (_client, jack_static_latency, (void *) this);
    jack_set_freewheel_callback (_client, jack_static_freewheel, (void *) this);
    jack_set_buffer_size_callback (_client, jack_static_buffsize, (void *) this);
    jack_on_shutdown (_client, jack_static_shutdown, (void *) this);

    _bsize = 0;
    _fsamp = 0;
    if (jack_activate (_client))
    {
        fprintf(stderr, "Can't activate Jack");
        exit (1);
    }
    _jname = jack_get_client_name (_client);
    _bsize = jack_get_buffer_size (_client);
    _fsamp = jack_get_sample_rate (_client);

    flags = JackPortIsTerminal | JackPortIsPhysical;
    if (_nchan > Netdata::MAXCHAN) _nchan = Netdata::MAXCHAN;
    for (i = 0; i < _nchan; i++)
    {
        sprintf (s, "out_%d", clist [i] + 1);
        _ports [i] = jack_port_register (_client, s, JACK_DEFAULT_AUDIO_TYPE,
                                         flags | JackPortIsOutput, 0);
    }
    pthread_getschedparam (jack_client_thread_id (_client), &spol, &spar);
    _rprio = spar.sched_priority -  sched_get_priority_max (spol);
    _buff = new float [_bsize * _nchan];
    _state = IDLE;
}


void Jackrx::fini (void)
{
    if (_client)
    {
        jack_deactivate (_client);
        jack_client_close (_client);
    }
    delete[] _buff;
}


void Jackrx::jack_static_shutdown (void *arg)
{
    ((Jackrx *) arg)->sendinfo (FATAL, 0, 0, 0);
}


int Jackrx::jack_static_buffsize (jack_nframes_t bsize, void *arg)
{
    ((Jackrx *) arg)->jack_buffsize (bsize);
    return 0;
}


void Jackrx::jack_static_freewheel (int state, void *arg)
{
    ((Jackrx *) arg)->jack_freewheel (state);
}


void Jackrx::jack_static_latency (jack_latency_callback_mode_t jlcm, void *arg)
{
}


int Jackrx::jack_static_process (jack_nframes_t nframes, void *arg)
{
    return ((Jackrx *) arg)->jack_process (nframes);
}


void Jackrx::start (Lfq_audio      *audioq,
                    Lfq_int32      *commq, 
                    Lfq_timedata   *timeq,
                    Lfq_infodata   *infoq,
                    double         ratio,
                    int            delay,
                    int            rqual)
{
    _audioq = audioq;
    _commq = commq;
    _timeq = timeq;
    _infoq = infoq;
    _quant = ldexp (1e-6f, 28);
    _ratio = ratio;
    _rcorr = 1.0;
    _resamp.setup (_ratio, _nchan, rqual);
    _resamp.set_rrfilt (100);
    _delay = delay + _resamp.inpsize () / 2.0;
    _ppsec = (_fsamp + _bsize / 2) / _bsize;
    _tnext = 0;
    _limit = (int)(_fsamp / _ratio);
    initwait (_ppsec / 2);
}


void Jackrx::initwait (int nwait)
{
    _count = -nwait;
    _commq->wr_int32 (Netrx::WAIT);
    _state = WAIT;
    if (nwait > _ppsec) sendinfo (_state, 0, 0, 0);
}


void Jackrx::initsync (void)
{
//  Reset all lock-free queues.
    _commq->reset ();
    _timeq->reset ();
    _audioq->reset ();
    // Reset and prefill the resampler.
    _resamp.reset ();
    _resamp.inp_count = _resamp.inpsize () / 2 - 1;
    _resamp.out_count = 10000;
    _resamp.process ();
    // Initiliase state variables.
    _t_a0 = _t_a1 = 0;
    _k_a0 = _k_a1 = 0;
    _k_j0 = 0;
    // Initialise loop filter state.
    _z1 = _z2 = _z3 = 0;
    // Activate the netrx thread,
    _commq->wr_int32 (Netrx::PROC);
    _state = SYNC0;
    sendinfo (_state, 0, 0, 0);
}


void Jackrx::setloop (double bw)
{
    double w;

    // Set the loop bandwidth to bw Hz.
    w = 6.28f * 20 * bw * _bsize / _fsamp;
    _w0 = 1.0 - exp (-w);
    w = 6.28f * bw * _ratio / _fsamp;
    _w1 = w * 1.6;
    _w2 = w * _bsize / 1.6;
}


void Jackrx::capture (int nframes)
{
    int    i, j, k1, k2;
    float  *p, *q;

    // Read from audio queue and resample.
    // The while loop takes care of wraparound.
    _resamp.out_count = _bsize;
    _resamp.out_data  = _buff;
    while (_resamp.out_count)
    {
	// Allow the audio queue to underrun, but
	// use zero valued samples in that case.
	// This will happen when the sender skips
	// some cycles and the receiver is not
	// configured for additional latency.
        k1 = _audioq->rd_avail ();
        k2 = _audioq->rd_linav ();
	if (k1 > 0)
	{
	    _resamp.inp_count = (k1 < k2) ? k1 : k2;
	    _resamp.inp_data  = _audioq->rd_datap ();
	}
	else
	{
	    _resamp.inp_count = 999999;
	    _resamp.inp_data = 0;
	}
	// Resample up to a full output buffer.
	k1 = _resamp.inp_count;
	_resamp.process ();
	k1 -= _resamp.inp_count;
	// Adjust audio queue and state by the
        // number of frames consumed.
	_audioq->rd_commit (k1);
	_k_j0 += k1;
    }
    // Deinterleave _buff to outputs.
    for (j = 0; j < _nchan; j++)
    {
	p = _buff + j;
	q = (float *)(jack_port_get_buffer (_ports [j], nframes));
	for (i = 0; i < _bsize; i++) q [i] = p [i * _nchan];
    }       
}


void Jackrx::silence (int nframes)
{
    int    i;
    float  *q;

    // Write silence to all jack ports.
    for (i = 0; i < _nchan; i++)
    {
        q = (float *)(jack_port_get_buffer (_ports [i], nframes));
        memset (q, 0, nframes * sizeof (float));
    }
}


void Jackrx::sendinfo (int state, double error, double ratio, int nfram)
{
    Infodata *I;

    if (_infoq->wr_avail ())
    {
	I = _infoq->wr_datap ();
	I->_state = state;
	I->_error = error;
	I->_ratio = ratio;
	I->_nfram = nfram;
	_infoq->wr_commit ();
    }
}


void Jackrx::jack_freewheel (int yesno)
{
    _freew = yesno ? true : false;
    if (_freew) initwait (_ppsec / 4);
}


void Jackrx::jack_buffsize (int bsize)
{
    if (_bsize == 0) _bsize = bsize;
    else if (_bsize != bsize) _state = Jackrx::FATAL;
}


int Jackrx::jack_process (int nframes)
{
    int             k, nskip;
    double          tj, err, d1, d2;
    jack_time_t     t0, t1;
    jack_nframes_t  ft;
    float           usecs;
    Timedata        *D;

    // Skip cylce if ports may not yet exist.
    if (_state < IDLE) return 0;

    // Buffer size change, no data, or other evil.
    if (_state >= TXEND)
    {
	sendinfo (_state, 0, 0, 0);
	_state = IDLE;
	return 0;
    }
    // Output silence if idle.
    if (_state < WAIT)
    {
	silence (nframes);
        return 0;
    }
    // Start synchronisation 1/2 second after entering
    // the WAIT state. This delay allows the netrx thread
    // to drop any accumulated packets and restart cleanly.
    // Disabled while freewheeling.
    if (_state == WAIT)
    {
	silence (nframes);
	if (_freew) return 0;
        if (++_count == 0) initsync ();
        else return 0;
    }

    // Get local timing info.
    jack_get_cycle_times (_client, &ft, &t0, &t1, &usecs);
    usecs = (float)(t0 - _tnext);
    nskip = (int)(_fsamp * usecs * 1e-6f / _ratio + 0.5f);
    _tnext = t1;
    tj = 1e-6 * (t0 & 0x0FFFFFFF);

    // Check if we have timing data from the netrx thread.
    // If the queue is full restart synchronisation.
    // This can happen e.g. on a jack engine timeout,
    // or when too many cycles have been skipped.
    k = _timeq->rd_avail ();
    if (k == _timeq->nelm ())
    {
        initwait (_ppsec / 2);
        return 0;
    }
    if (k)
    {
        // Else move interval end to start, and update the
        // interval end keeping only the most recent data.
        if (_state < SYNC2)
	{
            _state++;
            sendinfo (_state, 0, 0, 0);
	}
        _t_a0 = _t_a1;
        _k_a0 = _k_a1;
        while (_timeq->rd_avail ())
        {
            D = _timeq->rd_datap ();
	    // Check status of network thread.
	    switch (D->_state)
	    {
            case Netrx::FAIL:
                _state = FATAL;
                return 0;
            case Netrx::TERM:
                _state = TXEND;
                return 0;
            case Netrx::WAIT:
                // Restart synchronisation in case the netrx
                // thread signals a problem. This will happen
                // when the sender goes into freewheeling mode.
                initwait (_ppsec / 2);
                return 0;
            }
            _t_a1 = D->_timer;
            _k_a1 = D->_nsamp;
            _timeq->rd_commit ();
        }
    }

    err = 0;
    if (_state >= SYNC2)
    {
        // Compute the delay error.
        d1 = modtime (tj - _t_a0);
        d2 = modtime (_t_a1 - _t_a0);
	// This must be done as integer as both terms will overflow.
        k = _k_a0 - _k_j0;
        err = k + (_k_a1 - _k_a0) * d1 / d2  + _resamp.inpdist () - _delay ;

        k = (int)(floor (err + 0.5));
        if (_state == SYNC2)
        {
            // We have the first delay error value. Adjust both the state
            // variables and the audio queue by the same number of frames
            // to obtain the actually wanted delay, and start tracking.
            _audioq->rd_commit (k);
            _k_j0 += k;
            err -= k;
            setloop (0.5);
            _state = PROC1;
        }    
        else if (_state >= PROC1)
        {
	    // Check error conditions.
	    if (nskip)
	    {
                // Jack skipped some cycles. Adjust both the state
                // and the audio queue so we can just continue.
		// The loop will correct the remaining fraction
		// of a frame.
                _audioq->rd_commit (nskip);
                _k_j0 += nskip;
                err -= nskip;
            }
            if (fabs (err) >= 100)
            {
                // Something is really wrong, wait 10 seconds then restart.
//                initwait (10 * _ppsec);
//		return 0;
            }
        }
    }

    // Switch to lower bandwidth after 4 seconds.
    if ((_state == PROC1) && (++_count == 4 * _ppsec))
    {
        _state = PROC2;
        setloop (0.05);
    } 

    if (_state >= PROC1)
    {
        // Run loop filter and set resample ratio.
        _z1 += _w0 * (_w1 * err - _z1);
        _z2 += _w0 * (_z1 - _z2);
        _z3 += _w2 * _z2;
        _rcorr = 1 - _z2 - _z3;
	if (_rcorr > 1.05) _rcorr = 1.05;
	if (_rcorr < 0.95) _rcorr = 0.95;
        _resamp.set_rratio (_rcorr);

	// Resample and transfer between audio
        // queue and jack ports.
	capture (nframes);
	k = _audioq->rd_avail ();
	sendinfo (_state, err, _rcorr, k);
	if (k < -_limit) _state = TXEND;
    }
    else silence (nframes);

    return 0; 
}
