% $Id: CurryParser.lhs,v 1.40 2004/09/10 13:26:24 berrueta Exp $
%
% Copyright (c) 1999-2003, Wolfgang Lux
% Copyright (c) 2003-2004, Diego Berrueta
% See LICENSE for the full license.
%
\nwfilename{CurryParser.lhs}
\codesection{A parser for Curry}
The Curry parser is implemented using the (mostly) LL(1) parsing
combinators described in section~\ref{sec:ll-parsecomb}.
\begin{lstlisting}

> module CurryParser where
> import Ident
> import Position
> import Error
> import LexComb
> import LLParseComb
> import CurrySyntax hiding (infixOp)
> import CurryLexer
> import TypeExpr hiding (typeExprContext)
> import Kind

> instance Symbol Token where
>   isEOF (Token c _) = c == EOF

\end{lstlisting}
\codesubsection{Modules}
\begin{lstlisting}

> parseSource :: FilePath -> String -> Error Module
> parseSource = applyParser parseModule lexer

> parseModule :: Parser Token Module a
> parseModule = (moduleHeader `opt` Module mainMIdent Nothing) <*> decls

> moduleHeader :: Parser Token ([Decl] -> Module) a
> moduleHeader =
>   Module <$-> token KW_module
>          <*> (mIdent <?> "module name expected")
>          <*> (Just <$> exportSpec `opt` Nothing)
>          <*-> (token KW_where <?> "where expected")

> exportSpec :: Parser Token ExportSpec a
> exportSpec = Exporting <$> position <*> parens (export `sepBy` comma)

> export :: Parser Token Export a
> export = qFunctionName <**> (parens spec `opt` Export)
>      <|> ExportModule <$-> token KW_module <*> mIdent
>      <|> (token KW_class <-*> qIdent) <**> (parens typeClassSpec)
>   where spec = ExportTypeAll <$-> token DotDot
>            <|> flip ExportTypeWith <$> functionName `sepBy` comma
>         typeClassSpec = ExportTypeClassAll <$-> token DotDot
>            <|> flip ExportTypeClassWith <$> functionName `sepBy` comma

\end{lstlisting}
\codesubsection{Interfaces}
\begin{lstlisting}

> parseInterface :: FilePath -> String -> Error Interface
> parseInterface fn s = applyParser parseIface lexer fn s

> parseIface :: Parser Token Interface a
> parseIface =
>   Interface <$-> token Id_interface
>             <*> (mIdent <?> "module name expected")
>             <*-> (token KW_where <?> "where expected")
>             <*> braces (intfDecl `sepBy` semicolon)

\end{lstlisting}
\codesubsection{Goals}
\begin{lstlisting}

> parseGoal :: String -> Error Goal
> parseGoal s = applyParser goal lexer "" s

> goal :: Parser Token Goal a
> goal = Goal <$> position <*> expr <*> localDefs

\end{lstlisting}
\codesubsection{Declarations}
\begin{lstlisting}

> decls :: Parser Token [Decl] a
> decls = layout globalDecl

> localDefs :: Parser Token [Decl] a
> localDefs = token KW_where <-*> layout valueDecl `opt` []

> globalDecl :: Parser Token Decl a
> globalDecl = importDecl <|> infixDecl
>          <|> typeClassDecl <|> instanceDecl
>          <|> dataDecl <|> newtypeDecl <|> typeDecl
>          <|> functionDecl

> importDecl :: Parser Token Decl a
> importDecl =
>   flip . ImportDecl <$> position <*-> token KW_import 
>                     <*> (True <$-> token Id_qualified `opt` False)
>                     <*> mIdent
>                     <*> (Just <$-> token Id_as <*> mIdent `opt` Nothing)
>                     <*> (Just <$> importSpec `opt` Nothing)

> importSpec :: Parser Token ImportSpec a
> importSpec =
>   position <**> (Hiding <$-> token Id_hiding `opt` Importing)
>            <*> parens (spec `sepBy` comma)
>   where spec = functionName <**> (parens constrs `opt` Import)
>         constrs = ImportTypeAll <$-> token DotDot
>               <|> flip ImportTypeWith <$> functionName `sepBy` comma

> infixDecl :: Parser Token Decl a
> infixDecl = infixDeclLhs InfixDecl <*> infixOpId `sepBy1` comma

> infixDeclLhs :: (Position -> Infix -> Int -> a) -> Parser Token a b
> infixDeclLhs f = f <$> position <*> tokenOps infixKW <*> int
>   where infixKW = [(KW_infix,Infix),(KW_infixl,InfixL),(KW_infixr,InfixR)]

\end{lstlisting}
Parsing of type class declarations and instance declarations is
very similar, so we introduce a new combinator.
\begin{lstlisting}

> classInstHead :: (Position -> TypeExprContext -> a -> b -> c) -> 
>                  Category ->
>                  Parser Token a d -> Parser Token b d ->
>                  Parser Token c d
> classInstHead f kw classIdent argument =
>   mkDecl <$> position <*-> token kw <*> declHead
>   where mkDecl p (ctxt,tc,x) = f p ctxt tc x
>         declHead = (,,) <$> typeExprContext <*-> token ContextRightArrow
>                         <*> classIdent <*> argument
>                  <|?>
>                    (,,) emptyTypeExprContext
>                         <$> classIdent <*> argument

> typeClassDecl :: Parser Token Decl a
> typeClassDecl =
>   classInstHead TypeClassDecl KW_class ident ident
>     <*> (token KW_where <-*> layout methodDecl `opt` [])
>   where methodDecl :: Parser Token Decl a
>         methodDecl = functionDecl

> instanceDecl :: Parser Token Decl a
> instanceDecl =
>   classInstHead InstanceDecl KW_instance qIdent type2
>     <*> (token KW_where <-*> layout methodDef `opt` [])
>   where methodDef :: Parser Token Decl a
>         methodDef = functionDecl

\end{lstlisting}
Parsing of type declarations are also very similar. A new combinator
is introduced.
\begin{lstlisting}

> dataDecl :: Parser Token Decl a
> dataDecl = typeDeclLhs DataDecl KW_data
>            <*> (equals <-*> constrDecl `sepBy1` bar `opt` [])
>            <*> optDeriving

> optDeriving :: Parser Token [Ident] a
> optDeriving = token KW_deriving <-*> classes `opt` []
>   where classes = return <$> ident
>               <|> parens (ident `sepBy1` comma)

> newtypeDecl :: Parser Token Decl a
> newtypeDecl = typeDeclLhs NewtypeDecl KW_newtype <*-> equals <*> nconstrDecl
>               <*> optDeriving

> typeDecl :: Parser Token Decl a
> typeDecl = typeDeclLhs TypeDecl KW_type <*-> equals <*> type0

> typeDeclLhs :: (Position -> Ident -> [Ident] -> a) -> Category
>             -> Parser Token a b
> typeDeclLhs f kw = f <$> position <*-> token kw <*> ident <*> many typeVar
>   where typeVar = ident <|> anonId <$-> token Underscore

> constrDecl :: Parser Token (ConstrDecl [TypeExpr]) a
> constrDecl = constr <*> many type2

> nconstrDecl :: Parser Token (ConstrDecl TypeExpr) a
> nconstrDecl = constr <*> type2

> constr :: Parser Token (a -> ConstrDecl a) b
> constr = flip . ConstrDecl <$> position <*> existVars <*> ident
>   where {- existVars = token Id_forall <-*> many1 ident <*-> dot `opt` [] -}
>         existVars = succeed []

> functionDecl :: Parser Token Decl a
> functionDecl = position <**> decl <|> externalDecl
>   where decl = ident <**> identDecl
>            <|> leftParen <-*> parenDecl
>           <|!> constrTerm0 <**> opDecl
>         identDecl = many (comma <-*> functionName) <**> funListDecl
>                 <|> as <$-> token At <*> constrTerm2 <*> opDecl
>                 <|> many constrTerm2 <**> (funRhs <|> con <$> opDecl)
>         parenDecl = opId <*-> rightParen <**> identDecl
>                <|!> tuple <$> (constrTerm0 `sepBy` comma) <*-> rightParen
>                           <**> opDecl
>         opDecl = opFn <$> infixOpId <*> constrTerm0 <*> funRhs
>         funRhs = fun <$> declRhs
>         fun rhs ts f p = FunctionDecl p f [Equation p ts rhs]
>         as t f v = f (AsPattern v t)
>         con f ts x
>           | null ts = f (VariablePattern x)
>           | otherwise = f (ConstructorPattern (qualify x) ts)
>         opFn op t2 f t1  = f [t1,t2] op
>         tuple [t] = t
>         tuple ts = TuplePattern ts

> valueDecl :: Parser Token Decl a
> valueDecl = position <**> decl <|> externalDecl
>   where decl = ident <**> identDecl
>            <|> leftParen <-*> parenDecl
>           <|!> constrTerm0 <**> patDecl
>         identDecl = (many (comma <-*> functionName)) <**> valListDecl
>                 <|> as <$-> token At <*> constrTerm2 <*> patDecl
>                 <|> var <$> patDecl
>                 <|> many1 constrTerm2 <**> (funRhs <|> con <$> opDecl)
>         parenDecl = opId <*-> rightParen <**> identDecl
>                <|!> tuple <$> (constrTerm0 `sepBy` comma) <*-> rightParen
>                           <**> patDecl
>         patDecl = pat <$> declRhs <|> opDecl
>         opDecl = opFn <$> infixOpId <*> constrTerm0 <*> funRhs
>         funRhs = fun <$> declRhs
>         fun rhs ts f p = FunctionDecl p f [Equation p ts rhs]
>         pat rhs t p = PatternDecl p t rhs
>         var f v = f (VariablePattern v)
>         as t f v = f (AsPattern v t)
>         con f ts c = f (ConstructorPattern (qualify c) ts)
>         opFn op t2 f t1 = f [t1,t2] op
>         tuple [t] = t
>         tuple ts = TuplePattern ts

> funListDecl :: Parser Token ([Ident] -> Ident -> Position -> Decl) a
> funListDecl = typeSig <$-> token DoubleColon <*> typeExprWithContext
>           <|> evalAnnot <$-> token KW_eval <*> tokenOps evalKW
>   where typeSig tyWC vs v p = TypeSig p (v:vs) tyWC
>         evalAnnot ev vs v p = EvalAnnot p (v:vs) ev
>         evalKW = [(KW_rigid,EvalRigid),(KW_choice,EvalChoice)]

> valListDecl :: Parser Token ([Ident] -> Ident -> Position -> Decl) a
> valListDecl = funListDecl <|> extraVars <$-> token KW_free
>   where extraVars vs v p = ExtraVariables p (v:vs)

> declRhs :: Parser Token Rhs a
> declRhs = rhs equals

> rhs :: Parser Token a b -> Parser Token Rhs b
> rhs eq = (SimpleRhs <$-> eq <*> position <*> expr
>           <|> GuardedRhs <$> many1 (condExpr eq))
>       <*> localDefs

> externalDecl :: Parser Token Decl a
> externalDecl =
>   ExternalDecl <$> position <*-> token KW_external
>                <*> callConv <*> (Just <$> string `opt` Nothing)
>                <*> functionName <*-> token DoubleColon <*> type0
>   where callConv = CallConvPrimitive <$-> token Id_primitive
>                <|> CallConvCCall <$-> token Id_ccall
>                <?> "Unsupported calling convention"

\end{lstlisting}
\codesubsection{Interface declarations}
\begin{lstlisting}

> intfDecl :: Parser Token IDecl a
> intfDecl = iImportDecl <|> iInfixDecl
>       <|> iTypeClassDecl <|> iInstanceDecl
>       <|> iHidingDecl <|> iDataDecl <|> iNewtypeDecl <|> iTypeDecl
>       <|> iKindDecl
>      <|!> iFunctionDecl

> iImportDecl :: Parser Token IDecl a
> iImportDecl = IImportDecl <$> position <*-> token KW_import <*> mIdent

> iInfixDecl :: Parser Token IDecl a
> iInfixDecl = infixDeclLhs IInfixDecl <*> qInfixOpId

> iTypeClassDecl :: Parser Token IDecl a
> iTypeClassDecl =
>   classInstHead ITypeClassDecl KW_class qIdent ident
>     <*> (token KW_where <-*> layout methodDecl `opt` [])
>   where methodDecl :: Parser Token IDecl a
>         methodDecl = iFunctionDecl

> iInstanceDecl :: Parser Token IDecl a
> iInstanceDecl =
>   classInstHead IInstanceDecl KW_instance qIdent type2

> iHidingDecl :: Parser Token IDecl a
> iHidingDecl = position <*-> token Id_hiding <**> (dataDecl <|> funcDecl)
>   where dataDecl = hiddenData <$-> token KW_data <*> ident <*> many ident
>         funcDecl = hidingFunc <$-> token DoubleColon <*> typeExprWithContext
>         hiddenData tc tvs p = HidingDataDecl p tc tvs
>         hidingFunc tyWC p = IFunctionDecl p hidingId tyWC
>         hidingId = qualify (mkIdent "hiding")

> iDataDecl :: Parser Token IDecl a
> iDataDecl = iTypeDeclLhs IDataDecl KW_data
>             <*> (equals <-*> iConstrDecl `sepBy1` bar `opt` [])
>   where iConstrDecl = Just <$> constrDecl
>                   <|> Nothing <$-> token Underscore

> iNewtypeDecl :: Parser Token IDecl a
> iNewtypeDecl =
>   iTypeDeclLhs INewtypeDecl KW_newtype <*-> equals <*> nconstrDecl

> iTypeDecl :: Parser Token IDecl a
> iTypeDecl = iTypeDeclLhs ITypeDecl KW_type <*-> equals <*> type0

> iTypeDeclLhs :: (Position -> QualIdent -> [Ident] -> a) -> Category
>              -> Parser Token a b
> iTypeDeclLhs f kw = f <$> position <*-> token kw <*> qIdent <*> many ident

> iFunctionDecl :: Parser Token IDecl a
> iFunctionDecl =
>   IFunctionDecl <$> position <*> qFunctionName <*-> token DoubleColon
>                 <*> typeExprWithContext

> iKindDecl :: Parser Token IDecl a
> iKindDecl = position <*-> token Id_kind <**> (kindDecl <|> funcDecl)
>   where kindDecl = intfKind <$> qIdent <*> kind
>         funcDecl = kindFunc <$-> token DoubleColon <*> typeExprWithContext
>         intfKind f k p = IKindDecl p f k
>         kindFunc tyWC p = IFunctionDecl p kindId tyWC
>         kindId = qualify (mkIdent "kind")

\end{lstlisting}
\codesubsection{Types}
\begin{lstlisting}

> type0 :: Parser Token TypeExpr a
> type0 = type1 `chainr1` (typeExprApplyArrowConstructor <$-> token RightArrow)

> type1 :: Parser Token TypeExpr a
> type1 = apply <$> qIdent <*> many type2
>     <|> leftBracket <-*> bracketType
>    <|!> type2
>   where apply id tys
>           | isQualified id = typeExprApplyConstructor id tys
>           | otherwise = typeExprApply (TypeExprVariable $ unqualify id) tys
>         bracketType =
>               typeExprApply typeExprListConstructor <$-> rightBracket
>                                                     <*> many type2
>           <|> typeExprApplyListConstructor <$> type0 <*-> rightBracket

> type2 :: Parser Token TypeExpr a
> type2 = anonType <|> varType <|> parenType <|> listType

> anonType :: Parser Token TypeExpr a
> anonType = TypeExprVariable anonId <$-> token Underscore

> varType :: Parser Token TypeExpr a
> varType = typeVar <$> qIdent
>   where typeVar v
>           | isQualified v = TypeExprConstructor v
>           | otherwise = TypeExprVariable (unqualify v)

> parenType :: Parser Token TypeExpr a
> parenType = tuple <$> parens (type0 `sepBy` comma)
>   where tuple []   = typeExprUnitConstructor
>         tuple [ty] = ty
>         tuple tys  = typeExprApplyTupleConstructor tys

> listType :: Parser Token TypeExpr a
> listType = brackets (typeExprApplyListConstructor <$> type0 `opt`
>                      typeExprListConstructor)

\end{lstlisting}
\codesubsection{Type expressions with contexts}
\begin{lstlisting}

> typeExprWithContext :: Parser Token TypeExprWithContext a
> typeExprWithContext =
>    TypeExprWithContext <$> typeExprContext <*-> token ContextRightArrow <*> type0
>    <|?> TypeExprWithContext emptyTypeExprContext <$> type0

> typeExprContext :: Parser Token TypeExprContext a
> typeExprContext =   TypeExprContext <$> parens (typeExprClassConstraint `sepBy` comma)
>                 <|> (\x -> TypeExprContext [x]) <$> typeExprClassConstraint

> typeExprClassConstraint :: Parser Token TypeExprClassConstraint a
> typeExprClassConstraint = TypeExprClassConstraint <$> qIdent <*>
>                             (TypeExprVariable <$> ident)

\end{lstlisting}
\codesubsection{Literals}
\begin{lstlisting}

> literal :: Parser Token Literal a
> literal = Char <$> char
>       <|> Int <$> int
>       <|> Float <$> float
>       <|> String <$> string

\end{lstlisting}
\codesubsection{Kinds}
The kind expression language contains parentesis.
\begin{lstlisting}

> kind :: Parser Token Kind a
> kind = (Star <$-> token At <|> parens kind) `chainr1`
>        (KFun <$-> token RightArrow)

\end{lstlisting}
\codesubsection{Patterns}
\begin{lstlisting}

> constrTerm0 :: Parser Token ConstrTerm a
> constrTerm0 = constrTerm1 `chainr1` (flip InfixPattern qConsId <$-> colon)

> constrTerm1 :: Parser Token ConstrTerm a
> constrTerm1 = ident <**> (flip AsPattern <$-> token At <*> constrTerm2
>                           <|> con <$> many constrTerm2)
>          <|!> ConstructorPattern <$> qIdent <*> many constrTerm2
>           <|> LiteralPattern . Int . negate <$-> minus <*> checkInt
>           <|> LiteralPattern . Float . negate <$-> fminus <*> checkFloat
>           <|!> constrTerm2
>   where con ts x
>           | null ts = VariablePattern x
>           | otherwise = ConstructorPattern (qualify x) ts

> constrTerm2 :: Parser Token ConstrTerm a
> constrTerm2 = literalPattern <|> anonPattern <|> varPattern
>           <|> parenPattern <|> listPattern <|> lazyPattern

> literalPattern :: Parser Token ConstrTerm a
> literalPattern = LiteralPattern <$> literal

> anonPattern :: Parser Token ConstrTerm a
> anonPattern = VariablePattern anonId <$-> token Underscore

> varPattern :: Parser Token ConstrTerm a
> varPattern = ident <**> (flip AsPattern <$-> token At <*> constrTerm2
>                          `opt` VariablePattern)
>         <|!> flip ConstructorPattern [] <$> qIdent

> parenPattern :: Parser Token ConstrTerm a
> parenPattern = tuple <$> parens (constrTerm0 `sepBy` comma)
>   where tuple [t] = t
>         tuple ts = TuplePattern ts

> listPattern :: Parser Token ConstrTerm a
> listPattern = ListPattern <$> brackets (constrTerm0 `sepBy` comma)

> lazyPattern :: Parser Token ConstrTerm a
> lazyPattern = LazyPattern <$-> token Tilde <*> constrTerm2

\end{lstlisting}
\codesubsection{Expressions}
\begin{lstlisting}

> condExpr :: Parser Token a b -> Parser Token CondExpr b
> condExpr eq = CondExpr <$> position <*-> bar <*> expr0 <*-> eq <*> expr

> expr :: Parser Token Expression a
> expr = expr0 <??> (flip Typed <$-> token DoubleColon <*> typeExprWithContext)

> expr0 :: Parser Token Expression a
> expr0 = expr1 `chainr1` (flip InfixApply <$> infixOp)

> expr1 :: Parser Token Expression a
> expr1 = UnaryMinus <$> (minus <|> fminus) <*> expr2
>     <|> expr2

> expr2 :: Parser Token Expression a
> expr2 = lambdaExpr <|> letExpr <|> doExpr <|> ifExpr <|> caseExpr
>     <|> foldl1 Apply <$> many1 expr3

> expr3 :: Parser Token Expression a
> expr3 = constant <|> variable <|> parenExpr <|> listExpr

> constant :: Parser Token Expression a
> constant = Literal <$> literal

> variable :: Parser Token Expression a
> variable = Variable <$> qIdent

> parenExpr :: Parser Token Expression a
> parenExpr = parens pExpr
>   where pExpr = (minus <|> fminus) <**> minusOrTuple
>            <|> tupleConstr . length <$> many comma
>           <|!> leftSectionOrTuple
>           <|!> opOrRightSection
>         minusOrTuple = expr1 <**> (unaryMin <$> infixOrTuple)
>                  `opt` Variable . qualify
>         leftSectionOrTuple = expr1 <**> (($ id) <$> infixOrTuple)
>         infixOrTuple = infixOp <**> leftSectionOrExp
>                    <|> tuple <$> many (comma <-*> expr)
>                    <|> tyTuple <$-> token DoubleColon <*> typeExprWithContext
>                                <*> many (comma <-*> expr)
>         leftSectionOrExp = expr1 <**> (infixApp <$> infixOrTuple)
>                            `opt` leftSection
>         opOrRightSection = infixOp <**> optRightSection
>         optRightSection = flip RightSection <$> expr0 `opt` Variable . opName
>         unaryMin f e op = f id (UnaryMinus op e)
>         infixApp f e2 op g e1 = f (g . InfixApply e1 op) e2
>         leftSection op f e = LeftSection (f e) op
>         tuple es f e = if null es then Paren (f e) else Tuple (f e:es)
>         tyTuple ty es f = tuple es (flip Typed ty . f)
>         tupleConstr n =
>           if n == 0 then Tuple [] else Constructor (qTupleId (n + 1))

> listExpr :: Parser Token Expression a
> listExpr = brackets (elements `opt` List [])
>   where elements = expr <**> rest
>         rest = comprehension
>            <|> enumeration (flip EnumFromTo) EnumFrom
>            <|> comma <-*> expr <**>
>                (enumeration (flip3 EnumFromThenTo) (flip EnumFromThen)
>                <|> (\es e2 e1 -> List (e1:e2:es)) <$> many (comma <-*> expr))
>            `opt` (\e -> List [e])
>         comprehension =
>           flip ListCompr <$-> bar <*> (statement `sepBy1` comma)
>         enumeration enumTo enum =
>           token DotDot <-*> (enumTo <$> expr `opt` enum)
>         flip3 f x y z = f z y x

> lambdaExpr :: Parser Token Expression a
> lambdaExpr =
>   Lambda <$-> token Backslash <*> many1 constrTerm2 <*->
>   (token RightArrow <?> "-> expected") <*> expr

> letExpr :: Parser Token Expression a
> letExpr =
>   Let <$-> token KW_let <*> layout valueDecl <*->
>   (token KW_in <?> "in expected") <*> expr

> doExpr :: Parser Token Expression a
> doExpr = Do <$-> token KW_do <*> layout statement

> ifExpr :: Parser Token Expression a
> ifExpr =
>   IfThenElse <$-> token KW_if <*> expr <*->
>   (token KW_then <?> "then expected") <*> expr <*->
>   (token KW_else <?> "else expected") <*> expr

> caseExpr :: Parser Token Expression a
> caseExpr =
>   Case <$-> token KW_case <*> expr <*-> (token KW_of <?> "of expected")
>        <*> layout alt

> alt :: Parser Token Alt a
> alt =
>   Alt <$> position <*> constrTerm0
>       <*> rhs (token RightArrow <?> "-> expected")

\end{lstlisting}
\codesubsection{Statements in list comprehensions and \texttt{do} expressions}
Parsing statements is a bit difficult because the syntax of patterns
and expressions largely overlaps. The parser will first try to
recognize the prefix \emph{Pattern}~\texttt{<-} of a binding statement
and if this fails fall back into parsing an expression statement. In
addition, we have to be prepared that the sequence
\texttt{let}~\emph{LocalDefs} can be either a let-statement or the
prefix of a let expression.

\textbf{Do not change the order in the statement parser. Let
  statements must be tried before the expression parser is
  used. Otherwise the parser will always complain about a missing
  \texttt{in} keyword when parsing a let statement.}
\begin{lstlisting}

> statement :: Parser Token Statement a
> statement = letStmt <|!> exprOrBindStmt

> letStmt :: Parser Token Statement a
> letStmt =
>   token KW_let <-*> layout valueDecl <**>
>     ((\e ds -> StmtExpr (Let ds e)) <$-> token KW_in <*> expr
>      `opt` StmtDecl)

> exprOrBindStmt :: Parser Token Statement a
> exprOrBindStmt =
>   StmtBind <$> constrTerm0 <*-> leftArrow <*> expr <|?> StmtExpr <$> expr

\end{lstlisting}
\codesubsection{Literals, identifiers, and (infix) operators}
\begin{lstlisting}

> char :: Parser Token Char a
> char = cval <$> token CharTok

> int, checkInt :: Parser Token Int a
> int = ival <$> token IntTok
> checkInt = int <?> "integer number expected"

> float, checkFloat :: Parser Token Double a
> float = fval <$> token FloatTok
> checkFloat = float <?> "floating point number expected"

> string :: Parser Token String a
> string = sval <$> token StringTok

> infixOpId :: Parser Token Ident a
> infixOpId = opId <|> backquotes (ident <?> "function name expected")

> qInfixOpId :: Parser Token QualIdent a
> qInfixOpId = qOpId <|> backquotes (qIdent <?> "function name expected")

> infixOp :: Parser Token InfixOp a
> infixOp = InfixOp <$> qInfixOpId

> functionName :: Parser Token Ident a
> functionName = ident <|> parens (opId <?> "infix operator expected")

> qFunctionName :: Parser Token QualIdent a
> qFunctionName = qIdent <|> parens (qOpId  <?> "infix operator expected")

> ident :: Parser Token Ident a
> ident = mkIdent . sval <$> tokens [Id,Id_as,Id_ccall,Id_forall,Id_hiding,
>                                    Id_interface,Id_kind,Id_primitive,
>                                    Id_qualified]

> opId :: Parser Token Ident a
> opId = mkIdent . sval <$> tokens [Sym,Colon,Sym_Dot,Sym_Minus,Sym_MinusDot]

> mIdent :: Parser Token ModuleIdent a
> mIdent = mIdent <$> tokens [Id,QId,Id_as,Id_ccall,Id_forall,Id_hiding,
>                             Id_interface,Id_kind,Id_primitive,Id_qualified]
>   where mIdent a = mkMIdent (qual a ++ [sval a])

> qIdent :: Parser Token QualIdent a
> qIdent = qualify <$> ident <|> mkQIdent <$> token QId
>   where mkQIdent a = qualifyWith (mkMIdent (qual a)) (mkIdent (sval a))

> qOpId :: Parser Token QualIdent a
> qOpId = qualify <$> opId <|> mkQIdent <$> token QSym
>   where mkQIdent a = qualifyWith (mkMIdent (qual a)) (mkIdent (sval a))

> minus :: Parser Token Ident a
> minus = minusId <$-> token Sym_Minus

> fminus :: Parser Token Ident a
> fminus = fminusId <$-> token Sym_MinusDot

\end{lstlisting}
\codesubsection{Layout}
\begin{lstlisting}

> layout :: Parser Token a b -> Parser Token [a] b
> layout p = layoutBraces (p `sepBy` semicolon)

> layoutBraces :: Parser Token a b -> Parser Token a b
> layoutBraces p =
>   layoutOff <-*> braces p <|>
>   layoutOn <-*> p <*-> (token VRightBrace <|> layoutEnd)

\end{lstlisting}
\codesubsection{More combinators}
\begin{lstlisting}

> braces, brackets, parens, backquotes :: Parser Token a b -> Parser Token a b
> braces p = bracket leftBrace p rightBrace
> brackets p = bracket leftBracket p rightBracket
> parens p = bracket leftParen p rightParen
> backquotes p = bracket backquote p checkBackquote

\end{lstlisting}
\codesubsection{Simple token parsers}
\begin{lstlisting}

> token :: Category -> Parser Token Attributes a
> token c = attr <$> symbol (Token c NoAttributes)
>   where attr (Token _ a) = a

> tokens :: [Category] -> Parser Token Attributes a
> tokens cs = foldr1 (<|>) (map token cs)

> tokenOps :: [(Category,a)] -> Parser Token a b
> tokenOps cs = ops [(Token c NoAttributes,x) | (c,x) <- cs]

> dot, colon, comma, semicolon, bar, equals :: Parser Token Attributes a
> dot = token Sym_Dot
> colon = token Colon
> comma = token Comma
> semicolon = token Semicolon <|> token VSemicolon
> bar = token Bar
> equals = token Equals

> backquote, checkBackquote :: Parser Token Attributes a
> backquote = token Backquote
> checkBackquote = backquote <?> "backquote (`) expected"

> leftParen, rightParen :: Parser Token Attributes a
> leftParen = token LeftParen
> rightParen = token RightParen

> leftBracket, rightBracket :: Parser Token Attributes a
> leftBracket = token LeftBracket
> rightBracket = token RightBracket

> leftBrace, rightBrace :: Parser Token Attributes a
> leftBrace = token LeftBrace
> rightBrace = token RightBrace

> leftArrow :: Parser Token Attributes a
> leftArrow = token LeftArrow

\end{lstlisting}
