/* This is for emacs: -*-Mode: C++;-*- */
/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#if !defined(__INC_UCXX_SCRIPT_TRAITS_H)
#define __INC_UCXX_SCRIPT_TRAITS_H

#include <typeinfo>

#include <yehia/plugin.h>
#include <yehia/script/wrapobj.h>

namespace Yehia
{

namespace Script
{


template <class T>
struct MarshalTraits
{
    typedef T ArgType;
    typedef T SigType;
    
    static Any marshal(T t) { return Any(t); }
    static T unmarshal(const Any& a) { return any_cast<T>(a); }
};

template <class T>
struct MarshalTraits<T *>
{
    typedef T *ArgType;
    typedef T SigType;
    
    static Any marshal(T *t) { 
      if (t == 0)
        return Any();
      
      Any::InstanceValue h;
      if (!(h.instance = dynamic_cast<SigC::Object *>(t)))
      {
        h.instance = SigC::manage(new WrapObject(t));
        h.instance->reference();
      }
      h.tinfo = &typeid(T);
      return Any(h);
    }
    static T *unmarshal(const Any& a) {
      if (a.typecode() == Any::TC_VOID)
        return 0;
      Any::InstanceValue inst = any_cast<Any::InstanceValue>(a);
      if (typeid(*inst.instance) != typeid(WrapObject))
        return dynamic_cast<T *>(inst.instance); 
      WrapObject *wrap = static_cast<WrapObject *>(inst.instance);
      return wrap->type() == typeid(T) ?
        static_cast<WrapObject::Holder<T> *>(wrap->content_)->held_ : 0;
    }
};

template <class T>
struct MarshalTraits<const T *>
{
    typedef const T *ArgType;
    typedef T SigType;
    
    static Any marshal(const T *t) { 
      return MarshalTraits<T *>::marshal(const_cast<T *>(t)); 
    }
    static const T *unmarshal(const Any& a) { 
      return MarshalTraits<T *>::unmarshal(a); 
    }
};

template <class T>
struct MarshalTraits<T&>
{
    typedef T& ArgType;
    typedef T SigType;
    
    static Any marshal(T& t) { 
      return MarshalTraits<T *>::marshal(&t); 
    }
    static T& unmarshal(const Any& a) { 
      return *MarshalTraits<T *>::unmarshal(a); 
    }
};

template <class T>
struct MarshalTraits<const T&>
{
    typedef const T& ArgType;
    typedef T SigType;
    
    static Any marshal(const T& t) { 
      return MarshalTraits<T *>::marshal(const_cast<T *>(&t)); 
    }
    static const T& unmarshal(const Any& a) { 
      return *MarshalTraits<T *>::unmarshal(a); 
    }
};


template <>
struct MarshalTraits<int>
{
    typedef long ArgType;
    typedef long SigType;
    
    static Any marshal(int i) { return Any((long)i); }
    static int unmarshal(const Any& a) { return any_cast<long>(a); }
};

template <>
struct MarshalTraits<unsigned int>
{
    typedef unsigned long ArgType;
    typedef unsigned long SigType;
    
    static Any marshal(unsigned int i) { return Any((unsigned long)i); }
    static unsigned int unmarshal(const Any& a) { 
      return any_cast<unsigned long>(a); 
    }
};

template <>
struct MarshalTraits<const char *>
{
    typedef const char *ArgType;
    typedef std::string SigType;
    
    static Any marshal(const char *cp) { return Any(cp); }
    static const char *unmarshal(const Any& a) { return any_cast<char *>(a); }
};

template <>
struct MarshalTraits<const std::string&>
{
    typedef std::string ArgType;
    typedef std::string SigType;
    
    static Any marshal(const std::string& s) { return Any(s); }
    static std::string unmarshal(const Any& a) { 
      return any_cast<std::string>(a); 
    }
};

template <class T>
struct MarshalTraits<std::list<T> >
{
    typedef std::list<T> ArgType;
    typedef std::list<Any> SigType;
    
    static Any marshal(const std::list<T>& l) {
      std::list<Any> anylist;
      for (typename std::list<T>::const_iterator it = l.begin(); it != l.end(); ++it)
        anylist.push_back(MarshalTraits<T>::marshal(*it));
      return Any(anylist); 
    }
    static std::list<T> unmarshal(const Any& a) {
      std::list<Any> anylist = any_cast<std::list<Any> >(a);
      std::list<T> result;
      for (std::list<Any>::iterator it = anylist.begin(); it != anylist.end(); ++it)
        result.push_back(MarshalTraits<T>::unmarshal(*it));
      return result;
    }
};

template <class T>
struct MarshalTraits<const std::list<T>& >
{
    typedef std::list<T> ArgType;
    typedef std::list<Any> SigType;
    
    static Any marshal(const std::list<T>& l) {
      return MarshalTraits<std::list<T> >::marshal(l); 
    }
    static std::list<T> unmarshal(const Any& a) {
      return MarshalTraits<std::list<T> >::unmarshal(a);
    }
};

// FIXME: add vectors to built-in types instead of treating them as lists

template <class T>
struct MarshalTraits<std::vector<T> >
{
    typedef std::vector<T> ArgType;
    typedef std::list<Any> SigType;

    static Any marshal(const std::vector<T>& l) {
      return MarshalTraits<std::list<T> >::marshal(
              std::list<T>(l.begin(), l.end())); 
    }
    static std::vector<T> unmarshal(const Any& a) {
      std::list<T> l = MarshalTraits<std::list<T> >::unmarshal(a);
      return std::vector<T>(l.begin(), l.end());
    }
};

template <class T>
struct MarshalTraits<const std::vector<T>& >
{
    typedef std::vector<T> ArgType;
    typedef std::list<Any> SigType;
    
    static Any marshal(const std::vector<T>& l) {
      return MarshalTraits<std::list<T> >::marshal(
              std::list<T>(l.begin(), l.end())); 
    }
    static std::vector<T> unmarshal(const Any& a) {
      std::list<T> l = MarshalTraits<std::list<T> >::unmarshal(a);
      return std::vector<T>(l.begin(), l.end());
    }
};

template <>
struct MarshalTraits<const Any::SlotValue&>
{
    typedef Any::SlotValue ArgType;
    typedef Any SigType; // FIXME
    
    static Any marshal(const Any::SlotValue& s) { return Any(s); }
    static Any::SlotValue unmarshal(const Any& a) { 
      return any_cast<Any::SlotValue>(a); 
    }
};

}

}

#endif
