/*  
  Copyright 2002, Andreas Rottmann

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public  License
  along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <string.h>
#include <stdlib.h>

#include <gc/gc.h>

#include "tree.h"
#include "process.h"

#define DEFTREECODE(SYM, STRING) STRING, 
static const char *tree_code_name[] =
{
#include "tree.def"
  NULL
};
#undef DEFTREECODE

#define TREE_INIT(node, code) { \
  TREE_SET_CODE((node), (code)); \
  TREE_LINENO(node) = process_get_infile_lineno(); \
  TREE_FILENAME(node) = process_get_infile_name(); \
}

Tree tree_cons(Tree car, Tree cdr)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeList));
  
  memset(node, 0, sizeof(TreeList));
  
  TREE_INIT(node, TREE_LIST);
  TREE_CHAIN(node) = cdr;
  TREE_VALUE(node) = car;

  return node;
}

Tree tree_append(Tree list, Tree val)
{
  Tree node, newnode = (Tree)GC_malloc(sizeof(TreeList));
  
  memset(newnode, 0, sizeof(TreeList));
  
  TREE_INIT(newnode, TREE_LIST);
  TREE_VALUE(newnode) = val;
  TREE_CHAIN(newnode) = NULL_TREE;

  if (list == NULL_TREE)
    return newnode;

  for (node = list; TREE_CHAIN(node); node = TREE_CHAIN(node))
    ;

  TREE_CHAIN(node) = newnode;

  return list;
}

int tree_list_length(Tree list)
{
  Tree node;
  int len = 0;
  
  for (node = list; node; node = TREE_CHAIN(node))
    len++;

  return len;
}

Tree tree_build_value(Tree identifier, Tree val)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeValue));
  
  memset(node, 0, sizeof(TreeValue));

  TREE_INIT(node, VALUE_STMT);
  TREE_VALUE_IDENTIFIER(node) = identifier;
  TREE_VALUE_VAL(node) = val;

  return node;
}

Tree tree_build_class(Tree identifier, int wrapped, 
                      Tree defined_by, Tree supers, Tree members)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeClass));
  
  memset(node, 0, sizeof(TreeClass));

  TREE_INIT(node, CLASS_STMT);
  TREE_CLASS_IDENTIFIER(node) = identifier;
  if (defined_by == NULL)
    TREE_CLASS_DEFINED_BY(node) = tree_cons(
            tree_build_typename(identifier, NULL_TREE), NULL_TREE);
  else
    TREE_CLASS_DEFINED_BY(node) = defined_by;
  TREE_CLASS_WRAPPED(node) = wrapped;
  TREE_CLASS_SUPERS(node) = supers;
  TREE_CLASS_MEMBERS(node) = members;

  return node;
}

Tree tree_build_class_decl(Tree ident, Tree defby)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeClassDecl));
  
  memset(node, 0, sizeof(TreeClassDecl));

  TREE_INIT(node, CLASS_DECL);
  TREE_CLASSDECL_IDENTIFIER(node) = ident;
  if (defby == NULL_TREE)
    TREE_CLASSDECL_DEFINED_BY(node) =
      tree_cons(tree_build_typename(ident, NULL_TREE), NULL_TREE);
  else
    TREE_CLASSDECL_DEFINED_BY(node) = defby;

  return node;
}

Tree tree_build_attribute(AttributeType type, Tree identifier,
                          Tree defby, Tree sig, Tree retspec,
                          gboolean v, gboolean pure)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeAttribute));
  
  memset(node, 0, sizeof(TreeAttribute));

  TREE_INIT(node, ATTR_STMT);
  TREE_ATTR_TYPE(node) = type;
  TREE_ATTR_IDENTIFIER(node) = identifier;
  TREE_ATTR_DEFINED_BY(node) = defby;
  TREE_ATTR_SIGNATURE(node) = sig;
  TREE_ATTR_RET_SPEC(node) = retspec;
  TREE_ATTR_VIRTUAL(node) = v;
  TREE_ATTR_PURE(node) = pure;
  
  return node;
}

Tree tree_build_typename(Tree identifier, Tree template_spec)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeTypename));
  
  memset(node, 0, sizeof(TreeTypename));

  TREE_INIT(node, TYPE_NAME);
  TREE_TYPENAME_IDENTIFIER(node) = identifier;
  TREE_TYPENAME_TEMPLATE_SPEC(node) = template_spec;
  
  return node;
}  

Tree tree_build_string(const char *s)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeStringCst));
  
  memset(node, 0, sizeof(TreeStringCst));

  TREE_INIT(node, STRING_CST);
  TREE_STRING_CST_LEN(node) = strlen(s) - 1;
  TREE_STRING_CST_PTR(node) = (char *)GC_malloc(TREE_STRING_CST_LEN(node) + 1);
  memcpy((char *)TREE_STRING_CST_PTR(node), s + 1,
         TREE_STRING_CST_LEN(node) - 1);
  ((char *)TREE_STRING_CST_PTR(node))[TREE_STRING_CST_LEN(node)] = '\0';
  
  return node;
}

Tree tree_build_integer(const char *s)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeIntCst));
  
  memset(node, 0, sizeof(TreeIntCst));

  TREE_INIT(node, INTEGER_CST);
  
  if (s[0] == '-')
    TREE_INT_CST(node) = (unsigned long)strtol(s, NULL, 0);
  else
  {
    TREE_INT_CST(node) = strtoul(s, NULL, 0);
    TREE_UNSIGNED(node) = 1;
  }

  return node;
}

Tree tree_build_identifier(const char *s)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeIdentifier));
  
  memset(node, 0, sizeof(TreeIdentifier));

  TREE_INIT(node, IDENTIFIER_NODE);
  TREE_IDENTIFIER_LEN(node) = strlen(s);
  TREE_IDENTIFIER_STR(node) = (char *)GC_malloc(TREE_IDENTIFIER_LEN(node) + 1);
  memcpy((char *)TREE_IDENTIFIER_STR(node), s, TREE_IDENTIFIER_LEN(node) + 1);

  return node;
}

Tree tree_build_typespec(const char *s)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeTypeSpec));
  
  memset(node, 0, sizeof(TreeTypeSpec));

  TREE_INIT(node, TYPE_SPEC);
  
  switch (s[0])
  {
    case 'u':
      TREE_TYPESPEC(node) = TYPESPEC_UNSIGNED;
      break;
    case 's':
      switch (s[1])
      {
        case 'i':
          TREE_TYPESPEC(node) = TYPESPEC_SIGNED;
          break;
        case 'h':
          TREE_TYPESPEC(node) = TYPESPEC_SHORT;
          break;
        default:
          process_internal_error("invalid typespec");
      }
    case 'l':
      TREE_TYPESPEC(node) = TYPESPEC_LONG;
      break;
    case 'i':
      TREE_TYPESPEC(node) = TYPESPEC_INT;
      break;
    case 'c':
      TREE_TYPESPEC(node) = TYPESPEC_CHAR;
      break;
    case 'd':
      TREE_TYPESPEC(node) = TYPESPEC_DOUBLE;
      break;
    case 'b':
      TREE_TYPESPEC(node) = TYPESPEC_BOOL;
      break;
    default:
      process_internal_error("invalid typespec");
  }
  
  return node;
}

Tree tree_build_typedescr(int is_const, Tree specs, TypeDescr type)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeTypeDescr));
  
  memset(node, 0, sizeof(TreeTypeDescr));

  TREE_INIT(node, TYPE_DESCR);
  TREE_TYPEDESCR(node) = type;
  TREE_TYPEDESCR_SPECS(node) = specs;
  TREE_TYPEDESCR_CONST(node) = is_const;

  return node;
}

Tree tree_build_paramdescr(Tree type, Tree name)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeParamDescr));
  
  memset(node, 0, sizeof(TreeParamDescr));

  TREE_INIT(node, PARAM_DESCR);
  TREE_PARAMDESCR_TYPE(node) = type;
  TREE_PARAMDESCR_NAME(node) = name;

  return node;
}

Tree tree_build_namespace(Tree ident, Tree members)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeNamespace));
  
  memset(node, 0, sizeof(TreeNamespace));

  TREE_INIT(node, NAMESPACE_STMT);
  TREE_NAMESPACE_IDENTIFIER(node) = ident;
  TREE_NAMESPACE_MEMBERS(node) = members;

  return node;
}

Tree tree_build_meta_plugin(Tree ident, Tree descr, Tree used_modules)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeMetaPlugin));
  
  memset(node, 0, sizeof(TreeMetaPlugin));

  TREE_INIT(node, META_PLUGIN);
  TREE_META_PLUGIN_IDENTIFIER(node) = ident;
  TREE_META_PLUGIN_DESCR(node) = descr;
  TREE_META_PLUGIN_USED_MODULES(node) = used_modules;

  return node;
}

Tree tree_build_verbatim(const char *s)
{
  Tree node = (Tree)GC_malloc(sizeof(TreeVerbatim));
  
  memset(node, 0, sizeof(TreeVerbatim));

  TREE_INIT(node, VERBATIM);
  TREE_VERBATIM_TEXT(node) = s;

  return node;
}

#if (__GNUC__ >= 2)

void tree_check_failed(const Tree node, 
                       TreeCode code, 
                       const char *filename, 
                       int line, 
                       const char *funcname)
{
#ifdef DEBUG
  abort();
#else
  process_internal_error("Tree check: expected %s, have %s in %s, at %s:%d",
                         tree_code_name[code], 
                         tree_code_name[TREE_CODE(node)],
                         funcname, filename, line);
#endif
}

#endif
