/* This is for emacs: -*-Mode: C++;-*- */
/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#if !defined(_INC_UCXX_SCRIPT_BASE_H)
#define _INC_UCXX_SCRIPT_BASE_H

#include <string>
#include <list>
#include <vector>
#include <stdexcept>

#include <sigc++/slot.h>
#include <sigc++/signal.h>

#include <sigcx/tunnel.h>
#include <sigcx/dispatch.h>

#include <yehia/error-handler.h>
#include <yehia/script/types.h>
#include <yehia/script/any.h>

/** \defgroup ucxxscript Scripting */

namespace Yehia
{

namespace Script
{

class Object;
class ObjectFactory;
class Language;

/** \addtogroup ucxx
 *@{*/

/** \addtogroup ucxxscript
 *@{*/

/** Script exception.
 *
 * Thrown on errors that happen during the execution of scripting code or
 * the C++-scripting glue.
 */
class Exception : public std::runtime_error
{
  public:
    Exception(const std::string& s) : std::runtime_error(s) { }
};

/** Bad parameter exception.
 *
 * Thrown when trying to pass a false number or the wrong type of arguments
 * to either a C++ function, called from scripting code or a script function,
 * called from C++ code. 
 */
class BadParam : public Exception
{
  public:
    BadParam(const std::string& s = "bad parameter") : Exception(s) { }
};


/** Namespace interface.
 * 
 * Implemented by objects that are similiar to a C++ namescape, like 
 * (in Python) packages, modules, classes and instances. */
class Namespace
{
  public:
    /** Get a namespace member.
     * \param name Name of the namespace member. 
     * \return The script object if found, 0 otherwise.
     */
    virtual Object *member(const std::string& name) = 0;

    /** Insert an object into the namespace.
     * \param name Name of the object.
     * \param object Object to insert. */
    virtual void insert(const std::string& name, Object& object) = 0;
};

/** Instance interface.
 *
 * This interface is implemented by instances of wrapped C++ classes.
 */
class Instance
{
  public:
    virtual SigC::Object& object() = 0;
    virtual Object& class_object() = 0;
};

/** Function interface.
 *
 * This interface is implemented by callable objects (methods,
 * functions and the like). 
 */
class Function
{
  public:
    virtual Object *call(const ParamList& params) = 0;
};
    
/** Generic Function interface.
 *
 * A generic function is a group of methods that have the same name
 * but have different signatures.
 */
class GenFunc : public Function
{
  public:
    virtual void add_method(const Slot& s, const Signature& sig) = 0;
};

/** Class interface.
 *
 * This interface is implemented by wrapped C++ class object.
 */
class Class
{
  public:
    /** Set the class constructor. 
     * \param genfunc Constructor generic function. */
    virtual void set_constructor(Object& genfunc) = 0;
    
    /** Add a generic function to the instances of the class.
     * The generic function may be called for instances of the class. 
     * It must take at least one argument, its type being of this class.
     * \param name Name of the generic function.
     * \param genfunc Generic function.
     */
    virtual void add_genfunc(const std::string& name, Object& genfunc) = 0;

    /** Add a generic function to the class itself.
     * The generic function implements a method that behave 
     * like a C++ static method, that is, it belongs to the class itself, 
     * not it's instances. The generic function should take no arguments. 
     * \param name Name of the generic function.
     * \param genfunc Generic function.
     **/
    virtual void add_class_genfunc(const std::string& name, Object& genfunc) = 0;

    /** Add a getter function to the instances of the class.
     * \param name Name of the generic function.
     * \param genfunc Generic function.
     */
    virtual void add_setter(const std::string& name, Object& genfunc) = 0;

    /** Add a getter function to the instances of the class.
     * \param name Name of the generic function.
     * \param genfunc Generic function.
     */
    virtual void add_getter(const std::string& name, Object& genfunc) = 0;
};

/** A scripting object.
 *
 * A scripting object represents an entity of a scripting language, like 
 * (in Python) modules, classes, methods. Scripting objects are 
 * reference-counted. If a object's reference count reaches zero it 
 * is destroyed.
 */
class Object
{
  public:
    /** Constructor.
     * The initial reference count is set to one.
     * \param factory The corresponding object factory. */
    Object(ObjectFactory& factory) : factory_(factory), refcnt_(1) { }
    /** Destructor. */
    virtual ~Object();
    
    /** Increase reference count. */
    void reference() {
      refcnt_++;
    }
    /** Descrease reference count and destroy \c this if it reaches zero. */
    void unreference();

    /** Get object factory. 
     * \return The object's factory. */
    ObjectFactory& factory() const { return factory_; }

    /*@{*/
    /** Get interface. 
     * \return A pointer to the interface or 0 if this interface is 
     *   not available. */

    virtual Namespace *namespace_interface();
    virtual Instance *instance_interface();
    virtual Function *function_interface();
    virtual GenFunc *genfunc_interface();
    virtual Class *class_interface();
    /*@}*/
    
    /** Get object value. 
     * \return Any object with the value of this object. */
    virtual Any value() const = 0;
  private:
    ObjectFactory& factory_;
    unsigned refcnt_;
};

//
// All objects are created and destroyed here
//
class ObjectFactory
{
    friend class Object;
  public:
    ObjectFactory();
    virtual ~ObjectFactory();
    
    virtual Object& create_value(const Any& v) = 0;
    virtual Object& create_genfunc() = 0;
    virtual Object& create_class(const Signature& supers = Signature()) = 0;
    virtual Object& create_namespace() = 0;

    virtual Object& wrap_instance(const Object& klass, SigC::Object& obj) = 0;
    
    virtual Object& null_object() = 0;
    
    void register_class(const std::type_info& ti, Object& klass);
    Object *find_class(const std::type_info& ti);

    bool operator==(const ObjectFactory& fact) {
      return this == &fact;
    }
  private:
    virtual void destroy(Object& object) = 0;

    GHashTable *class_hash_;
};


class Language : public SigC::Object
{
  public:
    Language(Namespace& ns, ObjectFactory& fact, SigCX::Tunnel *t);
    Language(const Language& l);

    Language& operator=(const Language& l);
    
    Namespace& root_namespace() { return ns_; }
    ObjectFactory& factory() { return fact_; }
    
    SigCX::Tunnel *tunnel() { return tunnel_; }
  private:
    Namespace& ns_;
    ObjectFactory& fact_;
    SigCX::Tunnel *tunnel_;
};

class LanguageManager : public ErrorHandler
{
  public:
    LanguageManager(ErrorHandler *parent = 0, SigCX::Tunnel *main_tunnel = 0);
    
    /** Get \c LanguageManager instance. 
     * The \c LanguageManager class is a singleton, that is, 
     * only one instance exists in a program.
     * \return If a \c LanguageManager instance has already been created, 
     *    this will return a reference to that instance, else a new instance 
     *    will be created by invoking the constructor with default 
     *    arguments. */
    static LanguageManager& instance();

    /** Get main tunnel.
     * The main tunnel is used for calls from the scripting plugins 
     * to application code.
     * \return The tunnel to the main thread of the program as specified 
     *   in the constructor or with PluginManager::set_main_tunnel(tunnel). */
    SigCX::Tunnel *main_tunnel() { return main_tunnel_; }
    /** Set main tunnel.
     * Set the tunnel to the main thread of the program. */
    void set_main_tunnel(SigCX::Tunnel *main_tunnel) { 
      main_tunnel_ = main_tunnel;
    }
    
    /** Get a language by factory
     *  \param factory The ObjectFactory of the language.
     *  \result The language with ObjectFactory \a factory, 0 if not found.
     */
    Script::Language *language(const Script::ObjectFactory& factory);
    /** Get the list of the scripting languages registered.
     * \return A list of pointers to the languages registered. */
    std::list<Script::Language *> languages() const;
    /** Get the list of names of the scripting languages registered.
     * \return A list of strings with the names of the languages registered. */
    std::list<std::string> language_names() const;
    
    /** Emitted when a new scripting language becomes available.
     * \param id The name of the new language. */
    SigC::Signal1<void, const std::string&> language_registered;
    /** Register a new scripting language. */
    void register_language(const std::string& name, Script::Language& lang);
    
    /** Return a language by name
     *  \param name The name of the language.
     *  \result The language named \c name, 0 if not found.
     */
    Script::Language *language(const std::string& name) {
      std::map<std::string, Script::Language *>::iterator it 
        = languages_.find(name);
      return it == languages_.end() ? 0 : (*it).second;
    }
  private:
    static LanguageManager *instance_;
    std::map<std::string, Script::Language *> languages_;
    
    SigCX::Tunnel *main_tunnel_;
};

/*@}*/

/*@}*/

} // Script

} // Yehia

#endif
