/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include "config.h"

#include <dirent.h>
#include <sys/stat.h>

#include <gmodule.h>

#include <iostream>

#include <sigc++/bind.h>
#include <sigc++/object.h>

#include <sigcx/thread_tunnel.h>

#include <yehia/plugin.h>

namespace Yehia
{

using namespace std;

class ShLibPluginLoader : public PluginLoader
{
  public:
    ShLibPluginLoader(ErrorHandler *parent) : PluginLoader(parent) { }
    virtual Plugin *load(PluginManager& mgr, const string& name);
    virtual string id() const { return "native"; }
    virtual void scan(PluginManager& mgr) const;
};

PluginManager *PluginManager::instance_ = 0;

PluginManager::PluginManager(ErrorHandler *parent)
    :  ErrorHandler(parent), plugin_tree_("root")
{
  if (instance_)
    throw std::runtime_error("an instance of PluginManager already exists");
  
  instance_ = this;
  scan_time_ = false;
  register_plugin_loader(*SigC::manage(new ShLibPluginLoader(this)));
}

PluginManager::~PluginManager()
{
  list<PluginLoader *>::iterator loader_it;
  
  //plugin_tree_.print();
  for (loader_it = loaders_.begin(); loader_it != loaders_.end(); ++loader_it)
    (*loader_it)->unreference();
  
  instance_ = 0;
}

PluginManager& PluginManager::instance()
{
  if (!instance_)
    instance_ = new PluginManager();

  return *instance_;
}

void PluginManager::register_plugin_loader(PluginLoader& loader)
{
  loaders_.push_back(&loader);
  loader.reference();

  scan();
}

void PluginManager::unregister_plugin_loader(PluginLoader& loader)
{
  list<PluginLoader *>::iterator it;
  
  for (it = loaders_.begin(); it != loaders_.end(); ++it)
    if ((*it) == &loader)
    {
      loaders_.erase(it);
      loader.unreference();
      break;
    }
  scan();
}

Plugin *PluginManager::load_plugin(const string& name)
{
  Plugin *plugin = plugins()[name];
  list<PluginLoader *>::iterator it;

  if (plugin)
    return plugin;
  
  emit_log(LOG_INFO, "load request: " + name);
  
  for (it = loaders_.begin(); it != loaders_.end(); ++it)
  {
    emit_log(LOG_DEBUG, string("using loader: ") + (*it)->id());
    if ((plugin = (*it)->load(*this, name)) != 0)
    {
      plugin_tree_.insert(name, plugin);
      plugin->unreference();
      plugin_loaded.emit(name);
      emit_log(LOG_INFO, "loaded: " + name);
      
      return plugin;
    }
  }
  
  if (last_error_.length() != 0)
    emit_error(last_error_);

  set_error();
  
  return 0;
}

void PluginManager::set_error(const string& err)
{
  last_error_ = err;
}


namespace
{

list<string> split(const string& str, char c)
{
  list<string> result;
  
  string::size_type lastpos = 0;
  string::size_type pos;
  do
  {
    pos = str.find(c, lastpos);
    result.push_back(str.substr(lastpos, pos - lastpos));
    lastpos = pos + 1;
  } while (pos != string::npos);

  return result;
}

}

const list<string>& PluginManager::arch_dep_paths()
{
  if (ad_paths_.size() > 0)
    return ad_paths_;

  char *env = getenv("YEHIA_ARCH_DEP_PLUGIN_PATH");
  string env_path = env ? env : "";
  if (env_path.size() != 0)
    ad_paths_ = split(env_path, ':');
  else
  {
    char *home = getenv("HOME"); 
    if (home != 0)
      ad_paths_.push_back(string(home) + "/" YEHIA_AD_USERDIR "/plugins");
    ad_paths_.push_back(YEHIA_AD_DIR "/plugins");
  }
  return ad_paths_;
}

const list<string>& PluginManager::arch_indep_paths()
{
  if (ai_paths_.size() > 0)
    return ai_paths_;

  char *env = getenv("YEHIA_ARCH_INDEP_PLUGIN_PATH");
  string env_path = env ? env : "";
  if (env_path.size() != 0)
    ai_paths_ = split(env_path, ':');
  else
  {
    char *home = getenv("HOME"); 
    if (home != 0)
      ai_paths_.push_back(string(home) + "/" YEHIA_AI_USERDIR "/plugins");
    ai_paths_.push_back(YEHIA_AI_DIR "/plugins");
  }
  return ai_paths_;
}

void PluginManager::scan()
{
  //PluginNode old_tree = plugin_tree_.copy();
  
  scan_time_ = true;

  for (list<PluginLoader *>::iterator it = loaders_.begin();
       it != loaders_.end(); ++it)
    (*it)->scan(*this);

  // Must find out anavaible drivers here!
  
  scan_time_ = false; 
}

void PluginManager::plugin_found(const string& name)
{
  if (!scan_time_)
    return;
  
  PluginNode::iterator it = plugin_tree_.find(name);
  if (it == plugin_tree_.end())
  {
    plugin_tree_.insert(name, 0); 
    emit_log(LOG_INFO, "found: " + name);
    plugin_available.emit(name);
  }
}

void PluginManager::release_plugin(const string& id)
{
  PluginNode::iterator it = plugin_tree_.find(id);
  
  if (it != plugin_tree_.end())
  {
    if ((*it).plugin())
    {
      //cout << "set_plugin(0)" << endl;
      (*it).set_plugin(0);
      plugin_unloaded.emit(id);
    }
  }
}

struct ShLibInfo
{
    PluginManager *plugin_manager;
    string name;
    GModule *module;
};

namespace
{

void plugin_destroy(const string& name, ShLibInfo info)
{
  if (name == info.name && info.module)
  {
#if 1
    if (!g_module_close(info.module))
      info.plugin_manager->emit_error(g_module_error());
#endif
    info.plugin_manager->emit_log(ErrorHandler::LOG_DEBUG, 
                                  "unloaded plugin: " + name);
  }
}

}

Plugin *ShLibPluginLoader::load(PluginManager& mgr, const string& name)
{
  using SigC::slot;
  using SigC::bind;
  
  GModule *module = 0;
  Plugin *plugin = 0;
  Plugin *(*init_func)(PluginManager *);
  list<string>::const_iterator it;
  const list<string>& paths = mgr.arch_dep_paths();
  string path(name), realname;
  int pos = 0;

  for (string::size_type i = 0; i < path.size(); i++)
    if (path[i] == '.')
    {
      pos = i;
      path[i] = '/';
    }

  if (pos != 0)
  {
    realname = path.substr(pos + 1);
    path = path.substr(0, pos);
  }
  else
  {
    realname = path;
    path = "";
  }
  
  for (it = paths.begin(); it != paths.end(); ++it)
  {
    string modulepath = (*it) + (path.length() == 0 ? "" : "/" + path);
    string fullname = modulepath + "/" + realname + ".so"; // FIXME: unportable
    if ((module = g_module_open(fullname.c_str(), G_MODULE_BIND_LAZY)) != 0)
      break;
    else
      mgr.set_error(g_module_error());
  }
  
  if (module)
  {
    for (string::size_type i = 0; i < realname.length(); i++)
      if (realname[i] == '-')
        realname[i] = '_';
    
    string funcname = string("yehia_") + realname + "_plugin_init";
    if (!g_module_symbol(module, funcname.c_str(), (gpointer *)&init_func))
      mgr.set_error(g_module_error());
    else if ((plugin = (*init_func)(&mgr)) != 0)
    {
      ShLibInfo info;
      info.plugin_manager = &mgr;
      info.name = name;
      info.module = module;
      mgr.plugin_unloaded.connect(SigC::bind(SigC::slot(&plugin_destroy),
                                              info));
    }
  }
  
  return plugin;
}

namespace
{

void do_scan(PluginManager& mgr, const string& path, const string& basename)
{
  DIR *dir;
  struct dirent *file;

  if ((dir = opendir(path.c_str())) != NULL)
  {
    while ((file = readdir(dir)) != NULL)
    {
      int len;
      string fullname = path + file->d_name;
      struct stat statinfo;
      
      if (file->d_name[0] != '.' &&
          stat(fullname.c_str(), &statinfo) == 0 && S_ISDIR(statinfo.st_mode))
        do_scan(mgr,
                path + file->d_name + "/", basename + file->d_name + ".");
      
      len = strlen(file->d_name);
      if (len > 3 && strcmp(file->d_name + len - 3, ".so") == 0)
        mgr.plugin_found(basename + string(file->d_name).substr(0, len - 3));
    }
    closedir(dir);
  }
}

}

void ShLibPluginLoader::scan(PluginManager& mgr) const
{
  const list<string>& paths = mgr.arch_dep_paths();
  list<string>::const_iterator it;
  
  try
  {
    for (it = paths.begin(); it != paths.end(); ++it)
      do_scan(mgr, (*it) + "/", "");
  }
  // FIXME: do something meaningful here (ErrorHandler?)
  catch (const std::exception& e) { std::cerr << "EXC: " << e.what() << endl; }
  catch (...) { std::cerr << "EXC!!!" << endl; }
}

Plugin::~Plugin()
{
}

} // Yehia
