/* This is for emacs: -*-Mode: C++;-*- */
#if !defined(_INC_YEHIA_IO_H)
#define _INC_YEHIA_IO_H

#include <sigc++/signal.h>
#include <sigcx/dispatch.h>

namespace Yehia
{

class IOException : public std::runtime_error
{
  public:
    IOException(const std::string& msg) : runtime_error(msg) { }
};

class DataSource : virtual public SigC::Object
{
  public:
    typedef unsigned int size_type;
    
    DataSource();
    virtual ~DataSource();
    
    virtual size_type read(void *data, size_type len) throw (IOException) = 0;
    virtual void set_notification(bool do_notify) = 0;

    void attach(const SigC::Slot0<void> ready_cb, 
                const SigC::Slot0<void> exhausted_cb) {
      ready_cb_ = ready_cb;
      exhausted_cb_ =  exhausted_cb;
    }
    void detach() {
      ready_cb_ = SigC::Slot0<void>();
      exhausted_cb_ = SigC::Slot0<void>();
    }
  protected:
    void ready() { ready_cb_(); }
    void exhausted() { exhausted_cb_(); }
  private:
    SigC::Slot0<void> ready_cb_;
    SigC::Slot0<void> exhausted_cb_;
};

class DataSink : virtual public SigC::Object
{
  public:
    typedef unsigned int size_type;
    
    DataSink();
    virtual ~DataSink();
    
    virtual size_type write(const void *data, size_type len) 
      throw (IOException) = 0;
    virtual void set_notification(bool do_notify) = 0;
    
    void attach(const SigC::Slot0<void> ready_cb) {
      ready_cb_ = ready_cb;
    }
    void detach() {  ready_cb_ = SigC::Slot0<void>(); }
  protected:
    void ready() { ready_cb_(); }
  private:
    SigC::Slot0<void> ready_cb_;
};

class CircularBuffer
{
  public:
    typedef unsigned int size_type;
    
    CircularBuffer(size_type bufsz = 1024);
    ~CircularBuffer();
    
    unsigned char *get_ptr() const { return bufget_; }
    unsigned char *put_ptr() const { return bufput_; }
    
    size_type get_size() const { 
      return empty_ ? 0 : bufput_ <= bufget_ ? bufend_ - bufget_ 
        : bufput_ - bufget_; 
    }
    size_type put_size() const { 
      return (empty_ || bufget_ < bufput_) ? bufend_ - bufput_ 
        : bufget_ - bufput_;
    }

    void get_advance(size_type n) { 
      bufget_ += n;
      if (bufget_ == bufend_)
        bufget_ = buffer_;
      if (n > 0)
        empty_ = bufget_ == bufput_;
    }
    void put_advance(size_type n) { 
      bufput_ += n;
      if (bufput_ == bufend_)
        bufput_ = buffer_;
      if (n > 0)
        empty_ = false;
    }
    
    bool empty() { return empty_; }
    bool full() { return !empty_ && bufput_ == bufget_; }
  private:
    unsigned char *buffer_, *bufput_, *bufget_, *bufend_;
    bool empty_;
};

class StackedDataSource : public DataSource
{
  public:
    typedef CircularBuffer::size_type size_type;
    
    StackedDataSource(size_type bufsz = 1024);

    void attach(DataSource& src);
    void detach();
    bool attached() const  { return src_ != 0; }

    virtual void set_notification(bool do_notify);
  protected:
    CircularBuffer buffer;
    bool is_exhausted() const { return exhausted_; }
  private:
    bool notify_;
    bool exhausted_;
    DataSource *src_;
    
    void source_ready();
    void source_exhausted();
};

class DataStream : public SigC::Object
{
    typedef DataSource::size_type size_type;
  public:
    DataStream(DataSource& src, DataSink& sink, int bufsz = 1024);
    ~DataStream();
    
    SigC::Signal0<void> done;
  private:
    DataSource& src_;
    DataSink& sink_;
    CircularBuffer buffer_;
    bool input_exhausted_;
    
    void input_exhausted();
    void input_ready();
    void output_ready();
};

class DispatcherDataSource : public DataSource
{
  public:
    DispatcherDataSource(SigCX::Dispatcher& disp, int fd);
    virtual ~DispatcherDataSource();

    virtual size_type read(void *data, size_type len) throw (IOException);
    virtual void set_notification(bool do_notify);

    SigCX::Dispatcher& dispatcher() { return disp_; }
    const SigCX::Dispatcher& dispatcher() const { return disp_; }
  private:
    void input_handler();
    
    SigCX::Dispatcher& disp_;
    SigCX::Dispatcher::HandlerID hid_;
    int fd_;
};

class DispatcherDataSink : public DataSink
{
  public:
    DispatcherDataSink(SigCX::Dispatcher& disp, int fd);
    virtual ~DispatcherDataSink();

    virtual size_type write(const void *data, size_type len) 
      throw (IOException);
    virtual void set_notification(bool do_notify);
  private:
    void output_handler();
    
    SigCX::Dispatcher& disp_;
    int fd_;
    SigCX::Dispatcher::HandlerID hid_;
};

}

#endif
