/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <glib.h>

#include <sigc++/class_slot.h>

#include "python-script.h"
#include "python-generic.h"

#include "yehia/script/class-builder.h"
#include "yehia/script/traits.h"

namespace Yehia
{

namespace Script
{

using namespace std;

pythonObjectFactory::pythonObjectFactory()
{
  memchunk_ = g_mem_chunk_new("yehia python object mem chunks",
                              sizeof(pythonObject),
                              sizeof(pythonObject) * 256,
                              G_ALLOC_AND_FREE);
  sigc_obj_class_ = 0;
  
  gpointer mem = g_mem_chunk_alloc(memchunk_);
  null_obj_ = new (mem) pythonObject(*this);

  // We must register the following types: bool, long, unsigned long, 
  // double, std::string, Any, std::list<Any>
  
  Py_INCREF((PyObject *)&PyInt_Type);
  register_class(typeid(bool), create_object((PyObject *)&PyInt_Type));

  Py_INCREF((PyObject *)&PyLong_Type);
  register_class(typeid(long), create_object((PyObject *)&PyLong_Type));
  
  Py_INCREF((PyObject *)&PyLong_Type);
  register_class(typeid(unsigned long), 
                 create_object((PyObject *)&PyLong_Type));
  
  Py_INCREF((PyObject *)&PyFloat_Type);
  register_class(typeid(double), create_object((PyObject *)&PyFloat_Type));

  Py_INCREF((PyObject *)&PyString_Type);
  register_class(typeid(std::string), 
                 create_object((PyObject *)&PyString_Type));
  
  Py_INCREF(Py_None);
  register_class(typeid(Any), create_object(Py_None));

  Py_INCREF((PyObject *)&PyList_Type);
  register_class(typeid(std::list<Any>), 
                 create_object((PyObject *)&PyList_Type));

  instance_hash_ = g_hash_table_new(&g_direct_hash, &g_direct_equal);
}

pythonObjectFactory::~pythonObjectFactory()
{
  g_mem_chunk_destroy(memchunk_);
  g_hash_table_destroy(instance_hash_);
}

PyObject *pythonObjectFactory::to_pyobj(const Any& v)
{
  PyObject *pyobj = NULL;
  
  try
  {
    switch (v.typecode())
    {
      case Any::TC_BOOL:
        pyobj = PyInt_FromLong(any_cast<bool>(v));
        break;
      case Any::TC_LONG:
        pyobj = PyInt_FromLong(any_cast<long>(v));
        break;
      case Any::TC_ULONG:
        pyobj = PyLong_FromUnsignedLong(any_cast<unsigned long>(v));
        break;
      case Any::TC_REAL:
        pyobj = PyFloat_FromDouble(any_cast<double>(v));
        break;
      case Any::TC_STRING:
        pyobj = PyString_FromString(any_cast<char *>(v));
        break;
      case Any::TC_LIST:
      {
        std::list<Any> the_list = any_cast<std::list<Any> >(v);
        std::list<Any>::const_iterator it;
        int i;
        
        pyobj = PyTuple_New(the_list.size());
        for (it = the_list.begin(), i = 0; it != the_list.end(); ++it, i++)
        {
          PyObject *pyelement = to_pyobj(*it);
          if (!pyelement)
          {
            pyelement = Py_None;
            Py_INCREF(pyelement);
          }
          PyTuple_SetItem(pyobj, i, pyelement);
        }
        break;
      }
      case Any::TC_SLOT:
      {
        Slot slot = any_cast<Slot>(v);
        pythonObject& pyfunc = dynamic_cast<pythonObject&>(create_genfunc());
        // We can use an empty signature, since the python generic function
        // implementation doesn't care aboute sigs.
        pyfunc.genfunc_interface()->add_method(slot, Signature());

        pyobj = pyfunc.pyobj();
        Py_INCREF(pyobj);
        pyfunc.unreference();
        break;
      }
      case Any::TC_INSTANCE:
      {
        Any::InstanceValue ih = any_cast<Any::InstanceValue>(v);
        pythonObject *instance;
        Object *class_object;
        
        if (ih.instance == 0)
        {
          pyobj = Py_None;
          Py_INCREF(pyobj);
          break;
        }
        
        if ((class_object = find_class(*ih.tinfo)) == 0)
        {
          if (!sigc_obj_class_)
          {
            PyObject *yehiamodule = PyImport_ImportModule("yehia");
            PyObject *yehiadict = yehiamodule ? 
              PyModule_GetDict(yehiamodule) : NULL;
            PyObject *objclass = 
              yehiadict ? PyDict_GetItemString(yehiadict, "Object") : NULL;
        
            sigc_obj_class_ = objclass ? &create_object(objclass) : 0;
            if (!sigc_obj_class_)
              throw Exception("cannot find python object for SigC::Object");
          }
          instance = sigc_obj_class_ ? 
            dynamic_cast<pythonObject *>(
                    &wrap_instance(*sigc_obj_class_, *ih.instance)) : 0;
        }
        else
          instance = dynamic_cast<pythonObject *>(
                  &wrap_instance(*class_object, *ih.instance));
        
        pyobj = instance ? instance->pyobj() : NULL;
        Py_XINCREF(pyobj);
        if (instance)
          instance->unreference();
        break;
      }
      case Any::TC_VOID:
        pyobj = Py_None;
        Py_INCREF(Py_None);
        break;
      default:
        g_assert_not_reached();
    }
  }
  catch (BadAnyCast)
  {
    pyobj = NULL;
  }
  return pyobj;
}

Object& pythonObjectFactory::create_value(const Any& v)
{
  PyObject *pyobj = to_pyobj(v);
  
  if (pyobj)
    return create_object(pyobj);
  else
  {
    null_object().reference();
    return null_object();
  }
}

Object& pythonObjectFactory::create_genfunc()
{
  PyObject *pyfunc = pythonGenericFunction_New();
  
  return create_object(pyfunc);
}

namespace
{

PyObject *py_string_from_pointer(const void *p)
{
  char address[64];
  sprintf(address, "%p", p);
  return PyString_FromString(address);
}

SigC::Object *py_get_instance(PyObject *obj)
{
  SigC::Object *instance;
  PyObject *pyaddr =
    obj ? PyObject_GetAttrString(obj, "__yehiainstance__") : 0;
  char *addr;
  
  addr = pyaddr && PyString_Check(pyaddr) ? PyString_AsString(pyaddr) : 0;
  
  if (!addr || sscanf(addr, "%p", &instance) != 1)
    return 0;

  return instance;
}

PyObject *py_init_instance(PyObject *self, PyObject *args)
{
  SigC::Object *instance = 0;
  Language *lang = LanguageManager::instance().language("python");
  pythonObjectFactory *factory = lang ? 
    dynamic_cast<pythonObjectFactory *>(&lang->factory()) : 0;

  g_return_val_if_fail(factory, NULL);

  if (!PyTuple_Check(args) || PyTuple_Size(args) < 1)
    return NULL;
  else
  {
    self = PyTuple_GetItem(args, 0);
    if (PyInstance_Check(self))
      self = (PyObject *)((PyInstanceObject *)self)->in_class;
  }
  
  PyObject *yehiainit = PyObject_GetAttrString(self, "__yehiainit__");
  PyObject *yehiaargs;
  PyObject *pyinstance;
  
  if (!yehiainit || !PyCallable_Check(yehiainit))
  {
    PyErr_SetString(PyExc_NotImplementedError, "no yehia constructor");
    return NULL;
  }
  if (!PyTuple_Check(args) || PyTuple_Size(args) < 1)
    return NULL;
  
  yehiaargs = PyTuple_New(PyTuple_Size(args) + 2);
  pyinstance = PyTuple_GetItem(args, 0);
  Py_INCREF(self);
  Py_INCREF(pyinstance);
  Script::Object& classobj = factory->create_object(self);
  Script::Object& instobj = factory->create_object(pyinstance);
  PyTuple_SET_ITEM(yehiaargs, 0, 
                   py_string_from_pointer(
                           LanguageManager::instance().language(*factory)));
  PyTuple_SET_ITEM(yehiaargs, 1, 
                   py_string_from_pointer(classobj.namespace_interface()));
  PyTuple_SET_ITEM(yehiaargs, 2, py_string_from_pointer(&instobj));
  for (int i = 3; i < PyTuple_Size(yehiaargs) ; i++)
  {
    PyObject *item = PyTuple_GET_ITEM(args, i - 2);
    Py_INCREF(item);
    PyTuple_SET_ITEM(yehiaargs, i, item);
  }
  
  PyObject *retval = PyObject_CallObject(yehiainit, yehiaargs);
  Py_DECREF(yehiaargs);
  // FIXME: change type of 1st constructor arg to Object to avoid leakin'
  /* classobj->unreference(); */
  instobj.unreference();
  
  if (!retval)
    return NULL;
    
  instance = py_get_instance(retval);
  
  if (instance)
    instance->reference();
  
    // 'move' dictionary 
  self = PyTuple_GET_ITEM(args, 0);
  Py_XDECREF(((PyInstanceObject *)self)->in_dict);
  ((PyInstanceObject *)self)->in_dict = 
    ((PyInstanceObject *)retval)->in_dict;
  Py_XINCREF(((PyInstanceObject *)self)->in_dict);
  
  Py_DECREF(retval);
  
  if (!instance)
    return NULL;
  
  factory->register_instance(instance, self);
  
  Py_INCREF(Py_None);
  return Py_None;
}

PyObject *py_del_instance(PyObject *self, PyObject *args)
{
  SigC::Object *instance;
  Language *lang = LanguageManager::instance().language("python");
  pythonObjectFactory *factory = lang ? 
    dynamic_cast<pythonObjectFactory *>(&lang->factory()) : 0;
  
  g_return_val_if_fail(factory, NULL);
  
  if (!PyTuple_Check(args) || PyTuple_Size(args) != 1)
    return NULL;
  
  if ((instance = py_get_instance(PyTuple_GET_ITEM(args, 0))) == 0)
    return NULL;
  
  instance->unreference();
  
  factory->unregister_instance(instance);
  
  Py_INCREF(Py_None);
  return Py_None;
}

PyObject *py_instance_detach(PyObject *self, PyObject *args)
{
  if (!PyArg_ParseTuple(args, ""))
    return NULL;
  
  WrapObject *instance = dynamic_cast<WrapObject *>(py_get_instance(self));
  
  if (instance)
    instance->detach();

  Py_INCREF(Py_None);
  return Py_None;
}

PyObject *py_instance_getattr(PyObject *self, PyObject *args)
{
  static PyMethodDef detach_method = 
    { "detach__", py_instance_detach, METH_VARARGS, "yehia detach" };
  
  PyObject *pyinstance;
  PyObject *getter_args, *getter, *result;
  std::string name;
  char *attname;
  
  if (!PyArg_ParseTuple(args, "Os", &pyinstance, &attname))
    return NULL;

  if (!PyInstance_Check(pyinstance))
    return NULL;
  else
    self = (PyObject *)((PyInstanceObject *)pyinstance)->in_class;

  if (strcmp(attname, detach_method.ml_name) == 0)
    return PyCFunction_New(&detach_method, pyinstance);

  name = std::string("__yehiaget_") + attname;
  
  if ((getter = PyObject_GetAttrString(self, 
                                       (char *)name.c_str())) == NULL ||
      !PyCallable_Check(getter))
    return NULL;

  getter_args = PyTuple_New(1);
  Py_INCREF(pyinstance);
  PyTuple_SET_ITEM(getter_args, 0, pyinstance);
  
  result = PyObject_CallObject(getter, getter_args);
  Py_DECREF(getter_args);
            
  return result;
}

PyObject *py_instance_setattr(PyObject *self, PyObject *args)
{
  if (!PyTuple_Check(args))
    return NULL;
  if (PyTuple_Size(args) != 3)
    return NULL;

  PyObject *pyinstance = PyTuple_GetItem(args, 0);
  PyInstanceObject *pyinst = (PyInstanceObject *)pyinstance;
  PyObject *pyname = PyTuple_GetItem(args, 1);
  PyObject *pyval = PyTuple_GetItem(args, 2);
  PyObject *setter_args, *setter, *result;
  std::string name;
  
  if (!PyString_Check(pyname))
    return NULL;

  if (!PyInstance_Check(pyinstance))
    return NULL;
  else
    self = (PyObject *)(pyinst->in_class);

  name = std::string("__yehiaset_") + PyString_AsString(pyname);

  if ((setter = PyObject_GetAttrString(self, 
                                       (char *)name.c_str())) == NULL ||
      !PyCallable_Check(setter))
  {
    /* This is copied from the python setattr implementation, since I
     * know no way how to invoke the orignal setattr without invoking
     * ourselves again ;-) */
    if (pyval == NULL)
    {
      int rv = PyDict_DelItem(pyinst->in_dict, pyname);
      if (rv < 0)
      {
        PyErr_Format(PyExc_AttributeError,
                     "%.50s instance has no attribute '%.400s'",
                     PyString_AS_STRING(pyinst->in_class->cl_name),
                     PyString_AS_STRING(pyname));
        return NULL;
      }
    }
    else
      PyDict_SetItem(pyinst->in_dict, pyname, pyval);
    
    Py_INCREF(Py_None);
    return Py_None;
  }  
  
  if (pyval == NULL)
    return NULL;
  
  setter_args = PyTuple_New(2);
  Py_INCREF(pyinstance);
  PyTuple_SET_ITEM(setter_args, 0, pyinstance);
  Py_INCREF(pyval);
  PyTuple_SET_ITEM(setter_args, 1, pyval);
  
  result = PyObject_CallObject(setter, setter_args);
  Py_DECREF(setter_args);
            
  return result;
}

} // namespace


Object& pythonObjectFactory::create_class(const Signature& supers)
{
  static PyMethodDef methods[] = {
    { "__init__", py_init_instance, METH_VARARGS, "yehia constructor" },
    { "__del__", py_del_instance, METH_VARARGS, "yehia destructor" },
    { "__getattr__", py_instance_getattr, METH_VARARGS, "yehia get attribute" },
    { "__setattr__", py_instance_setattr, METH_VARARGS, "yehia set attribute" },
    { NULL }
  };
  
  PyObject *pymethods[sizeof(methods) / sizeof(PyMethodDef) - 1];
  PyObject *pysupers, *dict, *name, *pyclass;
  int i = 0;
  
  // Create and fill tuple of superclasses
  pysupers = PyTuple_New(supers.size());
  for (Signature::const_iterator it = supers.begin(); it != supers.end(); 
       ++it, i++)
  {
    PyObject *pysuper = dynamic_cast<const pythonObject&>(*find_class(**it)).pyobj();
    PyTuple_SET_ITEM(pysupers, i, pysuper);
    Py_INCREF(pysuper);
  }
  
  dict = PyDict_New();
  name = PyString_FromString("unnamed yehia class");
  
  // insert methods
  for (PyMethodDef *def = methods; def->ml_name != NULL; def++)
  {
    PyObject *func = PyCFunction_New(def, NULL);
    PyObject *method = PyMethod_New(func, NULL, Py_None);
    PyDict_SetItemString(dict, def->ml_name, method);
    Py_DECREF(func);
    Py_DECREF(method);
    pymethods[def - methods] = method;
  }

  pyclass = PyClass_New(pysupers, dict, name);

  Py_DECREF(dict);
  Py_DECREF(name);
  Py_DECREF(pysupers);
  
  // Set class pointer
  for (PyMethodDef *def = methods; def->ml_name != NULL; def++)
  {
    PyMethod_GET_CLASS(pymethods[def - methods]) = pyclass;
  }
  
  return create_object(pyclass);
}

Object& pythonObjectFactory::create_namespace()
{
  return create_object(PyModule_New("__yehia_unnamed"));
}

Object& pythonObjectFactory::wrap_instance(const Object& klass,
                                           SigC::Object& obj)
{
  PyObject *instance = 0;
  PyObject *pyaddr = 0;
  char address[64];
  const pythonObject *scriptclass = &dynamic_cast<const pythonObject&>(klass);
  
  sprintf(address, "%p", &obj);
  pyaddr = PyString_FromString(address);
  
  if ((instance = (PyObject *)g_hash_table_lookup(instance_hash_, &obj)))
    Py_INCREF(instance);
  else
  {
    instance = PyInstance_NewRaw(scriptclass->pyobj(), NULL);
  
    if (!instance)
    {
      Py_XDECREF(pyaddr);
      throw Script::Exception("pythonObjectFactory::wrap_instance failed");
    }
    
    PyObject_SetAttrString(instance, "__yehiainstance__", pyaddr);
  
    obj.reference();
  }
  Py_XDECREF(pyaddr);
  
  return create_object(instance);
}

pythonObject& pythonObjectFactory::create_object(PyObject *pyobj)
{
  gpointer mem = g_mem_chunk_alloc(memchunk_);
  return *(new (mem) pythonObject(*this, pyobj));
}

Object& pythonObjectFactory::null_object()
{
  Py_INCREF(null_obj_->pyobj());
  return *null_obj_;
}

void pythonObjectFactory::destroy(Object& obj)
{
  pythonObject *pyobj = &dynamic_cast<pythonObject&>(obj);
  pyobj->~pythonObject();
  g_mem_chunk_free(memchunk_, pyobj);
}

Object *pythonObjectFactory::call_pyobj(const ObjectContainer& args, 
                                        PyObject *callable)
{
  PyObject *pyargs = PyTuple_New(args.size());
  PyObject *pyresult;
  ObjectContainer::const_iterator it;
  int i;
  
  for (i = 0, it = args.begin(); it != args.end(); ++it, i++)
  {
    pythonObject *pyoarg = dynamic_cast<pythonObject *>(*it);
    PyObject *pyarg = pyoarg ? pyoarg->pyobj() : NULL;
    if (pyarg == NULL)
    {
      pyarg = Py_None;
    }
    Py_INCREF(pyarg);
    PyTuple_SET_ITEM(pyargs, i, pyarg);
  }
  
  pyresult = PyObject_CallObject(callable, pyargs);
  
  Py_DECREF(pyargs);
  
  return &create_object(pyresult);
}

Any pythonObjectFactory::pyobj_to_any(PyObject *pyobj)
{
  if (PyInstance_Check(pyobj))
  {
    Any::InstanceValue ih;
    if ((ih.instance = py_get_instance(pyobj)) != 0)
    {
      Py_XINCREF((PyObject *)((PyInstanceObject *)pyobj)->in_class);
      // XXX: how do we get the class here *ARGL*
      //ih.class_object = 
      //create_object((PyObject *)((PyInstanceObject *)pyobj)->in_class);
      ih.tinfo = &typeid(Any);
      
      return Any(ih);
    }
  }
  if (PyString_Check(pyobj))
    return Any(std::string(PyString_AsString(pyobj)));
  
  if (PySequence_Check(pyobj))
  {
    std::list<Any> anylist;
    for (int i = 0; i < PyObject_Length(pyobj); i++)
      anylist.push_back(pyobj_to_any(PySequence_GetItem(pyobj, i)));
    return Any(anylist);
  }
  if (PyInt_Check(pyobj))
    return Any(PyInt_AsLong(pyobj));
  
  if (PyLong_Check(pyobj))
    return Any(PyLong_AsLong(pyobj));

  if (PyCallable_Check(pyobj))
  {
    // FIXME: refcounting
    Py_XINCREF(pyobj);
    return Any(Any::SlotValue(lang_convert(SigC::bind(SigC::slot_class(*this, &pythonObjectFactory::call_pyobj), pyobj), *LanguageManager::instance().language("python")), Signature()));
  }  
  return Any();
}

pythonObject::pythonObject(pythonObjectFactory& factory, PyObject *pyobj)
    : Script::Object(factory)
{
  if (pyobj == 0)
  {
    pyobj = Py_None;
    Py_INCREF(pyobj);
  }
  
  pyobj_ = pyobj;
}

pythonObject::~pythonObject()
{
  Py_XDECREF(pyobj_);
}

inline PyObject *pythonObject::pyobj() const
{
  if (!pyobj_)
    throw Exception("oops");

  return pyobj_;
}

inline bool pythonObject::is_null()
{
  return pyobj_ == Py_None;
}

inline pythonObject& pythonObject::operator=(PyObject *pyobj)
{
  Py_XDECREF(pyobj_);
  Py_XINCREF(pyobj);
  pyobj_ = pyobj;
  return *this;
}

inline void pythonObject::ensure(bool cond)
{
  if (!cond) throw Exception("hooo");
}

Namespace *pythonObject::namespace_interface()
{
  return this;
}

Instance *pythonObject::instance_interface()
{
  return PyInstance_Check(pyobj_) ? this : 0;
}

Function *pythonObject::function_interface()
{
  return PyCallable_Check(pyobj_) ? this : 0;
}

GenFunc *pythonObject::genfunc_interface()
{
  return pythonGenericFunction_Check(pyobj_) ? this : 0;
}

Class *pythonObject::class_interface()
{
  return PyClass_Check(pyobj_) ? this : 0;
}

Any pythonObject::value() const
{
  return factory().pyobj_to_any(pyobj_);
}

SigC::Object& pythonObject::object()
{
  return *py_get_instance(pyobj_);
}

Object& pythonObject::class_object()
{
  throw 0;
}

Object *pythonObject::call(const ParamList& args)
{
  PyObject *pyargs, *item = Py_None;
  ParamList::const_iterator it;
  int i;
  
  if (!PyCallable_Check(pyobj_))
    return 0;
  
  pyargs = args.size() > 0 ? PyTuple_New(args.size()) : NULL;
  for (i = 0, it = args.begin(); it != args.end(); ++i, ++it)
  {
    pythonObject *pyobj = dynamic_cast<pythonObject *>(*it);
    item = pyobj ? pyobj->pyobj() : 0;
    if (!item)
      break;
    Py_INCREF(item);
    PyTuple_SetItem(pyargs, i, item);
  }
  
  if (!item)
  {
    Py_XDECREF(pyargs);
    return 0;
  }
  
  PyObject *pyresult = PyObject_CallObject(pyobj_, pyargs);
  Py_XDECREF(pyargs);

  if (pyresult == NULL)
  {
    PyObject *type, *value, *tracebk, *errstr;
    
    PyErr_Fetch(&type, &value, &tracebk);
    PyErr_NormalizeException(&type, &value, &tracebk);

    if (type && PyErr_GivenExceptionMatches(type, PyExc_Exception) &&
        value && ((errstr = PyObject_Str(value)) != 0))

    {
      std::string errmsg = PyString_AsString(errstr);
      
      Py_DECREF(type);
      Py_DECREF(value);
      Py_DECREF(errstr);

      throw Exception(errmsg);
    }
    
    Py_XDECREF(type);
    Py_XDECREF(value);
    Py_XDECREF(tracebk);
    
    throw Exception("unable to call python method");
  }
    
  Py_INCREF(pyresult);
  
  return &factory().create_object(pyresult);
}

Object *pythonObject::member(const std::string& name)
{
  PyObject *item = PyObject_GetAttrString(pyobj_, 
                                          const_cast<char *>(name.c_str()));
  if (item)
  {
    Py_INCREF(item);
    return &factory().create_object(item);
  }
  else
  {
    PyErr_Clear();
    return 0;
  }
}

namespace {

void do_nothing()
{
}

}

void pythonObject::insert(const std::string& name, Script::Object& object)
{
  pythonObject *obj = &dynamic_cast<pythonObject&>(object);
  PyObject *sysmdict = PyImport_GetModuleDict();
  
  // insert module?
  if (PyModule_Check(obj->pyobj_))
  {
    string fullname = 
      is_null() ? name : string(PyModule_GetName(pyobj_)) + "." +  name;
    char *fullname_cstr = const_cast<char *>(fullname.c_str());
    
    // make  an entry in the modules dict
    PyDict_SetItemString(sysmdict, fullname_cstr, obj->pyobj_);
    
    // set insert module name, since this wasn't known at the start
    PyObject_SetAttrString(obj->pyobj_, "__name__", 
                           PyString_FromString(fullname_cstr));
    PyImport_AppendInittab(g_strdup(fullname_cstr), &do_nothing);
#if 0
    for (int i = 0; PyImport_Inittab[i].name != NULL; i++)
      fprintf(stderr, " entry: (%s, %p)\n", PyImport_Inittab[i].name, 
              PyImport_Inittab[i].initfunc);
#endif
  }
  
  if (PyClass_Check(obj->pyobj()))
  {
    PyObject *pyname = PyString_FromString(name.c_str());
    PyObject_SetAttrString(obj->pyobj(), "__name__", pyname);
    if (PyModule_Check(pyobj_))
    {
      char *ourname = PyModule_GetName(pyobj_);
      if (!ourname)
        throw Exception("how");
      pyname = PyString_FromString(ourname);
      PyObject_SetAttrString(obj->pyobj(), "__module__", pyname);
    }
  }
  
  if (!is_null())
    PyObject_SetAttrString(pyobj_, 
                           const_cast<char *>(name.c_str()), obj->pyobj());
}

void pythonObject::add_method(const Slot& slot, const Signature& sig)
{
  Language *lang = LanguageManager::instance().language("python");
 
  g_return_if_fail(lang != 0);
  
  pythonGenericFunction_AddSlot(pyobj(), lang_convert(slot, *lang));
}

void pythonObject::set_constructor(Object& genfunc)
{
  if (!PyClass_Check(pyobj_))
    return;
  
  PyObject *dict = ((PyClassObject *)pyobj_)->cl_dict;
  PyObject *pyconstr = dynamic_cast<pythonObject&>(genfunc).pyobj();
  
  PyDict_SetItemString(dict, "__yehiainit__", pyconstr);
}

void pythonObject::add_genfunc(const std::string& name, Object& genfunc)
{
  if (!PyClass_Check(pyobj_))
    return;

  pythonObject& pyfunc = dynamic_cast<pythonObject&>(genfunc);
  PyObject *method = PyMethod_New(pyfunc.pyobj(), NULL, pyobj_);
  
  PyObject_SetAttrString(pyobj_, (char *)name.c_str(), method);
  Py_DECREF(method);
}

void pythonObject::add_class_genfunc(const std::string& name, Object& genfunc)
{
  if (!PyClass_Check(pyobj_))
    return;

  pythonObject& pyfunc = dynamic_cast<pythonObject&>(genfunc);

  PyObject_SetAttrString(pyobj_, (char *)name.c_str(), pyfunc.pyobj());
}

void pythonObject::add_setter(const std::string& name, Object& genfunc)
{
  if (!PyClass_Check(pyobj_))
    return;

  pythonObject& pyfunc = dynamic_cast<pythonObject&>(genfunc);

  PyObject_SetAttrString(pyobj_, (char *)("__yehiaset_" + name).c_str(), 
                         pyfunc.pyobj());
}

void pythonObject::add_getter(const std::string& name, Object& genfunc)
{
  if (!PyClass_Check(pyobj_))
    return;

  pythonObject& pyfunc = dynamic_cast<pythonObject&>(genfunc);

  PyObject_SetAttrString(pyobj_, (char *)("__yehiaget_" + name).c_str(), 
                         pyfunc.pyobj());
}

} // namespace Script

}
