#include <stdlib.h>
#include <stdio.h>

#include <X11/xpm.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include "../wmgeneral/wmgeneral.h"
#include "wmcb.h"
#include "wmcb-master.xpm"

void usage()
{
    fprintf(stderr, "\nwmcb - manipulate cut buffers.\n");
    fprintf(stderr, "usage:\n");
    fprintf(stderr, "-display <display name>\n");
    fprintf(stderr, "-h                       this help screen\n");
    fprintf(stderr, "-v                       print version\n");
    fprintf(stderr, "-n <buffers>             number of buffers, max. %d\n",
	    MAX_BUFFERS);
    fprintf(stderr, "-f <font>                name of font\n\n");
}

void init(int argc,char *argv[]) 
{
    XGCValues my_gcv;
    unsigned long my_gcm;
    XWindowAttributes attributes;
    int i;
    Font f;
    char *arg;
    char *font_name="6x13";
    
    /*parse args*/
    for (i=1; i<argc; i++) 
    {
	arg = argv[i];
	if (*arg=='-')
	{
	    arg++;
	    switch(*arg)
	    {
	     case 'd' :
		if (strcmp(arg, "display")) 
		{
		    usage();
		    exit(1);
		}
		break;
	     case 'v':
		fprintf(stderr,"wmcb %s\n",WMCB_VERSION);
		exit(0);
		break;
	     case 'f':
		arg++;
		if(strlen(arg)>0)
		  font_name=arg;
		else
		  if(i<(argc-1))
		{
		    i++;
		    font_name=argv[i];
		}
		else
		{
		    usage();
		    exit(1);
		}
		break;
	     case 'n':
		arg++;
		if(strlen(arg)>0)
		  n_buffers=atoi(arg);
		else
		  if(i<(argc-1))
		{
		    i++;
		    n_buffers=atoi(argv[i]);
		}
		else
		{	
		    usage();
		    exit(1);
		}
		if(n_buffers>MAX_BUFFERS)
		{
		    usage();
		    exit(1);
		}
		break;
	     case 'h':
	     default:
		usage();
		exit(0);
	    }
	}
    }
    
    createXBMfromXPM(wmcb_mask_bits,wmcb_master_xpm,MASK_SIZE,
		     MASK_SIZE);
    openXwindow(argc,argv,wmcb_master_xpm,wmcb_mask_bits,MASK_SIZE,
		MASK_SIZE);
    
    XSelectInput(display, Root, PropertyChangeMask | StructureNotifyMask);

    for(i=0;i<2;i++)
      AddMouseRegion(i,pos[i].x,pos[i].y,pos[i].x+prs[i].w,pos[i].y+prs[i].h);
    AddMouseRegion(2, cb.x, cb.y, cb.w, cb.h);
    
    XGetWindowAttributes(display, Root, &attributes);

    my_gcm = GCForeground | GCBackground | GCFont; /*| GCGraphicsExposures;
    my_gcv.graphics_exposures = 0;*/

    color1.red = 0x2000;
    color1.green = 0x2000;
    color1.blue = 0x2000;
    color1.pixel = 0;
    if (!XAllocColor(display, attributes.colormap, &color1)) 
    {	
	fprintf(stderr, "wmcb.app: can't allocate color.\n");
    }
    
    color2.red = 0x2000;
    color2.green = 0xb200;
    color2.blue = 0xae00;
    color2.pixel = 0;
    if (!XAllocColor(display, attributes.colormap, &color2)) 
    {	
	fprintf(stderr, "wmcb.app: can't allocate color.\n");
    }
    
    f=XLoadFont(display, font_name);
    font_info = XQueryFont(display,f);
    
    my_gcv.background = color1.pixel;
    my_gcv.foreground = color2.pixel;
    my_gcv.font = font_info->fid;

    norm_gc = XCreateGC(display, Root, my_gcm, &my_gcv);
    
    my_gcv.background = color2.pixel;
    my_gcv.foreground = color1.pixel;
    my_gcv.font = font_info->fid;
    
    inv_gc = XCreateGC(display, Root, my_gcm, &my_gcv);
    
    bn_font = XLoadQueryFont(display, "6x13");
    my_gcv.background = color1.pixel;
    my_gcv.foreground = color2.pixel;
    my_gcv.font = bn_font->fid;
    
    bn_gc = XCreateGC(display, Root, my_gcm, &my_gcv);    
}

void draw_buffer(int buf,int sbuf)
{
    char *cut_buffer,*pos,*ppos;
    int n_bytes;
    int length;
    int y=0;
    int cols,rows,height;
    
    cols = cb.w/font_info->max_bounds.width;
    height = font_info->ascent + font_info->descent;
    rows = cb.h/height;
    pos=ppos=cut_buffer=XFetchBuffer(display,&n_bytes,buf);

    if(buf==sbuf)
      XFillRectangle(display,wmgen.pixmap,norm_gc,cb.x,cb.y,cb.w,cb.h);
    else
      XFillRectangle(display,wmgen.pixmap,inv_gc,cb.x,cb.y,cb.w,cb.h);
    
    while (ppos < cut_buffer + n_bytes)
    {
	if ((*ppos == '\n') && (y<rows)) 
	{
	    length=ppos-pos;
	    if(length>cols)
	      length=cols;

	    if(buf==sbuf)
	      XDrawString(display,wmgen.pixmap,inv_gc,cb.x,
			cb.y+(height*y)+font_info->ascent,pos,length);
	    else
	      XDrawString(display,wmgen.pixmap,norm_gc,cb.x,
			  cb.y+(height*y)+font_info->ascent,pos,length);
	    pos = ppos + 1;
	    y++;
	}
	ppos++;
    }
    
    if(y<rows)
    {
	length=ppos-pos;
	if(length>cols)
	  length=cols;
	if(buf==sbuf)
	  XDrawString(display,wmgen.pixmap,inv_gc,cb.x,
		      cb.y+(height*y)+font_info->ascent,pos,length);
	else
	  XDrawString(display,wmgen.pixmap,norm_gc,cb.x,cb.y+(height*y)+
		      font_info->ascent,pos,length);
    }
    
    XFree(cut_buffer);
}

void button_press(int b, int buf)
{
    char tbuf[8];

    sprintf(tbuf,"%1d",buf);

    XFillRectangle(display,wmgen.pixmap,inv_gc,bn.x,bn.y,bn.w,bn.h);
    XDrawString(display,wmgen.pixmap,bn_gc,bn.x+(bn.w/2)-
		(bn_font->max_bounds.width/2),
		bn.y+bn.h-bn_font->descent,tbuf,1);
    if(!(b<0))
      copyXPMArea(prs[b].x,prs[b].y,prs[b].w,prs[b].h,pos[b].x,pos[b].y);
}

int selection_notify(int buf,XEvent event)
{
    unsigned char* data;
    Atom act_type;
    int format;
    unsigned long n_bytes;
    unsigned long remaining;
    int ret;

    ret=XGetWindowProperty(display, event.xselection.requestor,
			   event.xselection.property, 0L,10000000L,True,
			   AnyPropertyType,&act_type, &format, &n_bytes,
			   &remaining,&data);
    
    if (ret != Success)
      printf("wmcb: XGetWindowProperty failed in selection_notify\n");
    else
    {
	if (act_type != XA_STRING)
	{
	    printf("wmcb: Selection type different than requested.\n");
	    return -1;
	}
	else
	{
	    XStoreBuffer(display,data,n_bytes,buf);
	}
	XFree(data);
    }
    return 0;
}

void selection_request(int buf,XEvent event)
{
    char *cut_buffer;
    int n_bytes;
    XSelectionEvent sel_ev;
    int ret;
    Atom supported_targets[3];
    Atom multiple;
    
#ifdef DEBUG
    char *atom_name;
    
    atom_name = XGetAtomName(display,event.xselectionrequest.selection);
    printf("wmcb: Requested selection is %s %d.\n",atom_name,
	   event.xselectionrequest.selection);    
    XFree(atom_name);
    
    atom_name = XGetAtomName(display,event.xselectionrequest.target);
    printf("wmcb: Requested target is %s %d.\n",atom_name,event.xselectionrequest.target);    
    XFree(atom_name);
    
    atom_name = XGetAtomName(display,event.xselectionrequest.property);
    printf("wmcb: Requested property is %s.\n",atom_name);    
    XFree(atom_name);    
#endif
    
    multiple=XInternAtom(display,"MULTIPLE",False);

    supported_targets[0]=XInternAtom(display,"TARGETS",False);
    supported_targets[1]=XA_STRING;
    supported_targets[2]=XInternAtom(display,"TEXT",False);
    
    if((event.xselectionrequest.target == XA_STRING) ||
       (event.xselectionrequest.target == supported_targets[2]))
    {
	if(event.xselectionrequest.property == None)
	{
	    printf("wmcb: Communicating with an obsolete client.\n");
	    sel_ev.property = event.xselectionrequest.target;
	}
	else
	  sel_ev.property = event.xselectionrequest.property;
	
	cut_buffer=XFetchBuffer(display,&n_bytes,buf);
	selection_buffer=realloc(selection_buffer,n_bytes);
	memcpy(selection_buffer,cut_buffer,n_bytes);
	XFree(cut_buffer);
	
	ret=XChangeProperty(display,event.xselectionrequest.requestor,
			    sel_ev.property,event.xselectionrequest.target,
			    8,PropModeReplace,selection_buffer,n_bytes);
	
	switch(ret)
	{
	 case BadAlloc:
	 case BadAtom:
	 case BadMatch:
	 case BadValue:
	 case BadWindow:
	    printf("wmcb: XChangeProperty Failed\n");
	}
    }
    else
    {
	if(event.xselectionrequest.target == supported_targets[0])
	{
	    sel_ev.property = event.xselectionrequest.property;

	    ret=XChangeProperty(display,event.xselectionrequest.requestor,
				sel_ev.property,XA_ATOM,32,PropModeReplace,
				(unsigned char *) supported_targets,3);
	    
	    switch(ret)
	    {
	     case BadAlloc:
	     case BadAtom:
	     case BadMatch:
	     case BadValue:
	     case BadWindow:
		printf("wmcb: XChangeProperty Failed\n");
	    }
	}
	else
	{
	    if(event.xselectionrequest.target == multiple)
	    {
		printf("wmcb: Sorry, no full ICCCM compliance yet.\n");
		sel_ev.property = None;
	    }
	    else
	    {
		sel_ev.property = None;
	    }
	}
    }

    sel_ev.display = event.xselectionrequest.display;
    sel_ev.selection = event.xselectionrequest.selection;
    sel_ev.target = event.xselectionrequest.target;
    sel_ev.type = SelectionNotify;       
    sel_ev.requestor = event.xselectionrequest.requestor;
    sel_ev.time  = event.xselectionrequest.time;
    sel_ev.send_event = True;
    
    ret=XSendEvent(display, event.xselectionrequest.requestor,
		   False,0L, (XEvent*) &sel_ev);
    if ((ret==BadValue) || (ret==BadWindow))
      printf("wmcb: SendEvent failed\n");
}

void kludge(int nbuf) /*see ICCCM sect 3*/
{
    int i;
    int n_bytes;
    char *data;
    
    for(i=0;i<nbuf;i++)
    {
	data=XFetchBuffer(display,&n_bytes,i);
	if(n_bytes==0)
	  XStoreBuffer(display,"",0,i);
	XFree(data);
    }
}

int main(int argc,char *argv[]) 
{    
    XEvent Event;
    
    init(argc,argv);
    
    button_press(-1,0);

    kludge(n_buffers);
    
    draw_buffer(0,-1);

    while(1)
    {
	XNextEvent(display, &Event);
	switch (Event.type) 
	{
	 case Expose:
	    RedrawWindow();
	    break;
	 case DestroyNotify:
	    XCloseDisplay(display);
	    exit(0);
	    break;
	 case SelectionClear:
	    selected_buffer=-1;
	    draw_buffer(cur_buffer,selected_buffer);	    
	    RedrawWindow();
	    break;
	 case SelectionNotify:
	    selection_notify(cur_buffer,Event);
	    break;
	 case SelectionRequest:
	    selection_request(selected_buffer,Event);
	    break;
	 case PropertyNotify:
	    draw_buffer(cur_buffer,selected_buffer);
	    RedrawWindow();
	    break;
	 case ButtonPress:
	    region = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);

	    switch(region) 
	    {
	     case 0:
		cur_buffer--;
		if(cur_buffer<0)
		  cur_buffer=n_buffers-1;
		button_press (0, cur_buffer);
		break;
	     case 1:
		cur_buffer++;
		if(cur_buffer>=n_buffers)
		  cur_buffer=0;
		button_press (1, cur_buffer);
		break;
	     case 2:
		switch (Event.xbutton.button)
		{
		 case 1:
		    if(cur_buffer!=selected_buffer) /*ICCCM sect 2.2, page 7*/
		    {
			XSetSelectionOwner(display, XA_PRIMARY, win,
					   Event.xbutton.time);
			if (XGetSelectionOwner(display,XA_PRIMARY) == win)
			{
			    selected_buffer=cur_buffer;
			    wmcb_stamp=Event.xbutton.time;
			}
			else
			{
			    printf("wmcb: Could not get selection ownership.\n");
			    selected_buffer=-1;
			}
		    }
		    break;
		 case 2:
		    if(Event.xbutton.state & ShiftMask)
		      XStoreBuffer(display, "", 0, cur_buffer);
		    else
		      XConvertSelection(display, XA_PRIMARY, XA_STRING,
					XA_STRING, win, 
					Event.xbutton.time);
		    break;
		 case 3:
		    if(Event.xbutton.state & ShiftMask)
		      XRotateBuffers(display, -1);
		    else
		      XRotateBuffers(display, 1);
		    break;
		 default:
		    break;
		}
		break;
	    }
	    draw_buffer(cur_buffer,selected_buffer);
	    RedrawWindow();
	    break;
	 case ButtonRelease:
/*	    i = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);*/
	    if ((region == 0) || (region == 1))
	      copyXPMArea(rel[region].x,rel[region].y,rel[region].w,
			  rel[region].h,pos[region].x,pos[region].y);
	    RedrawWindow();
	    break;
	}
    }
    return 0;
}
