/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "terminal.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"

namespace wftk {


Terminal::Terminal(const std::string& text, const Font &font, bool readonly, bool fill):
  MultiLineEdit(text, font, readonly, fill),
  scrollTo_(65535),
  currentLine_(65535)
{
  if(textFont_.getHeight())
    allLines_ = height()/textFont_.getHeight();  
  else
    allLines_ = 1;

  getResourceBackground("terminal");
}

  
Terminal::~Terminal()
{
}
  
unsigned int Terminal::linesCount() const
{
  return allLines_;
}
 
unsigned int Terminal::currentLine() const
{
  return currentLine_;
}

void Terminal::updateText()
{  
      /* recalculate the number of lines to be able to display
       * in case we have been resized
       */
      unsigned int max_lines = height()/textFont_.getHeight();
      
      /* layout the text ...
       * MultiLineEdit::breakText will create array of lines 
       * containing max maxLinesToStore_ + max_lines
       */
      std::vector<std::string> all_lines;
      MultiLineEdit::breakText(text_, all_lines, maxLinesToStore_ + max_lines);
      
      /* how many lines at all ?
       */
      allLines_ = all_lines.size() - max_lines;

      /* now check for offset
	 offset_ = 0, means same behavior like MultiLineEdit. 
      */
      std::vector<std::string> lines;
      if(all_lines.size() > max_lines)
	{
	  /* we cannot display all text lines (some are hidden)
	   */
	  if(scrollTo_ != currentLine_)
	    {
	      /* if there is a request to scroll ...
	       * limit it to the possible maximum
	       */
	      if(scrollTo_ > all_lines.size() - max_lines)
		scrollTo_ = all_lines.size() - max_lines; 
	    }
	  else
	    {
	      /* if there is no request update the 
	       * value to reflect the new situation
	       */
	      scrollTo_ = all_lines.size() - max_lines; 
	    }

	  /* if scroll request update current pointer 
	   * and emit signal
	   */
	  if(scrollTo_ != currentLine_)
	    {
	      currentLine_ = scrollTo_;
	      scrolled.emit();
	    }
	  /* copy the lines we want to display into lines */
	  for(unsigned n=currentLine_; n < currentLine_+max_lines; n++)
	    lines.push_back(all_lines[n]);
	}
      else
	{
	  /* all lines fit on the display .. so just show all ;) */
	  if(currentLine_)
	    {
	      currentLine_ = 0;
	    }
	  lines = all_lines;
	}

      Debug out(Debug::TEXT_WIDGETS);
      for(std::vector<std::string>::const_iterator itr = lines.begin();
        itr != lines.end(); ++itr)
	  out << *itr << Debug::endl;
      out << "----------------------------" << Debug::endl;

      /* let superclass render the text */
      MultiLineEdit::renderTextLines(lines);
}

void Terminal::scrollTo(int line)
{
  if(line < 0)
    return;

  if((unsigned)line != currentLine_)
    {
      scrollTo_ = line;
      textUpdate();
    }
}

void Terminal::scrollUp(int lines)
{
  if(lines <= 0 || scrollTo_ == 0)
    return;

  if(currentLine_ < (unsigned)lines)
    scrollTo_ = 0;
  else
    scrollTo_ = currentLine_ - lines;

  textUpdate();
}
  
void Terminal::scrollDown(int lines)
{  
  if(lines <= 0)
    return;
  scrollTo_ = currentLine_ + lines;

  textUpdate();
}
 
void Terminal::scroll2Top()
{
  if(scrollTo_ != 0)
    {
      scrollTo_ = 0;
      textUpdate();
    }
}  

void Terminal::scroll2Bottom()
{
  if(scrollTo_ != allLines_)
    {
      scrollTo_ = allLines_;
      textUpdate();
    }
}
 
}

