/*
  libwftk - Worldforge Toolkit - a widget library

  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.


  The read_png() routine has been copied from SDL_image-0.9.tar.gz
      IMGLIB:  An example image loading library for use with SDL
      Copyright (C) 1999-2002  Sam Lantinga
      Changes: 
        - replace IMG_error by std::cerr
  
  The save_png() routine was originally created by  
  Philippe Lavoie <lavoie@zeus.genie.uottawa.ca> on 2 November 1998

  here is its copyright notice:
=============================================================================
        File: SDL_png.c
     Purpose: A PNG loader and saver for the SDL library      
    Revision: 
  Created by: Philippe Lavoie          (2 November 1998)
              lavoie@zeus.genie.uottawa.ca
 Modified by: 

 Copyright notice:
          Copyright (C) 1998,1999 Philippe Lavoie
 
          This library is free software; you can redistribute it and/or
          modify it under the terms of the GNU Lesser General Public
          License as published by the Free Software Foundation; either
          version 2.1 of the License, or (at your option) any later version.
 
          This library is distributed in the hope that it will be useful,
          but WITHOUT ANY WARRANTY; without even the implied warranty of
          MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
          Lesser General Public License for more details.
 
          You should have received a copy of the GNU Lesser General Public
          License along with this library; if not, write to the Free
          Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, SA.

    Comments: The load and save routine are basically the ones you can find
             in the example.c file from the libpng distribution.

=============================================================================


  Modifications by Karsten Laux:
  * replaced SDL_SetError by std::cerr 
  * filehandling and writing merged to one function
  * moved vars declaration beneath setjmp to get rid of compiler warnings
*/


#include "surface.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "color.h"
 
#include "debug.h"

#include <iostream>
#include <cstdlib>
#include <cassert>
#include <cctype>
#include <map>

#include <png.h>
#include <SDL/SDL_endian.h>

namespace wftk {

static SDL_Surface* read_png(const char* filename)
{
  FILE *fp;
  fp = fopen(filename, "rb");
  if( fp == 0)
    {
      Debug::channel(Debug::DRAWING) << "     ! Couldn't open " << filename << " for reading." 
	     << Debug::endl;
      return 0;
    }
  else
    {
      Debug::channel(Debug::DRAWING) << "     * reading file " << filename << Debug::endl;
    }

  png_structp png_ptr = NULL;
  png_infop info_ptr = NULL; 

  SDL_Surface *surface;
  
  png_uint_32 width, height;
  int bit_depth, color_type, interlace_type;
  Uint32 Rmask;
  Uint32 Gmask;
  Uint32 Bmask;
  Uint32 Amask;
  SDL_Palette *palette;
  png_bytep *row_pointers = NULL;
  unsigned int row, i;
  int ckey = -1;
  png_color_16 *transv;


  png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, (png_voidp)0,0,0);

  if (png_ptr == 0){
    std::cerr << "     ! Couldn't allocate the memory for a PNG info structs." << std::endl;
		fclose(fp);
    return 0;
  }
  
  /* Allocate/initialize the memory for image information. */
  info_ptr = png_create_info_struct(png_ptr);
  if (!info_ptr)
    {
      std::cerr << "      ! Couldn't create the image information for a PNG file" << std::endl;
      png_destroy_read_struct(&png_ptr, (png_infopp)0, (png_infopp)0);
			fclose(fp);
      return 0;
    }

  /* Set error handling if you are using the setjmp/longjmp method (this is
   * the normal method of doing things with libpng).  REQUIRED unless you
   * set up your own error handlers in the png_create_read_struct() earlier.
   */
//  if (setjmp(png_ptr->jmpbuf))
	if (setjmp(png_jmpbuf(png_ptr)))
    {
      /* Free all of the memory associated with the png_ptr and info_ptr */
      png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
      std::cerr << "     ! Error reading the PNG file."<< std::endl;
      /* If we get here, we had a problem reading the file */
			fclose(fp);
      return 0;
   }
  

  /* Set up the input control if you are using standard C streams */
  png_init_io(png_ptr, fp); 

  /* Read PNG header info */
  png_read_info(png_ptr, info_ptr);
  png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
	       &color_type, &interlace_type, NULL, NULL);


  /* tell libpng to strip 16 bit/color files down to 8 bits/color */
  png_set_strip_16(png_ptr) ;
  
  /* Extract multiple pixels with bit depths of 1, 2, and 4 from a single
   * byte into separate bytes (useful for paletted and grayscale images).
   */
  png_set_packing(png_ptr);

  /* For images with a single "transparent colour", set colour key;
     if more than one index has transparency, use full alpha channel */
  if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS)) 
    {
      int num_trans;
      Uint8 *trans;
      png_get_tRNS(png_ptr, info_ptr, &trans, &num_trans,
		   &transv);
      if(color_type == PNG_COLOR_TYPE_PALETTE) 
	{
	  if(num_trans == 1) 
	    {
	      /* exactly one transparent value: set colour key */
	      ckey = trans[0];
	    } 
	  else
	    png_set_expand(png_ptr);
	} 
      else
	ckey = 0; /* actual value will be set later */
    }
  
  if ( (color_type == PNG_COLOR_TYPE_GRAY) ||
       (color_type == PNG_COLOR_TYPE_GRAY_ALPHA) )
    png_set_gray_to_rgb(png_ptr);
  
  png_read_update_info(png_ptr, info_ptr);
  
  png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth,
	       &color_type, &interlace_type, NULL, NULL);
  
  /* Allocate the SDL surface to hold the image */
  Rmask = Gmask = Bmask = Amask = 0 ; 
  if ( color_type != PNG_COLOR_TYPE_PALETTE ) 
    {
      if ( SDL_BYTEORDER == SDL_LIL_ENDIAN ) 
	{
	  Rmask = 0x000000FF;
	  Gmask = 0x0000FF00;
	  Bmask = 0x00FF0000;
	  Amask = (info_ptr->channels == 4) ? 0xFF000000 : 0;
	} 
      else 
	{
	  int s = (info_ptr->channels == 4) ? 0 : 8;
	  Rmask = 0xFF000000 >> s;
	  Gmask = 0x00FF0000 >> s;
	  Bmask = 0x0000FF00 >> s;
	  Amask = 0x000000FF >> s;
	}
    }
  surface = SDL_AllocSurface(SDL_SWSURFACE, width, height,
			     bit_depth*info_ptr->channels, 
			     Rmask,Gmask,Bmask,Amask);
  
  if ( surface == NULL ) 
    {
      std::cerr << "      ! out of memory"<<std::endl;
      goto done;
    }
  
  if(ckey != -1) 
    {
      if(color_type != PNG_COLOR_TYPE_PALETTE)
	ckey = SDL_MapRGB(surface->format, transv->red,
			  transv->green, transv->blue);
      SDL_SetColorKey(surface, SDL_SRCCOLORKEY, ckey);
    }
  
  /* Create the array of pointers to image data */
  row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*height);
  if ( (row_pointers == NULL) ) 
    {
      std::cerr << "      ! out of memory !" << std::endl;
      SDL_FreeSurface(surface);
      surface = NULL;
      goto done;
    }
  for (row = 0; row < height; row++) 
    {
      row_pointers[row] = (png_bytep)
	(Uint8 *)surface->pixels + row*surface->pitch;
    }
  
  /* Read the entire image in one go */
  png_read_image(png_ptr, row_pointers);
  
  /* read rest of file, get additional chunks in info_ptr - REQUIRED */
  png_read_end(png_ptr, info_ptr);
  
  /* Load the palette, if any */
  palette = surface->format->palette;
  if ( palette && (info_ptr->num_palette > 0) ) 
    {
      palette->ncolors = info_ptr->num_palette; 
      for( i=0; i<info_ptr->num_palette; ++i ) 
	{
	  palette->colors[i].b =(Uint8)info_ptr->palette[i].blue;
	  palette->colors[i].g =(Uint8)info_ptr->palette[i].green;
	  palette->colors[i].r =(Uint8)info_ptr->palette[i].red;
	}
    }
  
 done:	/* Clean up and return */
  png_destroy_read_struct(&png_ptr, info_ptr ? &info_ptr : (png_infopp)0,
			  (png_infopp)0);
  if ( row_pointers ) 
    {
      free(row_pointers);
    }
  
  //close the file
  fclose(fp);

  return surface ; 
}



static int write_png(const char* filename, SDL_Surface *surface)
{
  FILE *fp;

  fp = fopen(filename, "wb");
  if ( fp == 0 ) {
    std::cerr << "Couldn't open " << filename << "for writing" << std::endl;
    return -1;
  }

  png_structp png_ptr;
  png_infop info_ptr;
  
  png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,0,0,0);
  
  if (png_ptr == 0){
    std::cerr << "Couldn't create a write structure for the PNG file." << std::endl;
    fclose(fp);
    return -1;
  }

  /* Allocate/initialize the image information data.  REQUIRED */
  info_ptr = png_create_info_struct(png_ptr);
  if (info_ptr == 0){
    std::cerr << "Couldn't create the image information data." << std::endl;
    png_destroy_write_struct(&png_ptr,  (png_infopp)0);
    fclose(fp);
    return -1;
  }
  
  /* Set error handling.  REQUIRED if you aren't supplying your own
   * error hadnling functions in the png_create_write_struct() call.
   */
  if (setjmp(png_ptr->jmpbuf)){
    /* If we get here, we had a problem reading the file */
    std::cerr << "Error reading the file." << std::endl;
    png_destroy_write_struct(&png_ptr,  (png_infopp)0);
    fclose(fp);
    return -1;
  }

  int retval = 0;
  int number_passes ;
  png_colorp palette = 0;
  int i,y,pass;
  png_bytepp datap = 0;
  png_bytep *row_pointers = 0;

  /* set up the output control if you are using standard C streams */
  png_init_io(png_ptr, fp);
  
  /* Set the image information here.  Width and height are up to 2^31,
   * bit_depth is one of 1, 2, 4, 8, or 16, but valid values also depend on
   * the color_type selected. color_type is one of PNG_COLOR_TYPE_GRAY,
   * PNG_COLOR_TYPE_GRAY_ALPHA, PNG_COLOR_TYPE_PALETTE, PNG_COLOR_TYPE_RGB,
   * or PNG_COLOR_TYPE_RGB_ALPHA.  interlace is either PNG_INTERLACE_NONE or
   * PNG_INTERLACE_ADAM7, and the compression_type and filter_type MUST
   * currently be PNG_COMPRESSION_TYPE_BASE and PNG_FILTER_TYPE_BASE. 
   */
  if(surface->format->palette){
    png_set_IHDR(png_ptr,info_ptr,
		 surface->w,
		 surface->h,
		 surface->format->BitsPerPixel,
		 PNG_COLOR_TYPE_PALETTE,
		 PNG_INTERLACE_ADAM7,
		 PNG_COMPRESSION_TYPE_BASE, 
		 PNG_FILTER_TYPE_BASE);

    palette = (png_colorp)png_malloc(png_ptr, surface->format->palette->ncolors * sizeof (png_color));
    png_set_PLTE(png_ptr, info_ptr, palette, surface->format->palette->ncolors);
    for(i=0;i<info_ptr->num_palette;++i){
	    info_ptr->palette[i].blue = surface->format->palette->colors[i].b ;
	    info_ptr->palette[i].green = surface->format->palette->colors[i].g ;
	    info_ptr->palette[i].red = surface->format->palette->colors[i].r ;
    }       
  }
  else{
    if(surface->format->BitsPerPixel != 32 ){
      std::cerr << "The author of the PNG saver was lazy and doesn't support something other than 32bpp in non palette images." << std::endl;
      fclose(fp);
      return -1 ; 
    }
    if(surface->format->BitsPerPixel == 32){
      png_set_IHDR(png_ptr,info_ptr,
		   surface->w,
		   surface->h,
		   8,
		   PNG_COLOR_TYPE_RGB_ALPHA,
		   PNG_INTERLACE_NONE,
		   PNG_COMPRESSION_TYPE_BASE, 
		   PNG_FILTER_TYPE_BASE);
    }
    else{
      png_set_IHDR(png_ptr,info_ptr,
		   surface->w,
		   surface->h,
		   8,
		   PNG_COLOR_TYPE_RGB,
		   PNG_INTERLACE_NONE,
		   PNG_COMPRESSION_TYPE_BASE, 
		   PNG_FILTER_TYPE_BASE);
    }
    
  }
  
  /* Optionally write comments into the image */
  /*
    text_ptr[0].key = "Title";
    text_ptr[0].text = "unknown";
    text_ptr[0].compression = PNG_TEXT_COMPRESSION_NONE;
    text_ptr[1].key = "Author";
    text_ptr[1].text = "SDL_png";
    text_ptr[1].compression = PNG_TEXT_COMPRESSION_NONE;
    text_ptr[2].key = "Description";
    text_ptr[2].text = "This file was generated by the SDL library.";
    text_ptr[2].compression = PNG_TEXT_COMPRESSION_zTXt;
    png_set_text(png_ptr, info_ptr, text_ptr, 3);
  */
  
  /* Write the file header information.  REQUIRED */
  png_write_info(png_ptr, info_ptr);
  
  /* Once we write out the header, the compression type on the text
   * chunks gets changed to PNG_TEXT_COMPRESSION_NONE_WR or
   * PNG_TEXT_COMPRESSION_zTXt_WR, so it doesn't get written out again
   * at the end.
   */
  

  
  /* The number of passes is either 1 for non-interlaced images,
   * or 7 for interlaced images.
   */
  number_passes = png_set_interlace_handling(png_ptr);
  
  if(surface->format->BitsPerPixel==32){
    row_pointers = (png_bytep*) malloc(sizeof(png_bytep)*surface->h) ;
    for (y = 0; y < surface->h; y++){
      row_pointers[y] = (png_bytep) malloc(png_get_rowbytes(png_ptr, info_ptr));
    }
  }
  Uint32 pixel;
  wftk::Color col;

  for (pass = 0; pass < number_passes; ++pass)
    {
      for (y = 0; y < surface->h; ++y)
	{
	  if(surface->format->BitsPerPixel == 32)
	    {
	      for(i=0;i< surface->w*4 ;i+=4)
		{
		  pixel = *((Uint32*)((Uint8*)surface->pixels + i + y*surface->pitch));
		  SDL_GetRGBA(pixel, surface->format, &col.r, &col.g, &col.b, &col.a);
		  if ( SDL_BYTEORDER == SDL_BIG_ENDIAN ) 
		    {
		      row_pointers[y][i+3] = col.r;
		      row_pointers[y][i+2] = col.g;
		      row_pointers[y][i+1] = col.b;
		      
#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
		      row_pointers[y][i] = col.a ;
#else
		      row_pointers[y][i] = 255-col.a ;
#endif
		    } 
		  else 
		    {
		      row_pointers[y][i] = col.r;
		      row_pointers[y][i+1] = col.g;
		      row_pointers[y][i+2] = col.b;
		      
#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
		      row_pointers[y][i+3] = col.a ;
#else
		      row_pointers[y][i+3] = 255-col.a ;
#endif
		    }
		}

	      png_write_rows(png_ptr, &row_pointers[y], 1);
	    }
	  else
	    {
	      *datap = ((Uint8 *)surface->pixels + y*surface->pitch) ;
	      png_write_rows(png_ptr, datap, 1);
	    }
	}
    }
  
  /* It is REQUIRED to call this to finish writing the rest of the file */
  png_write_end(png_ptr, info_ptr);
  
  /* if you malloced the palette, free it here */
  if(surface->format->palette){
    free(info_ptr->palette);
  }
  
  /* if you allocated any text comments, free them here */
  
  /* clean up after the write, and free any memory allocated */
  png_destroy_write_struct(&png_ptr, (png_infopp)0);
  
  
  if(surface->format->BitsPerPixel==32){
    for (y = 0; y < surface->h; y++){
      free(row_pointers[y]) ;
    }
    free(row_pointers) ;
  }
  
  if ( fclose(fp) == EOF ) 
    {
      std::cerr << "Couldn't close the file %s" << std::endl;
      retval = -1;
    }
  
  return retval;
}

bool Surface::writeToFile(const std::string& filename)
{
  if(!sdlSurface_)
    return false;
  
  bool success = false;
  
  if(filename.find(".bmp") != std::string::npos ||
     filename.find(".BMP") != std::string::npos)
    {
      success = !SDL_SaveBMP(sdlSurface_, filename.c_str());
    }
  else
    {
      success = !write_png(filename.c_str(), sdlSurface_);
    }

  if(success)
    {    
      std::cerr << "Surface wrote \""<<filename<<"\" successfully." << std::endl;
    }
  else
    {
      std::cerr << "Surface could not write \""<<filename<<"\" to disk."  << std::endl;
    }

  return success;
}


bool Surface::readFromFile(const std::string& filename)
{
  Debug::channel(Debug::DRAWING) <<"Surface::readFromFile("<<filename<<")"<<Debug::endl;

  //free old surface data (if any)
  if(sdlSurface_)
    SDL_FreeSurface(sdlSurface_);

  if(filename.find(".bmp") != std::string::npos ||
     filename.find(".BMP") != std::string::npos)
    {
      sdlSurface_ = SDL_LoadBMP(filename.c_str());
    }
  else
    {
      //default is to try to load PNG
      sdlSurface_ = read_png(filename.c_str());
    }

  if(sdlSurface_)
    {
      Debug::channel(Debug::DRAWING) << "     * successfully read file \"" << filename 
	    << "\""<<Debug::endl;

      Debug::channel(Debug::DRAWING) << "     * got pixelformat:"
    	 << Pixelformat(sdlSurface_).asString() << Debug::endl;
 
      return true;
    }
  else
    return false;
}

bool Surface::readFromHeader(unsigned char* header_data, unsigned int w, unsigned int h)
{
  bool success = false;
  
  //free old surface data (if any)
  if(sdlSurface_)
    SDL_FreeSurface(sdlSurface_);
  
  Pixelformat format = Pixelformat::BGR888;
  sdlSurface_ = SDL_AllocSurface(SDL_SWSURFACE |SDL_SRCALPHA ,
				 w,
				 h,
				 format.bitspp(),
				 format.rMask(),
				 format.gMask(),
				 format.bMask(),
				 format.aMask());
  
  
  success = true;
  
  Debug::channel(Debug::DRAWING) << "parsing header_data ..."<< Debug::endl;
  
  //header_data is the address of the image data
  unsigned char *data = header_data;

  lock();
  
  for(unsigned int y = 0; y < h; y++) {
    Uint32 pixeladdr = pitch() * y;
    for(unsigned int x = 0; x < w; x++)
      {
        writePixel(pixeladdr, pixelformat().mapToPixel(Color(data[0], data[1], data[2])));
	data += 3; // advance to next pixel
        pixeladdr += pixelformat().bpp();
      }
  }

  unlock();
  
  Debug::channel(Debug::DRAWING)<<"OK."<<Debug::endl;
  
  return success;
}

static char hexDecode(char c)
{
  return isdigit(c) ? c - '0' : toupper(c)-'A'+10;
}

bool 
Surface::readFromXPM(char** data)
{
  //free old surface data (if any)
  if(sdlSurface_) {
    SDL_FreeSurface(sdlSurface_);
    sdlSurface_ = 0;
  }
  
  int lineNr = 0;
  const char* current = data[lineNr++];

  std::string line;
  line = std::string(current); 
  
  // parse the header line (using goold old sscanf)
  int w, h;
  int colorCount;   // number of individual colors
  int cpp;      // "chars per pixel"
  int num = sscanf(line.c_str(), "%d %d %d %d",&w, &h, &colorCount, &cpp);
  
  if(num != 4)
    {
      Debug::channel(Debug::DRAWING)<<"Surface::readFromXPM failed to parse header." << Debug::endl;
      return false;
    }

  Debug::channel(Debug::DRAWING)<<"Surface::readFromXPM going to parse colortable with "
	 << colorCount <<" entries ..." << Debug::endl;

  // now go for the color table
  std::map<std::string, Color> colors;
  Color color;
  std::string code;
  for(int i = 0; i < colorCount; i++)
    {
      current = data[lineNr++];
      
      // get the color code
      code = std::string(current, cpp);
      // skip color code
      current += cpp;

      // search for 'c'
      while(*current != 'c' && *current != 'g')
	current++;
      current++;
      current++;
      if(*current == '#')
	{
	  // this is a "normal" colortable entry
	  current++; // skip '#'

	  Debug::channel(Debug::DRAWING) << current << "--> ";

	  // who needs sscanf ?
	  color.r = hexDecode(*current++)*16 + hexDecode(*current++);
	  color.g = hexDecode(*current++)*16 + hexDecode(*current++);
	  color.b = hexDecode(*current++)*16 + hexDecode(*current++);
#if SDL_VERSIONNUM(SDL_MAJOR_VERSION, SDL_MINOR_VERSION, SDL_PATCHLEVEL) >= \
    SDL_VERSIONNUM(1, 1, 5)
	  color.a = 255;
#else
	  color.a = 0;
#endif

	  Debug::channel(Debug::DRAWING) << "("<<(int)color.r<<","<<(int)color.g<<","<<(int)color.b<<")"<<Debug::endl;
	}
      else
	{
	  // transparent color ... some else here ??
	  color = "transparent";
	}
      colors[code] = color;
    }
  
  if(*current != '\0')
    {
      Debug::channel(Debug::DRAWING)<<"[ERROR] " << Debug::endl;
      return false;
    }
	   
  Debug::channel(Debug::DRAWING)<<" [OK]"<< Debug::endl;

  /* Now we know enough to build the surface.
   * We will always build a 32bit surface encoding the transparency 
   * using the alpha channel. 
   */
  Pixelformat format = Pixelformat::ABGR8888;
  sdlSurface_ = SDL_AllocSurface(SDL_SWSURFACE | SDL_SRCALPHA ,
				 w,
				 h,
				 format.bitspp(),
				 format.rMask(),
				 format.gMask(),
				 format.bMask(),
				 format.aMask());
 
    
  Debug::channel(Debug::DRAWING) << "parsing XPM data  ..."<< Debug::endl;

  lock();
    
  for(int y = 0; y < h; y++)
    {
      Uint32 pixeladdr = pitch() * y;
      current = data[lineNr++];
      for(int x = 0; x < w; x++)
	{
	  // extract pixel code (cpp chars)
	  code = std::string(current, cpp);
	  //next pixel
	  current+=cpp;
	  
          writePixel(pixeladdr, pixelformat().mapToPixel(colors[code]));
          pixeladdr += pixelformat().bpp();
	}
    }

  unlock();

  bool success;
  if(*current != '\0')
    {
      Debug::channel(Debug::DRAWING)<<"[ERROR] " << Debug::endl;
      success= false;
    }
  else
    {
      Debug::channel(Debug::DRAWING)<<" [OK]"<< Debug::endl;
      success = true;
    }

  return success;

}

}

