/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "pointer.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include <assert.h>

#include "mouse_data.h"

#include <sigc++/object_slot.h>

wftk::ResourceRegistry<wftk::Pointer*,wftk::Pointer::ResLoad,wftk::Pointer::ResInval>
	wftk::Pointer::registry;

wftk::Pointer::ResInval::OutType
wftk::Pointer::ResInval::operator()(const std::string&) const
{
  // have to use get() instead of find(), since find() defaults
  // to calling this function if it can't find the font
  Resource* res = registry.get("default");

  if(!res) { // must not be created yet
    res = new Resource(new HardPointer());
    registry.insert("default", res);
    res->free();
  }

  return res->res();
}

wftk::HardPointer::HardPointer(const Surface& surf, const Point& hotspot)
{
  // This code is based on that found at:
  // http://sdldoc.csn.ul.ie/sdlcreatecursor.php

  // minimum number of bytes to get surf->width() bits
  int bytes_width = (surf.width() + 7) / 8;
 
  Uint8 *data = new Uint8[bytes_width * surf.height()];
  Uint8 *mask = new Uint8[bytes_width * surf.height()];

  surf.lock();

  Point p;
  int i = -1;
  for ( p.y = 0; (unsigned) p.y < surf.height(); ++p.y ) {
    for ( p.x = 0; (unsigned) p.x < surf.width(); ++p.x ) {
      // Checking p.x % 8 not only catches new bytes in
      // the same line, it also catches newlines.
      if ( p.x % 8 ) {
        data[i] <<= 1;
        mask[i] <<= 1;
      } else {
        ++i;
        // While there may be some padding in the last byte on each line,
        // each byte has at least one bit that corresponds to a pixel in
        // the surface. Thus, each byte gets initialized.
        data[i] = mask[i] = 0;
      }

      Uint32 pixeladdr = surf.pixelformat().bpp() * p.x + surf.pitch() * p.y;
      Color col = surf.pixelformat().mapToColor(surf.readPixel(pixeladdr));

#if SDL_ALPHA_OPAQUE
      if(col.a < 128) // opaque is 255
#else
      if(col.a >= 128) // opaque is 0
#endif
        continue; // tranparent

      mask[i] |= 0x01;
      if(((int) col.r + (int) col.g + (int) col.b) >= (3 * 128 - 1))
        data[i] |= 0x01; // white instead of black
    }
  }

  surf.unlock();

  ptr_ = SDL_CreateCursor(data, mask, 8 * bytes_width, surf.height(),
     hotspot.x, hotspot.y);
  assert(ptr_);

  delete data;
  delete mask;
}

wftk::AnimatedPointer::AnimatedPointer(const std::vector<Data>& data,
	unsigned update) :
  SoftPointer(data.front().first, data.front().second),
  update_(update),
  data_(data),
  image_num_(0)
{
  update_.halt();
  update_.alarm.connect(SigC::slot(*this, &AnimatedPointer::switchImage));
  assert(!data_.empty());
  for(unsigned i = 0; i < data_.size(); ++i)
    data_[i].first->bind();
}

wftk::AnimatedPointer::~AnimatedPointer()
{
  for(unsigned i = 0; i < data_.size(); ++i)
    data_[i].first->free();
}

void
wftk::AnimatedPointer::switchImage()
{
  if(++image_num_ == data_.size())
    image_num_ = 0;

  Data& data = data_[image_num_];
  image_->free();
  image_ = data.first;
  image_->bind();
  hotspot_ = data.second;
}

void
wftk::HardPointer::insert(const std::string& name,
	const Surface& surf, const Point& hotspot)
{
  Resource *res = new Resource(new HardPointer(surf, hotspot));
  registry.insert(name, res);
  res->free();
}

void
wftk::SoftPointer::insert(const std::string& name,
	Surface::Resource* surf, const Point& hotspot)
{
  Resource *res = new Resource(new SoftPointer(surf, hotspot));
  registry.insert(name, res);
  res->free();
}

void
wftk::AnimatedPointer::insert(const std::string& name,
	const std::vector<Data>& data, unsigned update)
{
  Resource *res = new Resource(new AnimatedPointer(data, update));
  registry.insert(name, res);
  res->free();
}

  
  /* XPM */
  static const char * mouse_xpm[] = {
    "16 16 3 1",
    "       c None",
    ".      c #000000",
    "+      c #FFFFFF",
    "+               ",
    "++              ",
    "+.+             ",
    "+..+            ",
    "+...+           ",
    "+....+          ",
    "+.....+         ",
    "+......+        ",
    "+.......+       ",
    "+......+++      ",
    "+.....+         ",
    "+..+..+         ",
    "+.+ +..+        ",
    "++  +..+        ",
    "     +.++       ",
    "     ++         ",
		"0,0"
  };
  
static SDL_Cursor* wftk_cursor(const char* ptr_image[]) {
	
  int i, row, col;
	// 2 byte row width, 16 rows -> 32 bytes
  Uint8* data = new Uint8[32];
  Uint8* mask = new Uint8[32];
	int hot_x = 0;
	int hot_y = 0;
	int ptr_w = 16;
	int ptr_h = 16;

  i = -1;
  for ( row = 0; row < ptr_w; ++row ) {
    for ( col = 0; col < ptr_h; ++col ) {
      if ( col % 8 ) {
        data[i] <<= 1;
        mask[i] <<= 1;
      } else {
        ++i;
        data[i] = mask[i] = 0;
      }
      switch (ptr_image[4+row][col]) {
        case '.':
          // black
          data[i] |= 0x01;
          mask[i] |= 0x01;
        break;
        case '+':
          // white
          mask[i] |= 0x01;
        break;
        case ' ':
        break;
      }
    }
  }
	sscanf (ptr_image[4+row], "%d,%d", &hot_x, &hot_y);
  return SDL_CreateCursor(data, mask, ptr_w, ptr_h, hot_x, hot_y);
}

// this creates the default mouse pointer
wftk::HardPointer::HardPointer()
{
  // Set the SDL hardware-accelled mouse pointer too, while we're at it
  // This code is based on that found at:
  // http://sdldoc.csn.ul.ie/sdlcreatecursor.php
  ptr_ = wftk_cursor(mouse_xpm);
	assert(ptr_);
}
