#!/usr/bin/perl -w 

=head1 DESCRIPTION

This module is used for editing netgroups, but it is limited to editing the hostnames of a netgroup.
It is part of the Skolelinux webmin modules, and its goal is to make it easier to limit what machines
are allowed to mount certain NFS volumes.

=cut

=head1 AUTHOR

Alex Brasetvik

=cut

use strict;
use Socket;

use File::Temp qw/ tempfile /;;
use Digest::MD5;
use Storable qw( store_fd retrieve);

use warnings;
use diagnostics;

#Warning: Horrible use of global variables ahead.
use vars qw($ldap %text %config %in $tb $cb $struct @log);

use constant ADD_NETGROUP => 1;
use constant DELETE_NETGROUP => 2;
use constant ADD_SUBGROUP => 3;
use constant DELETE_SUBGROUP => 4;
use constant ADD_TRIPLE => 5;
use constant DELETE_TRIPLE => 6;

require "ldap-netgroups.pl";

# Webmin style is messy, but at least separate some of the HTML from the logic. ;)
require "log.pl";
require "list.pl";
require "edit.pl";
require "head.pl";

#Webmin-stuff.
require "/usr/share/webmin/web-lib.pl";
&init_config();

#Application defaults. Use the webmin configuration interface to override these!
$config{rootdn} ||= 'cn=admin,ou=People,dc=skole,dc=skolelinux,dc=no';
$config{basedn} ||= 'dc=skole,dc=skolelinux,dc=no';
$config{tmpdir} ||= '/tmp';
$config{server} ||= 'ldap';

&ReadParse();

#Figure out if this is a new session or not.
#If it is, generate a new temporary file.
#If it isn't, read the temporary file.
my ($filename, $digest);

#We've got a cookie.
if($ENV{HTTP_COOKIE} =~ /filename/ && $ENV{HTTP_COOKIE} =~ /digest/) {

   $ENV{HTTP_COOKIE} =~ /filename=([^\s;]+)/;
   $filename = $1;
   $filename = undef if($filename !~ /^$config{tmpdir}\/(?:\w+)\.wln$/);

   $ENV{HTTP_COOKIE} =~ /digest=([^\s;]+)/;
   $digest = $1;
   $digest = undef if($digest =~ /[^a-f0-9]/);
}

my $all_okay = 0;

#Verify that the file hasn't been modified.
if($filename && $digest) {

  $all_okay++ if(-r $filename); #XXX - If a request is interrupted, which may happen frequently when invalid hostnames are input, the md5hash cookie isn't updated,
   		#      thus giving the user a new session with an empty queue. Ditch the digest until a better way of resolving is found.
		#      Resolving seems to be way faster on testing and unstable.

   if( -r $filename && 0) { #Disabled.
      my $fh;
      open($fh, "<", $filename) or return undef;
      binmode($fh);
      seek($fh,0,0);

      my $Md5 = Digest::MD5->new();
      $Md5->addfile($fh);

      $all_okay++ if($Md5->hexdigest eq $digest);
    } 
}


if($all_okay) {
   #Read the data from the file.
   $struct = retrieve $filename;
} else {
   #Generete the data structure and store it in a new temporary file.
   rebuild_struct();
}

#We've now got some data to work with.
#Now do something useful, save the results, store something in $output, deal out a new cookie and then print the output.

my $output;

$in{group} = undef if($in{edit} and not $struct->{netgroups}->{$in{group}});

#Decide what to do.
if ( $in{editqueue} ) {
    my @objects = split(/\0/, $in{delete});

    foreach my $qid (@objects) {

      my $Object = $struct->{queue}->[$qid];

      if($Object->{action} == ADD_NETGROUP) {
        # Removing this action will affect every action depending on that netgroup.
	
        $struct->{netgroups}->{$Object->{netgroup}}->{deleted} = 1;	
	
	for(my $x=0; $x <= $#{ $struct->{queue} }; $x++) {
	   my $QObject = $struct->{queue}->[$x];
	   next if($QObject->{disregard} || ! $QObject->{action}); #Probably already ignored.

	   if($QObject->{netgroup} eq $Object->{netgroup}) {
	     #This actually just makes it undef. It's easier than splice()-ing it away, as array element ID-s changes for every splice.
	     #delete $struct->{queue}->[$x];

	     #Ignore it instead.
	     $QObject->{disregard} = 1;
	   }
	   
	   if($QObject->{action} == ADD_SUBGROUP) {
	     @{$QObject->{subgroups}} = grep(!/^$Object->{netgroup}$/, @{ $QObject->{subgroups} });

	     #If that was the only subgroup in that action, disregard it. If not, add the others:
	     delete $struct->{queue}->[$x] unless(@{$QObject->{subgroups}});

	   }

	} #for

	delete $struct->{queue}->[$qid];

      } elsif($Object->{action} == DELETE_NETGROUP) {
        #Unhide the netgroup and make depending actions effective.

        delete $struct->{netgroups}->{$Object->{netgroup}}->{deleted};

         for(my $x=0; $x <= $#{ $struct->{queue} }; $x++) {
	  #Unignore the actions.
          delete $struct->{queue}->[$x]->{disregard} if($struct->{queue}->[$x]->{netgroup} eq $Object->{netgroup}); # && $struct->{queue}->{action} != DELETE_NETGROUP);
         }

	delete $struct->{queue}->[$qid];

      } else {
        delete $struct->{queue}->[$qid];
      }
    } #foreach

    $output .= netgroup_list();

} elsif ( $in{execute} && $in{password} ) {

   my($basedn, $rootdn, $rootpw) = ($config{basedn}, $config{rootdn}, $in{password});

   &ldap_connect( $config{'server'}, $config{'rootdn'} ) unless($ldap);
   if ($ldap->bind($rootdn, password => $rootpw)->code) {
   	$output .= Error($text{invalid_credentials});
	$output .= netgroup_list();
   } else {
      foreach my $action ( @{$struct->{queue}} ) {

         next if( ! $action->{action} || $action->{disregard} );

	 my $netgroup = $action->{netgroup};
         
         if($action->{action} == ADD_NETGROUP) {
             my $log = $text{add_netgroup};
             $log =~ s/\%group/$netgroup/g;
             $log .= ".. " . ldap_netgroup_add({ basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} })->error;
	     webmin_log("ADD_NETGROUP", "netgroup", $action->{netgroup}, { basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} });
             push(@log, $log);
         } elsif($action->{action} == DELETE_NETGROUP) {
             my $log = $text{delete_netgroup};
             $log =~ s/\%group/$netgroup/g;
             $log .= ".. " . ldap_netgroup_delete({ basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} })->error;
	     webmin_log("DELETE_NETGROUP", "netgroup", $action->{netgroup}, { basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} });
             push(@log, $log);
         } elsif($action->{action} == ADD_TRIPLE) {
             my $log = $text{add_triple};
             $log =~ s/\%group/$netgroup/g;
             my $triples = join(", ", map {/\((.*?),(.*?),(.*?)\)/; $1} @{$action->{triples}});
             $triples = substr($triples, 0, 30) . ".. " if(length($triples) > 30);
             $log =~ s/\%triple/$triples/g;
             $log .= ".. " . ldap_netgroup_addtriple({ basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup}, triples => $action->{triples} })->error;
	     webmin_log("ADD_TRIPLE", "netgroup", $action->{netgroup}, { basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} });
             push(@log, $log);
         } elsif($action->{action} == DELETE_TRIPLE) {
             my $log = $text{delete_triple};
             $log =~ s/\%group/$netgroup/g;
             my $triples = join(", ", map {/\((.*?),(.*?),(.*?)\)/; $1} @{$action->{triples}});
             $triples = substr($triples, 0, 30) . ".. " if(length($triples) > 30);
             $log =~ s/\%triple/$triples/g;
             $log .= ".. " . ldap_netgroup_rmtriple({ basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup}, triples => $action->{triples} })->error;
	     webmin_log("DELETE_TRIPLE", "netgroup", $action->{netgroup}, { basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} });
             push(@log, $log);
         } elsif($action->{action} == ADD_SUBGROUP) {
             my $log = $text{add_subgroup};
             $log =~ s/\%group/$netgroup/g;
             my $subgroups = join(", ", @{$action->{subgroups}});
             $subgroups = substr($subgroups, 0, 30) . ".. " if(length($subgroups) > 30);
             $log =~ s/\%subgroup/$subgroups/g;
             $log .= ".. " . ldap_netgroup_addsubgroup({ basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup}, subgroups => $action->{subgroups} })->error;
	     webmin_log("ADD_SUBGROUP", "netgroup", $action->{netgroup}, { basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} });
             push(@log, $log);
         } elsif($action->{action} == DELETE_SUBGROUP) {
             my $log = $text{delete_subgroup};
             $log =~ s/\%group/$netgroup/g;
             my $subgroups = join(", ", @{$action->{subgroups}});
             $subgroups = substr($subgroups, 0, 30) . ".. " if(length($subgroups) > 30);
             $log =~ s/\%subgroup/$subgroups/g;
             $log .= ".. " . ldap_netgroup_rmsubgroup({ basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup}, subgroups => $action->{subgroups}})->error;
	     webmin_log("DELETE_SUBGROUP", "netgroup", $action->{netgroup}, { basedn => $basedn, rootdn => $rootdn, rootpw => $rootpw, cn => $action->{netgroup} });
             push(@log, $log);
         } else {
             next;
         }
      }
      $output = view_log();

      $filename =~ s/$config{tmpdir}//;
      $filename =~ s/\///g;

      if($filename =~ /^(?:\w+)\.wln$/) {
        unlink $config{tmpdir} . '/' . $filename;
      }

      rebuild_struct();
   }
   
} elsif ( $in{create} ) {
    #Create the netgroup
    #
    # XXX : Errorchecking!
    
    my $error;
    if ( $in{group} =~ /[^\w\-_]/ || length( $in{group} ) < 9 ) {
        if ( $in{group} =~ /\W/ ) {
            $output .= Error( $text{invalid_chars} );
	    $error++;
        }
        if ( length( $in{group} ) < 9 ) {
            $output .= Error( $text{invalid_length} );
	    $error++;
        }
    }

    if ( ldap_netgroup_exists({ basedn => $config{basedn}, cn => $in{group} }) ) {
       $output .= Error( $text{exists} );
       $error++;
    }

    unless($error) {

       push(@{ $struct->{queue} }, { action => ADD_NETGROUP, netgroup => $in{group} });
   
       $struct->{netgroups}->{$in{group}} = { hosts => {}, membergroups => {} };

       for(my $x=0; $x <= $#{ $struct->{queue} }; $x++) {
   
          if($struct->{queue}->[$x]->{netgroup} eq $in{group} && $struct->{queue}->[$x]->{action} != ADD_NETGROUP && $struct->{queue}->[$x]->{disregard}) {
            #Actions on a previously deleted, but now readded, netgroup. They must be executed after it is created, though.
   	 
            delete $struct->{queue}->[$x]->{disregard};
            push(@{$struct->{queue}}, $struct->{queue}->[$x]);
            delete $struct->{queue}->[$x];

          }    
       }
    }

    $output .= netgroup_list();

} elsif ( $in{rmnetgroup} ) {
    #The user has answered yes to "Are you really sure", so delete.
    my @netgroups = split(/\0/, $in{delete});

    foreach my $netgroup (@netgroups) {
       push(@{ $struct->{queue} }, { action => DELETE_NETGROUP, netgroup => $netgroup });

       #delete $struct->{netgroups}->{$netgroup};
       $struct->{netgroups}->{$netgroup}->{deleted} = 1;

       for(my $x=0; $x <= $#{ $struct->{queue} }; $x++) {
          $struct->{queue}->[$x]->{disregard}++ if($struct->{queue}->[$x]->{netgroup} eq $netgroup && $struct->{queue}->[$x]->{action} != DELETE_NETGROUP);
       }

    }

	
    #Added to queue. Draw a list of existing netgroups.
    $output .= netgroup_list();
} elsif ( $in{edit} && $in{group} ) {
    if ( $in{rmentry} ) {
       #Remove a triple. 

       my @triples = split(/\0/, $in{delete});

       push(@{ $struct->{queue} }, { action => DELETE_TRIPLE, netgroup => $in{group}, triples => \@triples  });

       for(@triples) {
         delete $struct->{netgroups}->{$in{group}}->{hosts}->{$_};
       }

    } elsif ( $in{rmsubgroup} ) {

       my @subgroups = split(/\0/, $in{delete});

       push(@{ $struct->{queue} }, { action => DELETE_SUBGROUP, netgroup => $in{group}, subgroups => \@subgroups  });

       for(@subgroups) {
         delete $struct->{netgroups}->{$in{group}}->{membergroups}->{$_};
       }

    } elsif ( $in{addsubgroup} ) {

       my @subgroups = split(/\0/, $in{add});

       push(@{ $struct->{queue} }, { action => ADD_SUBGROUP, netgroup => $in{group}, subgroups => \@subgroups  });

       for(@subgroups) {
         $struct->{netgroups}->{$in{group}}->{membergroups}->{$_} = {};
       }

    } elsif ( $in{addentry} ) {
    
        #Add a triple to the netgroup.

	#Every hostname is a single line in a textbox.
	my @hostnames = split(/\n/, $in{hostnames});

	# The user might specify multiple domains for a single machine without knowing so. We only want to have one entry
	# for each machine, so we make a hash key for each unique hostname.
	
	my (%hosts, @invalid);

	foreach my $hostname (@hostnames) {
                #For every hostname in the textbox.

		#Ignore empty lines.
		next if($hostname !~ /\S/);

		#Remove any surrounding whitespace.
		$hostname =~ /(\S*)/;
		$hostname = $1;

		push(@invalid, $hostname) and next if($hostname =~ /[^\w\-.]/);

		$hosts{$hostname}++ and next unless($in{verify_forward} || $in{verify_reverse});


                my $ip;
		if($in{verify_forward} || $in{verify_reverse}) {
		
                   #This is SLOW if the hostname isn't found. I've tried several ways to improve it without luck.
                   my @ip = gethostbyname($hostname);

                   if($ip[4]) {
                     @ip = map { inet_ntoa($_) } @ip[4 .. $#ip];
                     $ip = $ip[0];
                   } else {
                     #Nothing is found, so we break out of the loop.
                     push(@invalid, $hostname);
                     next;
                   }
		}

		if($in{verify_reverse}) {

                   #It's valid - now we want the reverse.
                   $hostname = gethostbyaddr(inet_aton($ip), AF_INET);

		   unless($hostname) {
		     #If the reverse isn't found, it won't work with NFS.
                     push(@invalid, $hostname);
                     next;
		   }
		}

		#Okay - add it to the hash.
  		$hosts{$hostname}++; 
	}

	my @triples = map { $_ = "($_,-,-)" } keys(%hosts);

	foreach my $triple (@triples) {
   
	   $triple =~ /\((.*?),(.*?),(.*?)\)/;

	   $struct->{netgroups}->{$in{group}}->{hosts}->{$triple} = { altered => 1, hostname => $1, username => $2, domainname => $3 };
	}

	push(@{ $struct->{queue} }, { action => ADD_TRIPLE, netgroup => $in{group}, triples => \@triples } );
		
       # my $result = ldap_netgroup_addentry($config{basedn}, $in{group}, \%hosts, $config{rootdn}, $in{password});

  #	if( $result->code() ) {
#	   $output .= Error( $text{ldap_failed} .  $result->error );
#	}

	if(@invalid) {
	   $output .= Error("$text{invalid_hostnames}:");
	   $output .= Error(join("<br>\n", @invalid));
	}
    }
    #Okay. It's added - now display the list of triples in the group again.
    $output .= edit_netgroup();
}
else {
    #List all netgroups
    $output .= netgroup_list();
}

#Store the structure.
my $fh;

if($filename) {
   open($fh, ">", $filename) or die "Couldn't open temporary file: $!";
   binmode($fh);
   store_fd( $struct, $fh);
   close($fh);
} else {
   ($fh, $filename) = tempfile( DIR => $config{tmpdir}, SUFFIX => '.wln');
   binmode($fh);
   store_fd( $struct, $fh);
   close($fh);
}

#Get the new digest.
open($fh, "<", $filename) or die "Couldn't open temporary file: $!";
seek($fh,0,0);
my $Md5 = Digest::MD5->new();
$Md5->addfile($fh);
$digest = $Md5->hexdigest;

#Feed new cookie. 
print "Set-Cookie: filename=$filename; secure; path=/ldap-netgroups\r\n";
print "Set-Cookie: digest=$digest; secure; path=/ldap-netgroups\r\n";

print "Content-type: text/html\n\n";
print head();
print "<hr>";
print $output;
footer();

#Display errors in red.
sub Error {
    return '<font color="#ff0000">' . shift(@_) . "</font><br>\n";
}

sub rebuild_struct
{
   $filename = undef;
   $digest = undef;

   #Connect to the LDAP-server.
   &ldap_connect( $config{'server'}, $config{'rootdn'} ) unless($ldap);

   my @netgroups = ldap_netgroups_get($config{basedn});

   my $netgroups = {};

   foreach my $Netgroup ( @netgroups ) {
   
           my $cn;
           my %hosts;
           my %membergroups;
   
           foreach my $attribute ( @{ $Netgroup->{asn}->{attributes} } ) {

                   if($attribute->{type} eq 'cn') {

                           $cn = $attribute->{vals}->[0];

                   } elsif($attribute->{type} eq 'nisNetgroupTriple') {
   
                           foreach my $hostname ( @{ $attribute->{vals} } ) {
                                   #We may want more info on a per-host basis in the future, so why not make it a hash.
   
                                   $hostname =~ /\((.*?),(.*?),(.*?)\)/;
   
                                   $hosts{$hostname} = { altered => 0, hostname => $1, username => $2, domainname => $3 };
   
                           }
   
                   } elsif($attribute->{type} eq 'memberNisNetgroup') {
   
                           foreach my $netgroup ( @{ $attribute->{vals} } ) {
   
                                   #Ditto.
                                   $membergroups{$netgroup} = {};
   
                           }
                   } else {
                           next;
                   }
           } # attribute-loop.
   
           next unless($cn); #This shouldn't happen, really.
   
           $netgroups->{$cn} = {
                                hosts => \%hosts,
                                membergroups => \%membergroups,
           };
   } # Netgroup-loop.


   $struct = {
	netgroups => $netgroups,
	queue => []
   };

}
