/* -*- C++ -*-
 
  This file is part of ViPEC
  Copyright (C) 1991-2000 Johan Rossouw (jrossouw@alcatel.altech.co.za)
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/

#include <GridDefinition.h>

#include <Strings.h>
#include <Utils.h>
#include <Setup.h>
#include <GridView.h>
#include <DimensionDefinition.h>

#include <qdom.h>
#include <qtextstream.h>
#include <qapplication.h>

//----------------------------------------------------------------------------
GridDefinition::GridDefinition( )
    : GraphDefinition( gridType ),
    xAxisTitle_( 0 ),
    linXAxis_( TRUE ),
    xMin_( 0.1 ),
    xMax_( 10 ),
    xSteps_( 5 ),
    yMin_( -50 ),
    yMax_( 0 ),
    ySteps_( 5 ),
    trackProjFreq_( TRUE ),
    view_( 0 )
{
  Setup* setup = Setup::instance();
  setLinXAxis( setup->isLinearSweep() );
  setXAxisMin( setup->getStartFrequency() );
  setXAxisMax( setup->getStopFrequency() );
}

//----------------------------------------------------------------------------
GridDefinition::~GridDefinition()
{
  if ( !view_.isNull() )
    {
      delete view_;
    }
}

//----------------------------------------------------------------------------
GraphView* GridDefinition::getView()
{
  return view_;
}

//----------------------------------------------------------------------------
void GridDefinition::updateGraph()
{
  if ( view_.isNull() )
    {
      view_ = new GridView();
    }

  Setup* setup = Setup::instance();
  setLinXAxis( setup->isLinearSweep() );

  if ( trackProjFreq_ )
    {
      setXAxisMin( setup->getStartFrequency() );
      setXAxisMax( setup->getStopFrequency() );
    }

  view_->setName( getName() );
  view_->setTitle( getTitle() );
  view_->setXAxisMin( xMin_ );
  view_->setXAxisMax( xMax_ );
  view_->setYAxisMin( yMin_ );
  view_->setYAxisMax( yMax_ );
  view_->setNoXSteps( xSteps_ );
  view_->setNoYSteps( ySteps_ );
  view_->setLinXAxis( linXAxis_ );
  DimensionDefinition* def = Setup::instance()->getDimensionDefinition( Strings::FrequencyDim );
  const QString& freqUnit = def->getActiveValueName();
  QString label = Strings::translate( Strings::FrequencyDim );
  label += "(" + freqUnit + ")";
  view_->setXAxisTitle( label );
}

//----------------------------------------------------------------------------
void GridDefinition::setLinXAxis(bool value)
{
  linXAxis_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setXAxisMin(TReal value)
{
  xMin_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setXAxisMax(TReal value)
{
  xMax_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setNoXSteps(uint value)
{
  xSteps_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setYAxisMin(TReal value)
{
  yMin_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setYAxisMax(TReal value)
{
  yMax_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setNoYSteps(uint value)
{
  ySteps_ = value;
}

//----------------------------------------------------------------------------
void GridDefinition::setTracking( bool value )
{
  trackProjFreq_ = value;
}

//----------------------------------------------------------------------------
bool  GridDefinition::getLinXAxis() const
  {
    return linXAxis_;
  }

//----------------------------------------------------------------------------
TReal GridDefinition::getXAxisMin() const
  {
    return xMin_;
  }

//----------------------------------------------------------------------------
TReal GridDefinition::getXAxisMax() const
  {
    return xMax_;
  }

//----------------------------------------------------------------------------
uint  GridDefinition::getNoXSteps() const
  {
    return xSteps_;
  }

//----------------------------------------------------------------------------
TReal GridDefinition::getYAxisMin() const
  {
    return yMin_;
  }

//----------------------------------------------------------------------------
TReal GridDefinition::getYAxisMax() const
  {
    return yMax_;
  }

//----------------------------------------------------------------------------
uint  GridDefinition::getNoYSteps() const
  {
    return ySteps_;
  }

//----------------------------------------------------------------------------
bool  GridDefinition::getTracking() const
  {
    return trackProjFreq_;
  }

//----------------------------------------------------------------------------
void GridDefinition::writeGraphSettingToStream( QTextStream& stream )
{
  stream << "<XAXIS TRACKING=\"" << (int) trackProjFreq_ << "\" ";
  stream << "MIN=\"" << xMin_ << "\" ";
  stream << "MAX=\"" << xMax_ << "\" ";
  stream << "STEPS=\"" << xSteps_ << "\" />" << endl;
  stream << "<YAXIS MIN=\"" << yMin_ << "\" MAX=\"" << yMax_ << "\" ";
  stream << "STEPS=\"" << ySteps_ << "\" />" << endl;
}

//----------------------------------------------------------------------------
bool GridDefinition::readGraphSettingFromDOM( QDomElement& element )
{
  QDomNode node = element.firstChild();
  while( !node.isNull() )
    {
      QDomElement childElement = node.toElement();
      if( !childElement.isNull() )
        {
          if ( childElement.tagName() == "XAXIS" )
            {
              QString minStr = childElement.attribute( "MIN" );
              QString maxStr = childElement.attribute( "MAX" );
              QString stepStr = childElement.attribute( "STEPS" );
              QString trackingStr = childElement.attribute( "TRACKING" );
              xMin_ = minStr.toDouble();
              xMax_ = maxStr.toDouble();
              xSteps_ = stepStr.toUInt();
              trackProjFreq_ = (bool) trackingStr.toInt();
            }
          else if ( childElement.tagName() == "YAXIS" )
            {
              QString minStr = childElement.attribute( "MIN" );
              QString maxStr = childElement.attribute( "MAX" );
              QString stepStr = childElement.attribute( "STEPS" );
              yMin_ = minStr.toDouble();
              yMax_ = maxStr.toDouble();
              ySteps_ = stepStr.toUInt();
            }
        }
      node = node.nextSibling();
    }
  return TRUE;
}
