/* -*- C++ -*-

  This file is part of ViPEC
  Copyright (C) 1991-2000 Johan Rossouw (jrossouw@alcatel.altech.co.za)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <ModifyGridWindow.h>

#include <Setup.h>
#include <Strings.h>
#include <GridDefinition.h>
#include <MainWindow.h>
#include <NavigationWindow.h>

#include <qlabel.h>
#include <qlayout.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qapplication.h>
#include <qvalidator.h>

#include <iostream.h>

//----------------------------------------------------------------------------
ModifyGridWindow::ModifyGridWindow( QWidget* parent, const char* name )
  : QDialog(parent, name, TRUE),
    topLayout_(0),
    grid_(0)
{
  setCaption( name );
  
  topLayout_ = new QVBoxLayout( this );
  QGridLayout* gridLayout = new QGridLayout ( 0 );
  QHBoxLayout* buttonLayout = new QHBoxLayout( 0 );
  
  topLayout_->addSpacing( 10 );
  topLayout_->addItem( gridLayout );
  topLayout_->addSpacing( 10 );
  topLayout_->addItem( buttonLayout );

  QString okLabel = Strings::translate( Strings::LabelOk );
  QString cancelLabel = Strings::translate( Strings::LabelCancel );
  QPushButton* ok = new QPushButton( okLabel, this );
  QPushButton* cancel = new QPushButton( cancelLabel, this );
  
  buttonLayout->addWidget( ok );
  buttonLayout->addWidget( cancel );
  
  connect ( ok, SIGNAL( clicked() ),
	    this, SLOT( ok() ) );
  connect ( cancel, SIGNAL( clicked() ),
	    this, SLOT( cancel() ) );
  
  QStringList labelText;
  labelText += Strings::LabelGridTitle;
  labelText += Strings::LabelMinXAxisInput;
  labelText += Strings::LabelMaxXAxisInput;
  labelText += Strings::LabelNoXAxisTicks;
  labelText += Strings::LabelMinYAxisInput;
  labelText += Strings::LabelMaxYAxisInput;
  labelText += Strings::LabelNoYAxisTicks;

  gridLayout->expand( 4, labelText.count()+1 );
  gridLayout->addColSpacing( 0, 30 );
  gridLayout->addColSpacing( 1, 100 );
  gridLayout->addColSpacing( 2, 10 );
  gridLayout->addColSpacing( 3, 100 );
  gridLayout->addColSpacing( 4, 30 );

  xRangeValidator_ = new QDoubleValidator( this );
  xRangeValidator_->setRange( 1E-10, 1E15, 5 );
  yRangeValidator_ = new QDoubleValidator( this );
  yRangeValidator_->setRange( -1E10, 1E10, 5 );
  intValidator_ = new QIntValidator( this );
  intValidator_->setRange( 1, 100);

  for ( uint n=0; n<labelText.count(); n++ )
    {
       gridLayout->addRowSpacing( n, 35 );
    }

  for ( uint i = 0; i < labelText.count(); i++ ) 
    {
      QString labelTxt = Strings::translate( labelText[i] );
      QLabel* label = new QLabel( labelTxt, this );
      labelList_.append( label );
      gridLayout->addWidget( label, i, 1 );
      QLineEdit* edit = new QLineEdit( this );
      lineEditMap_[ labelText[i] ] = edit;
      gridLayout->addWidget( edit, i, 3 );
    }

  uint row = labelText.count();
  tracking_ = new QCheckBox( Strings::translate( Strings::LabelTrackProjFreq ), this );
  gridLayout->addMultiCellWidget( tracking_, row, row, 1, 3, Qt::AlignHCenter);

  lineEditMap_[Strings::LabelMinXAxisInput]->setValidator(xRangeValidator_);
  lineEditMap_[Strings::LabelMaxXAxisInput]->setValidator(xRangeValidator_);
  lineEditMap_[Strings::LabelMinYAxisInput]->setValidator(yRangeValidator_);
  lineEditMap_[Strings::LabelMaxYAxisInput]->setValidator(yRangeValidator_);
  lineEditMap_[Strings::LabelNoXAxisTicks]->setValidator(intValidator_);
  lineEditMap_[Strings::LabelNoYAxisTicks]->setValidator(intValidator_);

  resize( topLayout_->sizeHint() );
}

//----------------------------------------------------------------------------
ModifyGridWindow::~ModifyGridWindow()
{
  delete xRangeValidator_;
  delete yRangeValidator_;
  delete intValidator_;
  delete topLayout_;
}

//----------------------------------------------------------------------------
void ModifyGridWindow::destroyWidgets()
{
  QMap< QString, QLineEdit* >::Iterator it;
  for( it = lineEditMap_.begin(); it != lineEditMap_.end(); it++ )
    {
      delete (QLineEdit*) it.data();
      it.data() = 0;
    }
  lineEditMap_.clear();
  labelList_.clear();
}

//----------------------------------------------------------------------------
void ModifyGridWindow::ok()
{
  
  bool ok = TRUE;
  TReal x1, x2, y1, y2;
  uint xstep, ystep;

  x1 = lineEditMap_[Strings::LabelMinXAxisInput]->text().toDouble( &ok );
  if ( ok )
    x2 = lineEditMap_[Strings::LabelMaxXAxisInput]->text().toDouble( &ok );
  if ( ok )
    y1 = lineEditMap_[Strings::LabelMinYAxisInput]->text().toDouble( &ok );
  if ( ok )
    y2 = lineEditMap_[Strings::LabelMaxYAxisInput]->text().toDouble( &ok );
  if ( ok )
    xstep = lineEditMap_[Strings::LabelNoXAxisTicks]->text().toUInt( &ok );
  if ( ok )
    ystep = lineEditMap_[Strings::LabelNoYAxisTicks]->text().toUInt( &ok );

  if ( !ok )
    {
      return;
    }

  grid_->setTitle( lineEditMap_[Strings::LabelGridTitle]->text() );
  grid_->setXAxisMin( x1 );
  grid_->setXAxisMax( x2 );
  grid_->setYAxisMin( y1 );
  grid_->setYAxisMax( y2 );
  grid_->setNoXSteps( xstep );
  grid_->setNoYSteps( ystep );
  grid_->setTracking( tracking_->isChecked() );
  grid_->reDraw();
  grid_ = 0;
  close();
}

//----------------------------------------------------------------------------
void ModifyGridWindow::cancel()
{
  grid_ = 0;
  close();
}

//----------------------------------------------------------------------------
void ModifyGridWindow::initialize( const QString& name )
{
  grid_ = MainWindow::instance()->findGrid( name );
  ASSERT( grid_ != 0 );
  QString text;
  lineEditMap_[Strings::LabelGridTitle]->setText( grid_->getTitle() );
  text.setNum( grid_->getXAxisMin() );
  lineEditMap_[Strings::LabelMinXAxisInput]->setText( text );
  text.setNum( grid_->getXAxisMax() );
  lineEditMap_[Strings::LabelMaxXAxisInput]->setText( text );
  text.setNum( grid_->getYAxisMin() );
  lineEditMap_[Strings::LabelMinYAxisInput]->setText( text );
  text.setNum( grid_->getYAxisMax() );
  lineEditMap_[Strings::LabelMaxYAxisInput]->setText( text );
  text.setNum( grid_->getNoXSteps() );
  lineEditMap_[Strings::LabelNoXAxisTicks]->setText( text );
  text.setNum( grid_->getNoYSteps() );
  lineEditMap_[Strings::LabelNoYAxisTicks]->setText( text );
  tracking_->setChecked( grid_->getTracking() );
}
