/*
 * GLX Hardware Device Driver for Matrox G200/G400
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* $Id: mgaglx.c,v 1.35 2000/08/11 11:34:03 ocrbj Exp $ */

#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "glx_config.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "mm.h"
#include "mgadd.h"
#include "g200_mac.h"
#include "mgalib.h"
#include "hwlog.h"
#include "mgadirect.h"
#include "mgapipeline.h"
#include "mgarender.h"
#include "mgatris.h"

extern XSMesaContext XSMesa;

/* our collected globals */
mgaGlx_t	mgaglx;

mgaBufferPtr mgaDB = NULL;
mgaBufferPtr mgaFrontBuffer = NULL;
mgaContextPtr mgaCtx = NULL;



/*
 * ForceSoftwareBuffers
 * Free any hardware buffers and reallocate only in
 * virtual memory, so hardware acceleration can't be used.
 */
static void ForceSoftwareBuffers( mgaBufferPtr buf ) {
	if ( buf->backBufferBlock ) {
		/* free the hw buffer */
		mmFreeMem( buf->backBufferBlock );
		buf->backBufferBlock = NULL;
	}

	/* allocate a main memory buffer */
	hwMsg( 1, "moving back buffer to system memory to disable hw accel.\n" );
	buf->backBuffer = malloc( buf->pitch * buf->height * buf->bytesPerPixel );
	if ( !buf->backBuffer ) {
		FatalError( "Malloc for back buffer failed" );
	}

	if ( buf->depthBufferBlock ) {
		/* free the hw buffer */
		mmFreeMem( buf->depthBufferBlock );
		buf->depthBufferBlock = NULL;
	}

	/* allocate a main memory buffer */
	hwMsg( 1, "moving depth buffer to system memory to disable hw accel.\n" );
	buf->depthBuffer = malloc( buf->pitch * buf->height * buf->bytesPerDepth );
	if ( !buf->depthBuffer ) {
		FatalError( "Malloc for depth buffer failed" );
	}
}


void mgaGLXCreateDepthBuffer( GLcontext* ctx ) {
	/* we create the depth buffer when creating the image,
	 * so there is nothing to do here.
	 */
}

/*
 * mgaGLXDestroyImage
 */
void mgaGLXDestroyImage( GLXImage* image ) {
	mgaBufferPtr	buf;

	hwMsg( 1, "mgaGLXDestroyImage( %p )\n", image->devPriv );

	buf = (mgaBufferPtr)image->devPriv;
	if(buf == mgaDB)mgaDB = 0;
	
	/* free memory either in card space or VM */
	if ( buf->backBufferBlock ) {
		mmFreeMem( buf->backBufferBlock );
	} else {
		free( buf->backBuffer );
	}

	if ( buf->depthBufferBlock ) {
		mmFreeMem( buf->depthBufferBlock );
	} else {
		free( buf->depthBuffer );
	}

	buf->magic = 0;
	free( buf );

	if ( hwGetLogLevel() >= 1 )  {
		mmDumpMemInfo( cardHeap );
	}

	xfree( image );
}

/*
 * mgaAllocateBufferInHeap
 *
 * The same allocation will be tried in the card heap and the system heap.
 */
static GLboolean mgaAllocateBufferInHeap( mgaBufferPtr buf, memHeap_t *heap, 
							void *virtualBase, int startAt ) {
	int	size;

	if ( !heap ) {
		return 0;
	}
		
	/* the G400 seems to have a depth buffer problem with non-page-aligned buffers
	 * so we round everything up in size
	 */
	
	/* allocate the back buffer */
	size = buf->pitch * buf->height * buf->bytesPerPixel;
	size = (size + 4095 ) & ~4095;
	/* first try looking in the last half of memory so the buffers
	 * can be in different sgram banks if possible.
	 */ 
	buf->backBufferBlock = mmAllocMem( heap, size, 7, startAt );
	if ( !buf->backBufferBlock ) {
		buf->backBufferBlock = mmAllocMem( heap, size, 7, 0 );
		if ( !buf->backBufferBlock ) {
			return 0;
		}	
	}
	
	/* allocate the depth buffer */
	size = buf->pitch * buf->height * buf->bytesPerDepth;
	size = (size + 4095 ) & ~4095;
	if ( !size ) {
		// no depth buffer case
		buf->depthBufferBlock = 0;
	} else {
		size = (size + 4095 ) & ~4095;
	 	buf->depthBufferBlock = mmAllocMem( heap, size, 7, 0 );
		if ( !buf->depthBufferBlock ) {
			mmFreeMem( buf->backBufferBlock );
			buf->backBufferBlock = NULL;		
			return 0;
		}
	}
	
	if ( hwGetLogLevel() >= 1 )  {
		mmDumpMemInfo( heap );
	}

	buf->backBuffer = (unsigned char *)
		virtualBase + buf->backBufferBlock->ofs;
	if ( !buf->depthBufferBlock ) {
		buf->depthBuffer = NULL;
	} else {	
		buf->depthBuffer = (unsigned char *)
			virtualBase + buf->depthBufferBlock->ofs;
	}
	return 1;
}
 
/*
 * mgaGLXCreateImage
 *
 * This will always succeed, but if the buffer couldn't be allocated in
 * card memory, HW accel won't be usable.
 */
GLXImage *mgaGLXCreateImage( WindowPtr pwindow, GLvisual *visual,
				int width, int height, GLXImage *old_image )
{
	mgaBufferPtr	buf;
	GLXImage		*image;
	int			depth;
	int			maccess;
	
	hwMsg( 1, "mgaGLXCreateImage( %i, %i )\n", width, height );

	/* free the old image */
	if ( old_image && old_image->devPriv ) {
		mgaGLXDestroyImage( old_image );
	}

	depth = visual->RedBits + visual->GreenBits + visual->BlueBits;

	buf = (mgaBufferPtr)calloc(1,sizeof(mgaBuffer));
	if ( !buf ) {
		FatalError( "Malloc for buf failed\n" );
	}

	buf->magic = mgaBufferMagic;
	buf->pwindow = pwindow;
	buf->visual = *visual;
	buf->width = width;
	buf->height = height;
	buf->pitch = ( width + 63 ) & ~63;

	// if we want hardware stencil bits, we get a 24 bit depth buffer
	if ( buf->visual.StencilBits && mgaglx.isG400 ) {
		buf->visual.DepthBits = 24;		
	}
		
	maccess = 0;
	
	if ( depth == 15 ) {
		buf->bytesPerPixel = 2;
		buf->colorPixelFormat = MGA_PF_555;
		maccess |= MA_pwidth_16 | MA_dit555_enable;
	} else if ( depth == 16 ) {
		buf->bytesPerPixel = 2;
		buf->colorPixelFormat = MGA_PF_565;
		maccess |= MA_pwidth_16;
	} else if ( depth == 24 ) {
		buf->colorPixelFormat = MGA_PF_8888;
		buf->bytesPerPixel = 4;
		maccess |= MA_pwidth_32;
	} else {
		FatalError( "Bad color depth in mgaGLXCreateImage" );
	}
	
	if ( buf->visual.DepthBits == 16 ) {
		buf->bytesPerDepth = 2;
		maccess |= MA_zwidth_16;	
	} else if ( buf->visual.DepthBits == 24 ) {
		buf->bytesPerDepth = 4;
		maccess |= MA_zwidth_24;
		buf->hasStencil = 1;	
	} else if ( buf->visual.DepthBits == 32 ) {
		buf->bytesPerDepth = 4;
		maccess |= MA_zwidth_32;	
	} else if ( buf->visual.DepthBits == 0 ) {
		buf->bytesPerDepth = 0;
	} else {
		FatalError( "Bad DepthBits in mgaGLXCreateImage" );
	}

	/* throw out all texture to make sure we have as much
	 * room as possible.
	 */
	 mgaFlushAllTextures();

	hwMsg( 1, "   %i bytesPerPixel  %i bytesPerDepth\n", buf->bytesPerPixel,
		buf->bytesPerDepth );

	/* first check in card memory, then system memory, then fail to software */
	if ( mgaAllocateBufferInHeap( buf, cardHeap, cardVirtual, (GLXSYM(vga256InfoRec).videoRam * 1024) / 2 ) ) {
		buf->SystemMemory = 0;
	} else if ( mgaAllocateBufferInHeap( buf, sysmemHeap, sysmemVirtual, 0 ) ) {
		buf->SystemMemory = 1;
	} else {
		ForceSoftwareBuffers( buf );
	}

	// if we are going to hardware render, set up the default register values
	if ( buf->backBufferBlock ) {
		buf->SetupSize = MGA_SETUP_SIZE;
		buf->Setup[0] = MGA_SETUP_0;
		buf->Setup[5] = MGA_SETUP_5;
		buf->Setup[MGA_SETUP_PITCH] = buf->pitch;
		buf->Setup[MGA_SETUP_CXBNDRY] = 0x0fff0000;
		buf->Setup[MGA_SETUP_YTOP] = 0;
		buf->Setup[MGA_SETUP_YBOT] = 0x00ffffff;
		buf->Setup[MGA_SETUP_DSTORG] = 0;	
		buf->Setup[MGA_SETUP_MACCESS] = maccess;
		buf->Setup[MGA_SETUP_PLNWT] = ~0;
	
		if ( buf->SystemMemory ) {
			buf->Setup[MGA_SETUP_DSTORG] = ( buf->backBufferBlock->ofs + sysmemPhysical ) | buf->SystemMemory;
			if ( buf->depthBufferBlock ) {
				buf->Setup[MGA_SETUP_ZORG] = ( buf->depthBufferBlock->ofs + sysmemPhysical ) | buf->SystemMemory;
			}	
		} else {
			buf->Setup[MGA_SETUP_DSTORG] = buf->backBufferBlock->ofs;
			if ( buf->depthBufferBlock ) {
				buf->Setup[MGA_SETUP_ZORG] = buf->depthBufferBlock->ofs;
			}	
		}
	}
	// the image structure is just for GLX's use, the hardware renderer
	// doesn't use it for much of anything
	  
	image = (GLXImage *) xalloc( sizeof(GLXImage) );
	if ( !image ) {
		FatalError( "Malloc for back ximage failed" );
	}
	image->pwin = pwindow;
	// X's drawing routines won't take arbitrary width / pitch combinations,
	// so set width equal to pitch
	image->width = buf->pitch;	
	image->height = height;
	image->bits_per_pixel = depth;
	image->data = buf->backBuffer;
	image->bytes_per_line = buf->pitch * buf->bytesPerPixel;
	image->devPriv = buf;

	return image;
}

/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 */
static void DoMakeCurrent(XSMesaContext c, XSMesaBuffer b) {
	mgaContextPtr ctx;
	mgaBufferPtr  buf;
  
	hwMsg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

	XSMesa = c;                                                                 

	mgaCtx = NULL;
	mgaDB = NULL;
	
	if ( !c ) {
		gl_make_current(NULL, NULL);
		return;
	}
		
	ctx = c->hw_ctx;
	if (!VALID_MGA_CONTEXT(ctx)) {
		FatalError( "DoMakeCurrent: !VALID_MGA_CONTEXT" );
	}

	if ( !b ) {
		gl_make_current(ctx->gl_ctx, NULL);
		return;
	}
		
	/* if we don't have a backimage, create one now.  This happens
	 * because of the way software GLX doesn't allocate a
	 * buffer when front buffer rendering, but we need one
	 * anyway
	 */
	if ( !b->backimage ) {
		b->backimage = mgaGLXCreateImage( (WindowPtr)b->frontbuffer,
						   b->xsm_visual->gl_visual, 
						   b->width, b->height, NULL );
	}
	
	buf = (mgaBufferPtr) b->backimage->devPriv;
	if (!buf || buf->magic != mgaBufferMagic) {
		FatalError( "DoMakeCurrent: !VALID_MGA_BUFFER" );
	}

	/* Need to push context information into buf->Setup values...
	 */
	ctx->new_state = ~0;
	ctx->DB = buf;

	/* valid conditions to render with */
	mgaCtx = ctx;
	mgaDB = ctx->DB;
	mgaDB->xsBuf = b;	/* save the xsbuffer so we can get the window for front swaps */

	mga_setup_DD_pointers( ctx->gl_ctx );
	
	gl_make_current( ctx->gl_ctx, 
		  c->xsm_buffer ? c->xsm_buffer->gl_buffer : NULL);

	hwMsg( 10, "DoMakeCurrent end\n");
}


/*
 * mgaGLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */ 
GLboolean mgaGLXMakeCurrent( XSMesaContext c ) {
	if (c == XSMesa) {
		return (GL_TRUE);
	}
	
 	hwMsg( 10, "mgaGLXMakeCurrent( %p )\n", c );

	/* mesa 3.1 will sometimes leave unflushed vertexes */
	if ( XSMesa ) {
		glFlush();
	}

	DoMakeCurrent(c,c->xsm_buffer);
	return(GL_TRUE); 
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean mgaGLXBindBuffer( XSMesaContext c, XSMesaBuffer b ) {
	if ( c == XSMesa && c->xsm_buffer == b ) {
		return 1;
	}
	
	hwMsg( 10, "mgaGLXBindBuffer( %p, %p )\n", c, b );

	/* mesa 3.1 will sometimes leave unflushed vertexes */
	if ( XSMesa ) {
		glFlush();
	}

	if ( c ) {
		c->xsm_buffer = b;
	}
	
	DoMakeCurrent(c,b);

	return XSMesaBindBuffer(c,b);
}



/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext mgaGLXCreateContext( XSMesaVisual v,  
                                   XSMesaContext share_list )
{
	XSMesaContext	xc;
	GLcontext		*ctx;
	mgaContextPtr	c;

	hwMsg (0, "mgaGLXCreateContext()\n" );

	xc = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
	if ( !xc ) {
		FatalError( "calloc failed" );
	}

	ctx = xc->gl_ctx = gl_create_context(v->gl_visual,    
				  share_list ? share_list->gl_ctx : NULL,
				  (void *) xc, GL_TRUE /* direct rendering */);
	if ( !ctx ) {
	        free( xc );
	        return NULL;
	}

	c = (mgaContextPtr) calloc(1,sizeof(mgaContext));
	if ( !c ) {
		FatalError( "calloc failed" );
	}
		
	c->magic = mgaContextMagic;
	c->gl_ctx = ctx;
	c->regDWGCTL = ( DC_clipdis_disable | (0xC << DC_bop_SHIFT) |
		   DC_shftzero_enable | DC_zmode_nozcmp | 
		   DC_atype_zi );

	c->regALPHACTRL = AC_src_one | AC_dst_zero | AC_amode_FCOL | 
		    AC_astipple_disable | AC_aten_disable | AC_atmode_noacmp |
		    AC_alphasel_fromtex;

	c->regFOGCOLOR = MGAPACKCOLOR888((hwUI8)(ctx->Fog.Color[0]*255.0F), 
				   (hwUI8)(ctx->Fog.Color[1]*255.0F), 
				   (hwUI8)(ctx->Fog.Color[2]*255.0F));

	c->regSTENCIL = 0;
	c->regSTENCILCTL = 0;
   
	c->renderindex = -1;		/* impossible value */
	c->new_state = ~0;
	c->reg_dirty = ~0;


	/* XXX Fix me: callback not registered when main VB is created.
	*/
	if (ctx->VB) {
		mgaDDRegisterVB( ctx->VB );
	} else {
		fprintf(stderr, "**** Didn't create vb driver data\n");
	}
	
	if ( ctx->NrPipelineStages ) {
		ctx->NrPipelineStages = mgaDDRegisterPipelineStages(ctx->PipelineStage,
							 ctx->PipelineStage,
							 ctx->NrPipelineStages);
	}
		
	if (!glx_getint("mga_no_fast_path")) {  	
		ctx->Driver.BuildPrecalcPipeline = mgaDDBuildPrecalcPipeline;
	} else {
		hwMsg( 1, "enabling mga_no_fast_path\n" );
	}

	/* Ask mesa to clip fog coordinates for us.
	*/
	ctx->TriangleCaps |= DD_CLIP_FOG_COORD;

	mgaDDRenderInit();
	mgaDDTrifuncInit();
	mgaDDFastPathInit();
	mgaDDSetupInit();
	mgaDDExtensionsInit( ctx );

	xc->xsm_visual = v;
	xc->xsm_buffer = NULL;       /* set later by XSMesaMakeCurrent */
	xc->pixelformat = v->dithered_pf;    /* Dithering is enabled by default */
	xc->hw_ctx = c;
 	xc->gl_ctx->Driver.UpdateState = mga_setup_DD_pointers;

	hwMsg( 1, "mgaGLXCreateContext succeeded: %p\n", xc );

	return xc;
}


/*
 * mgaGLXDestroyContext
 */
void mgaGLXDestroyContext( XSMesaContext c ) {
	mgaContextPtr ctx;
	
	hwMsg( 1, "mgaGLXDestroyContext( %p )\n", c );
	
	/* make sure all drawing is completed */
        if(mgaCtx) mgaWaitDrawingEngine();
	
	ctx = (mgaContextPtr)c->hw_ctx;

	/* if this is our current context, make sure we clear it */
	if ( ctx == mgaCtx ) {
		mgaCtx = NULL;
	}
	if ( c == XSMesa ) {
		XSMesa = 0;
	}

	if ( ctx ) {
		if (ctx->magic != mgaContextMagic) {
			hwError("mgaDestroyContext(): ctx->magic != mgaContextMagic\n");
		} else {
			ctx->magic = 0;
			free(ctx);
			hwMsg( 1,"mgaDestroyContext(): successfully destroyed.\n" );
		}
	}
			
	XSMesaDestroyContext(c);
}
				       
