
/* this file deals with allowing software mesa to interact with the buffers
 * used for hardware rendering
 */

#include <stdlib.h>
#include <string.h>
#include "mesaglx/context.h"
#include "mesaglx/types.h"
#include "types.h"
#include "hw_buffer.h"
#include "glx_symbols.h"

/*

future improvements:

more fast path cases
we could implement dithering on the pixel writes.
we could do bit replication on reads from 16 bit buffers

we are assuming sizeof( GLdepth ) == 2

*/


/* Every time a context or buffer is changed, these will be updated
 * to account for the upside down way that GL looks at buffers, the
 * buffer pointer will point at the BOTTOM line, and pitch will have
 * a negative value.
 */
static	GLubyte	*dbBuffer;
static	dbFormat_t	dbFormat;
static	int	dbPitch;
static	int	dbHeight;

static	GLubyte	*cbBuffer;
static	cbFormat_t	cbFormat;
static	int	cbPitch;
static	int	cbHeight;
static	int	cbWidth;
	
static	void (*hwFinishFunc)(void);

/* when rasterizing flat shaded primitives, this fixed color will be set */
static GLubyte	monoColor[4];

/*
 * Return the address of the Z-buffer value for window coordinate (x,y):
 */
#define Z_ADDRESS( X, Y ) ((unsigned short *)dbBuffer + dbPitch * (Y) + (X))
#define Z_ADDRESS32( X, Y ) ((unsigned *)dbBuffer + dbPitch * (Y) + (X))


/**********************************************************************/
/*****                   Depth Testing Functions                  *****/
/**********************************************************************/


/*
 * Depth test horizontal spans of fragments.  These functions are called
 * via ctx->Driver.depth_test_span only.
 *
 * Input:  n - number of pixels in the span
 *         x, y - location of leftmost pixel in span in window coords
 *         z - array [n] of integer depth values
 * In/Out:  mask - array [n] of flags (1=draw pixel, 0=don't draw) 
 * Return:  number of pixels which passed depth test
 */


/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static GLuint depth_test_span_generic_16( GLcontext* ctx,
                                   GLuint n, GLint x, GLint y,
                                   const GLdepth z[],
                                   GLubyte mask[] )
{
   unsigned short *zptr;
   GLubyte *m = mask;
   GLuint i;
   GLuint passed = 0;

	zptr = (unsigned short *)dbBuffer + dbPitch * y + x;
	
  hwFinishFunc();

   /* switch cases ordered from most frequent to less frequent */
   switch (ctx->Depth.Func) {
      case GL_LESS:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] < *zptr) {
		     /* pass */
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     /* fail */
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] < *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_LEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] <= *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] <= *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] >= *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] >= *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GREATER:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] > *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] > *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_NOTEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] != *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] != *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_EQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] == *zptr) {
		     *zptr = z[i];
		     passed++;
		  }
		  else {
		     *m =0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] == *zptr) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m =0;
		  }
	       }
	    }
	 }
	 break;
      case GL_ALWAYS:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  *zptr = z[i];
		  passed++;
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer or mask */
	    passed = n;
	 }
	 break;
      case GL_NEVER:
	 for (i=0;i<n;i++) {
	    mask[i] = 0;
	 }
	 break;
      default:
         gl_problem(ctx, "Bad depth func in gl_depth_test_span_generic");
   } /*switch*/

   return passed;
}

/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static GLuint depth_test_span_generic_32( GLcontext* ctx,
                                   GLuint n, GLint x, GLint y,
                                   const GLdepth z[],
                                   GLubyte mask[] )
{
   unsigned  *zptr;
   GLubyte *m = mask;
   GLuint i;
   GLuint passed = 0;
	int	shift;
	
	zptr = (unsigned  *)dbBuffer + dbPitch * y + x;
	
	shift = 16;

#define	READZ	((*zptr)>>16)
					
  hwFinishFunc();

   /* switch cases ordered from most frequent to less frequent */
   switch (ctx->Depth.Func) {
      case GL_LESS:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] < READZ) {
		     /* pass */
		     *zptr = z[i] << shift;
		     passed++;
		  }
		  else {
		     /* fail */
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] < READZ) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_LEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] <= READZ) {
		     *zptr = z[i] << shift;
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] <= READZ) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] >= READZ) {
		     *zptr = z[i] << shift;
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] >= READZ) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GREATER:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] > READZ) {
		     *zptr = z[i] << shift;
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] > READZ) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_NOTEQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] != READZ) {
		     *zptr = z[i] << shift;
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] != READZ) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_EQUAL:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] == READZ) {
		     *zptr = z[i] << shift;
		     passed++;
		  }
		  else {
		     *m =0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  if (z[i] == READZ) {
		     /* pass */
		     passed++;
		  }
		  else {
		     *m =0;
		  }
	       }
	    }
	 }
	 break;
      case GL_ALWAYS:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0;i<n;i++,zptr++,m++) {
	       if (*m) {
		  *zptr = z[i] << shift;
		  passed++;
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer or mask */
	    passed = n;
	 }
	 break;
      case GL_NEVER:
	 for (i=0;i<n;i++) {
	    mask[i] = 0;
	 }
	 break;
      default:
         gl_problem(ctx, "Bad depth func in gl_depth_test_span_generic");
   } /*switch*/

   return passed;
}



/*
 * Depth test an array of randomly positioned fragments.
 */

/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static void depth_test_pixels_generic_16( GLcontext* ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   const GLdepth z[], GLubyte mask[] )
{
   register GLdepth *zptr;
   register GLuint i;

  hwFinishFunc();

   /* switch cases ordered from most frequent to less frequent */
   switch (ctx->Depth.Func) {
      case GL_LESS:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] < *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] < *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_LEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] <= *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] <= *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] >= *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] >= *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GREATER:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] > *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] > *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_NOTEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] != *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] != *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_EQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] == *zptr) {
		     /* pass */
		     *zptr = z[i];
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  if (z[i] == *zptr) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_ALWAYS:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS(x[i],y[i]);
		  *zptr = z[i];
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer or mask */
	 }
	 break;
      case GL_NEVER:
	 /* depth test never passes */
	 for (i=0;i<n;i++) {
	    mask[i] = 0;
	 }
	 break;
      default:
         gl_problem(ctx, "Bad depth func in gl_depth_test_pixels_generic");
   } /*switch*/
}

/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static void depth_test_pixels_generic_32( GLcontext* ctx,
                                   GLuint n, const GLint x[], const GLint y[],
                                   const GLdepth z[], GLubyte mask[] )
{
   register unsigned *zptr;
   register GLuint i;

  hwFinishFunc();

   /* switch cases ordered from most frequent to less frequent */
   switch (ctx->Depth.Func) {
      case GL_LESS:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] < ( *zptr >> 16 ) ) {
		     /* pass */
		     *zptr = z[i] << 16;
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] < ( *zptr >> 16 )  ) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_LEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] <= ( *zptr >> 16 ) ) {
		     /* pass */
		     *zptr = z[i] << 16;
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] <= ( *zptr >> 16 ) ) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] >= ( *zptr >> 16 ) ) {
		     /* pass */
		     *zptr = z[i] << 16;
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] >= ( *zptr >> 16 ) ) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_GREATER:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] > ( *zptr >> 16 ) ) {
		     /* pass */
		     *zptr = z[i] << 16;
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] > ( *zptr >> 16 ) ) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_NOTEQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] != ( *zptr >> 16 ) ) {
		     /* pass */
		     *zptr = z[i] << 16;
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] != ( *zptr >> 16 ) ) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_EQUAL:
         if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] == ( *zptr >> 16 ) ) {
		     /* pass */
		     *zptr = z[i] << 16;
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  if (z[i] == ( *zptr >> 16 ) ) {
		     /* pass */
		  }
		  else {
		     /* fail */
		     mask[i] = 0;
		  }
	       }
	    }
	 }
	 break;
      case GL_ALWAYS:
	 if (ctx->Depth.Mask) {
	    /* Update Z buffer */
	    for (i=0; i<n; i++) {
	       if (mask[i]) {
		  zptr = Z_ADDRESS32(x[i],y[i]);
		  *zptr = z[i] << 16;
	       }
	    }
	 }
	 else {
	    /* Don't update Z buffer or mask */
	 }
	 break;
      case GL_NEVER:
	 /* depth test never passes */
	 for (i=0;i<n;i++) {
	    mask[i] = 0;
	 }
	 break;
      default:
         gl_problem(ctx, "Bad depth func in gl_depth_test_pixels_generic");
   } /*switch*/
}

/**********************************************************************/
/*****                      Read Depth Buffer                     *****/
/**********************************************************************/


/*
 * Return a span of depth values from the depth buffer as floats in [0,1].
 * This function is only called through Driver.read_depth_span_float()
 * Input:  n - how many pixels
 *         x,y - location of first pixel
 * Output:  depth - the array of depth values
 */
static void read_depth_span_float_16( GLcontext* ctx,
                               GLuint n, GLint x, GLint y, GLfloat depth[] )
{
	unsigned short *zptr;
	GLfloat scale;
	GLuint i;

	hwFinishFunc();

	scale = 1.0F / 0xffff;

	zptr = (unsigned short *)dbBuffer + dbPitch * y + x;
	for (i=0;i<n;i++) {
		depth[i] = (GLfloat) zptr[i] * scale;
	}
}

static void read_depth_span_float_32( GLcontext* ctx,
                               GLuint n, GLint x, GLint y, GLfloat depth[] )
{
	unsigned *zptr;
	double scale;
	GLuint i;

	hwFinishFunc();

	scale = 1.0F / (double)0xffffffffu;

	zptr = (unsigned *)dbBuffer + dbPitch * y + x;
	for (i=0;i<n;i++) {
		depth[i] = zptr[i] * scale;
	}
}


static void read_depth_span_float_24( GLcontext* ctx,
                               GLuint n, GLint x, GLint y, GLfloat depth[] )
{
	unsigned *zptr;
	double scale;
	GLuint i;

	hwFinishFunc();

	scale = 1.0F / (double)0xffffffu;

	zptr = (unsigned *)dbBuffer + dbPitch * y + x;
	for (i=0;i<n;i++) {
		depth[i] = ( zptr[i] & 0xffffff ) * scale;
	}
}


static void read_depth_span_float_0( GLcontext* ctx,
                               GLuint n, GLint x, GLint y, GLfloat depth[] )
{
	GLuint i;

	for (i=0;i<n;i++) {
		depth[i] = 0.0F;
	}
}



/*
 * Return a span of depth values from the depth buffer as integers in
 * [0,MAX_DEPTH].
 * This function is only called through Driver.read_depth_span_int()
 * Input:  n - how many pixels
 *         x,y - location of first pixel
 * Output:  depth - the array of depth values
 */
static void read_depth_span_int_16( GLcontext* ctx,
                             GLuint n, GLint x, GLint y, GLdepth depth[] )
{
	unsigned short	*zptr;
	
	hwFinishFunc();

	zptr = (unsigned short *)dbBuffer + dbPitch * y + x;
	MEMCPY( depth, zptr, n * sizeof(GLdepth) );
}

static void read_depth_span_int_32( GLcontext* ctx,
                             GLuint n, GLint x, GLint y, GLdepth depth[] )
{
	unsigned	*zptr;
	int		i;
	
	hwFinishFunc();

	zptr = (unsigned *)dbBuffer + dbPitch * y + x;
	for (i=0;i<n;i++) {
		depth[i] = zptr[i] >> 16;
	}
}

static void read_depth_span_int_24( GLcontext* ctx,
                             GLuint n, GLint x, GLint y, GLdepth depth[] )
{
	unsigned	*zptr;
	int		i;
	
	hwFinishFunc();

	zptr = (unsigned *)dbBuffer + dbPitch * y + x;
	for (i=0;i<n;i++) {
		depth[i] = zptr[i] >> 8;
	}
}

static void read_depth_span_int_0( GLcontext* ctx,
                             GLuint n, GLint x, GLint y, GLdepth depth[] )
{
	GLuint i;
	
	for (i=0;i<n;i++) {
		depth[i] = 0;
	}
}

//======================================================

/* this is extremely inefficient, but it should only be hit when conformance testing */

static void WritePixel( int x, int y, const GLubyte rgba[4] ) {
        switch ( cbFormat ) {
       	case CB_15BIT:
       		{
                    unsigned short *ptr = (unsigned short *)cbBuffer + 
                    	cbPitch * y + x;
			*ptr = ( rgba[2] >> 3 )
				| ( ( rgba[1] >> 3 ) << 5 )
				| ( ( rgba[0] >> 3 ) << 10 )
				| ( ( rgba[3] >> 7 ) << 15 );
                }    
       	case CB_16BIT:
       		{
                    unsigned short *ptr = (unsigned short *)cbBuffer + 
                    	cbPitch * y + x;
			*ptr = ( rgba[2] >> 3 )
				| ( ( rgba[1] >> 2 ) << 5 )
				| ( ( rgba[0] >> 3 ) << 11 );
                }    
                break;
	case CB_32BIT_BGRA:                
		{
		unsigned pix;
		
                GLuint *ptr = (unsigned *)cbBuffer + 
                	cbPitch * y + x;
		// swap to bgra
		((GLubyte *)&pix)[0] = rgba[2];
		((GLubyte *)&pix)[1] = rgba[1];
		((GLubyte *)&pix)[2] = rgba[0];
		((GLubyte *)&pix)[3] = rgba[3];
			                            
	            *ptr = pix;
                }
                break;
	}                
}


static void WriteRGBASpan( const GLcontext *ctx,
			   GLuint n, GLint x, GLint y,
			   CONST GLubyte rgba[][4], 
			   const GLubyte mask[] )
{
	int	i;
	
	hwFinishFunc();
	
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			WritePixel( x + i, y, rgba[i] );
		}
	}		
}

static void WriteRGBASpan_16( const GLcontext *ctx,
			   GLuint n, GLint x, GLint y,
			   CONST GLubyte rgba[][4], 
			   const GLubyte mask[] )
{
	int	i;
	unsigned short *ptr;
	CONST GLubyte *rgba_p;

	hwFinishFunc();
	
        ptr = (unsigned short *)cbBuffer + cbPitch * y + x;
	rgba_p = rgba[0];
	
	for ( i = 0 ; i < n ; i++, rgba_p += 4 ) {
		if ( !mask || mask[i] ) {
			ptr[i] = ( rgba_p[2] >> 3 )
				| ( ( rgba_p[1] >> 2 ) << 5 )
				| ( ( rgba_p[0] >> 3 ) << 11 );
                }    
	}		
}

static void WriteRGBSpan( const GLcontext *ctx,
			  GLuint n, GLint x, GLint y,
			  CONST GLubyte rgb[][3], const GLubyte mask[] )
{
	int	i;
	GLubyte	rgba[4];
	
	hwFinishFunc();
	
	rgba[3] = 255;
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			rgba[0] = rgb[i][0];
			rgba[1] = rgb[i][1];
			rgba[2] = rgb[i][2];
			WritePixel( x + i, y, rgba );
		}
	}		
}

static void WriteRGBSpan_16( const GLcontext *ctx,
			   GLuint n, GLint x, GLint y,
			   CONST GLubyte rgb[][3], 
			   const GLubyte mask[] )
{
	int	i;
	unsigned short *ptr;
	CONST GLubyte *rgba_p;
	
	hwFinishFunc();
	
        ptr = (unsigned short *)cbBuffer + cbPitch * y + x;
	rgba_p = rgb[0];
	
	for ( i = 0 ; i < n ; i++, rgba_p += 3 ) {
		if ( !mask || mask[i] ) {
			ptr[i] = ( rgba_p[2] >> 3 )
				| ( ( rgba_p[1] >> 2 ) << 5 )
				| ( ( rgba_p[0] >> 3 ) << 11 );
                }    
	}		
}


static void WriteMonoRGBASpan(const GLcontext *ctx, GLuint n, 
			      GLint x, GLint y,const GLubyte mask[] )
{
	int	i;
	
	hwFinishFunc();
	
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			WritePixel( x + i, y, monoColor );
		}
	}		
}

static void WriteMonoRGBASpan_16( const GLcontext *ctx, GLuint n, 
				GLint x, GLint y, const GLubyte mask[] )
{
	int	i;
	unsigned short *ptr, pix;
	
	hwFinishFunc();
	
        ptr = (unsigned short *)cbBuffer + cbPitch * y + x;
	pix = ( monoColor[2] >> 3 )
				| ( ( monoColor[1] >> 2 ) << 5 )
				| ( ( monoColor[0] >> 3 ) << 11 );
	
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			ptr[i] = pix;
                }    
	}		
}

static void WriteRGBAPixels(const GLcontext *ctx,
			    GLuint n, const GLint x[], const GLint y[],
			    CONST GLubyte rgba[][4], 
			    const GLubyte mask[] )
{
	int	i;
	
	hwFinishFunc();
	
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			WritePixel( x[i], y[i], rgba[i] );
		}
	}		
}

static void WriteRGBAPixels_16(const GLcontext *ctx,
			    GLuint n, const GLint x[], const GLint y[],
			    CONST GLubyte rgba[][4], 
			    const GLubyte mask[] )
{
	int	i;
	CONST GLubyte	*rgba_p;
	unsigned short *ptr;
	
	hwFinishFunc();
	
	rgba_p = rgba[0];
	for ( i = 0 ; i < n ; i++, rgba_p+=4 ) {
		if ( !mask || mask[i] ) {
			ptr = (unsigned short *)cbBuffer + cbPitch * y[i] + x[i];
			*ptr = ( rgba_p[2] >> 3 )
				| ( ( rgba_p[1] >> 2 ) << 5 )
				| ( ( rgba_p[0] >> 3 ) << 11 );
		}
	}		
}

static void WriteMonoRGBAPixels(const GLcontext *ctx,GLuint n, 
				const GLint x[], const GLint y[],
				const GLubyte mask[] )
{
	int	i;
	
	hwFinishFunc();
	
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			WritePixel( x[i], y[i], monoColor );
		}
	}		
}

static void WriteMonoRGBAPixels_16(const GLcontext *ctx,GLuint n, 
				const GLint x[], const GLint y[],
				const GLubyte mask[] )
{
	int	i;
	unsigned short *ptr, pix;
	
	hwFinishFunc();
	
	pix = ( monoColor[2] >> 3 )
				| ( ( monoColor[1] >> 2 ) << 5 )
				| ( ( monoColor[0] >> 3 ) << 11 );
	
	for ( i = 0 ; i < n ; i++ ) {
		if ( !mask || mask[i] ) {
			ptr = (unsigned short *)cbBuffer + cbPitch * y[i] + x[i];
			*ptr = pix;
                }    
	}		
}

static void ReadRGBASpan_24(const GLcontext *ctx, GLuint n, 
			 GLint x, GLint y,GLubyte rgba[][4] )
{
	int		i;
	GLuint		*ptr;
	
	hwFinishFunc();

        ptr = (unsigned *)cbBuffer + cbPitch * y + x;
       	for (i=0;i<n;i++) {
	        GLuint p4 = ptr[i];
                rgba[i][0] = (GLubyte) ((p4 >> 16) & 0xff);
                rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                rgba[i][2] = (GLubyte) ((p4 >> 0)  & 0xff);
                rgba[i][3] = (GLubyte) ((p4 >> 24) & 0xff);
        }
}

static void ReadRGBASpan_16(const GLcontext *ctx, GLuint n, 
			 GLint x, GLint y,GLubyte rgba[][4] )
{
	int		i;
	unsigned short	*ptr;
	
	hwFinishFunc();

	/* to be really correct, we should replicate the bits we are missing
	 * instead of 0 filling
	 */
        ptr = (unsigned short *)cbBuffer + cbPitch * y + x;
        for (i=0;i<n;i++) {
              	unsigned short p = ptr[i];	
                rgba[i][2] = (p & 0x1f) << 3;
                rgba[i][1] = (p & (0x3f << 5) ) >> 3;
                rgba[i][0] = (p & (0x1f << 11) ) >> 8;
                rgba[i][3] = 255;
        }
}

static void ReadRGBASpan_15(const GLcontext *ctx, GLuint n, 
			 GLint x, GLint y,GLubyte rgba[][4] )
{
	int		i;
	unsigned short	*ptr;
	
	hwFinishFunc();

	/* to be really correct, we should replicate the bits we are missing
	 * instead of 0 filling
	 */
        ptr = (unsigned short *)cbBuffer + cbPitch * y + x;
        for (i=0;i<n;i++) {
              	unsigned short p = ptr[i];	
                rgba[i][2] = (p & 0x1f) << 3;
                rgba[i][1] = (p & (0x1f << 5) ) >> 2;
                rgba[i][0] = (p & (0x1f << 10) ) >> 7;
                rgba[i][3] = 255;
        }
}

static void ReadRGBAPixels_24(const GLcontext *ctx,GLuint n, 
			   const GLint x[], const GLint y[],
			   GLubyte rgba[][4], const GLubyte mask[] )
{
	int		i;
	
	hwFinishFunc();

        for (i=0;i<n;i++) {
		if (!mask || mask[i]) {
                        GLuint p4 = *( (unsigned *)cbBuffer + cbPitch * y[i] + x[i] );
                        rgba[i][0] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ((p4 >> 0) & 0xff);
                        rgba[i][3] = (GLubyte) ((p4 >> 24) & 0xff);
                }
	}                
}

static void ReadRGBAPixels_16(const GLcontext *ctx,GLuint n, 
			   const GLint x[], const GLint y[],
			   GLubyte rgba[][4], const GLubyte mask[] )
{
	int		i;
	
	hwFinishFunc();
	
	/* to be really correct, we should replicate the bits we are missing
	 * instead of 0 filling
	 */
	for (i=0;i<n;i++) {
                if ( !mask || mask[i] ) {
                    unsigned short p = *( (unsigned short *)cbBuffer + cbPitch * y[i] + x[i] );
	            rgba[i][2] = (p & 0x1f) << 3;
	          	rgba[i][1] = (p & (0x3f << 5) ) >> 3;
	          	rgba[i][0] = (p & (0x1f << 11) ) >> 8;
 	           	rgba[i][3] = 255;
                }
        }
}

static void ReadRGBAPixels_15(const GLcontext *ctx,GLuint n, 
			   const GLint x[], const GLint y[],
			   GLubyte rgba[][4], const GLubyte mask[] )
{
	int		i;
	
	hwFinishFunc();
	
	/* to be really correct, we should replicate the bits we are missing
	 * instead of 0 filling
	 */
	for (i=0;i<n;i++) {
                if ( !mask || mask[i] ) {
                    unsigned short p = *( (unsigned short *)cbBuffer + cbPitch * y[i] + x[i] );
	            rgba[i][2] = (p & 0x1f) << 3;
	          	rgba[i][1] = (p & (0x1f << 5) ) >> 2;
	          	rgba[i][0] = (p & (0x1f << 10) ) >> 7;
 	           	rgba[i][3] = 255;
                }
        }
}



static void SetMonoSpanColor( GLcontext *ctx,
                  GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha ) {
	monoColor[0] = red;	                  
	monoColor[1] = green;	                  
	monoColor[2] = blue;	                  
	monoColor[3] = alpha;	                  
}                  


/*
 * SetDriverBufferFunctions
 *
 * This sets all the buffer access functions in ctx->Driver
 *
 * hwFinish is a function pointer to a routine that will wait
 * until all hardware rendering is completed, allowing the cpu
 * to safely read buffer contents.
 *
 * This is per-context, so contexts with different
 * color or depth bits can coexist.
 * This should be called every time a buffer is bound, because
 * the same context can be bound to different depth buffers at
 * different times.
 *
 * pitch is in PIXELS, not bytes
 */
void SetDriverBufferFunctions( GLcontext *ctx, void (*hwFinish)(void),
	GLubyte *colorBuffer, int colorPitch, int colorHeight, cbFormat_t colorFormat,
	GLubyte *depthBuffer, int depthPitch, int depthHeight, dbFormat_t depthFormat ) {

	hwFinishFunc = hwFinish;
	 
	// flip buffer references for GL's bottom to top frame of reference
	cbWidth = colorPitch;
	cbPitch = -colorPitch;
	cbHeight = colorHeight;
	cbFormat = colorFormat;
	
	dbPitch = -depthPitch;
	dbHeight = depthHeight; 
	dbFormat = depthFormat;
	
	ctx->Driver.Color = SetMonoSpanColor;

        switch ( cbFormat ) {
        case CB_15BIT:
		cbBuffer = colorBuffer - 2 * ( cbHeight - 1 ) * cbPitch;
		ctx->Driver.WriteRGBASpan = WriteRGBASpan;
		ctx->Driver.WriteRGBSpan = WriteRGBSpan;
		ctx->Driver.WriteMonoRGBASpan = WriteMonoRGBASpan;
		ctx->Driver.WriteRGBAPixels = WriteRGBAPixels;
		ctx->Driver.WriteMonoRGBAPixels = WriteMonoRGBAPixels;
		
		ctx->Driver.ReadRGBASpan = ReadRGBASpan_15;
		ctx->Driver.ReadRGBAPixels = ReadRGBAPixels_15;
		break;
        case CB_16BIT:
		cbBuffer = colorBuffer - 2 * ( cbHeight - 1 ) * cbPitch;
		ctx->Driver.WriteRGBASpan = WriteRGBASpan_16;
		ctx->Driver.WriteRGBSpan = WriteRGBSpan_16;
		ctx->Driver.WriteMonoRGBASpan = WriteMonoRGBASpan_16;
		ctx->Driver.WriteRGBAPixels = WriteRGBAPixels_16;
		ctx->Driver.WriteMonoRGBAPixels = WriteMonoRGBAPixels_16;
		
		ctx->Driver.ReadRGBASpan = ReadRGBASpan_16;
		ctx->Driver.ReadRGBAPixels = ReadRGBAPixels_16;
		break;
	case CB_32BIT_BGRA:	
		cbBuffer = colorBuffer - 4 * ( cbHeight - 1 ) * cbPitch;
		ctx->Driver.WriteRGBASpan = WriteRGBASpan;
		ctx->Driver.WriteRGBSpan = WriteRGBSpan;
		ctx->Driver.WriteMonoRGBASpan = WriteMonoRGBASpan;
		ctx->Driver.WriteRGBAPixels = WriteRGBAPixels;
		ctx->Driver.WriteMonoRGBAPixels = WriteMonoRGBAPixels;
		
		ctx->Driver.ReadRGBASpan = ReadRGBASpan_24;
		ctx->Driver.ReadRGBAPixels = ReadRGBAPixels_24;
		break;
	}


	if ( !depthBuffer ) {
		/* we can sometimes have no depth buffer */
		dbBuffer = NULL;
	 	ctx->Driver.ReadDepthSpanFloat = read_depth_span_float_0;
		ctx->Driver.ReadDepthSpanInt = read_depth_span_int_0;
		ctx->Driver.DepthTestSpan = NULL;
		ctx->Driver.DepthTestPixels = NULL;
	} else {
		switch( dbFormat ) {
		case DB_24BIT_SB_8BIT:
			dbBuffer = depthBuffer - 4 * ( dbHeight - 1 ) * dbPitch;
		 	ctx->Driver.ReadDepthSpanFloat = read_depth_span_float_24;
			ctx->Driver.ReadDepthSpanInt = read_depth_span_int_24;
			ctx->Driver.DepthTestSpan = depth_test_span_generic_32;
			ctx->Driver.DepthTestPixels = depth_test_pixels_generic_32;
			break;
		case DB_32BIT:
			dbBuffer = depthBuffer - 4 * ( dbHeight - 1 ) * dbPitch;
		 	ctx->Driver.ReadDepthSpanFloat = read_depth_span_float_32;
			ctx->Driver.ReadDepthSpanInt = read_depth_span_int_32;
			ctx->Driver.DepthTestSpan = depth_test_span_generic_32;
			ctx->Driver.DepthTestPixels = depth_test_pixels_generic_32;
			break;
		case DB_16BIT:
			dbBuffer = depthBuffer - 2 * ( dbHeight - 1 ) * dbPitch;
		 	ctx->Driver.ReadDepthSpanFloat = read_depth_span_float_16;
			ctx->Driver.ReadDepthSpanInt = read_depth_span_int_16;
			ctx->Driver.DepthTestSpan = depth_test_span_generic_16;
			ctx->Driver.DepthTestPixels = depth_test_pixels_generic_16;
			break;
		default:
			FatalError( "Unsupported depth buffer format" );
			break;
		}
	}			
}


GLbitfield FallbackBufferClear( GLcontext *ctx, GLbitfield mask, GLboolean all,
		     GLint x, GLint y, GLint width, GLint height ) { 
	int		i, j;
	int		pixel;
	int		r, g, b,a;

	if (all == GL_TRUE) {
		x = 0;
		y = 0;
		width = cbWidth;
		height = cbHeight;
	}
	
	if ( y + height > cbHeight ) {
		height = cbHeight - y;
	}
	if ( x + width > cbWidth ) {
		width = cbWidth - x;
	}
	if ( x < 0 ) {
		width += x;
		x = 0;
	}
	if ( y < 0 ) {
		height += y;
		y = 0;
	}
	if ( x >= cbWidth || y >= cbHeight || width < 1 || height < 1 ) {
		return 0;
	}

	if ( mask & GL_COLOR_BUFFER_BIT ) {
		// FIXME: do we have to check colormask?
		r = 255 * ctx->Color.ClearColor[0];
		g = 255 * ctx->Color.ClearColor[1];
		b = 255 * ctx->Color.ClearColor[2];
		
		if ( cbFormat == CB_32BIT_BGRA ) {
			pixel = PACKCOLOR8888(r,g,b,a);
			
			for ( i = 0 ; i < height ; i++ ) {
				unsigned *dest;
	
				dest = (unsigned *)cbBuffer + ( y + i ) * cbPitch + x;
				
				for ( j = 0 ; j < width ; j++ ) {
					dest[j] = pixel;
				}
			}	
		} else {
			if ( cbFormat == CB_15BIT ) {
				pixel = PACKCOLOR555(r,g,b,a);
			} else {
				pixel = PACKCOLOR565(r,g,b);
			}
			
			for ( i = 0 ; i < height ; i++ ) {
				unsigned short *dest;
	
				dest = (unsigned short *)cbBuffer + ( y + i ) * cbPitch + x;
				
				for ( j = 0 ; j < width ; j++ ) {
					dest[j] = pixel;
				}
			}	
		}	
		mask &= ~GL_COLOR_BUFFER_BIT;
	}
	
	// FIXME: stencil + depth case
	
	// if we don't have a depth buffer, clears do nothing
	if ( !dbBuffer ) {
		mask &= ~GL_DEPTH_BUFFER_BIT;
	}
	
	if ( mask & GL_DEPTH_BUFFER_BIT ) {
		// FIXME: do we have to check depth mask?
		unsigned	zval;

		if ( dbFormat == DB_16BIT ) {	
			zval = (unsigned) (ctx->Depth.Clear * 0xffff);

			for ( i = 0 ; i < height ; i++ ) {
				unsigned short *dest;

				dest = (unsigned short *)dbBuffer + ( y + i ) * dbPitch + x;
	
				for ( j = 0 ; j < width ; j++ ) {
					dest[j] = zval;
				}
			}
			mask &= ~GL_DEPTH_BUFFER_BIT;
		} else if ( dbFormat == DB_32BIT ) {
			zval = (unsigned) (ctx->Depth.Clear * 0xffffffff);

			for ( i = 0 ; i < height ; i++ ) {
				unsigned *dest;

				dest = (unsigned *)dbBuffer + ( y + i ) * dbPitch + x;
	
				for ( j = 0 ; j < width ; j++ ) {
					dest[j] = zval;
				}
			}
			mask &= ~GL_DEPTH_BUFFER_BIT;
		}	
	}	

	return mask;
}


static void swDDClearColor( GLcontext *ctx, GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha )
{
	/* we just look in the context when we clear, but mesa
	 * requires this function to exist
	 */
}

static void swDDAllocDepthBuffer( GLcontext *ctx )
{
	/* we always allocate the depth buffer with the color buffer */
}

static void swDDUpdateState( GLcontext *ctx )
{
	/* we always allocate the depth buffer with the color buffer */
}

static GLboolean swDDSetBuffer( GLcontext *ctx, GLenum mode )
{
	return 1;
}

const GLubyte *swDDGetString( GLcontext *ctx, GLenum name )
{
   switch (name) {
   case GL_VENDOR:
      return "Utah GLX";
   case GL_RENDERER:
      return "software";
   default:
      return "";
   }
}

/*
 * This will clear and set all ctx->Driver values for a software only
 * rendering.  Must be followed by a call to SetDriverBufferFunctions.
 */
void SetSoftwareDriverFunctions( GLcontext *ctx ) {
	memset( &ctx->Driver, 0, sizeof( ctx->Driver ) );
	
//	ctx->Driver.GetBufferSize = swDDGetBufferSize;
	ctx->Driver.AllocDepthBuffer = swDDAllocDepthBuffer;
	ctx->Driver.SetBuffer = swDDSetBuffer;
	ctx->Driver.ClearColor = swDDClearColor;
	ctx->Driver.UpdateState = swDDUpdateState;
	ctx->Driver.Clear = FallbackBufferClear;
	ctx->Driver.GetString = swDDGetString;
}
