/* $Id: hw_agp.h,v 1.3 2000/03/27 10:22:27 gareth Exp $ */

/*
 * AGP memory handling.
 *
 * The current interface allows blocks of AGP memory to be allocated and
 * deallocated as required.  There is no upper limit other than the size
 * of the aperture, and blocks will be obtained from the agpgart module
 * on demand.
 *
 * There is scope to change this allocation scheme to a growable-buffer
 * approach, but the typical use of AGP memory shouldn't require this.
 * That is, DMA buffers that reside in AGP space will be allocated on
 * startup, and AGP textures should be allocated on program startup and
 * freed on program termination.  There will be a performance hit if the
 * application creates and deletes a lot of textures.  glTexSubImage*()
 * is the preferred way to update textures.
 *
 * There will also be space lost when allocating small buffers, as only
 * multiples of 4k pages can be obtained.
 *
 * Gareth Hughes <gareth@precisioninsight.com>
 */

#ifndef __HW_AGP_H__
#define __HW_AGP_H__

#ifdef HAVE_LINUX_NEWAGP
#include <linux/agpgart.h>
#endif

#include "hwtypes.h"
#include "mm.h"

typedef struct {
	PMemBlock	memBlock;	/* heap info, offset */
#ifdef HAVE_LINUX_NEWAGP
	agp_allocate	entry;		/* agpgart module info */
#endif
	unsigned char	*buffer;	/* virtual buffer address */
	int		size;

	int		age;		/* age of block, taken care of by
					   the driver as required */
} agp_mem_t;


/*
 * Initialize the agpgart module and AGP aperture.  Map the agpgart device
 * into memory.  Set AGP 1X, 2X or 4X mode as required.  This function must
 * be called before any other access to AGP memory is made.  If limit is
 * non-zero, set the upper limit of available AGP memory to this amount (in
 * megabytes).
 *
 * If any stage of the initialization fails, close the device filehandle and
 * return an error code.
 */
int hwInitAGPMem( int mode, int limit );

/*
 * Get the current status of the AGP module.  Useful for obtaining the size
 * and base of the aperture for hardware-specific initialization, among
 * other things.  This function takes a void pointer that will be cast to an
 * agp_info pointer when the agpgart module is present, preventing dependence
 * on the agpgart headers when the module is not present.
 */
int hwGetAGPInfo( void *info );

/*
 * Allocate and deallocate blocks of memory in the AGP aperture.  Pointers to
 * the virtual memory mapping of the block will be set up, as well as offsets
 * from the base of the aperture for programming hardware access to the block.
 */
int hwAllocAGPMem( agp_mem_t *b, size_t size );
int hwFreeAGPMem( agp_mem_t *b );

/*
 * Clean up the agpgart module - direct clients must call this upon exit.
 *
 * Unmap the agpgart device and close the device filehandle.  Currently, any
 * memory allocated must be explicitly deallocated and unbound using
 * hwFreeAGPMem(), but this may change.
 */
int hwReleaseAGPMem( void );

/*
 * Display information about the AGP heap, including all blocks of memory
 * currently allocated and bound.
 */
void hwDumpAGPMemInfo( void );


#endif


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
