/* usplash
 *
 * Copyright © 2006 Canonical Ltd.
 * Copyright © 2006 Dennis Kaarsemaker <dennis@kaarsemaker.net>
 * Copyright © 2005 Matthew Garrett <mjg59@srcf.ucam.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

#include <linux/vt.h>

#include <sys/select.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#include <dlfcn.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>

#include "bogl.h"
#include "usplash.h"
#include "usplash-theme.h"


/* Prototypes of static functions */
static void switch_console    (int vt);
static void restore_console   (void);

static int  main_loop         (void);
static int  read_command      (void);
static int  parse_command     (const char *string, size_t len);

static void clear_screen      (void);

static void clear_progressbar (void);
static void draw_progressbar  (int percentage);

static void clear_text        (void);
static void draw_text         (const char *string, size_t len);
static void draw_line         (const char *string, size_t len);
static void draw_status       (const char *string, size_t len, int mode);


/* Default theme, used when no suitable alternative can be found */
extern struct usplash_theme testcard_theme;

/* Theme being used */
static struct usplash_theme *theme;

/* Distance to themed area from the top-left corner of the screen */
static int left_edge, top_edge;


/* Virtual terminal we switched away from */
static int saved_vt = 0;

/* Virtual terminal we switched to */
static int new_vt = 0;


/* File descriptor to read commands from */
static int fifo_fd;

/* Number of seconds to wait for a command before exiting */
static int timeout = 15;


int
main (int   argc,
      char *argv[])
{
	void *theme_handle;
	int   i, ret = 0;

	for (i = 1; i < argc; i++) {
		if (! strcmp (argv[i], "-c")) {
			switch_console (8);
		} else {
			fprintf (stderr, "unknown argument: %s\n", argv[i]);
			exit (1);
		}
	}


	if (chdir (USPLASH_DIR) < 0) {
		perror ("chdir");
		ret = 1;
		goto exit;
	}

	if (mkfifo (USPLASH_FIFO, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP) < 0) {
		if (errno != EEXIST) {
			perror ("mkfifo");
			ret = 1;
			goto exit;
		}
	}

	fifo_fd = open (USPLASH_FIFO, O_RDONLY|O_NONBLOCK);
	if (fifo_fd < 0) {
		perror ("open");
		ret = 2;
		goto exit;
	}


	theme_handle = dlopen (USPLASH_THEME, RTLD_LAZY);
	if (theme_handle) {
		theme = dlsym (theme_handle, "usplash_theme");
		if ((theme == NULL) || (theme->version != THEME_VERSION)) {
			dlclose (theme_handle);
			theme = &testcard_theme;
		}
	} else {
		theme = &testcard_theme;
	}

	if (! bogl_init ()) {
		fprintf (stderr, "bogl_init failed\n");
		ret = 3;
		goto exit;
	}


	left_edge = (bogl_xres - theme->pixmap->width)  / 2;
	top_edge  = (bogl_yres - theme->pixmap->height) / 2;

	bogl_set_palette (0, theme->pixmap->ncols, theme->pixmap->palette);

	clear_screen ();
	clear_progressbar ();
	clear_text ();

	ret = main_loop ();

	bogl_done ();
exit:
	restore_console ();
	return ret;
}


static void
switch_console (int vt)
{
	char           vtname[10];
	struct vt_stat state;
	int            fd;

	assert ((vt >= 0) && (vt < 10));
	sprintf (vtname, "/dev/tty%d", vt);

	fd = open ("/dev/console", O_RDWR);
	ioctl (fd, VT_GETSTATE, &state);
	close (fd);

	saved_vt = state.v_active;
	new_vt = vt;

	fd = open (vtname, O_RDWR);
	ioctl (fd, VT_ACTIVATE, vt);
	ioctl (fd, VT_WAITACTIVE, vt);
	close (fd);
}

static void
restore_console (void)
{
	struct vt_stat state;
	int            fd;

	if (saved_vt != 0) {
                fd = open ("/dev/console", O_RDWR);
                ioctl (fd, VT_GETSTATE, &state);
                close (fd);

		/* Switch back if we're still on the console we switched to */
                if (state.v_active == new_vt)
			switch_console (saved_vt);
	}
}


static int
main_loop (void)
{

	for (;;) {
		struct timeval tv;
		fd_set         rfds;
		int            retval;

		FD_ZERO (&rfds);
		FD_SET (fifo_fd, &rfds);

		tv.tv_sec = timeout;
		tv.tv_usec = 0;

		retval = select (fifo_fd + 1, &rfds, NULL, NULL, &tv);

		if (retval < 0) {
			/* Error */
			return 1;
		} else if (retval > 0) {
			int ret;

			/* Data available */
			ret = read_command ();
			if (ret)
				return ret;
		} else {
			/* Timeout */
			return 0;
		}
	}

	/* Not reached */
	return 0;
}

static int
read_command (void)
{
	static char    buf[4096], *ptr;
	static size_t  buflen;
	static ssize_t len;

	len = read (fifo_fd, buf + buflen, sizeof (buf) - buflen);
	if (len < 0) {
		if (errno != EAGAIN) {
			/* Try opening again */
			close (fifo_fd);
			fifo_fd = open (USPLASH_FIFO, O_RDONLY|O_NONBLOCK);
			if (fifo_fd < 0)
				return 2;
		}

		return 0;
	} else if (len == 0) {
		/* Reopen to see if there's anything more for us */
		close (fifo_fd);
		fifo_fd = open (USPLASH_FIFO, O_RDONLY|O_NONBLOCK);
		if (fifo_fd < 0)
			return 2;

		return 0;
	}

	buflen += len;
	while ((ptr = memchr (buf, '\0', buflen)) != NULL) {
		int ret;

		ret = parse_command (buf, ptr - buf);
		if (ret)
			return ret;

		/* Move the rest of the buffer up */
		buflen -= ptr - buf + 1;
		memcpy (buf, ptr + 1, buflen);
	}

	return 0;
}

static size_t
strncspn (const char *s, size_t n, const char *reject)
{
	register size_t l;

	for (l = 0; l < n; l++)
		if (strchr (reject, s[l]))
			break;

	return l;
}

static int
parse_command (const char *string, size_t len)
{
	const char *command;
	size_t      commandlen;

	command = string;
	commandlen = strncspn (string, len, " ");

	if (string[commandlen] == ' ') {
		string += commandlen + 1;
		len -= commandlen + 1;
	} else {
		len = 0;
	}


	if (! strncmp (command, "QUIT", commandlen)) {
		return 1;

	} else if (! strncmp (command, "TIMEOUT", commandlen)) {
		timeout = atoi (string);

	} else if (! strncmp (command, "CLEAR", commandlen)) {
		clear_text ();

	} else if (! strncmp (command, "TEXT", commandlen)) {
		draw_text (string, len);

	} else if (! strncmp (command, "STATUS", commandlen)) {
		draw_status (string, len, 0);

	} else if (! strncmp (command, "SUCCESS", commandlen)) {
		draw_status (string, len, 1);

	} else if (! strncmp (command, "FAILURE", commandlen)) {
		draw_status (string, len, -1);

	} else if (! strncmp (command, "PROGRESS", commandlen)) {
		draw_progressbar (atoi (string));
	}

	return 0;
}


static void
clear_screen (void)
{
	const int colour_map[] = { 0x0, 0x1, 0x2, 0x3, 0x4, 0x5, 0x6, 0x7,
				   0x8, 0x9, 0xa, 0xb, 0xc, 0xd, 0xe, 0xf };

	bogl_clear (0, 0, bogl_xres, bogl_yres, theme->background);

	bogl_put (left_edge, top_edge, theme->pixmap, colour_map);
}


static void
clear_progressbar (void)
{
	int x1, y1, x2, y2;

	x1 = left_edge + theme->progressbar_x;
	y1 = top_edge + theme->progressbar_y;

	x2 = x1 + theme->progressbar_width;
	y2 = y1 + theme->progressbar_height;

	bogl_clear (x1, y1, x2, y2, theme->progressbar_background);
}

static void
draw_progressbar (int percentage)
{
	int x1, y1, x2, y2, xx, bg, fg;

	if (percentage < 0) {
		bg = theme->progressbar_foreground;
		fg = theme->progressbar_background;
		percentage = -percentage;
	} else {
		bg = theme->progressbar_background;
		fg = theme->progressbar_foreground;
	}

	if (percentage > 100)
		return;


	x1 = left_edge + theme->progressbar_x;
	y1 = top_edge + theme->progressbar_y;

	x2 = x1 + theme->progressbar_width;
	y2 = y1 + theme->progressbar_height;

	xx = x1 + ((theme->progressbar_width * percentage) / 100);

	bogl_clear (x1, y1, xx, y2, fg);
	bogl_clear (xx, y1, x2, y2, bg);
}


static void
clear_text (void)
{
	int x1, y1, x2, y2;

	x1 = left_edge + theme->text_x;
	y1 = top_edge + theme->text_y;

	x2 = x1 + theme->text_width;
	y2 = y1 + theme->text_height;

	bogl_clear (x1, y1, x2, y2, theme->text_background);
}

static void
draw_text (const char *string, size_t len)
{
	/* FIXME some kind of better word wrapping here, perhaps? */
#if 0
	while (len > theme->line_length) {
		draw_line (string, theme->line_length);
		string += theme->line_length;
		len -= theme->line_length;
	}
#endif

	draw_line (string, len);
}

static void
draw_line (const char *string, size_t len)
{
	int x1, y1, x2, y2;

	x1 = left_edge + theme->text_x;
	x2 = x1 + theme->text_width;

	y2 = top_edge + theme->text_y + theme->text_height;
	y1 = y2 - theme->line_height;

	/* Move existing text up */
	bogl_move (x1, top_edge + theme->text_y + theme->line_height,
		   x1, top_edge + theme->text_y,
		   theme->text_width, theme->text_height - theme->line_height);


	bogl_clear (x1, y1, x2, y2, theme->text_background);

	bogl_text (x1, y1, string, len,
		   theme->text_foreground, theme->text_background,
		   0, theme->font);
}

static void
draw_status (const char *string, size_t len, int mode)
{
	int x1, y1, x2, y2, fg;

	if (mode < 0) {
		fg = theme->text_failure;
	} else if (mode > 0) {
		fg = theme->text_success;
	} else {
		fg = theme->text_foreground;
	}

	x2 = left_edge + theme->text_x + theme->text_width;
	y2 = top_edge + theme->text_y + theme->text_height;

	x1 = x2 - theme->status_width;
	y1 = y2 - theme->line_height;

	bogl_clear (x1, y1, x2, y2, theme->text_background);

	bogl_text (x1, y1, string, len, fg, theme->text_background,
		   0, theme->font);
}
