/*****************************************************************************/

/*
 *      lsusb.c  --  lspci like utility for the USB bus
 *
 *      Copyright (C) 1999, 2000, 2001  Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 */

/*****************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <stdarg.h>

#include <linux/types.h>
#ifdef HAVE_LINUX_USBDEVICE_FS_H
#include <linux/usbdevice_fs.h>
#else
#include "usbdevice_fs.h"
#endif
#ifdef HAVE_LINUX_USB_H
#include <linux/usb.h>
#else
#include "usb.h"
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "names.h"
#include "devtree.h"

#define CTRL_RETRIES	 50
#define CTRL_TIMEOUT	100	/* milliseconds */

#define USB_DT_CS_DEVICE	       	0x21
#define USB_DT_CS_CONFIG	       	0x22
#define USB_DT_CS_STRING		0x23
#define USB_DT_CS_INTERFACE		0x24
#define USB_DT_CS_ENDPOINT		0x25

static const char *procbususb = "/proc/bus/usb";
static unsigned int verblevel = 1;
static int do_report_desc = 1;

static void dump_junk2(unsigned char *, unsigned int);

/* ---------------------------------------------------------------------- */

int lprintf(unsigned int vl, const char *format, ...)
{
        va_list ap;
        int r;

        if (vl > verblevel)
                return 0;
        va_start(ap, format);
	r = vfprintf(stderr, format, ap);
        va_end(ap);
        if (!vl)
                exit(1);
        return r;
}

/* ---------------------------------------------------------------------- */

static int usb_control_msg(int fd, u_int8_t requesttype, u_int8_t request, u_int16_t value,
			   u_int16_t index, unsigned int size, void *data)
{
	struct usbdevfs_ctrltransfer ctrl;
	int result, try;

	ctrl.requesttype = requesttype;
	ctrl.request = request;
	ctrl.value = value;
	ctrl.index = index;
	ctrl.length = size;
	ctrl.timeout = 1000;
	ctrl.data = data;
 	ctrl.timeout = CTRL_TIMEOUT; 
 	try = 0;
 	do {
 	  	result = ioctl(fd, USBDEVFS_CONTROL, &ctrl);
 		try++;
 	} while (try < CTRL_RETRIES && result == -1 && errno == ETIMEDOUT);
 	return result;
}

/* ---------------------------------------------------------------------- */

static int get_string(int fd, char *buf, size_t size, u_int8_t id, u_int16_t lang)
{
	unsigned char b[256];
	wchar_t w[128];
	unsigned int i;
	int ret;

	if (size < 1)
		return 0;
	*buf = 0;
	/* string ID 0 means no string */
	if (!id || fd == -1)
		return 0;

	b[0] = b[1] = 0xbf;
	ret = usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR, (USB_DT_STRING << 8) | id, 0, sizeof(b), b);
	if (ret < 0) {
		fprintf(stderr, "cannot get string descriptor %d, error = %s(%d)\n", id, strerror(errno), errno);
		return 0;
	}
	if (ret < 2 || b[0] < 2 || b[1] != USB_DT_STRING) {
		fprintf(stderr, "string descriptor %d invalid (%02x %02x; len=%d)\n", id, b[0], b[1], ret);
		return 0;
	}
#if 0
	for (i = 0; i < ((b[0] - 2) / 2); i++)
		w[i] = b[2+2*i] | (b[3+2*i] << 8);
	w[i] = 0;
	return wcstombs(buf, w, size);
#else
        for (i = 0; i < ((b[0] - 2) / 2); i++)
                buf[i] = b[2+2*i];
        buf[i] = 0;
        return i;
#endif        
}

static int get_vendor_string(char *buf, size_t size, u_int16_t vid)
{
	const char *cp;

	if (size < 1)
		return 0;
	*buf = 0;
	if (!(cp = names_vendor(vid)))
		return 0;
	return snprintf(buf, size, "%s", cp);
}

static int get_product_string(char *buf, size_t size, u_int16_t vid, u_int16_t pid)
{
	const char *cp;

	if (size < 1)
		return 0;
	*buf = 0;
	if (!(cp = names_product(vid, pid)))
		return 0;
	return snprintf(buf, size, "%s", cp);
}

static int get_class_string(char *buf, size_t size, u_int8_t cls)
{
	const char *cp;

	if (size < 1)
		return 0;
	*buf = 0;
	if (!(cp = names_class(cls)))
		return 0;
	return snprintf(buf, size, "%s", cp);
}

static int get_subclass_string(char *buf, size_t size, u_int8_t cls, u_int8_t subcls)
{
	const char *cp;

	if (size < 1)
		return 0;
	*buf = 0;
	if (!(cp = names_subclass(cls, subcls)))
		return 0;
	return snprintf(buf, size, "%s", cp);
}

static int get_protocol_string(char *buf, size_t size, u_int8_t cls, u_int8_t subcls, u_int8_t proto)
{
	const char *cp;

	if (size < 1)
		return 0;
	*buf = 0;
	if (!(cp = names_protocol(cls, subcls, proto)))
		return 0;
	return snprintf(buf, size, "%s", cp);
}

static int get_audioterminal_string(char *buf, size_t size, u_int16_t termt)
{
	const char *cp;

	if (size < 1)
		return 0;
	*buf = 0;
	if (!(cp = names_audioterminal(termt)))
		return 0;
	return snprintf(buf, size, "%s", cp);
}

/* ---------------------------------------------------------------------- */

static void dump_junk2(unsigned char *buf, unsigned int len)
{
	unsigned int i;

	for (i = 0; i < len; i++)
		printf(" %02x", buf[i]);
	printf("\n");
}

static void dump_junk(unsigned char *buf, const char *indent, unsigned int len)
{
	unsigned int i;

	if (buf[0] <= len)
		return;
	printf("%sjunk at descriptor end:", indent);
	for (i = len; i < buf[0]; i++)
		printf(" %02x", buf[i]);
	printf("\n");
}

/*
 * General config descriptor dump
 */

static void dump_device(int fd, unsigned char *buf, u_int16_t lang)
{
	unsigned int vid, pid;
	char vendor[128], product[128];
	char cls[128], subcls[128], proto[128];
	char mfg[128], prod[128], serial[128];

	if (buf[1] != USB_DT_DEVICE)
		printf("  Warning: Invalid descriptor\n");
	else if (buf[0] < 18)
		printf("  Warning: Descriptor too short\n");
	vid = buf[8] | (buf[9] << 8);
	pid = buf[10] | (buf[11] << 8);
	get_vendor_string(vendor, sizeof(vendor), vid);
	get_product_string(product, sizeof(product), vid, pid);
	get_class_string(cls, sizeof(cls), buf[4]);
	get_subclass_string(subcls, sizeof(subcls), buf[4], buf[5]);
	get_protocol_string(proto, sizeof(proto), buf[4], buf[5], buf[6]);
	get_string(fd, mfg, sizeof(mfg), buf[14], lang);
	get_string(fd, prod, sizeof(prod), buf[15], lang);
	get_string(fd, serial, sizeof(serial), buf[16], lang);
	printf("Device Descriptor:\n"
	       "  bLength             %5u\n"
	       "  bDescriptorType     %5u\n"
	       "  bcdUSB              %2x.%02x\n"
	       "  bDeviceClass        %5u %s\n"
	       "  bDeviceSubClass     %5u %s\n"
	       "  bDeviceProtocol     %5u %s\n"
	       "  bMaxPacketSize0     %5u\n"
	       "  idVendor           0x%04x %s\n"
	       "  idProduct          0x%04x %s\n"
	       "  bcdDevice           %2x.%02x\n"
	       "  iManufacturer       %5u %s\n"
	       "  iProduct            %5u %s\n"
	       "  iSerial             %5u %s\n"
	       "  bNumConfigurations  %5u\n",
	       buf[0], buf[1], buf[3], buf[2], buf[4], cls, buf[5], subcls, buf[6], proto, buf[7],
	       vid, vendor, pid, product, buf[13], buf[12], buf[14], mfg, buf[15], prod, buf[16], serial, buf[17]);
	dump_junk(buf, "  ", 18);
}

static void dump_config(int fd, unsigned char *buf)
{
	if (buf[1] != USB_DT_CONFIG)
		printf("  Warning: Invalid descriptor\n");
	else if (buf[0] < 9)
		printf("  Warning: Descriptor too short\n");
	printf("  Configuration Descriptor:\n"
	       "    bLength             %5u\n"
	       "    bDescriptorType     %5u\n"
	       "    wTotalLength        %5u\n"
	       "    bNumInterfaces      %5u\n"
	       "    bConfigurationValue %5u\n"
	       "    iConfiguration      %5u\n"
	       "    bmAttributes         0x%02x\n",
	       buf[0], buf[1], buf[2] | (buf[3] << 8), buf[4], buf[5], buf[6], buf[7]);
	if (buf[7] & 0x40)
		printf("      Self Powered\n");
	if (buf[7] & 0x20)
		printf("      Remote Wakeup\n");
	printf("    MaxPower            %5umA\n", buf[8] * 2);
	dump_junk(buf, "    ", 9);
}

static void dump_interface(int fd, unsigned char *buf, u_int16_t lang)
{
	char cls[128], subcls[128], proto[128];
	char ifstr[128];

	if (buf[1] != USB_DT_INTERFACE)
		printf("    Warning: Invalid descriptor\n");
	else if (buf[0] < 9)
		printf("    Warning: Descriptor too short\n");
	get_class_string(cls, sizeof(cls), buf[5]);
	get_subclass_string(subcls, sizeof(subcls), buf[5], buf[6]);
	get_protocol_string(proto, sizeof(proto), buf[5], buf[6], buf[7]);
	get_string(fd, ifstr, sizeof(ifstr), buf[8], lang);
	printf("    Interface Descriptor:\n"
	       "      bLength             %5u\n"
	       "      bDescriptorType     %5u\n"
	       "      bInterfaceNumber    %5u\n"
	       "      bAlternateSetting   %5u\n"
	       "      bNumEndpoints       %5u\n"
	       "      bInterfaceClass     %5u %s\n"
	       "      bInterfaceSubClass  %5u %s\n"
	       "      bInterfaceProtocol  %5u %s\n"
	       "      iInterface          %5u %s\n",
	       buf[0], buf[1], buf[2], buf[3], buf[4], buf[5], cls, 
	       buf[6], subcls, buf[7], proto, buf[8], ifstr);
	dump_junk(buf, "      ", 9);
}

static void dump_endpoint(int fd, unsigned char *buf)
{
	static const char *typeattr[] = { "Control", "Isochronous", "Bulk", "Interrupt" };
	static const char *syncattr[] = { "none", "Asynchronous", "Adaptive", "Synchronous" };

	if (buf[1] != USB_DT_ENDPOINT)
		printf("      Warning: Invalid descriptor\n");
	else if (buf[0] < 7)
		printf("      Warning: Descriptor too short\n");
	printf("      Endpoint Descriptor:\n"
	       "        bLength             %5u\n"
	       "        bDescriptorType     %5u\n"
	       "        bEndpointAddress     0x%02x  EP %u %s\n"
	       "        bmAttributes        %5u\n"
	       "          Transfer Type            %s\n"
	       "          Synch Type               %s\n"
	       "        wMaxPacketSize      %5u\n"
	       "        bInterval           %5u\n",
	       buf[0], buf[1], buf[2], buf[2] & 15, (buf[2] & 0x80) ? "IN" : "OUT", 
	       buf[3], typeattr[buf[3] & 3], syncattr[(buf[3] >> 2) & 3],
	       buf[4] | (buf[5] << 8), buf[6]);
	if (buf[0] < 9) {
		dump_junk(buf, "        ", 7);
		return;
	}
	printf("        bRefresh            %5u\n"
	       "        bSynchAddress       %5u\n",
	       buf[7], buf[8]);
	dump_junk(buf, "        ", 9);
}

/* ---------------------------------------------------------------------- */

/*
 * Audio Class descriptor dump
 */

static void dump_audiocontrol_interface(int fd, unsigned char *buf, u_int16_t lang)
{
	static const char *chconfig[] = {
		"Left Front (L)", "Right Front (R)", "Center Front (C)", "Low Freqency Enhancement (LFE)",
		"Left Surround (LS)", "Right Surround (RS)", "Left of Center (LC)", "Right of Center (RC)",
		"Surround (S)", "Side Left (SL)", "Side Right (SR)", "Top (T)"
	};
	static const char *chftrcontrols[] = {
		"Mute", "Volume", "Bass", "Mid", "Treble", "Graphic Equalizer", "Automatic Gain", "Delay", "Bass Boost", "Loudness"
	};
	unsigned int i, chcfg, j, k, N, termt;
	char chnames[128], term[128], termts[128];

	if (buf[1] != USB_DT_CS_INTERFACE)
		printf("      Warning: Invalid descriptor\n");
	else if (buf[0] < 3)
		printf("      Warning: Descriptor too short\n");
	printf("      AudioControl Interface Descriptor:\n"
	       "        bLength             %5u\n"
	       "        bDescriptorType     %5u\n"
	       "        bDescriptorSubtype  %5u ",
	       buf[0], buf[1], buf[2]);
	switch (buf[2]) {
	case 0x01:  /* HEADER */
		printf("(HEADER)\n");
		if (buf[0] < 8+buf[7])
			printf("      Warning: Descriptor too short\n");
		printf("        bcdADC              %2x.%02x\n"
		       "        wTotalLength        %5u\n"
		       "        bInCollection       %5u\n",
		       buf[4], buf[3], buf[5] | (buf[6] << 8), buf[7]);
		for(i = 0; i < buf[7]; i++)
			printf("        baInterfaceNr(%2u)   %5u\n", i, buf[8+i]);
		dump_junk(buf, "        ", 8+buf[7]);
		break;

	case 0x02:  /* INPUT_TERMINAL */
		printf("(INPUT_TERMINAL)\n");
		get_string(fd, chnames, sizeof(chnames), buf[10], lang);
		get_string(fd, term, sizeof(term), buf[11], lang);
		termt = buf[4] | (buf[5] << 8);
		get_audioterminal_string(termts, sizeof(termts), termt);
		if (buf[0] < 12)
			printf("      Warning: Descriptor too short\n");
		chcfg = buf[8] | (buf[9] << 8);
		printf("        bTerminalID         %5u\n"
		       "        wTerminalType      0x%04x %s\n"
		       "        bAssocTerminal      %5u\n"
		       "        bNrChannels         %5u\n"
		       "        wChannelConfig     0x%04x\n",
		       buf[3], termt, termts, buf[6], buf[7], chcfg);
		for (i = 0; i < 12; i++)
			if ((chcfg >> i) & 1)
				printf("          %s\n", chconfig[i]);
		printf("        iChannelNames       %5u %s\n"
		       "        iTerminal           %5u %s\n",
		       buf[10], chnames, buf[11], term);	
		dump_junk(buf, "        ", 12);
		break;

	case 0x03:  /* OUTPUT_TERMINAL */
		printf("(OUTPUT_TERMINAL)\n");
		get_string(fd, term, sizeof(term), buf[8], lang);
		termt = buf[4] | (buf[5] << 8);
		get_audioterminal_string(termts, sizeof(termts), termt);
		if (buf[0] < 9)
			printf("      Warning: Descriptor too short\n");
		printf("        bTerminalID         %5u\n"
		       "        wTerminalType      0x%04x %s\n"
		       "        bAssocTerminal      %5u\n"
		       "        bSourceID           %5u\n"
		       "        iTerminal           %5u %s\n",
		       buf[3], termt, termts, buf[6], buf[7], buf[8], term);	
		dump_junk(buf, "        ", 9);
		break;

	case 0x04:  /* MIXER_UNIT */
		printf("(MIXER_UNIT)\n");
		j = buf[4];
		k = buf[j+5];
		if (j == 0 || k == 0) { 
		  printf("      Warning: mixer with %5u input and %5u output channels.\n", j, k);
		  N = 0;
		} else {
		  N = 1+(j*k-1)/8;
		}
		get_string(fd, chnames, sizeof(chnames), buf[8+j], lang);
		get_string(fd, term, sizeof(term), buf[9+j+N], lang);
		if (buf[0] < 10+j+N)
			printf("      Warning: Descriptor too short\n");
		chcfg = buf[6+j] | (buf[7+j] << 8);
		printf("        bUnitID             %5u\n"
		       "        bNrInPins           %5u\n",
		       buf[3], buf[4]);
		for (i = 0; i < j; i++)
			printf("        baSourceID(%2u)      %5u\n", i, buf[5+i]);
		printf("        bNrChannels         %5u\n"
		       "        wChannelConfig     0x%04x\n",
		       buf[5+j], chcfg);
		for (i = 0; i < 12; i++)
			if ((chcfg >> i) & 1)
				printf("          %s\n", chconfig[i]);
		printf("        iChannelNames       %5u %s\n",
		       buf[8+j], chnames);
		for (i = 0; i < N; i++)
			printf("        bmControls         0x%02x\n", buf[9+j+i]);
		printf("        iMixer              %5u %s\n", buf[9+j+N], term);	
		dump_junk(buf, "        ", 10+j+N);
		break;

	case 0x05:  /* SELECTOR_UNIT */
		printf("(SELECTOR_UNIT)\n");
		if (buf[0] < 6+buf[4])
			printf("      Warning: Descriptor too short\n");
		get_string(fd, term, sizeof(term), buf[5+buf[4]], lang);

		printf("        bUnitID             %5u\n"
		       "        bNrInPins           %5u\n",
		       buf[3], buf[4]);
		for (i = 0; i < buf[4]; i++)
			printf("        baSource(%2u)        %5u\n", i, buf[5+i]);
		printf("        iSelector           %5u %s\n",
		       buf[5+buf[4]], term);
		dump_junk(buf, "        ", 6+buf[4]);
		break;

	case 0x06:  /* FEATURE_UNIT */
		printf("(FEATURE_UNIT)\n");
		j = buf[5];
		if (!j)
			j = 1;
		k = (buf[0] - 7) / j;
		if (buf[0] < 7+buf[5]*k)
			printf("      Warning: Descriptor too short\n");
		get_string(fd, term, sizeof(term), buf[6+buf[5]*k], lang);
		printf("        bUnitID             %5u\n"
		       "        bSourceID           %5u\n"
		       "        bControlSize        %5u\n",
		       buf[3], buf[4], buf[5]);
		for (i = 0; i < k; i++) {
			chcfg = buf[6+buf[5]*i];
			if (buf[5] > 1)
				chcfg |= (buf[7+buf[5]*i] << 8);
			for (j = 0; j < buf[5]; j++)
				printf("        bmaControls(%2u)      0x%02x\n", j, buf[6+buf[5]*i+j]);
			for (j = 0; j < 10; j++)
				if ((chcfg >> j) & 1)
					printf("          %s\n", chftrcontrols[j]);
		}
		printf("        iFeature            %5u %s\n", buf[6+buf[5]*k], term);
		dump_junk(buf, "        ", 7+buf[5]*k);
		break;

	case 0x07:  /* PROCESSING_UNIT */
		printf("(PROCESSING_UNIT)\n");
		j = buf[6];
		k = buf[11+j];
		get_string(fd, chnames, sizeof(chnames), buf[10+j], lang);
		get_string(fd, term, sizeof(term), buf[12+j+k], lang);
		chcfg = buf[8+j] | (buf[9+j] << 8);
		if (buf[0] < 13+j+k)
			printf("      Warning: Descriptor too short\n");
		printf("        bUnitID             %5u\n"
		       "        wProcessType        %5u\n"
		       "        bNrPins             %5u\n",
		       buf[3], buf[4] | (buf[5] << 8), buf[6]);
		for (i = 0; i < j; i++)
			printf("        baSourceID(%2u)      %5u\n", i, buf[7+i]);
		printf("        bNrChannels         %5u\n"
		       "        wChannelConfig     0x%04x\n", buf[7+j], chcfg);
		for (i = 0; i < 12; i++)
			if ((chcfg >> i) & 1)
				printf("          %s\n", chconfig[i]);
		printf("        iChannelNames       %5u %s\n"
		       "        bControlSize        %5u\n", buf[10+j], chnames, buf[11+j]);
		for (i = 0; i < k; i++)
			printf("        bmControls(%2u)       0x%02x\n", i, buf[12+j+i]);
		if (buf[12+j] & 1)
			printf("          Enable Processing\n");
		printf("        iProcessing         %5u %s\n"
		       "        Process-Specific    ", buf[12+j+k], term);
		dump_junk2(buf+(13+j+k), buf[0]-(13+j+k));
		break;

	case 0x08:  /* EXTENSION_UNIT */
		printf("(EXTENSION_UNIT)\n");
		j = buf[6];
		k = buf[11+j];
		get_string(fd, chnames, sizeof(chnames), buf[10+j], lang);
		get_string(fd, term, sizeof(term), buf[12+j+k], lang);
		chcfg = buf[8+j] | (buf[9+j] << 8);
		if (buf[0] < 13+j+k)
			printf("      Warning: Descriptor too short\n");
		printf("        bUnitID             %5u\n"
		       "        wExtensionCode      %5u\n"
		       "        bNrPins             %5u\n",
		       buf[3], buf[4] | (buf[5] << 8), buf[6]);
		for (i = 0; i < j; i++)
			printf("        baSourceID(%2u)      %5u\n", i, buf[7+i]);
		printf("        bNrChannels         %5u\n"
		       "        wChannelConfig      %5u\n", buf[7+j], chcfg);
		for (i = 0; i < 12; i++)
			if ((chcfg >> i) & 1)
				printf("          %s\n", chconfig[i]);
		printf("        iChannelNames       %5u %s\n"
		       "        bControlSize        %5u\n", buf[10+j], chnames, buf[11+j]);
		for (i = 0; i < k; i++)
			printf("        bmControls(%2u)       0x%02x\n", i, buf[12+j+i]);
		if (buf[12+j] & 1)
			printf("          Enable Processing\n");
		printf("        iExtension          %5u %s\n",
		       buf[12+j+k], term);
		dump_junk(buf, "        ", 13+j+k);
		break;

	default:
		printf("(unknown)\n"
		       "        Invalid desc subtype:");
		dump_junk2(buf+3, buf[0]-3);
		break;
	}
}

static void dump_audiostreaming_interface(int fd, unsigned char *buf)
{
	static const char *fmtItag[] = { "TYPE_I_UNDEFINED", "PCM", "PCM8", "IEEE_FLOAT", "ALAW", "MULAW" };
	static const char *fmtIItag[] = { "TYPE_II_UNDEFINED", "MPEG", "AC-3" };
	static const char *fmtIIItag[] = { "TYPE_III_UNDEFINED", "IEC1937_AC-3", "IEC1937_MPEG-1_Layer1", 
					   "IEC1937_MPEG-Layer2/3/NOEXT", "IEC1937_MPEG-2_EXT",
					   "IEC1937_MPEG-2_Layer1_LS", "IEC1937_MPEG-2_Layer2/3_LS" };
	unsigned int i, j, fmttag;
	const char *fmtptr = "undefined";

	if (buf[1] != USB_DT_CS_INTERFACE)
		printf("      Warning: Invalid descriptor\n");
	else if (buf[0] < 3)
		printf("      Warning: Descriptor too short\n");
	printf("      AudioControl Interface Descriptor:\n"
	       "        bLength             %5u\n"
	       "        bDescriptorType     %5u\n"
	       "        bDescriptorSubtype  %5u ",
	       buf[0], buf[1], buf[2]);
	switch (buf[2]) {
	case 0x01: /* AS_GENERAL */
		printf("(AS_GENERAL)\n");
		if (buf[0] < 7)
			printf("      Warning: Descriptor too short\n");
		fmttag = buf[5] | (buf[6] << 8);
		if (fmttag >= 0 && fmttag <= 5)
			fmtptr = fmtItag[fmttag];
		else if (fmttag >= 0x1000 && fmttag <= 0x1002)
			fmtptr = fmtIItag[fmttag & 0xfff];
		else if (fmttag >= 0x2000 && fmttag <= 0x2006)
			fmtptr = fmtIIItag[fmttag & 0xfff];
		printf("        bTerminalLink       %5u\n"
		       "        bDelay              %5u frames\n"
		       "        wFormatTag          %5u %s\n",
		       buf[3], buf[4], fmttag, fmtptr);
		dump_junk(buf, "        ", 7);
		break;

	case 0x02: /* FORMAT_TYPE */
		printf("(FORMAT_TYPE)\n");
		if (buf[0] < 8)
			printf("      Warning: Descriptor too short\n");
		printf("        bFormatType         %5u ", buf[3]);
		switch (buf[3]) {
		case 0x01: /* FORMAT_TYPE_I */
			printf("(FORMAT_TYPE_I)\n");
			j = buf[7] ? (buf[7]*3+8) : 14;
			if (buf[0] < j)
				printf("      Warning: Descriptor too short\n");
			printf("        bNrChannels         %5u\n"
			       "        bSubframeSize       %5u\n"
			       "        bBitResolution      %5u\n"
			       "        bSamFreqType        %5u %s\n",
			       buf[4], buf[5], buf[6], buf[7], buf[7] ? "Discrete" : "Continuous");
			if (!buf[7])
				printf("        tLowerSamFreq     %7u\n"
				       "        tUpperSamFreq     %7u\n",
				       buf[8] | (buf[9] << 8) | (buf[10] << 16), buf[11] | (buf[12] << 8) | (buf[13] << 16));
			else
				for (i = 0; i < buf[7]; i++)
					printf("        tSamFreq[%2u]      %7u\n", i,
					       buf[8+3*i] | (buf[9+3*i] << 8) | (buf[10+3*i] << 16));
			dump_junk(buf, "        ", j);
			break;

		case 0x02: /* FORMAT_TYPE_II */
			printf("(FORMAT_TYPE_II)\n");
			j = buf[8] ? (buf[7]*3+9) : 15;
			if (buf[0] < j)
				printf("      Warning: Descriptor too short\n");
			printf("        wMaxBitRate         %5u\n"
			       "        wSamplesPerFrame    %5u\n"
			       "        bSamFreqType        %5u %s\n",
			       buf[4] | (buf[5] << 8), buf[6] | (buf[7] << 8), buf[8], buf[8] ? "Discrete" : "Continuous");
			if (!buf[8])
				printf("        tLowerSamFreq     %7u\n"
				       "        tUpperSamFreq     %7u\n",
				       buf[9] | (buf[10] << 8) | (buf[11] << 16), buf[12] | (buf[13] << 8) | (buf[14] << 16));
			else
				for (i = 0; i < buf[8]; i++)
					printf("        tSamFreq[%2u]      %7u\n", i,
					       buf[9+3*i] | (buf[10+3*i] << 8) | (buf[11+3*i] << 16));
			dump_junk(buf, "        ", j);
			break;

		case 0x03: /* FORMAT_TYPE_III */
			printf("(FORMAT_TYPE_III)\n");
			j = buf[7] ? (buf[7]*3+8) : 14;
			if (buf[0] < j)
				printf("      Warning: Descriptor too short\n");
			printf("        bNrChannels         %5u\n"
			       "        bSubframeSize       %5u\n"
			       "        bBitResolution      %5u\n"
			       "        bSamFreqType        %5u %s\n",
			       buf[4], buf[5], buf[6], buf[7], buf[7] ? "Discrete" : "Continuous");
			if (!buf[7])
				printf("        tLowerSamFreq     %7u\n"
				       "        tUpperSamFreq     %7u\n",
				       buf[8] | (buf[9] << 8) | (buf[10] << 16), buf[11] | (buf[12] << 8) | (buf[13] << 16));
			else
				for (i = 0; i < buf[7]; i++)
					printf("        tSamFreq[%2u]      %7u\n", i,
					       buf[8+3*i] | (buf[9+3*i] << 8) | (buf[10+3*i] << 16));
			dump_junk(buf, "        ", j);
			break;

		default:
			printf("(unknown)\n"
			       "        Invalid desc format type:");
			dump_junk2(buf+4, buf[0]-4);
		}
		break;

	case 0x03: /* FORMAT_SPECIFIC */
		printf("(FORMAT_SPECIFIC)\n");
		if (buf[0] < 5)
			printf("      Warning: Descriptor too short\n");
		fmttag = buf[3] | (buf[4] << 8);
		if (fmttag >= 0 && fmttag <= 5)
			fmtptr = fmtItag[fmttag];
		else if (fmttag >= 0x1000 && fmttag <= 0x1002)
			fmtptr = fmtIItag[fmttag & 0xfff];
		else if (fmttag >= 0x2000 && fmttag <= 0x2006)
			fmtptr = fmtIIItag[fmttag & 0xfff];
		printf("        wFormatTag          %5u %s\n", fmttag, fmtptr);
		switch (fmttag) {
		case 0x1001: /* MPEG */
			if (buf[0] < 8)
				printf("      Warning: Descriptor too short\n");
			printf("        bmMPEGCapabilities 0x%04x\n",
			       buf[5] | (buf[6] << 8));
			if (buf[5] & 0x01)
				printf("          Layer I\n");
			if (buf[5] & 0x02)
				printf("          Layer II\n");
			if (buf[5] & 0x04)
				printf("          Layer III\n");
			if (buf[5] & 0x08)
				printf("          MPEG-1 only\n");
			if (buf[5] & 0x10)
				printf("          MPEG-1 dual-channel\n");
			if (buf[5] & 0x20)
				printf("          MPEG-2 second stereo\n");
			if (buf[5] & 0x40)
				printf("          MPEG-2 7.1 channel augmentation\n");
			if (buf[5] & 0x80)
				printf("          Adaptive multi-channel prediction\n");
			printf("          MPEG-2 multilingual support: ");
			switch (buf[6] & 3) {
			case 0:
				printf("Not supported\n");
				break;
			
			case 1:
				printf("Supported at Fs\n");
				break;
			
			case 2:
				printf("Reserved\n");
				break;
			
			default:
				printf("Supported at Fs and 1/2Fs\n");
				break;
			}
			printf("        bmMPEGFeatures       0x%02x\n", buf[7]);
			printf("          Internal Dynamic Range Control: ");
			switch ((buf[7] << 4) & 3) {
			case 0:
				printf("not supported\n");
				break;
			
			case 1:
				printf("supported but not scalable\n");
				break;
			
			case 2:
				printf("scalable, common boost and cut scaling value\n");
				break;
			
			default:
				printf("scalable, separate boost and cut scaling value\n");
				break;
			}
			dump_junk(buf, "        ", 8);
			break;

		case 0x1002: /* AC-3 */
			if (buf[0] < 10)
				printf("      Warning: Descriptor too short\n");
			printf("        bmBSID         0x%08x\n"
			       "        bmAC3Features        0x%02x\n",
			       buf[5] | (buf[6] << 8) | (buf[7] << 16) | (buf[8] << 24), buf[9]);
			if (buf[9] & 0x0)
				printf("          RF mode\n");
			if (buf[9] & 0x0)
				printf("          Line mode\n");
			if (buf[9] & 0x0)
				printf("          Custom0 mode\n");
			if (buf[9] & 0x0)
				printf("          Custom1 mode\n");
			printf("          Internal Dynamic Range Control: ");
			switch ((buf[9] >> 4) & 3) {
			case 0:
				printf("not supported\n");
				break;
				
			case 1:
				printf("supported but not scalable\n");
				break;
				
			case 2:
				printf("scalable, common boost and cut scaling value\n");
				break;
				
			default:
				printf("scalable, separate boost and cut scaling value\n");
				break;
			}
			dump_junk(buf, "        ", 8);
			break;

		default:
			printf("(unknown)\n"
			       "        Invalid desc format type:");
			dump_junk2(buf+4, buf[0]-4);
		}
		break;

	default:
		printf("        Invalid desc subtype:");
		dump_junk2(buf+3, buf[0]-3);
		break;
	}
}

static void dump_audiostreaming_endpoint(int fd, unsigned char *buf)
{
	static const char *lockdelunits[] = { "Undefined", "Milliseconds", "Decoded PCM samples", "Reserved" };
	unsigned int lckdelidx;

	if (buf[1] != USB_DT_CS_ENDPOINT)
		printf("      Warning: Invalid descriptor\n");
	else if (buf[0] < 7)
		printf("      Warning: Descriptor too short\n");
	printf("        AudioControl Endpoint Descriptor:\n"
	       "          bLength             %5u\n"
	       "          bDescriptorType     %5u\n"
	       "          bDescriptorSubtype  %5u (%s)\n"
	       "          bmAttributes         0x%02x\n",
	       buf[0], buf[1], buf[2], buf[2] == 1 ? "EP_GENERAL" : "invalid", buf[3]);
	if (buf[3] & 1)
		printf("            Sampling Frequency\n");
	if (buf[3] & 2)
		printf("            Pitch\n");
	if (buf[3] & 128)
		printf("            MaxPacketsOnly\n");
	lckdelidx = buf[4];
	if (lckdelidx > 3)
		lckdelidx = 3;
	printf("          bLockDelayUnits     %5u %s\n"
	       "          wLockDelay          %5u %s\n",
	       buf[4], lockdelunits[lckdelidx], buf[5] | (buf[6] << 8), lockdelunits[lckdelidx]);
	dump_junk(buf, "        ", 7);
}

static void dump_hub(char *p)
{
	unsigned int l, i, j;
	
	printf("       Hub Descriptor:\n");
	printf("         bLength             %3u\n",p[0]);
	printf("         bDesriptorType      %3u\n",p[1]);
	printf("         nNbrPorts           %3u\n",p[2]); 
	printf("         wHubCharacteristic 0x%02x 0x%02x\n", p[3],p[4]);
	printf("         bPwrOn2PwrGood      %3u * 2 milli seconds\n",p[5]);
	printf("         bHubContrCurrent    %3u milli Ampere\n",p[6]);
	l= (p[2] >> 3) + 1; /* this determines the variable number of bytes following */
	printf("         DeviceRemovable   ");
	for(i = 0; i < l; i++) 
		printf(" 0x%02x", p[7+i]);
	printf("\n         PortPwrCtrlMask   ");
	for(j = 0; j < l; j++)
		printf(" 0x%02x ", p[7+i+j]);
	printf("\n");
}

/* ---------------------------------------------------------------------- */

/*
 * HID descriptor
 */

static void dump_report_desc(unsigned char *b, int l)
{
        unsigned int t, j, bsize, btag, btype, data, hut;
	int i;
	char *types[4] = { "Main", "Global", "Local", "reserved" };
	char indent[] = "                            ";

	printf("	  Report Descriptor: (length is %d)\n", l);
	for(i = 0; i < l; ) {
		t = b[i];
		bsize= b[i] & 0x03;
		if (bsize == 3)
                        bsize=4;
		btype = b[i] & (0x03 << 2);
		btag = b[i] & ~0x03; /* 2 LSB bits encode length */
		printf("            Item(%-6s): %s, data=", types[btype>>2], names_reporttag(btag));
		if(bsize > 0) {
			printf(" [ ");
			data = 0;
			for(j = 0; j < bsize; j++) {
				printf("0x%02x ", b[i+1+j]);
				data += (b[i+1+j] << (8*j));
			}
			printf("] %d", data);
		} else
                        printf("none");
		printf("\n");
		switch(btag) {
		case 0x04 : /* Usage Page */
                        printf("%s%s\n", indent, names_huts(data));
			hut = data;
                        break;
                        
		case 0x08 : /* Usage */
		case 0x18 : /* Usage Minimum */
		case 0x28 : /* Usage Maximum */
			printf("%s%s\n", indent, names_hutus((hut << 16) + data));
			break;
                        
		case 0xa0 : /* Collection */
			printf("%s", indent);
			switch (data) {
			case 0x00:
                                printf("Physical\n");
                                break;
                                
			case 0x01:
                                printf("Application\n");
                                break;
                                
			case 0x02:
                                printf("Logical\n");
                                break;
                                
			default:
				if(data & 0x80)
				    printf("Vendor definened\n");
				else
				    printf("Reserved for future use.\n");
			}
			break;
		case 0x80: /* Input */
		case 0x90: /* Output */
		case 0xb0: /* Feature */
			printf("%s%s %s %s %s %s\n%s%s %s %s %s\n",
			       indent,
			       data & 0x01 ? "Constant": "Data",
			       data & 0x02 ? "Variable": "Array",
			       data & 0x04 ? "Relative": "Absolute",
			       data & 0x08 ? "Wrap" : "No_Wrap",
			       data & 0x10 ? "Non_Linear": "Linear",
			       indent,
			       data & 0x20 ? "No_Preferred_State": "Preferred_State",
			       data & 0x40 ? "Null_State": "No_Null_Position",
			       data & 0x80 ? "Volatile": "Non_Volatile",
			       data &0x100 ? "Buffered Bytes": "Bitfield"
			);
		}
		i += 1 + bsize;
	}
}

static void dump_hid_device(int fd, unsigned char *buf,int interface_number)
{
	unsigned int i, len;
	int n;
	unsigned char dbuf[8192];

	if (buf[1] != USB_DT_CS_DEVICE)
		printf("      Warning: Invalid descriptor\n");
	else if (buf[0] < 6+3*buf[5])
		printf("      Warning: Descriptor too short\n");
	printf("        HID Device Descriptor:\n"
	       "          bLength             %5u\n"
	       "          bDescriptorType     %5u\n"
	       "          bcdHID              %2x.%02x\n"
	       "          bCountryCode        %5u\n"
	       "          bNumDescriptors     %5u\n",
	       buf[0], buf[1], buf[3], buf[2], buf[4], buf[5]);
	for (i = 0; i < buf[5]; i++)
		printf("          bDescriptorType     %5u %s\n"
		       "          wDescriptorLength   %5u\n", 
		       buf[6+3*i], names_hid(buf[6+3*i]), buf[7+3*i] | (buf[8+3*i] << 8));
	dump_junk(buf, "        ", 6+3*buf[5]);
	if (!do_report_desc)
		return;
	for (i = 0; i < buf[5]; i++) {
		if (buf[6+3*i] != 0x22) /* we are just interested in report descriptors*/
			continue;
		len = buf[7+3*i] | (buf[8+3*i] << 8);
		if (len > sizeof(dbuf)) {
			printf("report descriptor too long\n");
			continue;
		}
		if ((n = usb_control_msg(fd, 0x81 , 0x06, 0x22<<8, interface_number, len, dbuf)) < 0) {
			printf("cannot get report descriptor\n");
			continue;
		}
		dump_report_desc(dbuf, n);
	}
}

/* ---------------------------------------------------------------------- */

static void do_config(int fd, unsigned int nr, u_int16_t lang)
{
	unsigned char buf[1024],*p;
	unsigned int sz,curinterface;
	int l;
	u_int8_t curclass = 0xff, cursubclass = 0xff;

	if (usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR, (USB_DT_CONFIG << 8) | nr,
			    0, USB_DT_CONFIG_SIZE, buf) < 0) {
		fprintf(stdout, "cannot get config descriptor %d, %s (%d)\n", nr, strerror(errno), errno);
		return;
	}
	if (buf[0] < USB_DT_CONFIG_SIZE || buf[1] != USB_DT_CONFIG)
		fprintf(stderr, "Warning: invalid config descriptor\n");
	sz = buf[2] | buf[3] << 8;
	if (sz > sizeof(buf)) {
		fprintf(stderr, "Config %d descriptor too long, truncating\n", nr);
		sz = sizeof(buf);
	}
	if (usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR, (USB_DT_CONFIG << 8) | nr,
			    0, sz, buf) < 0) {
		fprintf(stdout, "cannot get config descriptor %d, %s (%d)\n", nr, strerror(errno), errno);
		return;
	}
	p = buf;
	while (sz >= 2) {
		if (p[0] < 2)
			break;
		if (p[0] > sz) {
			printf("  descriptor length past end:");
			dump_junk2(p, sz);
			sz = 0;
			break;
		}
		switch (p[1]) {
		case USB_DT_DEVICE:
			dump_device(fd, p, lang);
			curclass = p[4];
			cursubclass = p[5];
			break;
			
		case USB_DT_CONFIG:
			dump_config(fd, p);
			break;
			
		case USB_DT_INTERFACE:
			dump_interface(fd, p, lang);
			curclass = p[5];
			cursubclass = p[6];
			curinterface = p[2];
			break;
			
		case USB_DT_ENDPOINT:
			dump_endpoint(fd, p);
			break;

		case USB_DT_CS_DEVICE:
			if (curclass == 3) {
				dump_hid_device(fd, p, curinterface);
				break;
			}
			printf("  unknown descriptor type:");
			dump_junk2(p, p[0]);
			break;

		case USB_DT_CS_CONFIG:
			printf("  unknown descriptor type:");
			dump_junk2(p, p[0]);
			break;

		case USB_DT_CS_STRING:
			printf("  unknown descriptor type:");
			dump_junk2(p, p[0]);
			break;

		case USB_DT_CS_INTERFACE:
			if (curclass == 1 && cursubclass == 1) {
				dump_audiocontrol_interface(fd, p, lang);
				break;
			}
			if (curclass == 1 && cursubclass == 2) {
				dump_audiostreaming_interface(fd, p);
				break;
			}
			printf("  unknown descriptor type:");
			dump_junk2(p, p[0]);
			break;

		case USB_DT_CS_ENDPOINT:
			if (curclass == 1 && cursubclass == 2) {
				dump_audiostreaming_endpoint(fd, p);
				break;
			}
			printf("  unknown descriptor type:");
			dump_junk2(p, p[0]);
			break;

		case USB_DT_HUB:
			dump_hub(p);
			break;

		default:
			printf("  unknown descriptor type:");
			dump_junk2(p, p[0]);
		}
		sz -= p[0];
		p += p[0];
	}
	if (sz > 0) {
		printf("  junk at config descriptor end:");
		dump_junk2(p, sz);
	}
}

/* ---------------------------------------------------------------------- */

/* returns first lang ID */
static u_int16_t dump_langids(int fd, int quiet)
{
	unsigned char b[256];
	int i, l;
        u_int16_t lang;

	b[0] = b[1] = 0xbf;
	l = usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR, USB_DT_STRING << 8, 0, sizeof(b), b);

	if (l < 0) {
		printf("  Language IDs: none (cannot get min. string descriptor; got len=%d, error=%d:%s)\n",
			l, errno, strerror(errno));
		return 0;
	}
	if (l < 4 || b[0] != l) {
		printf("  Language IDs: none (invalid length string descriptor %02x; len=%d)\n", b[0], l);
		return 0;
	}
        /* save first language ID for further get_string_descriptors */
	lang =  b[2] | (b[3] << 8);
#if 0
	printf ("dump_langids: ret=%d:%d, lang=0x%x, length=%d\n", l, errno, lang, b[0]);
	dump_junk2 (b, 32);
#endif
	if(quiet)
		return lang;
	printf("  Language IDs: (length=%d)\n", b[0]);
	for (i = 0; i < ((b[0] - 2) / 2); i++) {
		l = b[2+2*i] | (b[3+2*i] << 8);
		printf("     %04x %s(%s)\n", l, names_langid(l & 0x3ff), names_langid(l));
	}
        return lang;
}

/* ---------------------------------------------------------------------- */

static void dumpdev(unsigned char *devdesc, int fd, unsigned int flags)
{
	unsigned int i, maxcfg;
	u_int16_t lang;

	maxcfg = devdesc[17];
	if (devdesc[0] < 18 || devdesc[1] != USB_DT_DEVICE)
		maxcfg = 1;
	lang = dump_langids(fd, 1);
	dump_device(fd, devdesc, lang);
	for (i = 0; i < maxcfg; i++)
		do_config(fd, i, lang);
	lang = dump_langids(fd, 0);
}

/* ---------------------------------------------------------------------- */

static void dump_one_device(const char *path, unsigned int flags)
{
        unsigned char buf[USB_DT_DEVICE_SIZE];
	unsigned int vid, pid;
	char vendor[128], product[128];
        int fd;

	if ((fd = open(path, O_RDWR)) == -1) {
		fprintf(stderr, "cannot open %s, %s (%d)\n", path, strerror(errno), errno);
		return;
	}
	if (usb_control_msg(fd, USB_DIR_IN, USB_REQ_GET_DESCRIPTOR, (USB_DT_DEVICE << 8), 0, USB_DT_DEVICE_SIZE, buf) < 0) {
		fprintf(stderr, "cannot get config descriptor, %s (%d)\n", strerror(errno), errno);
		goto err;
	}
	vid = buf[8] | (buf[9] << 8);
	pid = buf[10] | (buf[11] << 8);
	get_vendor_string(vendor, sizeof(vendor), vid);
	get_product_string(product, sizeof(product), vid, pid);
	printf("Device: ID %04x:%04x %s %s\n", vid, pid, vendor, product);
	dumpdev(buf, fd, flags);
 err:
	close(fd);
}

/* ---------------------------------------------------------------------- */

static void list_devices(int bus, int devnum, int vendorid, int productid, unsigned int flags)
{
        DIR *d, *d2;
        struct dirent *de, *de2;
        unsigned char buf[256];
	unsigned int vid, pid;
	char vendor[128], product[128];
        int fd;

        d = opendir(procbususb);
        if (!d) {
                fprintf(stderr, "cannot open %s, %s (%d)\n", procbususb, strerror(errno), errno);
                return;
        }
        while ((de = readdir(d))) {
                if (de->d_name[0] < '0' || de->d_name[0] > '9')
                        continue;
		if (bus != -1 && strtoul(de->d_name, NULL, 0) != bus)
			continue;
                snprintf(buf, sizeof(buf), "%s/%s/", procbususb, de->d_name);
                if (!(d2 = opendir(buf)))
                        continue;
                while ((de2 = readdir(d2))) {
                        if (de2->d_name[0] == '.')
                                continue;
			if (devnum != -1 && strtoul(de2->d_name, NULL, 0) != devnum)
				continue;
                        snprintf(buf, sizeof(buf), "%s/%s/%s", procbususb, de->d_name, de2->d_name);
			if ((fd = open(buf, O_RDWR)) == -1) {
				fprintf(stderr, "cannot open %s, %s (%d)\n", buf, strerror(errno), errno);
				continue;
			}
			if (read(fd, buf, USB_DT_DEVICE_SIZE) != USB_DT_DEVICE_SIZE) {
				fprintf(stderr, "cannot read device descriptor %s (%d)\n", strerror(errno), errno);
				goto err;
			}
			vid = buf[8] | (buf[9] << 8);
			pid = buf[10] | (buf[11] << 8);
			if (buf[0] >= USB_DT_DEVICE_SIZE && ((vendorid != -1 && vendorid != vid) || (productid != -1 && productid != pid)))
				goto err;
			get_vendor_string(vendor, sizeof(vendor), vid);
			get_product_string(product, sizeof(product), vid, pid);
			printf("\nBus %s Device %s: ID %04x:%04x %s %s\n", de->d_name, de2->d_name, vid, pid, vendor, product);
			dumpdev(buf, fd, flags);
		err:
			close(fd);
                }
                closedir(d2);
        }
        closedir(d);
}

/* ---------------------------------------------------------------------- */

void devtree_busconnect(struct usbbusnode *bus)
{
}

void devtree_busdisconnect(struct usbbusnode *bus)
{
}

void devtree_devconnect(struct usbdevnode *dev)
{
}

void devtree_devdisconnect(struct usbdevnode *dev)
{
}

static void treedump(void)
{ 
	int fd;
	char buf[512];

	snprintf(buf, sizeof(buf), "%s/devices", procbususb);
	if ((fd = open(buf, O_RDONLY)) == -1) {
                fprintf(stderr, "cannot open %s, %s (%d)\n", buf, strerror(errno), errno);
                return;
	}
	devtree_parsedevfile(fd);
	close(fd);
        devtree_dump();
}

/* ---------------------------------------------------------------------- */

int main(int argc, char *argv[])
{
        static const struct option long_options[] = {
                { "version", 0, 0, 'V' },
                { 0, 0, 0, 0 }
        };
	int c, err = 0;
	unsigned int allowctrlmsg = 0, treemode = 0;
	int bus = -1, devnum = -1, vendor = -1, product = -1;
	const char *devdump = NULL;
	char *cp;

	while ((c = getopt_long(argc, argv, "D:vxtp:s:d:V", long_options, NULL)) != EOF) {
		switch(c) {
                case 'V':
                        printf("lsusb (" PACKAGE ")  " VERSION "\n");
                        exit(0);
                        
		case 'v':
			verblevel++;
			break;

		case 'x':
			allowctrlmsg = 1;
			break;

		case 't':
			treemode = 1;
			break;

		case 'p':
			procbususb = optarg;
			break;

		case 's':
			cp = strchr(optarg, ':');
			if (cp) {
				*cp++ = 0;
				if (*optarg)
					bus = strtoul(optarg, NULL, 0);
				if (*cp)
					devnum = strtoul(cp, NULL, 0);
			} else {
				if (*optarg)
					devnum = strtoul(optarg, NULL, 0);
			}
			break;

		case 'd':
			cp = strchr(optarg, ':');
			if (!cp) {
				err++;
				break;
			}
			*cp++ = 0;
			if (*optarg)
				vendor = strtoul(optarg, NULL, 0);
			if (*cp)
				product = strtoul(cp, NULL, 0);
			break;

		case 'D':
			devdump = optarg;
			break;

		default:
			err++;
			break;
		}
	}
	if (err) {
		fprintf(stderr, "usage: lsusb [-v] [-x] [-p <procpath>] [-s [<bus>:][<devnum>]] [-d [<vendor>]:[<device>]]\n");
		exit(1);
	}
	if ((err = names_init("./usb.ids")) != 0)
		if(err = names_init(USBIDS_FILE)) {
			printf("Error, cannot open USBIDS File \"%s\", %s\n", USBIDS_FILE, strerror(err));
			exit(1);
	}
	if (treemode)
		treedump();
	else if (devdump)
		dump_one_device(devdump, 0);
	else
		list_devices(bus, devnum, vendor, product, 0);
	return 0;
}
