# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Library stuff for Unperish"""


import collections
import imp
import inspect
import logging
import optparse
import os
import subprocess
import sys


class FakeOptions:

    """Fake an optparse Options object for unit tests."""

    def __init__(self, **kwargs):
        self.directory = "."

        for name, value in kwargs.iteritems():
            setattr(self, name, value)

    def __getattr__(self, name):
        return None


class Stack:

    """Simple stack."""
    
    def __init__(self, iterable=None):
        self._deque = collections.deque(iterable or [])

    def push(self, value):
        """Put value on top of the stack."""
        self._deque.append(value)
        
    def pop(self):
        """Remove topmost value from stack and return it."""
        return self._deque.pop()
        
    def top(self):
        """Return topmost value from stack without removing it."""
        return self._deque[-1]
        
    def is_empty(self):
        """Is the stack empty?"""
        return len(self._deque) == 0


class Operation:

    """Base class for all operations.
    
    Plugins should define one or more classes that inherit this class.
    Every sub-class MUST have an initializer without arguments, and
    creating an instance of the sub-class MUST not have any side
    effects. All side effects MUST happen only in the do_it method.
    
    """
    
    def __init__(self):
        pass
        
    def get_name(self):
        """Return name of this operation.
        
        Sub-classes may override this method, or just define the "name"
        attribute to be the name of the operation. Note that an operation
        is NOT guaranteed to have a name attribute. The defined interface
        for getting the name of an operation is this method. The default
        implementation merely makes it easy for an operation to define
        the name.
        
        """
        if hasattr(self, "name"):
            return self.name
        else:
            return None

    def add_options(self, option_parser):
        """Add operation specific options to OptionParser instance."""
        
    def get_required_options(self):
        """Return list of option names that must be set for this operation.
        
        The name should be the attribute name in the optparse.Options
        object, not the name used on the command line. Of course, many
        times they're the same.
        
        Sub-classes may override this method, or just define the
        "required_options" attribute to be the name of the operation.
        Note that an operation is NOT guaranteed to have that attribute.
        The defined interface for getting the dependencies is this
        method. The default implementation merely makes it easy for a
        sub-class to define them.
        
        """
        if hasattr(self, "required_options"):
            return self.required_options
        else:
            return []

    def get_provided_options(self):
        """Return list of option names that this operation MAY set.
        
        This is a similar list to the one returned by get_required_options,
        but names options that this operation maybe sets. For example,
        an operation to deduce the project version number from a NEWS
        file would add "project_version" to the list returned by this
        method, but only actually set the option if it actually finds the
        version number in the NEWS file (there might not even be one).
        
        This is used to determine dependencies between bits of information
        and operations without having to explicitly list the operations
        that provide the bits of information. This allows several operations
        to try to do the same thing. The first one that succeeds wins.
        
        Sub-classes may override this method, or just define the
        "provided_options" attribute to be the name of the operation.
        Note that an operation is NOT guaranteed to have that attribute.
        The defined interface for getting the provided options is this
        method. The default implementation merely makes it easy for a
        sub-class to define them.
        
        """
        if hasattr(self, "provided_options"):
            return self.provided_options
        else:
            return []

    def set_application(self, app):
        """Set the Application instance to be used.
        
        For a few special plugins it is necessary to access the Application
        that is being run. For example, a plugin that wants to list all
        other plugins needs this. Any plugin that can, should do without
        this.
        
        This method gets called by the Application after all plugins
        have been loaded and before the do_it methods start getting called.
        
        """

    def do_it(self, options):
        """Actually do the operation.
        
        options is an optparse.parse_args return value giving the current
        settings. The method MAY change it.
        
        """


class PluginManager:

    """Manage plugins and operations therein.
    
    A plugin is a Python file (*.py) located in one of the plugin
    directories set with the set_directories method. It must contain
    zero or more classes that inherit Operation from this file, and
    it must not contain import-time statement that have side effects,
    or things will go very badly during unit testing.
    
    """
    
    def __init__(self):
        self._plugin_dirs = []
        self._operations = []
        self._plugins = []
        self._loaded = False

    def has_loaded(self):
        """Have we loaded plugins already?"""
        return self._loaded
    
    def get_plugin_directories(self):
        """Return list of directories in which to search for plugins."""
        return self._plugin_dirs
    
    def set_plugin_directories(self, plugin_dirs):
        """Return list of directories in which to search for plugins."""
        self._plugin_dirs = plugin_dirs

    def find_plugin_files(self):
        """Return list of paths to plugin files."""
        files = []
        for plugin_dir in self.get_plugin_directories():
            if os.path.isdir(plugin_dir):
                files += [os.path.join(plugin_dir, x) 
                          for x in os.listdir(plugin_dir) 
                          if x.endswith(".py")]
        return files

    def get_operations(self):
        """Return list of all operations known so far."""
        return self._operations

    def get_plugins(self):
        """Return list of all currently loaded plugins."""
        return self._plugins

    def find_operations(self, module):
        """Find all classes in a module which inherit Operation."""
        operations = []
        for _, member in inspect.getmembers(module):
            if inspect.isclass(member) and issubclass(member, Operation):
                operations.append(member)
        return [oper() for oper in operations]

    def load_plugins(self):
        """Load all plugins found by find_plugin_files."""
        logging.debug("Loading plugins")
        for filename in self.find_plugin_files():
            logging.debug("Loading plugins from %s" % filename)
            if filename.endswith(".py"):
                module_name, _ = os.path.splitext(os.path.basename(filename))
                f = file(filename, "r")
                module = imp.load_module(module_name, f, filename,
                                         (".py", "r", imp.PY_SOURCE))
                f.close()
                self._plugins.append(module)
                self._operations += self.find_operations(module)
        self._loaded = True
        logging.debug("Finished loading plugins")

    def unload_plugins(self):
        """Unload all currently loaded plugins."""
        # We assume the Python modules are well written, so importing them
        # multiple times is OK. Thus, unloading means we just forget about
        # them.
        self._plugins = []
        self._operations = []
        self._loaded = False

    def get_operation(self, name):
        """Return operation with the desired name, or None."""
        for oper in self.get_operations():
            if oper.get_name() == name:
                return oper
        return None


class InternalOptionsGroup(optparse.OptionGroup):

    """OptionGroup to hide/unhide internal options from --help output.

    Most of the options defined by plugins are for communication between
    plugins, and are useless for most users. We need a way of hiding them
    from --help output. The way we do that is by adding the internal options
    to this option group, and not printing them when help output is printed.
    Unless, of course, the --help-all option is uses.

    """

    def __init__(self, parser):
        optparse.OptionGroup.__init__(self, parser, "Internal options")
        self.show = False

    def format_help(self, *args):
        if self.show:
            return optparse.OptionGroup.format_help(self, *args)
        else:
            return ""
        

class CommandLineParser:

    """Command line parser for Unperish."""

    def __init__(self):
        self._parser, self._internal_options = self._create_parser()
        
    def _create_parser(self):
        """Create an optparse.OptionParser instance with default options."""

        p = optparse.OptionParser()
        
        p.add_option("--help-all", action="callback", callback=self.help_all,
                     help="Like --help, but include internal options.")

        p.add_option("--directory", default=".", metavar="DIR",
                     help="Use DIR as project root, instead of current "
                          "directory.")
        
        p.add_option("-v", "--verbose", action="store_true",
                     help="Print out name of each operation before it's "
                          "excecuted.")
        
        g = InternalOptionsGroup(p)
        p.add_option_group(g)
        
        return p, g

    def has_option(self, *args, **kwargs):
        return self._parser.has_option(*args, **kwargs)

    def add_public_option(self, *args, **kwargs):
        self._parser.add_option(*args, **kwargs)
    
    def add_option(self, *args, **kwargs):
        self._internal_options.add_option(*args, **kwargs)
    
    def set_defaults(self, *args, **kwargs):
        self._parser.set_defaults(*args, **kwargs)

    def show_internal_options(self, *args, **kwargs):
        """Make internal options visible in help output."""
        self._internal_options.show = True

    def format_help(self):
        """Return the formatted help text."""
        return self._parser.format_help()
    
    def help_all(self, *args, **kwargs):
        self.show_internal_options()
        self._parser.print_help(file=kwargs.get("file", None))
    
    def parse_args(self, args):
        """Return optparse options object and list of operation names."""
        return self._parser.parse_args(args)


class UnperishException(Exception):

    """Base class for all exceptions specific to Unperish.
    
    Doing a str() on these exceptions gives a useful error message.
    
    """
    
    def __str__(self):
        return self._str


class UnknownOperation(UnperishException):

    def __init__(self, oper_name):
        self._str = "Unknown operation %s" % oper_name


class MissingOption(UnperishException):

    def __init__(self, name):
        self._str = ("Option %s has not been set and has no default value" %
                     name)


class Application:

    """The Unperish command line application."""
    
    default_plugin_dirs = ( #pragma: no cover
        "/usr/share/unperish/plugins",
        "/usr/local/share/unperish/plugins",
        "~/.unperish/plugins",
    )
    
    def __init__(self, plugins=None):
        """Initialize the object.
        
        If plugins is set, it is the list of plugin directories that the
        plugin manager should look at.
        
        """
        self._pm = PluginManager()
        if plugins is None:
            plugins = [os.path.expanduser(dir) 
                       for dir in self.default_plugin_dirs]
        self._pm.set_plugin_directories(plugins)
        self._cli = CommandLineParser()
        self._verbose_file = sys.stdout
        
    def get_plugin_manager(self):
        """Return the PluginManager this application uses."""
        return self._pm
        
    def get_command_line_parser(self):
        """Return the CommandLineParser this application uses."""
        return self._cli

    def prepare_plugins(self):
        """Make sure all plugins are loaded and set up."""
        if not self._pm.has_loaded():
            self._pm.load_plugins()
            for oper in self._pm.get_operations():
                oper.add_options(self._cli)

    def parse_args(self, args):
        """Parse the command line given in args."""
        self.prepare_plugins()
        return self._cli.parse_args(args)

    def find_missing_providers(self, oper, options):
        """Find all operations that provide the first missing option."""
        for option in oper.get_required_options():
            if getattr(options, option) is None:
                logging.debug("Finding providers for %s" % option)
                providers = [x for x in self._pm.get_operations()
                             if option in x.get_provided_options()]
                return option, providers
        return None, []

    def map_names_to_operations(self, names):
        """Return list of operations matching names."""
        list = []
        for name in names:
            oper = self._pm.get_operation(name)
            if oper is None:
                raise UnknownOperation(name)
            oper.set_application(self)
            list.append(oper)
        return list

    def attempt_operation(self, oper, options, done):
        """Try to perform an operation, it possible.
        
        If the operation needs options that have not yet been provided,
        we find potential providers and perform those first. If it turns
        out we can't perform the operation, or any of its option
        providers, then we bail out, and return the name of the option
        that can't be provided. If we managed to perform the operation,
        we return None.
        
        It is the caller who needs to decide whether failure to perform
        is a problem or not.
        
        """

        if oper in done: #pragma: no cover
            logging.debug("Not attempting operation %s, already done" %
                          oper.get_name())
            return True

        logging.debug("Attempting operation %s" % oper.get_name())

        # Provide missing options.
        option, providers = self.find_missing_providers(oper, options)
        while option:
            logging.debug("Need missing option: %s" % option)
            for provider in providers:
                self.attempt_operation(provider, options, done)
            if getattr(options, option) is None:
                logging.debug("Nobody provided %s, oh well." % option)
                return option
            logging.debug("Missing option %s was provided" % option)
            option, providers = self.find_missing_providers(oper, options)

        # Now we have all missing options and can perform the operation.
        self.run_operation(options, oper)
        done.add(oper)

    def run(self, args):
        """Run the application with the given set of command line args."""
        logging.debug("Application.run: %s" % args)
        self.prepare_plugins()
        options, oper_names = self.parse_args(args)
        logging.debug("Operations: %s" % oper_names)
        
        opers = self.map_names_to_operations(oper_names)
        done = set()
        for oper in opers:
            if oper in done:
                logging.debug("Skipping %s, since it's done already." %
                              oper.get_name())
            else:
                missing = self.attempt_operation(oper, options, done)
                if missing:
                    raise MissingOption(missing)

    def set_verbose_file(self, file):
        """Set file to which --verbose output should go."""
        self._verbose_file = file
        
    def get_verbose_file(self):
        """Return file to which --verbose output should go."""
        return self._verbose_file

    def run_operation(self, options, oper):
        """Run one operation.
        
        This is a separate method so that unit tests may easily override
        it.
        
        """
        
        logging.debug("Executing: %s" % oper.get_name())
        if options.verbose:
            self.get_verbose_file().write("Executing %s\n" % oper.get_name())
        oper.do_it(options)


class CommandFailed(UnperishException):

    """Running an external command failed."""

    def indent(self, str, amount):
        return "\n".join((" " * amount) + line for line in str.splitlines())

    def __init__(self, argv, returncode, stdout, stderr):
        self._str = (("Command failed: %s\n"
                      "Return code was %d\n"
                      "Standard output:\n%s\n"
                      "Error output:\n%s") %
                     (argv, returncode, 
                      self.indent(stdout, 2),
                      self.indent(stderr, 2)))


def run(argv, cwd=None):
    """Run an external command.
    
    If cwd is not None, change the directory to it in the sub-process
    that executes the command (but not in the parent process which
    spawns the sub-process).
    
    """
    logging.debug("Executing (in %s): %s" % (cwd, repr(argv)))
    p = subprocess.Popen(argv, cwd=cwd, stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE)
    stdout, stderr = p.communicate()
    if p.returncode != 0:
        raise CommandFailed(argv, p.returncode, stdout, stderr)
    return stdout
