/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.  
                  Let a thousand taggers bloom!
   
   Copyright (C) Robert Kaye 2003
   
   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: tagger.c,v 1.24 2004/03/25 09:20:15 robert Exp $

----------------------------------------------------------------------------*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <tunepimp/tp_c.h>
#include "../config.h"

#ifdef HAVE_LIBREADLINE
#  include <readline/readline.h>
#  include <readline/history.h>
#endif

const char *fileStatusText[] =
{
    "Unrecognized",
    "Recognized",
    "Pending",
    "TRM Lookup",
    "TRM Collision",
    "File Lookup",
    "User Selection",
    "Verified",
    "Saved",
    "Deleted",
    "Error"
};
static tunepimp_t pimp = NULL;

#ifdef HAVE_LIBREADLINE
                                      
/*
 * Read a line from standard input using libreadline and put it into
 * the history buffer.
 */
char *getCommand(void)                
{                                     
	static char *line = NULL;

        if ( line != NULL )           
                free(line);           
                                      
        line = readline("tag> ");     

	/* line can be NULL if the user typed Ctrl-D. */
	if ( line == NULL )
		line = strdup("q");	/* quit */
	else
		if ( line[0] != '\0' )
			add_history(line);
                                      
        return line;                  
}                                     
                                      
#else                                 

static char cmdLine[256];
char *getCommand(void)
{
    printf("tag> ");
    fflush(stdout);
    if (fgets(cmdLine, 256, stdin))
        cmdLine[strlen(cmdLine) - 1] = 0;

    return cmdLine;
}

#endif /* HAVE_LIBREADLINE */

void printList(tunepimp_t pimp)
{
    char      fileName[255];
    track_t  *track;
    int       num, *ids, i;

    num = tp_GetNumFileIds(pimp);
    if (num == 0)
        printf("No files in the tagger.\n");
    else
    {
        ids = malloc(sizeof(int) * num);
        tp_GetFileIds(pimp, ids, num);

        for(i = 0; i != num; i++)
        {
            track = tp_GetTrack(pimp, ids[i]);
            if (!track)
            {
                printf("Can't get track %d info\n", ids[i]);
                continue;
            }

            tr_Lock(track);
            tr_GetFileName(track, fileName, 255);
            printf("%d: %s, %3d%%: %s\n", i, 
                                   fileStatusText[tr_GetStatus(track)],
                                   tr_GetSimilarity(track),
                                   fileName);
            if (tr_GetStatus(track) == eError)
            {
                char err[255];

                tr_GetError(track, err, 255);
                printf("  Error: %s\n", err);
            }
            tr_Unlock(track);

            tp_ReleaseTrack(pimp, track);
        }
        free(ids);
    }

    printf("\n%d unsubmitted TRMs.\n\n", tp_GetNumUnsubmitted(pimp));
}

void printTrack(tunepimp_t pimp, int id)
{
    track_t    track;
    char       fileName[255], trm[64];
    metadata_t *data;

    data = md_New();

    track = tp_GetTrack(pimp, id);
    if (!track)
    {
        printf("Can't get track %d\n\n", id);
        return;
    }

    tr_Lock(track);
    tr_GetFileName(track, fileName, 255);
    printf("%d: %s: %s\n", id, 
                           fileStatusText[tr_GetStatus(track)],
                           fileName);

    tr_GetTRM(track, trm, 64);
    printf("       TRM: %s\n", trm);
    printf("Similarity: %d\n", tr_GetSimilarity(track));
    printf("   Changed: %d\n\n", tr_HasChanged(track));

    tr_GetLocalMetadata(track, data);
    printf("File metadata:\n");
    printf("     Format: %s\n", data->fileFormat);
    printf("     Artist: %s\n", data->artist);
    printf("   SortName: %s\n", data->sortName);
    printf("      Album: %s\n", data->album);
    printf("      Track: %s\n", data->track);
    printf("   TrackNum: %d\n", data->trackNum);
    printf("   Duration: %ld\n", data->duration);
    printf("   ArtistId: %s\n", data->artistId);
    printf("    AlbumId: %s\n", data->albumId);
    printf("    TrackId: %s\n", data->trackId);
    printf("         VA: %d\n", data->variousArtist);
    if (data->albumType != eAlbumType_Error)
    {
        char type[255];

        md_ConvertFromAlbumType(data->albumType, type, 255);
        printf("  AlbumType: %s\n", type);
    }
    if (data->albumStatus != eAlbumStatus_Error)
    {
        char status[255];

        md_ConvertFromAlbumStatus(data->albumStatus, status, 255);
        printf("AlbumStatus: %s\n", status);
    }
    if (data->releaseYear != 0)
    {
        printf("       Date: %d-%02d-%02d (%s)\n", data->releaseYear,
               data->releaseMonth, data->releaseDay, data->releaseCountry);
    }

    printf("\n");

    tr_GetServerMetadata(track, data);
    printf("Server metadata:\n");
    printf("     Artist: %s\n", data->artist);
    printf("   SortName: %s\n", data->sortName);
    printf("      Album: %s\n", data->album);
    printf("      Track: %s\n", data->track);
    printf("   TrackNum: %d\n", data->trackNum);
    printf("   Duration: %ld\n", data->duration);
    printf("   ArtistId: %s\n", data->artistId);
    printf("    AlbumId: %s\n", data->albumId);
    printf("    TrackId: %s\n", data->trackId);
    printf("         VA: %d\n", data->variousArtist);
    if (data->albumType != eAlbumType_Error)
    {
        char type[255];

        md_ConvertFromAlbumType(data->albumType, type, 255);
        printf("  AlbumType: %s\n", type);
    }
    if (data->albumStatus != eAlbumStatus_Error)
    {
        char status[255];

        md_ConvertFromAlbumStatus(data->albumStatus, status, 255);
        printf("AlbumStatus: %s\n", status);
    }
    if (data->releaseYear != 0)
    {
        printf("       Date: %d-%02d-%02d (%s)\n", data->releaseYear,
               data->releaseMonth, data->releaseDay, data->releaseCountry);
    }

    printf("\n");
    tr_Unlock(track);

    tp_ReleaseTrack(pimp, track);

    md_Delete(data);
}

void printArtistResult(track_t track, artistresult_t **res, int resCount)
{
    char   fileName[255];
    int    i;

    tr_Lock(track);
    tr_GetFileName(track, fileName, 255);
    tr_Unlock(track);

    printf("Artists for %s:\n--------------------------------------------\n", fileName);
    for(i = 0; i != resCount; i++, (*res)++)
    {
         printf("%2d. %d%% %-32s %-32s\n", i, (*res)->relevance, (*res)->name, (*res)->sortName);
    }
    printf("\n");
}

void printAlbumResult(track_t track, albumresult_t **res, int resCount)
{
    char   fileName[255];
    int    i;

    tr_Lock(track);
    tr_GetFileName(track, fileName, 255);
    tr_Unlock(track);

    for(i = 0; i != resCount; i++, (*res)++)
    {
        if (i == 0)
        {
            printf("Artist: %s (%s)\n", (*res)->artist->name, (*res)->artist->sortName);
            printf("Albums for: %s\n---------------------------------------------\n", 
                   fileName);
        }

        printf("%2d. %d%% %-40s (%d trks %d cd)\n", i,
            (*res)->relevance, (*res)->name, (*res)->numTracks, (*res)->numCDIndexIds);
    } 
    printf("\n");
}

void printTrackResult(track_t track, albumtrackresult_t **res, int resCount)
{
    char   fileName[255];
    int    i;

    tr_Lock(track);
    tr_GetFileName(track, fileName, 255);
    tr_Unlock(track);

    for(i = 0; i != resCount; i++, res++)
    {
        if (i == 0)
        {
            printf("Artist: %s (%s)\n", (*res)->artist->name, (*res)->artist->sortName);
            printf("Album/Tracks for: %s\n---------------------------------------------\n", 
                   fileName);
        }
        printf("%2d %d%% %-32s %-32s (", i, (*res)->relevance, (*res)->name, (*res)->album->name);
        if ((*res)->album->numTracks >= 0)
            printf("%d trks ", (*res)->album->numTracks);
        if ((*res)->album->numCDIndexIds >= 0)
            printf("%d cd ", (*res)->album->numCDIndexIds);

        printf("%d trm)\n", (*res)->numTRMIds);
    }
    printf("\n");
}

void printResults(tunepimp_t pimp, int id)
{
    track_t      track;
    TPResultType type;
    char         fileName[255];
    int          num;
    result_t    *results;

    track = tp_GetTrack(pimp, id);
    if (!track)
    {
        printf("Can't get track %d\n\n", id);
        return;
    }

    tr_Lock(track);
    tr_GetFileName(track, fileName, 255);

    num = tr_GetNumResults(track);
    if (num == 0)
    {
        printf("No results available for this track.\n\n");
    }
    else
    {
        results = (result_t  *)malloc(sizeof(result_t *) * num);
        tr_GetResults(track, &type, results, &num);
        switch(type)
        {
            case eArtistList:
                printArtistResult(track, (artistresult_t **)results, num);
                break;
            case eAlbumList:
                printAlbumResult(track, (albumresult_t **)results, num);
                break;
            case eTrackList:
                printTrackResult(track, (albumtrackresult_t **)results, num);
                break;
            default:
                printf("No clue what this result is.\n");
        }
        rs_Delete(type, results, num);
        free(results);
    }

    tr_Unlock(track);
    tp_ReleaseTrack(pimp, track);
}

void toggleVA(tunepimp_t pimp, int id)
{
    track_t track;

    track = tp_GetTrack(pimp, id);
    if (!track)
    {
        printf("Can't get track %d\n\n", id);
        return;
    }

    tr_Lock(track);
    if (tr_GetStatus(track) != eUserSelection && tr_GetStatus(track) != eRecognized)
    {
        tr_Unlock(track);
        tp_ReleaseTrack(pimp, track);
        printf("Only tracks waiting for a user selection or recognized tracks "
               "can be looked up as various artist.\n");
        return;
    }
    else
    {
        metadata_t data;

        tr_GetLocalMetadata(track, &data);
        free(data.artistId);
        data.artistId = strdup("89ad4ac3-39f7-470e-963a-56509c546377");
        tr_SetLocalMetadata(track, &data);
        tr_SetStatus(track, eFileLookup);
    }
    tr_Unlock(track);

    tp_Wake(pimp, track);
    tp_ReleaseTrack(pimp, track);
}

void setNewMetadata(tunepimp_t pimp, int id)
{
    track_t    track;
    metadata_t *data;
    char       temp[256];

    data = md_New();

    track = tp_GetTrack(pimp, id);
    if (!track)
    {
        printf("Can't get track %d\n\n", id);
        return;
    }

    tr_Lock(track);
    tr_GetServerMetadata(track, data);
    tr_Unlock(track);

    printf("Artist: ");
    fflush(stdout);
    fgets(temp, 255, stdin);
    temp[strlen(temp) - 1] = 0;
    free(data->artist);
    data->artist = strdup(temp);
    
    printf("Album: ");
    fflush(stdout);
    fgets(temp, 255, stdin);
    temp[strlen(temp) - 1] = 0;
    free(data->album);
    data->album = strdup(temp);
    
    printf("Track: ");
    fflush(stdout);
    fgets(temp, 255, stdin);
    temp[strlen(temp) - 1] = 0;
    free(data->track);
    data->track = strdup(temp);

    data->trackNum = 0;
    data->duration = 0;
    free(data->artistId);
    free(data->albumId);
    free(data->trackId);
    data->artistId = strdup("");
    data->albumId = strdup("");
    data->trackId = strdup("");

    tr_Lock(track);
    tr_SetServerMetadata(track, data);
    tr_SetStatus(track, eFileLookup);
    tr_Unlock(track);

    tp_Wake(pimp, track);
    tp_ReleaseTrack(pimp, track);

    md_Delete(data);
}

void selectTrack(tunepimp_t pimp, int id, int result)
{
    track_t    track;

    track = tp_GetTrack(pimp, id);
    if (!track)
    {
        printf("Can't get track %d\n\n", id);
        return;
    }

    tp_SelectResult(pimp, track, result);
    tp_ReleaseTrack(pimp, track);
}

void notify(tunepimp_t pimp, void *data, TPCallbackEnum type, int id)
{
    track_t    track;
    char       fileName[255];

    track = tp_GetTrack(pimp, id);
    if (track)
    {
        tr_Lock(track);
        tr_GetFileName(track, fileName, 255);
        tr_Unlock(track);
        tp_ReleaseTrack(pimp, track);
    }
    else
        strcpy(fileName, "<unknown>");

    switch(type)
    {
        case tpFileAdded:
            printf("File added: %s\n", fileName);
            break;
        case tpFileRemoved:
            printf("File removed: %d\n", id);
            break;
        case tpFileChanged:
            if (track)
                printf("%s: %s\n", fileStatusText[(int)tr_GetStatus(track)], fileName);
            break;
        case tpWriteTagsComplete:
            printf("Writing tags complete!\n");
            break;
        default:
            break;
    }
}

int main(int argc, char *argv[])
{
    char           *cmd;
    int             debugOpt = 0;
    int             renameOpt = 1;
    int             moveOpt = 1;
    int             clearTagsOpt = 0;
    char            server[255], outputDir[255], fileMask[255], variousFileMask[255];
    int             collThreshold = 80;
    int             autoSaveThreshold = -1, num, i, fileId;
    char            ext[10][32], user[255], passwd[255];
    TPCallbackEnum  type;

    strcpy(outputDir, "music");
    strcpy(server, "test.musicbrainz.org");
    strcpy(fileMask, "%abc/%sortname/%album/%sortname-%album-%0num-%track");
    strcpy(variousFileMask, "Various Artists/%album-%type-%status/%album-%0num-%sortname-%track");
    strcpy(user, "test");
    strcpy(passwd, "mb");

    pimp = tp_New("tp_tagger", "0.2.1");
    tp_SetUseUTF8(pimp, 1);
    // Not used right now -- we'll use the polling version
    //tp_SetNotifyCallback(pimp, notify);
    tp_SetDestDir(pimp, outputDir);
    tp_SetRenameFiles(pimp, 1);
    tp_SetMoveFiles(pimp, 1);
    tp_SetTopSrcDir(pimp, ".");
    tp_SetClearTags(pimp, clearTagsOpt);
    tp_SetFileMask(pimp, fileMask);
    tp_SetVariousFileMask(pimp, variousFileMask);
    tp_SetAutoSaveThreshold(pimp, autoSaveThreshold);
    tp_SetTRMCollisionThreshold(pimp, collThreshold);
    // Uncomment the following line if you object to vowels in your filenames
    //tp_SetAllowedFileCharacters(pimp, ".-bcdfghjklmnpqrstvwxyzBCDFGHJKLMNPQRSTVWXYZ");

    printf("Supported file extensions\n");
    num = tp_GetNumSupportedExtensions(pimp);

    tp_GetSupportedExtensions(pimp, ext);
    for(i = 0; i < num; i++)
        printf("%s ", ext[i]);

    printf("\nEnter ? or h for help\n\n");

    for(i = 1; i < argc; i++)
       tp_AddFile(pimp, argv[i]);

    tp_SetServer(pimp, server, 80);
    for(;;)
    {
        while(tp_GetNotification(pimp, &type, &fileId))
            notify(pimp, NULL, type, fileId);

        cmd = getCommand();
        if (*cmd == 0)
            continue;

        if (*cmd == 'q')
        {
            printf("cleaning up...\n");
            tp_Delete(pimp);
            return 0;
        }
        if (*cmd == 'l')
        {
            printList(pimp);
            continue;
        }
        if (*cmd == 'a')
        {
            tp_AddFile(pimp, cmd + 2);
            continue;
        }
        if (*cmd == 'd')
        {
            tp_AddDir(pimp, cmd + 2);
            continue;
        }
        if (*cmd == 'n')
        {
            if (strlen(cmd) <= 2)
                printf("Usage: n <file #>\n");
            else
                setNewMetadata(pimp, atoi(cmd + 2));
            continue;
        }
        if (*cmd == 'm')
        {
            if (strlen(cmd) <= 2)
                printf("Usage: m <file #>\n");
            else
                tp_Misidentified(pimp, atoi(cmd + 2));
            continue;
        }
        if (*cmd == 'i')
        {
            if (strlen(cmd) <= 2)
                printf("Usage: i <file #>\n");
            else
                tp_IdentifyAgain(pimp, atoi(cmd + 2));
            continue;
        }
        if (*cmd == 'p')
        {
            if (strlen(cmd) <= 2)
                printf("Usage: p <file #>\n");
            else
                printTrack(pimp, atoi(cmd + 2));
            continue;
        }
        if (*cmd == 'r')
        {
            if (strlen(cmd) <= 2)
                printf("Usage: r <file #>\n");
            else
                printResults(pimp, atoi(cmd + 2));
            continue;
        }
        if (*cmd == 'v')
        {
            if (strlen(cmd) <= 2)
                printf("Usage: v <file #>\n");
            else
                toggleVA(pimp, atoi(cmd + 2));
            continue;
        }
        if (*cmd == 'w')
        {
            int ret;

            if (strlen(cmd) <= 2)
                ret = tp_WriteTags(pimp, NULL, 0);
            else
            {
                int id;

                id = atoi(cmd + 2);
                ret = tp_WriteTags(pimp, &id, 1);
            }
            continue;
        }
        if (*cmd == 'c')
        {
            int fileId, result;

            if (sscanf(cmd, " %*c %d %d", &fileId, &result) == 2)
                selectTrack(pimp, fileId, result);
            else
                printf("usage: c <file #> <choice>\n");
            continue;
        }
        if (*cmd == 's')
        {
            tp_SetUserInfo(pimp, user, passwd);
            if (tp_SubmitTRMs(pimp) == tpOk)
                printf("Submitted ok.\n");
            else
            {
                char err[255];

                tp_GetError(pimp, err, 255);
                printf("Submit error: %s\n", err);
            }
            continue;
        }
        if (*cmd == 'o')
        {
            moveOpt = !moveOpt;
            tp_SetMoveFiles(pimp, moveOpt);
            printf("Move is: %s\n", moveOpt ? "on" : "off");
            continue;
        }
        if (*cmd == 'e')
        {
            renameOpt = !renameOpt;
            tp_SetRenameFiles(pimp, renameOpt);
            printf("Rename is: %s\n", renameOpt ? "on" : "off");
            continue;
        }
        if (*cmd == 'b')
        {
            debugOpt = !debugOpt;
            tp_SetDebug(pimp, debugOpt);
            printf("Debug: %s\n", debugOpt ? "on" : "off");
            continue;
        }
        if (strncmp(cmd, "tc", 2) == 0)
        {
            if (strlen(cmd) <= 2)
                printf("Usage: tc <trm collision threshold> (0-100)\n");
            else
            {
                collThreshold = atoi(cmd + 2);
                tp_SetTRMCollisionThreshold(pimp, collThreshold);
            }
            continue;
        }
        if (strncmp(cmd, "ta", 2) == 0)
        {
            if (strlen(cmd) <= 2)
                printf("Usage: ta <auto save threshold> (0-100)\n");
            else
            {
                autoSaveThreshold = atoi(cmd + 2);
                tp_SetAutoSaveThreshold(pimp, autoSaveThreshold);
            }
            continue;
        }
        if (*cmd == 't')
        {
            char newPath[255];

            if (sscanf(cmd, " %*c %[^\n]", newPath) == 1)
            {
                strcpy(outputDir, newPath);
                tp_SetDestDir(pimp, outputDir);
                printf("New output dir is [%s]\n", newPath);
            }
            else
                printf("usage: t <output dir>\n");

            continue;
        }
        if (strncmp(cmd, "ui", 2) == 0)
        {
            char newUser[255];
            char newPasswd[255];

            if (sscanf(cmd + 2, " %s %s", newUser, newPasswd) == 2)
            {
                strcpy(user, newUser);
                strcpy(passwd, newPasswd);
                printf("New username and password: %s, %s\n", user, passwd);
            }
            else
                printf("Usage: ui <user> <passwd>\n");

            continue;
        }

        if (*cmd == 'u')
        {
            char url[255];

            if (sscanf(cmd, " %*c %[^\n]", url) == 1)
            {
                strcpy(server, url);
                tp_SetServer(pimp, server, 80);
                printf("New server is [%s]\n", url);
            }
            else
                printf("usage: u <server url>\n");

            continue;
        }
        if (*cmd == 'x')
        {
            clearTagsOpt = !clearTagsOpt;
            tp_SetDebug(pimp, debugOpt);
            printf("Clear tags: %s\n", clearTagsOpt ? "on" : "off");
            tp_SetClearTags(pimp, clearTagsOpt);
            continue;
        }
        if (strncmp(cmd, "fv", 2) == 0)
        {
            char mask[255];

            if (sscanf(cmd, " %*c %[^\n]", mask) == 1)
            {
                strcpy(variousFileMask, mask);
                tp_SetVariousFileMask(pimp, variousFileMask);
                printf("New various file mask is [%s]\n", mask);
            }
            else
                printf("usage: fv <various file mask>\n");

            continue;
        }
        if (*cmd == 'f')
        {
            char mask[255];

            if (sscanf(cmd, " %*c %[^\n]", mask) == 1)
            {
                strcpy(fileMask, mask);
                tp_SetFileMask(pimp, fileMask);
                printf("New file mask is [%s]\n", mask);
            }
            else
                printf("usage: f <file mask>\n");

            continue;
        }
        if (*cmd == 'h' || *cmd == '?')
        {
            printf("Help:\n"
                   "Command:  Arguments:        Function:\n"
                   "---------------------------------------------------------\n"
                   "   a      <filename>        Add file\n"
                   "   d      <dir>             Add files in dir\n"
                   "   l                        List files in tagger\n"
                   "   p      <file #>          Print details of file\n"
                   "   r      <file #>          Show the matches for a file lookup\n"
                   "   q                        Quit\n"
                   "   w      [file #s]         Write tags & rename files\n"
                   "   c      <file #> <choice> Choose a search result\n"
                   "   s                        Submit TRMs to server\n"
                   "   i      <file #>          Identify file again\n"
                   "   m      <file #>          File was misidentified\n"
                   "   n      <file #>          Provide new metadata for file\n"
                   "   v      <file #>          Toggle Various Artist flag for file\n\n"
                   "Options:\n"
                   "---------------------------------------------------------\n"
                   "   e                        Toggle file rename (%s)\n"
                   "   o                        Toggle file move (%s)\n"
                   "   t      <output dir>      Set output dir (%s)\n"
                   "   u      <server name>     Use MB server (%s)\n"
                   "   ui     <name> <passwd>   Set MB user name/passwd (%s, %s)\n"
                   "   tc     <#>               Set TRM collision threshold (%d)\n"
                   "   ta     <#>               Set auto save threshold (%d)\n"
                   "   x                        Clear tags before writing new ones (%s)\n"
                   "   f      <mask>            Set the filemask (%s)\n"
                   "   fv     <va mask>         Set the various artist filemask (%s)\n"
                   "\n", 
                   renameOpt ? "on" : "off",
                   moveOpt ? "on" : "off",
                   outputDir,
                   server,
                   user,
                   passwd,
                   collThreshold,
                   autoSaveThreshold,
                   clearTagsOpt ? "on" : "off",
                   fileMask,
                   variousFileMask
                   );
            continue;
        }
        printf("Unknown command: %d\n", *cmd);
    }

}
