/*  tilp - link program for TI calculators
 *  Copyright (C) 1999, 2000  Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
  This unit contains functions for managing TIGraphLink file format
  and some other stuffs.
*/

#include "hfiles.h"
#include "tiffep.h"

/************************************************/
/* File information functions (stat equivalent) */
/************************************************/

int load_file_informations_82_83_85_86(FILE *file, TiffepFileInfo *fi);
int load_file_informations_89_92_92p(FILE *file, TiffepFileInfo *fi);

/*
  Returns some informations about a TIGraphLink file
  - filename [in]: the file to stat
  - fi [out]: the informations stored in an allocated structure
  (must be freed when no longer used)
  - int [out]: 0 if success, an error code otherwise.
*/

TIEXPORT
int tiffep_load_file_informations(gchar *filename, TiffepFileInfo **fileinfo)
{
  FILE *f;
  gint err = 0;
  gchar *ext;
  TiffepFileInfo *fi;
  struct stat f_info;

  f = fopen(filename, "rb");
  if(f == NULL)
    {
      g_error("Unable to open this file: <%s>\n", filename);
      return ERR_FILE_OPEN;
    }
  
  *fileinfo = tiffep_alloc_fi_struct();
  fi = *fileinfo;

  FI_FILESIZE(fi) = 0;
  if(!stat(filename, &f_info))
    {
      FI_FILESIZE(fi) = f_info.st_size;
    }

  FI_FILENAME(fi) = g_strdup(filename);
  FI_NAME(fi) = g_strdup(g_basename(filename));
  ext = strrchr(filename, '.');
  FI_EXTENSION(fi) = g_strdup(ext++);

  if(tiffep_is_a_group_file(filename))
    {
      FI_FILETYPE(fi) = FILE_GROUP;
      FI_NUMVARS(fi) = 0;
    }
  else if(tiffep_is_a_backup_file(filename))
    {
      FI_FILETYPE(fi) = FILE_BACKUP;
      FI_NUMVARS(fi) = 1;
    }
  else if(tiffep_is_a_flash_file(filename))
    {
      FI_FILETYPE(fi) = FILE_FLASH;
      FI_NUMVARS(fi) = 1;
    }
  else
    {
      FI_FILETYPE(fi) = FILE_SINGLE;
      FI_NUMVARS(fi) = 1;
    }

  FI_CALCTYPE(fi) = tiffep_which_calc_type_from_file(filename);
  if(FI_CALCTYPE(fi) == TYPE_NONE)
    {
      g_warning("Unknown file type.\n");
      return ERR_INVALID_FILE;
    }

  switch(FI_CALCTYPE(fi))
    {
    case TYPE_TI82:
    case TYPE_TI83:
    case TYPE_TI83P:
    case TYPE_TI85:
    case TYPE_TI86:
      err = load_file_informations_82_83_85_86(f, fi);
      break;
    case TYPE_TI89:
    case TYPE_TI92:
    case TYPE_TI92P:
      err = load_file_informations_89_92_92p(f, fi);
      break;
    default:
      g_warning("Unhandled case. This is a bug !\n");
      break;
    }
  fclose(f);

  return err;
}

int load_file_informations_89_92_92p_single(FILE *file, TiffepFileInfo *fi);
int load_file_informations_89_92_92p_backup(FILE *file, TiffepFileInfo *fi);
int load_file_informations_89_92_92p_group(FILE *file, TiffepFileInfo *fi);
int load_file_informations_89_92_92p_flash(FILE *file, TiffepFileInfo *fi);

/* 
   Load file informations for TI89..92+ calcs
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
   - int [out]: 0 if success, an error code otherwise.
*/
int load_file_informations_89_92_92p(FILE *file, TiffepFileInfo *fi)
{
  gchar buffer[MAXCHARS];
  gchar *header = buffer;
  gint i;

  fgets(header, 9, file);
  if(strcmp(header, "**TI92**") && strcmp(header, "**TI89**") &&
     strcmp(header, "**TI92P*"))
    return ERR_INVALID_FILE;
  fgetc(file);
  fgetc(file);

  for(i=0; i<8; i++) // skip folder name
    fgetc(file);
  fgets(buffer, 41, file);
  FI_COMMENT(fi) = g_strdup(buffer);
  FI_NUMVARS(fi) = fgetc(file);

  switch(FI_FILETYPE(fi))
    {
    case FILE_SINGLE:
      return load_file_informations_89_92_92p_single(file, fi);
      break;
    case FILE_GROUP:
      return load_file_informations_89_92_92p_group(file, fi);
      break;
    case FILE_FLASH:
      return load_file_informations_89_92_92p_flash(file, fi);
      break;
    case FILE_BACKUP:
      return load_file_informations_89_92_92p_backup(file, fi);
      break;
    default:
      g_warning("Unhandled case. This is a bug !\n");
      break;
    }

  return 0;
}

/* 
   Load file informations for a TI89..92+ single file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_89_92_92p_single(FILE *file, TiffepFileInfo *fi)
{
  gint i;
  gchar buffer[MAXCHARS];
  TiffepVarInfo *vi;
  TiffepFolderInfo *di;
  byte vartype;
  gchar trans[9];

  FI_NUMVARS(fi) = 1;
  rewind(file);
  for(i=0; i<10; i++)
    fgetc(file);
  fgets(buffer, 9, file);
  FI_VARLIST(fi) = NULL;
  FI_DIRLIST(fi) = NULL;
  vi = tiffep_alloc_vi_struct();
  di = tiffep_alloc_di_struct();
  FI_VARLIST(fi) = g_list_append(FI_VARLIST(fi), vi);
  FI_DIRLIST(fi) = g_list_append(FI_DIRLIST(fi), di);
  VI_PATH(vi) = g_strdup(buffer);
  DI_NAME(di) = g_strdup(buffer);
  DI_NUMVARS(di) = 1;
  fgets(buffer, 41, file);
  for(i=0; i<2; i++)
    fgetc(file);
  VI_OFFSET(vi) = BYTE2LONGWORD(fgetc(file), fgetc(file), 
			     fgetc(file), fgetc(file));
  DI_OFFSET(di) = VI_OFFSET(vi);
  fgets(buffer, 9, file);
  VI_CALCNAME(vi) = g_strdup(buffer);
  vartype = fgetc(file);
  VI_REALNAME(vi) = g_strdup(ti92_translate_varname(VI_CALCNAME(vi), 
						   trans, vartype));
  for(i=0; i<3; i++)
    fgetc(file);
  VI_TYPE(vi) = TIxx_TYPES_TO_COMMON_TYPES(vartype, FI_CALCTYPE(fi));
  VI_ATTRIBUTE(vi) = 0;
  VI_SIZE(vi) = BYTE2WORD(fgetc(file), fgetc(file)) - VI_OFFSET(vi) - 6;
  //VI_SIZE(vi) -= (0x52 + 6);
  VI_CONTENT(vi) = NULL;
  VI_CALCTYPE(vi) = FI_CALCTYPE(fi);
  FI_DATASIZE(fi) = VI_SIZE(vi);

  return 0;
}

/* 
   Load file informations for a TI89..92+ backup file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_89_92_92p_backup(FILE *file, TiffepFileInfo *fi)
{
  gint i;
  gchar buffer[MAXCHARS];
  TiffepVarInfo *vi;
  byte vartype;

  FI_NUMVARS(fi) = 1;
  rewind(file);
  for(i=0; i<10; i++)
    fgetc(file);
  fgets(buffer, 9, file);
  FI_VARLIST(fi) = NULL;
  vi = tiffep_alloc_vi_struct();
  FI_VARLIST(fi) = g_list_append(FI_VARLIST(fi), vi);
  VI_PATH(vi) = g_strdup(buffer);
  fgets(buffer, 41, file);
  for(i=0; i<2; i++)
    fgetc(file);
  VI_OFFSET(vi) = BYTE2LONGWORD(fgetc(file), fgetc(file), 
			     fgetc(file), fgetc(file));
  fgets(buffer, 9, file);
  VI_CALCNAME(vi) = g_strdup(buffer);
  VI_REALNAME(vi) = g_strdup(buffer);
  vartype = fgetc(file);
  for(i=0; i<3; i++)
    fgetc(file);
  VI_TYPE(vi) = TIxx_TYPES_TO_COMMON_TYPES(vartype, FI_CALCTYPE(fi));
  VI_ATTRIBUTE(vi) = 0;
  VI_SIZE(vi) = BYTE2WORD(fgetc(file), fgetc(file)) - VI_OFFSET(vi) - 2; //2\6?
  VI_CONTENT(vi) = NULL;
  VI_CALCTYPE(vi) = FI_CALCTYPE(fi);
  FI_DATASIZE(fi) = VI_SIZE(vi);

  return 0;
}

/* 
   Load file informations for a TI89..92+ group file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_89_92_92p_group(FILE *file, TiffepFileInfo *fi)
{
  gint i, j=0, k;
  gchar buffer[MAXCHARS];
  TiffepVarInfo *vi;
  TiffepFolderInfo *di;
  gchar *default_folder_name;
  gint num_of_entries = 0;
  gint num_of_vars_in_folder = 0;
  TiffepVarInfo *prev_var = NULL;
  gchar trans[9];
  byte vartype;

  rewind(file);
  for(i=0; i<10; i++)
    fgetc(file);
  fgets(buffer, 9, file);
  FI_VARLIST(fi) = NULL;
  FI_DIRLIST(fi) = NULL;
  default_folder_name = g_strdup(buffer);
  fgets(buffer, 41, file);
  num_of_entries = BYTE2WORD(fgetc(file), fgetc(file));
  FI_NUMVARS(fi) = 0;
  FI_DATASIZE(fi) = 0;

  k=0;
  while(k < num_of_entries)
    {
      di = tiffep_alloc_di_struct();
      FI_DIRLIST(fi) = g_list_append(FI_DIRLIST(fi), di);
      k++;

      DI_OFFSET(di) = BYTE2LONGWORD(fgetc(file), fgetc(file), 
				 fgetc(file), fgetc(file));
      for(i=0; i<8; i++) buffer[i]=fgetc(file);
      buffer[i]='\0';
      default_folder_name = g_strdup(buffer);
      DI_NAME(di) = g_strdup(buffer);
      vartype = fgetc(file);
      DI_TYPE(di) = TIxx_TYPES_TO_COMMON_TYPES(vartype, FI_CALCTYPE(fi));
      fgetc(file);
      num_of_vars_in_folder = BYTE2WORD(fgetc(file), fgetc(file));
      FI_NUMVARS(fi) += num_of_vars_in_folder;
      DI_SIZE(di) = num_of_vars_in_folder;

      for(j=0; j<num_of_vars_in_folder; j++)
	{
	  vi = tiffep_alloc_vi_struct();
	  FI_VARLIST(fi) = g_list_append(FI_VARLIST(fi), vi);  
	  k++;

	  VI_OFFSET(vi) = BYTE2LONGWORD(fgetc(file), fgetc(file), 
				     fgetc(file), fgetc(file));
	  for(i=0; i<8; i++) buffer[i]=fgetc(file);
	  buffer[i]='\0';
	  VI_PATH(vi)   = g_strdup(default_folder_name);
	  VI_CALCNAME(vi) = g_strdup(buffer);
	  vartype   = fgetc(file);
	  VI_TYPE(vi) = TIxx_TYPES_TO_COMMON_TYPES(vartype, FI_CALCTYPE(fi));
	  VI_REALNAME(vi) = g_strdup(ti92_translate_varname(VI_CALCNAME(vi), 
						       trans, VI_TYPE(vi)));
	  VI_ATTRIBUTE(vi) = fgetc(file);
	  VI_CALCTYPE(vi) = FI_CALCTYPE(fi);
	  for(i=0; i<2; i++) fgetc(file);
	  
	  if((j > 0) || (k > 2))
	    {
	      VI_SIZE(prev_var) = VI_OFFSET(vi) - VI_OFFSET(prev_var) - 6;
	      FI_DATASIZE(fi) += VI_SIZE(prev_var);
	      DISPLAY2("name = %s, size = %i\n", VI_CALCNAME(vi), 
		      VI_SIZE(prev_var));
	    }
	  prev_var = vi;
	}
    }
  // read the last offset (byte after the checksum
  /*
  VI_SIZE(prev_var) = BYTE2WORD(fgetc(file), fgetc(file)) - 
    VI_OFFSET(prev_var) - 6;
  FI_DATASIZE(fi) += VI_SIZE(prev_var);
  */
  /*
    bug here
  DISPLAY("name = %s, size = %i\n", VI_CALCNAME(vi), 
	  VI_SIZE(prev_var));
  */
  return 0;
}

/* 
   Load file informations for a TI89..92+ flash file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_89_92_92p_flash(FILE *file, TiffepFileInfo *fi)
{
  return 0;
}

int load_file_informations_82_86_single(FILE *file, TiffepFileInfo *fi);
int load_file_informations_82_86_backup(FILE *file, TiffepFileInfo *fi);
int load_file_informations_82_86_group(FILE *file, TiffepFileInfo *fi);
int load_file_informations_82_86_flash(FILE *file, TiffepFileInfo *fi);

/* 
   Load file informations for TI82..86 calcs
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
   - int [out]: 0 if success, an error code otherwise.
*/
int load_file_informations_82_83_85_86(FILE *file, TiffepFileInfo *fi)
{
  gchar buffer[MAXCHARS];
  gchar *header = buffer;
  gint i;

  fgets(header, 9, file);
  if(strcmp(header, "**TI82**") && strcmp(header, "**TI83**") &&
     strcmp(header, "**TI93P*") && strcmp(header, "**TI85**") &&
     strcmp(header, "**TI86**") )
    return ERR_INVALID_FILE;
  for(i=0; i<3; i++) fgetc(file);

  fgets(buffer, 43, file);
  FI_COMMENT(fi) = g_strdup(buffer);

  switch(FI_FILETYPE(fi))
    {
    case FILE_SINGLE:
      return load_file_informations_82_86_single(file, fi);
      break;
    case FILE_GROUP:
      return load_file_informations_82_86_group(file, fi);
      break;
    case FILE_FLASH:
      return load_file_informations_82_86_flash(file, fi);
      break;
    case FILE_BACKUP:
      return load_file_informations_82_86_backup(file, fi);
      break;
    default:
      g_warning("Unhandled case. This is a bug !\n");
      break;
    }

  return 0;
}

/* 
   Load file informations for a TI82..86 single file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_82_86_single(FILE *file, TiffepFileInfo *fi)
{
  gint i;
  gchar buffer[MAXCHARS];
  TiffepVarInfo *vi;
  byte vartype;
  gchar trans[9];

  FI_NUMVARS(fi) = 1;
  rewind(file);
  for(i=0; i<11; i++) fgetc(file);
  for(i=0; i<42; i++) fgetc(file);

  FI_VARLIST(fi) = NULL;
  vi = tiffep_alloc_vi_struct();
  FI_VARLIST(fi) = g_list_append(FI_VARLIST(fi), vi);

  FI_DATASIZE(fi) = BYTE2WORD(fgetc(file), fgetc(file));
  fgetc(file); fgetc(file);
  VI_SIZE(vi) = BYTE2WORD(fgetc(file), fgetc(file));
  vartype = fgetc(file);
  VI_TYPE(vi) = TIxx_TYPES_TO_COMMON_TYPES(vartype, FI_CALCTYPE(fi));

  fgets(buffer, 9, file);
  VI_CALCNAME(vi) = g_strdup(buffer);
  VI_REALNAME(vi) = g_strdup(ti92_translate_varname(VI_CALCNAME(vi), 
						   trans, vartype));
  fgetc(file); fgetc(file); //varsize redundancy
  VI_OFFSET(vi) = ftell(file);
  VI_PATH(vi) = g_strdup("");
  VI_ATTRIBUTE(vi) = 0;
  VI_CONTENT(vi) = NULL;
  VI_CALCTYPE(vi) = FI_CALCTYPE(fi);

  return 0;
}

/* 
   Load file informations for a TI82..86 backup file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_82_86_backup(FILE *file, TiffepFileInfo *fi)
{
  gint i;
  //gchar buffer[MAXCHARS];
  TiffepVarInfo *vi;
  byte vartype;
  //gchar trans[9];

  FI_NUMVARS(fi) = 1;
  rewind(file);
  for(i=0; i<11; i++) fgetc(file);
  for(i=0; i<42; i++) fgetc(file);

  FI_VARLIST(fi) = NULL;
  vi = tiffep_alloc_vi_struct();
  FI_VARLIST(fi) = g_list_append(FI_VARLIST(fi), vi);

  FI_DATASIZE(fi) = BYTE2WORD(fgetc(file), fgetc(file));
  VI_SIZE(vi) = FI_DATASIZE(fi);

  vartype = TI82_BKUP;
  VI_TYPE(vi) = TIxx_TYPES_TO_COMMON_TYPES(vartype, FI_CALCTYPE(fi));

  fgetc(file); fgetc(file);
  for(i=0; i<9; i++) fgetc(file);
  VI_OFFSET(vi) = ftell(file);
  VI_PATH(vi) = g_strdup("");
  VI_ATTRIBUTE(vi) = 0;
  VI_CONTENT(vi) = NULL;
  VI_CALCTYPE(vi) = FI_CALCTYPE(fi);

  return 0;
}

/* 
   Load file informations for a TI82..86 group file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_82_86_group(FILE *file, TiffepFileInfo *fi)
{
  gint i, j, k;
  gchar buffer[MAXCHARS];
  TiffepVarInfo *vi;
  gchar *default_folder_name;
  gint num_of_entries = 0;
  gint num_of_vars_in_folder = 0;
  TiffepVarInfo *prev_var = NULL;
  gchar trans[9];

  rewind(file);
  for(i=0; i<10; i++)
    fgetc(file);
  fgets(buffer, 9, file);
  FI_VARLIST(fi) = NULL;
  default_folder_name = g_strdup(buffer);
  fgets(buffer, 41, file);
  num_of_entries = BYTE2WORD(fgetc(file), fgetc(file));
  FI_NUMVARS(fi) = 0;
  FI_DATASIZE(fi) = 0;

  k=0;
  while(k < num_of_entries)
    {
      vi = tiffep_alloc_vi_struct();
      FI_DIRLIST(fi) = g_list_append(FI_DIRLIST(fi), vi);
      //FI_VARINFO(fi) = (FI_VARLIST(fi))->data;
      k++;

      VI_OFFSET(vi) = BYTE2LONGWORD(fgetc(file), fgetc(file), 
				 fgetc(file), fgetc(file));
      for(i=0; i<8; i++) buffer[i]=fgetc(file);
      buffer[i]='\0';
      default_folder_name = g_strdup(buffer);
      VI_PATH(vi) = g_strdup(buffer);
      VI_CALCNAME(vi) = g_strdup(buffer);
      VI_TYPE(vi) = fgetc(file);
      VI_REALNAME(vi) = g_strdup(ti92_translate_varname(VI_CALCNAME(vi), 
						       trans, VI_TYPE(vi)));
      fgetc(file);
      num_of_vars_in_folder = BYTE2WORD(fgetc(file), fgetc(file));
      FI_NUMVARS(fi) += num_of_vars_in_folder;
      VI_SIZE(vi) = num_of_vars_in_folder;
      VI_CALCTYPE(vi) = FI_CALCTYPE(fi);
      
      for(j=0; j<num_of_vars_in_folder; j++)
	{
	  vi = tiffep_alloc_vi_struct();
	  FI_VARLIST(fi) = g_list_append(FI_VARLIST(fi), vi);
	  k++;

	  VI_OFFSET(vi) = BYTE2LONGWORD(fgetc(file), fgetc(file), 
				     fgetc(file), fgetc(file));
	  for(i=0; i<8; i++) buffer[i]=fgetc(file);
	  buffer[i]='\0';
	  VI_PATH(vi) = g_strdup(default_folder_name);
	  VI_CALCNAME(vi) = g_strdup(buffer);
	  VI_TYPE(vi) = fgetc(file);
	  VI_REALNAME(vi) = g_strdup(ti92_translate_varname(VI_CALCNAME(vi), 
						       trans, VI_TYPE(vi)));
	  VI_CALCTYPE(vi) = FI_CALCTYPE(fi);
	  fgetc(file);
	  for(i=0; i<2; i++) fgetc(file);
	  
	  if(j != 0)
	    {
	      VI_SIZE(prev_var) = VI_OFFSET(vi) - VI_OFFSET(prev_var) - 6;
	      FI_DATASIZE(fi) += VI_SIZE(prev_var);
	    }
	  prev_var = vi;
	}
      // read the last offset (byte after the checksum
      VI_SIZE(prev_var) = BYTE2WORD(fgetc(file), fgetc(file)) - 
	VI_OFFSET(prev_var) - 6;
      FI_DATASIZE(fi) += VI_SIZE(prev_var);
    }

  /* Point the first element and not the last ! */
  //FI_VARINFO(fi) = (FI_VARLIST(fi))->data;
  
  return 0;
}

/* 
   Load file informations for a TI82..86 flash file
   - file [in]: a pointer on the file to stat 
   - fi [out]: the informations to complete
*/
int load_file_informations_82_86_flash(FILE *file, TiffepFileInfo *fi)
{
  return 0;
}


/********************************************/
/* File content functions (read equivalent) */
/********************************************/


int load_file_content_82_83_85_86(FILE *file, TiffepFileInfo *fi);
int load_file_content_89_92_92p(FILE *file, TiffepFileInfo *fi);

/*
  Load the content of the specified TIGraphLink file.
  - filename [in]: the file to load
  - fileinfo [out]: the informations and content stored in an 
  allocated structure. Must be freed when no longer used.
  - int [out]: 0 if success, an error code otherwise.
*/
TIEXPORT
int tiffep_load_file_content(gchar *filename, TiffepFileInfo **fileinfo)
{
  gint err = 0;
  TiffepFileInfo *fi;
  FILE *f;

  /* Load informations (open & close the file) */
  err = tiffep_load_file_informations(filename, fileinfo);
  fi = *fileinfo;
  
  f = fopen(filename, "rb");
  if(f == NULL)
    {
      g_error("Unable to open this file: <%s>\n", filename);
      return ERR_FILE_OPEN;
    }

  switch(FI_CALCTYPE(fi))
    {
    case TYPE_TI82:
    case TYPE_TI83:
    case TYPE_TI83P:
    case TYPE_TI85:
    case TYPE_TI86:
      err = load_file_content_82_83_85_86(f, fi);
      break;
    case TYPE_TI89:
    case TYPE_TI92:
    case TYPE_TI92P:
      err = load_file_content_89_92_92p(f, fi);
      break;
    default:
      g_warning("Unhandled case. This is a bug !\n");
      break;
    }
  fclose(f);

  return err;
}

/* Load the var content from the file to memory for each variable */
static void load_var_content_1(gpointer data, gpointer user_data)
{
  TiffepVarInfo *vi = (TiffepVarInfo *)data;
  FILE *f = (FILE *)user_data;

  fseek(f, VI_OFFSET(vi), SEEK_SET);
  //fprintf(stderr, "varname = %s, varsize = %i\n", VI_CALCNAME(vi), VI_SIZE(vi));
  //if(VI_SIZE(vi) < 0) return;
  VI_CONTENT(vi) = (byte *)g_malloc(VI_SIZE(vi) * sizeof(byte));
  fgetc(f); fgetc(f); fgetc(f); fgetc(f);
  fread(VI_CONTENT(vi), sizeof(byte), VI_SIZE(vi), f);

  /*  
  printf("vi_size = %i\n", VI_SIZE(vi));
  for(i=0; i< VI_SIZE(vi); i++)
    printf("0x%02x ", VI_CONTENT(vi)[i]);
  printf("\n");
  */
}

/* 
   Load file content for TI89..92+ calcs
   - file [in]: a pointer on the file to load 
   - fi [out]: the informations to complete
*/
int load_file_content_89_92_92p(FILE *file, TiffepFileInfo *fi)
{
  GList *varlist = FI_VARLIST(fi);

  g_list_foreach(varlist, load_var_content_1, file);

  return 0;
}

/* Load the var content from the file to memory for each variable */
static void load_var_content_2(gpointer data, gpointer user_data)
{
  TiffepVarInfo *vi = (TiffepVarInfo *)data;
  FILE *f = (FILE *)user_data;

  fseek(f, VI_OFFSET(vi), SEEK_SET);
  VI_CONTENT(vi) = (byte *)g_malloc(VI_SIZE(vi) * sizeof(byte));
  fread(VI_CONTENT(vi), sizeof(byte), VI_SIZE(vi), f);
}

/* Load file content for TI82..86 calcs */
int load_file_content_82_83_85_86(FILE *file, TiffepFileInfo *fi)
{
  GList *varlist = FI_VARLIST(fi);

  g_list_foreach(varlist, load_var_content_2, file);

  return 0;
}





