module tango.text.locale.Data;

private import tango.core.Exception;

package void error(char[] msg) {
     throw new LocaleException (msg);
}

package int compareString(char[] strA, char[] strB) {
  // Comparison ignores case
  int strALength = strA.length;
  int strBLength = strB.length;
  int l = strALength;
  if (strBLength < l)
    l = strBLength;

  int result;

  for (int i = 0; i < l; i++) {
    if (strA[i] != strB[i]) {
      char chA = strA[i];
      char chB = strB[i];
      if (chA >= 'A' && chA <= 'Z')
        chA += cast(int)'a' - cast(int)'A';
      if (chB >= 'A' && chB <= 'Z')
        chB += cast(int)'a' - cast(int)'A';
      result = cast(int)chA - cast(int)chB;
      if (result != 0)
        break;
    }
  }
  if (result == 0)
    result = strALength - strBLength;
  return result;
}

package struct CultureData {

  package int lcid;
  package int parent;
  package char[] name;
  package char[] isoLangName;
  package char[] isoLangName2;
  package char[] ietfTag;
  package char[] englishName;
  package char[] nativeName;
  package bool isNeutral;

  package int geoId;
  package char[] regionName;
  package char[] isoRegionName;
  package char[] englishCountry;
  package char[] nativeCountry;
  package char[] intlSymbol;
  package char[] englishCurrency;
  package char[] nativeCurrency;
  package bool isMetric;

  package int digits;
  package int negativeNumber;
  package int currencyDigits;
  package int negativeCurrency;
  package int positiveCurrency;
  package int[] grouping;
  package int[] monetaryGrouping;
  package char[] decimal;
  package char[] thousand;
  package char[] monetaryDecimal;
  package char[] monetaryThousand;
  package char[] currency;
  package char[] negativeSign;
  package char[] positiveSign;
  package char[] nan;
  package char[] negInfinity;
  package char[] posInfinity;
  package char[][] nativeDigits;

  package int calendarType;
  package int[] optionalCalendars;
  package char[] nativeCalName;
  package int firstDayOfWeek;
  package int firstDayOfYear;
  package char[] date;
  package char[] time;
  package char[] am;
  package char[] pm;
  package char[] shortDate;
  package char[] shortTime;
  package char[] longDate;
  package char[] longTime;
  package char[] monthDay;
  package char[] yearMonth;
  package char[][] shortTimes;
  package char[][] shortDates;
  package char[][] longTimes;
  package char[][] longDates;
  package char[][] yearMonths;
  package char[][] abbrevDayNames;
  package char[][] dayNames;
  package char[][] abbrevMonthNames;
  package char[][] monthNames;

  package static const CultureData[] cultureDataTable = [
{ 0x0001, 0x007F, "ar", "ar", "ara", "ar", "\u0041\u0072\u0061\u0062\u0069\u0063", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629", true },
{ 0x0002, 0x007F, "bg", "bg", "bul", "bg", "\u0042\u0075\u006C\u0067\u0061\u0072\u0069\u0061\u006E", "\u0431\u044A\u043B\u0433\u0430\u0440\u0441\u043A\u0438", true },
{ 0x0003, 0x007F, "ca", "ca", "cat", "ca", "\u0043\u0061\u0074\u0061\u006C\u0061\u006E", "\u0063\u0061\u0074\u0061\u006C\u00E0", true },
{ 0x0004, 0x007F, "zh-CHS", "zh", "zho", "zh-Hans", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u0053\u0069\u006D\u0070\u006C\u0069\u0066\u0069\u0065\u0064\u0029", "\u4E2D\u6587\u0028\u7B80\u4F53\u0029", true },
{ 0x0005, 0x007F, "cs", "cs", "ces", "cs", "\u0043\u007A\u0065\u0063\u0068", "\u010D\u0065\u0161\u0074\u0069\u006E\u0061", true },
{ 0x0006, 0x007F, "da", "da", "dan", "da", "\u0044\u0061\u006E\u0069\u0073\u0068", "\u0064\u0061\u006E\u0073\u006B", true },
{ 0x0007, 0x007F, "de", "de", "deu", "de", "\u0047\u0065\u0072\u006D\u0061\u006E", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068", true },
{ 0x0008, 0x007F, "el", "el", "ell", "el", "\u0047\u0072\u0065\u0065\u006B", "\u03B5\u03BB\u03BB\u03B7\u03BD\u03B9\u03BA\u03AC", true },
{ 0x0009, 0x007F, "en", "en", "eng", "en", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068", true },
{ 0x000A, 0x007F, "es", "es", "spa", "es", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068", "\u0065\u0073\u0070\u0061\u00F1\u006F\u006C", true },
{ 0x000B, 0x007F, "fi", "fi", "fin", "fi", "\u0046\u0069\u006E\u006E\u0069\u0073\u0068", "\u0073\u0075\u006F\u006D\u0069", true },
{ 0x000C, 0x007F, "fr", "fr", "fra", "fr", "\u0046\u0072\u0065\u006E\u0063\u0068", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073", true },
{ 0x000D, 0x007F, "he", "he", "heb", "he", "\u0048\u0065\u0062\u0072\u0065\u0077", "\u05E2\u05D1\u05E8\u05D9\u05EA", true },
{ 0x000E, 0x007F, "hu", "hu", "hun", "hu", "\u0048\u0075\u006E\u0067\u0061\u0072\u0069\u0061\u006E", "\u006D\u0061\u0067\u0079\u0061\u0072", true },
{ 0x000F, 0x007F, "is", "is", "isl", "is", "\u0049\u0063\u0065\u006C\u0061\u006E\u0064\u0069\u0063", "\u00ED\u0073\u006C\u0065\u006E\u0073\u006B\u0061", true },
{ 0x0010, 0x007F, "it", "it", "ita", "it", "\u0049\u0074\u0061\u006C\u0069\u0061\u006E", "\u0069\u0074\u0061\u006C\u0069\u0061\u006E\u006F", true },
{ 0x0011, 0x007F, "ja", "ja", "jpn", "ja", "\u004A\u0061\u0070\u0061\u006E\u0065\u0073\u0065", "\u65E5\u672C\u8A9E", true },
{ 0x0012, 0x007F, "ko", "ko", "kor", "ko", "\u004B\u006F\u0072\u0065\u0061\u006E", "\uD55C\uAD6D\uC5B4", true },
{ 0x0013, 0x007F, "nl", "nl", "nld", "nl", "\u0044\u0075\u0074\u0063\u0068", "\u004E\u0065\u0064\u0065\u0072\u006C\u0061\u006E\u0064\u0073", true },
{ 0x0014, 0x007F, "no", "no", "nor", "no", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E", "\u006E\u006F\u0072\u0073\u006B", true },
{ 0x0015, 0x007F, "pl", "pl", "pol", "pl", "\u0050\u006F\u006C\u0069\u0073\u0068", "\u0070\u006F\u006C\u0073\u006B\u0069", true },
{ 0x0016, 0x007F, "pt", "pt", "por", "pt", "\u0050\u006F\u0072\u0074\u0075\u0067\u0075\u0065\u0073\u0065", "\u0050\u006F\u0072\u0074\u0075\u0067\u0075\u00EA\u0073", true },
{ 0x0018, 0x007F, "ro", "ro", "ron", "ro", "\u0052\u006F\u006D\u0061\u006E\u0069\u0061\u006E", "\u0072\u006F\u006D\u00E2\u006E\u0103", true },
{ 0x0019, 0x007F, "ru", "ru", "rus", "ru", "\u0052\u0075\u0073\u0073\u0069\u0061\u006E", "\u0440\u0443\u0441\u0441\u043A\u0438\u0439", true },
{ 0x001A, 0x007F, "hr", "hr", "hrv", "hr", "\u0043\u0072\u006F\u0061\u0074\u0069\u0061\u006E", "\u0068\u0072\u0076\u0061\u0074\u0073\u006B\u0069", true },
{ 0x001B, 0x007F, "sk", "sk", "slk", "sk", "\u0053\u006C\u006F\u0076\u0061\u006B", "\u0073\u006C\u006F\u0076\u0065\u006E\u010D\u0069\u006E\u0061", true },
{ 0x001C, 0x007F, "sq", "sq", "sqi", "sq", "\u0041\u006C\u0062\u0061\u006E\u0069\u0061\u006E", "\u0073\u0068\u0071\u0069\u0070\u0065", true },
{ 0x001D, 0x007F, "sv", "sv", "swe", "sv", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068", "\u0073\u0076\u0065\u006E\u0073\u006B\u0061", true },
{ 0x001E, 0x007F, "th", "th", "tha", "th", "\u0054\u0068\u0061\u0069", "\u0E44\u0E17\u0E22", true },
{ 0x001F, 0x007F, "tr", "tr", "tur", "tr", "\u0054\u0075\u0072\u006B\u0069\u0073\u0068", "\u0054\u00FC\u0072\u006B\u00E7\u0065", true },
{ 0x0020, 0x007F, "ur", "ur", "urd", "ur", "\u0055\u0072\u0064\u0075", "\u0627\u064F\u0631\u062F\u0648", true },
{ 0x0021, 0x007F, "id", "id", "ind", "id", "\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061\u006E", "\u0042\u0061\u0068\u0061\u0073\u0061\u0020\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061", true },
{ 0x0022, 0x007F, "uk", "uk", "ukr", "uk", "\u0055\u006B\u0072\u0061\u0069\u006E\u0069\u0061\u006E", "\u0443\u043A\u0440\u0430\u0457\u043D\u044C\u0441\u043A\u0430", true },
{ 0x0023, 0x007F, "be", "be", "bel", "be", "\u0042\u0065\u006C\u0061\u0072\u0075\u0073\u0069\u0061\u006E", "\u0411\u0435\u043B\u0430\u0440\u0443\u0441\u043A\u0456", true },
{ 0x0024, 0x007F, "sl", "sl", "slv", "sl", "\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u0061\u006E", "\u0073\u006C\u006F\u0076\u0065\u006E\u0073\u006B\u0069", true },
{ 0x0025, 0x007F, "et", "et", "est", "et", "\u0045\u0073\u0074\u006F\u006E\u0069\u0061\u006E", "\u0065\u0065\u0073\u0074\u0069", true },
{ 0x0026, 0x007F, "lv", "lv", "lav", "lv", "\u004C\u0061\u0074\u0076\u0069\u0061\u006E", "\u006C\u0061\u0074\u0076\u0069\u0065\u0161\u0075", true },
{ 0x0027, 0x007F, "lt", "lt", "lit", "lt", "\u004C\u0069\u0074\u0068\u0075\u0061\u006E\u0069\u0061\u006E", "\u006C\u0069\u0065\u0074\u0075\u0076\u0069\u0173", true },
{ 0x0029, 0x007F, "fa", "fa", "fas", "fa", "\u0050\u0065\u0072\u0073\u0069\u0061\u006E", "\u0641\u0627\u0631\u0633\u0649", true },
{ 0x002A, 0x007F, "vi", "vi", "vie", "vi", "\u0056\u0069\u0065\u0074\u006E\u0061\u006D\u0065\u0073\u0065", "\u0054\u0069\u00EA\u0301\u006E\u0067\u0020\u0056\u0069\u00EA\u0323\u0074", true },
{ 0x002B, 0x007F, "hy", "hy", "hye", "hy", "\u0041\u0072\u006D\u0065\u006E\u0069\u0061\u006E", "\u0540\u0561\u0575\u0565\u0580\u0565\u0576", true },
{ 0x002C, 0x007F, "az", "az", "aze", "az", "\u0041\u007A\u0065\u0072\u0069", "\u0041\u007A\u0259\u0072\u0062\u0061\u0079\u0063\u0061\u006E\u00AD\u0131\u006C\u0131", true },
{ 0x002D, 0x007F, "eu", "eu", "eus", "eu", "\u0042\u0061\u0073\u0071\u0075\u0065", "\u0065\u0075\u0073\u006B\u0061\u0072\u0061", true },
{ 0x002F, 0x007F, "mk", "mk", "mkd", "mk", "\u004D\u0061\u0063\u0065\u0064\u006F\u006E\u0069\u0061\u006E", "\u043C\u0430\u043A\u0435\u0434\u043E\u043D\u0441\u043A\u0438\u0020\u0458\u0430\u0437\u0438\u043A", true },
{ 0x0036, 0x007F, "af", "af", "afr", "af", "\u0041\u0066\u0072\u0069\u006B\u0061\u0061\u006E\u0073", "\u0041\u0066\u0072\u0069\u006B\u0061\u0061\u006E\u0073", true },
{ 0x0037, 0x007F, "ka", "ka", "kat", "ka", "\u0047\u0065\u006F\u0072\u0067\u0069\u0061\u006E", "\u10E5\u10D0\u10E0\u10D7\u10E3\u10DA\u10D8", true },
{ 0x0038, 0x007F, "fo", "fo", "fao", "fo", "\u0046\u0061\u0072\u006F\u0065\u0073\u0065", "\u0066\u00F8\u0072\u006F\u0079\u0073\u006B\u0074", true },
{ 0x0039, 0x007F, "hi", "hi", "hin", "hi", "\u0048\u0069\u006E\u0064\u0069", "\u0939\u093F\u0902\u0926\u0940", true },
{ 0x003E, 0x007F, "ms", "ms", "msa", "ms", "\u004D\u0061\u006C\u0061\u0079", "\u0042\u0061\u0068\u0061\u0073\u0061\u0020\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061", true },
{ 0x003F, 0x007F, "kk", "kk", "kaz", "kk", "\u004B\u0061\u007A\u0061\u006B\u0068", "\u049A\u0430\u0437\u0430\u0449\u0062", true },
{ 0x0040, 0x007F, "ky", "ky", "kir", "ky", "\u004B\u0079\u0072\u0067\u0079\u007A", "\u041A\u044B\u0440\u0433\u044B\u0437", true },
{ 0x0041, 0x007F, "sw", "sw", "swa", "sw", "\u004B\u0069\u0073\u0077\u0061\u0068\u0069\u006C\u0069", "\u004B\u0069\u0073\u0077\u0061\u0068\u0069\u006C\u0069", true },
{ 0x0043, 0x007F, "uz", "uz", "uzb", "uz", "\u0055\u007A\u0062\u0065\u006B", "\u0055\u0027\u007A\u0062\u0065\u006B", true },
{ 0x0044, 0x007F, "tt", "tt", "tat", "tt", "\u0054\u0061\u0074\u0061\u0072", "\u0422\u0430\u0442\u0430\u0440", true },
{ 0x0046, 0x007F, "pa", "pa", "pan", "pa", "\u0050\u0075\u006E\u006A\u0061\u0062\u0069", "\u0A2A\u0A70\u0A1C\u0A3E\u0A2C\u0A40", true },
{ 0x0047, 0x007F, "gu", "gu", "guj", "gu", "\u0047\u0075\u006A\u0061\u0072\u0061\u0074\u0069", "\u0A97\u0AC1\u0A9C\u0AB0\u0ABE\u0AA4\u0AC0", true },
{ 0x0049, 0x007F, "ta", "ta", "tam", "ta", "\u0054\u0061\u006D\u0069\u006C", "\u0BA4\u0BAE\u0BBF\u0BB4\u0BCD", true },
{ 0x004A, 0x007F, "te", "te", "tel", "te", "\u0054\u0065\u006C\u0075\u0067\u0075", "\u0C24\u0C46\u0C32\u0C41\u0C17\u0C41", true },
{ 0x004B, 0x007F, "kn", "kn", "kan", "kn", "\u004B\u0061\u006E\u006E\u0061\u0064\u0061", "\u0C95\u0CA8\u0CCD\u0CA8\u0CA1", true },
{ 0x004E, 0x007F, "mr", "mr", "mar", "mr", "\u004D\u0061\u0072\u0061\u0074\u0068\u0069", "\u092E\u0930\u093E\u0920\u0940", true },
{ 0x004F, 0x007F, "sa", "sa", "san", "sa", "\u0053\u0061\u006E\u0073\u006B\u0072\u0069\u0074", "\u0938\u0902\u0938\u094D\u0915\u0943\u0924", true },
{ 0x0050, 0x007F, "mn", "mn", "mon", "mn", "\u004D\u006F\u006E\u0067\u006F\u006C\u0069\u0061\u006E", "\u041C\u043E\u043D\u0433\u043E\u043B\u00A0\u0445\u044D\u043B", true },
{ 0x0056, 0x007F, "gl", "gl", "glg", "gl", "\u0047\u0061\u006C\u0069\u0063\u0069\u0061\u006E", "\u0067\u0061\u006C\u0065\u0067\u006F", true },
{ 0x0057, 0x007F, "kok", "kok", "kok", "kok", "\u004B\u006F\u006E\u006B\u0061\u006E\u0069", "\u0915\u094B\u0902\u0915\u0923\u0940", true },
{ 0x005A, 0x007F, "syr", "syr", "syr", "syr", "\u0053\u0079\u0072\u0069\u0061\u0063", "\u0723\u0718\u072A\u071D\u071D\u0710", true },
{ 0x0065, 0x007F, "div", "div", "div", "div", "\u0044\u0069\u0076\u0065\u0068\u0069", "\u078B\u07A8\u0788\u07AC\u0780\u07A8\u0784\u07A6\u0790\u07B0", true },
{ 0x007F, 0x007F, "", "iv", "IVL", "", "\u0049\u006E\u0076\u0061\u0072\u0069\u0061\u006E\u0074\u0020\u004C\u0061\u006E\u0067\u0075\u0061\u0067\u0065\u0020\u0028\u0049\u006E\u0076\u0061\u0072\u0069\u0061\u006E\u0074\u0020\u0043\u006F\u0075\u006E\u0074\u0072\u0079\u0029", "\u0049\u006E\u0076\u0061\u0072\u0069\u0061\u006E\u0074\u0020\u004C\u0061\u006E\u0067\u0075\u0061\u0067\u0065\u0020\u0028\u0049\u006E\u0076\u0061\u0072\u0069\u0061\u006E\u0074\u0020\u0043\u006F\u0075\u006E\u0074\u0072\u0079\u0029", false },
{ 0x0401, 0x0001, "ar-SA", "ar", "ara", "ar-SA", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0053\u0061\u0075\u0064\u0069\u0020\u0041\u0072\u0061\u0062\u0069\u0061\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0645\u0645\u0644\u0643\u0629\u0020\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0627\u0644\u0633\u0639\u0648\u062F\u064A\u0629\u0029", false, 0x00CD, "SA", "SAU", "\u0053\u0061\u0075\u0064\u0069\u0020\u0041\u0072\u0061\u0062\u0069\u0061", "\u0627\u0644\u0645\u0645\u0644\u0643\u0629\u0020\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0627\u0644\u0633\u0639\u0648\u062F\u064A\u0629", "\u0053\u0041\u0052", "\u0053\u0061\u0075\u0064\u0069\u0020\u0052\u0069\u0079\u0061\u006C", "\u0631\u064A\u0627\u0644\u00A0\u0633\u0639\u0648\u062F\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0631\u002E\u0633\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 6, [ 6, 23, 2, 9, 10, 1, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0647\u062C\u0631\u064A", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u002F\u004D\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u002F\u004D\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0645\u062D\u0631\u0645", "\u0635\u0641\u0631", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u0627\u0648\u0644", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u0627\u0648\u0644\u0649", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u062B\u0627\u0646\u064A\u0629", "\u0631\u062C\u0628", "\u0634\u0639\u0628\u0627\u0646", "\u0631\u0645\u0636\u0627\u0646", "\u0634\u0648\u0627\u0644", "\u0630\u0648\u00A0\u0627\u0644\u0642\u0639\u062F\u0629", "\u0630\u0648\u00A0\u0627\u0644\u062D\u062C\u0629", "" ], [ "\u0645\u062D\u0631\u0645", "\u0635\u0641\u0631", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u0623\u0648\u0644", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u0623\u0648\u0644\u0649", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u062B\u0627\u0646\u064A\u0629", "\u0631\u062C\u0628", "\u0634\u0639\u0628\u0627\u0646", "\u0631\u0645\u0636\u0627\u0646", "\u0634\u0648\u0627\u0644", "\u0630\u0648\u00A0\u0627\u0644\u0642\u0639\u062F\u0629", "\u0630\u0648\u00A0\u0627\u0644\u062D\u062C\u0629", "" ] },
{ 0x0402, 0x0002, "bg-BG", "bg", "bul", "bg-BG", "\u0042\u0075\u006C\u0067\u0061\u0072\u0069\u0061\u006E\u0020\u0028\u0042\u0075\u006C\u0067\u0061\u0072\u0069\u0061\u0029", "\u0431\u044A\u043B\u0433\u0430\u0440\u0441\u043A\u0438\u0020\u0028\u0411\u044A\u043B\u0433\u0430\u0440\u0438\u044F\u0029", false, 0x0023, "BG", "BGR", "\u0042\u0075\u006C\u0067\u0061\u0072\u0069\u0061", "\u0411\u044A\u043B\u0433\u0430\u0440\u0438\u044F", "\u0042\u0047\u004C", "\u0042\u0075\u006C\u0067\u0061\u0072\u0069\u0061\u006E\u0020\u004C\u0065\u0076", "\u043B\u0432\u002E", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u043B\u0432", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0413\u0440\u0435\u0433\u043E\u0440\u0438\u0430\u043D\u0441\u043A\u0438\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'" ], [ "\u041D\u0434", "\u041F\u043D", "\u0412\u0442", "\u0421\u0440", "\u0427\u0442", "\u041F\u0442", "\u0421\u0431" ], [ "\u043D\u0435\u0434\u0435\u043B\u044F", "\u043F\u043E\u043D\u0435\u0434\u0435\u043B\u043D\u0438\u043A", "\u0432\u0442\u043E\u0440\u043D\u0438\u043A", "\u0441\u0440\u044F\u0434\u0430", "\u0447\u0435\u0442\u0432\u044A\u0440\u0442\u044A\u043A", "\u043F\u0435\u0442\u044A\u043A", "\u0441\u044A\u0431\u043E\u0442\u0430" ], [ "\u042F\u043D\u0443\u0430\u0440\u0438", "\u0424\u0435\u0432\u0440\u0443\u0430\u0440\u0438", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0438\u043B", "\u041C\u0430\u0439", "\u042E\u043D\u0438", "\u042E\u043B\u0438", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043F\u0442\u0435\u043C\u0432\u0440\u0438", "\u041E\u043A\u0442\u043E\u043C\u0432\u0440\u0438", "\u041D\u043E\u0435\u043C\u0432\u0440\u0438", "\u0414\u0435\u043A\u0435\u043C\u0432\u0440\u0438", "" ], [ "\u042F\u043D\u0443\u0430\u0440\u0438", "\u0424\u0435\u0432\u0440\u0443\u0430\u0440\u0438", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0438\u043B", "\u041C\u0430\u0439", "\u042E\u043D\u0438", "\u042E\u043B\u0438", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043F\u0442\u0435\u043C\u0432\u0440\u0438", "\u041E\u043A\u0442\u043E\u043C\u0432\u0440\u0438", "\u041D\u043E\u0435\u043C\u0432\u0440\u0438", "\u0414\u0435\u043A\u0435\u043C\u0432\u0440\u0438", "" ] },
{ 0x0403, 0x0003, "ca-ES", "ca", "cat", "ca-ES", "\u0043\u0061\u0074\u0061\u006C\u0061\u006E\u0020\u0028\u0043\u0061\u0074\u0061\u006C\u0061\u006E\u0029", "\u0063\u0061\u0074\u0061\u006C\u00E0\u0020\u0028\u0063\u0061\u0074\u0061\u006C\u00E0\u0029", false, 0x00D9, "ES", "ESP", "\u0053\u0070\u0061\u0069\u006E", "\u0045\u0073\u0070\u0061\u006E\u0079\u0061", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u00E0", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u002F\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u002F\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u002F\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u005C\'\'" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u005C\'\'\u0073\u0073\'\u005C\'\u005C\'\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u002F\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u002F\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u002F\'\u004D\u004D\u004D\u004D\'\u002F\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\'\u004D\u004D\u004D\u004D\'\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u002F\u0020\'\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u0067\u002E", "\u0064\u006C\u002E", "\u0064\u0074\u002E", "\u0064\u0063\u002E", "\u0064\u006A\u002E", "\u0064\u0076\u002E", "\u0064\u0073\u002E" ], [ "\u0064\u0069\u0075\u006D\u0065\u006E\u0067\u0065", "\u0064\u0069\u006C\u006C\u0075\u006E\u0073", "\u0064\u0069\u006D\u0061\u0072\u0074\u0073", "\u0064\u0069\u006D\u0065\u0063\u0072\u0065\u0073", "\u0064\u0069\u006A\u006F\u0075\u0073", "\u0064\u0069\u0076\u0065\u006E\u0064\u0072\u0065\u0073", "\u0064\u0069\u0073\u0073\u0061\u0062\u0074\u0065" ], [ "\u0067\u0065\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072\u00E7", "\u0061\u0062\u0072", "\u006D\u0061\u0069\u0067", "\u006A\u0075\u006E\u0079", "\u006A\u0075\u006C", "\u0061\u0067", "\u0073\u0065\u0074", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0073", "" ], [ "\u0067\u0065\u006E\u0065\u0072", "\u0066\u0065\u0062\u0072\u0065\u0072", "\u006D\u0061\u0072\u00E7", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0069\u0067", "\u006A\u0075\u006E\u0079", "\u006A\u0075\u006C\u0069\u006F\u006C", "\u0061\u0067\u006F\u0073\u0074", "\u0073\u0065\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u0065\u0073\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x0404, 0x7C04, "zh-TW", "zh", "zho", "zh-TW", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u0054\u0061\u0069\u0077\u0061\u006E\u0029", "\u4E2D\u6587\u0028\u53F0\u7063\u0029", false, 0x00ED, "TW", "TWN", "\u0054\u0061\u0069\u0077\u0061\u006E", "\u53F0\u7063", "\u0054\u0057\u0044", "\u004E\u0065\u0077\u0020\u0054\u0061\u0069\u0077\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u65B0\u53F0\u5E63", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u004E\u0054\u0024", "\u002D", "\u002B", "\u4E0D\u662F\u4E00\u500B\u6578\u5B57", "\u8CA0\u7121\u7AAE\u5927", "\u6B63\u7121\u7AAE\u5927", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2, 4 ], "\u897F\u66C6\u0020\u0028\u4E2D\u6587\u0029", 0, 0, "\u002F", "\u003A", "\u4E0A\u5348", "\u4E0B\u5348", "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'", [ "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\u004D\'\u6708\'\u0064\u0064\'\u65E5\'" ], [ "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ] },
{ 0x0405, 0x0005, "cs-CZ", "cs", "ces", "cs-CZ", "\u0043\u007A\u0065\u0063\u0068\u0020\u0028\u0043\u007A\u0065\u0063\u0068\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0029", "\u010D\u0065\u0161\u0074\u0069\u006E\u0061\u0020\u0028\u010C\u0065\u0073\u006B\u00E1\u00A0\u0072\u0065\u0070\u0075\u0062\u006C\u0069\u006B\u0061\u0029", false, 0x004B, "CZ", "CZE", "\u0043\u007A\u0065\u0063\u0068\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063", "\u010C\u0065\u0073\u006B\u00E1\u00A0\u0072\u0065\u0070\u0075\u0062\u006C\u0069\u006B\u0061", "\u0043\u005A\u004B", "\u0043\u007A\u0065\u0063\u0068\u0020\u004B\u006F\u0072\u0075\u006E\u0061", "\u004B\u006F\u0072\u0075\u006E\u0061\u0020\u010C\u0065\u0073\u006B\u00E1", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u004B\u010D", "\u002D", "\u002B", "\u004E\u0065\u006E\u00ED\u0020\u010D\u00ED\u0073\u006C\u006F", "\u002D\u006E\u0065\u006B\u006F\u006E\u0065\u010D\u006E\u006F", "\u002B\u006E\u0065\u006B\u006F\u006E\u0065\u010D\u006E\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u00E1\u006E\u0073\u006B\u00FD\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u00E1\u0159", 1, 0, "\u002E", "\u003A", "\u0064\u006F\u0070\u002E", "\u006F\u0064\u0070\u002E", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u002E\u006D\u006D\u0020\u0074\u0074", "\u0048\u002E\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u002D\u0064", "\u0079\u0079\u002D\u004D\u002D\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u002E\u006D\u006D\u002E\u0073\u0073\u0020\u0074\u0074", "\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065", "\u0070\u006F", "\u00FA\u0074", "\u0073\u0074", "\u010D\u0074", "\u0070\u00E1", "\u0073\u006F" ], [ "\u006E\u0065\u0064\u011B\u006C\u0065", "\u0070\u006F\u006E\u0064\u011B\u006C\u00ED", "\u00FA\u0074\u0065\u0072\u00FD", "\u0073\u0074\u0159\u0065\u0064\u0061", "\u010D\u0074\u0076\u0072\u0074\u0065\u006B", "\u0070\u00E1\u0074\u0065\u006B", "\u0073\u006F\u0062\u006F\u0074\u0061" ], [ "\u0049", "\u0049\u0049", "\u0049\u0049\u0049", "\u0049\u0056", "\u0056", "\u0056\u0049", "\u0056\u0049\u0049", "\u0056\u0049\u0049\u0049", "\u0049\u0058", "\u0058", "\u0058\u0049", "\u0058\u0049\u0049", "" ], [ "\u006C\u0065\u0064\u0065\u006E", "\u00FA\u006E\u006F\u0072", "\u0062\u0159\u0065\u007A\u0065\u006E", "\u0064\u0075\u0062\u0065\u006E", "\u006B\u0076\u011B\u0074\u0065\u006E", "\u010D\u0065\u0072\u0076\u0065\u006E", "\u010D\u0065\u0072\u0076\u0065\u006E\u0065\u0063", "\u0073\u0072\u0070\u0065\u006E", "\u007A\u00E1\u0159\u00ED", "\u0159\u00ED\u006A\u0065\u006E", "\u006C\u0069\u0073\u0074\u006F\u0070\u0061\u0064", "\u0070\u0072\u006F\u0073\u0069\u006E\u0065\u0063", "" ] },
{ 0x0406, 0x0006, "da-DK", "da", "dan", "da-DK", "\u0044\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0044\u0065\u006E\u006D\u0061\u0072\u006B\u0029", "\u0064\u0061\u006E\u0073\u006B\u0020\u0028\u0044\u0061\u006E\u006D\u0061\u0072\u006B\u0029", false, 0x003D, "DK", "DNK", "\u0044\u0065\u006E\u006D\u0061\u0072\u006B", "\u0044\u0061\u006E\u006D\u0061\u0072\u006B", "\u0044\u004B\u004B", "\u0044\u0061\u006E\u0069\u0073\u0068\u0020\u004B\u0072\u006F\u006E\u0065", "\u0044\u0061\u006E\u0073\u006B\u0020\u006B\u0072\u006F\u006E\u0065", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0044\u0065\u006E\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0065\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002D", "\u003A", "", "", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u00F8", "\u006D\u0061", "\u0074\u0069", "\u006F\u006E", "\u0074\u006F", "\u0066\u0072", "\u006C\u00F8" ], [ "\u0073\u00F8\u006E\u0064\u0061\u0067", "\u006D\u0061\u006E\u0064\u0061\u0067", "\u0074\u0069\u0072\u0073\u0064\u0061\u0067", "\u006F\u006E\u0073\u0064\u0061\u0067", "\u0074\u006F\u0072\u0073\u0064\u0061\u0067", "\u0066\u0072\u0065\u0064\u0061\u0067", "\u006C\u00F8\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0074\u0073", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0407, 0x0007, "de-DE", "de", "deu", "de-DE", "\u0047\u0065\u0072\u006D\u0061\u006E\u0020\u0028\u0047\u0065\u0072\u006D\u0061\u006E\u0079\u0029", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u0020\u0028\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u006C\u0061\u006E\u0064\u0029", false, 0x005E, "DE", "DEU", "\u0047\u0065\u0072\u006D\u0061\u006E\u0079", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u006C\u0061\u006E\u0064", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0045\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u006E\u002E\u0020\u0064\u0065\u0066\u002E", "\u002D\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", "\u002B\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0069\u0073\u0063\u0068\u0065\u0072\u0020\u004B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\u0020\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\u0020\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u006F", "\u004D\u006F", "\u0044\u0069", "\u004D\u0069", "\u0044\u006F", "\u0046\u0072", "\u0053\u0061" ], [ "\u0053\u006F\u006E\u006E\u0074\u0061\u0067", "\u004D\u006F\u006E\u0074\u0061\u0067", "\u0044\u0069\u0065\u006E\u0073\u0074\u0061\u0067", "\u004D\u0069\u0074\u0074\u0077\u006F\u0063\u0068", "\u0044\u006F\u006E\u006E\u0065\u0072\u0073\u0074\u0061\u0067", "\u0046\u0072\u0065\u0069\u0074\u0061\u0067", "\u0053\u0061\u006D\u0073\u0074\u0061\u0067" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0072\u007A", "\u0041\u0070\u0072", "\u004D\u0061\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u007A", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072", "\u004D\u00E4\u0072\u007A", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0069", "\u004A\u0075\u006E\u0069", "\u004A\u0075\u006C\u0069", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u007A\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0408, 0x0008, "el-GR", "el", "ell", "el-GR", "\u0047\u0072\u0065\u0065\u006B\u0020\u0028\u0047\u0072\u0065\u0065\u0063\u0065\u0029", "\u03B5\u03BB\u03BB\u03B7\u03BD\u03B9\u03BA\u03AC\u0020\u0028\u0395\u03BB\u03BB\u03AC\u03B4\u03B1\u0029", false, 0x0062, "GR", "GRC", "\u0047\u0072\u0065\u0065\u0063\u0065", "\u0395\u03BB\u03BB\u03AC\u03B4\u03B1", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u03B5\u03C5\u03C1\u03CE", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u03BC\u03B7\u00A0\u03B1\u03C1\u03B9\u03B8\u03BC\u03CC\u03C2", "\u002D\u0386\u03C0\u03B5\u03B9\u03C1\u03BF", "\u0386\u03C0\u03B5\u03B9\u03C1\u03BF", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0393\u03C1\u03B7\u03B3\u03BF\u03C1\u03B9\u03B1\u03BD\u03CC\u0020\u0397\u03BC\u03B5\u03C1\u03BF\u03BB\u03CC\u03B3\u03B9\u03BF", 1, 0, "\u002F", "\u003A", "\u03C0\u03BC", "\u03BC\u03BC", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u039A\u03C5\u03C1", "\u0394\u03B5\u03C5", "\u03A4\u03C1\u03B9", "\u03A4\u03B5\u03C4", "\u03A0\u03B5\u03BC", "\u03A0\u03B1\u03C1", "\u03A3\u03B1\u03B2" ], [ "\u039A\u03C5\u03C1\u03B9\u03B1\u03BA\u03AE", "\u0394\u03B5\u03C5\u03C4\u03AD\u03C1\u03B1", "\u03A4\u03C1\u03AF\u03C4\u03B7", "\u03A4\u03B5\u03C4\u03AC\u03C1\u03C4\u03B7", "\u03A0\u03AD\u03BC\u03C0\u03C4\u03B7", "\u03A0\u03B1\u03C1\u03B1\u03C3\u03BA\u03B5\u03C5\u03AE", "\u03A3\u03AC\u03B2\u03B2\u03B1\u03C4\u03BF" ], [ "\u0399\u03B1\u03BD", "\u03A6\u03B5\u03B2", "\u039C\u03B1\u03C1", "\u0391\u03C0\u03C1", "\u039C\u03B1\u03CA", "\u0399\u03BF\u03C5\u03BD", "\u0399\u03BF\u03C5\u03BB", "\u0391\u03C5\u03B3", "\u03A3\u03B5\u03C0", "\u039F\u03BA\u03C4", "\u039D\u03BF\u03B5", "\u0394\u03B5\u03BA", "" ], [ "\u0399\u03B1\u03BD\u03BF\u03C5\u03AC\u03C1\u03B9\u03BF\u03C2", "\u03A6\u03B5\u03B2\u03C1\u03BF\u03C5\u03AC\u03C1\u03B9\u03BF\u03C2", "\u039C\u03AC\u03C1\u03C4\u03B9\u03BF\u03C2", "\u0391\u03C0\u03C1\u03AF\u03BB\u03B9\u03BF\u03C2", "\u039C\u03AC\u03B9\u03BF\u03C2", "\u0399\u03BF\u03CD\u03BD\u03B9\u03BF\u03C2", "\u0399\u03BF\u03CD\u03BB\u03B9\u03BF\u03C2", "\u0391\u03CD\u03B3\u03BF\u03C5\u03C3\u03C4\u03BF\u03C2", "\u03A3\u03B5\u03C0\u03C4\u03AD\u03BC\u03B2\u03C1\u03B9\u03BF\u03C2", "\u039F\u03BA\u03C4\u03CE\u03B2\u03C1\u03B9\u03BF\u03C2", "\u039D\u03BF\u03AD\u03BC\u03B2\u03C1\u03B9\u03BF\u03C2", "\u0394\u03B5\u03BA\u03AD\u03BC\u03B2\u03C1\u03B9\u03BF\u03C2", "" ] },
{ 0x0409, 0x0009, "en-US", "en", "eng", "en-US", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u0053\u0074\u0061\u0074\u0065\u0073\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u0053\u0074\u0061\u0074\u0065\u0073\u0029", false, 0x00F4, "US", "USA", "\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u0053\u0074\u0061\u0074\u0065\u0073", "\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u0053\u0074\u0061\u0074\u0065\u0073", "\u0055\u0053\u0044", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", false, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u002F\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x040B, 0x000B, "fi-FI", "fi", "fin", "fi-FI", "\u0046\u0069\u006E\u006E\u0069\u0073\u0068\u0020\u0028\u0046\u0069\u006E\u006C\u0061\u006E\u0064\u0029", "\u0073\u0075\u006F\u006D\u0069\u0020\u0028\u0053\u0075\u006F\u006D\u0069\u0029", false, 0x004D, "FI", "FIN", "\u0046\u0069\u006E\u006C\u0061\u006E\u0064", "\u0053\u0075\u006F\u006D\u0069", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0069\u006E\u0065\u006E\u0020\u006B\u0061\u006C\u0065\u006E\u0074\u0065\u0072\u0069", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\'\u0074\u0061\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\'\u0074\u0061\'", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\'\u0074\u0061\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\'\u0074\u0061\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u0075", "\u006D\u0061", "\u0074\u0069", "\u006B\u0065", "\u0074\u006F", "\u0070\u0065", "\u006C\u0061" ], [ "\u0073\u0075\u006E\u006E\u0075\u006E\u0074\u0061\u0069", "\u006D\u0061\u0061\u006E\u0061\u006E\u0074\u0061\u0069", "\u0074\u0069\u0069\u0073\u0074\u0061\u0069", "\u006B\u0065\u0073\u006B\u0069\u0076\u0069\u0069\u006B\u006B\u006F", "\u0074\u006F\u0072\u0073\u0074\u0061\u0069", "\u0070\u0065\u0072\u006A\u0061\u006E\u0074\u0061\u0069", "\u006C\u0061\u0075\u0061\u006E\u0074\u0061\u0069" ], [ "\u0074\u0061\u006D\u006D\u0069", "\u0068\u0065\u006C\u006D\u0069", "\u006D\u0061\u0061\u006C\u0069\u0073", "\u0068\u0075\u0068\u0074\u0069", "\u0074\u006F\u0075\u006B\u006F", "\u006B\u0065\u0073\u00E4", "\u0068\u0065\u0069\u006E\u00E4", "\u0065\u006C\u006F", "\u0073\u0079\u0079\u0073", "\u006C\u006F\u006B\u0061", "\u006D\u0061\u0072\u0072\u0061\u0073", "\u006A\u006F\u0075\u006C\u0075", "" ], [ "\u0074\u0061\u006D\u006D\u0069\u006B\u0075\u0075", "\u0068\u0065\u006C\u006D\u0069\u006B\u0075\u0075", "\u006D\u0061\u0061\u006C\u0069\u0073\u006B\u0075\u0075", "\u0068\u0075\u0068\u0074\u0069\u006B\u0075\u0075", "\u0074\u006F\u0075\u006B\u006F\u006B\u0075\u0075", "\u006B\u0065\u0073\u00E4\u006B\u0075\u0075", "\u0068\u0065\u0069\u006E\u00E4\u006B\u0075\u0075", "\u0065\u006C\u006F\u006B\u0075\u0075", "\u0073\u0079\u0079\u0073\u006B\u0075\u0075", "\u006C\u006F\u006B\u0061\u006B\u0075\u0075", "\u006D\u0061\u0072\u0072\u0061\u0073\u006B\u0075\u0075", "\u006A\u006F\u0075\u006C\u0075\u006B\u0075\u0075", "" ] },
{ 0x040C, 0x000C, "fr-FR", "fr", "fra", "fr-FR", "\u0046\u0072\u0065\u006E\u0063\u0068\u0020\u0028\u0046\u0072\u0061\u006E\u0063\u0065\u0029", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073\u0020\u0028\u0046\u0072\u0061\u006E\u0063\u0065\u0029", false, 0x0054, "FR", "FRA", "\u0046\u0072\u0061\u006E\u0063\u0065", "\u0046\u0072\u0061\u006E\u0063\u0065", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u004E\u0075\u006D\u00E9\u0072\u0069\u0071\u0075\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0069\u0065\u0072\u0020\u0067\u0072\u00E9\u0067\u006F\u0072\u0069\u0065\u006E", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u0069\u006D\u002E", "\u006C\u0075\u006E\u002E", "\u006D\u0061\u0072\u002E", "\u006D\u0065\u0072\u002E", "\u006A\u0065\u0075\u002E", "\u0076\u0065\u006E\u002E", "\u0073\u0061\u006D\u002E" ], [ "\u0064\u0069\u006D\u0061\u006E\u0063\u0068\u0065", "\u006C\u0075\u006E\u0064\u0069", "\u006D\u0061\u0072\u0064\u0069", "\u006D\u0065\u0072\u0063\u0072\u0065\u0064\u0069", "\u006A\u0065\u0075\u0064\u0069", "\u0076\u0065\u006E\u0064\u0072\u0065\u0064\u0069", "\u0073\u0061\u006D\u0065\u0064\u0069" ], [ "\u006A\u0061\u006E\u0076\u002E", "\u0066\u00E9\u0076\u0072\u002E", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u002E", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u002E", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u00E9\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0076\u0069\u0065\u0072", "\u0066\u00E9\u0076\u0072\u0069\u0065\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u006C\u0065\u0074", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u00E9\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x040D, 0x000D, "he-IL", "he", "heb", "he-IL", "\u0048\u0065\u0062\u0072\u0065\u0077\u0020\u0028\u0049\u0073\u0072\u0061\u0065\u006C\u0029", "\u05E2\u05D1\u05E8\u05D9\u05EA\u0020\u0028\u05D9\u05E9\u05E8\u05D0\u05DC\u0029", false, 0x0075, "IL", "ISR", "\u0049\u0073\u0072\u0061\u0065\u006C", "\u05D9\u05E9\u05E8\u05D0\u05DC", "\u0049\u004C\u0053", "\u0049\u0073\u0072\u0061\u0065\u006C\u0069\u0020\u004E\u0065\u0077\u0020\u0053\u0068\u0065\u006B\u0065\u006C", "\u05E9\u05E7\u05DC\u0020\u05D7\u05D3\u05E9", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u20AA", "\u002D", "\u002B", "\u05DC\u05D0\u0020\u05DE\u05E1\u05E4\u05E8", "\u05D0\u05D9\u05E0\u05E1\u05D5\u05E3\u00A0\u05E9\u05DC\u05D9\u05DC\u05D9", "\u05D0\u05D9\u05E0\u05E1\u05D5\u05E3\u00A0\u05D7\u05D9\u05D5\u05D1\u05D9", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 8 ], "\u05DC\u05D5\u05D7\u00A0\u05E9\u05E0\u05D4\u00A0\u05D2\u05E8\u05D2\u05D5\u05E8\u05D9\u05D0\u05E0\u05D9", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u0020\u05D1\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\u0064\u0020\u05D1\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0020\u0064\u0064\u0020\u05D1\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u05D1\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u05D9\u05D5\u05DD\u00A0\u05D0", "\u05D9\u05D5\u05DD\u00A0\u05D1", "\u05D9\u05D5\u05DD\u00A0\u05D2", "\u05D9\u05D5\u05DD\u00A0\u05D3", "\u05D9\u05D5\u05DD\u00A0\u05D4", "\u05D9\u05D5\u05DD\u00A0\u05D5", "\u05E9\u05D1\u05EA" ], [ "\u05D9\u05D5\u05DD\u00A0\u05E8\u05D0\u05E9\u05D5\u05DF", "\u05D9\u05D5\u05DD\u00A0\u05E9\u05E0\u05D9", "\u05D9\u05D5\u05DD\u00A0\u05E9\u05DC\u05D9\u05E9\u05D9", "\u05D9\u05D5\u05DD\u00A0\u05E8\u05D1\u05D9\u05E2\u05D9", "\u05D9\u05D5\u05DD\u00A0\u05D7\u05DE\u05D9\u05E9\u05D9", "\u05D9\u05D5\u05DD\u00A0\u05E9\u05D9\u05E9\u05D9", "\u05E9\u05D1\u05EA" ], [ "\u05D9\u05E0\u05D5", "\u05E4\u05D1\u05E8", "\u05DE\u05E8\u05E5", "\u05D0\u05E4\u05E8", "\u05DE\u05D0\u05D9", "\u05D9\u05D5\u05E0", "\u05D9\u05D5\u05DC", "\u05D0\u05D5\u05D2", "\u05E1\u05E4\u05D8", "\u05D0\u05D5\u05E7", "\u05E0\u05D5\u05D1", "\u05D3\u05E6\u05DE", "" ], [ "\u05D9\u05E0\u05D5\u05D0\u05E8", "\u05E4\u05D1\u05E8\u05D5\u05D0\u05E8", "\u05DE\u05E8\u05E5", "\u05D0\u05E4\u05E8\u05D9\u05DC", "\u05DE\u05D0\u05D9", "\u05D9\u05D5\u05E0\u05D9", "\u05D9\u05D5\u05DC\u05D9", "\u05D0\u05D5\u05D2\u05D5\u05E1\u05D8", "\u05E1\u05E4\u05D8\u05DE\u05D1\u05E8", "\u05D0\u05D5\u05E7\u05D8\u05D5\u05D1\u05E8", "\u05E0\u05D5\u05D1\u05DE\u05D1\u05E8", "\u05D3\u05E6\u05DE\u05D1\u05E8", "" ] },
{ 0x040E, 0x000E, "hu-HU", "hu", "hun", "hu-HU", "\u0048\u0075\u006E\u0067\u0061\u0072\u0069\u0061\u006E\u0020\u0028\u0048\u0075\u006E\u0067\u0061\u0072\u0079\u0029", "\u006D\u0061\u0067\u0079\u0061\u0072\u0020\u0028\u004D\u0061\u0067\u0079\u0061\u0072\u006F\u0072\u0073\u007A\u00E1\u0067\u0029", false, 0x006D, "HU", "HUN", "\u0048\u0075\u006E\u0067\u0061\u0072\u0079", "\u004D\u0061\u0067\u0079\u0061\u0072\u006F\u0072\u0073\u007A\u00E1\u0067", "\u0048\u0055\u0046", "\u0048\u0075\u006E\u0067\u0061\u0072\u0069\u0061\u006E\u0020\u0046\u006F\u0072\u0069\u006E\u0074", "\u0066\u006F\u0072\u0069\u006E\u0074", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0046\u0074", "\u002D", "\u002B", "\u006E\u0065\u006D\u0020\u0073\u007A\u00E1\u006D", "\u006E\u0065\u0067\u0061\u0074\u00ED\u0076\u0020\u0076\u00E9\u0067\u0074\u0065\u006C\u0065\u006E", "\u0076\u00E9\u0067\u0074\u0065\u006C\u0065\u006E", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0065\u0072\u0067\u0065\u006C\u0079\u002D\u006E\u0061\u0070\u0074\u00E1\u0072", 1, 0, "\u002E\u0020", "\u003A", "\u0064\u0065\u002E", "\u0064\u0075\u002E", "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u002E\u0020\u0064\u0064\u002E", "\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u002E", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u002E", "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u004D\u004D", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u002E\u0020\u0064\u0064\u002E", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u002E" ], [ "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u004D\u004D" ], [ "\u0056", "\u0048", "\u004B", "\u0053\u007A\u0065", "\u0043\u0073", "\u0050", "\u0053\u007A\u006F" ], [ "\u0076\u0061\u0073\u00E1\u0072\u006E\u0061\u0070", "\u0068\u00E9\u0074\u0066\u0151", "\u006B\u0065\u0064\u0064", "\u0073\u007A\u0065\u0072\u0064\u0061", "\u0063\u0073\u00FC\u0074\u00F6\u0072\u0074\u00F6\u006B", "\u0070\u00E9\u006E\u0074\u0065\u006B", "\u0073\u007A\u006F\u006D\u0062\u0061\u0074" ], [ "\u006A\u0061\u006E\u002E", "\u0066\u0065\u0062\u0072\u002E", "\u006D\u00E1\u0072\u0063\u002E", "\u00E1\u0070\u0072\u002E", "\u006D\u00E1\u006A\u002E", "\u006A\u00FA\u006E\u002E", "\u006A\u00FA\u006C\u002E", "\u0061\u0075\u0067\u002E", "\u0073\u007A\u0065\u0070\u0074\u002E", "\u006F\u006B\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u0065\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0075\u00E1\u0072", "\u0066\u0065\u0062\u0072\u0075\u00E1\u0072", "\u006D\u00E1\u0072\u0063\u0069\u0075\u0073", "\u00E1\u0070\u0072\u0069\u006C\u0069\u0073", "\u006D\u00E1\u006A\u0075\u0073", "\u006A\u00FA\u006E\u0069\u0075\u0073", "\u006A\u00FA\u006C\u0069\u0075\u0073", "\u0061\u0075\u0067\u0075\u0073\u007A\u0074\u0075\u0073", "\u0073\u007A\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u00F3\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x040F, 0x000F, "is-IS", "is", "isl", "is-IS", "\u0049\u0063\u0065\u006C\u0061\u006E\u0064\u0069\u0063\u0020\u0028\u0049\u0063\u0065\u006C\u0061\u006E\u0064\u0029", "\u00ED\u0073\u006C\u0065\u006E\u0073\u006B\u0061\u0020\u0028\u00CD\u0073\u006C\u0061\u006E\u0064\u0029", false, 0x006E, "IS", "ISL", "\u0049\u0063\u0065\u006C\u0061\u006E\u0064", "\u00CD\u0073\u006C\u0061\u006E\u0064", "\u0049\u0053\u004B", "\u0049\u0063\u0065\u006C\u0061\u006E\u0064\u0069\u0063\u0020\u004B\u0072\u006F\u006E\u0061", "\u004B\u0072\u00F3\u006E\u0061", true, 2, 1, 0, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u006B\u0072\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u00ED\u0073\u006B\u0074\u0020\u0074\u00ED\u006D\u0061\u0074\u0061\u006C", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079\u002E", "\u0064\u002E\u0020\u004D\u002E\u0020\'\u005C\'\'\u0079\u0079\u002E", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u0075\u006E\u002E", "\u006D\u00E1\u006E\u002E", "\u00FE\u0072\u0069\u002E", "\u006D\u0069\u00F0\u002E", "\u0066\u0069\u006D\u002E", "\u0066\u00F6\u0073\u002E", "\u006C\u0061\u0075\u002E" ], [ "\u0073\u0075\u006E\u006E\u0075\u0064\u0061\u0067\u0075\u0072", "\u006D\u00E1\u006E\u0075\u0064\u0061\u0067\u0075\u0072", "\u00FE\u0072\u0069\u00F0\u006A\u0075\u0064\u0061\u0067\u0075\u0072", "\u006D\u0069\u00F0\u0076\u0069\u006B\u0075\u0064\u0061\u0067\u0075\u0072", "\u0066\u0069\u006D\u006D\u0074\u0075\u0064\u0061\u0067\u0075\u0072", "\u0066\u00F6\u0073\u0074\u0075\u0064\u0061\u0067\u0075\u0072", "\u006C\u0061\u0075\u0067\u0061\u0072\u0064\u0061\u0067\u0075\u0072" ], [ "\u006A\u0061\u006E\u002E", "\u0066\u0065\u0062\u002E", "\u006D\u0061\u0072\u002E", "\u0061\u0070\u0072\u002E", "\u006D\u0061\u00ED", "\u006A\u00FA\u006E\u002E", "\u006A\u00FA\u006C\u002E", "\u00E1\u0067\u00FA\u002E", "\u0073\u0065\u0070\u002E", "\u006F\u006B\u0074\u002E", "\u006E\u00F3\u0076\u002E", "\u0064\u0065\u0073\u002E", "" ], [ "\u006A\u0061\u006E\u00FA\u0061\u0072", "\u0066\u0065\u0062\u0072\u00FA\u0061\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0070\u0072\u00ED\u006C", "\u006D\u0061\u00ED", "\u006A\u00FA\u006E\u00ED", "\u006A\u00FA\u006C\u00ED", "\u00E1\u0067\u00FA\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u00F3\u0062\u0065\u0072", "\u006E\u00F3\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0410, 0x0010, "it-IT", "it", "ita", "it-IT", "\u0049\u0074\u0061\u006C\u0069\u0061\u006E\u0020\u0028\u0049\u0074\u0061\u006C\u0079\u0029", "\u0069\u0074\u0061\u006C\u0069\u0061\u006E\u006F\u0020\u0028\u0049\u0074\u0061\u006C\u0069\u0061\u0029", false, 0x0076, "IT", "ITA", "\u0049\u0074\u0061\u006C\u0079", "\u0049\u0074\u0061\u006C\u0069\u0061", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 9, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u0075\u006E\u0020\u006E\u0075\u006D\u0065\u0072\u006F\u0020\u0072\u0065\u0061\u006C\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 2, "\u002F", "\u002E", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u002E\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u002E\u006D\u006D\u002E\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u002E\u006D\u006D\u002E\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073", "\u0048\u002E\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0065\u0072", "\u0067\u0069\u006F", "\u0076\u0065\u006E", "\u0073\u0061\u0062" ], [ "\u0064\u006F\u006D\u0065\u006E\u0069\u0063\u0061", "\u006C\u0075\u006E\u0065\u0064\u00EC", "\u006D\u0061\u0072\u0074\u0065\u0064\u00EC", "\u006D\u0065\u0072\u0063\u006F\u006C\u0065\u0064\u00EC", "\u0067\u0069\u006F\u0076\u0065\u0064\u00EC", "\u0076\u0065\u006E\u0065\u0072\u0064\u00EC", "\u0073\u0061\u0062\u0061\u0074\u006F" ], [ "\u0067\u0065\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u0067", "\u0067\u0069\u0075", "\u006C\u0075\u0067", "\u0061\u0067\u006F", "\u0073\u0065\u0074", "\u006F\u0074\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0067\u0065\u006E\u006E\u0061\u0069\u006F", "\u0066\u0065\u0062\u0062\u0072\u0061\u0069\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0070\u0072\u0069\u006C\u0065", "\u006D\u0061\u0067\u0067\u0069\u006F", "\u0067\u0069\u0075\u0067\u006E\u006F", "\u006C\u0075\u0067\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0074\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0074\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x0411, 0x0011, "ja-JP", "ja", "jpn", "ja-JP", "\u004A\u0061\u0070\u0061\u006E\u0065\u0073\u0065\u0020\u0028\u004A\u0061\u0070\u0061\u006E\u0029", "\u65E5\u672C\u8A9E\u0020\u0028\u65E5\u672C\u0029", false, 0x007A, "JP", "JPN", "\u004A\u0061\u0070\u0061\u006E", "\u65E5\u672C", "\u004A\u0050\u0059", "\u004A\u0061\u0070\u0061\u006E\u0065\u0073\u0065\u0020\u0059\u0065\u006E", "\u5186", true, 2, 1, 0, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u00A5", "\u002D", "\u002B", "\u004E\u0061\u004E\u0020\u0028\u975E\u6570\u5024\u0029", "\u002D\u221E", "\u002B\u221E", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 3, 2 ], "\u897F\u66A6\u0020\u0028\u65E5\u672C\u8A9E\u0029", 0, 0, "\u002F", "\u003A", "\u5348\u524D", "\u5348\u5F8C", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064\'\u0020\u0028\'\u0064\u0064\u0064\'\u0029\'", "\u0079\u0079\u002F\u004D\u002F\u0064\'\u0020\u0028\'\u0064\u0064\u0064\'\u0029\'", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064\'\u0020\u0028\'\u0064\u0064\u0064\'\u0029\'", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\u004D\'\u6708\'\u0064\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'\u0020\u0064\u0064\u0064\u0064", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\u004D\'\u6708\'\u0064\u0064\'\u65E5\'\u0020\u0064\u0064\u0064\u0064" ], [ "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'" ], [ "\u65E5", "\u6708", "\u706B", "\u6C34", "\u6728", "\u91D1", "\u571F" ], [ "\u65E5\u66DC\u65E5", "\u6708\u66DC\u65E5", "\u706B\u66DC\u65E5", "\u6C34\u66DC\u65E5", "\u6728\u66DC\u65E5", "\u91D1\u66DC\u65E5", "\u571F\u66DC\u65E5" ], [ "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039", "\u0031\u0030", "\u0031\u0031", "\u0031\u0032", "" ], [ "\u0031\u6708", "\u0032\u6708", "\u0033\u6708", "\u0034\u6708", "\u0035\u6708", "\u0036\u6708", "\u0037\u6708", "\u0038\u6708", "\u0039\u6708", "\u0031\u0030\u6708", "\u0031\u0031\u6708", "\u0031\u0032\u6708", "" ] },
{ 0x0412, 0x0012, "ko-KR", "ko", "kor", "ko-KR", "\u004B\u006F\u0072\u0065\u0061\u006E\u0020\u0028\u004B\u006F\u0072\u0065\u0061\u0029", "\uD55C\uAD6D\uC5B4\u0020\u0028\uB300\uD55C\uBBFC\uAD6D\u0029", false, 0x0086, "KR", "KOR", "\u004B\u006F\u0072\u0065\u0061", "\uB300\uD55C\uBBFC\uAD6D", "\u004B\u0052\u0057", "\u004B\u006F\u0072\u0065\u0061\u006E\u0020\u0057\u006F\u006E", "\uC6D0", true, 2, 1, 0, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u20A9", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 5, 2 ], "\uC11C\uAE30\u0020\u0028\uD55C\uAE00\u0029", 0, 0, "\u002D", "\u003A", "\uC624\uC804", "\uC624\uD6C4", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'\u0020\u0064\'\uC77C\'\u0020\u0064\u0064\u0064\u0064", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\'\uC6D4\'\u0020\u0064\'\uC77C\'", "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'", [ "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u002D\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u002D\u0064" ], [ "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'\u0020\u0064\'\uC77C\'\u0020\u0064\u0064\u0064\u0064", "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'\u0020\u0064\'\uC77C\'", "\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'\u0020\u0064\'\uC77C\'\u0020\u0064\u0064\u0064\u0064", "\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'\u0020\u0064\'\uC77C\'", "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\u004D\'\uC6D4\'\u0020\u0064\u0064\'\uC77C\'\u0020\u0064\u0064\u0064\u0064", "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\u004D\'\uC6D4\'\u0020\u0064\u0064\'\uC77C\'" ], [ "\u0079\u0079\u0079\u0079\'\uB144\'\u0020\u004D\'\uC6D4\'" ], [ "\uC77C", "\uC6D4", "\uD654", "\uC218", "\uBAA9", "\uAE08", "\uD1A0" ], [ "\uC77C\uC694\uC77C", "\uC6D4\uC694\uC77C", "\uD654\uC694\uC77C", "\uC218\uC694\uC77C", "\uBAA9\uC694\uC77C", "\uAE08\uC694\uC77C", "\uD1A0\uC694\uC77C" ], [ "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039", "\u0031\u0030", "\u0031\u0031", "\u0031\u0032", "" ], [ "\u0031\uC6D4", "\u0032\uC6D4", "\u0033\uC6D4", "\u0034\uC6D4", "\u0035\uC6D4", "\u0036\uC6D4", "\u0037\uC6D4", "\u0038\uC6D4", "\u0039\uC6D4", "\u0031\u0030\uC6D4", "\u0031\u0031\uC6D4", "\u0031\u0032\uC6D4", "" ] },
{ 0x0413, 0x0013, "nl-NL", "nl", "nld", "nl-NL", "\u0044\u0075\u0074\u0063\u0068\u0020\u0028\u004E\u0065\u0074\u0068\u0065\u0072\u006C\u0061\u006E\u0064\u0073\u0029", "\u004E\u0065\u0064\u0065\u0072\u006C\u0061\u006E\u0064\u0073\u0020\u0028\u004E\u0065\u0064\u0065\u0072\u006C\u0061\u006E\u0064\u0029", false, 0x00B0, "NL", "NLD", "\u004E\u0065\u0074\u0068\u0065\u0072\u006C\u0061\u006E\u0064\u0073", "\u004E\u0065\u0064\u0065\u0072\u006C\u0061\u006E\u0064", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0073\u0065\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002D", "\u003A", "", "", "\u0064\u002D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0075\u0075\u0072\'", "\u0048\u0048\u003A\u006D\u006D\'\u0020\u0075\u0075\u0072\'" ], [ "\u0064\u002D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073\'\u0020\u0075\u0075\u0072\'", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073\'\u0020\u0075\u0075\u0072\'" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u007A\u006F", "\u006D\u0061", "\u0064\u0069", "\u0077\u006F", "\u0064\u006F", "\u0076\u0072", "\u007A\u0061" ], [ "\u007A\u006F\u006E\u0064\u0061\u0067", "\u006D\u0061\u0061\u006E\u0064\u0061\u0067", "\u0064\u0069\u006E\u0073\u0064\u0061\u0067", "\u0077\u006F\u0065\u006E\u0073\u0064\u0061\u0067", "\u0064\u006F\u006E\u0064\u0065\u0072\u0064\u0061\u0067", "\u0076\u0072\u0069\u006A\u0064\u0061\u0067", "\u007A\u0061\u0074\u0065\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0072\u0074", "\u0061\u0070\u0072", "\u006D\u0065\u0069", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u006D\u0061\u0061\u0072\u0074", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0065\u0069", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074\u0075\u0073", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0414, 0x0014, "nb-NO", "nb", "nob", "nb-NO", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u002C\u0020\u0042\u006F\u006B\u006D\u00E5\u006C\u0020\u0028\u004E\u006F\u0072\u0077\u0061\u0079\u0029", "\u006E\u006F\u0072\u0073\u006B\u002C\u0020\u0062\u006F\u006B\u006D\u00E5\u006C\u0020\u0028\u004E\u006F\u0072\u0067\u0065\u0029", false, 0x00B1, "NO", "NOR", "\u004E\u006F\u0072\u0077\u0061\u0079", "\u004E\u006F\u0072\u0067\u0065", "\u004E\u004F\u004B", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u0020\u004B\u0072\u006F\u006E\u0065", "\u004E\u006F\u0072\u0073\u006B\u0020\u006B\u0072\u006F\u006E\u0065", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\'\u006B\u006C\u0020\'\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\'\u006B\u006C\u0020\'\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u00F8", "\u006D\u0061", "\u0074\u0069", "\u006F\u006E", "\u0074\u006F", "\u0066\u0072", "\u006C\u00F8" ], [ "\u0073\u00F8\u006E\u0064\u0061\u0067", "\u006D\u0061\u006E\u0064\u0061\u0067", "\u0074\u0069\u0072\u0073\u0064\u0061\u0067", "\u006F\u006E\u0073\u0064\u0061\u0067", "\u0074\u006F\u0072\u0073\u0064\u0061\u0067", "\u0066\u0072\u0065\u0064\u0061\u0067", "\u006C\u00F8\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u0069", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0073", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0415, 0x0015, "pl-PL", "pl", "pol", "pl-PL", "\u0050\u006F\u006C\u0069\u0073\u0068\u0020\u0028\u0050\u006F\u006C\u0061\u006E\u0064\u0029", "\u0070\u006F\u006C\u0073\u006B\u0069\u0020\u0028\u0050\u006F\u006C\u0073\u006B\u0061\u0029", false, 0x00BF, "PL", "POL", "\u0050\u006F\u006C\u0061\u006E\u0064", "\u0050\u006F\u006C\u0073\u006B\u0061", "\u0050\u004C\u004E", "\u0050\u006F\u006C\u0069\u0073\u0068\u0020\u005A\u006C\u006F\u0074\u0079", "\u005A\u0142\u006F\u0074\u0079", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u007A\u0142", "\u002D", "\u002B", "\u006E\u0069\u0065\u0020\u006A\u0065\u0073\u0074\u0020\u006C\u0069\u0063\u007A\u0062\u0105", "\u002D\u006E\u0069\u0065\u0073\u006B\u006F\u0144\u0063\u007A\u006F\u006E\u006F\u015B\u0107", "\u002B\u006E\u0069\u0065\u0073\u006B\u006F\u0144\u0063\u007A\u006F\u006E\u006F\u015B\u0107", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u004B\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u007A\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0144\u0073\u006B\u0069", 1, 2, "\u002D", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004E", "\u0050\u006E", "\u0057\u0074", "\u015A\u0072", "\u0043\u007A", "\u0050\u0074", "\u0053\u006F" ], [ "\u006E\u0069\u0065\u0064\u007A\u0069\u0065\u006C\u0061", "\u0070\u006F\u006E\u0069\u0065\u0064\u007A\u0069\u0061\u0142\u0065\u006B", "\u0077\u0074\u006F\u0072\u0065\u006B", "\u015B\u0072\u006F\u0064\u0061", "\u0063\u007A\u0077\u0061\u0072\u0074\u0065\u006B", "\u0070\u0069\u0105\u0074\u0065\u006B", "\u0073\u006F\u0062\u006F\u0074\u0061" ], [ "\u0073\u0074\u0079", "\u006C\u0075\u0074", "\u006D\u0061\u0072", "\u006B\u0077\u0069", "\u006D\u0061\u006A", "\u0063\u007A\u0065", "\u006C\u0069\u0070", "\u0073\u0069\u0065", "\u0077\u0072\u007A", "\u0070\u0061\u017A", "\u006C\u0069\u0073", "\u0067\u0072\u0075", "" ], [ "\u0073\u0074\u0079\u0063\u007A\u0065\u0144", "\u006C\u0075\u0074\u0079", "\u006D\u0061\u0072\u007A\u0065\u0063", "\u006B\u0077\u0069\u0065\u0063\u0069\u0065\u0144", "\u006D\u0061\u006A", "\u0063\u007A\u0065\u0072\u0077\u0069\u0065\u0063", "\u006C\u0069\u0070\u0069\u0065\u0063", "\u0073\u0069\u0065\u0072\u0070\u0069\u0065\u0144", "\u0077\u0072\u007A\u0065\u0073\u0069\u0065\u0144", "\u0070\u0061\u017A\u0064\u007A\u0069\u0065\u0072\u006E\u0069\u006B", "\u006C\u0069\u0073\u0074\u006F\u0070\u0061\u0064", "\u0067\u0072\u0075\u0064\u007A\u0069\u0065\u0144", "" ] },
{ 0x0416, 0x0016, "pt-BR", "pt", "por", "pt-BR", "\u0050\u006F\u0072\u0074\u0075\u0067\u0075\u0065\u0073\u0065\u0020\u0028\u0042\u0072\u0061\u007A\u0069\u006C\u0029", "\u0050\u006F\u0072\u0074\u0075\u0067\u0075\u00EA\u0073\u0020\u0028\u0042\u0072\u0061\u0073\u0069\u006C\u0029", false, 0x0020, "BR", "BRA", "\u0042\u0072\u0061\u007A\u0069\u006C", "\u0042\u0072\u0061\u0073\u0069\u006C", "\u0042\u0052\u004C", "\u0052\u0065\u0061\u006C", "\u0052\u0065\u0061\u006C", true, 2, 1, 2, 9, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0052\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E\u0020\u0028\u004E\u00E3\u006F\u0020\u00E9\u0020\u0075\u006D\u0020\u006E\u00FA\u006D\u0065\u0072\u006F\u0029", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0043\u0061\u006C\u0065\u006E\u0064\u00E1\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "", "", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u0073\u0065\u0067", "\u0074\u0065\u0072", "\u0071\u0075\u0061", "\u0071\u0075\u0069", "\u0073\u0065\u0078", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u0073\u0065\u0067\u0075\u006E\u0064\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0074\u0065\u0072\u00E7\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0071\u0075\u0061\u0072\u0074\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0071\u0075\u0069\u006E\u0074\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0073\u0065\u0078\u0074\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0076", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0069", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0074", "\u006F\u0075\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u007A", "" ], [ "\u006A\u0061\u006E\u0065\u0069\u0072\u006F", "\u0066\u0065\u0076\u0065\u0072\u0065\u0069\u0072\u006F", "\u006D\u0061\u0072\u00E7\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0069\u006F", "\u006A\u0075\u006E\u0068\u006F", "\u006A\u0075\u006C\u0068\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0074\u0065\u006D\u0062\u0072\u006F", "\u006F\u0075\u0074\u0075\u0062\u0072\u006F", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u006F", "\u0064\u0065\u007A\u0065\u006D\u0062\u0072\u006F", "" ] },
{ 0x0418, 0x0018, "ro-RO", "ro", "ron", "ro-RO", "\u0052\u006F\u006D\u0061\u006E\u0069\u0061\u006E\u0020\u0028\u0052\u006F\u006D\u0061\u006E\u0069\u0061\u0029", "\u0072\u006F\u006D\u00E2\u006E\u0103\u0020\u0028\u0052\u006F\u006D\u00E2\u006E\u0069\u0061\u0029", false, 0x00C8, "RO", "ROU", "\u0052\u006F\u006D\u0061\u006E\u0069\u0061", "\u0052\u006F\u006D\u00E2\u006E\u0069\u0061", "\u0052\u004F\u004C", "\u0052\u006F\u006D\u0061\u006E\u0069\u0061\u006E\u0020\u004C\u0065\u0075", "\u004C\u0065\u0075", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u006C\u0065\u0069", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0020\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0044", "\u004C", "\u004D\u0061", "\u004D\u0069", "\u004A", "\u0056", "\u0053" ], [ "\u0064\u0075\u006D\u0069\u006E\u0069\u0063\u0103", "\u006C\u0075\u006E\u0069", "\u006D\u0061\u0072\u0163\u0069", "\u006D\u0069\u0065\u0072\u0063\u0075\u0072\u0069", "\u006A\u006F\u0069", "\u0076\u0069\u006E\u0065\u0072\u0069", "\u0073\u00E2\u006D\u0062\u0103\u0074\u0103" ], [ "\u0069\u0061\u006E\u002E", "\u0066\u0065\u0062\u002E", "\u006D\u0061\u0072\u002E", "\u0061\u0070\u0072\u002E", "\u006D\u0061\u0069\u002E", "\u0069\u0075\u006E\u002E", "\u0069\u0075\u006C\u002E", "\u0061\u0075\u0067\u002E", "\u0073\u0065\u0070\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u0065\u0063\u002E", "" ], [ "\u0069\u0061\u006E\u0075\u0061\u0072\u0069\u0065", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072\u0069\u0065", "\u006D\u0061\u0072\u0074\u0069\u0065", "\u0061\u0070\u0072\u0069\u006C\u0069\u0065", "\u006D\u0061\u0069", "\u0069\u0075\u006E\u0069\u0065", "\u0069\u0075\u006C\u0069\u0065", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0069\u0065", "\u006F\u0063\u0074\u006F\u006D\u0062\u0072\u0069\u0065", "\u006E\u006F\u0069\u0065\u006D\u0062\u0072\u0069\u0065", "\u0064\u0065\u0063\u0065\u006D\u0062\u0072\u0069\u0065", "" ] },
{ 0x0419, 0x0019, "ru-RU", "ru", "rus", "ru-RU", "\u0052\u0075\u0073\u0073\u0069\u0061\u006E\u0020\u0028\u0052\u0075\u0073\u0073\u0069\u0061\u0029", "\u0440\u0443\u0441\u0441\u043A\u0438\u0439\u0020\u0028\u0420\u043E\u0441\u0441\u0438\u044F\u0029", false, 0x00CB, "RU", "RUS", "\u0052\u0075\u0073\u0073\u0069\u0061", "\u0420\u043E\u0441\u0441\u0438\u044F", "\u0052\u0055\u0052", "\u0052\u0075\u0073\u0073\u0069\u0061\u006E\u0020\u0052\u0075\u0062\u006C\u0065", "\u0440\u0443\u0431\u043B\u044C", true, 2, 1, 2, 5, 1, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0440\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0431\u0435\u0441\u043A\u043E\u043D\u0435\u0447\u043D\u043E\u0441\u0442\u044C", "\u0431\u0435\u0441\u043A\u043E\u043D\u0435\u0447\u043D\u043E\u0441\u0442\u044C", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0438\u0433\u043E\u0440\u0438\u0430\u043D\u0441\u043A\u0438\u0439\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440\u044C", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0433\u002E\'" ], [ "\u0412\u0441", "\u041F\u043D", "\u0412\u0442", "\u0421\u0440", "\u0427\u0442", "\u041F\u0442", "\u0421\u0431" ], [ "\u0432\u043E\u0441\u043A\u0440\u0435\u0441\u0435\u043D\u044C\u0435", "\u043F\u043E\u043D\u0435\u0434\u0435\u043B\u044C\u043D\u0438\u043A", "\u0432\u0442\u043E\u0440\u043D\u0438\u043A", "\u0441\u0440\u0435\u0434\u0430", "\u0447\u0435\u0442\u0432\u0435\u0440\u0433", "\u043F\u044F\u0442\u043D\u0438\u0446\u0430", "\u0441\u0443\u0431\u0431\u043E\u0442\u0430" ], [ "\u044F\u043D\u0432", "\u0444\u0435\u0432", "\u043C\u0430\u0440", "\u0430\u043F\u0440", "\u043C\u0430\u0439", "\u0438\u044E\u043D", "\u0438\u044E\u043B", "\u0430\u0432\u0433", "\u0441\u0435\u043D", "\u043E\u043A\u0442", "\u043D\u043E\u044F", "\u0434\u0435\u043A", "" ], [ "\u042F\u043D\u0432\u0430\u0440\u044C", "\u0424\u0435\u0432\u0440\u0430\u043B\u044C", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0435\u043B\u044C", "\u041C\u0430\u0439", "\u0418\u044E\u043D\u044C", "\u0418\u044E\u043B\u044C", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043D\u0442\u044F\u0431\u0440\u044C", "\u041E\u043A\u0442\u044F\u0431\u0440\u044C", "\u041D\u043E\u044F\u0431\u0440\u044C", "\u0414\u0435\u043A\u0430\u0431\u0440\u044C", "" ] },
{ 0x041A, 0x001A, "hr-HR", "hr", "hrv", "hr-HR", "\u0043\u0072\u006F\u0061\u0074\u0069\u0061\u006E\u0020\u0028\u0043\u0072\u006F\u0061\u0074\u0069\u0061\u0029", "\u0068\u0072\u0076\u0061\u0074\u0073\u006B\u0069\u0020\u0028\u0048\u0072\u0076\u0061\u0074\u0073\u006B\u0061\u0029", false, 0x006C, "HR", "HRV", "\u0043\u0072\u006F\u0061\u0074\u0069\u0061", "\u0048\u0072\u0076\u0061\u0074\u0073\u006B\u0061", "\u0048\u0052\u004B", "\u0043\u0072\u006F\u0061\u0074\u0069\u0061\u006E\u0020\u004B\u0075\u006E\u0061", "\u0068\u0072\u0076\u0061\u0074\u0073\u006B\u0061\u0020\u006B\u0075\u006E\u0061", true, 2, 2, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u006B\u006E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006A\u0061\u006E\u0073\u006B\u0069\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065\u0064", "\u0070\u006F\u006E", "\u0075\u0074\u006F", "\u0073\u0072\u0069", "\u010D\u0065\u0074", "\u0070\u0065\u0074", "\u0073\u0075\u0062" ], [ "\u006E\u0065\u0064\u006A\u0065\u006C\u006A\u0061", "\u0070\u006F\u006E\u0065\u0064\u006A\u0065\u006C\u006A\u0061\u006B", "\u0075\u0074\u006F\u0072\u0061\u006B", "\u0073\u0072\u0069\u006A\u0065\u0064\u0061", "\u010D\u0065\u0074\u0076\u0072\u0074\u0061\u006B", "\u0070\u0065\u0074\u0061\u006B", "\u0073\u0075\u0062\u006F\u0074\u0061" ], [ "\u0073\u0069\u006A", "\u0076\u006C\u006A", "\u006F\u017E\u0075", "\u0074\u0072\u0061", "\u0073\u0076\u0069", "\u006C\u0069\u0070", "\u0073\u0072\u0070", "\u006B\u006F\u006C", "\u0072\u0075\u006A", "\u006C\u0069\u0073", "\u0073\u0074\u0075", "\u0070\u0072\u006F", "" ], [ "\u0073\u0069\u006A\u0065\u010D\u0061\u006E\u006A", "\u0076\u0065\u006C\u006A\u0061\u010D\u0061", "\u006F\u017E\u0075\u006A\u0061\u006B", "\u0074\u0072\u0061\u0076\u0061\u006E\u006A", "\u0073\u0076\u0069\u0062\u0061\u006E\u006A", "\u006C\u0069\u0070\u0061\u006E\u006A", "\u0073\u0072\u0070\u0061\u006E\u006A", "\u006B\u006F\u006C\u006F\u0076\u006F\u007A", "\u0072\u0075\u006A\u0061\u006E", "\u006C\u0069\u0073\u0074\u006F\u0070\u0061\u0064", "\u0073\u0074\u0075\u0064\u0065\u006E\u0069", "\u0070\u0072\u006F\u0073\u0069\u006E\u0061\u0063", "" ] },
{ 0x041B, 0x001B, "sk-SK", "sk", "slk", "sk-SK", "\u0053\u006C\u006F\u0076\u0061\u006B\u0020\u0028\u0053\u006C\u006F\u0076\u0061\u006B\u0069\u0061\u0029", "\u0073\u006C\u006F\u0076\u0065\u006E\u010D\u0069\u006E\u0061\u0020\u0028\u0053\u006C\u006F\u0076\u0065\u006E\u0073\u006B\u00E1\u0020\u0072\u0065\u0070\u0075\u0062\u006C\u0069\u006B\u0061\u0029", false, 0x008F, "SK", "SVK", "\u0053\u006C\u006F\u0076\u0061\u006B\u0069\u0061", "\u0053\u006C\u006F\u0076\u0065\u006E\u0073\u006B\u00E1\u0020\u0072\u0065\u0070\u0075\u0062\u006C\u0069\u006B\u0061", "\u0053\u004B\u004B", "\u0053\u006C\u006F\u0076\u0061\u006B\u0020\u004B\u006F\u0072\u0075\u006E\u0061", "\u0053\u006C\u006F\u0076\u0065\u006E\u0073\u006B\u00E1\u0020\u006B\u006F\u0072\u0075\u006E\u0061", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0053\u006B", "\u002D", "\u002B", "\u004E\u0069\u0065\u0020\u006A\u0065\u0020\u010D\u00ED\u0073\u006C\u006F", "\u002D\u006E\u0065\u006B\u006F\u006E\u0065\u010D\u006E\u006F", "\u002B\u006E\u0065\u006B\u006F\u006E\u0065\u010D\u006E\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u00E1\u006E\u0073\u006B\u0079\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u00E1\u0072", 1, 0, "\u002E\u0020", "\u003A", "", "", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D" ], [ "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065", "\u0070\u006F", "\u0075\u0074", "\u0073\u0074", "\u0161\u0074", "\u0070\u0069", "\u0073\u006F" ], [ "\u006E\u0065\u0064\u0065\u013E\u0061", "\u0070\u006F\u006E\u0064\u0065\u006C\u006F\u006B", "\u0075\u0074\u006F\u0072\u006F\u006B", "\u0073\u0074\u0072\u0065\u0064\u0061", "\u0161\u0074\u0076\u0072\u0074\u006F\u006B", "\u0070\u0069\u0061\u0074\u006F\u006B", "\u0073\u006F\u0062\u006F\u0074\u0061" ], [ "\u0049", "\u0049\u0049", "\u0049\u0049\u0049", "\u0049\u0056", "\u0056", "\u0056\u0049", "\u0056\u0049\u0049", "\u0056\u0049\u0049\u0049", "\u0049\u0058", "\u0058", "\u0058\u0049", "\u0058\u0049\u0049", "" ], [ "\u006A\u0061\u006E\u0075\u00E1\u0072", "\u0066\u0065\u0062\u0072\u0075\u00E1\u0072", "\u006D\u0061\u0072\u0065\u0063", "\u0061\u0070\u0072\u00ED\u006C", "\u006D\u00E1\u006A", "\u006A\u00FA\u006E", "\u006A\u00FA\u006C", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u00F3\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x041C, 0x001C, "sq-AL", "sq", "sqi", "sq-AL", "\u0041\u006C\u0062\u0061\u006E\u0069\u0061\u006E\u0020\u0028\u0041\u006C\u0062\u0061\u006E\u0069\u0061\u0029", "\u0073\u0068\u0071\u0069\u0070\u0065\u0020\u0028\u0053\u0068\u0071\u0069\u0070\u00EB\u0072\u0069\u0061\u0029", false, 0x0006, "AL", "ALB", "\u0041\u006C\u0062\u0061\u006E\u0069\u0061", "\u0053\u0068\u0071\u0069\u0070\u00EB\u0072\u0069\u0061", "\u0041\u004C\u004C", "\u0041\u006C\u0062\u0061\u006E\u0069\u0061\u006E\u0020\u004C\u0065\u006B", "\u004C\u0065\u006B", true, 2, 1, 2, 5, 1, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u004C\u0065\u006B", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002D", "\u003A", "\u0050\u0044", "\u004D\u0044", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0068\u003A\u006D\u006D\u002E\u0074\u0074", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u002E\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D", [ "\u0068\u003A\u006D\u006D\u002E\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u002E\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D" ], [ "\u0044\u0069\u0065", "\u0048\u00EB\u006E", "\u004D\u0061\u0072", "\u004D\u00EB\u0072", "\u0045\u006E\u006A", "\u0050\u0072\u0065", "\u0053\u0068\u0074" ], [ "\u0065\u0020\u0064\u0069\u0065\u006C", "\u0065\u0020\u0068\u00EB\u006E\u00EB", "\u0065\u0020\u006D\u0061\u0072\u0074\u00EB", "\u0065\u0020\u006D\u00EB\u0072\u006B\u0075\u0072\u00EB", "\u0065\u0020\u0065\u006E\u006A\u0074\u0065", "\u0065\u0020\u0070\u0072\u0065\u006D\u0074\u0065", "\u0065\u0020\u0073\u0068\u0074\u0075\u006E\u00EB" ], [ "\u004A\u0061\u006E", "\u0053\u0068\u006B", "\u004D\u0061\u0072", "\u0050\u0072\u0069", "\u004D\u0061\u006A", "\u0051\u0065\u0072", "\u004B\u006F\u0072", "\u0047\u0073\u0068", "\u0053\u0068\u0074", "\u0054\u0065\u0074", "\u004E\u00EB\u006E", "\u0044\u0068\u006A", "" ], [ "\u006A\u0061\u006E\u0061\u0072", "\u0073\u0068\u006B\u0075\u0072\u0074", "\u006D\u0061\u0072\u0073", "\u0070\u0072\u0069\u006C\u006C", "\u006D\u0061\u006A", "\u0071\u0065\u0072\u0073\u0068\u006F\u0072", "\u006B\u006F\u0072\u0072\u0069\u006B", "\u0067\u0075\u0073\u0068\u0074", "\u0073\u0068\u0074\u0061\u0074\u006F\u0072", "\u0074\u0065\u0074\u006F\u0072", "\u006E\u00EB\u006E\u0074\u006F\u0072", "\u0064\u0068\u006A\u0065\u0074\u006F\u0072", "" ] },
{ 0x041D, 0x001D, "sv-SE", "sv", "swe", "sv-SE", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068\u0020\u0028\u0053\u0077\u0065\u0064\u0065\u006E\u0029", "\u0073\u0076\u0065\u006E\u0073\u006B\u0061\u0020\u0028\u0053\u0076\u0065\u0072\u0069\u0067\u0065\u0029", false, 0x00DD, "SE", "SWE", "\u0053\u0077\u0065\u0064\u0065\u006E", "\u0053\u0076\u0065\u0072\u0069\u0067\u0065", "\u0053\u0045\u004B", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068\u0020\u004B\u0072\u006F\u006E\u0061", "\u0053\u0076\u0065\u006E\u0073\u006B\u0020\u006B\u0072\u006F\u006E\u0061", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u002E", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002D", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D", "\'\u0064\u0065\u006E\u0020\'\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\'\u0064\u0065\u006E\u0020\'\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\'\u006B\u006C\u0020\'\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\'\u006B\u006C\u0020\'\u0048\u003A\u006D\u006D" ], [ "\'\u0064\u0065\u006E\u0020\'\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u00F6", "\u006D\u00E5", "\u0074\u0069", "\u006F\u006E", "\u0074\u006F", "\u0066\u0072", "\u006C\u00F6" ], [ "\u0073\u00F6\u006E\u0064\u0061\u0067", "\u006D\u00E5\u006E\u0064\u0061\u0067", "\u0074\u0069\u0073\u0064\u0061\u0067", "\u006F\u006E\u0073\u0064\u0061\u0067", "\u0074\u006F\u0072\u0073\u0064\u0061\u0067", "\u0066\u0072\u0065\u0064\u0061\u0067", "\u006C\u00F6\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u006D\u0061\u0072\u0073", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074\u0069", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x041E, 0x001E, "th-TH", "th", "tha", "th-TH", "\u0054\u0068\u0061\u0069\u0020\u0028\u0054\u0068\u0061\u0069\u006C\u0061\u006E\u0064\u0029", "\u0E44\u0E17\u0E22\u0020\u0028\u0E44\u0E17\u0E22\u0029", false, 0x00E3, "TH", "THA", "\u0054\u0068\u0061\u0069\u006C\u0061\u006E\u0064", "\u0E44\u0E17\u0E22", "\u0054\u0048\u0042", "\u0054\u0068\u0061\u0069\u0020\u0042\u0061\u0068\u0074", "\u0E1A\u0E32\u0E17", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0E3F", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0E50", "\u0E51", "\u0E52", "\u0E53", "\u0E54", "\u0E55", "\u0E56", "\u0E57", "\u0E58", "\u0E59" ], 7, [ 7, 1 ], "\u0E1E\u0E38\u0E17\u0E18\u0E28\u0E31\u0E01\u0E23\u0E32\u0E0A", 1, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\'\u0E27\u0E31\u0E19\'\u0064\u0064\u0064\u0064\'\u0E17\u0E35\u0E48\'\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0067\u0067\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0E2D\u0E32\u002E", "\u0E08\u002E", "\u0E2D\u002E", "\u0E1E\u002E", "\u0E1E\u0E24\u002E", "\u0E28\u002E", "\u0E2A\u002E" ], [ "\u0E2D\u0E32\u0E17\u0E34\u0E15\u0E22\u0E4C", "\u0E08\u0E31\u0E19\u0E17\u0E23\u0E4C", "\u0E2D\u0E31\u0E07\u0E04\u0E32\u0E23", "\u0E1E\u0E38\u0E18", "\u0E1E\u0E24\u0E2B\u0E31\u0E2A\u0E1A\u0E14\u0E35", "\u0E28\u0E38\u0E01\u0E23\u0E4C", "\u0E40\u0E2A\u0E32\u0E23\u0E4C" ], [ "\u0E21\u002E\u0E04\u002E", "\u0E01\u002E\u0E1E\u002E", "\u0E21\u0E35\u002E\u0E04\u002E", "\u0E40\u0E21\u002E\u0E22\u002E", "\u0E1E\u002E\u0E04\u002E", "\u0E21\u0E34\u002E\u0E22\u002E", "\u0E01\u002E\u0E04\u002E", "\u0E2A\u002E\u0E04\u002E", "\u0E01\u002E\u0E22\u002E", "\u0E15\u002E\u0E04\u002E", "\u0E1E\u002E\u0E22\u002E", "\u0E18\u002E\u0E04\u002E", "" ], [ "\u0E21\u0E01\u0E23\u0E32\u0E04\u0E21", "\u0E01\u0E38\u0E21\u0E20\u0E32\u0E1E\u0E31\u0E19\u0E18\u0E4C", "\u0E21\u0E35\u0E19\u0E32\u0E04\u0E21", "\u0E40\u0E21\u0E29\u0E32\u0E22\u0E19", "\u0E1E\u0E24\u0E29\u0E20\u0E32\u0E04\u0E21", "\u0E21\u0E34\u0E16\u0E38\u0E19\u0E32\u0E22\u0E19", "\u0E01\u0E23\u0E01\u0E0E\u0E32\u0E04\u0E21", "\u0E2A\u0E34\u0E07\u0E2B\u0E32\u0E04\u0E21", "\u0E01\u0E31\u0E19\u0E22\u0E32\u0E22\u0E19", "\u0E15\u0E38\u0E25\u0E32\u0E04\u0E21", "\u0E1E\u0E24\u0E28\u0E08\u0E34\u0E01\u0E32\u0E22\u0E19", "\u0E18\u0E31\u0E19\u0E27\u0E32\u0E04\u0E21", "" ] },
{ 0x041F, 0x001F, "tr-TR", "tr", "tur", "tr-TR", "\u0054\u0075\u0072\u006B\u0069\u0073\u0068\u0020\u0028\u0054\u0075\u0072\u006B\u0065\u0079\u0029", "\u0054\u00FC\u0072\u006B\u00E7\u0065\u0020\u0028\u0054\u00FC\u0072\u006B\u0069\u0079\u0065\u0029", false, 0x00EB, "TR", "TUR", "\u0054\u0075\u0072\u006B\u0065\u0079", "\u0054\u00FC\u0072\u006B\u0069\u0079\u0065", "\u0054\u0052\u0059", "\u004E\u0065\u0077\u0020\u0054\u0075\u0072\u006B\u0069\u0073\u0068\u0020\u004C\u0069\u0072\u0061", "\u0059\u0065\u006E\u0069\u0020\u0054\u00FC\u0072\u006B\u0020\u004C\u0069\u0072\u0061\u0073\u0131", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0059\u0054\u004C", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0079\u0065\u006E\u0020\u0054\u0061\u006B\u0076\u0069\u006D\u0069", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\u0064\u0064\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\u0064\u0064\u0064\u0064", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0050\u0061\u007A", "\u0050\u007A\u0074", "\u0053\u0061\u006C", "\u00C7\u0061\u0072", "\u0050\u0065\u0072", "\u0043\u0075\u006D", "\u0043\u006D\u0074" ], [ "\u0050\u0061\u007A\u0061\u0072", "\u0050\u0061\u007A\u0061\u0072\u0074\u0065\u0073\u0069", "\u0053\u0061\u006C\u0131", "\u00C7\u0061\u0072\u015F\u0061\u006D\u0062\u0061", "\u0050\u0065\u0072\u015F\u0065\u006D\u0062\u0065", "\u0043\u0075\u006D\u0061", "\u0043\u0075\u006D\u0061\u0072\u0074\u0065\u0073\u0069" ], [ "\u004F\u0063\u0061", "\u015E\u0075\u0062", "\u004D\u0061\u0072", "\u004E\u0069\u0073", "\u004D\u0061\u0079", "\u0048\u0061\u007A", "\u0054\u0065\u006D", "\u0041\u011F\u0075", "\u0045\u0079\u006C", "\u0045\u006B\u0069", "\u004B\u0061\u0073", "\u0041\u0072\u0061", "" ], [ "\u004F\u0063\u0061\u006B", "\u015E\u0075\u0062\u0061\u0074", "\u004D\u0061\u0072\u0074", "\u004E\u0069\u0073\u0061\u006E", "\u004D\u0061\u0079\u0131\u0073", "\u0048\u0061\u007A\u0069\u0072\u0061\u006E", "\u0054\u0065\u006D\u006D\u0075\u007A", "\u0041\u011F\u0075\u0073\u0074\u006F\u0073", "\u0045\u0079\u006C\u00FC\u006C", "\u0045\u006B\u0069\u006D", "\u004B\u0061\u0073\u0131\u006D", "\u0041\u0072\u0061\u006C\u0131\u006B", "" ] },
{ 0x0420, 0x0020, "ur-PK", "ur", "urd", "ur-PK", "\u0055\u0072\u0064\u0075\u0020\u0028\u0049\u0073\u006C\u0061\u006D\u0069\u0063\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u0050\u0061\u006B\u0069\u0073\u0074\u0061\u006E\u0029", "\u0627\u064F\u0631\u062F\u0648\u0020\u0028\u067E\u0627\u06A9\u0633\u062A\u0627\u0646\u0029", false, 0x00BE, "PK", "PAK", "\u0049\u0073\u006C\u0061\u006D\u0069\u0063\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u0050\u0061\u006B\u0069\u0073\u0074\u0061\u006E", "\u067E\u0627\u06A9\u0633\u062A\u0627\u0646", "\u0050\u004B\u0052", "\u0050\u0061\u006B\u0069\u0073\u0074\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0631\u0648\u067E\u064A\u0647", true, 2, 1, 2, 3, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052\u0073", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u06F0", "\u06F1", "\u06F2", "\u06F3", "\u06F4", "\u06F5", "\u06F6", "\u06F7", "\u06F8", "\u06F9" ], 1, [ 1, 2, 6 ], "\u0639\u064A\u0633\u0648\u0649\u00A0\u0633\u0627\u0644", 1, 1, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u062A\u0648\u0627\u0631", "\u067E\u064A\u0631", "\u0645\u0646\u06AF\u0644", "\u0628\u062F\u06BE", "\u062C\u0645\u0639\u0631\u0627\u062A", "\u062C\u0645\u0639\u0647", "\u0647\u0641\u062A\u0647" ], [ "\u0627\u062A\u0648\u0627\u0631", "\u067E\u064A\u0631", "\u0645\u0646\u06AF\u0644", "\u0628\u062F\u06BE", "\u062C\u0645\u0639\u0631\u0627\u062A", "\u062C\u0645\u0639\u0647", "\u0647\u0641\u062A\u0647" ], [ "\u062C\u0646\u0648\u0631\u0649", "\u0641\u0631\u0648\u0631\u0649", "\u0645\u0627\u0631\u0686", "\u0627\u067E\u0631\u064A\u0644", "\u0645\u0626", "\u062C\u0648\u0646", "\u062C\u0648\u0644\u0627\u0678", "\u0627\u06AF\u0633\u062A", "\u0633\u062A\u0645\u0628\u0631", "\u0627\u06A9\u062A\u0648\u0628\u0631", "\u0646\u0648\u0645\u0628\u0631", "\u062F\u0633\u0645\u0628\u0631", "" ], [ "\u062C\u0646\u0648\u0631\u0649", "\u0641\u0631\u0648\u0631\u0649", "\u0645\u0627\u0631\u0686", "\u0627\u067E\u0631\u064A\u0644", "\u0645\u0626", "\u062C\u0648\u0646", "\u062C\u0648\u0644\u0627\u0678", "\u0627\u06AF\u0633\u062A", "\u0633\u062A\u0645\u0628\u0631", "\u0627\u06A9\u062A\u0648\u0628\u0631", "\u0646\u0648\u0645\u0628\u0631", "\u062F\u0633\u0645\u0628\u0631", "" ] },
{ 0x0421, 0x0021, "id-ID", "id", "ind", "id-ID", "\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061\u006E\u0020\u0028\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061\u0029", "\u0042\u0061\u0068\u0061\u0073\u0061\u0020\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061\u0020\u0028\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061\u0029", false, 0x006F, "ID", "IDN", "\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061", "\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061", "\u0049\u0044\u0052", "\u0049\u006E\u0064\u006F\u006E\u0065\u0073\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0069\u0061\u0068", "\u0052\u0075\u0070\u0069\u0061\u0068", true, 2, 1, 0, 0, 0, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0052\u0070", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0020\u004D\u0061\u0073\u0065\u0068\u0069", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u0069\u006E\u0067\u0067\u0075", "\u0053\u0065\u006E", "\u0053\u0065\u006C", "\u0052\u0061\u0062\u0075", "\u004B\u0061\u006D\u0069\u0073", "\u004A\u0075\u006D\u0061\u0074", "\u0053\u0061\u0062\u0074\u0075" ], [ "\u004D\u0069\u006E\u0067\u0067\u0075", "\u0053\u0065\u006E\u0069\u006E", "\u0053\u0065\u006C\u0061\u0073\u0061", "\u0052\u0061\u0062\u0075", "\u004B\u0061\u006D\u0069\u0073", "\u004A\u0075\u006D\u0061\u0074", "\u0053\u0061\u0062\u0074\u0075" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0065\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0070", "\u0044\u0065\u0073", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u004D\u0061\u0072\u0065\u0074", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0065\u0069", "\u004A\u0075\u006E\u0069", "\u004A\u0075\u006C\u0069", "\u0041\u0067\u0075\u0073\u0074\u0075\u0073", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0070\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0422, 0x0022, "uk-UA", "uk", "ukr", "uk-UA", "\u0055\u006B\u0072\u0061\u0069\u006E\u0069\u0061\u006E\u0020\u0028\u0055\u006B\u0072\u0061\u0069\u006E\u0065\u0029", "\u0443\u043A\u0440\u0430\u0457\u043D\u044C\u0441\u043A\u0430\u0020\u0028\u0423\u043A\u0440\u0430\u0457\u043D\u0430\u0029", false, 0x00F1, "UA", "UKR", "\u0055\u006B\u0072\u0061\u0069\u006E\u0065", "\u0423\u043A\u0440\u0430\u0457\u043D\u0430", "\u0055\u0041\u0048", "\u0055\u006B\u0072\u0061\u0069\u006E\u0069\u0061\u006E\u0020\u0047\u0072\u0069\u0076\u006E\u0061", "\u0433\u0440\u0438\u0432\u043D\u044F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0433\u0440\u043D\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0431\u0435\u0437\u043C\u0435\u0436\u043D\u0456\u0441\u0442\u044C", "\u0431\u0435\u0437\u043C\u0435\u0436\u043D\u0456\u0441\u0442\u044C", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0413\u0440\u0438\u0433\u043E\u0440\u0456\u0430\u043D\u044C\u0441\u043A\u0438\u0439\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u0020\u0440\u002E\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u0020\u0440\u002E\'", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u0020\u0440\u002E\'" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u0020\u0440\u002E\'" ], [ "\u041D\u0434", "\u041F\u043D", "\u0412\u0442", "\u0421\u0440", "\u0427\u0442", "\u041F\u0442", "\u0421\u0431" ], [ "\u043D\u0435\u0434\u0456\u043B\u044F", "\u043F\u043E\u043D\u0435\u0434\u0456\u043B\u043E\u043A", "\u0432\u0456\u0432\u0442\u043E\u0440\u043E\u043A", "\u0441\u0435\u0440\u0435\u0434\u0430", "\u0447\u0435\u0442\u0432\u0435\u0440", "\u043F\u0027\u044F\u0442\u043D\u0438\u0446\u044F", "\u0441\u0443\u0431\u043E\u0442\u0430" ], [ "\u0421\u0456\u0447", "\u041B\u044E\u0442", "\u0411\u0435\u0440", "\u041A\u0432\u0456", "\u0422\u0440\u0430", "\u0427\u0435\u0440", "\u041B\u0438\u043F", "\u0421\u0435\u0440", "\u0412\u0435\u0440", "\u0416\u043E\u0432", "\u041B\u0438\u0441", "\u0413\u0440\u0443", "" ], [ "\u0421\u0456\u0447\u0435\u043D\u044C", "\u041B\u044E\u0442\u0438\u0439", "\u0411\u0435\u0440\u0435\u0437\u0435\u043D\u044C", "\u041A\u0432\u0456\u0442\u0435\u043D\u044C", "\u0422\u0440\u0430\u0432\u0435\u043D\u044C", "\u0427\u0435\u0440\u0432\u0435\u043D\u044C", "\u041B\u0438\u043F\u0435\u043D\u044C", "\u0421\u0435\u0440\u043F\u0435\u043D\u044C", "\u0412\u0435\u0440\u0435\u0441\u0435\u043D\u044C", "\u0416\u043E\u0432\u0442\u0435\u043D\u044C", "\u041B\u0438\u0441\u0442\u043E\u043F\u0430\u0434", "\u0413\u0440\u0443\u0434\u0435\u043D\u044C", "" ] },
{ 0x0423, 0x0023, "be-BY", "be", "bel", "be-BY", "\u0042\u0065\u006C\u0061\u0072\u0075\u0073\u0069\u0061\u006E\u0020\u0028\u0042\u0065\u006C\u0061\u0072\u0075\u0073\u0029", "\u0411\u0435\u043B\u0430\u0440\u0443\u0441\u043A\u0456\u0020\u0028\u0411\u0435\u043B\u0430\u0440\u0443\u0441\u044C\u0029", false, 0x001D, "BY", "BLR", "\u0042\u0065\u006C\u0061\u0072\u0075\u0073", "\u0411\u0435\u043B\u0430\u0440\u0443\u0441\u044C", "\u0042\u0059\u0042", "\u0042\u0065\u006C\u0061\u0072\u0075\u0073\u0069\u0061\u006E\u0020\u0052\u0075\u0062\u006C\u0065", "\u0440\u0443\u0431\u043B\u044C", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0440\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0438\u0433\u043E\u0440\u0438\u0430\u043D\u0441\u043A\u0438\u0439\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440\u044C", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u043D\u0434", "\u043F\u043D", "\u0430\u045E", "\u0441\u0440", "\u0447\u0446", "\u043F\u0442", "\u0441\u0431" ], [ "\u043D\u044F\u0434\u0437\u0435\u043B\u044F", "\u043F\u0430\u043D\u044F\u0434\u0437\u0435\u043B\u0430\u043A", "\u0430\u045E\u0442\u043E\u0440\u0430\u043A", "\u0441\u0435\u0440\u0430\u0434\u0430", "\u0447\u0430\u0446\u0432\u0435\u0440", "\u043F\u044F\u0442\u043D\u0456\u0446\u0430", "\u0441\u0443\u0431\u043E\u0442\u0430" ], [ "\u0421\u0442\u0443", "\u041B\u044E\u0442", "\u0421\u0430\u043A", "\u041A\u0440\u0430", "\u041C\u0430\u0439", "\u0427\u044D\u0440", "\u041B\u0456\u043F", "\u0416\u043D\u0456", "\u0412\u0435\u0440", "\u041A\u0430\u0441", "\u041B\u0456\u0441", "\u0421\u043D\u0435", "" ], [ "\u0421\u0442\u0443\u0434\u0437\u0435\u043D\u044C", "\u041B\u044E\u0442\u044B", "\u0421\u0430\u043A\u0430\u0432\u0456\u043A", "\u041A\u0440\u0430\u0441\u0430\u0432\u0456\u043A", "\u041C\u0430\u0439", "\u0427\u044D\u0440\u0432\u0435\u043D\u044C", "\u041B\u0456\u043F\u0435\u043D\u044C", "\u0416\u043D\u0456\u0432\u0435\u043D\u044C", "\u0412\u0435\u0440\u0430\u0441\u0435\u043D\u044C", "\u041A\u0430\u0441\u0442\u0440\u044B\u0447\u043D\u0456\u043A", "\u041B\u0456\u0441\u0442\u0430\u043F\u0430\u0434", "\u0421\u043D\u0435\u0436\u0430\u043D\u044C", "" ] },
{ 0x0424, 0x0024, "sl-SI", "sl", "slv", "sl-SI", "\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u0061\u006E\u0020\u0028\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u0061\u0029", "\u0073\u006C\u006F\u0076\u0065\u006E\u0073\u006B\u0069\u0020\u0028\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u006A\u0061\u0029", false, 0x00D4, "SI", "SVN", "\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u0061", "\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u006A\u0061", "\u0053\u0049\u0054", "\u0053\u006C\u006F\u0076\u0065\u006E\u0069\u0061\u006E\u0020\u0054\u006F\u006C\u0061\u0072", "\u0073\u006C\u006F\u0076\u0065\u006E\u0073\u006B\u0069\u0020\u0074\u006F\u006C\u0061\u0072", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0053\u0049\u0054", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u006E\u0065\u0073\u006B\u006F\u006E\u010D\u006E\u006F\u0073\u0074", "\u006E\u0065\u0073\u006B\u006F\u006E\u010D\u006E\u006F\u0073\u0074", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006A\u0061\u006E\u0073\u006B\u0069\u0020\u006B\u006F\u006C\u0065\u0064\u0061\u0072", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065\u0064", "\u0070\u006F\u006E", "\u0074\u006F\u0072", "\u0073\u0072\u0065", "\u010D\u0065\u0074", "\u0070\u0065\u0074", "\u0073\u006F\u0062" ], [ "\u006E\u0065\u0064\u0065\u006C\u006A\u0061", "\u0070\u006F\u006E\u0065\u0064\u0065\u006C\u006A\u0065\u006B", "\u0074\u006F\u0072\u0065\u006B", "\u0073\u0072\u0065\u0064\u0061", "\u010D\u0065\u0074\u0072\u0074\u0065\u006B", "\u0070\u0065\u0074\u0065\u006B", "\u0073\u006F\u0062\u006F\u0074\u0061" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0065\u0063", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E\u0069\u006A", "\u006A\u0075\u006C\u0069\u006A", "\u0061\u0076\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0425, 0x0025, "et-EE", "et", "est", "et-EE", "\u0045\u0073\u0074\u006F\u006E\u0069\u0061\u006E\u0020\u0028\u0045\u0073\u0074\u006F\u006E\u0069\u0061\u0029", "\u0065\u0065\u0073\u0074\u0069\u0020\u0028\u0045\u0065\u0073\u0074\u0069\u0029", false, 0x0046, "EE", "EST", "\u0045\u0073\u0074\u006F\u006E\u0069\u0061", "\u0045\u0065\u0073\u0074\u0069", "\u0045\u0045\u004B", "\u0045\u0073\u0074\u006F\u006E\u0069\u0061\u006E\u0020\u004B\u0072\u006F\u006F\u006E", "\u004B\u0072\u006F\u006F\u006E", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002E", "\u00A0", "\u006B\u0072", "\u002D", "\u002B", "\u0061\u0076\u0061\u006C\u0064\u0061\u006D\u0061\u0074\u0075", "\u006D\u0069\u0069\u006E\u0075\u0073\u006C\u00F5\u0070\u006D\u0061\u0074\u0075\u0073", "\u0070\u006C\u0075\u0073\u0073\u006C\u00F5\u0070\u006D\u0061\u0074\u0075\u0073", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0075\u0073\u0065\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "\u0045\u004C", "\u0050\u004C", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0061\u002E\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0061\u002E\'", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0061\u002E\'", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0061\u002E\'", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0061\u002E\'" ], [ "\u0050", "\u0045", "\u0054", "\u004B", "\u004E", "\u0052", "\u004C" ], [ "\u0070\u00FC\u0068\u0061\u0070\u00E4\u0065\u0076", "\u0065\u0073\u006D\u0061\u0073\u0070\u00E4\u0065\u0076", "\u0074\u0065\u0069\u0073\u0069\u0070\u00E4\u0065\u0076", "\u006B\u006F\u006C\u006D\u0061\u0070\u00E4\u0065\u0076", "\u006E\u0065\u006C\u006A\u0061\u0070\u00E4\u0065\u0076", "\u0072\u0065\u0065\u0064\u0065", "\u006C\u0061\u0075\u0070\u00E4\u0065\u0076" ], [ "\u006A\u0061\u0061\u006E", "\u0076\u0065\u0065\u0062\u0072", "\u006D\u00E4\u0072\u0074\u0073", "\u0061\u0070\u0072", "\u006D\u0061\u0069", "\u006A\u0075\u0075\u006E\u0069", "\u006A\u0075\u0075\u006C\u0069", "\u0061\u0075\u0067", "\u0073\u0065\u0070\u0074", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0074\u0073", "" ], [ "\u006A\u0061\u0061\u006E\u0075\u0061\u0072", "\u0076\u0065\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u00E4\u0072\u0074\u0073", "\u0061\u0070\u0072\u0069\u006C\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0075\u006E\u0069", "\u006A\u0075\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0074\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0426, 0x0026, "lv-LV", "lv", "lav", "lv-LV", "\u004C\u0061\u0074\u0076\u0069\u0061\u006E\u0020\u0028\u004C\u0061\u0074\u0076\u0069\u0061\u0029", "\u006C\u0061\u0074\u0076\u0069\u0065\u0161\u0075\u0020\u0028\u004C\u0061\u0074\u0076\u0069\u006A\u0061\u0029", false, 0x008C, "LV", "LVA", "\u004C\u0061\u0074\u0076\u0069\u0061", "\u004C\u0061\u0074\u0076\u0069\u006A\u0061", "\u004C\u0056\u004C", "\u004C\u0061\u0074\u0076\u0069\u0061\u006E\u0020\u004C\u0061\u0074\u0073", "\u004C\u0061\u0074\u0073", true, 2, 1, 2, 9, 2, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u004C\u0073", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0062\u0065\u007A\u0067\u0061\u006C\u012B\u0062\u0061", "\u0062\u0065\u007A\u0067\u0061\u006C\u012B\u0062\u0061", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0061\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0101\u0072\u0073", 1, 2, "\u002E", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064\u002E", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0067\u0061\u0064\u0061\u0020\'\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u004D\u004D", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064\u002E", "\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064\u002E", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079\'\u002E\u0020\u0067\u0061\u0064\u0061\u0020\'\u0064\u002E\u0020\u004D\u004D\u004D\u004D" ], [ "\u0079\u0079\u0079\u0079\u002E\u0020\u004D\u004D\u004D\u004D" ], [ "\u0053\u0076", "\u0050\u0072", "\u004F\u0074", "\u0054\u0072", "\u0043\u0065", "\u0050\u006B", "\u0053\u0065" ], [ "\u0073\u0076\u0113\u0074\u0064\u0069\u0065\u006E\u0061", "\u0070\u0069\u0072\u006D\u0064\u0069\u0065\u006E\u0061", "\u006F\u0074\u0072\u0064\u0069\u0065\u006E\u0061", "\u0074\u0072\u0065\u0161\u0064\u0069\u0065\u006E\u0061", "\u0063\u0065\u0074\u0075\u0072\u0074\u0064\u0069\u0065\u006E\u0061", "\u0070\u0069\u0065\u006B\u0074\u0064\u0069\u0065\u006E\u0061", "\u0073\u0065\u0073\u0074\u0064\u0069\u0065\u006E\u0061" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0069", "\u004A\u016B\u006E", "\u004A\u016B\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0076\u0101\u0072\u0069\u0073", "\u0066\u0065\u0062\u0072\u0075\u0101\u0072\u0069\u0073", "\u006D\u0061\u0072\u0074\u0073", "\u0061\u0070\u0072\u012B\u006C\u0069\u0073", "\u006D\u0061\u0069\u006A\u0073", "\u006A\u016B\u006E\u0069\u006A\u0073", "\u006A\u016B\u006C\u0069\u006A\u0073", "\u0061\u0075\u0067\u0075\u0073\u0074\u0073", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0069\u0073", "\u006F\u006B\u0074\u006F\u0062\u0072\u0069\u0073", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0069\u0073", "\u0064\u0065\u0063\u0065\u006D\u0062\u0072\u0069\u0073", "" ] },
{ 0x0427, 0x0027, "lt-LT", "lt", "lit", "lt-LT", "\u004C\u0069\u0074\u0068\u0075\u0061\u006E\u0069\u0061\u006E\u0020\u0028\u004C\u0069\u0074\u0068\u0075\u0061\u006E\u0069\u0061\u0029", "\u006C\u0069\u0065\u0074\u0075\u0076\u0069\u0173\u0020\u0028\u004C\u0069\u0065\u0074\u0075\u0076\u0061\u0029", false, 0x008D, "LT", "LTU", "\u004C\u0069\u0074\u0068\u0075\u0061\u006E\u0069\u0061", "\u004C\u0069\u0065\u0074\u0075\u0076\u0061", "\u004C\u0054\u004C", "\u004C\u0069\u0074\u0068\u0075\u0061\u006E\u0069\u0061\u006E\u0020\u004C\u0069\u0074\u0061\u0073", "\u004C\u0069\u0074\u0061\u0073", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u004C\u0074", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0062\u0065\u0067\u0061\u006C\u0079\u0062\u0117", "\u0062\u0065\u0067\u0061\u006C\u0079\u0062\u0117", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0069\u0067\u0061\u006C\u0069\u0061\u0075\u0073\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u006F\u0072\u0069\u0075\u0073", 1, 0, "\u002E", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\u0020\'\u006D\u002E\'\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0020\'\u0064\u002E\'", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0020\'\u0064\u002E\'", "\u0079\u0079\u0079\u0079\u0020\'\u006D\u002E\'\u0020\u004D\u004D\u004D\u004D", [ "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u006D\u002E\'\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0020\'\u0064\u002E\'" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u006D\u002E\'\u0020\u004D\u004D\u004D\u004D" ], [ "\u0053\u006B", "\u0050\u0072", "\u0041\u006E", "\u0054\u0072", "\u004B\u0074", "\u0050\u006E", "\u0160\u0074" ], [ "\u0073\u0065\u006B\u006D\u0061\u0064\u0069\u0065\u006E\u0069\u0073", "\u0070\u0069\u0072\u006D\u0061\u0064\u0069\u0065\u006E\u0069\u0073", "\u0061\u006E\u0074\u0072\u0061\u0064\u0069\u0065\u006E\u0069\u0073", "\u0074\u0072\u0065\u010D\u0069\u0061\u0064\u0069\u0065\u006E\u0069\u0073", "\u006B\u0065\u0074\u0076\u0069\u0072\u0074\u0061\u0064\u0069\u0065\u006E\u0069\u0073", "\u0070\u0065\u006E\u006B\u0074\u0061\u0064\u0069\u0065\u006E\u0069\u0073", "\u0161\u0065\u0161\u0074\u0061\u0064\u0069\u0065\u006E\u0069\u0073" ], [ "\u0053\u0061\u0075", "\u0056\u0061\u0073", "\u004B\u006F\u0076", "\u0042\u0061\u006C", "\u0047\u0065\u0067", "\u0042\u0069\u0072", "\u004C\u0069\u0065", "\u0052\u0067\u0070", "\u0052\u0067\u0073", "\u0053\u0070\u006C", "\u004C\u0061\u0070", "\u0047\u0072\u0064", "" ], [ "\u0073\u0061\u0075\u0073\u0069\u0073", "\u0076\u0061\u0073\u0061\u0072\u0069\u0073", "\u006B\u006F\u0076\u0061\u0073", "\u0062\u0061\u006C\u0061\u006E\u0064\u0069\u0073", "\u0067\u0065\u0067\u0075\u017E\u0117", "\u0062\u0069\u0072\u017E\u0065\u006C\u0069\u0073", "\u006C\u0069\u0065\u0070\u0061", "\u0072\u0075\u0067\u0070\u006A\u016B\u0074\u0069\u0073", "\u0072\u0075\u0067\u0073\u0117\u006A\u0069\u0073", "\u0073\u0070\u0061\u006C\u0069\u0073", "\u006C\u0061\u0070\u006B\u0072\u0069\u0074\u0069\u0073", "\u0067\u0072\u0075\u006F\u0064\u0069\u0073", "" ] },
{ 0x0429, 0x0029, "fa-IR", "fa", "fas", "fa-IR", "\u0050\u0065\u0072\u0073\u0069\u0061\u006E\u0020\u0028\u0049\u0072\u0061\u006E\u0029", "\u0641\u0627\u0631\u0633\u0649\u0020\u0028\u0627\u064A\u0631\u0627\u0646\u0029", false, 0x0074, "IR", "IRN", "\u0049\u0072\u0061\u006E", "\u0627\u064A\u0631\u0627\u0646", "\u0049\u0052\u0052", "\u0049\u0072\u0061\u006E\u0069\u0061\u006E\u0020\u0052\u0069\u0061\u006C", "\u0631\u0649\u0627\u0644", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002F", "\u002C", "\u0631\u064A\u0627\u0644", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u06F0", "\u06F1", "\u06F2", "\u06F3", "\u06F4", "\u06F5", "\u06F6", "\u06F7", "\u06F8", "\u06F9" ], 2, [ 2, 1, 6, 11 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0020\u0028\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0029", 0, 0, "\u002F", "\u003A", "\u0642\u002E\u0638", "\u0628\u002E\u0638", "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u002F\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x042A, 0x002A, "vi-VN", "vi", "vie", "vi-VN", "\u0056\u0069\u0065\u0074\u006E\u0061\u006D\u0065\u0073\u0065\u0020\u0028\u0056\u0069\u0065\u0074\u006E\u0061\u006D\u0029", "\u0054\u0069\u00EA\u0301\u006E\u0067\u0020\u0056\u0069\u00EA\u0323\u0074\u0020\u0028\u0056\u0069\u00EA\u0323\u0074\u0020\u004E\u0061\u006D\u0029", false, 0x00FB, "VN", "VNM", "\u0056\u0069\u0065\u0074\u006E\u0061\u006D", "\u0056\u0069\u00EA\u0323\u0074\u0020\u004E\u0061\u006D", "\u0056\u004E\u0044", "\u0056\u0069\u0065\u0074\u006E\u0061\u006D\u0065\u0073\u0065\u0020\u0044\u006F\u006E\u0067", "\u0110\u00F4\u0300\u006E\u0067", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AB", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0044\u01B0\u01A1\u006E\u0067\u00A0\u004C\u0069\u0323\u0063\u0068", 1, 0, "\u002F", "\u003A", "\u0053\u0041", "\u0043\u0048", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0043\u004E", "\u0048\u0061\u0069", "\u0042\u0061", "\u0054\u01B0", "\u004E\u0103\u006D", "\u0053\u0061\u0301\u0075", "\u0042\u0061\u0309\u0079" ], [ "\u0043\u0068\u0075\u0309\u0020\u004E\u0068\u00E2\u0323\u0074", "\u0054\u0068\u01B0\u0301\u0020\u0048\u0061\u0069", "\u0054\u0068\u01B0\u0301\u0020\u0042\u0061", "\u0054\u0068\u01B0\u0301\u0020\u0054\u01B0", "\u0054\u0068\u01B0\u0301\u0020\u004E\u0103\u006D", "\u0054\u0068\u01B0\u0301\u0020\u0053\u0061\u0301\u0075", "\u0054\u0068\u01B0\u0301\u0020\u0042\u0061\u0309\u0079" ], [ "\u0054\u0068\u0067\u0031", "\u0054\u0068\u0067\u0032", "\u0054\u0068\u0067\u0033", "\u0054\u0068\u0067\u0034", "\u0054\u0068\u0067\u0035", "\u0054\u0068\u0067\u0036", "\u0054\u0068\u0067\u0037", "\u0054\u0068\u0067\u0038", "\u0054\u0068\u0067\u0039", "\u0054\u0068\u0067\u0031\u0030", "\u0054\u0068\u0067\u0031\u0031", "\u0054\u0068\u0067\u0031\u0032", "" ], [ "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0047\u0069\u00EA\u006E\u0067", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0048\u0061\u0069", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0042\u0061", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0054\u01B0", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u004E\u0103\u006D", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0053\u0061\u0301\u0075", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0042\u0061\u0309\u0079", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0054\u0061\u0301\u006D", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u0043\u0068\u0069\u0301\u006E", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u004D\u01B0\u01A1\u0300\u0069", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u004D\u01B0\u01A1\u0300\u0069\u0020\u004D\u00F4\u0323\u0074", "\u0054\u0068\u0061\u0301\u006E\u0067\u0020\u004D\u01B0\u01A1\u0300\u0069\u0020\u0048\u0061\u0069", "" ] },
{ 0x042B, 0x002B, "hy-AM", "hy", "hye", "hy-AM", "\u0041\u0072\u006D\u0065\u006E\u0069\u0061\u006E\u0020\u0028\u0041\u0072\u006D\u0065\u006E\u0069\u0061\u0029", "\u0540\u0561\u0575\u0565\u0580\u0565\u0576\u0020\u0028\u0540\u0561\u0575\u0561\u057D\u057F\u0561\u0576\u0029", false, 0x0007, "AM", "ARM", "\u0041\u0072\u006D\u0065\u006E\u0069\u0061", "\u0540\u0561\u0575\u0561\u057D\u057F\u0561\u0576", "\u0041\u004D\u0044", "\u0041\u0072\u006D\u0065\u006E\u0069\u0061\u006E\u0020\u0044\u0072\u0061\u006D", "\u0564\u0580\u0561\u0574", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0564\u0580\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0555\u0580\u0561\u0581\u0578\u0582\u0575\u0581", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u002C\u0020\u0064\u002D\u004D\u004D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u002D\u004D\u004D\u004D\u004D\u002D\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u053F\u056B\u0580", "\u0535\u0580\u056F", "\u0535\u0580\u0584", "\u0549\u0580\u0584", "\u0540\u0576\u0563", "\u0548\u0552\u0580", "\u0547\u0562\u0569" ], [ "\u053F\u056B\u0580\u0561\u056F\u056B", "\u0535\u0580\u056F\u0578\u0582\u0577\u0561\u0562\u0569\u056B", "\u0535\u0580\u0565\u0584\u0577\u0561\u0562\u0569\u056B", "\u0549\u0578\u0580\u0565\u0584\u0577\u0561\u0562\u0569\u056B", "\u0540\u056B\u0576\u0563\u0577\u0561\u0562\u0569\u056B", "\u0548\u0552\u0580\u0562\u0561\u0569", "\u0547\u0561\u0562\u0561\u0569" ], [ "\u0540\u0546\u054E", "\u0553\u054F\u054E", "\u0544\u0550\u054F", "\u0531\u054A\u0550", "\u0544\u0545\u054D", "\u0540\u0546\u054D", "\u0540\u053C\u054D", "\u0555\u0533\u054D", "\u054D\u0535\u054A", "\u0540\u0548\u053F", "\u0546\u0548\u0545", "\u0534\u0535\u053F", "" ], [ "\u0540\u0578\u0582\u0576\u057E\u0561\u0580", "\u0553\u0565\u057F\u0580\u057E\u0561\u0580", "\u0544\u0561\u0580\u057F", "\u0531\u057A\u0580\u056B\u056C", "\u0544\u0561\u0575\u056B\u057D", "\u0540\u0578\u0582\u0576\u056B\u057D", "\u0540\u0578\u0582\u056C\u056B\u057D", "\u0555\u0563\u0578\u057D\u057F\u0578\u057D", "\u054D\u0565\u057A\u057F\u0565\u0574\u0562\u0565\u0580", "\u0540\u0578\u056F\u057F\u0565\u0574\u0562\u0565\u0580", "\u0546\u0578\u0575\u0565\u0574\u0562\u0565\u0580", "\u0534\u0565\u056F\u057F\u0565\u0574\u0562\u0565\u0580", "" ] },
{ 0x042C, 0x002C, "az-AZ-Latn", "az", "aze", "az-Latn-AZ", "\u0041\u007A\u0065\u0072\u0069\u0020\u0028\u004C\u0061\u0074\u0069\u006E\u002C\u0020\u0041\u007A\u0065\u0072\u0062\u0061\u0069\u006A\u0061\u006E\u0029", "\u0041\u007A\u0259\u0072\u0062\u0061\u0079\u0063\u0061\u006E\u00AD\u0131\u006C\u0131\u0020\u0028\u0041\u007A\u0259\u0072\u0062\u0061\u0079\u0063\u0061\u006E\u0063\u0061\u0029", false, 0x0005, "AZ", "AZE", "\u0041\u007A\u0065\u0072\u0062\u0061\u0069\u006A\u0061\u006E", "\u0041\u007A\u0259\u0072\u0062\u0061\u0079\u0063\u0061\u006E\u0063\u0061", "\u0041\u005A\u004D", "\u0041\u007A\u0065\u0072\u0062\u0061\u0069\u006A\u0061\u006E\u0069\u0061\u006E\u0020\u004D\u0061\u006E\u0061\u0074", "\u006D\u0061\u006E\u0061\u0074", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u006D\u0061\u006E\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0051\u0072\u0069\u0071\u006F\u0072\u0069\u0061\u006E", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0042", "\u0042\u0065", "\u00C7\u0061", "\u00C7", "\u0043\u0061", "\u0043", "\u015E" ], [ "\u0042\u0061\u007A\u0061\u0072", "\u0042\u0061\u007A\u0061\u0072\u00A0\u0065\u0072\u0074\u0259\u0073\u0069", "\u00C7\u0259\u0072\u015F\u0259\u006E\u0062\u0259\u00A0\u0061\u0078\u015F\u0061\u006D\u0131", "\u00C7\u0259\u0072\u015F\u0259\u006E\u0062\u0259", "\u0043\u00FC\u006D\u0259\u00A0\u0061\u0078\u015F\u0061\u006D\u0131", "\u0043\u00FC\u006D\u0259", "\u015E\u0259\u006E\u0062\u0259" ], [ "\u0059\u0061\u006E", "\u0046\u0065\u0076", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u0130\u0079\u0075\u006E", "\u0130\u0079\u0075\u006C", "\u0041\u0076\u0067", "\u0053\u0065\u006E", "\u004F\u006B\u0074", "\u004E\u006F\u0079", "\u0044\u0065\u006B", "" ], [ "\u0059\u0061\u006E\u0076\u0061\u0072", "\u0046\u0065\u0076\u0072\u0061\u006C", "\u004D\u0061\u0072\u0074", "\u0041\u0070\u0072\u0065\u006C", "\u004D\u0061\u0079", "\u0130\u0079\u0075\u006E", "\u0130\u0079\u0075\u006C", "\u0041\u0076\u0067\u0075\u0073\u0074", "\u0053\u0065\u006E\u0074\u0079\u0061\u0062\u0072", "\u004F\u006B\u0074\u0079\u0061\u0062\u0072", "\u004E\u006F\u0079\u0061\u0062\u0072", "\u0044\u0065\u006B\u0061\u0062\u0072", "" ] },
{ 0x042D, 0x002D, "eu-ES", "eu", "eus", "eu-ES", "\u0042\u0061\u0073\u0071\u0075\u0065\u0020\u0028\u0042\u0061\u0073\u0071\u0075\u0065\u0029", "\u0065\u0075\u0073\u006B\u0061\u0072\u0061\u0020\u0028\u0065\u0075\u0073\u006B\u0061\u0072\u0061\u0029", false, 0x00D9, "ES", "ESP", "\u0053\u0070\u0061\u0069\u006E", "\u0045\u0073\u0070\u0061\u0069\u006E\u0069\u0061", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u0045\u0064\u005A", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0075", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0075", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0065\u0067\u0075\u0074\u0065\u0067\u0069\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006F\u0074\u0061\u0072\u0072\u0061", 1, 0, "\u002F", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079\u002E\'\u0065\u006B\u006F\'\u0020\u004D\u004D\u004D\u004D\'\u006B\u0020\'\u0064", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u0079\u0079\u0079\u0079\u002E\'\u0065\u006B\u006F\'\u0020\u004D\u004D\u004D\u004D", [ "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079\u002E\'\u0065\u006B\u006F\'\u0020\u004D\u004D\u004D\u004D\'\u006B\u0020\'\u0064" ], [ "\u0079\u0079\u0079\u0079\u002E\'\u0065\u006B\u006F\'\u0020\u004D\u004D\u004D\u004D" ], [ "\u0069\u0067\u002E", "\u0061\u006C\u002E", "\u0061\u0073\u002E", "\u0061\u007A\u002E", "\u006F\u0067\u002E", "\u006F\u0072\u002E", "\u006C\u0072\u002E" ], [ "\u0069\u0067\u0061\u006E\u0064\u0065\u0061", "\u0061\u0073\u0074\u0065\u006C\u0065\u0068\u0065\u006E\u0061", "\u0061\u0073\u0074\u0065\u0061\u0072\u0074\u0065\u0061", "\u0061\u0073\u0074\u0065\u0061\u007A\u006B\u0065\u006E\u0061", "\u006F\u0073\u0074\u0065\u0067\u0075\u006E\u0061", "\u006F\u0073\u0074\u0069\u0072\u0061\u006C\u0061", "\u006C\u0061\u0072\u0075\u006E\u0062\u0061\u0074\u0061" ], [ "\u0075\u0072\u0074\u002E", "\u006F\u0074\u0073\u002E", "\u006D\u0061\u0072\u002E", "\u0061\u0070\u0069\u002E", "\u006D\u0061\u0069\u002E", "\u0065\u006B\u0061\u002E", "\u0075\u007A\u0074\u002E", "\u0061\u0062\u0075\u002E", "\u0069\u0072\u0061\u002E", "\u0075\u0072\u0072\u002E", "\u0061\u007A\u0061\u002E", "\u0061\u0062\u0065\u002E", "" ], [ "\u0075\u0072\u0074\u0061\u0072\u0072\u0069\u006C\u0061", "\u006F\u0074\u0073\u0061\u0069\u006C\u0061", "\u006D\u0061\u0072\u0074\u0078\u006F\u0061", "\u0061\u0070\u0069\u0072\u0069\u006C\u0061", "\u006D\u0061\u0069\u0061\u0074\u007A\u0061", "\u0065\u006B\u0061\u0069\u006E\u0061", "\u0075\u007A\u0074\u0061\u0069\u006C\u0061", "\u0061\u0062\u0075\u007A\u0074\u0075\u0061", "\u0069\u0072\u0061\u0069\u006C\u0061", "\u0075\u0072\u0072\u0069\u0061", "\u0061\u007A\u0061\u0072\u006F\u0061", "\u0061\u0062\u0065\u006E\u0064\u0075\u0061", "" ] },
{ 0x042F, 0x002F, "mk-MK", "mk", "mkd", "mk-MK", "\u004D\u0061\u0063\u0065\u0064\u006F\u006E\u0069\u0061\u006E\u0020\u0028\u0046\u006F\u0072\u006D\u0065\u0072\u0020\u0059\u0075\u0067\u006F\u0073\u006C\u0061\u0076\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u004D\u0061\u0063\u0065\u0064\u006F\u006E\u0069\u0061\u0029", "\u043C\u0430\u043A\u0435\u0434\u043E\u043D\u0441\u043A\u0438\u0020\u0458\u0430\u0437\u0438\u043A\u0020\u0028\u041C\u0430\u043A\u0435\u0434\u043E\u043D\u0438\u0458\u0430\u0029", false, 0x4CA2, "MK", "MKD", "\u004D\u0061\u0063\u0065\u0064\u006F\u006E\u0069\u0061\u0020\u0028\u0046\u0059\u0052\u004F\u004D\u0029", "\u041C\u0430\u043A\u0435\u0434\u043E\u043D\u0438\u0458\u0430", "\u004D\u004B\u0044", "\u004D\u0061\u0063\u0065\u0064\u006F\u006E\u0069\u0061\u006E\u0020\u0044\u0065\u006E\u0061\u0072", "\u0434\u0435\u043D\u0430\u0440", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0434\u0435\u043D\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0435\u0433\u043E\u0440\u0438\u0458\u0430\u043D\u0441\u043A\u0438\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u043D\u0435\u0434", "\u043F\u043E\u043D", "\u0432\u0442\u0440", "\u0441\u0440\u0434", "\u0447\u0435\u0442", "\u043F\u0435\u0442", "\u0441\u0430\u0431" ], [ "\u043D\u0435\u0434\u0435\u043B\u0430", "\u043F\u043E\u043D\u0435\u0434\u0435\u043B\u043D\u0438\u043A", "\u0432\u0442\u043E\u0440\u043D\u0438\u043A", "\u0441\u0440\u0435\u0434\u0430", "\u0447\u0435\u0442\u0432\u0440\u0442\u043E\u043A", "\u043F\u0435\u0442\u043E\u043A", "\u0441\u0430\u0431\u043E\u0442\u0430" ], [ "\u0458\u0430\u043D", "\u0444\u0435\u0432", "\u043C\u0430\u0440", "\u0430\u043F\u0440", "\u043C\u0430\u0458", "\u0458\u0443\u043D", "\u0458\u0443\u043B", "\u0430\u0432\u0433", "\u0441\u0435\u043F", "\u043E\u043A\u0442", "\u043D\u043E\u0435", "\u0434\u0435\u043A", "" ], [ "\u0458\u0430\u043D\u0443\u0430\u0440\u0438", "\u0444\u0435\u0432\u0440\u0443\u0430\u0440\u0438", "\u043C\u0430\u0440\u0442", "\u0430\u043F\u0440\u0438\u043B", "\u043C\u0430\u0458", "\u0458\u0443\u043D\u0438", "\u0458\u0443\u043B\u0438", "\u0430\u0432\u0433\u0443\u0441\u0442", "\u0441\u0435\u043F\u0442\u0435\u043C\u0432\u0440\u0438", "\u043E\u043A\u0442\u043E\u043C\u0432\u0440\u0438", "\u043D\u043E\u0435\u043C\u0432\u0440\u0438", "\u0434\u0435\u043A\u0435\u043C\u0432\u0440\u0438", "" ] },
{ 0x0436, 0x0036, "af-ZA", "af", "afr", "af-ZA", "\u0041\u0066\u0072\u0069\u006B\u0061\u0061\u006E\u0073\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", "\u0041\u0066\u0072\u0069\u006B\u0061\u0061\u006E\u0073\u0020\u0028\u0053\u0075\u0069\u0064\u0020\u0041\u0066\u0072\u0069\u006B\u0061\u0029", false, 0x00D1, "ZA", "ZAF", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u0053\u0075\u0069\u0064\u0020\u0041\u0066\u0072\u0069\u006B\u0061", "\u005A\u0041\u0052", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u006E\u0020\u0052\u0061\u006E\u0064", "\u0052\u0061\u006E\u0064", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0073\u0065\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 0, "\u002F", "\u003A", "", "\u006E\u006D", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u006F\u006E", "\u004D\u0061\u0061\u006E", "\u0044\u0069\u006E\u0073", "\u0057\u006F\u0065\u006E", "\u0044\u006F\u006E\u0064", "\u0056\u0072\u0079", "\u0053\u0061\u0074" ], [ "\u0053\u006F\u006E\u0064\u0061\u0067", "\u004D\u0061\u0061\u006E\u0064\u0061\u0067", "\u0044\u0069\u006E\u0073\u0064\u0061\u0067", "\u0057\u006F\u0065\u006E\u0073\u0064\u0061\u0067", "\u0044\u006F\u006E\u0064\u0065\u0072\u0064\u0061\u0067", "\u0056\u0072\u0079\u0064\u0061\u0067", "\u0053\u0061\u0074\u0065\u0072\u0064\u0061\u0067" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0065\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0073", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0069\u0065", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0069\u0065", "\u004D\u0061\u0061\u0072\u0074", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0065\u0069", "\u004A\u0075\u006E\u0069\u0065", "\u004A\u0075\u006C\u0069\u0065", "\u0041\u0075\u0067\u0075\u0073\u0074\u0075\u0073", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0437, 0x0037, "ka-GE", "ka", "kat", "ka-GE", "\u0047\u0065\u006F\u0072\u0067\u0069\u0061\u006E\u0020\u0028\u0047\u0065\u006F\u0072\u0067\u0069\u0061\u0029", "\u10E5\u10D0\u10E0\u10D7\u10E3\u10DA\u10D8\u0020\u0028\u10E1\u10D0\u10E5\u10D0\u10E0\u10D7\u10D5\u10D4\u10DA\u10DD\u0029", false, 0x0058, "GE", "GEO", "\u0047\u0065\u006F\u0072\u0067\u0069\u0061", "\u10E1\u10D0\u10E5\u10D0\u10E0\u10D7\u10D5\u10D4\u10DA\u10DD", "\u0047\u0045\u004C", "\u004C\u0061\u0072\u0069", "\u10DA\u10D0\u10E0\u10D8", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u004C\u0061\u0072\u0069", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u10D2\u10E0\u10D8\u10D2\u10DD\u10E0\u10D8\u10D0\u10DC\u10E3\u10DA\u10D8\u00A0\u10D9\u10D0\u10DA\u10D4\u10DC\u10D3\u10D0\u10E0\u10D8", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\u0020\'\u10EC\u10DA\u10D8\u10E1\'\u0020\u0064\u0064\u0020\u004D\u004D\u002C\u0020\u0064\u0064\u0064\u0064", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u10EC\u10DA\u10D8\u10E1\'\u0020\u0064\u0064\u0020\u004D\u004D\u002C\u0020\u0064\u0064\u0064\u0064" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u10D9\u10D5\u10D8\u10E0\u10D0", "\u10DD\u10E0\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10E1\u10D0\u10DB\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10DD\u10D7\u10EE\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10EE\u10E3\u10D7\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10DE\u10D0\u10E0\u10D0\u10E1\u10D9\u10D4\u10D5\u10D8", "\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8" ], [ "\u10D9\u10D5\u10D8\u10E0\u10D0", "\u10DD\u10E0\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10E1\u10D0\u10DB\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10DD\u10D7\u10EE\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10EE\u10E3\u10D7\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8", "\u10DE\u10D0\u10E0\u10D0\u10E1\u10D9\u10D4\u10D5\u10D8", "\u10E8\u10D0\u10D1\u10D0\u10D7\u10D8" ], [ "\u10D8\u10D0\u10DC", "\u10D7\u10D4\u10D1", "\u10DB\u10D0\u10E0", "\u10D0\u10DE\u10E0", "\u10DB\u10D0\u10D8\u10E1", "\u10D8\u10D5\u10DC", "\u10D8\u10D5\u10DA", "\u10D0\u10D2\u10D5", "\u10E1\u10D4\u10E5", "\u10DD\u10E5\u10E2", "\u10DC\u10DD\u10D4\u10DB", "\u10D3\u10D4\u10D9", "" ], [ "\u10D8\u10D0\u10DC\u10D5\u10D0\u10E0\u10D8", "\u10D7\u10D4\u10D1\u10D4\u10E0\u10D5\u10D0\u10DA\u10D8", "\u10DB\u10D0\u10E0\u10E2\u10D8", "\u10D0\u10DE\u10E0\u10D8\u10DA\u10D8", "\u10DB\u10D0\u10D8\u10E1\u10D8", "\u10D8\u10D5\u10DC\u10D8\u10E1\u10D8", "\u10D8\u10D5\u10DA\u10D8\u10E1\u10D8", "\u10D0\u10D2\u10D5\u10D8\u10E1\u10E2\u10DD", "\u10E1\u10D4\u10E5\u10E2\u10D4\u10DB\u10D1\u10D4\u10E0\u10D8", "\u10DD\u10E5\u10E2\u10DD\u10DB\u10D1\u10D4\u10E0\u10D8", "\u10DC\u10DD\u10D4\u10DB\u10D1\u10D4\u10E0\u10D8", "\u10D3\u10D4\u10D9\u10D4\u10DB\u10D1\u10D4\u10E0\u10D8", "" ] },
{ 0x0438, 0x0038, "fo-FO", "fo", "fao", "fo-FO", "\u0046\u0061\u0072\u006F\u0065\u0073\u0065\u0020\u0028\u0046\u0061\u0072\u006F\u0065\u0020\u0049\u0073\u006C\u0061\u006E\u0064\u0073\u0029", "\u0066\u00F8\u0072\u006F\u0079\u0073\u006B\u0074\u0020\u0028\u0046\u00F8\u0072\u006F\u0079\u0061\u0072\u0029", false, 0x0051, "FO", "FRO", "\u0046\u0061\u0072\u006F\u0065\u0020\u0049\u0073\u006C\u0061\u006E\u0064\u0073", "\u0046\u00F8\u0072\u006F\u0079\u0061\u0072", "\u0044\u004B\u004B", "\u0044\u0061\u006E\u0069\u0073\u0068\u0020\u004B\u0072\u006F\u006E\u0065", "\u0044\u0061\u006E\u0073\u006B\u0020\u006B\u0072\u006F\u006E\u0065", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0069\u0020\u00E1\u006C\u006D\u0061\u006E\u0061\u006B\u006B\u0069\u006E", 1, 2, "\u002D", "\u002E", "", "", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u002E\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u0075\u006E", "\u006D\u00E1\u006E", "\u0074\u00FD\u0073", "\u006D\u0069\u006B", "\u0068\u00F3\u0073", "\u0066\u0072\u00ED", "\u006C\u0065\u0079\u0067" ], [ "\u0073\u0075\u006E\u006E\u0075\u0064\u0061\u0067\u0075\u0072", "\u006D\u00E1\u006E\u0061\u0064\u0061\u0067\u0075\u0072", "\u0074\u00FD\u0073\u0064\u0061\u0067\u0075\u0072", "\u006D\u0069\u006B\u0075\u0064\u0061\u0067\u0075\u0072", "\u0068\u00F3\u0073\u0064\u0061\u0067\u0075\u0072", "\u0066\u0072\u00ED\u0067\u0067\u006A\u0061\u0064\u0061\u0067\u0075\u0072", "\u006C\u0065\u0079\u0067\u0061\u0072\u0064\u0061\u0067\u0075\u0072" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u0069", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0073", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0070\u0072\u00ED\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0439, 0x0039, "hi-IN", "hi", "hin", "hi-IN", "\u0048\u0069\u006E\u0064\u0069\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0939\u093F\u0902\u0926\u0940\u0020\u0028\u092D\u093E\u0930\u0924\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u092D\u093E\u0930\u0924", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0930\u0941\u092A\u092F\u093E", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0930\u0941", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0905\u0902\u0917\u094D\u0930\u0947\u091C\u093C\u0940\u00A0\u0915\u0948\u0932\u0947\u0928\u094D\u0921\u0930", 1, 0, "\u002D", "\u003A", "\u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928", "\u0905\u092A\u0930\u093E\u0939\u094D\u0928", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0930\u0935\u093F\u002E", "\u0938\u094B\u092E\u002E", "\u092E\u0902\u0917\u0932\u002E", "\u092C\u0941\u0927\u002E", "\u0917\u0941\u0930\u0941\u002E", "\u0936\u0941\u0915\u094D\u0930\u002E", "\u0936\u0928\u093F\u002E" ], [ "\u0930\u0935\u093F\u0935\u093E\u0930", "\u0938\u094B\u092E\u0935\u093E\u0930", "\u092E\u0902\u0917\u0932\u0935\u093E\u0930", "\u092C\u0941\u0927\u0935\u093E\u0930", "\u0917\u0941\u0930\u0941\u0935\u093E\u0930", "\u0936\u0941\u0915\u094D\u0930\u0935\u093E\u0930", "\u0936\u0928\u093F\u0935\u093E\u0930" ], [ "\u091C\u0928\u0935\u0930\u0940", "\u092B\u0930\u0935\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u0905\u092A\u094D\u0930\u0948\u0932", "\u092E\u0908", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u093E\u0908", "\u0905\u0917\u0938\u094D\u0924", "\u0938\u093F\u0924\u092E\u094D\u092C\u0930", "\u0905\u0915\u094D\u0924\u0942\u092C\u0930", "\u0928\u0935\u092E\u094D\u092C\u0930", "\u0926\u093F\u0938\u092E\u094D\u092C\u0930", "" ], [ "\u091C\u0928\u0935\u0930\u0940", "\u092B\u0930\u0935\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u0905\u092A\u094D\u0930\u0948\u0932", "\u092E\u0908", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u093E\u0908", "\u0905\u0917\u0938\u094D\u0924", "\u0938\u093F\u0924\u092E\u094D\u092C\u0930", "\u0905\u0915\u094D\u0924\u0942\u092C\u0930", "\u0928\u0935\u092E\u094D\u092C\u0930", "\u0926\u093F\u0938\u092E\u094D\u092C\u0930", "" ] },
{ 0x043E, 0x003E, "ms-MY", "ms", "msa", "ms-MY", "\u004D\u0061\u006C\u0061\u0079\u0020\u0028\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061\u0029", "\u0042\u0061\u0068\u0061\u0073\u0061\u0020\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061\u0020\u0028\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061\u0029", false, 0x00A7, "MY", "MYS", "\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061", "\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061", "\u004D\u0059\u0052", "\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061\u006E\u0020\u0052\u0069\u006E\u0067\u0067\u0069\u0074", "\u0052\u0069\u006E\u0067\u0067\u0069\u0074\u0020\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061", true, 2, 1, 0, 0, 0, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0020\u004D\u0061\u0073\u0065\u0068\u0069", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0041\u0068\u0061\u0064", "\u0049\u0073\u006E\u0069\u006E", "\u0053\u0065\u006C", "\u0052\u0061\u0062\u0075", "\u004B\u0068\u0061\u006D\u0069\u0073", "\u004A\u0075\u006D\u0061\u0061\u0074", "\u0053\u0061\u0062\u0074\u0075" ], [ "\u0041\u0068\u0061\u0064", "\u0049\u0073\u006E\u0069\u006E", "\u0053\u0065\u006C\u0061\u0073\u0061", "\u0052\u0061\u0062\u0075", "\u004B\u0068\u0061\u006D\u0069\u0073", "\u004A\u0075\u006D\u0061\u0061\u0074", "\u0053\u0061\u0062\u0074\u0075" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0063", "\u0041\u0070\u0072", "\u004D\u0065\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u004F\u0067\u006F\u0073", "\u0053\u0065\u0070\u0074", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0069\u0073", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u004D\u0061\u0063", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0065\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C\u0061\u0069", "\u004F\u0067\u006F\u0073", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0069\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x043F, 0x003F, "kk-KZ", "kk", "kaz", "kk-KZ", "\u004B\u0061\u007A\u0061\u006B\u0068\u0020\u0028\u004B\u0061\u007A\u0061\u006B\u0068\u0073\u0074\u0061\u006E\u0029", "\u049A\u0430\u0437\u0430\u049B\u0020\u0028\u049A\u0430\u0437\u0430\u049B\u0441\u0442\u0430\u043D\u0029", false, 0x0089, "KZ", "KAZ", "\u004B\u0061\u007A\u0061\u006B\u0068\u0073\u0074\u0061\u006E", "\u049A\u0430\u0437\u0430\u049B\u0441\u0442\u0430\u043D", "\u004B\u005A\u0054", "\u0054\u0065\u006E\u0067\u0065", "\u0422", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002D", "\u00A0", "\u0422", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0438\u0433\u043E\u0440\u0438\u0430\u043D\u043A\u04AF\u043D\u0442\u0456\u0437\u0431\u0435\u043A", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0436\u002E\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0436\u002E\'", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\'\u0436\u002E\'" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0416\u043A", "\u0414\u0441", "\u0421\u0441", "\u0421\u0440", "\u0411\u0441", "\u0416\u043C", "\u0421\u043D" ], [ "\u0416\u0435\u043A\u0441\u0435\u043D\u0431\u0456", "\u0414\u04AF\u0439\u0441\u0435\u043D\u0431\u0456", "\u0421\u0435\u0439\u0441\u0435\u043D\u0431\u0456", "\u0421\u04D9\u0440\u0441\u0435\u043D\u0431\u0456", "\u0411\u0435\u0439\u0441\u0435\u043D\u0431\u0456", "\u0416\u04B1\u043C\u0430", "\u0421\u0435\u043D\u0431\u0456" ], [ "\u049A\u0430\u04A3", "\u0410\u049B\u043F", "\u041D\u0430\u0443", "\u0421\u04D9\u0443", "\u041C\u0430\u043C", "\u041C\u0430\u0443", "\u0428\u0456\u043B", "\u0422\u0430\u043C", "\u049A\u044B\u0440", "\u049A\u0430\u0437", "\u049A\u0430\u0440", "\u0416\u0435\u043B", "" ], [ "\u049B\u0430\u04A3\u0442\u0430\u0440", "\u0430\u049B\u043F\u0430\u043D", "\u043D\u0430\u0443\u0440\u044B\u0437", "\u0441\u04D9\u0443\u0456\u0440", "\u043C\u0430\u043C\u044B\u0440", "\u043C\u0430\u0443\u0441\u044B\u043C", "\u0448\u0456\u043B\u0434\u0435", "\u0442\u0430\u043C\u044B\u0437", "\u049B\u044B\u0440\u043A\u04AF\u0439\u0435\u043A", "\u049B\u0430\u0437\u0430\u043D", "\u049B\u0430\u0440\u0430\u0448\u0430", "\u0436\u0435\u043B\u0442\u043E\u049B\u0441\u0430\u043D", "" ] },
{ 0x0440, 0x0040, "ky-KG", "ky", "kir", "ky-KG", "\u004B\u0079\u0072\u0067\u0079\u007A\u0020\u0028\u004B\u0079\u0072\u0067\u0079\u007A\u0073\u0074\u0061\u006E\u0029", "\u041A\u044B\u0440\u0433\u044B\u0437\u0020\u0028\u041A\u044B\u0440\u0433\u044B\u0437\u0441\u0442\u0430\u043D\u0029", false, 0x0082, "KG", "KGZ", "\u004B\u0079\u0072\u0067\u0079\u007A\u0073\u0074\u0061\u006E", "\u041A\u044B\u0440\u0433\u044B\u0437\u0441\u0442\u0430\u043D", "\u004B\u0047\u0053", "\u0073\u006F\u006D", "\u0441\u043E\u043C", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002D", "\u00A0", "\u0441\u043E\u043C", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0438\u0433\u043E\u0440\u0438\u0430\u043D\u00A0\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440\u044B", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\'\u002D\'\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002D\u0436\u002E\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002D\u0436\u002E\'", [ "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\'\u002D\'\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002D\u0436\u002E\'" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\'\u002D\u0436\u002E\'" ], [ "\u0416\u0448", "\u0414\u0448", "\u0428\u0448", "\u0428\u0440", "\u0411\u0448", "\u0416\u043C", "\u0418\u0448" ], [ "\u0416\u0435\u043A\u0448\u0435\u043C\u0431\u0438", "\u0414\u04AF\u0439\u0448\u04E9\u043C\u0431\u04AF", "\u0428\u0435\u0439\u0448\u0435\u043C\u0431\u0438", "\u0428\u0430\u0440\u0448\u0435\u043C\u0431\u0438", "\u0411\u0435\u0439\u0448\u0435\u043C\u0431\u0438", "\u0416\u0443\u043C\u0430", "\u0418\u0448\u0435\u043C\u0431\u0438" ], [ "\u042F\u043D\u0432", "\u0424\u0435\u0432", "\u041C\u0430\u0440", "\u0410\u043F\u0440", "\u041C\u0430\u0439", "\u0418\u044E\u043D", "\u0418\u044E\u043B", "\u0410\u0432\u0433", "\u0421\u0435\u043D", "\u041E\u043A\u0442", "\u041D\u043E\u044F", "\u0414\u0435\u043A", "" ], [ "\u042F\u043D\u0432\u0430\u0440\u044C", "\u0424\u0435\u0432\u0440\u0430\u043B\u044C", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0435\u043B\u044C", "\u041C\u0430\u0439", "\u0418\u044E\u043D\u044C", "\u0418\u044E\u043B\u044C", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043D\u0442\u044F\u0431\u0440\u044C", "\u041E\u043A\u0442\u044F\u0431\u0440\u044C", "\u041D\u043E\u044F\u0431\u0440\u044C", "\u0414\u0435\u043A\u0430\u0431\u0440\u044C", "" ] },
{ 0x0441, 0x0041, "sw-KE", "sw", "swa", "sw-KE", "\u004B\u0069\u0073\u0077\u0061\u0068\u0069\u006C\u0069\u0020\u0028\u004B\u0065\u006E\u0079\u0061\u0029", "\u004B\u0069\u0073\u0077\u0061\u0068\u0069\u006C\u0069\u0020\u0028\u004B\u0065\u006E\u0079\u0061\u0029", false, 0x0081, "KE", "KEN", "\u004B\u0065\u006E\u0079\u0061", "\u004B\u0065\u006E\u0079\u0061", "\u004B\u0045\u0053", "\u004B\u0065\u006E\u0079\u0061\u006E\u0020\u0053\u0068\u0069\u006C\u006C\u0069\u006E\u0067", "\u0053\u0068\u0069\u006C\u0069\u006E\u0067\u0069", false, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0053", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u002F\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0443, 0x0043, "uz-UZ-Latn", "uz", "uzb", "uz-Latn-UZ", "\u0055\u007A\u0062\u0065\u006B\u0020\u0028\u004C\u0061\u0074\u0069\u006E\u002C\u0020\u0055\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u0061\u006E\u0029", "\u0055\u0027\u007A\u0062\u0065\u006B\u0020\u0028\u0055\u0027\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u006F\u006E\u0020\u0052\u0065\u0073\u0070\u0075\u0062\u006C\u0069\u006B\u0061\u0073\u0069\u0029", false, 0x00F7, "UZ", "UZB", "\u0055\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u0061\u006E", "\u0055\u0027\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u006F\u006E\u0020\u0052\u0065\u0073\u0070\u0075\u0062\u006C\u0069\u006B\u0061\u0073\u0069", "\u0055\u005A\u0053", "\u0055\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u0061\u006E\u0020\u0053\u0075\u006D", "\u0440\u0443\u0431\u043B\u044C", true, 2, 1, 0, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0073\u0075\u0027\u006D", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0069\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0069", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\u0020\'\u0079\u0069\u006C\'\u0020\u0064\u002D\u004D\u004D\u004D\u004D", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002D\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u0079\u0069\u006C\'\u0020\u0064\u002D\u004D\u004D\u004D\u004D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0079\u0061\u006B\u002E", "\u0064\u0073\u0068\u002E", "\u0073\u0065\u0073\u0068\u002E", "\u0063\u0068\u0072\u002E", "\u0070\u0073\u0068\u002E", "\u006A\u006D\u002E", "\u0073\u0068\u002E" ], [ "\u0079\u0061\u006B\u0073\u0068\u0061\u006E\u0062\u0061", "\u0064\u0075\u0073\u0068\u0061\u006E\u0062\u0061", "\u0073\u0065\u0073\u0068\u0061\u006E\u0062\u0061", "\u0063\u0068\u006F\u0072\u0073\u0068\u0061\u006E\u0062\u0061", "\u0070\u0061\u0079\u0073\u0068\u0061\u006E\u0062\u0061", "\u006A\u0075\u006D\u0061", "\u0073\u0068\u0061\u006E\u0062\u0061" ], [ "\u0079\u0061\u006E\u0076\u0061\u0072", "\u0066\u0065\u0076\u0072\u0061\u006C", "\u006D\u0061\u0072\u0074", "\u0061\u0070\u0072\u0065\u006C", "\u006D\u0061\u0079", "\u0069\u0079\u0075\u006E", "\u0069\u0079\u0075\u006C", "\u0061\u0076\u0067\u0075\u0073\u0074", "\u0073\u0065\u006E\u0074\u0079\u0061\u0062\u0072", "\u006F\u006B\u0074\u0079\u0061\u0062\u0072", "\u006E\u006F\u0079\u0061\u0062\u0072", "\u0064\u0065\u006B\u0061\u0062\u0072", "" ], [ "\u0079\u0061\u006E\u0076\u0061\u0072", "\u0066\u0065\u0076\u0072\u0061\u006C", "\u006D\u0061\u0072\u0074", "\u0061\u0070\u0072\u0065\u006C", "\u006D\u0061\u0079", "\u0069\u0079\u0075\u006E", "\u0069\u0079\u0075\u006C", "\u0061\u0076\u0067\u0075\u0073\u0074", "\u0073\u0065\u006E\u0074\u0079\u0061\u0062\u0072", "\u006F\u006B\u0074\u0079\u0061\u0062\u0072", "\u006E\u006F\u0079\u0061\u0062\u0072", "\u0064\u0065\u006B\u0061\u0062\u0072", "" ] },
{ 0x0444, 0x0044, "tt-RU", "tt", "tat", "tt-RU", "\u0054\u0061\u0074\u0061\u0072\u0020\u0028\u0052\u0075\u0073\u0073\u0069\u0061\u0029", "\u0422\u0430\u0442\u0430\u0440\u0020\u0028\u0420\u043E\u0441\u0441\u0438\u044F\u0029", false, 0x00CB, "RU", "RUS", "\u0052\u0075\u0073\u0073\u0069\u0061", "\u0420\u043E\u0441\u0441\u0438\u044F", "\u0052\u0055\u0052", "\u0052\u0075\u0073\u0073\u0069\u0061\u006E\u0020\u0052\u0075\u0062\u006C\u0065", "\u0440\u0443\u0431\u043B\u044C", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0440\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0413\u0440\u0438\u0433\u043E\u0440\u0438\u0430\u043D\u00A0\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440\u0435", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u042F\u043A\u0448", "\u0414\u04AF\u0448", "\u0421\u0438\u0448", "\u0427\u04D9\u0440\u0448", "\u041F\u04D9\u043D\u0497", "\u0496\u043E\u043C", "\u0428\u0438\u043C" ], [ "\u042F\u043A\u0448\u04D9\u043C\u0431\u0435", "\u0414\u04AF\u0448\u04D9\u043C\u0431\u0435", "\u0421\u0438\u0448\u04D9\u043C\u0431\u0435", "\u0427\u04D9\u0440\u0448\u04D9\u043C\u0431\u0435", "\u041F\u04D9\u043D\u0497\u0435\u0448\u04D9\u043C\u0431\u0435", "\u0496\u043E\u043C\u0433\u0430", "\u0428\u0438\u043C\u0431\u04D9" ], [ "\u0413\u044B\u0439\u043D\u0432", "\u0424\u0435\u0432", "\u041C\u0430\u0440", "\u0410\u043F\u0440", "\u041C\u0430\u0439", "\u0418\u044E\u043D", "\u0418\u044E\u043B", "\u0410\u0432\u0433", "\u0421\u0435\u043D", "\u041E\u043A\u0442", "\u041D\u043E\u044F", "\u0414\u0435\u043A", "" ], [ "\u0413\u044B\u0439\u043D\u0432\u0430\u0440\u044C", "\u0424\u0435\u0432\u0440\u0430\u043B\u044C", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0435\u043B\u044C", "\u041C\u0430\u0439", "\u0418\u044E\u043D\u044C", "\u0418\u044E\u043B\u044C", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043D\u0442\u044F\u0431\u0440\u044C", "\u041E\u043A\u0442\u044F\u0431\u0440\u044C", "\u041D\u043E\u044F\u0431\u0440\u044C", "\u0414\u0435\u043A\u0430\u0431\u0440\u044C", "" ] },
{ 0x0446, 0x0046, "pa-IN", "pa", "pan", "pa-IN", "\u0050\u0075\u006E\u006A\u0061\u0062\u0069\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0A2A\u0A70\u0A1C\u0A3E\u0A2C\u0A40\u0020\u0028\u0A2D\u0A3E\u0A30\u0A24\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u0A2D\u0A3E\u0A30\u0A24", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0A30\u0A41\u0A2A\u0A3F\u0A06", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0A30\u0A41", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0A66", "\u0A67", "\u0A68", "\u0A69", "\u0A6A", "\u0A6B", "\u0A6C", "\u0A6D", "\u0A6E", "\u0A6F" ], 1, [ 1 ], "\u0A05\u0A70\u0A17\u0A4D\u0A30\u0A47\u0A5B\u0A40\u00A0\u0A15\u0A32\u0A70\u0A21\u0A30", 1, 0, "\u002D", "\u003A", "\u0A38\u0A35\u0A47\u0A30\u0A47", "\u0A36\u0A3E\u0A2E", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\u0064\u0064\u0064\u0064", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\u0064\u0064\u0064\u0064", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0A10\u0A24\u002E", "\u0A38\u0A4B\u0A2E\u002E", "\u0A2E\u0A70\u0A17\u0A32\u002E", "\u0A2C\u0A41\u0A27\u002E", "\u0A35\u0A40\u0A30\u002E", "\u0A36\u0A41\u0A15\u0A30\u002E", "\u0A36\u0A28\u0A40\u002E" ], [ "\u0A10\u0A24\u0A35\u0A3E\u0A30", "\u0A38\u0A4B\u0A2E\u0A35\u0A3E\u0A30", "\u0A2E\u0A70\u0A17\u0A32\u0A35\u0A3E\u0A30", "\u0A2C\u0A41\u0A27\u0A35\u0A3E\u0A30", "\u0A35\u0A40\u0A30\u0A35\u0A3E\u0A30", "\u0A36\u0A41\u0A71\u0A15\u0A30\u0A35\u0A3E\u0A30", "\u0A36\u0A28\u0A40\u0A1A\u0A30\u0A35\u0A3E\u0A30" ], [ "\u0A1C\u0A28\u0A35\u0A30\u0A40", "\u0A5E\u0A30\u0A35\u0A30\u0A40", "\u0A2E\u0A3E\u0A30\u0A1A", "\u0A05\u0A2A\u0A4D\u0A30\u0A48\u0A32", "\u0A2E\u0A08", "\u0A1C\u0A42\u0A28", "\u0A1C\u0A41\u0A32\u0A3E\u0A08", "\u0A05\u0A17\u0A38\u0A24", "\u0A38\u0A24\u0A70\u0A2C\u0A30", "\u0A05\u0A15\u0A24\u0A42\u0A2C\u0A30", "\u0A28\u0A35\u0A70\u0A2C\u0A30", "\u0A26\u0A38\u0A70\u0A2C\u0A30", "" ], [ "\u0A1C\u0A28\u0A35\u0A30\u0A40", "\u0A5E\u0A30\u0A35\u0A30\u0A40", "\u0A2E\u0A3E\u0A30\u0A1A", "\u0A05\u0A2A\u0A4D\u0A30\u0A48\u0A32", "\u0A2E\u0A08", "\u0A1C\u0A42\u0A28", "\u0A1C\u0A41\u0A32\u0A3E\u0A08", "\u0A05\u0A17\u0A38\u0A24", "\u0A38\u0A24\u0A70\u0A2C\u0A30", "\u0A05\u0A15\u0A24\u0A42\u0A2C\u0A30", "\u0A28\u0A35\u0A70\u0A2C\u0A30", "\u0A26\u0A38\u0A70\u0A2C\u0A30", "" ] },
{ 0x0447, 0x0047, "gu-IN", "gu", "guj", "gu-IN", "\u0047\u0075\u006A\u0061\u0072\u0061\u0074\u0069\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0A97\u0AC1\u0A9C\u0AB0\u0ABE\u0AA4\u0AC0\u0020\u0028\u0AAD\u0ABE\u0AB0\u0AA4\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u0AAD\u0ABE\u0AB0\u0AA4", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0AB0\u0AC2\u0AAA\u0ABF\u0AAF\u0ACB", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0AB0\u0AC2", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0AE6", "\u0AE7", "\u0AE8", "\u0AE9", "\u0AEA", "\u0AEB", "\u0AEC", "\u0AED", "\u0AEE", "\u0AEF" ], 1, [ 1 ], "\u0A85\u0A82\u0A97\u0ACD\u0AB0\u0AC7\u0A9C\u0AC0\u00A0\u0A95\u0AC5\u0AB2\u0AC7\u0AA8\u0ACD\u0AA1\u0AB0", 1, 0, "\u002D", "\u003A", "\u0AAA\u0AC2\u0AB0\u0ACD\u0AB5\u00A0\u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8", "\u0A89\u0AA4\u0ACD\u0AA4\u0AB0\u00A0\u0AAE\u0AA7\u0ACD\u0AAF\u0ABE\u0AB9\u0ACD\u0AA8", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0AB0\u0AB5\u0ABF", "\u0AB8\u0ACB\u0AAE", "\u0AAE\u0A82\u0A97\u0AB3", "\u0AAC\u0AC1\u0AA7", "\u0A97\u0AC1\u0AB0\u0AC1", "\u0AB6\u0AC1\u0A95\u0ACD\u0AB0", "\u0AB6\u0AA8\u0ABF" ], [ "\u0AB0\u0AB5\u0ABF\u0AB5\u0ABE\u0AB0", "\u0AB8\u0ACB\u0AAE\u0AB5\u0ABE\u0AB0", "\u0AAE\u0A82\u0A97\u0AB3\u0AB5\u0ABE\u0AB0", "\u0AAC\u0AC1\u0AA7\u0AB5\u0ABE\u0AB0", "\u0A97\u0AC1\u0AB0\u0AC1\u0AB5\u0ABE\u0AB0", "\u0AB6\u0AC1\u0A95\u0ACD\u0AB0\u0AB5\u0ABE\u0AB0", "\u0AB6\u0AA8\u0ABF\u0AB5\u0ABE\u0AB0" ], [ "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1", "\u0AAB\u0AC7\u0AAC\u0ACD\u0AB0\u0AC1", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A", "\u0A8F\u0AAA\u0ACD\u0AB0\u0ABF\u0AB2", "\u0AAE\u0AC7", "\u0A9C\u0AC2\u0AA8", "\u0A9C\u0AC1\u0AB2\u0ABE\u0A88", "\u0A91\u0A97\u0AB8\u0ACD\u0A9F", "\u0AB8\u0AAA\u0ACD\u0A9F\u0AC7", "\u0A91\u0A95\u0ACD\u0A9F\u0ACB", "\u0AA8\u0AB5\u0AC7", "\u0AA1\u0ABF\u0AB8\u0AC7", "" ], [ "\u0A9C\u0ABE\u0AA8\u0ACD\u0AAF\u0AC1\u0A86\u0AB0\u0AC0", "\u0AAB\u0AC7\u0AAC\u0ACD\u0AB0\u0AC1\u0A86\u0AB0\u0AC0", "\u0AAE\u0ABE\u0AB0\u0ACD\u0A9A", "\u0A8F\u0AAA\u0ACD\u0AB0\u0ABF\u0AB2", "\u0AAE\u0AC7", "\u0A9C\u0AC2\u0AA8", "\u0A9C\u0AC1\u0AB2\u0ABE\u0A88", "\u0A91\u0A97\u0AB8\u0ACD\u0A9F", "\u0AB8\u0AAA\u0ACD\u0A9F\u0AC7\u0AAE\u0ACD\u0AAC\u0AB0", "\u0A91\u0A95\u0ACD\u0A9F\u0ACD\u0AAC\u0AB0", "\u0AA8\u0AB5\u0AC7\u0AAE\u0ACD\u0AAC\u0AB0", "\u0AA1\u0ABF\u0AB8\u0AC7\u0AAE\u0ACD\u0AAC\u0AB0", "" ] },
{ 0x0449, 0x0049, "ta-IN", "ta", "tam", "ta-IN", "\u0054\u0061\u006D\u0069\u006C\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0BA4\u0BAE\u0BBF\u0BB4\u0BCD\u0020\u0028\u0B87\u0BA8\u0BCD\u0BA4\u0BBF\u0BAF\u0BBE\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u0B87\u0BA8\u0BCD\u0BA4\u0BBF\u0BAF\u0BBE", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0BB0\u0BC2\u0BAA\u0BBE\u0BAF\u0BCD", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0BB0\u0BC2", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0B86\u0B99\u0BCD\u0B95\u0BBF\u0BB2\u00A0\u0BB5\u0BB0\u0BC1\u0B9F\u0BAE\u0BCD", 1, 0, "\u002D", "\u003A", "\u0B95\u0BBE\u0BB2\u0BC8", "\u0BAE\u0BBE\u0BB2\u0BC8", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0B9E\u0BBE", "\u0BA4\u0BBF", "\u0B9A\u0BC6", "\u0BAA\u0BC1", "\u0BB5\u0BBF", "\u0BB5\u0BC6", "\u0B9A" ], [ "\u0B9E\u0BBE\u0BAF\u0BBF\u0BB1\u0BC1", "\u0BA4\u0BBF\u0B99\u0BCD\u0B95\u0BB3\u0BCD", "\u0B9A\u0BC6\u0BB5\u0BCD\u0BB5\u0BBE\u0BAF\u0BCD", "\u0BAA\u0BC1\u0BA4\u0BA9\u0BCD", "\u0BB5\u0BBF\u0BAF\u0BBE\u0BB4\u0BA9\u0BCD", "\u0BB5\u0BC6\u0BB3\u0BCD\u0BB3\u0BBF", "\u0B9A\u0BA9\u0BBF" ], [ "\u0B9C\u0BA9\u002E", "\u0BAA\u0BC6\u0BAA\u0BCD\u002E", "\u0BAE\u0BBE\u0BB0\u0BCD\u002E", "\u0B8F\u0BAA\u0BCD\u002E", "\u0BAE\u0BC7", "\u0B9C\u0BC2\u0BA9\u0BCD", "\u0B9C\u0BC2\u0BB2\u0BC8", "\u0B86\u0B95\u002E", "\u0B9A\u0BC6\u0BAA\u0BCD\u002E", "\u0B85\u0B95\u0BCD\u002E", "\u0BA8\u0BB5\u002E", "\u0B9F\u0BBF\u0B9A\u002E", "" ], [ "\u0B9C\u0BA9\u0BB5\u0BB0\u0BBF", "\u0BAA\u0BC6\u0BAA\u0BCD\u0BB0\u0BB5\u0BB0\u0BBF", "\u0BAE\u0BBE\u0BB0\u0BCD\u0B9A\u0BCD", "\u0B8F\u0BAA\u0BCD\u0BB0\u0BB2\u0BCD", "\u0BAE\u0BC7", "\u0B9C\u0BC2\u0BA9\u0BCD", "\u0B9C\u0BC2\u0BB2\u0BC8", "\u0B86\u0B95\u0BB8\u0BCD\u0B9F\u0BCD", "\u0B9A\u0BC6\u0BAA\u0BCD\u0B9F\u0BAE\u0BCD\u0BAA\u0BB0\u0BCD", "\u0B85\u0B95\u0BCD\u0B9F\u0BCB\u0BAA\u0BB0\u0BCD", "\u0BA8\u0BB5\u0BAE\u0BCD\u0BAA\u0BB0\u0BCD", "\u0B9F\u0BBF\u0B9A\u0BAE\u0BCD\u0BAA\u0BB0\u0BCD", "" ] },
{ 0x044A, 0x004A, "te-IN", "te", "tel", "te-IN", "\u0054\u0065\u006C\u0075\u0067\u0075\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0C24\u0C46\u0C32\u0C41\u0C17\u0C41\u0020\u0028\u0C2D\u0C3E\u0C30\u0C24\u00A0\u0C26\u0C47\u0C36\u0C02\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u0C2D\u0C3E\u0C30\u0C24\u00A0\u0C26\u0C47\u0C36\u0C02", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0C30\u0C42\u0C2A\u0C3E\u0C2F\u0C3F", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0C30\u0C42", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0C66", "\u0C67", "\u0C68", "\u0C69", "\u0C6A", "\u0C6B", "\u0C6C", "\u0C6D", "\u0C6E", "\u0C6F" ], 1, [ 1 ], "\u0C07\u0C02\u0C17\u0C4D\u0C32\u0C40\u0C37\u0C41\u00A0\u0C15\u0C4D\u0C2F\u0C3E\u0C32\u0C02\u0C21\u0C30\u0C4D", 1, 0, "\u002D", "\u003A", "\u0C2A\u0C42\u0C30\u0C4D\u0C35\u0C3E\u0C39\u0C4D\u0C28", "\u0C05\u0C2A\u0C30\u0C3E\u0C39\u0C4D\u0C28", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0C06\u0C26\u0C3F\u002E", "\u0C38\u0C4B\u0C2E\u002E", "\u0C2E\u0C02\u0C17\u0C33\u002E", "\u0C2C\u0C41\u0C27\u002E", "\u0C17\u0C41\u0C30\u0C41\u002E", "\u0C36\u0C41\u0C15\u0C4D\u0C30\u002E", "\u0C36\u0C28\u0C3F\u002E" ], [ "\u0C06\u0C26\u0C3F\u0C35\u0C3E\u0C30\u0C02", "\u0C38\u0C4B\u0C2E\u0C35\u0C3E\u0C30\u0C02", "\u0C2E\u0C02\u0C17\u0C33\u0C35\u0C3E\u0C30\u0C02", "\u0C2C\u0C41\u0C27\u0C35\u0C3E\u0C30\u0C02", "\u0C17\u0C41\u0C30\u0C41\u0C35\u0C3E\u0C30\u0C02", "\u0C36\u0C41\u0C15\u0C4D\u0C30\u0C35\u0C3E\u0C30\u0C02", "\u0C36\u0C28\u0C3F\u0C35\u0C3E\u0C30\u0C02" ], [ "\u0C1C\u0C28\u0C35\u0C30\u0C3F", "\u0C2B\u0C3F\u0C2C\u0C4D\u0C30\u0C35\u0C30\u0C3F", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F", "\u0C0F\u0C2A\u0C4D\u0C30\u0C3F\u0C32\u0C4D", "\u0C2E\u0C47", "\u0C1C\u0C42\u0C28\u0C4D", "\u0C1C\u0C42\u0C32\u0C48", "\u0C06\u0C17\u0C38\u0C4D\u0C1F\u0C41", "\u0C38\u0C46\u0C2A\u0C4D\u0C1F\u0C46\u0C02\u0C2C\u0C30\u0C4D", "\u0C05\u0C15\u0C4D\u0C1F\u0C4B\u0C2C\u0C30\u0C4D", "\u0C28\u0C35\u0C02\u0C2C\u0C30\u0C4D", "\u0C21\u0C3F\u0C38\u0C46\u0C02\u0C2C\u0C30\u0C4D", "" ], [ "\u0C1C\u0C28\u0C35\u0C30\u0C3F", "\u0C2B\u0C3F\u0C2C\u0C4D\u0C30\u0C35\u0C30\u0C3F", "\u0C2E\u0C3E\u0C30\u0C4D\u0C1A\u0C3F", "\u0C0F\u0C2A\u0C4D\u0C30\u0C3F\u0C32\u0C4D", "\u0C2E\u0C47", "\u0C1C\u0C42\u0C28\u0C4D", "\u0C1C\u0C42\u0C32\u0C48", "\u0C06\u0C17\u0C38\u0C4D\u0C1F\u0C41", "\u0C38\u0C46\u0C2A\u0C4D\u0C1F\u0C46\u0C02\u0C2C\u0C30\u0C4D", "\u0C05\u0C15\u0C4D\u0C1F\u0C4B\u0C2C\u0C30\u0C4D", "\u0C28\u0C35\u0C02\u0C2C\u0C30\u0C4D", "\u0C21\u0C3F\u0C38\u0C46\u0C02\u0C2C\u0C30\u0C4D", "" ] },
{ 0x044B, 0x004B, "kn-IN", "kn", "kan", "kn-IN", "\u004B\u0061\u006E\u006E\u0061\u0064\u0061\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0C95\u0CA8\u0CCD\u0CA8\u0CA1\u0020\u0028\u0CAD\u0CBE\u0CB0\u0CA4\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u0CAD\u0CBE\u0CB0\u0CA4", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0CB0\u0CC2\u0CAA\u0CBE\u0CAF\u0CBF", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0CB0\u0CC2", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0CE6", "\u0CE7", "\u0CE8", "\u0CE9", "\u0CEA", "\u0CEB", "\u0CEC", "\u0CED", "\u0CEE", "\u0CEF" ], 1, [ 1 ], "\u0C87\u0C82\u0C97\u0CCD\u0CB2\u0CBF\u0CB7\u0CCD\u00A0\u0C95\u0CCD\u0CAF\u0CBE\u0CB2\u0CC6\u0C82\u0CA1\u0CB0\u0CCD", 1, 0, "\u002D", "\u003A", "\u0CAA\u0CC2\u0CB0\u0CCD\u0CB5\u0CBE\u0CB9\u0CCD\u0CA8", "\u0C85\u0CAA\u0CB0\u0CBE\u0CB9\u0CCD\u0CA8", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0CAD\u0CBE\u0CA8\u0CC1\u002E", "\u0CB8\u0CCB\u0CAE\u002E", "\u0CAE\u0C82\u0C97\u0CB3\u002E", "\u0CAC\u0CC1\u0CA7\u002E", "\u0C97\u0CC1\u0CB0\u0CC1\u002E", "\u0CB6\u0CC1\u0C95\u0CCD\u0CB0\u002E", "\u0CB6\u0CA8\u0CBF\u002E" ], [ "\u0CAD\u0CBE\u0CA8\u0CC1\u0CB5\u0CBE\u0CB0", "\u0CB8\u0CCB\u0CAE\u0CB5\u0CBE\u0CB0", "\u0CAE\u0C82\u0C97\u0CB3\u0CB5\u0CBE\u0CB0", "\u0CAC\u0CC1\u0CA7\u0CB5\u0CBE\u0CB0", "\u0C97\u0CC1\u0CB0\u0CC1\u0CB5\u0CBE\u0CB0", "\u0CB6\u0CC1\u0C95\u0CCD\u0CB0\u0CB5\u0CBE\u0CB0", "\u0CB6\u0CA8\u0CBF\u0CB5\u0CBE\u0CB0" ], [ "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF", "\u0CAB\u0CC6\u0CAC\u0CCD\u0CB0\u0CB5\u0CB0\u0CBF", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD", "\u0C8E\u0CAA\u0CCD\u0CB0\u0CBF\u0CB2\u0CCD", "\u0CAE\u0CC7", "\u0C9C\u0CC2\u0CA8\u0CCD", "\u0C9C\u0CC1\u0CB2\u0CC8", "\u0C86\u0C97\u0CB8\u0CCD\u0C9F\u0CCD", "\u0CB8\u0CC6\u0CAA\u0CCD\u0C9F\u0C82\u0CAC\u0CB0\u0CCD", "\u0C85\u0C95\u0CCD\u0C9F\u0CCB\u0CAC\u0CB0\u0CCD", "\u0CA8\u0CB5\u0CC6\u0C82\u0CAC\u0CB0\u0CCD", "\u0CA1\u0CBF\u0CB8\u0CC6\u0C82\u0CAC\u0CB0\u0CCD", "" ], [ "\u0C9C\u0CA8\u0CB5\u0CB0\u0CBF", "\u0CAB\u0CC6\u0CAC\u0CCD\u0CB0\u0CB5\u0CB0\u0CBF", "\u0CAE\u0CBE\u0CB0\u0CCD\u0C9A\u0CCD", "\u0C8E\u0CAA\u0CCD\u0CB0\u0CBF\u0CB2\u0CCD", "\u0CAE\u0CC7", "\u0C9C\u0CC2\u0CA8\u0CCD", "\u0C9C\u0CC1\u0CB2\u0CC8", "\u0C86\u0C97\u0CB8\u0CCD\u0C9F\u0CCD", "\u0CB8\u0CC6\u0CAA\u0CCD\u0C9F\u0C82\u0CAC\u0CB0\u0CCD", "\u0C85\u0C95\u0CCD\u0C9F\u0CCB\u0CAC\u0CB0\u0CCD", "\u0CA8\u0CB5\u0CC6\u0C82\u0CAC\u0CB0\u0CCD", "\u0CA1\u0CBF\u0CB8\u0CC6\u0C82\u0CAC\u0CB0\u0CCD", "" ] },
{ 0x044E, 0x004E, "mr-IN", "mr", "mar", "mr-IN", "\u004D\u0061\u0072\u0061\u0074\u0068\u0069\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u092E\u0930\u093E\u0920\u0940\u0020\u0028\u092D\u093E\u0930\u0924\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u092D\u093E\u0930\u0924", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0930\u0941\u092A\u092F\u093E", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0930\u0941", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0966", "\u0967", "\u0968", "\u0969", "\u096A", "\u096B", "\u096C", "\u096D", "\u096E", "\u096F" ], 1, [ 1 ], "\u0907\u0902\u0917\u094D\u0930\u091C\u0940\u00A0\u0915\u0945\u0932\u0947\u0928\u094D\u0921\u0930", 1, 0, "\u002D", "\u003A", "\u092E\u002E\u092A\u0942\u002E", "\u092E\u002E\u0928\u0902\u002E", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0930\u0935\u093F\u002E", "\u0938\u094B\u092E\u002E", "\u092E\u0902\u0917\u0933\u002E", "\u092C\u0941\u0927\u002E", "\u0917\u0941\u0930\u0941\u002E", "\u0936\u0941\u0915\u094D\u0930\u002E", "\u0936\u0928\u093F\u002E" ], [ "\u0930\u0935\u093F\u0935\u093E\u0930", "\u0938\u094B\u092E\u0935\u093E\u0930", "\u092E\u0902\u0917\u0933\u0935\u093E\u0930", "\u092C\u0941\u0927\u0935\u093E\u0930", "\u0917\u0941\u0930\u0941\u0935\u093E\u0930", "\u0936\u0941\u0915\u094D\u0930\u0935\u093E\u0930", "\u0936\u0928\u093F\u0935\u093E\u0930" ], [ "\u091C\u093E\u0928\u0947\u002E", "\u092B\u0947\u092C\u094D\u0930\u0941\u002E", "\u092E\u093E\u0930\u094D\u091A", "\u090F\u092A\u094D\u0930\u093F\u0932", "\u092E\u0947", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u0948", "\u0911\u0917\u0938\u094D\u091F", "\u0938\u092A\u094D\u091F\u0947\u0902\u002E", "\u0911\u0915\u094D\u091F\u094B\u002E", "\u0928\u094B\u0935\u094D\u0939\u0947\u0902\u002E", "\u0921\u093F\u0938\u0947\u0902\u002E", "" ], [ "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940", "\u092B\u0947\u092C\u094D\u0930\u0941\u0935\u093E\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u090F\u092A\u094D\u0930\u093F\u0932", "\u092E\u0947", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u0948", "\u0911\u0917\u0938\u094D\u091F", "\u0938\u092A\u094D\u091F\u0947\u0902\u092C\u0930", "\u0911\u0915\u094D\u091F\u094B\u092C\u0930", "\u0928\u094B\u0935\u094D\u0939\u0947\u0902\u092C\u0930", "\u0921\u093F\u0938\u0947\u0902\u092C\u0930", "" ] },
{ 0x044F, 0x004F, "sa-IN", "sa", "san", "sa-IN", "\u0053\u0061\u006E\u0073\u006B\u0072\u0069\u0074\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0938\u0902\u0938\u094D\u0915\u0943\u0924\u0020\u0028\u092D\u093E\u0930\u0924\u092E\u094D\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u092D\u093E\u0930\u0924\u092E\u094D", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0930\u0941\u094D\u092F\u0915\u092E\u094D", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0930\u0941", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0966", "\u0967", "\u0968", "\u0969", "\u096A", "\u096B", "\u096C", "\u096D", "\u096E", "\u096F" ], 1, [ 1 ], "\u0916\u094D\u0930\u093F\u0938\u094D\u0924\u093E\u092C\u094D\u0926\u00A0\u092A\u091E\u094D\u091C\u093F\u0915\u093E", 0, 0, "\u002D", "\u003A", "\u092A\u0942\u0930\u094D\u0935\u093E\u0939\u094D\u0928", "\u0905\u092A\u0930\u093E\u0939\u094D\u0928", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\u0064\u0064\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079\u0020\u0064\u0064\u0064\u0064", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0930\u0935\u093F\u0935\u093E\u0938\u0930\u0903", "\u0938\u094B\u092E\u0935\u093E\u0938\u0930\u0903", "\u092E\u0919\u094D\u0917\u0932\u0935\u093E\u0938\u0930\u0903", "\u092C\u0941\u0927\u0935\u093E\u0938\u0930\u0903", "\u0917\u0941\u0930\u0941\u0935\u093E\u0938\u0930\u0903", "\u0936\u0941\u0915\u094D\u0930\u0935\u093E\u0938\u0930\u0903", "\u0936\u0928\u093F\u0935\u093E\u0938\u0930\u0903" ], [ "\u0930\u0935\u093F\u0935\u093E\u0938\u0930\u0903", "\u0938\u094B\u092E\u0935\u093E\u0938\u0930\u0903", "\u092E\u0919\u094D\u0917\u0932\u0935\u093E\u0938\u0930\u0903", "\u092C\u0941\u0927\u0935\u093E\u0938\u0930\u0903", "\u0917\u0941\u0930\u0941\u0935\u093E\u0938\u0930\u0903", "\u0936\u0941\u0915\u094D\u0930\u0935\u093E\u0938\u0930\u0903", "\u0936\u0928\u093F\u0935\u093E\u0938\u0930\u0903" ], [ "\u091C\u0928\u0935\u0930\u0940", "\u092B\u0930\u0935\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u0905\u092A\u094D\u0930\u0948\u0932", "\u092E\u0908", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u093E\u0908", "\u0905\u0917\u0938\u094D\u0924", "\u0938\u093F\u0924\u092E\u094D\u092C\u0930", "\u0905\u0915\u094D\u0924\u0942\u092C\u0930", "\u0928\u0935\u092E\u094D\u092C\u0930", "\u0926\u093F\u0938\u092E\u094D\u092C\u0930", "" ], [ "\u091C\u0928\u0935\u0930\u0940", "\u092B\u0930\u0935\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u0905\u092A\u094D\u0930\u0948\u0932", "\u092E\u0908", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u093E\u0908", "\u0905\u0917\u0938\u094D\u0924", "\u0938\u093F\u0924\u092E\u094D\u092C\u0930", "\u0905\u0915\u094D\u0924\u0942\u092C\u0930", "\u0928\u0935\u092E\u094D\u092C\u0930", "\u0926\u093F\u0938\u092E\u094D\u092C\u0930", "" ] },
{ 0x0450, 0x0050, "mn-MN", "mn", "mon", "mn-MN", "\u004D\u006F\u006E\u0067\u006F\u006C\u0069\u0061\u006E\u0020\u0028\u0043\u0079\u0072\u0069\u006C\u006C\u0069\u0063\u002C\u0020\u004D\u006F\u006E\u0067\u006F\u006C\u0069\u0061\u0029", "\u041C\u043E\u043D\u0433\u043E\u043B\u00A0\u0445\u044D\u043B\u0020\u0028\u041C\u043E\u043D\u0433\u043E\u043B\u00A0\u0443\u043B\u0441\u0029", false, 0x009A, "MN", "MNG", "\u004D\u006F\u006E\u0067\u006F\u006C\u0069\u0061", "\u041C\u043E\u043D\u0433\u043E\u043B\u00A0\u0443\u043B\u0441", "\u004D\u004E\u0054", "\u0054\u0075\u0067\u0072\u0069\u006B", "\u0422\u04E9\u0433\u0440\u04E9\u0433", true, 2, 1, 2, 5, 1, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AE", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0413\u0440\u0435\u0433\u043E\u0440\u0438\u0439\u043D\u00A0\u043E\u043D\u00A0\u0442\u043E\u043E\u043B\u043E\u043B", 1, 0, "\u002E", "\u003A", "", "", "\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064", "\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\u0020\'\u043E\u043D\u044B\'\u0020\u004D\u004D\u004D\u004D\u0020\u0064", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u0079\u0079\u0079\u0079\u0020\'\u043E\u043D\'\u0020\u004D\u004D\u004D\u004D", [ "\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u043E\u043D\u044B\'\u0020\u004D\u004D\u004D\u004D\u0020\u0064" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u043E\u043D\'\u0020\u004D\u004D\u004D\u004D" ], [ "\u041D\u044F", "\u0414\u0430", "\u041C\u044F", "\u041B\u0445", "\u041F\u04AF", "\u0411\u0430", "\u0411\u044F" ], [ "\u041D\u044F\u043C", "\u0414\u0430\u0432\u0430\u0430", "\u041C\u044F\u0433\u043C\u0430\u0440", "\u041B\u0445\u0430\u0433\u0432\u0430", "\u041F\u04AF\u0440\u044D\u0432", "\u0411\u0430\u0430\u0441\u0430\u043D", "\u0411\u044F\u043C\u0431\u0430" ], [ "\u0049", "\u0049\u0049", "\u0049\u0049\u0049", "\u0049\u0056", "\u0056", "\u0056\u0049", "\u0056\u0049\u0049", "\u0056\u0428", "\u0049\u0058", "\u0058", "\u0058\u0049", "\u0058\u0049\u0049", "" ], [ "\u0031\u00A0\u0434\u04AF\u0433\u044D\u044D\u0440\u00A0\u0441\u0430\u0440", "\u0032\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0033\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0034\u00A0\u0434\u04AF\u0433\u044D\u044D\u0440\u00A0\u0441\u0430\u0440", "\u0035\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0036\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0037\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0038\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0039\u00A0\u0434\u04AF\u0433\u044D\u044D\u0440\u00A0\u0441\u0430\u0440", "\u0031\u0030\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "\u0031\u0031\u00A0\u0434\u04AF\u0433\u044D\u044D\u0440\u00A0\u0441\u0430\u0440", "\u0031\u0032\u00A0\u0434\u0443\u0433\u0430\u0430\u0440\u00A0\u0441\u0430\u0440", "" ] },
{ 0x0456, 0x0056, "gl-ES", "gl", "glg", "gl-ES", "\u0047\u0061\u006C\u0069\u0063\u0069\u0061\u006E\u0020\u0028\u0047\u0061\u006C\u0069\u0063\u0069\u0061\u006E\u0029", "\u0067\u0061\u006C\u0065\u0067\u006F\u0020\u0028\u0067\u0061\u006C\u0065\u0067\u006F\u0029", false, 0x00D9, "ES", "ESP", "\u0053\u0070\u0061\u0069\u006E", "\u0045\u0073\u0070\u0061\u00F1\u0061", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u005C\'\'" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u005C\'\'\u0073\u0073\'\u005C\'\u005C\'\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E\u0073", "\u006D\u0061\u0072", "\u006D\u00E9\u0072", "\u0078\u006F\u0076", "\u0076\u0065\u006E", "\u0073\u0061\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u00E9\u0072\u0063\u006F\u0072\u0065\u0073", "\u0078\u006F\u0076\u0065\u0073", "\u0076\u0065\u006E\u0072\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0078\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0069\u006F", "\u0078\u0075\u00F1", "\u0078\u0075\u006C\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0074", "\u006F\u0075\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u0078\u0061\u006E\u0065\u0069\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0069\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0069\u006F", "\u0078\u0075\u00F1\u006F", "\u0078\u0075\u006C\u006C\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0074\u0065\u006D\u0062\u0072\u006F", "\u006F\u0075\u0074\u0075\u0062\u0072\u006F", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u006F", "\u0064\u0065\u0063\u0065\u006D\u0062\u0072\u006F", "" ] },
{ 0x0457, 0x0057, "kok-IN", "kok", "kok", "kok-IN", "\u004B\u006F\u006E\u006B\u0061\u006E\u0069\u0020\u0028\u0049\u006E\u0064\u0069\u0061\u0029", "\u0915\u094B\u0902\u0915\u0923\u0940\u0020\u0028\u092D\u093E\u0930\u0924\u0029", false, 0x0071, "IN", "IND", "\u0049\u006E\u0064\u0069\u0061", "\u092D\u093E\u0930\u0924", "\u0049\u004E\u0052", "\u0049\u006E\u0064\u0069\u0061\u006E\u0020\u0052\u0075\u0070\u0065\u0065", "\u0930\u0941\u092A\u092F", true, 2, 1, 2, 12, 2, [ 3, 2 ], [ 3, 2 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0930\u0941", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0966", "\u0967", "\u0968", "\u0969", "\u096A", "\u096B", "\u096C", "\u096D", "\u096E", "\u096F" ], 1, [ 1 ], "\u0907\u0902\u0917\u094D\u0930\u091C\u0940\u00A0\u0915\u0945\u0932\u0947\u0928\u094D\u0921\u0930", 1, 0, "\u002D", "\u003A", "\u092E\u002E\u092A\u0942\u002E", "\u092E\u002E\u0928\u0902\u002E", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0906\u092F\u002E", "\u0938\u094B\u092E\u002E", "\u092E\u0902\u0917\u0933\u002E", "\u092C\u0941\u0927\u002E", "\u092C\u093F\u0930\u0947\u002E", "\u0938\u0941\u0915\u094D\u0930\u002E", "\u0936\u0947\u0928\u002E" ], [ "\u0906\u092F\u0924\u093E\u0930", "\u0938\u094B\u092E\u093E\u0930", "\u092E\u0902\u0917\u0933\u093E\u0930", "\u092C\u0941\u0927\u0935\u093E\u0930", "\u092C\u093F\u0930\u0947\u0938\u094D\u0924\u093E\u0930", "\u0938\u0941\u0915\u094D\u0930\u093E\u0930", "\u0936\u0947\u0928\u0935\u093E\u0930" ], [ "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940", "\u092B\u0947\u092C\u094D\u0930\u0941\u0935\u093E\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u090F\u092A\u094D\u0930\u093F\u0932", "\u092E\u0947", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u0948", "\u0911\u0917\u0938\u094D\u091F", "\u0938\u092A\u094D\u091F\u0947\u0902\u092C\u0930", "\u0911\u0915\u094D\u091F\u094B\u092C\u0930", "\u0928\u094B\u0935\u0947\u092E\u094D\u092C\u0930", "\u0921\u093F\u0938\u0947\u0902\u092C\u0930", "" ], [ "\u091C\u093E\u0928\u0947\u0935\u093E\u0930\u0940", "\u092B\u0947\u092C\u094D\u0930\u0941\u0935\u093E\u0930\u0940", "\u092E\u093E\u0930\u094D\u091A", "\u090F\u092A\u094D\u0930\u093F\u0932", "\u092E\u0947", "\u091C\u0942\u0928", "\u091C\u0941\u0932\u0948", "\u0911\u0917\u0938\u094D\u091F", "\u0938\u092A\u094D\u091F\u0947\u0902\u092C\u0930", "\u0911\u0915\u094D\u091F\u094B\u092C\u0930", "\u0928\u094B\u0935\u0947\u092E\u094D\u092C\u0930", "\u0921\u093F\u0938\u0947\u0902\u092C\u0930", "" ] },
{ 0x045A, 0x005A, "syr-SY", "syr", "syr", "syr-SY", "\u0053\u0079\u0072\u0069\u0061\u0063\u0020\u0028\u0053\u0079\u0072\u0069\u0061\u0029", "\u0723\u0718\u072A\u071D\u071D\u0710\u0020\u0028\u0633\u0648\u0631\u064A\u0627\u0029", false, 0x00DE, "SY", "SYR", "\u0053\u0079\u0072\u0069\u0061", "\u0633\u0648\u0631\u064A\u0627", "\u0053\u0059\u0050", "\u0053\u0079\u0072\u0069\u0061\u006E\u0020\u0050\u006F\u0075\u006E\u0064", "\u062C\u0646\u064A\u0647\u00A0\u0633\u0648\u0631\u064A", true, 2, 1, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0644\u002E\u0633\u002E\u200F", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0723\u0718\u072A\u0713\u0715\u0710\u00A0\u0713\u072A\u071D\u0713\u0718\u072A\u071D\u0710", 6, 0, "\u002F", "\u003A", "\u0729\u002E\u071B", "\u0712\u002E\u071B", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u070F\u0710\u00A0\u070F\u0712\u072B", "\u070F\u0712\u00A0\u070F\u0712\u072B", "\u070F\u0713\u00A0\u070F\u0712\u072B", "\u070F\u0715\u00A0\u070F\u0712\u072B", "\u070F\u0717\u00A0\u070F\u0712\u072B", "\u070F\u0725\u072A\u0718\u0712", "\u070F\u072B\u0712" ], [ "\u071A\u0715\u00A0\u0712\u072B\u0712\u0710", "\u072C\u072A\u071D\u0722\u00A0\u0712\u072B\u0712\u0710", "\u072C\u0720\u072C\u0710\u00A0\u0712\u072B\u0712\u0710", "\u0710\u072A\u0712\u0725\u0710\u00A0\u0712\u072B\u0712\u0710", "\u071A\u0721\u072B\u0710\u00A0\u0712\u072B\u0712\u0710", "\u0725\u072A\u0718\u0712\u072C\u0710", "\u072B\u0712\u072C\u0710" ], [ "\u070F\u071F\u0722\u00A0\u070F\u0712", "\u072B\u0712\u071B", "\u0710\u0715\u072A", "\u0722\u071D\u0723\u0722", "\u0710\u071D\u072A", "\u071A\u0719\u071D\u072A\u0722", "\u072C\u0721\u0718\u0719", "\u0710\u0712", "\u0710\u071D\u0720\u0718\u0720", "\u070F\u072C\u072B\u00A0\u070F\u0710", "\u070F\u072C\u072B\u00A0\u070F\u0712", "\u070F\u071F\u0722\u00A0\u070F\u0710", "" ], [ "\u071F\u0722\u0718\u0722\u00A0\u0710\u071A\u072A\u071D", "\u072B\u0712\u071B", "\u0710\u0715\u072A", "\u0722\u071D\u0723\u0722", "\u0710\u071D\u072A", "\u071A\u0719\u071D\u072A\u0722", "\u072C\u0721\u0718\u0719", "\u0710\u0712", "\u0710\u071D\u0720\u0718\u0720", "\u072C\u072B\u072A\u071D\u00A0\u0729\u0715\u071D\u0721", "\u072C\u072B\u072A\u071D\u00A0\u0710\u071A\u072A\u071D", "\u071F\u0722\u0718\u0722\u00A0\u0729\u0715\u071D\u0721", "" ] },
{ 0x0465, 0x0065, "div-MV", "div", "div", "div-MV", "\u0044\u0069\u0076\u0065\u0068\u0069\u0020\u0028\u004D\u0061\u006C\u0064\u0069\u0076\u0065\u0073\u0029", "\u078B\u07A8\u0788\u07AC\u0780\u07A8\u0784\u07A6\u0790\u07B0\u0020\u0028\u078B\u07A8\u0788\u07AC\u0780\u07A8\u0020\u0783\u07A7\u0787\u07B0\u0796\u07AC\u0029", false, 0x00A5, "MV", "MDV", "\u004D\u0061\u006C\u0064\u0069\u0076\u0065\u0073", "\u078B\u07A8\u0788\u07AC\u0780\u07A8\u0020\u0783\u07A7\u0787\u07B0\u0796\u07AC", "\u004D\u0056\u0052", "\u0052\u0075\u0066\u0069\u0079\u0061\u0061", "\u0783\u07AA\u078A\u07A8\u0794\u07A7", true, 2, 1, 2, 10, 3, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0783\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 6, [ 6, 1 ], "\u0780\u07A8\u0796\u07B0\u0783\u07A9\u0020\u0786\u07A6\u078D\u07A6\u0782\u07B0\u0791\u07A6\u0783\u07AA", 0, 0, "\u002F", "\u003A", "\u0789\u0786", "\u0789\u078A", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u002F\u004D\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u002F\u004D\u004D\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0645\u062D\u0631\u0645", "\u0635\u0641\u0631", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u0627\u0648\u0644", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u0627\u0648\u0644\u0649", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u062B\u0627\u0646\u064A\u0629", "\u0631\u062C\u0628", "\u0634\u0639\u0628\u0627\u0646", "\u0631\u0645\u0636\u0627\u0646", "\u0634\u0648\u0627\u0644", "\u0630\u0648\u00A0\u0627\u0644\u0642\u0639\u062F\u0629", "\u0630\u0648\u00A0\u0627\u0644\u062D\u062C\u0629", "" ], [ "\u0645\u062D\u0631\u0645", "\u0635\u0641\u0631", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u0623\u0648\u0644", "\u0631\u0628\u064A\u0639\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u0623\u0648\u0644\u0649", "\u062C\u0645\u0627\u062F\u0649\u00A0\u0627\u0644\u062B\u0627\u0646\u064A\u0629", "\u0631\u062C\u0628", "\u0634\u0639\u0628\u0627\u0646", "\u0631\u0645\u0636\u0627\u0646", "\u0634\u0648\u0627\u0644", "\u0630\u0648\u00A0\u0627\u0644\u0642\u0639\u062F\u0629", "\u0630\u0648\u00A0\u0627\u0644\u062D\u062C\u0629", "" ] },
{ 0x0801, 0x0001, "ar-IQ", "ar", "ara", "ar-IQ", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0049\u0072\u0061\u0071\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0639\u0631\u0627\u0642\u0029", false, 0x0079, "IQ", "IRQ", "\u0049\u0072\u0061\u0071", "\u0627\u0644\u0639\u0631\u0627\u0642", "\u0049\u0051\u0044", "\u0049\u0072\u0061\u0071\u0069\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u0639\u0631\u0627\u0642\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0639\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 11, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u0639\u0631\u0628\u064A\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ] },
{ 0x0804, 0x0004, "zh-CN", "zh", "zho", "zh-CN", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u0050\u0065\u006F\u0070\u006C\u0065\u0027\u0073\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u0043\u0068\u0069\u006E\u0061\u0029", "\u4E2D\u6587\u0028\u4E2D\u534E\u4EBA\u6C11\u5171\u548C\u56FD\u0029", false, 0x002D, "CN", "CHN", "\u0050\u0065\u006F\u0070\u006C\u0065\u0027\u0073\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u0043\u0068\u0069\u006E\u0061", "\u4E2D\u534E\u4EBA\u6C11\u5171\u548C\u56FD", "\u0043\u004E\u0059", "\u0050\u0052\u0043\u0020\u0059\u0075\u0061\u006E\u0020\u0052\u0065\u006E\u006D\u0069\u006E\u0062\u0069", "\u4EBA\u6C11\u5E01", true, 2, 1, 2, 2, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\uFFE5", "\u002D", "\u002B", "\u975E\u6570\u5B57", "\u8D1F\u65E0\u7A77\u5927", "\u6B63\u65E0\u7A77\u5927", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u516C\u5386", 0, 0, "\u002F", "\u003A", "\u4E0A\u5348", "\u4E0B\u5348", "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u002D\u0064", "\u0079\u0079\u0079\u0079\u002E\u004D\u002E\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064", "\u0079\u0079\u002D\u004D\u002D\u0064", "\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u002E\u004D\u002E\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'\u0064\'\u65E5\'" ], [ "\u0079\u0079\u0079\u0079\'\u5E74\'\u004D\'\u6708\'", "\u0079\u0079\u0079\u0079\u002E\u004D" ], [ "\u65E5", "\u4E00", "\u4E8C", "\u4E09", "\u56DB", "\u4E94", "\u516D" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ] },
{ 0x0807, 0x0007, "de-CH", "de", "deu", "de-CH", "\u0047\u0065\u0072\u006D\u0061\u006E\u0020\u0028\u0053\u0077\u0069\u0074\u007A\u0065\u0072\u006C\u0061\u006E\u0064\u0029", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u0020\u0028\u0053\u0063\u0068\u0077\u0065\u0069\u007A\u0029", false, 0x00DF, "CH", "CHE", "\u0053\u0077\u0069\u0074\u007A\u0065\u0072\u006C\u0061\u006E\u0064", "\u0053\u0063\u0068\u0077\u0065\u0069\u007A", "\u0043\u0048\u0046", "\u0053\u0077\u0069\u0073\u0073\u0020\u0046\u0072\u0061\u006E\u0063", "\u0053\u0063\u0068\u0077\u0065\u0069\u007A\u0065\u0072\u0020\u0046\u0072\u0061\u006E\u006B\u0065\u006E", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u0027", "\u002E", "\u0027", "\u0053\u0046\u0072\u002E", "\u002D", "\u002B", "\u006E\u002E\u0020\u0064\u0065\u0066\u002E", "\u002D\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", "\u002B\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0069\u0073\u0063\u0068\u0065\u0072\u0020\u004B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u0020\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u006F", "\u004D\u006F", "\u0044\u0069", "\u004D\u0069", "\u0044\u006F", "\u0046\u0072", "\u0053\u0061" ], [ "\u0053\u006F\u006E\u006E\u0074\u0061\u0067", "\u004D\u006F\u006E\u0074\u0061\u0067", "\u0044\u0069\u0065\u006E\u0073\u0074\u0061\u0067", "\u004D\u0069\u0074\u0074\u0077\u006F\u0063\u0068", "\u0044\u006F\u006E\u006E\u0065\u0072\u0073\u0074\u0061\u0067", "\u0046\u0072\u0065\u0069\u0074\u0061\u0067", "\u0053\u0061\u006D\u0073\u0074\u0061\u0067" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0072\u007A", "\u0041\u0070\u0072", "\u004D\u0061\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u007A", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072", "\u004D\u00E4\u0072\u007A", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0069", "\u004A\u0075\u006E\u0069", "\u004A\u0075\u006C\u0069", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u007A\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0809, 0x0009, "en-GB", "en", "eng", "en-GB", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u004B\u0069\u006E\u0067\u0064\u006F\u006D\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u004B\u0069\u006E\u0067\u0064\u006F\u006D\u0029", false, 0x00F2, "GB", "GBR", "\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u004B\u0069\u006E\u0067\u0064\u006F\u006D", "\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u004B\u0069\u006E\u0067\u0064\u006F\u006D", "\u0047\u0042\u0050", "\u0055\u004B\u0020\u0050\u006F\u0075\u006E\u0064\u0020\u0053\u0074\u0065\u0072\u006C\u0069\u006E\u0067", "\u0050\u006F\u0075\u006E\u0064\u0020\u0053\u0074\u0065\u0072\u006C\u0069\u006E\u0067", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u00A3", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x080A, 0x000A, "es-MX", "es", "spa", "es-MX", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u004D\u0065\u0078\u0069\u0063\u006F\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u004D\u00E9\u0078\u0069\u0063\u006F\u0029", false, 0x00A6, "MX", "MEX", "\u004D\u0065\u0078\u0069\u0063\u006F", "\u004D\u00E9\u0078\u0069\u0063\u006F", "\u004D\u0058\u004E", "\u004D\u0065\u0078\u0069\u0063\u0061\u006E\u0020\u0050\u0065\u0073\u006F", "\u0050\u0065\u0073\u006F", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x080C, 0x000C, "fr-BE", "fr", "fra", "fr-BE", "\u0046\u0072\u0065\u006E\u0063\u0068\u0020\u0028\u0042\u0065\u006C\u0067\u0069\u0075\u006D\u0029", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073\u0020\u0028\u0042\u0065\u006C\u0067\u0069\u0071\u0075\u0065\u0029", false, 0x0015, "BE", "BEL", "\u0042\u0065\u006C\u0067\u0069\u0075\u006D", "\u0042\u0065\u006C\u0067\u0069\u0071\u0075\u0065", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u004E\u0075\u006D\u00E9\u0072\u0069\u0071\u0075\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0069\u0065\u0072\u0020\u0067\u0072\u00E9\u0067\u006F\u0072\u0069\u0065\u006E", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D", "\u0048\'\u0020\u0068\u0020\'\u006D\u006D", "\u0048\'\u0020\u0068\u0020\'\u006D\'\u0020\u006D\u0069\u006E\u0020\'" ], [ "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D", "\u0048\'\u0020\u0068\u0020\'\u006D\u006D", "\u0048\'\u0020\u0068\u0020\'\u006D\'\u0020\u006D\u0069\u006E\u0020\'\u0073\'\u0020\u0073\u0020\'" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u0069\u006D\u002E", "\u006C\u0075\u006E\u002E", "\u006D\u0061\u0072\u002E", "\u006D\u0065\u0072\u002E", "\u006A\u0065\u0075\u002E", "\u0076\u0065\u006E\u002E", "\u0073\u0061\u006D\u002E" ], [ "\u0064\u0069\u006D\u0061\u006E\u0063\u0068\u0065", "\u006C\u0075\u006E\u0064\u0069", "\u006D\u0061\u0072\u0064\u0069", "\u006D\u0065\u0072\u0063\u0072\u0065\u0064\u0069", "\u006A\u0065\u0075\u0064\u0069", "\u0076\u0065\u006E\u0064\u0072\u0065\u0064\u0069", "\u0073\u0061\u006D\u0065\u0064\u0069" ], [ "\u006A\u0061\u006E\u0076\u002E", "\u0066\u00E9\u0076\u0072\u002E", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u002E", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u002E", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u00E9\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0076\u0069\u0065\u0072", "\u0066\u00E9\u0076\u0072\u0069\u0065\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u006C\u0065\u0074", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u00E9\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x0810, 0x0010, "it-CH", "it", "ita", "it-CH", "\u0049\u0074\u0061\u006C\u0069\u0061\u006E\u0020\u0028\u0053\u0077\u0069\u0074\u007A\u0065\u0072\u006C\u0061\u006E\u0064\u0029", "\u0069\u0074\u0061\u006C\u0069\u0061\u006E\u006F\u0020\u0028\u0053\u0076\u0069\u007A\u007A\u0065\u0072\u0061\u0029", false, 0x00DF, "CH", "CHE", "\u0053\u0077\u0069\u0074\u007A\u0065\u0072\u006C\u0061\u006E\u0064", "\u0053\u0076\u0069\u007A\u007A\u0065\u0072\u0061", "\u0043\u0048\u0046", "\u0053\u0077\u0069\u0073\u0073\u0020\u0046\u0072\u0061\u006E\u0063", "\u0046\u0072\u0061\u006E\u0063\u006F\u0020\u0073\u0076\u0069\u007A\u007A\u0065\u0072\u006F", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u0027", "\u002E", "\u0027", "\u0053\u0046\u0072\u002E", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u0075\u006E\u0020\u006E\u0075\u006D\u0065\u0072\u006F\u0020\u0072\u0065\u0061\u006C\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0068\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D\'\u0020\u0068\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0065\u0072", "\u0067\u0069\u006F", "\u0076\u0065\u006E", "\u0073\u0061\u0062" ], [ "\u0064\u006F\u006D\u0065\u006E\u0069\u0063\u0061", "\u006C\u0075\u006E\u0065\u0064\u00EC", "\u006D\u0061\u0072\u0074\u0065\u0064\u00EC", "\u006D\u0065\u0072\u0063\u006F\u006C\u0065\u0064\u00EC", "\u0067\u0069\u006F\u0076\u0065\u0064\u00EC", "\u0076\u0065\u006E\u0065\u0072\u0064\u00EC", "\u0073\u0061\u0062\u0061\u0074\u006F" ], [ "\u0067\u0065\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u0067", "\u0067\u0069\u006F", "\u006C\u0075\u0067", "\u0061\u0067\u006F", "\u0073\u0065\u0074", "\u006F\u0074\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0067\u0065\u006E\u006E\u0061\u0069\u006F", "\u0066\u0065\u0062\u0062\u0072\u0061\u0069\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0070\u0072\u0069\u006C\u0065", "\u006D\u0061\u0067\u0067\u0069\u006F", "\u0067\u0069\u0075\u0067\u006E\u006F", "\u006C\u0075\u0067\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0074\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0074\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x0813, 0x0013, "nl-BE", "nl", "nld", "nl-BE", "\u0044\u0075\u0074\u0063\u0068\u0020\u0028\u0042\u0065\u006C\u0067\u0069\u0075\u006D\u0029", "\u004E\u0065\u0064\u0065\u0072\u006C\u0061\u006E\u0064\u0073\u0020\u0028\u0042\u0065\u006C\u0067\u0069\u00EB\u0029", false, 0x0015, "BE", "BEL", "\u0042\u0065\u006C\u0067\u0069\u0075\u006D", "\u0042\u0065\u006C\u0067\u0069\u00EB", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E\u0020\u0028\u004E\u0069\u0065\u0074\u002D\u0065\u0065\u006E\u002D\u0067\u0065\u0074\u0061\u006C\u0029", "\u002D\u006F\u006E\u0065\u0069\u006E\u0064\u0069\u0067", "\u006F\u006E\u0065\u0069\u006E\u0064\u0069\u0067", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0073\u0065\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0075\u002E\'" ], [ "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D\'\u0020\u0075\u002E\'", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u007A\u006F", "\u006D\u0061", "\u0064\u0069", "\u0077\u006F", "\u0064\u006F", "\u0076\u0072", "\u007A\u0061" ], [ "\u007A\u006F\u006E\u0064\u0061\u0067", "\u006D\u0061\u0061\u006E\u0064\u0061\u0067", "\u0064\u0069\u006E\u0073\u0064\u0061\u0067", "\u0077\u006F\u0065\u006E\u0073\u0064\u0061\u0067", "\u0064\u006F\u006E\u0064\u0065\u0072\u0064\u0061\u0067", "\u0076\u0072\u0069\u006A\u0064\u0061\u0067", "\u007A\u0061\u0074\u0065\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0072\u0074", "\u0061\u0070\u0072", "\u006D\u0065\u0069", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u006D\u0061\u0061\u0072\u0074", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0065\u0069", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074\u0075\u0073", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0814, 0x0014, "nn-NO", "nn", "nno", "nn-NO", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u002C\u0020\u004E\u0079\u006E\u006F\u0072\u0073\u006B\u0020\u0028\u004E\u006F\u0072\u0077\u0061\u0079\u0029", "\u006E\u006F\u0072\u0073\u006B\u002C\u0020\u006E\u0079\u006E\u006F\u0072\u0073\u006B\u0020\u0028\u004E\u006F\u0072\u0065\u0067\u0029", false, 0x00B1, "NO", "NOR", "\u004E\u006F\u0072\u0077\u0061\u0079", "\u004E\u006F\u0072\u0065\u0067", "\u004E\u004F\u004B", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u0020\u004B\u0072\u006F\u006E\u0065", "\u004E\u006F\u0072\u0073\u006B\u0020\u006B\u0072\u006F\u006E\u0065", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\'\u006B\u006C\u0020\'\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\'\u006B\u006C\u0020\'\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u00F8", "\u006D\u00E5", "\u0074\u0079", "\u006F\u006E", "\u0074\u006F", "\u0066\u0072", "\u006C\u0061" ], [ "\u0073\u00F8\u006E\u0064\u0061\u0067", "\u006D\u00E5\u006E\u0064\u0061\u0067", "\u0074\u0079\u0073\u0064\u0061\u0067", "\u006F\u006E\u0073\u0064\u0061\u0067", "\u0074\u006F\u0072\u0073\u0064\u0061\u0067", "\u0066\u0072\u0065\u0064\u0061\u0067", "\u006C\u0061\u0075\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u0069", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0073", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0816, 0x0016, "pt-PT", "pt", "por", "pt-PT", "\u0050\u006F\u0072\u0074\u0075\u0067\u0075\u0065\u0073\u0065\u0020\u0028\u0050\u006F\u0072\u0074\u0075\u0067\u0061\u006C\u0029", "\u0070\u006F\u0072\u0074\u0075\u0067\u0075\u00EA\u0073\u0020\u0028\u0050\u006F\u0072\u0074\u0075\u0067\u0061\u006C\u0029", false, 0x00C1, "PT", "PRT", "\u0050\u006F\u0072\u0074\u0075\u0067\u0061\u006C", "\u0050\u006F\u0072\u0074\u0075\u0067\u0061\u006C", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E\u0020\u0028\u004E\u00E3\u006F\u0020\u00E9\u0020\u0075\u006D\u0020\u006E\u00FA\u006D\u0065\u0072\u006F\u0029", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0043\u0061\u006C\u0065\u006E\u0064\u00E1\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 0, "\u002D", "\u003A", "", "", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002F\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u006D\'" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u002E\u004D\u004D\u002E\u0064\u0064", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u006D\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u004D\u002F\u0079\u0079", "\u0064\u002E\u004D\u004D\u004D\u002E\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u0073\u0065\u0067", "\u0074\u0065\u0072", "\u0071\u0075\u0061", "\u0071\u0075\u0069", "\u0073\u0065\u0078", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u0073\u0065\u0067\u0075\u006E\u0064\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0074\u0065\u0072\u00E7\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0071\u0075\u0061\u0072\u0074\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0071\u0075\u0069\u006E\u0074\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0073\u0065\u0078\u0074\u0061\u002D\u0066\u0065\u0069\u0072\u0061", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0076", "\u004D\u0061\u0072", "\u0041\u0062\u0072", "\u004D\u0061\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0067\u006F", "\u0053\u0065\u0074", "\u004F\u0075\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u007A", "" ], [ "\u004A\u0061\u006E\u0065\u0069\u0072\u006F", "\u0046\u0065\u0076\u0065\u0072\u0065\u0069\u0072\u006F", "\u004D\u0061\u0072\u00E7\u006F", "\u0041\u0062\u0072\u0069\u006C", "\u004D\u0061\u0069\u006F", "\u004A\u0075\u006E\u0068\u006F", "\u004A\u0075\u006C\u0068\u006F", "\u0041\u0067\u006F\u0073\u0074\u006F", "\u0053\u0065\u0074\u0065\u006D\u0062\u0072\u006F", "\u004F\u0075\u0074\u0075\u0062\u0072\u006F", "\u004E\u006F\u0076\u0065\u006D\u0062\u0072\u006F", "\u0044\u0065\u007A\u0065\u006D\u0062\u0072\u006F", "" ] },
{ 0x081A, 0x7C1A, "sr-SP-Latn", "sr", "srp", "sr-Latn-SP", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E\u0020\u0028\u004C\u0061\u0074\u0069\u006E\u002C\u0020\u0053\u0065\u0072\u0062\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u004D\u006F\u006E\u0074\u0065\u006E\u0065\u0067\u0072\u006F\u0029", "\u0073\u0072\u0070\u0073\u006B\u0069\u0020\u0028\u0053\u0072\u0062\u0069\u006A\u0061\u0020\u0069\u0020\u0043\u0072\u006E\u0061\u0020\u0047\u006F\u0072\u0061\u0029", false, 0x010D, "SP", "SPB", "\u0053\u0065\u0072\u0062\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u004D\u006F\u006E\u0074\u0065\u006E\u0065\u0067\u0072\u006F", "\u0053\u0072\u0062\u0069\u006A\u0061\u0020\u0069\u0020\u0043\u0072\u006E\u0061\u0020\u0047\u006F\u0072\u0061", "\u0043\u0053\u0044", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E\u0020\u0044\u0069\u006E\u0061\u0072", "\u0064\u0069\u006E\u0061\u0072", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0044\u0069\u006E\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0062\u0065\u0073\u006B\u006F\u006E\u0061\u010D\u006E\u006F\u0073\u0074", "\u002B\u0062\u0065\u0073\u006B\u006F\u006E\u0061\u010D\u006E\u006F\u0073\u0074", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006A\u0061\u006E\u0073\u006B\u0069\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065\u0064", "\u0070\u006F\u006E", "\u0075\u0074\u006F", "\u0073\u0072\u0065", "\u010D\u0065\u0074", "\u0070\u0065\u0074", "\u0073\u0075\u0062" ], [ "\u006E\u0065\u0064\u0065\u006C\u006A\u0061", "\u0070\u006F\u006E\u0065\u0064\u0065\u006C\u006A\u0061\u006B", "\u0075\u0074\u006F\u0072\u0061\u006B", "\u0073\u0072\u0065\u0064\u0061", "\u010D\u0065\u0074\u0076\u0072\u0074\u0061\u006B", "\u0070\u0065\u0074\u0061\u006B", "\u0073\u0075\u0062\u006F\u0074\u0061" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0074", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0061\u0072", "\u006F\u006B\u0074\u006F\u0062\u0061\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0061\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0061\u0072", "" ] },
{ 0x081D, 0x001D, "sv-FI", "sv", "swe", "sv-FI", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068\u0020\u0028\u0046\u0069\u006E\u006C\u0061\u006E\u0064\u0029", "\u0073\u0076\u0065\u006E\u0073\u006B\u0061\u0020\u0028\u0046\u0069\u006E\u006C\u0061\u006E\u0064\u0029", false, 0x004D, "FI", "FIN", "\u0046\u0069\u006E\u006C\u0061\u006E\u0064", "\u0046\u0069\u006E\u006C\u0061\u006E\u0064", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u004E\u0046", "\u0049\u004E\u0046", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\'\u0064\u0065\u006E\u0020\'\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\'\u0064\u0065\u006E\u0020\'\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\'\u006B\u006C\u0020\'\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\'\u006B\u006C\u0020\'\u0048\u003A\u006D\u006D" ], [ "\'\u0064\u0065\u006E\u0020\'\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u00F6", "\u006D\u00E5", "\u0074\u0069", "\u006F\u006E", "\u0074\u006F", "\u0066\u0072", "\u006C\u00F6" ], [ "\u0073\u00F6\u006E\u0064\u0061\u0067", "\u006D\u00E5\u006E\u0064\u0061\u0067", "\u0074\u0069\u0073\u0064\u0061\u0067", "\u006F\u006E\u0073\u0064\u0061\u0067", "\u0074\u006F\u0072\u0073\u0064\u0061\u0067", "\u0066\u0072\u0065\u0064\u0061\u0067", "\u006C\u00F6\u0072\u0064\u0061\u0067" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0075\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u006D\u0061\u0072\u0073", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E\u0069", "\u006A\u0075\u006C\u0069", "\u0061\u0075\u0067\u0075\u0073\u0074\u0069", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u006F\u006B\u0074\u006F\u0062\u0065\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x082C, 0x002C, "az-AZ-Cyrl", "az", "aze", "az-Cyrl-AZ", "\u0041\u007A\u0065\u0072\u0069\u0020\u0028\u0043\u0079\u0072\u0069\u006C\u006C\u0069\u0063\u002C\u0020\u0041\u007A\u0065\u0072\u0062\u0061\u0069\u006A\u0061\u006E\u0029", "\u0410\u0437\u04D9\u0440\u0431\u0430\u0458\u04B9\u0430\u043D\u0020\u0028\u0410\u0437\u04D9\u0440\u0431\u0430\u0458\u04B9\u0430\u043D\u0029", false, 0x0005, "AZ", "AZE", "\u0041\u007A\u0065\u0072\u0062\u0061\u0069\u006A\u0061\u006E", "\u0410\u0437\u04D9\u0440\u0431\u0430\u0458\u04B9\u0430\u043D", "\u0041\u005A\u004D", "\u0041\u007A\u0065\u0072\u0062\u0061\u0069\u006A\u0061\u006E\u0069\u0061\u006E\u0020\u004D\u0061\u006E\u0061\u0074", "\u0440\u0443\u0431\u043B\u044C", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u043C\u0430\u043D\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0413\u0440\u0438\u0443\u043E\u0440\u0438\u0430\u043D", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0411", "\u0411\u0435", "\u0427\u0430", "\u0427", "\u04B8\u0430", "\u04B8", "\u0428" ], [ "\u0411\u0430\u0437\u0430\u0440", "\u0411\u0430\u0437\u0430\u0440\u00A0\u0435\u0440\u0442\u04D9\u0441\u0438", "\u0427\u04D9\u0440\u0448\u04D9\u043D\u0431\u04D9\u00A0\u0430\u0445\u0448\u0430\u043C\u044B", "\u0427\u04D9\u0440\u0448\u04D9\u043D\u0431\u04D9", "\u04B8\u04AF\u043C\u04D9\u00A0\u0430\u0445\u0448\u0430\u043C\u044B", "\u04B8\u04AF\u043C\u04D9", "\u0428\u04D9\u043D\u0431\u04D9" ], [ "\u0408\u0430\u043D", "\u0424\u0435\u0432", "\u041C\u0430\u0440", "\u0410\u043F\u0440", "\u041C\u0430\u0458", "\u0418\u0458\u0443\u043D", "\u0418\u0458\u0443\u043B", "\u0410\u0432\u0433", "\u0421\u0435\u043D", "\u041E\u043A\u0442", "\u041D\u043E\u044F", "\u0414\u0435\u043A", "" ], [ "\u0408\u0430\u043D\u0432\u0430\u0440", "\u0424\u0435\u0432\u0440\u0430\u043B", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0435\u043B", "\u041C\u0430\u0458", "\u0418\u0458\u0443\u043D", "\u0418\u0458\u0443\u043B", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043D\u0442\u0458\u0430\u0431\u0440", "\u041E\u043A\u0442\u0458\u0430\u0431\u0440", "\u041D\u043E\u0458\u0430\u0431\u0440", "\u0414\u0435\u043A\u0430\u0431\u0440", "" ] },
{ 0x083E, 0x003E, "ms-BN", "ms", "msa", "ms-BN", "\u004D\u0061\u006C\u0061\u0079\u0020\u0028\u0042\u0072\u0075\u006E\u0065\u0069\u0020\u0044\u0061\u0072\u0075\u0073\u0073\u0061\u006C\u0061\u006D\u0029", "\u0042\u0061\u0068\u0061\u0073\u0061\u0020\u004D\u0061\u006C\u0061\u0079\u0073\u0069\u0061\u0020\u0028\u0042\u0072\u0075\u006E\u0065\u0069\u0020\u0044\u0061\u0072\u0075\u0073\u0073\u0061\u006C\u0061\u006D\u0029", false, 0x0025, "BN", "BRN", "\u0042\u0072\u0075\u006E\u0065\u0069\u0020\u0044\u0061\u0072\u0075\u0073\u0073\u0061\u006C\u0061\u006D", "\u0042\u0072\u0075\u006E\u0065\u0069\u0020\u0044\u0061\u0072\u0075\u0073\u0073\u0061\u006C\u0061\u006D", "\u0042\u004E\u0044", "\u0042\u0072\u0075\u006E\u0065\u0069\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0052\u0069\u006E\u0067\u0067\u0069\u0074\u0020\u0042\u0072\u0075\u006E\u0065\u0069", true, 2, 1, 0, 0, 0, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0020\u004D\u0061\u0073\u0065\u0068\u0069", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0041\u0068\u0061\u0064", "\u0049\u0073\u006E\u0069\u006E", "\u0053\u0065\u006C", "\u0052\u0061\u0062\u0075", "\u004B\u0068\u0061\u006D\u0069\u0073", "\u004A\u0075\u006D\u0061\u0061\u0074", "\u0053\u0061\u0062\u0074\u0075" ], [ "\u0041\u0068\u0061\u0064", "\u0049\u0073\u006E\u0069\u006E", "\u0053\u0065\u006C\u0061\u0073\u0061", "\u0052\u0061\u0062\u0075", "\u004B\u0068\u0061\u006D\u0069\u0073", "\u004A\u0075\u006D\u0061\u0061\u0074", "\u0053\u0061\u0062\u0074\u0075" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0063", "\u0041\u0070\u0072", "\u004D\u0065\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u004F\u0067\u006F\u0073", "\u0053\u0065\u0070\u0074", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0069\u0073", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0069", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0069", "\u004D\u0061\u0063", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0065\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C\u0061\u0069", "\u004F\u0067\u006F\u0073", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0069\u0073\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0843, 0x0043, "uz-UZ-Cyrl", "uz", "uzb", "uz-Cyrl-UZ", "\u0055\u007A\u0062\u0065\u006B\u0020\u0028\u0043\u0079\u0072\u0069\u006C\u006C\u0069\u0063\u002C\u0020\u0055\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u0061\u006E\u0029", "\u040E\u0437\u0431\u0435\u043A\u0020\u0028\u040E\u0437\u0431\u0435\u043A\u0438\u0441\u0442\u043E\u043D\u0029", false, 0x00F7, "UZ", "UZB", "\u0055\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u0061\u006E", "\u040E\u0437\u0431\u0435\u043A\u0438\u0441\u0442\u043E\u043D\u00A0\u0420\u0435\u0441\u043F\u0443\u0431\u043B\u0438\u043A\u0430\u0441\u0438", "\u0055\u005A\u0053", "\u0055\u007A\u0062\u0065\u006B\u0069\u0073\u0074\u0061\u006E\u0020\u0053\u0075\u006D", "\u0440\u0443\u0431\u043B\u044C", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0441\u045E\u043C", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0438\u0433\u043E\u0440\u0438\u0430\u043D\u0441\u043A\u0438\u0439\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440\u044C", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0079\u0079\u0079\u0079\u0020\'\u0439\u0438\u043B\'\u0020\u0064\u002D\u004D\u004D\u004D\u004D", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002D\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u0020\'\u0439\u0438\u043B\'\u0020\u0064\u002D\u004D\u004D\u004D\u004D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u044F\u043A\u0448", "\u0434\u0448", "\u0441\u0448", "\u0447\u0448", "\u043F\u0448", "\u0436", "\u0448" ], [ "\u044F\u043A\u0448\u0430\u043D\u0431\u0430", "\u0434\u0443\u0448\u0430\u043D\u0431\u0430", "\u0441\u0435\u0448\u0430\u043D\u0431\u0430", "\u0447\u043E\u0440\u0448\u0430\u043D\u0431\u0430", "\u043F\u0430\u0439\u0448\u0430\u043D\u0431\u0430", "\u0436\u0443\u043C\u0430", "\u0448\u0430\u043D\u0431\u0430" ], [ "\u042F\u043D\u0432", "\u0424\u0435\u0432", "\u041C\u0430\u0440", "\u0410\u043F\u0440", "\u041C\u0430\u0439", "\u0418\u044E\u043D", "\u0418\u044E\u043B", "\u0410\u0432\u0433", "\u0421\u0435\u043D", "\u041E\u043A\u0442", "\u041D\u043E\u044F", "\u0414\u0435\u043A", "" ], [ "\u042F\u043D\u0432\u0430\u0440", "\u0424\u0435\u0432\u0440\u0430\u043B", "\u041C\u0430\u0440\u0442", "\u0410\u043F\u0440\u0435\u043B", "\u041C\u0430\u0439", "\u0418\u044E\u043D", "\u0418\u044E\u043B", "\u0410\u0432\u0433\u0443\u0441\u0442", "\u0421\u0435\u043D\u0442\u044F\u0431\u0440", "\u041E\u043A\u0442\u044F\u0431\u0440", "\u041D\u043E\u044F\u0431\u0440", "\u0414\u0435\u043A\u0430\u0431\u0440", "" ] },
{ 0x0C01, 0x0001, "ar-EG", "ar", "ara", "ar-EG", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0045\u0067\u0079\u0070\u0074\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0645\u0635\u0631\u0029", false, 0x0043, "EG", "EGY", "\u0045\u0067\u0079\u0070\u0074", "\u0645\u0635\u0631", "\u0045\u0047\u0050", "\u0045\u0067\u0079\u0070\u0074\u0069\u0061\u006E\u0020\u0050\u006F\u0075\u006E\u0064", "\u062C\u0646\u064A\u0647\u00A0\u0645\u0635\u0631\u064A", true, 3, 3, 3, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062C\u002E\u0645\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x0C04, 0x7C04, "zh-HK", "zh", "zho", "zh-HK", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u0048\u006F\u006E\u0067\u0020\u004B\u006F\u006E\u0067\u0020\u0053\u002E\u0041\u002E\u0052\u002E\u0029", "\u4E2D\u6587\u0028\u9999\u6E2F\u7279\u522B\u884C\u653F\u5340\u0029", false, 0x0068, "HK", "HKG", "\u0048\u006F\u006E\u0067\u0020\u004B\u006F\u006E\u0067\u0020\u0053\u002E\u0041\u002E\u0052\u002E", "\u9999\u6E2F\u7279\u522B\u884C\u653F\u5340", "\u0048\u004B\u0044", "\u0048\u006F\u006E\u0067\u0020\u004B\u006F\u006E\u0067\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u6E2F\u5E63", true, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0048\u004B\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u65E5\u66C6", 0, 0, "\u002F", "\u003A", "", "", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064", "\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0C07, 0x0007, "de-AT", "de", "deu", "de-AT", "\u0047\u0065\u0072\u006D\u0061\u006E\u0020\u0028\u0041\u0075\u0073\u0074\u0072\u0069\u0061\u0029", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u0020\u0028\u00D6\u0073\u0074\u0065\u0072\u0072\u0065\u0069\u0063\u0068\u0029", false, 0x000E, "AT", "AUT", "\u0041\u0075\u0073\u0074\u0072\u0069\u0061", "\u00D6\u0073\u0074\u0065\u0072\u0072\u0065\u0069\u0063\u0068", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0045\u0075\u0072\u006F", true, 2, 1, 2, 9, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u006E\u002E\u0020\u0064\u0065\u0066\u002E", "\u002D\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", "\u002B\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0069\u0073\u0063\u0068\u0065\u0072\u0020\u004B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D\'\u0020\u0055\u0068\u0072\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D\'\u0020\u0055\u0068\u0072\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u004D\u004D\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u006F", "\u004D\u006F", "\u0044\u0069", "\u004D\u0069", "\u0044\u006F", "\u0046\u0072", "\u0053\u0061" ], [ "\u0053\u006F\u006E\u006E\u0074\u0061\u0067", "\u004D\u006F\u006E\u0074\u0061\u0067", "\u0044\u0069\u0065\u006E\u0073\u0074\u0061\u0067", "\u004D\u0069\u0074\u0074\u0077\u006F\u0063\u0068", "\u0044\u006F\u006E\u006E\u0065\u0072\u0073\u0074\u0061\u0067", "\u0046\u0072\u0065\u0069\u0074\u0061\u0067", "\u0053\u0061\u006D\u0073\u0074\u0061\u0067" ], [ "\u004A\u00E4\u006E", "\u0046\u0065\u0062", "\u004D\u00E4\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u007A", "" ], [ "\u004A\u00E4\u006E\u006E\u0065\u0072", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072", "\u004D\u00E4\u0072\u007A", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0069", "\u004A\u0075\u006E\u0069", "\u004A\u0075\u006C\u0069", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u007A\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0C09, 0x0009, "en-AU", "en", "eng", "en-AU", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0041\u0075\u0073\u0074\u0072\u0061\u006C\u0069\u0061\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0041\u0075\u0073\u0074\u0072\u0061\u006C\u0069\u0061\u0029", false, 0x000C, "AU", "AUS", "\u0041\u0075\u0073\u0074\u0072\u0061\u006C\u0069\u0061", "\u0041\u0075\u0073\u0074\u0072\u0061\u006C\u0069\u0061", "\u0041\u0055\u0044", "\u0041\u0075\u0073\u0074\u0072\u0061\u006C\u0069\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0041\u0075\u0073\u0074\u0072\u0061\u006C\u0069\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x0C0A, 0x000A, "es-ES", "es", "spa", "es-ES", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0053\u0070\u0061\u0069\u006E\u0029", "\u0065\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0045\u0073\u0070\u0061\u00F1\u0061\u0029", false, 0x00D9, "ES", "ESP", "\u0053\u0070\u0061\u0069\u006E", "\u0045\u0073\u0070\u0061\u00F1\u0061", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u005C\'\'" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u0048\'\u0048\'\u006D\u006D\'\u005C\'\'\u0073\u0073\'\u005C\'\u005C\'\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x0C0C, 0x000C, "fr-CA", "fr", "fra", "fr-CA", "\u0046\u0072\u0065\u006E\u0063\u0068\u0020\u0028\u0043\u0061\u006E\u0061\u0064\u0061\u0029", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073\u0020\u0028\u0043\u0061\u006E\u0061\u0064\u0061\u0029", false, 0x0027, "CA", "CAN", "\u0043\u0061\u006E\u0061\u0064\u0061", "\u0043\u0061\u006E\u0061\u0064\u0061", "\u0043\u0041\u0044", "\u0043\u0061\u006E\u0061\u0064\u0069\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0044\u006F\u006C\u006C\u0061\u0072\u0020\u0063\u0061\u006E\u0061\u0064\u0069\u0065\u006E", true, 2, 1, 2, 15, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u0024", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u004E\u0075\u006D\u00E9\u0072\u0069\u0071\u0075\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0069\u0065\u0072\u0020\u0067\u0072\u00E9\u0067\u006F\u0072\u0069\u0065\u006E", 0, 0, "\u002D", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\'\u0020\u0068\u0020\'\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u0069\u006D\u002E", "\u006C\u0075\u006E\u002E", "\u006D\u0061\u0072\u002E", "\u006D\u0065\u0072\u002E", "\u006A\u0065\u0075\u002E", "\u0076\u0065\u006E\u002E", "\u0073\u0061\u006D\u002E" ], [ "\u0064\u0069\u006D\u0061\u006E\u0063\u0068\u0065", "\u006C\u0075\u006E\u0064\u0069", "\u006D\u0061\u0072\u0064\u0069", "\u006D\u0065\u0072\u0063\u0072\u0065\u0064\u0069", "\u006A\u0065\u0075\u0064\u0069", "\u0076\u0065\u006E\u0064\u0072\u0065\u0064\u0069", "\u0073\u0061\u006D\u0065\u0064\u0069" ], [ "\u006A\u0061\u006E\u0076\u002E", "\u0066\u00E9\u0076\u0072\u002E", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u002E", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u002E", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u00E9\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0076\u0069\u0065\u0072", "\u0066\u00E9\u0076\u0072\u0069\u0065\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u006C\u0065\u0074", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u00E9\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x0C1A, 0x7C1A, "sr-SP-Cyrl", "sr", "srp", "sr-Cyrl-SP", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E\u0020\u0028\u0043\u0079\u0072\u0069\u006C\u006C\u0069\u0063\u002C\u0020\u0053\u0065\u0072\u0062\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u004D\u006F\u006E\u0074\u0065\u006E\u0065\u0067\u0072\u006F\u0029", "\u0441\u0440\u043F\u0441\u043A\u0438\u0020\u0028\u0421\u0440\u0431\u0438\u0458\u0430\u00A0\u0438\u00A0\u0426\u0440\u043D\u0430\u00A0\u0413\u043E\u0440\u0430\u0029", false, 0x010D, "SP", "SPB", "\u0053\u0065\u0072\u0062\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u004D\u006F\u006E\u0074\u0065\u006E\u0065\u0067\u0072\u006F", "\u0421\u0440\u0431\u0438\u0458\u0430\u00A0\u0438\u00A0\u0426\u0440\u043D\u0430\u00A0\u0413\u043E\u0440\u0430", "\u0043\u0053\u0044", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E\u0020\u0044\u0069\u006E\u0061\u0072", "\u0434\u0438\u043D\u0430\u0440", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0414\u0438\u043D\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0431\u0435\u0441\u043A\u043E\u043D\u0430\u0447\u043D\u043E\u0441\u0442", "\u002B\u0431\u0435\u0441\u043A\u043E\u043D\u0430\u0447\u043D\u043E\u0441\u0442", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0435\u0433\u043E\u0440\u0438\u0458\u0430\u043D\u0441\u043A\u0438\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440", 1, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u043D\u0435\u0434", "\u043F\u043E\u043D", "\u0443\u0442\u043E", "\u0441\u0440\u0435", "\u0447\u0435\u0442", "\u043F\u0435\u0442", "\u0441\u0443\u0431" ], [ "\u043D\u0435\u0434\u0435\u0459\u0430", "\u043F\u043E\u043D\u0435\u0434\u0435\u0459\u0430\u043A", "\u0443\u0442\u043E\u0440\u0430\u043A", "\u0441\u0440\u0435\u0434\u0430", "\u0447\u0435\u0442\u0432\u0440\u0442\u0430\u043A", "\u043F\u0435\u0442\u0430\u043A", "\u0441\u0443\u0431\u043E\u0442\u0430" ], [ "\u0458\u0430\u043D", "\u0444\u0435\u0431", "\u043C\u0430\u0440", "\u0430\u043F\u0440", "\u043C\u0430\u0458", "\u0458\u0443\u043D", "\u0458\u0443\u043B", "\u0430\u0432\u0433", "\u0441\u0435\u043F", "\u043E\u043A\u0442", "\u043D\u043E\u0432", "\u0434\u0435\u0446", "" ], [ "\u0458\u0430\u043D\u0443\u0430\u0440", "\u0444\u0435\u0431\u0440\u0443\u0430\u0440", "\u043C\u0430\u0440\u0442", "\u0430\u043F\u0440\u0438\u043B", "\u043C\u0430\u0458", "\u0458\u0443\u043D", "\u0458\u0443\u043B", "\u0430\u0432\u0433\u0443\u0441\u0442", "\u0441\u0435\u043F\u0442\u0435\u043C\u0431\u0430\u0440", "\u043E\u043A\u0442\u043E\u0431\u0430\u0440", "\u043D\u043E\u0432\u0435\u043C\u0431\u0430\u0440", "\u0434\u0435\u0446\u0435\u043C\u0431\u0430\u0440", "" ] },
{ 0x1001, 0x0001, "ar-LY", "ar", "ara", "ar-LY", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u004C\u0069\u0062\u0079\u0061\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0644\u064A\u0628\u064A\u0627\u0029", false, 0x0094, "LY", "LBY", "\u004C\u0069\u0062\u0079\u0061", "\u0644\u064A\u0628\u064A\u0627", "\u004C\u0059\u0044", "\u004C\u0069\u0062\u0079\u0061\u006E\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u0644\u064A\u0628\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0644\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 6, 23, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x1004, 0x0004, "zh-SG", "zh", "zho", "zh-SG", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u0053\u0069\u006E\u0067\u0061\u0070\u006F\u0072\u0065\u0029", "\u4E2D\u6587\u0028\u65B0\u52A0\u5761\u0029", false, 0x00D7, "SG", "SGP", "\u0053\u0069\u006E\u0067\u0061\u0070\u006F\u0072\u0065", "\u65B0\u52A0\u5761", "\u0053\u0047\u0044", "\u0053\u0069\u006E\u0067\u0061\u0070\u006F\u0072\u0065\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u65B0\u5E01", false, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u897F\u5386", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0074\u0074\u0020\u0068\u003A\u006D\u006D", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0074\u0074\u0020\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0074\u0074\u0020\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064", "\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ] },
{ 0x1007, 0x0007, "de-LU", "de", "deu", "de-LU", "\u0047\u0065\u0072\u006D\u0061\u006E\u0020\u0028\u004C\u0075\u0078\u0065\u006D\u0062\u006F\u0075\u0072\u0067\u0029", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u0020\u0028\u004C\u0075\u0078\u0065\u006D\u0062\u0075\u0072\u0067\u0029", false, 0x0093, "LU", "LUX", "\u004C\u0075\u0078\u0065\u006D\u0062\u006F\u0075\u0072\u0067", "\u004C\u0075\u0078\u0065\u006D\u0062\u0075\u0072\u0067", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0045\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20AC", "\u002D", "\u002B", "\u006E\u002E\u0020\u0064\u0065\u0066\u002E", "\u002D\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", "\u002B\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0069\u0073\u0063\u0068\u0065\u0072\u0020\u004B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\u0020\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\u0020\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u006F", "\u004D\u006F", "\u0044\u0069", "\u004D\u0069", "\u0044\u006F", "\u0046\u0072", "\u0053\u0061" ], [ "\u0053\u006F\u006E\u006E\u0074\u0061\u0067", "\u004D\u006F\u006E\u0074\u0061\u0067", "\u0044\u0069\u0065\u006E\u0073\u0074\u0061\u0067", "\u004D\u0069\u0074\u0074\u0077\u006F\u0063\u0068", "\u0044\u006F\u006E\u006E\u0065\u0072\u0073\u0074\u0061\u0067", "\u0046\u0072\u0065\u0069\u0074\u0061\u0067", "\u0053\u0061\u006D\u0073\u0074\u0061\u0067" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0072\u007A", "\u0041\u0070\u0072", "\u004D\u0061\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u007A", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072", "\u004D\u00E4\u0072\u007A", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0069", "\u004A\u0075\u006E\u0069", "\u004A\u0075\u006C\u0069", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u007A\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x1009, 0x0009, "en-CA", "en", "eng", "en-CA", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0043\u0061\u006E\u0061\u0064\u0061\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0043\u0061\u006E\u0061\u0064\u0061\u0029", false, 0x0027, "CA", "CAN", "\u0043\u0061\u006E\u0061\u0064\u0061", "\u0043\u0061\u006E\u0061\u0064\u0061", "\u0043\u0041\u0044", "\u0043\u0061\u006E\u0061\u0064\u0069\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0043\u0061\u006E\u0061\u0064\u0069\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u004D\u002F\u0064\u0064\u002F\u0079\u0079" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x100A, 0x000A, "es-GT", "es", "spa", "es-GT", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0047\u0075\u0061\u0074\u0065\u006D\u0061\u006C\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0047\u0075\u0061\u0074\u0065\u006D\u0061\u006C\u0061\u0029", false, 0x0063, "GT", "GTM", "\u0047\u0075\u0061\u0074\u0065\u006D\u0061\u006C\u0061", "\u0047\u0075\u0061\u0074\u0065\u006D\u0061\u006C\u0061", "\u0047\u0054\u0051", "\u0047\u0075\u0061\u0074\u0065\u006D\u0061\u006C\u0061\u006E\u0020\u0051\u0075\u0065\u0074\u007A\u0061\u006C", "\u0051\u0075\u0065\u0074\u007A\u0061\u006C", true, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0051", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x100C, 0x000C, "fr-CH", "fr", "fra", "fr-CH", "\u0046\u0072\u0065\u006E\u0063\u0068\u0020\u0028\u0053\u0077\u0069\u0074\u007A\u0065\u0072\u006C\u0061\u006E\u0064\u0029", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073\u0020\u0028\u0053\u0075\u0069\u0073\u0073\u0065\u0029", false, 0x00DF, "CH", "CHE", "\u0053\u0077\u0069\u0074\u007A\u0065\u0072\u006C\u0061\u006E\u0064", "\u0053\u0075\u0069\u0073\u0073\u0065", "\u0043\u0048\u0046", "\u0053\u0077\u0069\u0073\u0073\u0020\u0046\u0072\u0061\u006E\u0063", "\u0046\u0072\u0061\u006E\u0063\u0020\u0073\u0075\u0069\u0073\u0073\u0065", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u0027", "\u002E", "\u0027", "\u0053\u0046\u0072\u002E", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u004E\u0075\u006D\u00E9\u0072\u0069\u0071\u0075\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0069\u0065\u0072\u0020\u0067\u0072\u00E9\u0067\u006F\u0072\u0069\u0065\u006E", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0068\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0068\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u0069\u006D\u002E", "\u006C\u0075\u006E\u002E", "\u006D\u0061\u0072\u002E", "\u006D\u0065\u0072\u002E", "\u006A\u0065\u0075\u002E", "\u0076\u0065\u006E\u002E", "\u0073\u0061\u006D\u002E" ], [ "\u0064\u0069\u006D\u0061\u006E\u0063\u0068\u0065", "\u006C\u0075\u006E\u0064\u0069", "\u006D\u0061\u0072\u0064\u0069", "\u006D\u0065\u0072\u0063\u0072\u0065\u0064\u0069", "\u006A\u0065\u0075\u0064\u0069", "\u0076\u0065\u006E\u0064\u0072\u0065\u0064\u0069", "\u0073\u0061\u006D\u0065\u0064\u0069" ], [ "\u006A\u0061\u006E\u0076\u002E", "\u0066\u00E9\u0076\u0072\u002E", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u002E", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u002E", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u00E9\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0076\u0069\u0065\u0072", "\u0066\u00E9\u0076\u0072\u0069\u0065\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u006C\u0065\u0074", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u00E9\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x1401, 0x0001, "ar-DZ", "ar", "ara", "ar-DZ", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0041\u006C\u0067\u0065\u0072\u0069\u0061\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u062C\u0632\u0627\u0626\u0631\u0029", false, 0x0004, "DZ", "DZA", "\u0041\u006C\u0067\u0065\u0072\u0069\u0061", "\u0627\u0644\u062C\u0632\u0627\u0626\u0631", "\u0044\u005A\u0044", "\u0041\u006C\u0067\u0065\u0072\u0069\u0061\u006E\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u062C\u0632\u0627\u0626\u0631\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u062C\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 6, 23, 2, 9, 10, 11 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0641\u0631\u0646\u0633\u064A\u0629\u0029\u200F", 6, 0, "\u002D", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u062C\u0627\u0646\u0641\u064A\u064A\u0647", "\u0641\u064A\u0641\u0631\u064A\u064A\u0647", "\u0645\u0627\u0631\u0633", "\u0623\u0641\u0631\u064A\u0644", "\u0645\u064A", "\u062C\u0648\u0627\u0646", "\u062C\u0648\u064A\u064A\u0647", "\u0623\u0648\u062A", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u062C\u0627\u0646\u0641\u064A\u064A\u0647", "\u0641\u064A\u0641\u0631\u064A\u064A\u0647", "\u0645\u0627\u0631\u0633", "\u0623\u0641\u0631\u064A\u0644", "\u0645\u064A", "\u062C\u0648\u0627\u0646", "\u062C\u0648\u064A\u064A\u0647", "\u0623\u0648\u062A", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x1404, 0x7C04, "zh-MO", "zh", "zho", "zh-MO", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u004D\u0061\u0063\u0061\u006F\u0020\u0053\u002E\u0041\u002E\u0052\u002E\u0029", "\u4E2D\u6587\u0028\u6FB3\u9580\u7279\u522B\u884C\u653F\u5340\u0029", false, 0x0097, "MO", "MAC", "\u004D\u0061\u0063\u0061\u006F\u0020\u0053\u002E\u0041\u002E\u0052\u002E", "\u6FB3\u9580\u7279\u522B\u884C\u653F\u5340", "\u004D\u004F\u0050", "\u004D\u0061\u0063\u0061\u006F\u0020\u0050\u0061\u0074\u0061\u0063\u0061", "\u0050\u0061\u0074\u0061\u0063\u0061", true, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u004D\u004F\u0050", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u897F\u66C6", 0, 0, "\u002F", "\u003A", "", "", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u002F\u0064", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064", "\u0079\u0079\u0079\u0079\u0020\u004D\u004D\u0020\u0064\u0064" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u661F\u671F\u65E5", "\u661F\u671F\u4E00", "\u661F\u671F\u4E8C", "\u661F\u671F\u4E09", "\u661F\u671F\u56DB", "\u661F\u671F\u4E94", "\u661F\u671F\u516D" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ], [ "\u4E00\u6708", "\u4E8C\u6708", "\u4E09\u6708", "\u56DB\u6708", "\u4E94\u6708", "\u516D\u6708", "\u4E03\u6708", "\u516B\u6708", "\u4E5D\u6708", "\u5341\u6708", "\u5341\u4E00\u6708", "\u5341\u4E8C\u6708", "" ] },
{ 0x1407, 0x0007, "de-LI", "de", "deu", "de-LI", "\u0047\u0065\u0072\u006D\u0061\u006E\u0020\u0028\u004C\u0069\u0065\u0063\u0068\u0074\u0065\u006E\u0073\u0074\u0065\u0069\u006E\u0029", "\u0044\u0065\u0075\u0074\u0073\u0063\u0068\u0020\u0028\u004C\u0069\u0065\u0063\u0068\u0074\u0065\u006E\u0073\u0074\u0065\u0069\u006E\u0029", false, 0x0091, "LI", "LIE", "\u004C\u0069\u0065\u0063\u0068\u0074\u0065\u006E\u0073\u0074\u0065\u0069\u006E", "\u004C\u0069\u0065\u0063\u0068\u0074\u0065\u006E\u0073\u0074\u0065\u0069\u006E", "\u0043\u0048\u0046", "\u0053\u0077\u0069\u0073\u0073\u0020\u0046\u0072\u0061\u006E\u0063", "\u0053\u0063\u0068\u0077\u0065\u0069\u007A\u0065\u0072\u0020\u0046\u0072\u0061\u006E\u006B\u0065\u006E", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u0027", "\u002E", "\u0027", "\u0043\u0048\u0046", "\u002D", "\u002B", "\u006E\u002E\u0020\u0064\u0065\u0066\u002E", "\u002D\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", "\u002B\u0075\u006E\u0065\u006E\u0064\u006C\u0069\u0063\u0068", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0069\u0073\u0063\u0068\u0065\u0072\u0020\u004B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 1, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\'" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u0048\u002E\u006D\u006D\'\u0020\u0068\'", "\u0048\u002E\u006D\u006D\'\u0020\u0055\u0068\u0072\'" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u004D\u004D\u0020\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u006F", "\u004D\u006F", "\u0044\u0069", "\u004D\u0069", "\u0044\u006F", "\u0046\u0072", "\u0053\u0061" ], [ "\u0053\u006F\u006E\u006E\u0074\u0061\u0067", "\u004D\u006F\u006E\u0074\u0061\u0067", "\u0044\u0069\u0065\u006E\u0073\u0074\u0061\u0067", "\u004D\u0069\u0074\u0074\u0077\u006F\u0063\u0068", "\u0044\u006F\u006E\u006E\u0065\u0072\u0073\u0074\u0061\u0067", "\u0046\u0072\u0065\u0069\u0074\u0061\u0067", "\u0053\u0061\u006D\u0073\u0074\u0061\u0067" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0072\u007A", "\u0041\u0070\u0072", "\u004D\u0061\u0069", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u006B\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u007A", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072", "\u004D\u00E4\u0072\u007A", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0069", "\u004A\u0075\u006E\u0069", "\u004A\u0075\u006C\u0069", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u006B\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u007A\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x1409, 0x0009, "en-NZ", "en", "eng", "en-NZ", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064\u0029", false, 0x00B7, "NZ", "NZL", "\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064", "\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064", "\u004E\u005A\u0044", "\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064\u0020\u0044\u006F\u006C\u006C\u0061\u0072", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x140A, 0x000A, "es-CR", "es", "spa", "es-CR", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0043\u006F\u0073\u0074\u0061\u0020\u0052\u0069\u0063\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0043\u006F\u0073\u0074\u0061\u0020\u0052\u0069\u0063\u0061\u0029", false, 0x0036, "CR", "CRI", "\u0043\u006F\u0073\u0074\u0061\u0020\u0052\u0069\u0063\u0061", "\u0043\u006F\u0073\u0074\u0061\u0020\u0052\u0069\u0063\u0061", "\u0043\u0052\u0043", "\u0043\u006F\u0073\u0074\u0061\u0020\u0052\u0069\u0063\u0061\u006E\u0020\u0043\u006F\u006C\u006F\u006E", "\u0043\u006F\u006C\u00F3\u006E", true, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u20A1", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x140C, 0x000C, "fr-LU", "fr", "fra", "fr-LU", "\u0046\u0072\u0065\u006E\u0063\u0068\u0020\u0028\u004C\u0075\u0078\u0065\u006D\u0062\u006F\u0075\u0072\u0067\u0029", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073\u0020\u0028\u004C\u0075\u0078\u0065\u006D\u0062\u006F\u0075\u0072\u0067\u0029", false, 0x0093, "LU", "LUX", "\u004C\u0075\u0078\u0065\u006D\u0062\u006F\u0075\u0072\u0067", "\u004C\u0075\u0078\u0065\u006D\u0062\u006F\u0075\u0072\u0067", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u004E\u0075\u006D\u00E9\u0072\u0069\u0071\u0075\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0069\u0065\u0072\u0020\u0067\u0072\u00E9\u0067\u006F\u0072\u0069\u0065\u006E", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u0069\u006D\u002E", "\u006C\u0075\u006E\u002E", "\u006D\u0061\u0072\u002E", "\u006D\u0065\u0072\u002E", "\u006A\u0065\u0075\u002E", "\u0076\u0065\u006E\u002E", "\u0073\u0061\u006D\u002E" ], [ "\u0064\u0069\u006D\u0061\u006E\u0063\u0068\u0065", "\u006C\u0075\u006E\u0064\u0069", "\u006D\u0061\u0072\u0064\u0069", "\u006D\u0065\u0072\u0063\u0072\u0065\u0064\u0069", "\u006A\u0065\u0075\u0064\u0069", "\u0076\u0065\u006E\u0064\u0072\u0065\u0064\u0069", "\u0073\u0061\u006D\u0065\u0064\u0069" ], [ "\u006A\u0061\u006E\u0076\u002E", "\u0066\u00E9\u0076\u0072\u002E", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u002E", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u002E", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u00E9\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0076\u0069\u0065\u0072", "\u0066\u00E9\u0076\u0072\u0069\u0065\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u006C\u0065\u0074", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u00E9\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x1801, 0x0001, "ar-MA", "ar", "ara", "ar-MA", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u004D\u006F\u0072\u006F\u0063\u0063\u006F\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0645\u0645\u0644\u0643\u0629\u0020\u0627\u0644\u0645\u063A\u0631\u0628\u064A\u0629\u0029", false, 0x009F, "MA", "MAR", "\u004D\u006F\u0072\u006F\u0063\u0063\u006F", "\u0627\u0644\u0645\u0645\u0644\u0643\u0629\u0020\u0627\u0644\u0645\u063A\u0631\u0628\u064A\u0629", "\u004D\u0041\u0044", "\u004D\u006F\u0072\u006F\u0063\u0063\u0061\u006E\u0020\u0044\u0069\u0072\u0068\u0061\u006D", "\u062F\u0631\u0647\u0645\u00A0\u0645\u063A\u0631\u0628\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0645\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 6, 23, 2, 9, 10, 11 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0641\u0631\u0646\u0633\u064A\u0629\u0029\u200F", 1, 0, "\u002D", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648\u0632", "\u063A\u0634\u062A", "\u0634\u062A\u0646\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0646\u0628\u0631", "\u062F\u062C\u0646\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648\u0632", "\u063A\u0634\u062A", "\u0634\u062A\u0646\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0646\u0628\u0631", "\u062F\u062C\u0646\u0628\u0631", "" ] },
{ 0x1809, 0x0009, "en-IE", "en", "eng", "en-IE", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0049\u0072\u0065\u006C\u0061\u006E\u0064\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0045\u0069\u0072\u0065\u0029", false, 0x0044, "IE", "IRL", "\u0049\u0072\u0065\u006C\u0061\u006E\u0064", "\u0045\u0069\u0072\u0065", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0045\u0075\u0072\u006F", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x180A, 0x000A, "es-PA", "es", "spa", "es-PA", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0050\u0061\u006E\u0061\u006D\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0050\u0061\u006E\u0061\u006D\u00E1\u0029", false, 0x00C0, "PA", "PAN", "\u0050\u0061\u006E\u0061\u006D\u0061", "\u0050\u0061\u006E\u0061\u006D\u00E1", "\u0050\u0041\u0042", "\u0050\u0061\u006E\u0061\u006D\u0061\u006E\u0069\u0061\u006E\u0020\u0042\u0061\u006C\u0062\u006F\u0061", "\u0042\u0061\u006C\u0062\u006F\u0061", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0042\u002F\u002E", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x180C, 0x000C, "fr-MC", "fr", "fra", "fr-MC", "\u0046\u0072\u0065\u006E\u0063\u0068\u0020\u0028\u0050\u0072\u0069\u006E\u0063\u0069\u0070\u0061\u006C\u0069\u0074\u0079\u0020\u006F\u0066\u0020\u004D\u006F\u006E\u0061\u0063\u006F\u0029", "\u0066\u0072\u0061\u006E\u00E7\u0061\u0069\u0073\u0020\u0028\u0050\u0072\u0069\u006E\u0063\u0069\u0070\u0061\u0075\u0074\u00E9\u0020\u0064\u0065\u0020\u004D\u006F\u006E\u0061\u0063\u006F\u0029", false, 0x009E, "MC", "MCO", "\u0050\u0072\u0069\u006E\u0063\u0069\u0070\u0061\u006C\u0069\u0074\u0079\u0020\u006F\u0066\u0020\u004D\u006F\u006E\u0061\u0063\u006F", "\u0050\u0072\u0069\u006E\u0063\u0069\u0070\u0061\u0075\u0074\u00E9\u0020\u0064\u0065\u0020\u004D\u006F\u006E\u0061\u0063\u006F", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u006F\u006E\u0020\u004E\u0075\u006D\u00E9\u0072\u0069\u0071\u0075\u0065", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069", "\u002B\u0049\u006E\u0066\u0069\u006E\u0069", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0069\u0065\u0072\u0020\u0067\u0072\u00E9\u0067\u006F\u0072\u0069\u0065\u006E", 1, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\'\u0020\u0068\u0020\'\u006D\u006D" ], [ "\u0064\u0064\u0064\u0064\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0064\u0069\u006D\u002E", "\u006C\u0075\u006E\u002E", "\u006D\u0061\u0072\u002E", "\u006D\u0065\u0072\u002E", "\u006A\u0065\u0075\u002E", "\u0076\u0065\u006E\u002E", "\u0073\u0061\u006D\u002E" ], [ "\u0064\u0069\u006D\u0061\u006E\u0063\u0068\u0065", "\u006C\u0075\u006E\u0064\u0069", "\u006D\u0061\u0072\u0064\u0069", "\u006D\u0065\u0072\u0063\u0072\u0065\u0064\u0069", "\u006A\u0065\u0075\u0064\u0069", "\u0076\u0065\u006E\u0064\u0072\u0065\u0064\u0069", "\u0073\u0061\u006D\u0065\u0064\u0069" ], [ "\u006A\u0061\u006E\u0076\u002E", "\u0066\u00E9\u0076\u0072\u002E", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u002E", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u002E", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u002E", "\u006F\u0063\u0074\u002E", "\u006E\u006F\u0076\u002E", "\u0064\u00E9\u0063\u002E", "" ], [ "\u006A\u0061\u006E\u0076\u0069\u0065\u0072", "\u0066\u00E9\u0076\u0072\u0069\u0065\u0072", "\u006D\u0061\u0072\u0073", "\u0061\u0076\u0072\u0069\u006C", "\u006D\u0061\u0069", "\u006A\u0075\u0069\u006E", "\u006A\u0075\u0069\u006C\u006C\u0065\u0074", "\u0061\u006F\u00FB\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u006F\u0062\u0072\u0065", "\u006E\u006F\u0076\u0065\u006D\u0062\u0072\u0065", "\u0064\u00E9\u0063\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x1C01, 0x0001, "ar-TN", "ar", "ara", "ar-TN", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0054\u0075\u006E\u0069\u0073\u0069\u0061\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u062A\u0648\u0646\u0633\u0029", false, 0x00EA, "TN", "TUN", "\u0054\u0075\u006E\u0069\u0073\u0069\u0061", "\u062A\u0648\u0646\u0633", "\u0054\u004E\u0044", "\u0054\u0075\u006E\u0069\u0073\u0069\u0061\u006E\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u062A\u0648\u0646\u0633\u064A", true, 3, 3, 3, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u062A\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 6, 23, 2, 9, 10, 11 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0641\u0631\u0646\u0633\u064A\u0629\u0029\u200F", 1, 0, "\u002D", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u062C\u0627\u0646\u0641\u064A", "\u0641\u064A\u0641\u0631\u064A", "\u0645\u0627\u0631\u0633", "\u0627\u0641\u0631\u064A\u0644", "\u0645\u0627\u064A", "\u062C\u0648\u0627\u0646", "\u062C\u0648\u064A\u0644\u064A\u0629", "\u0627\u0648\u062A", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u062C\u0627\u0646\u0641\u064A", "\u0641\u064A\u0641\u0631\u064A", "\u0645\u0627\u0631\u0633", "\u0627\u0641\u0631\u064A\u0644", "\u0645\u0627\u064A", "\u062C\u0648\u0627\u0646", "\u062C\u0648\u064A\u0644\u064A\u0629", "\u0627\u0648\u062A", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x1C09, 0x0009, "en-ZA", "en", "eng", "en-ZA", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", false, 0x00D1, "ZA", "ZAF", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u005A\u0041\u0052", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u006E\u0020\u0052\u0061\u006E\u0064", "\u0052\u0061\u006E\u0064", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x1C0A, 0x000A, "es-DO", "es", "spa", "es-DO", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0044\u006F\u006D\u0069\u006E\u0069\u0063\u0061\u006E\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0052\u0065\u0070\u00FA\u0062\u006C\u0069\u0063\u0061\u0020\u0044\u006F\u006D\u0069\u006E\u0069\u0063\u0061\u006E\u0061\u0029", false, 0x0041, "DO", "DOM", "\u0044\u006F\u006D\u0069\u006E\u0069\u0063\u0061\u006E\u0020\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063", "\u0052\u0065\u0070\u00FA\u0062\u006C\u0069\u0063\u0061\u0020\u0044\u006F\u006D\u0069\u006E\u0069\u0063\u0061\u006E\u0061", "\u0044\u004F\u0050", "\u0044\u006F\u006D\u0069\u006E\u0069\u0063\u0061\u006E\u0020\u0050\u0065\u0073\u006F", "\u0050\u0065\u0073\u006F", true, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052\u0044\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x2001, 0x0001, "ar-OM", "ar", "ara", "ar-OM", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u004F\u006D\u0061\u006E\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0639\u0645\u0627\u0646\u0029", false, 0x00A4, "OM", "OMN", "\u004F\u006D\u0061\u006E", "\u0639\u0645\u0627\u0646", "\u004F\u004D\u0052", "\u0052\u0069\u0061\u006C\u0020\u004F\u006D\u0061\u006E\u0069", "\u0631\u064A\u0627\u0644\u00A0\u0639\u0645\u0627\u0646\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0631\u002E\u0639\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 6, 23, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x2009, 0x0009, "en-JM", "en", "eng", "en-JM", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u004A\u0061\u006D\u0061\u0069\u0063\u0061\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u004A\u0061\u006D\u0061\u0069\u0063\u0061\u0029", false, 0x007C, "JM", "JAM", "\u004A\u0061\u006D\u0061\u0069\u0063\u0061", "\u004A\u0061\u006D\u0061\u0069\u0063\u0061", "\u004A\u004D\u0044", "\u004A\u0061\u006D\u0061\u0069\u0063\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u004A\u0061\u006D\u0061\u0069\u0063\u0061\u006E\u0020\u0044\u006F\u006C\u006C\u0061\u0072", false, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u004A\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x200A, 0x000A, "es-VE", "es", "spa", "es-VE", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0056\u0065\u006E\u0065\u007A\u0075\u0065\u006C\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0061\u0020\u0042\u006F\u006C\u0069\u0076\u0061\u0072\u0069\u0061\u006E\u0061\u0020\u0064\u0065\u0020\u0056\u0065\u006E\u0065\u007A\u0075\u0065\u006C\u0061\u0029", false, 0x00F9, "VE", "VEN", "\u0056\u0065\u006E\u0065\u007A\u0075\u0065\u006C\u0061", "\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0061\u0020\u0042\u006F\u006C\u0069\u0076\u0061\u0072\u0069\u0061\u006E\u0061\u0020\u0064\u0065\u0020\u0056\u0065\u006E\u0065\u007A\u0075\u0065\u006C\u0061", "\u0056\u0045\u0042", "\u0056\u0065\u006E\u0065\u007A\u0075\u0065\u006C\u0061\u006E\u0020\u0042\u006F\u006C\u0069\u0076\u0061\u0072", "\u0042\u006F\u006C\u00ED\u0076\u0061\u0072", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0042\u0073", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x2401, 0x0001, "ar-YE", "ar", "ara", "ar-YE", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0059\u0065\u006D\u0065\u006E\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u064A\u0645\u0646\u0029", false, 0x0105, "YE", "YEM", "\u0059\u0065\u006D\u0065\u006E", "\u0627\u0644\u064A\u0645\u0646", "\u0059\u0045\u0052", "\u0059\u0065\u006D\u0065\u006E\u0069\u0020\u0052\u0069\u0061\u006C", "\u0631\u064A\u0627\u0644\u00A0\u064A\u0645\u0646\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0631\u002E\u064A\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x2409, 0x0009, "en-CB", "en", "eng", "en-029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0043\u0061\u0072\u0069\u0062\u0062\u0065\u0061\u006E\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0043\u0061\u0072\u0069\u0062\u0062\u0065\u0061\u006E\u0029", false, 0x007C, "CB", "CAR", "\u0043\u0061\u0072\u0069\u0062\u0062\u0065\u0061\u006E", "\u0043\u0061\u0072\u0069\u0062\u0062\u0065\u0061\u006E", "\u0055\u0053\u0044", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", false, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 1, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x240A, 0x000A, "es-CO", "es", "spa", "es-CO", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0043\u006F\u006C\u006F\u006D\u0062\u0069\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0043\u006F\u006C\u006F\u006D\u0062\u0069\u0061\u0029", false, 0x0033, "CO", "COL", "\u0043\u006F\u006C\u006F\u006D\u0062\u0069\u0061", "\u0043\u006F\u006C\u006F\u006D\u0062\u0069\u0061", "\u0043\u004F\u0050", "\u0043\u006F\u006C\u006F\u006D\u0062\u0069\u0061\u006E\u0020\u0050\u0065\u0073\u006F", "\u0050\u0065\u0073\u006F", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x2801, 0x0001, "ar-SY", "ar", "ara", "ar-SY", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0053\u0079\u0072\u0069\u0061\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0633\u0648\u0631\u064A\u0627\u0029", false, 0x00DE, "SY", "SYR", "\u0053\u0079\u0072\u0069\u0061", "\u0633\u0648\u0631\u064A\u0627", "\u0053\u0059\u0050", "\u0053\u0079\u0072\u0069\u0061\u006E\u0020\u0050\u006F\u0075\u006E\u0064", "\u0644\u064A\u0631\u0629\u0020\u0633\u0648\u0631\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0644\u002E\u0633\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 11, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u0639\u0631\u0628\u064A\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ] },
{ 0x2809, 0x0009, "en-BZ", "en", "eng", "en-BZ", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0042\u0065\u006C\u0069\u007A\u0065\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0042\u0065\u006C\u0069\u007A\u0065\u0029", false, 0x0018, "BZ", "BLZ", "\u0042\u0065\u006C\u0069\u007A\u0065", "\u0042\u0065\u006C\u0069\u007A\u0065", "\u0042\u005A\u0044", "\u0042\u0065\u006C\u0069\u007A\u0065\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0042\u0065\u006C\u0069\u007A\u0065\u0020\u0044\u006F\u006C\u006C\u0061\u0072", true, 2, 1, 2, 0, 0, [ 3 ], [ 3, 0 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0042\u005A\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x280A, 0x000A, "es-PE", "es", "spa", "es-PE", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0050\u0065\u0072\u0075\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0050\u0065\u0072\u00FA\u0029", false, 0x00BB, "PE", "PER", "\u0050\u0065\u0072\u0075", "\u0050\u0065\u0072\u00FA", "\u0050\u0045\u004E", "\u0050\u0065\u0072\u0075\u0076\u0069\u0061\u006E\u0020\u004E\u0075\u0065\u0076\u006F\u0020\u0053\u006F\u006C", "\u004E\u0075\u0065\u0076\u006F\u0020\u0053\u006F\u006C", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0053\u002F\u002E", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x2C01, 0x0001, "ar-JO", "ar", "ara", "ar-JO", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u004A\u006F\u0072\u0064\u0061\u006E\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0623\u0631\u062F\u0646\u0029", false, 0x007E, "JO", "JOR", "\u004A\u006F\u0072\u0064\u0061\u006E", "\u0627\u0644\u0623\u0631\u062F\u0646", "\u004A\u004F\u0044", "\u004A\u006F\u0072\u0064\u0061\u006E\u0069\u0061\u006E\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u0627\u0631\u062F\u0646\u064A", true, 3, 3, 3, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0627\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 11, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u0639\u0631\u0628\u064A\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ] },
{ 0x2C09, 0x0009, "en-TT", "en", "eng", "en-TT", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0054\u0072\u0069\u006E\u0069\u0064\u0061\u0064\u0020\u0061\u006E\u0064\u0020\u0054\u006F\u0062\u0061\u0067\u006F\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0054\u0072\u0069\u006E\u0069\u0064\u0061\u0064\u0020\u0079\u0020\u0054\u006F\u0062\u0061\u0067\u006F\u0029", false, 0x00E1, "TT", "TTO", "\u0054\u0072\u0069\u006E\u0069\u0064\u0061\u0064\u0020\u0061\u006E\u0064\u0020\u0054\u006F\u0062\u0061\u0067\u006F", "\u0054\u0072\u0069\u006E\u0069\u0064\u0061\u0064\u0020\u0079\u0020\u0054\u006F\u0062\u0061\u0067\u006F", "\u0054\u0054\u0044", "\u0054\u0072\u0069\u006E\u0069\u0064\u0061\u0064\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0054\u0072\u0069\u006E\u0069\u0064\u0061\u0064\u0020\u0044\u006F\u006C\u006C\u0061\u0072", true, 2, 1, 2, 0, 0, [ 3 ], [ 3, 0 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0054\u0054\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x2C0A, 0x000A, "es-AR", "es", "spa", "es-AR", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0041\u0072\u0067\u0065\u006E\u0074\u0069\u006E\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0041\u0072\u0067\u0065\u006E\u0074\u0069\u006E\u0061\u0029", false, 0x000B, "AR", "ARG", "\u0041\u0072\u0067\u0065\u006E\u0074\u0069\u006E\u0061", "\u0041\u0072\u0067\u0065\u006E\u0074\u0069\u006E\u0061", "\u0041\u0052\u0053", "\u0041\u0072\u0067\u0065\u006E\u0074\u0069\u006E\u0065\u0020\u0050\u0065\u0073\u006F", "\u0050\u0065\u0073\u006F", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x3001, 0x0001, "ar-LB", "ar", "ara", "ar-LB", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u004C\u0065\u0062\u0061\u006E\u006F\u006E\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0644\u0628\u0646\u0627\u0646\u0029", false, 0x008B, "LB", "LBN", "\u004C\u0065\u0062\u0061\u006E\u006F\u006E", "\u0644\u0628\u0646\u0627\u0646", "\u004C\u0042\u0050", "\u004C\u0065\u0062\u0061\u006E\u0065\u0073\u0065\u0020\u0050\u006F\u0075\u006E\u0064", "\u0644\u064A\u0631\u0629\u0020\u0644\u0628\u0646\u0627\u0646\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0644\u002E\u0644\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 11, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u0639\u0631\u0628\u064A\u0029\u200F", 1, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ], [ "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0634\u0628\u0627\u0637", "\u0622\u0630\u0627\u0631", "\u0646\u064A\u0633\u0627\u0646", "\u0623\u064A\u0627\u0631", "\u062D\u0632\u064A\u0631\u0627\u0646", "\u062A\u0645\u0648\u0632", "\u0622\u0628", "\u0623\u064A\u0644\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "\u062A\u0634\u0631\u064A\u0646\u00A0\u0627\u0644\u062B\u0627\u0646\u064A", "\u0643\u0627\u0646\u0648\u0646\u00A0\u0627\u0644\u0623\u0648\u0644", "" ] },
{ 0x3009, 0x0009, "en-ZW", "en", "eng", "en-ZW", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u005A\u0069\u006D\u0062\u0061\u0062\u0077\u0065\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u005A\u0069\u006D\u0062\u0061\u0062\u0077\u0065\u0029", false, 0x0108, "ZW", "ZWE", "\u005A\u0069\u006D\u0062\u0061\u0062\u0077\u0065", "\u005A\u0069\u006D\u0062\u0061\u0062\u0077\u0065", "\u005A\u0057\u0044", "\u005A\u0069\u006D\u0062\u0061\u0062\u0077\u0065\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u005A\u0069\u006D\u0062\u0061\u0062\u0077\u0065\u0020\u0044\u006F\u006C\u006C\u0061\u0072", false, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u005A\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u002F\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x300A, 0x000A, "es-EC", "es", "spa", "es-EC", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0045\u0063\u0075\u0061\u0064\u006F\u0072\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0045\u0063\u0075\u0061\u0064\u006F\u0072\u0029", false, 0x0042, "EC", "ECU", "\u0045\u0063\u0075\u0061\u0064\u006F\u0072", "\u0045\u0063\u0075\u0061\u0064\u006F\u0072", "\u0055\u0053\u0044", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0055\u0053\u0020\u0044\u006F\u006C\u0061\u0072", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x3401, 0x0001, "ar-KW", "ar", "ara", "ar-KW", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u004B\u0075\u0077\u0061\u0069\u0074\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0643\u0648\u064A\u062A\u0029", false, 0x0088, "KW", "KWT", "\u004B\u0075\u0077\u0061\u0069\u0074", "\u0627\u0644\u0643\u0648\u064A\u062A", "\u004B\u0057\u0044", "\u004B\u0075\u0077\u0061\u0069\u0074\u0069\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u0643\u0648\u064A\u062A\u064A", true, 3, 3, 3, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0643\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 6, 23, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x3409, 0x0009, "en-PH", "en", "eng", "en-PH", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u0074\u0068\u0065\u0020\u0050\u0068\u0069\u006C\u0069\u0070\u0070\u0069\u006E\u0065\u0073\u0029", "\u0045\u006E\u0067\u006C\u0069\u0073\u0068\u0020\u0028\u0050\u0068\u0069\u006C\u0069\u0070\u0070\u0069\u006E\u0065\u0073\u0029", false, 0x00C9, "PH", "PHL", "\u0052\u0065\u0070\u0075\u0062\u006C\u0069\u0063\u0020\u006F\u0066\u0020\u0074\u0068\u0065\u0020\u0050\u0068\u0069\u006C\u0069\u0070\u0070\u0069\u006E\u0065\u0073", "\u0050\u0068\u0069\u006C\u0069\u0070\u0070\u0069\u006E\u0065\u0073", "\u0050\u0048\u0050", "\u0050\u0068\u0069\u006C\u0069\u0070\u0070\u0069\u006E\u0065\u0020\u0050\u0065\u0073\u006F", "\u0050\u0068\u0069\u006C\u0069\u0070\u0070\u0069\u006E\u0065\u0020\u0050\u0065\u0073\u006F", false, 2, 1, 2, 0, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0050\u0068\u0070", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1, 2 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u004D\u002F\u0064\u002F\u0079\u0079\u0079\u0079", "\u004D\u002F\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079", "\u004D\u004D\u002F\u0064\u0064\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0064\u0064\u002D\u004D\u004D\u004D\u002D\u0079\u0079" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0053\u0075\u006E\u0064\u0061\u0079", "\u004D\u006F\u006E\u0064\u0061\u0079", "\u0054\u0075\u0065\u0073\u0064\u0061\u0079", "\u0057\u0065\u0064\u006E\u0065\u0073\u0064\u0061\u0079", "\u0054\u0068\u0075\u0072\u0073\u0064\u0061\u0079", "\u0046\u0072\u0069\u0064\u0061\u0079", "\u0053\u0061\u0074\u0075\u0072\u0064\u0061\u0079" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u004A\u0061\u006E\u0075\u0061\u0072\u0079", "\u0046\u0065\u0062\u0072\u0075\u0061\u0072\u0079", "\u004D\u0061\u0072\u0063\u0068", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0061\u0079", "\u004A\u0075\u006E\u0065", "\u004A\u0075\u006C\u0079", "\u0041\u0075\u0067\u0075\u0073\u0074", "\u0053\u0065\u0070\u0074\u0065\u006D\u0062\u0065\u0072", "\u004F\u0063\u0074\u006F\u0062\u0065\u0072", "\u004E\u006F\u0076\u0065\u006D\u0062\u0065\u0072", "\u0044\u0065\u0063\u0065\u006D\u0062\u0065\u0072", "" ] },
{ 0x340A, 0x000A, "es-CL", "es", "spa", "es-CL", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0043\u0068\u0069\u006C\u0065\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0043\u0068\u0069\u006C\u0065\u0029", false, 0x002E, "CL", "CHL", "\u0043\u0068\u0069\u006C\u0065", "\u0043\u0068\u0069\u006C\u0065", "\u0043\u004C\u0050", "\u0043\u0068\u0069\u006C\u0065\u0061\u006E\u0020\u0050\u0065\u0073\u006F", "\u0050\u0065\u0073\u006F", true, 2, 1, 2, 9, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002D", "\u003A", "", "", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x3801, 0x0001, "ar-AE", "ar", "ara", "ar-AE", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0055\u002E\u0041\u002E\u0045\u002E\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0625\u0645\u0627\u0631\u0627\u062A\u0020\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0627\u0644\u0645\u062A\u062D\u062F\u0629\u0029", false, 0x00E0, "AE", "ARE", "\u0055\u002E\u0041\u002E\u0045\u002E", "\u0627\u0644\u0625\u0645\u0627\u0631\u0627\u062A\u0020\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0627\u0644\u0645\u062A\u062D\u062F\u0629", "\u0041\u0045\u0044", "\u0055\u0041\u0045\u0020\u0044\u0069\u0072\u0068\u0061\u006D", "\u062F\u0631\u0647\u0645\u00A0\u0627\u0645\u0627\u0631\u0627\u062A\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0625\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x380A, 0x000A, "es-UY", "es", "spa", "es-UY", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0055\u0072\u0075\u0067\u0075\u0061\u0079\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0055\u0072\u0075\u0067\u0075\u0061\u0079\u0029", false, 0x00F6, "UY", "URY", "\u0055\u0072\u0075\u0067\u0075\u0061\u0079", "\u0055\u0072\u0075\u0067\u0075\u0061\u0079", "\u0055\u0059\u0055", "\u0050\u0065\u0073\u006F\u0020\u0055\u0072\u0075\u0067\u0075\u0061\u0079\u006F", "\u0050\u0065\u0073\u006F", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024\u0055", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x3C01, 0x0001, "ar-BH", "ar", "ara", "ar-BH", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0042\u0061\u0068\u0072\u0061\u0069\u006E\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0627\u0644\u0628\u062D\u0631\u064A\u0646\u0029", false, 0x0011, "BH", "BHR", "\u0042\u0061\u0068\u0072\u0061\u0069\u006E", "\u0627\u0644\u0628\u062D\u0631\u064A\u0646", "\u0042\u0048\u0044", "\u0042\u0061\u0068\u0072\u0061\u0069\u006E\u0069\u0020\u0044\u0069\u006E\u0061\u0072", "\u062F\u064A\u0646\u0627\u0631\u00A0\u0628\u062D\u0631\u064A\u0646\u064A", true, 3, 3, 3, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u062F\u002E\u0628\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x3C0A, 0x000A, "es-PY", "es", "spa", "es-PY", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0050\u0061\u0072\u0061\u0067\u0075\u0061\u0079\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0050\u0061\u0072\u0061\u0067\u0075\u0061\u0079\u0029", false, 0x00B9, "PY", "PRY", "\u0050\u0061\u0072\u0061\u0067\u0075\u0061\u0079", "\u0050\u0061\u0072\u0061\u0067\u0075\u0061\u0079", "\u0050\u0059\u0047", "\u0050\u0061\u0072\u0061\u0067\u0075\u0061\u0079\u0020\u0047\u0075\u0061\u0072\u0061\u006E\u0069", "\u0047\u0075\u0061\u0072\u0061\u006E\u00ED", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0047\u0073", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 1, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x4001, 0x0001, "ar-QA", "ar", "ara", "ar-QA", "\u0041\u0072\u0061\u0062\u0069\u0063\u0020\u0028\u0051\u0061\u0074\u0061\u0072\u0029", "\u0627\u0644\u0639\u0631\u0628\u064A\u0629\u0020\u0028\u0642\u0637\u0631\u0029", false, 0x00C5, "QA", "QAT", "\u0051\u0061\u0074\u0061\u0072", "\u0642\u0637\u0631", "\u0051\u0041\u0052", "\u0051\u0061\u0074\u0061\u0072\u0069\u0020\u0052\u0069\u0061\u006C", "\u0631\u064A\u0627\u0644\u00A0\u0642\u0637\u0631\u064A", true, 2, 3, 2, 3, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0631\u002E\u0642\u002E\u200F", "\u002D", "\u002B", "\u0644\u064A\u0633\u0020\u0628\u0631\u0642\u0645", "\u002D\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", "\u002B\u0644\u0627\u0020\u0646\u0647\u0627\u064A\u0629", [ "\u0660", "\u0661", "\u0662", "\u0663", "\u0664", "\u0665", "\u0666", "\u0667", "\u0668", "\u0669" ], 1, [ 1, 23, 6, 2, 9, 10, 12 ], "\u0627\u0644\u062A\u0642\u0648\u064A\u0645\u00A0\u0627\u0644\u0645\u064A\u0644\u0627\u062F\u064A\u00A0\u0028\u062A\u0633\u0645\u064A\u0629\u00A0\u0625\u0646\u062C\u0644\u064A\u0632\u064A\u0629\u0029\u200F", 6, 0, "\u002F", "\u003A", "\u0635", "\u0645", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u0627\u0644\u0627\u062D\u062F", "\u0627\u0644\u0627\u062B\u0646\u064A\u0646", "\u0627\u0644\u062B\u0644\u0627\u062B\u0627\u0621", "\u0627\u0644\u0627\u0631\u0628\u0639\u0627\u0621", "\u0627\u0644\u062E\u0645\u064A\u0633", "\u0627\u0644\u062C\u0645\u0639\u0629", "\u0627\u0644\u0633\u0628\u062A" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ], [ "\u064A\u0646\u0627\u064A\u0631", "\u0641\u0628\u0631\u0627\u064A\u0631", "\u0645\u0627\u0631\u0633", "\u0627\u0628\u0631\u064A\u0644", "\u0645\u0627\u064A\u0648", "\u064A\u0648\u0646\u064A\u0648", "\u064A\u0648\u0644\u064A\u0648", "\u0627\u063A\u0633\u0637\u0633", "\u0633\u0628\u062A\u0645\u0628\u0631", "\u0627\u0643\u062A\u0648\u0628\u0631", "\u0646\u0648\u0641\u0645\u0628\u0631", "\u062F\u064A\u0633\u0645\u0628\u0631", "" ] },
{ 0x400A, 0x000A, "es-BO", "es", "spa", "es-BO", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u0029", false, 0x001A, "BO", "BOL", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061", "\u0042\u004F\u0042", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u006E\u006F", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u006E\u006F", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024\u0062", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x440A, 0x000A, "es-SV", "es", "spa", "es-SV", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0045\u006C\u0020\u0053\u0061\u006C\u0076\u0061\u0064\u006F\u0072\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0045\u006C\u0020\u0053\u0061\u006C\u0076\u0061\u0064\u006F\u0072\u0029", false, 0x0048, "SV", "SLV", "\u0045\u006C\u0020\u0053\u0061\u006C\u0076\u0061\u0064\u006F\u0072", "\u0045\u006C\u0020\u0053\u0061\u006C\u0076\u0061\u0064\u006F\u0072", "\u0055\u0053\u0044", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0055\u0053\u0020\u0044\u006F\u006C\u0061\u0072", true, 2, 1, 2, 0, 0, [ 3 ], [ 3, 0 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u004D\u004D\u002D\u0064\u0064\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x480A, 0x000A, "es-HN", "es", "spa", "es-HN", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0048\u006F\u006E\u0064\u0075\u0072\u0061\u0073\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0048\u006F\u006E\u0064\u0075\u0072\u0061\u0073\u0029", false, 0x006A, "HN", "HND", "\u0048\u006F\u006E\u0064\u0075\u0072\u0061\u0073", "\u0048\u006F\u006E\u0064\u0075\u0072\u0061\u0073", "\u0048\u004E\u004C", "\u0048\u006F\u006E\u0064\u0075\u0072\u0061\u006E\u0020\u004C\u0065\u006D\u0070\u0069\u0072\u0061", "\u004C\u0065\u006D\u0070\u0069\u0072\u0061", true, 2, 1, 2, 12, 2, [ 3 ], [ 3, 0 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u004C\u002E", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u004D\u004D\u002D\u0064\u0064\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x4C0A, 0x000A, "es-NI", "es", "spa", "es-NI", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u004E\u0069\u0063\u0061\u0072\u0061\u0067\u0075\u0061\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u004E\u0069\u0063\u0061\u0072\u0061\u0067\u0075\u0061\u0029", false, 0x00B6, "NI", "NIC", "\u004E\u0069\u0063\u0061\u0072\u0061\u0067\u0075\u0061", "\u004E\u0069\u0063\u0061\u0072\u0061\u0067\u0075\u0061", "\u004E\u0049\u004F", "\u004E\u0069\u0063\u0061\u0072\u0061\u0067\u0075\u0061\u006E\u0020\u0043\u006F\u0072\u0064\u006F\u0062\u0061\u0020\u004F\u0072\u006F", "\u0043\u00F3\u0072\u0064\u006F\u0062\u0061", true, 2, 1, 2, 14, 2, [ 3 ], [ 3, 0 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0043\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u004D\u004D\u002D\u0064\u0064\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x500A, 0x000A, "es-PR", "es", "spa", "es-PR", "\u0053\u0070\u0061\u006E\u0069\u0073\u0068\u0020\u0028\u0050\u0075\u0065\u0072\u0074\u006F\u0020\u0052\u0069\u0063\u006F\u0029", "\u0045\u0073\u0070\u0061\u00F1\u006F\u006C\u0020\u0028\u0050\u0075\u0065\u0072\u0074\u006F\u0020\u0052\u0069\u0063\u006F\u0029", false, 0x00CA, "PR", "PRI", "\u0050\u0075\u0065\u0072\u0074\u006F\u0020\u0052\u0069\u0063\u006F", "\u0050\u0075\u0065\u0072\u0074\u006F\u0020\u0052\u0069\u0063\u006F", "\u0055\u0053\u0044", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", true, 2, 1, 2, 14, 2, [ 3 ], [ 3, 0 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0065\u0075\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u006F", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u004D\u004D\u002D\u0064\u0064\u002D\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0064\u006F\u006D", "\u006C\u0075\u006E", "\u006D\u0061\u0072", "\u006D\u0069\u00E9", "\u006A\u0075\u0065", "\u0076\u0069\u0065", "\u0073\u00E1\u0062" ], [ "\u0064\u006F\u006D\u0069\u006E\u0067\u006F", "\u006C\u0075\u006E\u0065\u0073", "\u006D\u0061\u0072\u0074\u0065\u0073", "\u006D\u0069\u00E9\u0072\u0063\u006F\u006C\u0065\u0073", "\u006A\u0075\u0065\u0076\u0065\u0073", "\u0076\u0069\u0065\u0072\u006E\u0065\u0073", "\u0073\u00E1\u0062\u0061\u0064\u006F" ], [ "\u0065\u006E\u0065", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0062\u0072", "\u006D\u0061\u0079", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0067\u006F", "\u0073\u0065\u0070", "\u006F\u0063\u0074", "\u006E\u006F\u0076", "\u0064\u0069\u0063", "" ], [ "\u0065\u006E\u0065\u0072\u006F", "\u0066\u0065\u0062\u0072\u0065\u0072\u006F", "\u006D\u0061\u0072\u007A\u006F", "\u0061\u0062\u0072\u0069\u006C", "\u006D\u0061\u0079\u006F", "\u006A\u0075\u006E\u0069\u006F", "\u006A\u0075\u006C\u0069\u006F", "\u0061\u0067\u006F\u0073\u0074\u006F", "\u0073\u0065\u0070\u0074\u0069\u0065\u006D\u0062\u0072\u0065", "\u006F\u0063\u0074\u0075\u0062\u0072\u0065", "\u006E\u006F\u0076\u0069\u0065\u006D\u0062\u0072\u0065", "\u0064\u0069\u0063\u0069\u0065\u006D\u0062\u0072\u0065", "" ] },
{ 0x7C04, 0x007F, "zh-CHT", "zh", "zho", "zh-Hant", "\u0043\u0068\u0069\u006E\u0065\u0073\u0065\u0020\u0028\u0054\u0072\u0061\u0064\u0069\u0074\u0069\u006F\u006E\u0061\u006C\u0029", "\u4E2D\u6587\u0028\u7E41\u9AD4\u0029", true },
{ 0x7C1A, 0x007F, "sr", "sr", "srp", "sr", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E", "\u0073\u0072\u0070\u0073\u006B\u0069", true },
{ 0x183B, 0x007F, "sma-NO", "sma", "sma", "sma-NO", "\u0053\u0061\u006D\u0069\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0065\u0072\u006E\u0029\u0020\u0028\u004E\u006F\u0072\u0077\u0061\u0079\u0029", "\u00E5\u0061\u0072\u006A\u0065\u006C\u0073\u0061\u0065\u006D\u0069\u0065\u006E\u0067\u0069\u0065\u006C\u0065\u0020\u0028\u004E\u00F6\u00F6\u0072\u006A\u0065\u0029", false, 0x00B1, "NO", "NO", "\u004E\u006F\u0072\u0077\u0061\u0079", "\u004E\u00F6\u00F6\u0072\u006A\u0065", "\u004E\u004F\u004B", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u0020\u004B\u0072\u006F\u006E\u0065", "\u006B\u0072\u00E5\u0076\u006E\u006F\u0065", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0061\u0065\u006A", "\u006D\u00E5\u0061", "\u0064\u00E6\u006A", "\u0067\u0061\u0073\u006B", "\u0064\u0075\u0061\u0072", "\u0062\u0065\u0061\u0072\u006A", "\u006C\u0061\u0061\u0076" ], [ "\u0061\u0065\u006A\u006C\u0065\u0067\u0065", "\u006D\u00E5\u0061\u006E\u0074\u0061", "\u0064\u00E6\u006A\u0073\u0074\u0061", "\u0067\u0061\u0073\u006B\u0065\u0076\u00E5\u0068\u006B\u006F\u0065", "\u0064\u0075\u0061\u0072\u0073\u0074\u0061", "\u0062\u0065\u0061\u0072\u006A\u0061\u0064\u0061\u0068\u006B\u0065", "\u006C\u0061\u0061\u0076\u0076\u0061\u0072\u0064\u0061\u0068\u006B\u0065" ], [ "\u0074\u0073\u00EF\u0065\u006E", "\u0067\u006F\u0065\u0076\u0074", "\u006E\u006A\u006F\u006B", "\u0076\u006F\u0065\u0072", "\u0073\u0075\u0065\u0068", "\u0072\u0075\u0066\u0066", "\u0073\u006E\u006A\u0061", "\u006D\u00EF\u0065\u0074", "\u0073\u006B\u00EF\u0065\u0072", "\u0067\u006F\u006C\u006B", "\u0072\u0061\u0068\u006B", "\u0067\u006F\u0065\u0076", "" ], [ "\u0074\u0073\u00EF\u0065\u006E\u0067\u0065\u006C\u0065", "\u0067\u006F\u0065\u0076\u0074\u0065", "\u006E\u006A\u006F\u006B\u0074\u006A\u0065", "\u0076\u006F\u0065\u0072\u0068\u0074\u006A\u0065", "\u0073\u0075\u0065\u0068\u0070\u0065\u0064\u0065", "\u0072\u0075\u0066\u0066\u0069\u0065", "\u0073\u006E\u006A\u0061\u006C\u0074\u006A\u0065", "\u006D\u00EF\u0065\u0074\u0073\u006B\u0065", "\u0073\u006B\u00EF\u0065\u0072\u0065\u0064\u0065", "\u0067\u006F\u006C\u006B\u0065", "\u0072\u0061\u0068\u006B\u0061", "\u0067\u006F\u0065\u0076\u0065", "" ] },
{ 0x1C1A, 0x007F, "sr-BA-Cyrl", "sr", "sr", "sr-Cyrl-BA", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E\u0020\u0028\u0043\u0079\u0072\u0069\u006C\u006C\u0069\u0063\u0029\u0020\u0028\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", "\u0441\u0440\u043F\u0441\u043A\u0438\u0020\u0028\u0411\u043E\u0441\u043D\u0430\u0020\u0438\u0020\u0425\u0435\u0440\u0446\u0435\u0433\u043E\u0432\u0438\u043D\u0430\u0029", false, 0x0019, "BA", "BA", "\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0411\u043E\u0441\u043D\u0430\u0020\u0438\u0020\u0425\u0435\u0440\u0446\u0435\u0433\u043E\u0432\u0438\u043D\u0430", "\u0042\u0041\u004D", "\u0043\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u006C\u0065\u0020\u004D\u0061\u0072\u006B\u0073", "\u043A\u043E\u043D\u0432\u0435\u0440\u0442\u0438\u0431\u0438\u043B\u043D\u0430\u0020\u043C\u0430\u0440\u043A\u0430", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u041A\u041C", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0433\u0440\u0435\u0433\u043E\u0440\u0438\u0458\u0430\u043D\u0441\u043A\u0438\u0020\u043A\u0430\u043B\u0435\u043D\u0434\u0430\u0440", 0, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u043D\u0435\u0434", "\u043F\u043E\u043D", "\u0443\u0442\u043E", "\u0441\u0440\u0435", "\u0447\u0435\u0442", "\u043F\u0435\u0442", "\u0441\u0443\u0431" ], [ "\u043D\u0435\u0434\u0435\u0459\u0430", "\u043F\u043E\u043D\u0435\u0434\u0435\u0459\u0430\u043A", "\u0443\u0442\u043E\u0440\u0430\u043A", "\u0441\u0440\u0435\u0434\u0430", "\u0447\u0435\u0442\u0432\u0440\u0442\u0430\u043A", "\u043F\u0435\u0442\u0430\u043A", "\u0441\u0443\u0431\u043E\u0442\u0430" ], [ "\u0458\u0430\u043D", "\u0444\u0435\u0431", "\u043C\u0430\u0440", "\u0430\u043F\u0440", "\u043C\u0430\u0458", "\u0458\u0443\u043D", "\u0458\u0443\u043B", "\u0430\u0432\u0433", "\u0441\u0435\u043F", "\u043E\u043A\u0442", "\u043D\u043E\u0432", "\u0434\u0435\u0446", "" ], [ "\u0458\u0430\u043D\u0443\u0430\u0440", "\u0444\u0435\u0431\u0440\u0443\u0430\u0440", "\u043C\u0430\u0440\u0442", "\u0430\u043F\u0440\u0438\u043B", "\u043C\u0430\u0458", "\u0458\u0443\u043D", "\u0458\u0443\u043B", "\u0430\u0432\u0433\u0443\u0441\u0442", "\u0441\u0435\u043F\u0442\u0435\u043C\u0431\u0430\u0440", "\u043E\u043A\u0442\u043E\u0431\u0430\u0440", "\u043D\u043E\u0432\u0435\u043C\u0431\u0430\u0440", "\u0434\u0435\u0446\u0435\u043C\u0431\u0430\u0440", "" ] },
{ 0x0435, 0x007F, "zu-ZA", "zu", "zu", "zu-ZA", "\u005A\u0075\u006C\u0075\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", "\u0069\u0073\u0069\u005A\u0075\u006C\u0075\u0020\u0028\u0069\u004E\u0069\u006E\u0067\u0069\u007A\u0069\u006D\u0075\u0020\u0041\u0066\u0072\u0069\u006B\u0061\u0029", false, 0x00D1, "ZA", "ZA", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u0069\u004E\u0069\u006E\u0067\u0069\u007A\u0069\u006D\u0075\u0020\u0041\u0066\u0072\u0069\u006B\u0061", "\u005A\u0041\u0052", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u006E\u0020\u0052\u0061\u006E\u0064", "\u0052\u0061\u006E\u0064", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 6, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0069\u0053\u006F\u006E\u0074\u006F", "\u0075\u004D\u0073\u006F\u006D\u0062\u0075\u006C\u0075\u006B\u006F", "\u0075\u004C\u0077\u0065\u0073\u0069\u0062\u0069\u006C\u0069", "\u0075\u004C\u0077\u0065\u0073\u0069\u0074\u0068\u0061\u0074\u0068\u0075", "\u0075\u004C\u0077\u0065\u0073\u0069\u006E\u0065", "\u0075\u004C\u0077\u0065\u0073\u0069\u0068\u006C\u0061\u006E\u0075", "\u0075\u004D\u0067\u0071\u0069\u0062\u0065\u006C\u006F" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u0075\u004A\u0061\u006E\u0075\u0077\u0061\u0072\u0069", "\u0075\u0046\u0065\u0062\u0075\u0077\u0061\u0072\u0069", "\u0075\u004D\u0061\u0073\u0068\u0069", "\u0075\u0041\u0070\u0072\u0068\u0069\u006C\u0069", "\u0075\u004D\u0065\u0079\u0069", "\u0075\u004A\u0075\u006E\u0069", "\u0075\u004A\u0075\u006C\u0061\u0079\u0069", "\u0075\u0041\u0067\u0061\u0073\u0074\u0065", "\u0075\u0053\u0065\u0070\u0074\u0068\u0065\u006D\u0062\u0061", "\u0075\u004F\u006B\u0074\u0068\u006F\u0062\u0061", "\u0075\u004E\u006F\u0076\u0065\u006D\u0062\u0061", "\u0075\u0044\u0069\u0073\u0065\u006D\u0062\u0061", "" ] },
{ 0x0434, 0x007F, "xh-ZA", "xh", "xh", "xh-ZA", "\u0058\u0068\u006F\u0073\u0061\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", "\u0069\u0073\u0069\u0058\u0068\u006F\u0073\u0061\u0020\u0028\u0075\u004D\u007A\u0061\u006E\u0074\u0073\u0069\u0020\u0041\u0066\u0072\u0069\u006B\u0061\u0029", false, 0x00D1, "ZA", "ZA", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u0075\u004D\u007A\u0061\u006E\u0074\u0073\u0069\u0020\u0041\u0066\u0072\u0069\u006B\u0061", "\u005A\u0041\u0052", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u006E\u0020\u0052\u0061\u006E\u0064", "\u0052\u0061\u006E\u0064", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 6, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u0069\u0043\u0061\u0077\u0061", "\u0075\u004D\u0076\u0075\u006C\u006F", "\u0075\u004C\u0077\u0065\u0073\u0069\u0062\u0069\u006E\u0069", "\u0075\u004C\u0077\u0065\u0073\u0069\u0074\u0068\u0061\u0074\u0068\u0075", "\u0075\u004C\u0077\u0065\u0073\u0069\u006E\u0065", "\u0075\u004C\u0077\u0065\u0073\u0069\u0068\u006C\u0061\u006E\u0075", "\u0075\u004D\u0067\u0071\u0069\u0062\u0065\u006C\u006F" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u0065\u0079\u006F\u004D\u0071\u0075\u006E\u0067\u0075", "\u0065\u0079\u006F\u004D\u0064\u0075\u006D\u0062\u0061", "\u0065\u0079\u006F\u004B\u0077\u0069\u006E\u0064\u006C\u0061", "\u0054\u0073\u0068\u0061\u007A\u0069\u006D\u0070\u0075\u007A\u0069", "\u0043\u0061\u006E\u007A\u0069\u0062\u0065", "\u0065\u0079\u0065\u0053\u0069\u006C\u0069\u006D\u0065\u006C\u0061", "\u0065\u0079\u0065\u004B\u0068\u0061\u006C\u0061", "\u0065\u0079\u0065\u0054\u0068\u0075\u0070\u0068\u0061", "\u0065\u0079\u006F\u004D\u0073\u0069\u006E\u0074\u0073\u0069", "\u0065\u0079\u0065\u0044\u0077\u0061\u0072\u0061", "\u0065\u0079\u0065\u004E\u006B\u0061\u006E\u0067\u0061", "\u0065\u0079\u006F\u004D\u006E\u0067\u0061", "" ] },
{ 0x0432, 0x007F, "tn-ZA", "tn", "tn", "tn-ZA", "\u0054\u0073\u0077\u0061\u006E\u0061\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", "\u0053\u0065\u0074\u0073\u0077\u0061\u006E\u0061\u0020\u0028\u0041\u0066\u006F\u0072\u0069\u006B\u0061\u0020\u0042\u006F\u0072\u0077\u0061\u0029", false, 0x00D1, "ZA", "ZA", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u0041\u0066\u006F\u0072\u0069\u006B\u0061\u0020\u0042\u006F\u0072\u0077\u0061", "\u005A\u0041\u0052", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u006E\u0020\u0052\u0061\u006E\u0064", "\u0052\u0061\u006E\u0064", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 6, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u004C\u0061\u0074\u0073\u0068\u0069\u0070\u0069", "\u004D\u006F\u0073\u0075\u0070\u006F\u006C\u006F\u0067\u006F", "\u004C\u0061\u0062\u006F\u0062\u0065\u0064\u0069", "\u004C\u0061\u0062\u006F\u0072\u0061\u0072\u006F", "\u004C\u0061\u0062\u006F\u006E\u0065", "\u004C\u0061\u0062\u006F\u0074\u006C\u0068\u0061\u006E\u006F", "\u004C\u0061\u006D\u0061\u0074\u006C\u0068\u0061\u0074\u0073\u006F" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u0046\u0065\u0072\u0069\u006B\u0067\u006F\u006E\u0067", "\u0054\u006C\u0068\u0061\u006B\u006F\u006C\u0065", "\u004D\u006F\u0070\u0069\u0074\u006C\u006F\u0065", "\u004D\u006F\u0072\u0061\u006E\u0061\u006E\u0067", "\u004D\u006F\u0074\u0073\u0068\u0065\u0067\u0061\u006E\u006F\u006E\u0067", "\u0053\u0065\u0065\u0074\u0065\u0062\u006F\u0073\u0069\u0067\u006F", "\u0050\u0068\u0075\u006B\u0077\u0069", "\u0050\u0068\u0061\u0074\u0077\u0065", "\u004C\u0077\u0065\u0074\u0073\u0065", "\u0044\u0069\u0070\u0068\u0061\u006C\u0061\u006E\u0065", "\u004E\u0067\u0077\u0061\u006E\u0061\u0074\u0073\u0065\u006C\u0065", "\u0053\u0065\u0064\u0069\u006D\u006F\u0074\u0068\u006F\u006C\u0065", "" ] },
{ 0x083B, 0x007F, "se-SE", "se", "se", "se-SE", "\u0053\u0061\u006D\u0069\u0020\u0028\u004E\u006F\u0072\u0074\u0068\u0065\u0072\u006E\u0029\u0020\u0028\u0053\u0077\u0065\u0064\u0065\u006E\u0029", "\u0064\u0061\u0076\u0076\u0069\u0073\u00E1\u006D\u0065\u0067\u0069\u0065\u006C\u006C\u0061\u0020\u0028\u0052\u0075\u006F\u0167\u0167\u0061\u0029", false, 0x00DD, "SE", "SE", "\u0053\u0077\u0065\u0064\u0065\u006E", "\u0052\u0075\u006F\u0167\u0167\u0061", "\u0053\u0045\u004B", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068\u0020\u004B\u0072\u006F\u006E\u0061", "\u006B\u0072\u0075\u0076\u0064\u006E\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u002E", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 2, "\u002D", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u006F\u0074\u006E", "\u006D\u00E1\u006E", "\u0064\u0069\u0073", "\u0067\u0061\u0073\u006B", "\u0064\u0075\u006F\u0072", "\u0062\u0065\u0061\u0072", "\u006C\u00E1\u0076" ], [ "\u0073\u006F\u0074\u006E\u0061\u0062\u0065\u0061\u0069\u0076\u0069", "\u006D\u00E1\u006E\u006E\u006F\u0064\u0061\u0074", "\u0064\u0069\u0073\u0064\u0061\u0074", "\u0067\u0061\u0073\u006B\u0061\u0076\u0061\u0068\u006B\u006B\u0075", "\u0064\u0075\u006F\u0072\u0061\u0073\u0074\u0061\u0074", "\u0062\u0065\u0061\u0072\u006A\u0061\u0064\u0061\u0074", "\u006C\u00E1\u0076\u0076\u0061\u0072\u0064\u0061\u0074" ], [ "\u006F\u0111\u0111\u006A", "\u0067\u0075\u006F\u0076", "\u006E\u006A\u0075\u006B", "\u0063\u0075\u006F", "\u006D\u0069\u0065\u0073", "\u0067\u0065\u0061\u0073", "\u0073\u0075\u006F\u0069", "\u0062\u006F\u0072\u0067", "\u010D\u0061\u006B\u010D", "\u0067\u006F\u006C\u0067", "\u0073\u006B\u00E1\u0062", "\u006A\u0075\u006F\u0076", "" ], [ "\u006F\u0111\u0111\u0061\u006A\u0061\u0067\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0067\u0075\u006F\u0076\u0076\u0061\u006D\u00E1\u006E\u006E\u0075", "\u006E\u006A\u0075\u006B\u010D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u0063\u0075\u006F\u014B\u006F\u006D\u00E1\u006E\u006E\u0075", "\u006D\u0069\u0065\u0073\u0073\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0067\u0065\u0061\u0073\u0073\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0073\u0075\u006F\u0069\u0064\u006E\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0062\u006F\u0072\u0067\u0065\u006D\u00E1\u006E\u006E\u0075", "\u010D\u0061\u006B\u010D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u0067\u006F\u006C\u0067\u0067\u006F\u0074\u006D\u00E1\u006E\u006E\u0075", "\u0073\u006B\u00E1\u0062\u006D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u006A\u0075\u006F\u0076\u006C\u0061\u006D\u00E1\u006E\u006E\u0075", "" ] },
{ 0x1C3B, 0x007F, "sma-SE", "sma", "sma", "sma-SE", "\u0053\u0061\u006D\u0069\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0065\u0072\u006E\u0029\u0020\u0028\u0053\u0077\u0065\u0064\u0065\u006E\u0029", "\u00E5\u0061\u0072\u006A\u0065\u006C\u0073\u0061\u0065\u006D\u0069\u0065\u006E\u0067\u0069\u0065\u006C\u0065\u0020\u0028\u0053\u0076\u0065\u0065\u0072\u006A\u0065\u0029", false, 0x00DD, "SE", "SE", "\u0053\u0077\u0065\u0064\u0065\u006E", "\u0053\u0076\u0065\u0065\u0072\u006A\u0065", "\u0053\u0045\u004B", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068\u0020\u004B\u0072\u006F\u006E\u0061", "\u006B\u0072\u00E5\u0076\u006E\u006F\u0065", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u002E", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 2, "\u002D", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0061\u0065\u006A", "\u006D\u00E5\u0061", "\u0064\u00E6\u006A", "\u0067\u0061\u0073\u006B", "\u0064\u0075\u0061\u0072", "\u0062\u0065\u0061\u0072\u006A", "\u006C\u0061\u0061\u0076" ], [ "\u0061\u0065\u006A\u006C\u0065\u0067\u0065", "\u006D\u00E5\u0061\u006E\u0074\u0061", "\u0064\u00E6\u006A\u0073\u0074\u0061", "\u0067\u0061\u0073\u006B\u0065\u0076\u00E5\u0068\u006B\u006F\u0065", "\u0064\u0075\u0061\u0072\u0073\u0074\u0061", "\u0062\u0065\u0061\u0072\u006A\u0061\u0064\u0061\u0068\u006B\u0065", "\u006C\u0061\u0061\u0076\u0076\u0061\u0072\u0064\u0061\u0068\u006B\u0065" ], [ "\u0074\u0073\u00EF\u0065\u006E", "\u0067\u006F\u0065\u0076\u0074", "\u006E\u006A\u006F\u006B", "\u0076\u006F\u0065\u0072", "\u0073\u0075\u0065\u0068", "\u0072\u0075\u0066\u0066", "\u0073\u006E\u006A\u0061", "\u006D\u00EF\u0065\u0074", "\u0073\u006B\u00EF\u0065\u0072", "\u0067\u006F\u006C\u006B", "\u0072\u0061\u0068\u006B", "\u0067\u006F\u0065\u0076", "" ], [ "\u0074\u0073\u00EF\u0065\u006E\u0067\u0065\u006C\u0065", "\u0067\u006F\u0065\u0076\u0074\u0065", "\u006E\u006A\u006F\u006B\u0074\u006A\u0065", "\u0076\u006F\u0065\u0072\u0068\u0074\u006A\u0065", "\u0073\u0075\u0065\u0068\u0070\u0065\u0064\u0065", "\u0072\u0075\u0066\u0066\u0069\u0065", "\u0073\u006E\u006A\u0061\u006C\u0074\u006A\u0065", "\u006D\u00EF\u0065\u0074\u0073\u006B\u0065", "\u0073\u006B\u00EF\u0065\u0072\u0065\u0064\u0065", "\u0067\u006F\u006C\u006B\u0065", "\u0072\u0061\u0068\u006B\u0061", "\u0067\u006F\u0065\u0076\u0065", "" ] },
{ 0x101A, 0x007F, "hr-BA", "hr", "hr", "hr-BA", "\u0043\u0072\u006F\u0061\u0074\u0069\u0061\u006E\u0020\u0028\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", "\u0068\u0072\u0076\u0061\u0074\u0073\u006B\u0069\u0020\u0028\u0042\u006F\u0073\u006E\u0061\u0020\u0069\u0020\u0048\u0065\u0072\u0063\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", false, 0x0019, "BA", "BA", "\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0042\u006F\u0073\u006E\u0061\u0020\u0069\u0020\u0048\u0065\u0072\u0063\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0042\u0041\u004D", "\u0043\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u006C\u0065\u0020\u004D\u0061\u0072\u006B\u0073", "\u006B\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u0069\u006C\u006E\u0061\u0020\u006D\u0061\u0072\u006B\u0061", true, 2, 2, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u004B\u004D", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006A\u0061\u006E\u0073\u006B\u0069\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u002C\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065\u0064", "\u0070\u006F\u006E", "\u0075\u0074\u006F", "\u0073\u0072\u0069", "\u010D\u0065\u0074", "\u0070\u0065\u0074", "\u0073\u0075\u0062" ], [ "\u006E\u0065\u0064\u006A\u0065\u006C\u006A\u0061", "\u0070\u006F\u006E\u0065\u0064\u006A\u0065\u006C\u006A\u0061\u006B", "\u0075\u0074\u006F\u0072\u0061\u006B", "\u0073\u0072\u0069\u006A\u0065\u0064\u0061", "\u010D\u0065\u0074\u0076\u0072\u0074\u0061\u006B", "\u0070\u0065\u0074\u0061\u006B", "\u0073\u0075\u0062\u006F\u0074\u0061" ], [ "\u0073\u0069\u006A", "\u0076\u006C\u006A", "\u006F\u017E\u0075", "\u0074\u0072\u0061", "\u0073\u0076\u0069", "\u006C\u0069\u0070", "\u0073\u0072\u0070", "\u006B\u006F\u006C", "\u0072\u0075\u006A", "\u006C\u0069\u0073", "\u0073\u0074\u0075", "\u0070\u0072\u006F", "" ], [ "\u0073\u0069\u006A\u0065\u010D\u0061\u006E\u006A", "\u0076\u0065\u006C\u006A\u0061\u010D\u0061", "\u006F\u017E\u0075\u006A\u0061\u006B", "\u0074\u0072\u0061\u0076\u0061\u006E\u006A", "\u0073\u0076\u0069\u0062\u0061\u006E\u006A", "\u006C\u0069\u0070\u0061\u006E\u006A", "\u0073\u0072\u0070\u0061\u006E\u006A", "\u006B\u006F\u006C\u006F\u0076\u006F\u007A", "\u0072\u0075\u006A\u0061\u006E", "\u006C\u0069\u0073\u0074\u006F\u0070\u0061\u0064", "\u0073\u0074\u0075\u0064\u0065\u006E\u0069", "\u0070\u0072\u006F\u0073\u0069\u006E\u0061\u0063", "" ] },
{ 0x243B, 0x007F, "smn-FI", "smn", "smn", "smn-FI", "\u0053\u0061\u006D\u0069\u0020\u0028\u0049\u006E\u0061\u0072\u0069\u0029\u0020\u0028\u0046\u0069\u006E\u006C\u0061\u006E\u0064\u0029", "\u0073\u00E4\u006D\u0069\u006B\u0069\u0065\u006C\u00E2\u0020\u0028\u0053\u0075\u006F\u006D\u00E2\u0029", false, 0x004D, "FI", "FI", "\u0046\u0069\u006E\u006C\u0061\u006E\u0064", "\u0053\u0075\u006F\u006D\u00E2", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0076\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0069\u006E\u0065\u006E\u0020\u006B\u0061\u006C\u0065\u006E\u0074\u0065\u0072\u0069", 0, 2, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0070\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0070\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0070\u0061", "\u0076\u0075", "\u006D\u0061", "\u006B\u006F", "\u0074\u0075", "\u0076\u00E1", "\u006C\u00E1" ], [ "\u0070\u0061\u0073\u0065\u0070\u0065\u0069\u0076\u0069", "\u0076\u0075\u006F\u0073\u0073\u0061\u0072\u0067\u00E2", "\u006D\u0061\u006A\u0065\u0062\u0061\u0072\u0067\u00E2", "\u006B\u006F\u0073\u006B\u006F\u006B\u006B\u006F", "\u0074\u0075\u006F\u0072\u00E2\u0073\u0074\u00E2\u0068", "\u0076\u00E1\u0073\u0074\u0075\u0070\u0070\u0065\u0069\u0076\u0069", "\u006C\u00E1\u0076\u00E1\u0072\u0064\u00E2\u0068" ], [ "\u0075\u0111\u0069\u0076", "\u006B\u0075\u006F\u0076", "\u006E\u006A\u0075\u0068", "\u0063\u0075\u006F\u014B", "\u0076\u0079\u0065\u0073", "\u006B\u0065\u0073\u0069", "\u0073\u0079\u0065\u0069", "\u0070\u006F\u0072\u0067", "\u010D\u006F\u0068", "\u0072\u006F\u006F\u0076", "\u0073\u006B\u0061", "\u006A\u0075\u006F\u0076", "" ], [ "\u0075\u0111\u0111\u00E2\u0069\u0076\u0065\u006D\u00E1\u00E1\u006E\u0075", "\u006B\u0075\u006F\u0076\u00E2\u006D\u00E1\u00E1\u006E\u0075", "\u006E\u006A\u0075\u0068\u010D\u00E2\u006D\u00E1\u00E1\u006E\u0075", "\u0063\u0075\u00E1\u014B\u0075\u0069\u006D\u00E1\u00E1\u006E\u0075", "\u0076\u0079\u0065\u0073\u0069\u006D\u00E1\u00E1\u006E\u0075", "\u006B\u0065\u0073\u0069\u006D\u00E1\u00E1\u006E\u0075", "\u0073\u0079\u0065\u0069\u006E\u0069\u006D\u00E1\u00E1\u006E\u0075", "\u0070\u006F\u0072\u0067\u0065\u006D\u00E1\u00E1\u006E\u0075", "\u010D\u006F\u0068\u010D\u00E2\u006D\u00E1\u00E1\u006E\u0075", "\u0072\u006F\u006F\u0076\u0076\u00E2\u0064\u006D\u00E1\u00E1\u006E\u0075", "\u0073\u006B\u0061\u006D\u006D\u00E2\u006D\u00E1\u00E1\u006E\u0075", "\u006A\u0075\u006F\u0076\u006C\u00E2\u006D\u00E1\u00E1\u006E\u0075", "" ] },
{ 0x0C6B, 0x007F, "quz-PE", "quz", "quz", "quz-PE", "\u0051\u0075\u0065\u0063\u0068\u0075\u0061\u0020\u0028\u0050\u0065\u0072\u0075\u0029", "\u0072\u0075\u006E\u0061\u0073\u0069\u006D\u0069\u0020\u0028\u0050\u0065\u0072\u0075\u0020\u0053\u0075\u0079\u0075\u0029", false, 0x00BB, "PE", "PE", "\u0050\u0065\u0072\u0075", "\u0050\u0065\u0072\u0075\u0020\u0053\u0075\u0079\u0075", "\u0050\u0045\u004E", "\u0050\u0065\u0072\u0075\u0076\u0069\u0061\u006E\u0020\u004E\u0075\u0065\u0076\u006F\u0020\u0053\u006F\u006C", "\u004E\u0075\u0065\u0076\u006F\u0020\u0053\u006F\u006C", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002E", "\u002C", "\u0053\u002F\u002E", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 6, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0069\u006E\u0074", "\u006B\u0069\u006C", "\u0061\u0074\u0069", "\u0071\u0075\u0079", "\u0043\u0068\u0092", "\u0049\u006C\u006C", "\u006B\u0027\u0075" ], [ "\u0069\u006E\u0074\u0069\u0063\u0068\u0061\u0077", "\u006B\u0069\u006C\u006C\u0061\u0063\u0068\u0061\u0077", "\u0061\u0074\u0069\u0070\u0061\u0063\u0068\u0061\u0077", "\u0071\u0075\u0079\u006C\u006C\u0075\u0072\u0063\u0068\u0061\u0077", "\u0043\u0068\u0027\u0020\u0061\u0073\u006B\u0061\u0063\u0068\u0061\u0077", "\u0049\u006C\u006C\u0061\u0070\u0061\u0063\u0068\u0061\u0077", "\u006B\u0027\u0075\u0079\u0063\u0068\u0069\u0063\u0068\u0061\u0077" ], [ "\u0051\u0075\u006C", "\u0048\u0061\u0074", "\u0050\u0061\u0075", "\u0061\u0079\u0072", "\u0041\u0079\u006D", "\u0049\u006E\u0074", "\u0041\u006E\u0074", "\u0051\u0068\u0061", "\u0055\u006D\u0061", "\u004B\u0061\u006E", "\u0041\u0079\u0061", "\u004B\u0061\u0070", "" ], [ "\u0051\u0075\u006C\u006C\u0061\u0020\u0070\u0075\u0071\u0075\u0079", "\u0048\u0061\u0074\u0075\u006E\u0020\u0070\u0075\u0071\u0075\u0079", "\u0050\u0061\u0075\u0071\u0061\u0072\u0020\u0077\u0061\u0072\u0061\u0079", "\u0061\u0079\u0072\u0069\u0077\u0061", "\u0041\u0079\u006D\u0075\u0072\u0061\u0079", "\u0049\u006E\u0074\u0069\u0020\u0072\u0061\u0079\u006D\u0069", "\u0041\u006E\u0074\u0061\u0020\u0053\u0069\u0074\u0077\u0061", "\u0051\u0068\u0061\u0070\u0061\u0071\u0020\u0053\u0069\u0074\u0077\u0061", "\u0055\u006D\u0061\u0020\u0072\u0061\u0079\u006D\u0069", "\u004B\u0061\u006E\u0074\u0061\u0072\u0061\u0079", "\u0041\u0079\u0061\u006D\u0061\u0072\u0071\u0027\u0061", "\u004B\u0061\u0070\u0061\u0071\u0020\u0052\u0061\u0079\u006D\u0069", "" ] },
{ 0x0C3B, 0x007F, "se-FI", "se", "se", "se-FI", "\u0053\u0061\u006D\u0069\u0020\u0028\u004E\u006F\u0072\u0074\u0068\u0065\u0072\u006E\u0029\u0020\u0028\u0046\u0069\u006E\u006C\u0061\u006E\u0064\u0029", "\u0064\u0061\u0076\u0076\u0069\u0073\u00E1\u006D\u0065\u0067\u0069\u0065\u006C\u006C\u0061\u0020\u0028\u0053\u0075\u006F\u0070\u006D\u0061\u0029", false, 0x004D, "FI", "FI", "\u0046\u0069\u006E\u006C\u0061\u006E\u0064", "\u0053\u0075\u006F\u0070\u006D\u0061", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0069\u006E\u0065\u006E\u0020\u006B\u0061\u006C\u0065\u006E\u0074\u0065\u0072\u0069", 0, 2, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u006F\u0074\u006E", "\u0076\u0075\u006F\u0073", "\u006D\u0061\u014B", "\u0067\u0061\u0073\u006B", "\u0064\u0075\u006F\u0072", "\u0062\u0065\u0061\u0072", "\u006C\u00E1\u0076" ], [ "\u0073\u006F\u0074\u006E\u0061\u0062\u0065\u0061\u0069\u0076\u0069", "\u0076\u0075\u006F\u0073\u0073\u00E1\u0072\u0067\u0061", "\u006D\u0061\u014B\u014B\u0065\u0062\u00E1\u0072\u0067\u0061", "\u0067\u0061\u0073\u006B\u0061\u0076\u0061\u0068\u006B\u006B\u0075", "\u0064\u0075\u006F\u0072\u0061\u0073\u0074\u0061\u0074", "\u0062\u0065\u0061\u0072\u006A\u0061\u0064\u0061\u0074", "\u006C\u00E1\u0076\u0076\u0061\u0072\u0064\u0061\u0074" ], [ "\u006F\u0111\u0111\u006A", "\u0067\u0075\u006F\u0076", "\u006E\u006A\u0075\u006B", "\u0063\u0075\u006F", "\u006D\u0069\u0065\u0073", "\u0067\u0065\u0061\u0073", "\u0073\u0075\u006F\u0069", "\u0062\u006F\u0072\u0067", "\u010D\u0061\u006B\u010D", "\u0067\u006F\u006C\u0067", "\u0073\u006B\u00E1\u0062", "\u006A\u0075\u006F\u0076", "" ], [ "\u006F\u0111\u0111\u0061\u006A\u0061\u0067\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0067\u0075\u006F\u0076\u0076\u0061\u006D\u00E1\u006E\u006E\u0075", "\u006E\u006A\u0075\u006B\u010D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u0063\u0075\u006F\u014B\u006F\u006D\u00E1\u006E\u006E\u0075", "\u006D\u0069\u0065\u0073\u0073\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0067\u0065\u0061\u0073\u0073\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0073\u0075\u006F\u0069\u0064\u006E\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0062\u006F\u0072\u0067\u0065\u006D\u00E1\u006E\u006E\u0075", "\u010D\u0061\u006B\u010D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u0067\u006F\u006C\u0067\u0067\u006F\u0074\u006D\u00E1\u006E\u006E\u0075", "\u0073\u006B\u00E1\u0062\u006D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u006A\u0075\u006F\u0076\u006C\u0061\u006D\u00E1\u006E\u006E\u0075", "" ] },
{ 0x203B, 0x007F, "sms-FI", "sms", "sms", "sms-FI", "\u0053\u0061\u006D\u0069\u0020\u0028\u0053\u006B\u006F\u006C\u0074\u0029\u0020\u0028\u0046\u0069\u006E\u006C\u0061\u006E\u0064\u0029", "\u0073\u00E4\u00E4\u006D\u00B4\u01E9\u0069\u00F5\u006C\u006C\u0020\u0028\u004C\u00E4\u00E4\u00B4\u0064\u0064\u006A\u00E2\u006E\u006E\u0061\u006D\u0029", false, 0x004D, "FI", "FI", "\u0046\u0069\u006E\u006C\u0061\u006E\u0064", "\u004C\u00E4\u00E4\u00B4\u0064\u0064\u006A\u00E2\u006E\u006E\u0061\u006D", "\u0045\u0055\u0052", "\u0045\u0075\u0072\u006F", "\u0065\u0075\u0072\u006F", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u20AC", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u0061\u006E\u0069\u006E\u0065\u006E\u0020\u006B\u0061\u006C\u0065\u006E\u0074\u0065\u0072\u0069", 0, 2, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0070\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0070\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0070\u00E2", "\u0076\u0075", "\u006D\u00E2", "\u0073\u0065", "\u006E\u0065", "\u0070\u0069", "\u0073\u0075" ], [ "\u0070\u00E2\u00B4\u0073\u0073\u0070\u0065\u0069\u00B4\u0076\u0076", "\u0076\u0075\u00F5\u0073\u0073\u0061\u0072\u0067\u0067", "\u006D\u00E2\u00E2\u0069\u0062\u0061\u0072\u0067\u0067", "\u0073\u0065\u00E4\u0072\u0061\u0064", "\u006E\u0065\u006C\u006C\u006A\u0064\u0070\u0065\u0069\u00B4\u0076\u0076", "\u0070\u0069\u00E2\u0074\u006E\u00E2\u0063", "\u0073\u0075\u0065\u00B4\u0076\u0065\u0074" ], [ "\u006F\u0111\u006A\u006D", "\u0074\u00E4\u00B4\u006C\u0076\u0076", "\u0070\u00E2\u007A\u006C", "\u006E\u006A\u0075\u0068", "\u0076\u0075\u0065", "\u01E9\u0069\u0065", "\u0073\u0075\u0065\u0069", "\u0070\u00E5\u00B4\u0072", "\u010D\u00F5\u0068", "\u006B\u00E5\u006C\u0067", "\u0073\u006B\u0061", "\u0072\u006F\u0073\u0074", "" ], [ "\u006F\u0111\u0111\u0065\u0065\u00B4\u006A\u006A\u006D\u00E4\u00E4\u006E", "\u0074\u00E4\u00B4\u006C\u0076\u0076\u006D\u00E4\u00E4\u006E", "\u0070\u00E2\u00B4\u007A\u007A\u006C\u00E2\u0161\u0074\u0074\u0061\u006D\u006D\u00E4\u00E4\u006E", "\u006E\u006A\u0075\u0068\u010D\u010D\u006D\u00E4\u00E4\u006E", "\u0076\u0075\u0065\u00B4\u0073\u0073\u006D\u00E4\u00E4\u006E", "\u01E9\u0069\u0065\u00B4\u0073\u0073\u006D\u00E4\u00E4\u006E", "\u0073\u0075\u0065\u0069\u00B4\u006E\u006E\u006D\u00E4\u00E4\u006E", "\u0070\u00E5\u00B4\u0072\u01E7\u01E7\u006D\u00E4\u00E4\u006E", "\u010D\u00F5\u0068\u010D\u010D\u006D\u00E4\u00E4\u006E", "\u006B\u00E5\u006C\u0067\u0067\u006D\u00E4\u00E4\u006E", "\u0073\u006B\u0061\u006D\u006D\u00B4\u006D\u00E4\u00E4\u006E", "\u0072\u006F\u0073\u0074\u0074\u006F\u0076\u006D\u00E4\u00E4\u006E", "" ] },
{ 0x0452, 0x007F, "cy-GB", "cy", "cy", "cy-GB", "\u0057\u0065\u006C\u0073\u0068\u0020\u0028\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u004B\u0069\u006E\u0067\u0064\u006F\u006D\u0029", "\u0043\u0079\u006D\u0072\u0061\u0065\u0067\u0020\u0028\u0079\u0020\u0044\u0065\u0079\u0072\u006E\u0061\u0073\u0020\u0055\u006E\u0065\u0064\u0069\u0067\u0029", false, 0x00F2, "GB", "GB", "\u0055\u006E\u0069\u0074\u0065\u0064\u0020\u004B\u0069\u006E\u0067\u0064\u006F\u006D", "\u0079\u0020\u0044\u0065\u0079\u0072\u006E\u0061\u0073\u0020\u0055\u006E\u0065\u0064\u0069\u0067", "\u0047\u0042\u0050", "\u0055\u004B\u0020\u0050\u006F\u0075\u006E\u0064\u0020\u0053\u0074\u0065\u0072\u006C\u0069\u006E\u0067", "\u0070\u0075\u006E\u0074", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u00A3", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0072\u0020\u0047\u0072\u0065\u0067\u006F\u0072\u0069", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006C", "\u004C\u006C\u0075\u006E", "\u004D\u0061\u0077", "\u004D\u0065\u0072", "\u0049\u0061\u0075", "\u0047\u0077\u0065", "\u0053\u0061\u0064" ], [ "\u0044\u0079\u0064\u0064\u0020\u0053\u0075\u006C", "\u0044\u0079\u0064\u0064\u0020\u004C\u006C\u0075\u006E", "\u0044\u0079\u0064\u0064\u0020\u004D\u0061\u0077\u0072\u0074\u0068", "\u0044\u0079\u0064\u0064\u0020\u004D\u0065\u0072\u0063\u0068\u0065\u0072", "\u0044\u0079\u0064\u0064\u0020\u0049\u0061\u0075", "\u0044\u0079\u0064\u0064\u0020\u0047\u0077\u0065\u006E\u0065\u0072", "\u0044\u0079\u0064\u0064\u0020\u0053\u0061\u0064\u0077\u0072\u006E" ], [ "\u0049\u006F\u006E", "\u0043\u0068\u0077\u0065", "\u004D\u0061\u0077", "\u0045\u0062\u0072", "\u004D\u0061\u0069", "\u004D\u0065\u0068", "\u0047\u006F\u0072", "\u0041\u0077\u0073", "\u004D\u0065\u0064", "\u0048\u0079\u0064", "\u0054\u0061\u0063\u0068", "\u0052\u0068\u0061\u0067", "" ], [ "\u0049\u006F\u006E\u0061\u0077\u0072", "\u0043\u0068\u0077\u0065\u0066\u0072\u006F\u0072", "\u004D\u0061\u0077\u0072\u0074\u0068", "\u0045\u0062\u0072\u0069\u006C\u006C", "\u004D\u0061\u0069", "\u004D\u0065\u0068\u0065\u0066\u0069\u006E", "\u0047\u006F\u0072\u0066\u0066\u0065\u006E\u006E\u0061\u0066", "\u0041\u0077\u0073\u0074", "\u004D\u0065\u0064\u0069", "\u0048\u0079\u0064\u0072\u0065\u0066", "\u0054\u0061\u0063\u0068\u0077\u0065\u0064\u0064", "\u0052\u0068\u0061\u0067\u0066\u0079\u0072", "" ] },
{ 0x141A, 0x007F, "bs-BA-Latn", "bs", "bs", "bs-Latn-BA", "\u0042\u006F\u0073\u006E\u0069\u0061\u006E\u0020\u0028\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", "\u0062\u006F\u0073\u0061\u006E\u0073\u006B\u0069\u0020\u0028\u0042\u006F\u0073\u006E\u0061\u0020\u0069\u0020\u0048\u0065\u0072\u0063\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", false, 0x0019, "BA", "BA", "\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0042\u006F\u0073\u006E\u0061\u0020\u0069\u0020\u0048\u0065\u0072\u0063\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0042\u0041\u004D", "\u0043\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u006C\u0065\u0020\u004D\u0061\u0072\u006B\u0073", "\u006B\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u0069\u006C\u006E\u0061\u0020\u006D\u0061\u0072\u006B\u0061", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u004B\u004D", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006A\u0061\u006E\u0073\u006B\u0069\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065\u0064", "\u0070\u006F\u006E", "\u0075\u0074\u006F", "\u0073\u0072\u0069", "\u010D\u0065\u0074", "\u0070\u0065\u0074", "\u0073\u0075\u0062" ], [ "\u006E\u0065\u0064\u006A\u0065\u006C\u006A\u0061", "\u0070\u006F\u006E\u0065\u0064\u006A\u0065\u006C\u006A\u0061\u006B", "\u0075\u0074\u006F\u0072\u0061\u006B", "\u0073\u0072\u0069\u006A\u0065\u0064\u0061", "\u010D\u0065\u0074\u0076\u0072\u0074\u0061\u006B", "\u0070\u0065\u0074\u0061\u006B", "\u0073\u0075\u0062\u006F\u0074\u0061" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0074", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0061\u0072", "\u006F\u006B\u0074\u006F\u0062\u0061\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0061\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0061\u0072", "" ] },
{ 0x103B, 0x007F, "smj-NO", "smj", "smj", "smj-NO", "\u0053\u0061\u006D\u0069\u0020\u0028\u004C\u0075\u006C\u0065\u0029\u0020\u0028\u004E\u006F\u0072\u0077\u0061\u0079\u0029", "\u006A\u0075\u006C\u0065\u0076\u0075\u0073\u00E1\u006D\u0065\u0067\u0069\u0065\u006C\u006C\u0061\u0020\u0028\u0056\u0075\u006F\u0064\u006E\u0061\u0029", false, 0x00B1, "NO", "NO", "\u004E\u006F\u0072\u0077\u0061\u0079", "\u0056\u0075\u006F\u0064\u006E\u0061", "\u004E\u004F\u004B", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u0020\u004B\u0072\u006F\u006E\u0065", "\u006B\u0072\u00E5\u0076\u006E\u006E\u00E5", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u00E5\u0064", "\u006D\u00E1\u006E", "\u0064\u0069\u0073", "\u0067\u0061\u0073", "\u0064\u0075\u006F\u0072", "\u0062\u0069\u0065\u0072", "\u006C\u00E1\u0076" ], [ "\u0073\u00E5\u0064\u006E\u00E5\u0062\u0069\u0065\u006A\u0076\u0076\u0065", "\u006D\u00E1\u006E\u006E\u006F\u0064\u0061\u0068\u006B\u0061", "\u0064\u0069\u006A\u0073\u0074\u0061\u0068\u006B\u0061", "\u0067\u0061\u0073\u0073\u006B\u0061\u0076\u0061\u0068\u006B\u006B\u006F", "\u0064\u0075\u006F\u0072\u0061\u0073\u0074\u0061\u0068\u006B\u0061", "\u0062\u0069\u0065\u0072\u006A\u006A\u0065\u0064\u0061\u0068\u006B\u0061", "\u006C\u00E1\u0076\u0076\u006F\u0064\u0061\u0068\u006B\u0061" ], [ "\u00E5\u0064\u00E5\u006A", "\u0067\u0075\u006F\u0076", "\u0073\u006E\u006A\u0075", "\u0076\u0075\u006F\u0072", "\u006D\u006F\u0061\u0072", "\u0062\u0069\u0065\u0068", "\u0073\u006E\u006A\u0069", "\u0062\u00E5\u0072\u0067", "\u0072\u0061\u0067\u00E1", "\u0067\u00E5\u006C\u0067", "\u0062\u0061\u0073\u00E1", "\u006A\u0061\u0076\u006C", "" ], [ "\u00E5\u0064\u00E5\u006A\u0061\u006B\u006D\u00E1\u006E\u006E\u006F", "\u0067\u0075\u006F\u0076\u0076\u0061\u006D\u00E1\u006E\u006E\u006F", "\u0073\u006A\u006E\u006A\u0075\u006B\u0074\u006A\u0061\u006D\u00E1\u006E\u006E\u006F", "\u0076\u0075\u006F\u0072\u0061\u0074\u006A\u0069\u0073\u006D\u00E1\u006E\u006E\u006F", "\u006D\u006F\u0061\u0072\u006D\u0065\u0073\u006D\u00E1\u006E\u006E\u006F", "\u0062\u0069\u0065\u0068\u0074\u0073\u0065\u006D\u00E1\u006E\u006E\u006F", "\u0073\u006A\u006E\u006A\u0069\u006C\u006C\u0074\u006A\u0061\u006D\u00E1\u006E\u006E\u006F", "\u0062\u00E5\u0072\u0067\u0067\u0065\u006D\u00E1\u006E\u006E\u006F", "\u0072\u0061\u0067\u00E1\u0074\u006D\u00E1\u006E\u006E\u006F", "\u0067\u00E5\u006C\u0067\u00E5\u0064\u0069\u0073\u006D\u00E1\u006E\u006E\u006F", "\u0062\u0061\u0073\u00E1\u0064\u0069\u0073\u006D\u00E1\u006E\u006E\u006F", "\u006A\u0061\u0076\u006C\u006C\u0061\u006D\u00E1\u006E\u006E\u006F", "" ] },
{ 0x0481, 0x007F, "mi-NZ", "mi", "mi", "mi-NZ", "\u004D\u0061\u006F\u0072\u0069\u0020\u0028\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064\u0029", "\u0052\u0065\u006F\u0020\u004D\u0101\u006F\u0072\u0069\u0020\u0028\u0041\u006F\u0074\u0065\u0061\u0072\u006F\u0061\u0029", false, 0x00B7, "NZ", "NZ", "\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064", "\u0041\u006F\u0074\u0065\u0061\u0072\u006F\u0061", "\u004E\u005A\u0044", "\u004E\u0065\u0077\u0020\u005A\u0065\u0061\u006C\u0061\u006E\u0064\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0074\u0101\u0072\u0061", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0054\u0061", "\u004D\u0061", "\u0054\u016B", "\u0057\u0065", "\u0054\u0101\u0069", "\u0050\u0061", "\u0048\u0101" ], [ "\u0052\u0101\u0074\u0061\u0070\u0075", "\u004D\u0061\u006E\u0065", "\u0054\u016B\u0072\u0065\u0069", "\u0057\u0065\u006E\u0065\u0072\u0065\u0069", "\u0054\u0101\u0069\u0074\u0065", "\u0050\u0061\u0072\u0061\u0069\u0072\u0065", "\u0048\u0101\u0074\u0061\u0072\u0065\u0069" ], [ "\u004B\u006F\u0068\u0069", "\u0048\u0075\u0069", "\u0050\u006F\u0075", "\u0050\u0061\u0065", "\u0048\u0061\u0072\u0061", "\u0050\u0069\u0070\u0069", "\u0048\u014D\u006E\u0067\u006F\u0069", "\u0048\u0065\u0072\u0065", "\u004D\u0061\u0068\u0075", "\u0057\u0068\u0069\u002D\u006E\u0075", "\u0057\u0068\u0069\u002D\u0072\u0061", "\u0048\u0061\u006B\u0069", "" ], [ "\u004B\u006F\u0068\u0069\u002D\u0074\u0101\u0074\u0065\u0061", "\u0048\u0075\u0069\u002D\u0074\u0061\u006E\u0067\u0075\u0072\u0075", "\u0050\u006F\u0075\u0074\u016B\u002D\u0074\u0065\u002D\u0072\u0061\u006E\u0067\u0069", "\u0050\u0061\u0065\u006E\u0067\u0061\u002D\u0077\u0068\u0101\u0077\u0068\u0101", "\u0048\u0061\u0072\u0061\u0074\u0075\u0061", "\u0050\u0069\u0070\u0069\u0072\u0069", "\u0048\u014D\u006E\u0067\u006F\u0069\u006E\u0067\u006F\u0069", "\u0048\u0065\u0072\u0065\u002D\u0074\u0075\u0072\u0069\u002D\u006B\u014D\u006B\u0101", "\u004D\u0061\u0068\u0075\u0072\u0075", "\u0057\u0068\u0069\u0072\u0069\u006E\u0067\u0061\u002D\u0101\u002D\u006E\u0075\u006B\u0075", "\u0057\u0068\u0069\u0072\u0069\u006E\u0067\u0061\u002D\u0101\u002D\u0072\u0061\u006E\u0067\u0069", "\u0048\u0061\u006B\u0069\u0068\u0065\u0061", "" ] },
{ 0x086B, 0x007F, "quz-EC", "quz", "quz", "quz-EC", "\u0051\u0075\u0065\u0063\u0068\u0075\u0061\u0020\u0028\u0045\u0063\u0075\u0061\u0064\u006F\u0072\u0029", "\u0072\u0075\u006E\u0061\u0073\u0069\u006D\u0069\u0020\u0028\u0045\u0063\u0075\u0061\u0064\u006F\u0072\u0020\u0053\u0075\u0079\u0075\u0029", false, 0x0042, "EC", "EC", "\u0045\u0063\u0075\u0061\u0064\u006F\u0072", "\u0045\u0063\u0075\u0061\u0064\u006F\u0072\u0020\u0053\u0075\u0079\u0075", "\u0055\u0053\u0044", "\u0055\u0053\u0020\u0044\u006F\u006C\u006C\u0061\u0072", "\u0055\u0053\u0020\u0044\u006F\u006C\u0061\u0072", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 6, 0, "\u002F", "\u003A", "", "", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0069\u006E\u0074", "\u006B\u0069\u006C", "\u0061\u0074\u0069", "\u0071\u0075\u0079", "\u0043\u0068\u0092", "\u0049\u006C\u006C", "\u006B\u0027\u0075" ], [ "\u0069\u006E\u0074\u0069\u0063\u0068\u0061\u0077", "\u006B\u0069\u006C\u006C\u0061\u0063\u0068\u0061\u0077", "\u0061\u0074\u0069\u0070\u0061\u0063\u0068\u0061\u0077", "\u0071\u0075\u0079\u006C\u006C\u0075\u0072\u0063\u0068\u0061\u0077", "\u0043\u0068\u0027\u0020\u0061\u0073\u006B\u0061\u0063\u0068\u0061\u0077", "\u0049\u006C\u006C\u0061\u0070\u0061\u0063\u0068\u0061\u0077", "\u006B\u0027\u0075\u0079\u0063\u0068\u0069\u0063\u0068\u0061\u0077" ], [ "\u0051\u0075\u006C", "\u0048\u0061\u0074", "\u0050\u0061\u0075", "\u0061\u0079\u0072", "\u0041\u0079\u006D", "\u0049\u006E\u0074", "\u0041\u006E\u0074", "\u0051\u0068\u0061", "\u0055\u006D\u0061", "\u004B\u0061\u006E", "\u0041\u0079\u0061", "\u004B\u0061\u0070", "" ], [ "\u0051\u0075\u006C\u006C\u0061\u0020\u0070\u0075\u0071\u0075\u0079", "\u0048\u0061\u0074\u0075\u006E\u0020\u0070\u0075\u0071\u0075\u0079", "\u0050\u0061\u0075\u0071\u0061\u0072\u0020\u0077\u0061\u0072\u0061\u0079", "\u0061\u0079\u0072\u0069\u0077\u0061", "\u0041\u0079\u006D\u0075\u0072\u0061\u0079", "\u0049\u006E\u0074\u0069\u0020\u0072\u0061\u0079\u006D\u0069", "\u0041\u006E\u0074\u0061\u0020\u0053\u0069\u0074\u0077\u0061", "\u0051\u0068\u0061\u0070\u0061\u0071\u0020\u0053\u0069\u0074\u0077\u0061", "\u0055\u006D\u0061\u0020\u0072\u0061\u0079\u006D\u0069", "\u004B\u0061\u006E\u0074\u0061\u0072\u0061\u0079", "\u0041\u0079\u0061\u006D\u0061\u0072\u0071\u0027\u0061", "\u004B\u0061\u0070\u0061\u0071\u0020\u0052\u0061\u0079\u006D\u0069", "" ] },
{ 0x181A, 0x007F, "sr-BA-Latn", "sr", "sr", "sr-Latn-BA", "\u0053\u0065\u0072\u0062\u0069\u0061\u006E\u0020\u0028\u004C\u0061\u0074\u0069\u006E\u0029\u0020\u0028\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", "\u0073\u0072\u0070\u0073\u006B\u0069\u0020\u0028\u0042\u006F\u0073\u006E\u0061\u0020\u0069\u0020\u0048\u0065\u0072\u0063\u0065\u0067\u006F\u0076\u0069\u006E\u0061\u0029", false, 0x0019, "BA", "BA", "\u0042\u006F\u0073\u006E\u0069\u0061\u0020\u0061\u006E\u0064\u0020\u0048\u0065\u0072\u007A\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0042\u006F\u0073\u006E\u0061\u0020\u0069\u0020\u0048\u0065\u0072\u0063\u0065\u0067\u006F\u0076\u0069\u006E\u0061", "\u0042\u0041\u004D", "\u0043\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u006C\u0065\u0020\u004D\u0061\u0072\u006B\u0073", "\u006B\u006F\u006E\u0076\u0065\u0072\u0074\u0069\u0062\u0069\u006C\u006E\u0061\u0020\u006D\u0061\u0072\u006B\u0061", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u004B\u004D", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u006A\u0061\u006E\u0073\u006B\u0069\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 0, 0, "\u002E", "\u003A", "", "", "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u002E\u0020\u004D\u002E\u0020\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u002E\u0020\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u002E\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u006E\u0065\u0064", "\u0070\u006F\u006E", "\u0075\u0074\u006F", "\u0073\u0072\u0065", "\u010D\u0065\u0074", "\u0070\u0065\u0074", "\u0073\u0075\u0062" ], [ "\u006E\u0065\u0064\u0065\u006C\u006A\u0061", "\u0070\u006F\u006E\u0065\u0064\u0065\u006C\u006A\u0061\u006B", "\u0075\u0074\u006F\u0072\u0061\u006B", "\u0073\u0072\u0065\u0064\u0061", "\u010D\u0065\u0074\u0076\u0072\u0074\u0061\u006B", "\u0070\u0065\u0074\u0061\u006B", "\u0073\u0075\u0062\u006F\u0074\u0061" ], [ "\u006A\u0061\u006E", "\u0066\u0065\u0062", "\u006D\u0061\u0072", "\u0061\u0070\u0072", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067", "\u0073\u0065\u0070", "\u006F\u006B\u0074", "\u006E\u006F\u0076", "\u0064\u0065\u0063", "" ], [ "\u006A\u0061\u006E\u0075\u0061\u0072", "\u0066\u0065\u0062\u0072\u0075\u0061\u0072", "\u006D\u0061\u0072\u0074", "\u0061\u0070\u0072\u0069\u006C", "\u006D\u0061\u006A", "\u006A\u0075\u006E", "\u006A\u0075\u006C", "\u0061\u0076\u0067\u0075\u0073\u0074", "\u0073\u0065\u0070\u0074\u0065\u006D\u0062\u0061\u0072", "\u006F\u006B\u0074\u006F\u0062\u0061\u0072", "\u006E\u006F\u0076\u0065\u006D\u0062\u0061\u0072", "\u0064\u0065\u0063\u0065\u006D\u0062\u0061\u0072", "" ] },
{ 0x143B, 0x007F, "smj-SE", "smj", "smj", "smj-SE", "\u0053\u0061\u006D\u0069\u0020\u0028\u004C\u0075\u006C\u0065\u0029\u0020\u0028\u0053\u0077\u0065\u0064\u0065\u006E\u0029", "\u006A\u0075\u006C\u0065\u0076\u0075\u0073\u00E1\u006D\u0065\u0067\u0069\u0065\u006C\u006C\u0061\u0020\u0028\u0053\u0076\u0069\u0065\u0072\u0069\u006B\u0029", false, 0x00DD, "SE", "SE", "\u0053\u0077\u0065\u0064\u0065\u006E", "\u0053\u0076\u0069\u0065\u0072\u0069\u006B", "\u0053\u0045\u004B", "\u0053\u0077\u0065\u0064\u0069\u0073\u0068\u0020\u004B\u0072\u006F\u006E\u0061", "\u006B\u0072\u00E5\u0076\u006E\u006E\u00E5", true, 2, 1, 2, 8, 3, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u002E", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 2, "\u002D", "\u003A", "", "", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u00E1\u006A\u006C", "\u006D\u00E1\u006E", "\u0064\u0069\u0073", "\u0067\u0061\u0073", "\u0064\u0075\u006F\u0072", "\u0062\u0069\u0065\u0072", "\u006C\u00E1\u0076" ], [ "\u00E1\u006A\u006C\u006C\u0065\u006B", "\u006D\u00E1\u006E\u006E\u006F\u0064\u0061\u0068\u006B\u0061", "\u0064\u0069\u006A\u0073\u0074\u0061\u0068\u006B\u0061", "\u0067\u0061\u0073\u0073\u006B\u0061\u0076\u0061\u0068\u006B\u006B\u006F", "\u0064\u0075\u006F\u0072\u0061\u0073\u0074\u0061\u0068\u006B\u0061", "\u0062\u0069\u0065\u0072\u006A\u006A\u0065\u0064\u0061\u0068\u006B\u0061", "\u006C\u00E1\u0076\u0076\u006F\u0064\u0061\u0068\u006B\u0061" ], [ "\u00E5\u0064\u00E5\u006A", "\u0067\u0075\u006F\u0076", "\u0073\u006E\u006A\u0075", "\u0076\u0075\u006F\u0072", "\u006D\u006F\u0061\u0072", "\u0062\u0069\u0065\u0068", "\u0073\u006E\u006A\u0069", "\u0062\u00E5\u0072\u0067", "\u0072\u0061\u0067\u00E1", "\u0067\u00E5\u006C\u0067", "\u0062\u0061\u0073\u00E1", "\u006A\u0061\u0076\u006C", "" ], [ "\u00E5\u0064\u00E5\u006A\u0061\u006B\u006D\u00E1\u006E\u006E\u006F", "\u0067\u0075\u006F\u0076\u0076\u0061\u006D\u00E1\u006E\u006E\u006F", "\u0073\u006A\u006E\u006A\u0075\u006B\u0074\u006A\u0061\u006D\u00E1\u006E\u006E\u006F", "\u0076\u0075\u006F\u0072\u0061\u0074\u006A\u0069\u0073\u006D\u00E1\u006E\u006E\u006F", "\u006D\u006F\u0061\u0072\u006D\u0065\u0073\u006D\u00E1\u006E\u006E\u006F", "\u0062\u0069\u0065\u0068\u0074\u0073\u0065\u006D\u00E1\u006E\u006E\u006F", "\u0073\u006A\u006E\u006A\u0069\u006C\u006C\u0074\u006A\u0061\u006D\u00E1\u006E\u006E\u006F", "\u0062\u00E5\u0072\u0067\u0067\u0065\u006D\u00E1\u006E\u006E\u006F", "\u0072\u0061\u0067\u00E1\u0074\u006D\u00E1\u006E\u006E\u006F", "\u0067\u00E5\u006C\u0067\u00E5\u0064\u0069\u0073\u006D\u00E1\u006E\u006E\u006F", "\u0062\u0061\u0073\u00E1\u0064\u0069\u0073\u006D\u00E1\u006E\u006E\u006F", "\u006A\u0061\u0076\u006C\u006C\u0061\u006D\u00E1\u006E\u006E\u006F", "" ] },
{ 0x046C, 0x007F, "ns-ZA", "ns", "ns", "ns-ZA", "\u004E\u006F\u0072\u0074\u0068\u0065\u0072\u006E\u0020\u0053\u006F\u0074\u0068\u006F\u0020\u0028\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u0029", "\u0053\u0065\u0073\u006F\u0074\u0068\u006F\u0020\u0073\u0061\u0020\u004C\u0065\u0062\u006F\u0061\u0020\u0028\u0041\u0066\u0072\u0069\u006B\u0061\u0020\u0042\u006F\u0072\u0077\u0061\u0029", false, 0x00D1, "ZA", "ZA", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061", "\u0041\u0066\u0072\u0069\u006B\u0061\u0020\u0042\u006F\u0072\u0077\u0061", "\u005A\u0041\u0052", "\u0053\u006F\u0075\u0074\u0068\u0020\u0041\u0066\u0072\u0069\u0063\u0061\u006E\u0020\u0052\u0061\u006E\u0064", "\u0052\u0061\u006E\u0064", true, 2, 1, 2, 2, 2, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u0052", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0047\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0020\u0043\u0061\u006C\u0065\u006E\u0064\u0061\u0072", 6, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0079\u0079\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u002F\u004D\u004D\u002F\u0064\u0064", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0020\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0053\u0075\u006E", "\u004D\u006F\u006E", "\u0054\u0075\u0065", "\u0057\u0065\u0064", "\u0054\u0068\u0075", "\u0046\u0072\u0069", "\u0053\u0061\u0074" ], [ "\u004C\u0061\u006D\u006F\u0072\u0065\u006E\u0061", "\u004D\u006F\u0161\u0075\u0070\u006F\u006C\u006F\u0067\u006F", "\u004C\u0061\u0062\u006F\u0062\u0065\u0064\u0069", "\u004C\u0061\u0062\u006F\u0072\u0061\u0072\u006F", "\u004C\u0061\u0062\u006F\u006E\u0065", "\u004C\u0061\u0062\u006F\u0068\u006C\u0061\u006E\u006F", "\u004D\u006F\u006B\u0069\u0062\u0065\u006C\u006F" ], [ "\u004A\u0061\u006E", "\u0046\u0065\u0062", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0061\u0079", "\u004A\u0075\u006E", "\u004A\u0075\u006C", "\u0041\u0075\u0067", "\u0053\u0065\u0070", "\u004F\u0063\u0074", "\u004E\u006F\u0076", "\u0044\u0065\u0063", "" ], [ "\u0050\u0068\u0065\u0072\u0065\u006B\u0067\u006F\u006E\u0067", "\u0048\u006C\u0061\u006B\u006F\u006C\u0061", "\u004D\u006F\u0070\u0069\u0074\u006C\u006F", "\u004D\u006F\u0072\u0061\u006E\u0061\u006E\u0067", "\u004D\u006F\u0073\u0065\u0067\u0061\u006D\u0061\u006E\u0079\u0065", "\u004E\u0067\u006F\u0061\u0074\u006F\u0062\u006F\u0161\u0065\u0067\u006F", "\u0050\u0068\u0075\u0070\u0068\u0075", "\u0050\u0068\u0061\u0074\u006F", "\u004C\u0065\u0077\u0065\u0064\u0069", "\u0044\u0069\u0070\u0068\u0061\u006C\u0061\u006E\u0061", "\u0044\u0069\u0062\u0061\u0074\u0073\u0065\u006C\u0061", "\u004D\u0061\u006E\u0074\u0068\u006F\u006C\u0065", "" ] },
{ 0x046B, 0x007F, "quz-BO", "quz", "quz", "quz-BO", "\u0051\u0075\u0065\u0063\u0068\u0075\u0061\u0020\u0028\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u0029", "\u0072\u0075\u006E\u0061\u0073\u0069\u006D\u0069\u0020\u0028\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u0020\u0053\u0075\u0079\u0075\u0029", false, 0x001A, "BO", "BO", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u0020\u0053\u0075\u0079\u0075", "\u0042\u004F\u0042", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u006E\u006F", "\u0042\u006F\u006C\u0069\u0076\u0069\u0061\u006E\u006F", true, 2, 1, 2, 14, 2, [ 3 ], [ 3 ], "\u002C", "\u002E", "\u002C", "\u002E", "\u0024\u0062", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0063\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u0069\u006F\u0020\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u006F", 6, 0, "\u002F", "\u003A", "\u0061\u002E\u006D\u002E", "\u0070\u002E\u006D\u002E", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079", "\u0064\u0064\u002D\u004D\u004D\u002D\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\u0064\u0064\u0064\u0020\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0064\u0065\u0020\'\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\'\u0020\u0064\u0065\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u0069\u006E\u0074", "\u006B\u0069\u006C", "\u0061\u0074\u0069", "\u0071\u0075\u0079", "\u0043\u0068\u0092", "\u0049\u006C\u006C", "\u006B\u0027\u0075" ], [ "\u0069\u006E\u0074\u0069\u0063\u0068\u0061\u0077", "\u006B\u0069\u006C\u006C\u0061\u0063\u0068\u0061\u0077", "\u0061\u0074\u0069\u0070\u0061\u0063\u0068\u0061\u0077", "\u0071\u0075\u0079\u006C\u006C\u0075\u0072\u0063\u0068\u0061\u0077", "\u0043\u0068\u0027\u0020\u0061\u0073\u006B\u0061\u0063\u0068\u0061\u0077", "\u0049\u006C\u006C\u0061\u0070\u0061\u0063\u0068\u0061\u0077", "\u006B\u0027\u0075\u0079\u0063\u0068\u0069\u0063\u0068\u0061\u0077" ], [ "\u0051\u0075\u006C", "\u0048\u0061\u0074", "\u0050\u0061\u0075", "\u0061\u0079\u0072", "\u0041\u0079\u006D", "\u0049\u006E\u0074", "\u0041\u006E\u0074", "\u0051\u0068\u0061", "\u0055\u006D\u0061", "\u004B\u0061\u006E", "\u0041\u0079\u0061", "\u004B\u0061\u0070", "" ], [ "\u0051\u0075\u006C\u006C\u0061\u0020\u0070\u0075\u0071\u0075\u0079", "\u0048\u0061\u0074\u0075\u006E\u0020\u0070\u0075\u0071\u0075\u0079", "\u0050\u0061\u0075\u0071\u0061\u0072\u0020\u0077\u0061\u0072\u0061\u0079", "\u0061\u0079\u0072\u0069\u0077\u0061", "\u0041\u0079\u006D\u0075\u0072\u0061\u0079", "\u0049\u006E\u0074\u0069\u0020\u0072\u0061\u0079\u006D\u0069", "\u0041\u006E\u0074\u0061\u0020\u0053\u0069\u0074\u0077\u0061", "\u0051\u0068\u0061\u0070\u0061\u0071\u0020\u0053\u0069\u0074\u0077\u0061", "\u0055\u006D\u0061\u0020\u0072\u0061\u0079\u006D\u0069", "\u004B\u0061\u006E\u0074\u0061\u0072\u0061\u0079", "\u0041\u0079\u0061\u006D\u0061\u0072\u0071\u0027\u0061", "\u004B\u0061\u0070\u0061\u0071\u0020\u0052\u0061\u0079\u006D\u0069", "" ] },
{ 0x043B, 0x007F, "se-NO", "se", "se", "se-NO", "\u0053\u0061\u006D\u0069\u0020\u0028\u004E\u006F\u0072\u0074\u0068\u0065\u0072\u006E\u0029\u0020\u0028\u004E\u006F\u0072\u0077\u0061\u0079\u0029", "\u0064\u0061\u0076\u0076\u0069\u0073\u00E1\u006D\u0065\u0067\u0069\u0065\u006C\u006C\u0061\u0020\u0028\u004E\u006F\u0072\u0067\u0061\u0029", false, 0x00B1, "NO", "NO", "\u004E\u006F\u0072\u0077\u0061\u0079", "\u004E\u006F\u0072\u0067\u0061", "\u004E\u004F\u004B", "\u004E\u006F\u0072\u0077\u0065\u0067\u0069\u0061\u006E\u0020\u004B\u0072\u006F\u006E\u0065", "\u006B\u0072\u0075\u0076\u0064\u006E\u006F", true, 2, 1, 2, 12, 2, [ 3 ], [ 3 ], "\u002C", "\u00A0", "\u002C", "\u00A0", "\u006B\u0072", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u0067\u0072\u0065\u0067\u006F\u0072\u0069\u0061\u006E\u0073\u006B\u0020\u006B\u0061\u006C\u0065\u006E\u0064\u0065\u0072", 0, 2, "\u002E", "\u003A", "", "", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079\u0079\u0079", "\u0064\u0064\u002E\u004D\u004D\u002E\u0079\u0079", "\u0064\u002E\u004D\u002E\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0048\u0048\u002E\u006D\u006D", "\u0048\u0048\u002E\u006D\u006D\u002E\u0073\u0073" ], [ "\u004D\u004D\u004D\u004D\u0020\u0064\'\u002E\u0020\u0062\u002E\u0020\'\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0073\u006F\u0074\u006E", "\u0076\u0075\u006F\u0073", "\u006D\u0061\u014B", "\u0067\u0061\u0073\u006B", "\u0064\u0075\u006F\u0072", "\u0062\u0065\u0061\u0072", "\u006C\u00E1\u0076" ], [ "\u0073\u006F\u0074\u006E\u0061\u0062\u0065\u0061\u0069\u0076\u0069", "\u0076\u0075\u006F\u0073\u0073\u00E1\u0072\u0067\u0061", "\u006D\u0061\u014B\u014B\u0065\u0062\u00E1\u0072\u0067\u0061", "\u0067\u0061\u0073\u006B\u0061\u0076\u0061\u0068\u006B\u006B\u0075", "\u0064\u0075\u006F\u0072\u0061\u0073\u0074\u0061\u0074", "\u0062\u0065\u0061\u0072\u006A\u0061\u0064\u0061\u0074", "\u006C\u00E1\u0076\u0076\u0061\u0072\u0064\u0061\u0074" ], [ "\u006F\u0111\u0111\u006A", "\u0067\u0075\u006F\u0076", "\u006E\u006A\u0075\u006B", "\u0063\u0075\u006F", "\u006D\u0069\u0065\u0073", "\u0067\u0065\u0061\u0073", "\u0073\u0075\u006F\u0069", "\u0062\u006F\u0072\u0067", "\u010D\u0061\u006B\u010D", "\u0067\u006F\u006C\u0067", "\u0073\u006B\u00E1\u0062", "\u006A\u0075\u006F\u0076", "" ], [ "\u006F\u0111\u0111\u0061\u006A\u0061\u0067\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0067\u0075\u006F\u0076\u0076\u0061\u006D\u00E1\u006E\u006E\u0075", "\u006E\u006A\u0075\u006B\u010D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u0063\u0075\u006F\u014B\u006F\u006D\u00E1\u006E\u006E\u0075", "\u006D\u0069\u0065\u0073\u0073\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0067\u0065\u0061\u0073\u0073\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0073\u0075\u006F\u0069\u0064\u006E\u0065\u006D\u00E1\u006E\u006E\u0075", "\u0062\u006F\u0072\u0067\u0065\u006D\u00E1\u006E\u006E\u0075", "\u010D\u0061\u006B\u010D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u0067\u006F\u006C\u0067\u0067\u006F\u0074\u006D\u00E1\u006E\u006E\u0075", "\u0073\u006B\u00E1\u0062\u006D\u0061\u006D\u00E1\u006E\u006E\u0075", "\u006A\u0075\u006F\u0076\u006C\u0061\u006D\u00E1\u006E\u006E\u0075", "" ] },
{ 0x043A, 0x007F, "mt-MT", "mt", "mt", "mt-MT", "\u004D\u0061\u006C\u0074\u0065\u0073\u0065\u0020\u0028\u004D\u0061\u006C\u0074\u0061\u0029", "\u004D\u0061\u006C\u0074\u0069\u0020\u0028\u004D\u0061\u006C\u0074\u0061\u0029", false, 0x00A3, "MT", "MT", "\u004D\u0061\u006C\u0074\u0061", "\u004D\u0061\u006C\u0074\u0061", "\u004D\u0054\u004C", "\u004D\u0061\u006C\u0074\u0065\u0073\u0065\u0020\u004C\u0069\u0072\u0061", "\u004C\u0069\u0072\u0061\u0020\u004D\u0061\u006C\u0074\u0069\u006A\u0061", true, 2, 1, 2, 1, 0, [ 3 ], [ 3 ], "\u002E", "\u002C", "\u002E", "\u002C", "\u004C\u006D", "\u002D", "\u002B", "\u004E\u0061\u004E", "\u002D\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", "\u0049\u006E\u0066\u0069\u006E\u0069\u0074\u0079", [ "\u0030", "\u0031", "\u0032", "\u0033", "\u0034", "\u0035", "\u0036", "\u0037", "\u0038", "\u0039" ], 1, [ 1 ], "\u004B\u0061\u006C\u0065\u006E\u0064\u0061\u0072\u006A\u0075", 0, 0, "\u002F", "\u003A", "\u0041\u004D", "\u0050\u004D", "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u0074\u0061\u005C\'\u0020\'\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u004D\u004D\u004D\u004D\u0020\u0064\u0064", "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u002F\u004D\u004D\u002F\u0079\u0079\u0079\u0079", "\u0079\u0079\u0079\u0079\u002D\u004D\u004D\u002D\u0064\u0064", "\u0064\u0064\u004D\u004D\u0079\u0079\u0079\u0079", "\u0064\u002D\u004D\u002D\u0079\u0079\u0079\u0079", "\u0064\u002F\u004D\u002F\u0079\u0079\u0079\u0079" ], [ "\u0048\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074", "\u0048\u003A\u006D\u006D\u003A\u0073\u0073", "\u0068\u0068\u003A\u006D\u006D\u003A\u0073\u0073\u0020\u0074\u0074" ], [ "\u0064\u0064\u0064\u0064\u002C\u0020\u0064\'\u0020\u0074\u0061\u005C\'\u0020\'\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\'\u0020\u0074\u0061\u005C\'\u0020\'\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079", "\u0064\u0064\u0020\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u004D\u004D\u004D\u004D\u0020\u0079\u0079\u0079\u0079" ], [ "\u0126\u0061\u0064", "\u0054\u006E\u0065", "\u0054\u006C\u0069", "\u0045\u0072\u0062", "\u0126\u0061\u006D", "\u0120\u0069\u006D", "\u0053\u0069\u0062" ], [ "\u0049\u006C\u002D\u0126\u0061\u0064\u0064", "\u0049\u0074\u002D\u0054\u006E\u0065\u006A\u006E", "\u0049\u0074\u002D\u0054\u006C\u0069\u0065\u0074\u0061", "\u004C\u002D\u0045\u0072\u0062\u0067\u0127\u0061", "\u0049\u006C\u002D\u0126\u0061\u006D\u0069\u0073", "\u0049\u006C\u002D\u0120\u0069\u006D\u0067\u0127\u0061", "\u0049\u0073\u002D\u0053\u0069\u0062\u0074" ], [ "\u004A\u0061\u006E", "\u0046\u0072\u0061", "\u004D\u0061\u0072", "\u0041\u0070\u0072", "\u004D\u0065\u006A", "\u0120\u0075\u006E", "\u004C\u0075\u006C", "\u0041\u0077\u0069", "\u0053\u0065\u0074", "\u004F\u0074\u0074", "\u004E\u006F\u0076", "\u0044\u0069\u010B", "" ], [ "\u004A\u0061\u006E\u006E\u0061\u0072", "\u0046\u0072\u0061\u0072", "\u004D\u0061\u0072\u007A\u0075", "\u0041\u0070\u0072\u0069\u006C", "\u004D\u0065\u006A\u006A\u0075", "\u0120\u0075\u006E\u006A\u0075", "\u004C\u0075\u006C\u006A\u0075", "\u0041\u0077\u0069\u0073\u0073\u0075", "\u0053\u0065\u0074\u0074\u0065\u006D\u0062\u0072\u0075", "\u004F\u0074\u0074\u0075\u0062\u0072\u0075", "\u004E\u006F\u0076\u0065\u006D\u0062\u0072\u0075", "\u0044\u0069\u010B\u0065\u006D\u0062\u0072\u0075", "" ] },
  ];

  private static CultureData*[char[]] nameTable;
  private static CultureData*[int] idTable;
  private static CultureData*[char[]] regionNameTable;

  static ~this() {
    nameTable = null;
    idTable = null;
    regionNameTable = null;
  }

  package static CultureData* getDataFromCultureName(char[] name) {
    if (CultureData** data = name in nameTable)
      return *data;

    char[] actualName;
    for (int i = 0; i < cultureDataTable.length; i++) {
      if (compareString(name, cultureDataTable[i].name) == 0) {
        nameTable[actualName = cultureDataTable[i].name] = &cultureDataTable[i];
        return nameTable[actualName];
      }
    }

    error ("Culture name '" ~ name ~ "' is not supported.");
    return null;
  }

  package static CultureData* getDataFromCultureID(int lcid) {
    if (CultureData** data = lcid in idTable)
      return *data;

    for (int i = 0; i < cultureDataTable.length; i++) {
      if (cultureDataTable[i].lcid == lcid) {
        idTable[lcid] = &cultureDataTable[i];
        return idTable[lcid];
      }
    }

version (Posix) {
    error ("Culture not found - if this was not tried set by the application, Tango "
            ~ "will expect that a locale is set via environment variables LANG or LC_ALL.");
}
else {
    error ("Culture not found - if this was not tried set by the application, Tango "
            ~ "will expect that a locale is set for the system .");
}
    return null;
  }

  package static CultureData* getDataFromRegionName(ref char[] name) {
    CultureData** data;
    if ((data = name in regionNameTable) !is null)
      return *data;

    for (int i = 0; i < cultureDataTable.length; i++) {
      if (compareString(name, cultureDataTable[i].regionName) == 0) {
        regionNameTable[name = cultureDataTable[i].regionName] = &cultureDataTable[i];
        return regionNameTable[name];
      }
    }

    // We also accept a culture name.
    if ((data = name in nameTable) !is null)
      return *data;

    for (int i = 0; i < cultureDataTable.length; i++) {
      if (compareString(name, cultureDataTable[i].name) == 0) {
        nameTable[name = cultureDataTable[i].name] = &cultureDataTable[i];
        return nameTable[name];
      }
    }

    error ("Region name '" ~ name ~ "' is not supported.");
    return null;
  }

  package static char[] getCultureNameFromIetfName(char[] name) {
    for (int i = 0; i < cultureDataTable.length; i++) {
      if (compareString(name, cultureDataTable[i].ietfTag) == 0)
        return cultureDataTable[i].name;
    }
    return null;
  }

}
