#include "osl/effect_util/additionalEffect.h"
#include "osl/record/csaString.h"
#include "osl/record/csaRecord.h"
#include "osl/record/record.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <fstream>
#include <iostream>
using namespace osl;
using namespace osl::effect_util;

class AdditionalEffectTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(AdditionalEffectTest);
  CPPUNIT_TEST(testHasEffect);
  CPPUNIT_TEST(testZero);
  CPPUNIT_TEST(testOne);
  CPPUNIT_TEST(testPile);
  CPPUNIT_TEST(testFiles);
  CPPUNIT_TEST(testCount2);
  CPPUNIT_TEST_SUITE_END();
public:
  void testHasEffect();
  void testZero();
  void testOne();
  void testPile();
  void testFile(const std::string&);
  void testFiles();
  void testCount2();
};

CPPUNIT_TEST_SUITE_REGISTRATION(AdditionalEffectTest);

void AdditionalEffectTest::testHasEffect()
{
  NumEffectState state(CsaString(
			 "P1+UM *  *  *  *  * -OU * -KY\n"
			 "P2 *  * -HI-KI * -GI-KI *  * \n"
			 "P3 *  *  * -FU-GI-FU *  * -KE\n"
			 "P4-FU *  *  * -FU * -FU-FU-FU\n"
			 "P5 * +KE * +FU * +FU *  *  * \n"
			 "P6+FU * -FU+KI+FU * +GI * +FU\n"
			 "P7 * +FU *  *  *  * +KE *  * \n"
			 "P8 * +OU+KI *  *  *  *  * +HI\n"
			 "P9+KY *  *  *  *  *  *  * +KY\n"
			 "P+00KA00KE00KY00FU\n"
			 "P-00GI00FU00FU00FU\n"
			 "-\n").getInitialState());
  CPPUNIT_ASSERT_EQUAL(true, AdditionalEffect::hasEffect
		       (state, Position(7,7), WHITE));
  CPPUNIT_ASSERT_EQUAL(false, AdditionalEffect::hasEffect
		       (state, Position(7,7), BLACK));

  CPPUNIT_ASSERT_EQUAL(false, AdditionalEffect::hasEffect
		       (state, Position(8,8), WHITE));
  CPPUNIT_ASSERT_EQUAL(true, AdditionalEffect::hasEffect
		       (state, Position(8,8), BLACK));
}

void AdditionalEffectTest::testFile(const std::string& filename)
{
  const Record record = CsaFile(filename).getRecord();
  NumEffectState state(record.getInitialState());
  const vector<Move> moves = record.getMoves();
  for (unsigned int i=0; i<moves.size(); ++i)
  {
    const Player turn = state.getTurn();
    const Position target = state.getKingPosition(turn);

    // 8近傍の利きの数をカウント
    int kagekiki = 0;
    for (int x=-1; x<2; x++)
    {
      for (int y=-1; y<2; y++){
	Position pos = Position(target.x()+x, target.y()+y);
	if (pos.isOnBoard())
	  kagekiki += AdditionalEffect::count(state, turn, pos);
      }
    }

    for (int x=1; x<=9; x++) {
      for (int y=1; y<=9; y++) {
	const Position p(x,y);
	CPPUNIT_ASSERT_EQUAL(AdditionalEffect::hasEffect(state, p, BLACK),
			     AdditionalEffect::hasEffectStable(state, p, BLACK));
	CPPUNIT_ASSERT_EQUAL(AdditionalEffect::hasEffect(state, p, WHITE),
			     AdditionalEffect::hasEffectStable(state, p, WHITE));
      }
    }
    ApplyMoveOfTurn::doMove(state, moves[i]);
  }
}

void AdditionalEffectTest::testFiles()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  const int count = isShortTest ? 10 : 100;
  std::string filename;
  while ((ifs >> filename) && (++i<count)) {
    if (filename == "") 
      break;
    if (! isShortTest)
      std::cerr << filename << " ";
    testFile(OslConfig::testCsaFile(filename));
  }
}

void AdditionalEffectTest::testZero()
{
  NumEffectState state(CsaString(
			 "P1-KY-KE-GI * +OU * -GI-KE * \n"
			 "P2 * -HI-KI *  *  * -KI-KA * \n"
			 "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			 "P4 *  *  *  *  *  *  *  *  * \n"
			 "P5 *  *  *  * -KY *  *  *  * \n"
			 "P6 *  *  *  *  *  *  *  *  * \n"
			 "P7+FU+FU+FU+FU+FU+FU+FU+FU+FU\n"
			 "P8 * +KA+KI *  *  * +KI+HI * \n"
			 "P9+KY+KE+GI * -OU * +GI+KE+KY\n"
			 "+\n").getInitialState());
  CPPUNIT_ASSERT_EQUAL(0, AdditionalEffect::count(state, WHITE, 
						Position(5,2)));
  CPPUNIT_ASSERT_EQUAL(0, AdditionalEffect::count(state, BLACK, 
						Position(5,2)));
  CPPUNIT_ASSERT_EQUAL(0, AdditionalEffect::count(state, BLACK, 
						Position(6,6)));
  CPPUNIT_ASSERT_EQUAL(0, AdditionalEffect::count(state, BLACK, 
						Position(7,6)));
  CPPUNIT_ASSERT_EQUAL(0, AdditionalEffect::count(state, BLACK,
						Position(8,4)));
}

void AdditionalEffectTest::testOne()
{
  NumEffectState state(CsaString(
			 "P1-KY-KE-GI * +OU * -GI-KE * \n"
			 "P2 * -HI-KI *  *  * -KI-KA * \n"
			 "P3-FU-FU-FU-FU-FU-FU-FU-FU-FU\n"
			 "P4 *  *  *  *  *  *  *  *  * \n"
			 "P5 *  *  *  * -KY *  *  *  * \n"
			 "P6 *  * +FU *  *  *  *  *  * \n"
			 "P7+FU+FU+KI+FU+FU+FU+FU+FU+FU\n"
			 "P8 * +KA *  *  *  * +KI+HI * \n"
			 "P9+KY+KE+GI * -OU * +GI+KE+KY\n"
			 "+\n").getInitialState());
  CPPUNIT_ASSERT_EQUAL(1, AdditionalEffect::count(state, BLACK, 
						Position(6,6)));
  CPPUNIT_ASSERT_EQUAL(1, AdditionalEffect::count(state, WHITE, 
						Position(6,2)));
}

void AdditionalEffectTest::testPile()
{
  NumEffectState state(CsaString(
			 "P1-KY-KE-GI * +OU * -GI-KE * \n"
			 "P2 *  * -KI *  *  * -KI * -KY\n"
			 "P3-FU-FU-FU-FU-FU-FU-FU-FU * \n"
			 "P4 *  *  *  *  * +GI *  * -FU\n"
			 "P5 *  *  *  *  *  *  *  *  * \n"
			 "P6 *  * +FU+KA *  *  *  * +FU\n"
			 "P7+FU+FU * +FU+FU+FU+FU+FU+HI\n"
			 "P8 * +KA *  *  *  * +KI * +KY\n"
			 "P9+KY+KE * +KI-OU * +GI+KE-HI\n"
			 "+\n").getInitialState());
  // KYO, HI
  CPPUNIT_ASSERT_EQUAL(2, AdditionalEffect::count(state, BLACK, 
						Position(1,5)));
  CPPUNIT_ASSERT_EQUAL(2, AdditionalEffect::count(state, WHITE, 
						Position(1,5)));
  CPPUNIT_ASSERT_EQUAL(2, AdditionalEffect::count(state, BLACK, 
						Position(3,3)));
}

void AdditionalEffectTest::testCount2()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY *  *  *  *  *  *  * -KY\n"
			   "P2 * -HI *  *  * -GI-KI-OU * \n"
			   "P3 *  * -KE * -KA-KI-KE-FU+FU\n"
			   "P4 * -GI-FU-FU-FU-FU *  *  * \n"
			   "P5-FU-FU *  *  *  * -FU+FU+HI\n"
			   "P6 *  * +FU+FU+FU+FU *  *  * \n"
			   "P7+FU+FU+GI+KI+GI * +KE *  * \n"
			   "P8 * +OU+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  * +KY\n"
			   "P+00FU\n"
			   "P-00KA00FU\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(2, AdditionalEffect::count2(state, Position(1,2),
						     BLACK));
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE * -KI *  *  *  * +RY\n"
			   "P2 * -OU-GI-KI *  *  *  *  * \n"
			   "P3 *  * -FU-FU * -FU * -FU * \n"
			   "P4-FU+FU *  * -FU-GI *  * +FU\n"
			   "P5 *  *  *  *  *  * -FU+FU * \n"
			   "P6+FU *  *  * +FU+FU *  *  * \n"
			   "P7 * +GI+KE+FU+KA * +FU *  * \n"
			   "P8 * +KY+OU+GI *  *  *  *  * \n"
			   "P9+KY *  *  * +KI *  *  * -RY\n"
			   "P+00KE\n"
			   "P-00KA00KI00KE00KY00FU00FU00FU\n"
			   "+\n").getInitialState());
    CPPUNIT_ASSERT_EQUAL(0, AdditionalEffect::count2(state, Position(8,3),
						     BLACK));
  }
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
