/* kakinoki.cc
 */
#include "osl/record/kakinoki.h"
#include "osl/record/kanjiMove.h"
#include "osl/record/kanjiCode.h"
#include "osl/misc/sjis2euc.h"
#include "osl/state/simpleState.h"
#include "osl/apply_move/applyMove.h"
#include <iostream>
#include <fstream>
#include <stdexcept>
#include <cassert>
#include <string>
#include <sstream>

namespace osl
{
  namespace record
  {
    void kakinokiParseLine(boost::shared_ptr<RecordVisitor>& rv,
			   std::string s)
    {
      Record *record=rv->getRecord();
      SimpleState* state=rv->getState();
      if (s[0] == '*') 
      {
	MoveRecord *mr = rv->getLastMove();
	if (mr)
	  mr->addComment(s.substr(1));
	return;
      }
      if (s.size() > 6)
      {
	if (equal(K_BLACK.begin(), K_BLACK.end(), s.begin())) {
	  record->setPlayer(BLACK, s.substr(6));
	  return;
	}
	if (equal(K_WHITE.begin(), K_WHITE.end(), s.begin())) {
	  record->setPlayer(WHITE, s.substr(6));
	  return;
	}
        if (s.substr(0,6) == (K_KISEN+K_COLON))
        {
          record->setTounamentName(s.substr(6));
          return;
        }
      }
      if (s[0] != ' ') 
      {
	if (rv->getLastMove() == 0)
	  record->addInitialComment(s);
	return;			// otherwise ignore
      }
      if (s.find(K_TORYO) != s.npos)
      {
	record->setResult((state->getTurn() == BLACK)
			  ? Record::WHITE_WIN : Record::BLACK_WIN);
	return;
      }

      {
	// replace '(' and ')' with white space if exists
	size_t p = s.find('(');
	if (p != s.npos)
	  s.replace(p, 1, 1, ' ');
	p = s.find(')');
	if (p != s.npos)
	  s.replace(p, 1, 1, ' ');
      }      
      Move last_move;
      if (rv->getLastMove())
	last_move = rv->getLastMove()->getMove();
      const Move m = kakinoki::strToMove(s, *state, last_move);
      if (m.isNormal()) {
	if (! state->isValidMove(m)) {
	  std::ostringstream ss;
	  ss << *state << s << "\n" << m;
	  std::cerr << ss.str();
	  throw KakinokiIOError(ss.str());
	}	
	rv->addMoveAndAdvance(m);
      }
    }
  }
}

osl::Move osl::record::
kakinoki::strToMove(const std::string& s, const SimpleState& state, Move last_move)
{
  static KanjiMove Kanji_Move;
  std::istringstream is(s);
  int move_number, from_number;
  std::string move_string;
  is >> move_number >> move_string;

  Position to, from;
  if (move_string.substr(0,2) == K_ONAZI)
    to = last_move.to();
  else
    to = Kanji_Move.toPosition(move_string.substr(0,4));
  if (to == Position())		// resign?
    return Move();
  
  Ptype ptype;
  size_t cur = 4;
  if (move_string.substr(cur,2) == K_NARU) // PLANCE, PKIGHT, PSILVER
  {
    assert(move_string.size() >= cur+4);
    ptype = Kanji_Move.toPtype(move_string.substr(cur,4));
    cur += 4;
  }
  else
  {
    ptype = Kanji_Move.toPtype(move_string.substr(cur,2));
    cur += 2;
  }
  if (move_string.size() >= cur+2 && move_string.substr(cur,2)
      == K_UTSU)
    from = Position();
  else 
  {
    if (! (is >> from_number))
      throw KakinokiIOError("error in move from");
    from = Position(from_number / 10, from_number % 10);
  }
  
  bool is_promote = false;
  if (move_string.size() >= cur+2 && move_string.substr(cur,2) == K_NARU)
    is_promote = true;

  if (from.isPieceStand())
    return Move(to, ptype, state.getTurn());
  Ptype captured = state.getPieceOnBoard(to).ptype();
  return Move(from, to, is_promote ? promote(ptype) : ptype,
	      captured, is_promote, state.getTurn());
}

osl::record::kakinoki::
InputStream::InputStream(std::istream& is) 
  : is(is), 
    rv(boost::shared_ptr<record::RecordVisitor>(new record::RecordVisitor()))
{
  if (! is)
  {
    std::cerr << "InputStream::InputStream cannot read \n";
    abort();
  }
}
  
osl::record::kakinoki::
InputStream::InputStream(std::istream& is, boost::shared_ptr<record::RecordVisitor> rv) 
  : is(is), rv(rv)
{
  if (! is)
  {
    std::cerr << "InputStream::InputStream cannot read \n";
    throw KakinokiIOError("file open failed");
  }
}
  
osl::record::kakinoki::
InputStream::~InputStream(){}
  
void osl::record::kakinoki::
InputStream::load(Record* rec)
{
  //  rec->init();
  state.init(HIRATE);
  rv->setState(&state);
  rv->setRecord(rec);
  std::string line;
  while (std::getline(is, line)) 
  {
    // quick hack for \r
    if ((! line.empty())
	&& (line[line.size()-1] == 13))
      line.erase(line.size()-1);
    if (line.length()==0) 
      continue;
    // to euc
    line = misc::sjis2euc(line);
    // skip variations
    if (line.find(K_HENKA) == 0)
      break;
    if (! line.empty() && line[0] == '#' 
	&& line.find("separator") != line.npos)
      break;			// tanase shogi
    
    kakinokiParseLine(rv, line);
  }
  assert(state.isConsistent());
}

osl::record::kakinoki::
KakinokiFile::KakinokiFile(const std::string& filename)
{
  std::ifstream ifs(filename.c_str());
  if (! ifs)
  {
    const std::string msg = "KakinokiFile::KakinokiFile file cannot read ";
    std::cerr << msg << filename << "\n";
    throw KakinokiIOError(msg + filename);
  }
  InputStream irs(ifs);
  irs.load(&rec);
}

osl::record::kakinoki::
KakinokiFile::~KakinokiFile()
{
}

const osl::record::Record& osl::record::kakinoki::
KakinokiFile::getRecord() const
{
  return rec;
}

const osl::NumEffectState osl::record::kakinoki::
KakinokiFile::getInitialState() const
{
  return NumEffectState(rec.getInitialState());
}

bool osl::record::kakinoki::
KakinokiFile::isKakinokiFile(const std::string& filename)
{
  std::ifstream is(filename.c_str());
  std::string line;
  if (! is || ! getline(is, line)) 
    return false;
  line = misc::sjis2euc(line);
  return line.find("Kifu for Windows") != line.npos
    || line.find("KIFU") != line.npos
    || line.find(K_SENKEI) == 0
    || (line.find("#") == 0 && line.find(K_KIFU) != line.npos);
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
