/***************************************************************************
 $RCSfile: mediumddv-09.cpp,v $
                             -------------------
    cvs         : $Id: mediumddv-09.cpp,v 1.11 2004/01/15 17:55:39 aquamaniac Exp $
    begin       : Fri Nov 16 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef __declspec
# if BUILDING_DLL
#  define DLLIMPORT __declspec (dllexport)
# else /* Not BUILDING_DLL */
#  define DLLIMPORT __declspec (dllimport)
# endif /* Not BUILDING_DLL */
#else
# define DLLIMPORT
#endif



#include <openssl/bn.h>
#include <openssl/ripemd.h>


#include <list>
#include <openhbci/error.h>
#include <openhbci/pointer.h>
#include <openhbci/hbcistring.h>
#include <openhbci/date.h>
#include <openhbci/value.h>
#include <openhbci/balance.h>
#include <openhbci/auth.h>
#include <openhbci/interactor.h>
#include <openhbci/hbci.h>
#include <openhbci/user.h>
#include <openhbci/account.h>
#include <openhbci/bpdjob.h>
#include <openhbci/bank.h>
#include <openhbci/cryptkey.h>
#include <openhbci/deskey.h>
#include <openhbci/medium.h>
#include <mediumddv.h>

#include <unistd.h> // for sleep


// for bad_cast exception
#include <typeinfo>

namespace HBCI {


/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                              DDVCard
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


DDVCard::CallBackResult DDVCard::callback(bool first){
  if (!_hbci)
    return CallBackAbort;
  if (_hbci->interactor().ref().keepAlive())
    return CallBackContinue;
  return CallBackAbort;
}


DDVCard::DDVCard(const CTCard &c, const Hbci *hbci)
:HBCICard(c)
,_hbci(hbci)
{
}


DDVCard::~DDVCard(){
}


/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                              DDVCardTrader
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */


DDVCardTrader::CallBackResult DDVCardTrader::callback(bool first){
  if (!_hbci)
    return CallBackAbort;
  if (_hbci->interactor().ref().keepAlive())
    return CallBackContinue;
  return CallBackAbort;
}


DDVCardTrader::DDVCardTrader(const Hbci *hbci,
			     bool next,
			     unsigned int readerFlags,
			     unsigned int readerFlagsMask,
			     unsigned int status,
			     unsigned int statusMask,
			     unsigned int statusDelta)
:CTCardTrader(next, readerFlags, readerFlagsMask,
	      status, statusMask, statusDelta)
  ,_hbci(hbci)
{
}


DDVCardTrader::~DDVCardTrader(){
}





/*_________________________________________________________________________
 *AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
 *                              MediumDDV
 *YYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYYY
 */



MediumDDV::MediumDDV(const Hbci *hbci,
                     const string &cardnumber)
:Medium(hbci)
,_ismounted(false)
,_mountcount(0)
,_loglevel(LoggerLevelError)
,_useKeyPad(true)
,_cardnumber(cardnumber)
,_country(280)
,_haveKeyVersions(false)
,_signKeyNumber(0)
,_signKeyVersion(0)
,_cryptKeyNumber(0)
,_cryptKeyVersion(0)
{
}


MediumDDV::~MediumDDV(){
}


const string& MediumDDV::mediumName() const {
  return _cardnumber;
}


const string& MediumDDV::serialNumber() const {
  return _cardnumber;
}


Error MediumDDV::verify(const string &data, const string &signature){
  string hash;
  string mac;

  // make RIPEMD-160
  hash=ripe(data);
  // create MAC by the chip card
  if (!_card.ref().hash2MAC(hash,mac))
    return Error("MediumDDV::verify",
		 "Error on hash2MAC()",
		 0);
  // compare both strings
  if (mac==signature)
    return Error();
  return Error("MediumDDV::verify",
	       "Bad signature()",
	       0);
}


string MediumDDV::sign(const string &data){
  string hash;
  string mac;

  // make RIPEMD-160
  hash=ripe(data);
  // create MAC by the chip card
  if (!_card.ref().hash2MAC(hash,mac))
    throw Error("MediumDDV::sign",
		"Error on hash2MAC()",
		0);
  // return MAC
  // that's all
  return mac;
}


string MediumDDV::createMessageKey() const {
  string kL;
  string kR;
  string key;

  // create left
  if (!const_cast<DDVCard*>(_card.ptr())->getRandom(kL))
    throw Error("MediumDDV::createMessageKey",
		"Error on getRandom()",
		0);
  // create right
  if (!const_cast<DDVCard*>(_card.ptr())->getRandom(kR))
    throw Error("MediumDDV::createMessageKey",
		"Error on getRandom()",
		0);
  key=kL + kR;
  return key;
}


string MediumDDV::encryptKey(const string &srckey){
    string kL;
    string kR;
    string kLenc;
    string kRenc;
    string dstkey;

    // check sizes
    if (srckey.length()!=16)
        throw Error("MediumDDV::encryptKey",
                        "Bad length of srckey",
                        0);
    // split key
    kL=srckey.substr(0,8);
    kR=srckey.substr(8);
    // encode left
    _card.ref().cryptBlock(kL,kLenc);

    // encode right
    _card.ref().cryptBlock(kR,kRenc);
    dstkey=kLenc + kRenc;
    return dstkey;
}


string MediumDDV::decryptKey(const string &srckey){
    return encryptKey(srckey);
}


Error MediumDDV::_checkCard(Pointer<DDVCard> hcard){
  Error err;
  HBCICard::CardData scid;
  string cardnumber;
  CTError cterr;

  try {
    // try to open the card
    cterr=hcard.ref().openCard();
    if (!cterr.isOk()) {
      Error lerr;

      lerr=Error(cterr.where(),
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_WRONG_MEDIUM,
		 ERROR_ADVISE_RETRY,
		 cterr.info(),
		 cterr.explanation());
      return Error("MediumDDV::_checkCard()", lerr);
    }

    // try to read the card identification data
    if (Hbci::debugLevel()>2)
      fprintf(stderr,"MediumDDV: Getting card id.\n");
    scid=hcard.ref().getCardId();
    if (Hbci::debugLevel()>2)
      fprintf(stderr,"MediumDDV: Reading card number.\n");
    cardnumber=scid.cardNumber();
  }
  catch (CTError xerr) {
    Error lerr;

    if (hbci()->debugLevel()>0)
      fprintf(stderr,"MediumDDV: Exception: %s\n",
	      xerr.errorString().c_str());
    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    // close card
    cterr=hcard.ref().closeCard();
    if (!cterr.isOk()) {
      if (hbci()->debugLevel()>0)
	fprintf(stderr,"MediumDDV: Error closing card: %s\n",
		cterr.errorString().c_str());
    }
    return Error("MediumDDV::_checkCard()", lerr);
  }

  // if cardnumber read is not the one we want
  if (Hbci::debugLevel()>1)
    fprintf(stderr,"MediumDDV: Card is open, checking card number.\n");
  if (cardnumber==_cardnumber || _cardnumber.empty()) {
    // set new card number if ours is empty
    if (_cardnumber.empty())
      _cardnumber=cardnumber;
  }
  else {
    if (hbci()->debugLevel()>1)
      fprintf(stderr,"MediumDDV: False card\n");
    // close card
    cterr=hcard.ref().closeCard();
    if (!cterr.isOk()) {
      if (hbci()->debugLevel()>0)
	fprintf(stderr,"MediumDDV: Error closing card: %s\n",
		cterr.errorString().c_str());
    }

    return Error("MediumDDV::_checkCard",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_WRONG_MEDIUM,
		 ERROR_ADVISE_RETRY,
		 "wrong card");
  }
  // everything's fine
  return Error();
}


Error MediumDDV::_enterPin(Pointer<DDVCard> hcard,
			   const string &pin) {
  Error herr;
  CTError err;
  string localpin;

  if ((hcard.ref().readerFlags() & CHIPCARD_READERFLAGS_KEYPAD) &&
      _useKeyPad) {
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"MediumDDV: Terminal has a keypad, will ask for pin.\n");
    // tell the user about pin verification
    _hbci->interactor().ref().
      msgStartInputPinViaKeypad(owner());
    err=hcard.ref().verifyPin();
    // tell the user about end of pin verification
    _hbci->interactor().ref().
      msgFinishedInputPinViaKeypad(owner());
    if (!err.isOk()) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr,"MediumDDV: Bad pin.\n");
      sleep(1);
      _hbci->interactor().ref().msgStateResponse("Error: "+
						 err.errorString());
      if (err.code()==k_CTERROR_OK) {
	int ec;

	if (err.subcode1()==0x63) {
	  switch (err.subcode2()) {
	  case 0xc0: ec=HBCI_ERROR_CODE_PIN_WRONG_0; break;
	  case 0xc1: ec=HBCI_ERROR_CODE_PIN_WRONG_1; break;
	  case 0xc2: ec=HBCI_ERROR_CODE_PIN_WRONG_2; break;
	  default:   ec=HBCI_ERROR_CODE_PIN_WRONG; break;
	  } // switch
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_CRITICAL,
		       ec,
		       ERROR_ADVISE_RETRY,
		       "Bad pin",
		       "verifyPin (1)");
	}
	else if (err.subcode1()==0x69 &&
		 err.subcode2()==0x83) {
	  ec=HBCI_ERROR_CODE_CARD_DESTROYED;
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_CRITICAL,
		       ec,
		       ERROR_ADVISE_ABORT,
		       "Card destroyed",
		       "verifyPin (2)");
	}
	else if (err.subcode1()==0x64 &&
		 err.subcode2()==0x01) {
	  ec=HBCI_ERROR_CODE_PIN_ABORTED;
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_NORMAL,
		       ec,
		       ERROR_ADVISE_ABORT,
		       "User aborted",
		       "verifyPin (3)");

	}
      }
      return Error("MediumDDV::_enterPin",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_PIN_ABORTED,
		   ERROR_ADVISE_ABORT,
		   err.errorString(),
		   "verifyPin (4)");
    }
  } // if hasKeyPad
  else {
    localpin=pin;
    if (Hbci::debugLevel()>2)
      fprintf(stderr,
	      "MediumDDV: No keypad (or disabled), will ask for pin.\n");
    if (localpin.length()<4) {
      herr=_hbci->authentificator().ref().getSecret(owner(),
						    _cardnumber,
						    localpin);
      if (!herr.isOk()) {
	if (Hbci::debugLevel()>1)
	  fprintf(stderr,"MediumDDV: No pin entered.\n");
	_hbci->interactor().ref().msgStateResponse("Error: "+
						   herr.errorString());
	return Error("MediumDDV::_enterPin",
		     ERROR_LEVEL_NORMAL,
		     herr.code(),
		     ERROR_ADVISE_ABORT,
		     "User aborted",
		     "verifyPin (5)");
      } // if user aborted
    } // if pin is not long enough

    // verify the pin
    if (Hbci::debugLevel()>1)
      fprintf(stderr,"MediumDDV: Verifying cardholder pin.\n");
    err=hcard.ref().verifyPin(localpin); // CARDHOLDER pin
    if (!err.isOk()) {
      if (Hbci::debugLevel()>1)
	fprintf(stderr,"MediumDDV: Bad pin.\n");
      _hbci->interactor().ref().msgStateResponse("Error: "+err.errorString());
      if (err.code()==k_CTERROR_OK) {
	int ec;

	if (err.subcode1()==0x63) {
	  switch (err.subcode2()) {
	  case 0xc0: ec=HBCI_ERROR_CODE_PIN_WRONG_0; break;
	  case 0xc1: ec=HBCI_ERROR_CODE_PIN_WRONG_1; break;
	  case 0xc2: ec=HBCI_ERROR_CODE_PIN_WRONG_2; break;
	  default:   ec=HBCI_ERROR_CODE_PIN_WRONG; break;
	  } // switch
	  return Error("MediumDDV::_enterPin",
		       ERROR_LEVEL_CRITICAL,
		       ec,
		       ERROR_ADVISE_RETRY,
		       "Bad pin",
		       "verifyPin (6)");
	}
        else if (err.subcode1()==0x69 &&
                 err.subcode2()==0x83) {
          ec=HBCI_ERROR_CODE_CARD_DESTROYED;
          return Error("MediumDDV::_enterPin",
                       ERROR_LEVEL_CRITICAL,
                       ec,
                       ERROR_ADVISE_ABORT,
                       "Card destroyed",
                       "verifyPin (7)");
        }
        else {
          return Error("MediumDDV::_enterPin",
                       ERROR_LEVEL_CRITICAL,
                       HBCI_ERROR_CODE_UNKNOWN,
                       ERROR_ADVISE_ABORT,
                       err.errorString(),
                       "verifyPin (8)");
        }
      }
      else {
        return Error("MediumDDV::_enterPin",
                     ERROR_LEVEL_CRITICAL,
                     HBCI_ERROR_CODE_PIN_ABORTED,
                     ERROR_ADVISE_ABORT,
                     err.errorString(),
                     "verifyPin (9)");
      }
    }
  } // if no keyPad

  return Error();
}


Error MediumDDV::mountMedium(const string &pin){
  Pointer<DDVCardTrader> trader;
  bool first;
  bool havecard;
  Pointer<DDVCard> hcard;
  CTCard *cardp;
  CTError err;
  Error herr;
  bool havepin;

  if (isMounted()) {
    _mountcount++;
    return Error();
  }

  Logger_SetLevel(_loglevel);

  trader=new DDVCardTrader(hbci(),
			   false,
			   0,
			   0,
			   CHIPCARD_STATUS_INSERTED,
			   CHIPCARD_STATUS_INSERTED |
			   CHIPCARD_STATUS_LOCKED_BY_OTHER,
			   CHIPCARD_STATUS_INSERTED);
  err=trader.ref().start();
  if (!err.isOk()) {
    fprintf(stderr, "MediumDDV: Could not initialize trader");
    return Error("MediumDDV::mountMedium()",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_NO_CARD,
		 ERROR_ADVISE_ABORT,
		 "could not start trader");
  }

  first=true;
  havecard=false;
  while (!havecard) {
    int timeout;

    // determine timeout value
    if (first) {
      timeout=3;
    }
    else
      timeout=5;

    err=trader.ref().getNext(cardp, timeout);
    if (!err.isOk()) {
      if (err.code()==k_CTERROR_API &&
	  (err.subcode1()==CHIPCARD_ERROR_NO_TRANSPORT ||
	   err.subcode1()==CHIPCARD_ERROR_NO_REQUEST)) {
	fprintf(stderr,
		"MediumDDV: Service unreachable, maybe \"chipcardd\" is not running?\n");
	trader.ref().stop();
	return Error("MediumDDV::mountMedium()",
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_NO_CARD,
		     ERROR_ADVISE_ABORT,
		     "service unreachable");
      }
      // no card inserted, simple timeout, so ask the user to insert card
      if (!_hbci->interactor().ref().
	  msgInsertMediumOrAbort(owner(),MediumTypeCard)) {
	if (Hbci::debugLevel()>1)
	  fprintf(stderr,"MediumDDV: User aborted.\n");
	trader.ref().stop();
	return Error("MediumDDV::mountMedium()",
		     ERROR_LEVEL_NORMAL,
                     HBCI_ERROR_CODE_USER_ABORT,
		     ERROR_ADVISE_ABORT,
		     "No card, user aborted.");
      }
    }
    else {
      // ok, we have a card, now check it
      hcard=new DDVCard(*cardp, hbci());
      herr=_checkCard(hcard);
      if (!herr.isOk()) {
	if (herr.code()==HBCI_ERROR_CODE_WRONG_MEDIUM) {
	  /* false card, check whether there is another card in the queue.
	   * If it is not, then ask user to insert the correct card
	   */
	  err=trader.ref().peekNext();
	  if (!err.isOk()) {
	    if (!_hbci->interactor().ref().
		msgInsertCorrectMediumOrAbort(owner(),
					      MediumTypeCard)) {
	      trader.ref().stop();
	      return Error("MediumDDV::mountMedium()",
			   ERROR_LEVEL_NORMAL,
			   HBCI_ERROR_CODE_USER_ABORT,
			   ERROR_ADVISE_ABORT,
			   "user aborted");
	    }
	  }
	  /* otherwise there already is another card in another reader,
	   * so no need to bother the user. This allows to insert all
	   * cards in all readers and let me choose the card ;-) */
	} // if wrong medium
	else {
	  // a severe error with the card, so stop now
	  trader.ref().stop();
	  return Error("MediumDDV::mountMedium()", herr);
	}
      } // if check card returns an error
      else {
	// card is the one we wanted, so flag it
	havecard=true;
      }
    }

    first=false;
  } // while !havecard

  // ok, now we have the card we wanted to have, now ask for the pin
  trader.ref().stop();
  havepin=false;
  while(!havepin) {
    herr=_enterPin(hcard, pin);
    if (!herr.isOk()) {
      hcard.ref().closeCard();
      return Error("mountMedium", herr);
    }
    else
      havepin=true;
  } // while !havepin

  // ok, now we have a card and the PIN is verified, that's it ;-)
  _mountcount++;
  _card=hcard;
  return Error();
}



Error MediumDDV::unmountMedium(const string&){
    CTError err;

    if (--_mountcount>0)
      return Error();
    _mountcount=0;
    err=_card.ref().closeCard();
    if (!err.isOk(0x62))
      // just check for IO error, do not check SW1/2
      return Error("MediumDDV::unmountMedium()",
		   "Could not unmount chip card.",
		   0);
    return Error();
}


bool MediumDDV::isMounted() {
  if (_mountcount<1)
    return false;
  return true;
}


string MediumDDV::mediumId() const {
    string result;
    const_cast<MediumDDV*>(this)->readCID();
    result="@";
    result+=String::num2string(_cid.length());
    result+="@";
    result+=_cid;
    return result;
}


void MediumDDV::readCID(){
  if (!_card.ref().getCID(_cid).isOk())
    throw Error("MediumDDV::readCID()",
		"Could not read chip card.",
		0);
}


unsigned int MediumDDV::nextSEQ(){
  unsigned int seq;

  if (!_card.ref().readSEQ(seq))
    throw Error("MediumDDV::nextSEQ()",
		"Could not read chip card.",
		0);
  seq++;
  if (!_card.ref().writeSEQ(seq))
    throw Error("MediumDDV::nextSEQ()",
		"Could not write chip card.",
		0);
  return seq;
}


int MediumDDV::signKeyNumber() const {
  if (!_haveKeyVersions) {
    Error err;

    err=const_cast<MediumDDV*>(this)->_readKeys();
    if (!err.isOk()) {
      throw Error("MediumDDV::signKeyNumber", err);
    }
  }
  return _signKeyNumber;
}


int MediumDDV::signKeyVersion() const {
  if (!_haveKeyVersions) {
    Error err;

    err=const_cast<MediumDDV*>(this)->_readKeys();
    if (!err.isOk()) {
      throw Error("MediumDDV::signKeyVersion", err);
    }
  }
  return _signKeyVersion;
}


int MediumDDV::cryptKeyNumber() const {
  if (!_haveKeyVersions) {
    Error err;

    err=const_cast<MediumDDV*>(this)->_readKeys();
    if (!err.isOk()) {
      throw Error("MediumDDV::cryptKeyNumber", err);
    }
  }
  return _cryptKeyNumber;
}


int MediumDDV::cryptKeyVersion() const {
  if (!_haveKeyVersions) {
    Error err;

    err=const_cast<MediumDDV*>(this)->_readKeys();
    if (!err.isOk()) {
      throw Error("MediumDDV::cryptKeyVersion", err);
    }
  }
  return _cryptKeyVersion;
}


Error MediumDDV::selectContext(int country,
			       const string &instcode,
			       const string &userid){
  HBCICard::instituteData sid;
  int i;
  int validentries;
  CTError cterr;

  validentries=0;
  // if the context is the same as the current then do nothing
  if (_instcode==instcode && _userid==userid)
    return Error();

  // try to find the context on chip card
  for (i=1; i<6; i++) {
    try {
      cterr=_card.ref().getInstituteData(i, sid);
      if (!cterr.isOk()) {
	return Error(cterr.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_MEDIUM,
		     ERROR_ADVISE_DONTKNOW,
		     cterr.info(),
		     cterr.explanation());
      }
    }
    catch (CTError xerr) {
      Error lerr;

      if (hbci()->debugLevel()>0)
	fprintf(stderr,"Exception: %s\n",
		xerr.errorString().c_str());
      lerr=Error(xerr.where(),
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_MEDIUM,
		 ERROR_ADVISE_DONTKNOW,
		 xerr.info(),
		 xerr.explanation());
      return Error("MediumDDV::selectContext()", lerr);
    }

    // fist check if the entry is valid
    if (sid.country()!=0 && sid.code()!="00000000") {
      validentries++;
      if (sid.code()==instcode)
	if (sid.user()==userid) {
	  // ok, we found it, set new context
	  _country=country;
	  _instcode=instcode;
	  _userid=userid;
	  _addr=sid.address();
	  return Error();
	}
    } // if valid
  } // for
  // ok, we did not find the context, now check why
  if (!validentries) {
#if DEBUGMODE>0
    fprintf(stderr,
	    "Warning: Your card has no valid institute entries.\n");
#endif
    _country=country;
    _instcode=instcode;
    _userid=userid;
    return Error();
  }
  return Error("MediumDDV::selectContext",
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_INVALID,
	       ERROR_ADVISE_DONTKNOW,
	       "no matching entry found");
}


Error MediumDDV::getContext(int num,
			    int &countrycode,
			    string &instcode,
			    string &userid,
			    string &server) const {
  HBCICard::instituteData sid;

  if (num<1)
    return Error("MediumDDV::selectContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "bad context number");
  if (!const_cast<MediumDDV*>(this)->isMounted()){
    return Error("MediumDDV::selectContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "medium not mounted");
  }
  try {
    sid=const_cast<DDVCard*>(_card.ptr())->getInstituteData(num);
  }
  catch (CTError xerr) {
    Error lerr;

    if (hbci()->debugLevel()>0)
      fprintf(stderr,"Exception: %s\n",
	      xerr.errorString().c_str());
    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    return Error("MediumDDV::getContext()", lerr);
  }

  countrycode=sid.country();
  instcode=sid.code();
  userid=sid.user();
  server=sid.address();
  return Error();
}

Error MediumDDV::changePIN() {
  return Error("MediumDDV::changePIN",
	       "can't change PIN on DDV cards",
	       0);
}


Error MediumDDV::changeContext(int context, int country,
			       const string instcode,
			       const string userid, const string custid,
			       const string server) {
  Error err;

  if (context<1)
    return Error("MediumDDV::changeContext",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_INVALID,
		 ERROR_ADVISE_DONTKNOW,
		 "bad context number");

  err=mountMedium();
  if (!err.isOk())
    return Error("MediumDDV::changeContext", err);

  HBCICard::instituteData sid;

  try {
    sid=const_cast<DDVCard*>(_card.ptr())->getInstituteData(context);

    if (0 != country)
      sid.setCountry(country);
    if (! instcode.empty())
      sid.setCode(instcode);
    if (! userid.empty())
      sid.setUser(userid);
    if (! server.empty())
      sid.setAddress(server);

    const_cast<DDVCard*>(_card.ptr())->putInstituteData(context, sid);
  }
  catch (CTError xerr) {
    Error lerr;
    unmountMedium();

    if (hbci()->debugLevel()>0)
      fprintf(stderr,"Exception: %s\n",
	      xerr.errorString().c_str());
    lerr=Error(xerr.where(),
	       ERROR_LEVEL_NORMAL,
	       HBCI_ERROR_CODE_MEDIUM,
	       ERROR_ADVISE_DONTKNOW,
	       xerr.info(),
	       xerr.explanation());
    return Error("MediumDDV::getContext()", lerr);
  }

  return unmountMedium();
}



LOGGER_LEVEL MediumDDV::_strToLogLevel(const string &s) const {
  LOGGER_LEVEL l;

  if (strcmp(s.c_str(), "emergency")==0)
    l=LoggerLevelEmergency;
  else if (strcmp(s.c_str(), "alert")==0)
    l=LoggerLevelAlert;
  else if (strcmp(s.c_str(), "critical")==0)
    l=LoggerLevelCritical;
  else if (strcmp(s.c_str(), "error")==0)
    l=LoggerLevelError;
  else if (strcmp(s.c_str(), "warning")==0)
    l=LoggerLevelWarning;
  else if (strcmp(s.c_str(), "notice")==0)
    l=LoggerLevelNotice;
  else if (strcmp(s.c_str(), "info")==0)
    l=LoggerLevelInfo;
  else if (strcmp(s.c_str(), "debug")==0)
    l=LoggerLevelDebug;
  else {
    fprintf(stderr,
	    "Unknown log level \"%s\", using \"error\"\n",
	    s.c_str());
    l=LoggerLevelError;
  }

  return l;
}


string MediumDDV::_logLevelToStr(LOGGER_LEVEL l) const {
  string s;

  switch(l) {
  case LoggerLevelEmergency: s="emergency"; break;
  case LoggerLevelAlert:     s="alert"; break;
  case LoggerLevelCritical:  s="critical"; break;
  case LoggerLevelError:     s="error"; break;
  case LoggerLevelWarning:   s="warning"; break;
  case LoggerLevelNotice:    s="notice"; break;
  case LoggerLevelInfo:      s="info"; break;
  case LoggerLevelDebug:     s="debug"; break;
  default:                   s="error"; break;
  } // switch

  return s;
}



Error MediumDDV::getProperty(const string &propertyName,
			     string &propertyValue) {
  if (strcasecmp(propertyName.c_str(), "loglevel")==0) {
    propertyValue=_logLevelToStr(_loglevel);
  }
  else if (strcasecmp(propertyName.c_str(), "usekeypad")==0) {
    propertyValue=_useKeyPad?"yes":"no";
  }
  else {
    return Error("MediumDDV::getProperty",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_UNKNOWN_PROPERTY,
		 ERROR_ADVISE_DONTKNOW,
		 "Unknown property",
		 propertyName);
  }

  return Error();
}


Error MediumDDV::setProperty(const string &propertyName,
			     const string &propertyValue){
  if (strcasecmp(propertyName.c_str(), "loglevel")==0) {
    _loglevel=_strToLogLevel(propertyValue);
  }
  else if (strcasecmp(propertyName.c_str(), "usekeypad")==0) {
    if (strcasecmp(propertyValue.c_str(),"yes")==0) {
      _useKeyPad=true;
    }
    else if (strcasecmp(propertyValue.c_str(),"true")==0) {
      _useKeyPad=true;
    }
    else if (strcasecmp(propertyValue.c_str(),"1")==0) {
      _useKeyPad=true;
    }
    else if (strcasecmp(propertyValue.c_str(),"no")==0) {
      _useKeyPad=false;
    }
    else if (strcasecmp(propertyValue.c_str(),"false")==0) {
      _useKeyPad=false;
    }
    else if (strcasecmp(propertyValue.c_str(),"0")==0) {
      _useKeyPad=false;
    }
    else {
      return Error("MediumDDV::setProperty",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_UNKNOWN_PROPERTY,
		   ERROR_ADVISE_DONTKNOW,
		   "Bad value for property \"usekeypad\"",
		   propertyValue);
    }
  }
  else {
    return Error("MediumDDV::setProperty",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_UNKNOWN_PROPERTY,
		 ERROR_ADVISE_DONTKNOW,
		 "Unknown property",
		 propertyName);
  }

  return Error();
}



Error MediumDDV::_keysDDV0() {
  CTError err;
  CTCommand cmd;
  string fcp;
  string data;

  /* signkey */
  err=_card.ref().selectEF(fcp, 0x0013);
  if (!err.isOk()) {
    Error lerr=Error(err.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_WRONG_MEDIUM,
		     ERROR_ADVISE_RETRY,
		     err.info(),
		     err.explanation());
    return Error("MediumDDV::_keysDDV0", lerr);
  }

  data.erase();
  fprintf(stderr, "MediumDDV: Will read record 1\n");
  err=_card.ref().readRecord(data, 1, 0);
  if (!err.isOk()) {
    Error lerr=Error(err.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_WRONG_MEDIUM,
		     ERROR_ADVISE_RETRY,
		     err.info(),
		     err.explanation());
    return Error("MediumDDV::_keysDDV0", lerr);
  }

  if (data.length()==5 &&
      (unsigned char)data[1]==16 &&
      (unsigned char)data[2]==7) {
    _signKeyNumber=(unsigned char)data[0];
    _signKeyVersion=(unsigned char)data[4];
  }
  else {
    return Error("MediumDDV::_keysDDV0",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_WRONG_MEDIUM,
		 ERROR_ADVISE_DONTKNOW,
		 "bad format of signkey cardfile");
  }

  /* cryptkey */
  err=_card.ref().selectEF(fcp, 0x0014);
  if (!err.isOk()) {
    Error lerr=Error(err.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_WRONG_MEDIUM,
		     ERROR_ADVISE_RETRY,
		     err.info(),
		     err.explanation());
    return Error("MediumDDV::_keysDDV0", lerr);
  }
  data.erase();
  err=_card.ref().readRecord(data, 1, 4);
  if (!err.isOk()) {
    Error lerr=Error(err.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_WRONG_MEDIUM,
		     ERROR_ADVISE_RETRY,
		     err.info(),
		     err.explanation());
    return Error("MediumDDV::_keysDDV0", lerr);
  }

  if (data.length()==4 &&
      (unsigned char)data[1]==16 &&
      (unsigned char)data[2]==7) {
    _cryptKeyNumber=(unsigned char)data[0];
    _cryptKeyVersion=(unsigned char)data[4];
  }
  else {
    return Error("MediumDDV::_keysDDV0",
		 ERROR_LEVEL_NORMAL,
		 HBCI_ERROR_CODE_WRONG_MEDIUM,
		 ERROR_ADVISE_DONTKNOW,
		 "bad format of cryptkey cardfile");
  }
  _haveKeyVersions=true;
  return Error();
}



Error MediumDDV::_keysDDV1() {
  CTError err;
  CTCommand cmd;
  string fcp;
  string data;
  int i;

  err=_card.ref().selectEF(fcp, 0x0013);
  if (!err.isOk()) {
    Error lerr=Error(err.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_WRONG_MEDIUM,
		     ERROR_ADVISE_RETRY,
		     err.info(),
		     err.explanation());
    return Error("MediumDDV::_keysDDV1", lerr);
  }

  for (i=1; i<4; i++) {
    unsigned int pos;
    CTPointer<CTTLV> tlv;
    int keyNum=0;
    int keyVersion=0;
    string kt;

    data.erase();
    err=_card.ref().readRecord(data, i, 0);
    if (!err.isOk()) {
      Error lerr=Error(err.where(),
		       ERROR_LEVEL_NORMAL,
		       HBCI_ERROR_CODE_WRONG_MEDIUM,
		       ERROR_ADVISE_RETRY,
		       err.info(),
		       err.explanation());
      return Error("MediumDDV::_keysDDV1", lerr);
    }

    pos=0;
    while (pos<data.length()) {
      tlv=new CTTLV(data, pos, true);
      if (tlv.ref().isValid()) {
	string tstr;

	tstr=tlv.ref().getValue();
	switch(tlv.ref().getTag()) {
	case 0x93:
	  if (tstr.length()==2) {
	    keyNum=(unsigned char)(tstr[0]);
	    keyVersion=(unsigned char)(tstr[1]);
	  }
	  break;

	default:
	  break;
	} /* switch */
      }
    } /* while */
    switch(i) {
    case 2: _signKeyNumber=keyNum; _signKeyVersion=keyVersion; break;
    case 3: _cryptKeyNumber=keyNum; _cryptKeyVersion=keyVersion; break;
    default: break;
    } // switch

  } /* for */

  _haveKeyVersions=true;
  return Error();
}



Error MediumDDV::_readKeys() {
  HBCICard::CardData cd;

  try {
    cd=_card.ref().getCardId();
    if (cd.cardType()==k_HBCICARD_TYPE_0)
      return _keysDDV0();
    else if (cd.cardType()==k_HBCICARD_TYPE_1)
      return _keysDDV1();
    else {
      fprintf(stderr, "Unknown DDV card.\n");
      return Error("MediumDDV::_readKeys",
		   ERROR_LEVEL_NORMAL,
		   HBCI_ERROR_CODE_WRONG_MEDIUM,
		   ERROR_ADVISE_DONTKNOW,
		   "unknown DDVCard type");
    }
  } /* try */
  catch(CTError cterr) {
    Error lerr=Error(cterr.where(),
		     ERROR_LEVEL_NORMAL,
		     HBCI_ERROR_CODE_WRONG_MEDIUM,
		     ERROR_ADVISE_RETRY,
		     cterr.info(),
		     cterr.explanation());
    return Error("MediumDDV::_readKeys()", lerr);
  }

  return Error();
}



} // namespace HBCI

HBCI_Medium *HBCI_MediumDDV_Medium(HBCI_MediumDDV *h)
{
    return h;
}
HBCI_MediumDDV *HBCI_Medium_MediumDDV(HBCI_Medium *h)
{
    try {
	return dynamic_cast<HBCI_MediumDDV*>(h);
    } 
    catch (bad_cast) {
	fprintf(stderr, 
		"HBCI_Medium_MediumDDV: Caught bad_cast exception.\n");
	return 0l;
    }
}



