(**
  A model for table-like data. This is an abstract baseclass. It just
  defines an interface which must be implemented by derived models.

  A database modl f.e. would directly work on the result set. Other
  model may work on lists as internal datarepresentation. These models may
  implement additional methods for direct access or for adding and removing
  lines or data.
**)

MODULE VO:Model:Table;

(*
    A model for table-like data.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT O   := VO:Base:Object,
       U   := VO:Base:Util,

       HM  := VO:Model:Header,

       G   := VO:Object,

       co  := IntStr,
       str := Strings;

CONST
  noSelect*         = 0;
  cellSelect*       = 1;
  singleLineSelect* = 2;

  (* Events *)
  selectionMsg* = 0;

TYPE
  TableModel*       = POINTER TO TableModelDesc;

  (**
    Abstract baseclass for all table models. Its purpose is, to define an
    common interface for all tablemodels. The table object itself will only
    talk with the model through this interface.
  **)
  TableModelDesc*   = RECORD (O.ModelDesc)
                        header* : HM.HeaderModel; (* Writeable only for inherting mdoels *)

                        (* selection stuff *)
                        select  : LONGINT; (* type of selection *)
                        sx-,
                        sy-     : LONGINT; (* coords of selected cell/row *)
                      END;


  (* Some internal datatstructures for the ASTableModel *)

  ASColumn          = POINTER TO ARRAY OF U.Text;
  ASRows            = POINTER TO ARRAY OF ASColumn;

  (**

  **)

  ASTableModel*     = POINTER TO ASTableModelDesc;

  (**
    This implementation stores the value internally as a two dimentional (a)rray
    of (s) strings.
  **)
  ASTableModelDesc* = RECORD (TableModelDesc)
                        rows : ASRows;
                      END;

  CellEntryDesc    = RECORD
                       object : G.Object;
                       text   : U.Text;
                     END;

  LRow             = POINTER TO ARRAY OF CellEntryDesc;

  ListEntry        = POINTER TO ListEntryDesc;
  ListEntryDesc    = RECORD
                       last,
                       next   : ListEntry;
                       row    : LRow;
                     END;

  LTableModel*     = POINTER TO LTableModelDesc;

  (**
    This implementation stores the value internally as a two dimentional (a)rray
    of (s) strings.
  **)
  LTableModelDesc* = RECORD (TableModelDesc)
                        first,
                        last,
                        current    : ListEntry;
                        currentPos : LONGINT;
                        rows       : LONGINT;
                      END;


  TestTable*        = POINTER TO TestModelDesc;

  (**
    A simple model implementation, just for testing purpose.
  **)
  TestModelDesc*    = RECORD (TableModelDesc)
                        buffer : U.Text;
                      END;

  (* --------------------------- *)

  RefreshCell*     = POINTER TO RefreshCellDesc;
  RefreshCellDesc* = RECORD (O.ResyncMsgDesc)
                       x*,y* : LONGINT;
                     END;

  RefreshRow*      = POINTER TO RefreshRowDesc;
  RefreshRowDesc*  = RECORD (O.ResyncMsgDesc)
                       y* : LONGINT;
                     END;

  InsertRow*        = POINTER TO InsertRowDesc;
  InsertRowDesc*    = RECORD (O.ResyncMsgDesc)
                        y*,count* : LONGINT;
                      END;

  DeleteRow*        = POINTER TO DeleteRowDesc;
  DeleteRowDesc*    = RECORD (O.ResyncMsgDesc)
                        y*,count* : LONGINT;
                      END;

  (* --------------------------- *)

  SelectionMsg*     = POINTER TO SelectionMsgDesc;
  SelectionMsgDesc* = RECORD (O.MessageDesc)
                        x*,y* : LONGINT;
                      END;


VAR
  refreshCell : RefreshCell; (* We allocate a global instance and reuse it everytime *)
  refreshRow  : RefreshRow; (* We allocate a global instance and reuse it everytime *)
  insertRow   : InsertRow;
  deleteRow   : DeleteRow;

  PROCEDURE (t : TableModel) Init*;

  BEGIN
    t.Init^;

    t.header:=NIL;

    t.select:=noSelect;
    t.sx:=-1;
    t.sy:=-1;
  END Init;

  PROCEDURE (t : TableModel) RedrawRow*(row : LONGINT);

  BEGIN
    refreshRow.y:=row;
    t.Notify(refreshRow);
  END RedrawRow;

  PROCEDURE (t : TableModel) RedrawCell*(column,row : LONGINT);

  BEGIN
    refreshCell.x:=column;
    refreshCell.y:=row;
    t.Notify(refreshCell);
  END RedrawCell;


  PROCEDURE (t : TableModel) NotifyInsert*(y,count : LONGINT);

  BEGIN
    insertRow.y:=y;
    insertRow.count:=count;

    t.Notify(insertRow);
  END NotifyInsert;

  PROCEDURE (t : TableModel) NotifyDelete*(y,count : LONGINT);

  BEGIN
    deleteRow.y:=y;
    deleteRow.count:=count;

    t.Notify(deleteRow);
  END NotifyDelete;

  PROCEDURE (t : TableModel) SendLineSelection*;

  VAR
    selection : SelectionMsg;

  BEGIN
    NEW(selection);
    selection.y:=t.sy;

    t.Send(selection,selectionMsg);
  END SendLineSelection;

  PROCEDURE (t : TableModel) SetSelectionType*(type : LONGINT);
  (**
    Set the type of selection. Current supported are:
    cellSelect - select an individual cell
    singleLineSelect - select one row
  **)

  BEGIN
    t.select:=type;
    (*
      TODO: If we change the type of the selection we must convert or clear the current
      selection somehow and then update the display.
    *)
  END SetSelectionType;

  PROCEDURE (t : TableModel) GetSelectionType*():LONGINT;

  BEGIN
    RETURN t.select;
  END GetSelectionType;

  PROCEDURE (t : TableModel) SelectRow*(row : LONGINT);

  VAR
    oy : LONGINT;

  BEGIN
    ASSERT(t.select=singleLineSelect);

    IF row#t.sy THEN
      oy:=t.sy;
      t.sy:=row;
      t.RedrawRow(oy);
      t.RedrawRow(row);
      t.SendLineSelection;
    END;
  END SelectRow;

  PROCEDURE (t : TableModel) ClearSelection*;

  VAR
    line : LONGINT;

  BEGIN
    line:=t.sy;

    t.sy:=-1;
    t.sx:=-1;

    IF (t.sy>=0) THEN (* TODO: Add check for upper bound *)
      t.RedrawRow(t.sy);
    END;
  END ClearSelection;

  PROCEDURE (t : TableModel) SelectCell*(column,row : LONGINT);

  VAR
    ox,oy : LONGINT;

  BEGIN
    ASSERT(t.select=cellSelect);

    IF (row#t.sy) OR (column#t.sx) THEN
      ox:=t.sx;
      oy:=t.sy;
      t.sx:=column;
      t.sy:=row;
      t.RedrawCell(ox,oy);
      t.RedrawCell(column,row);
(*      t.SendLineSelection;*)
    END;
  END SelectCell;

  PROCEDURE (t : TableModel) IsRowSelected*(row : LONGINT):BOOLEAN;

  BEGIN
    RETURN t.sy=row;
  END IsRowSelected;

  PROCEDURE (t : TableModel) IsCellSelected*(column,row : LONGINT):BOOLEAN;

  BEGIN
    RETURN (t.sx=column) & (t.sy=row);
  END IsCellSelected;

  PROCEDURE (t : TableModel) GetColumns*():LONGINT;

  BEGIN
    RETURN 1;
  END GetColumns;

  PROCEDURE (t : TableModel) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    RETURN MAX(INTEGER);
  END GetColumnWidth;

  PROCEDURE (t : TableModel) GetRows*():LONGINT;

  BEGIN
    RETURN 0;
  END GetRows;

  PROCEDURE (t : TableModel) GetText*(x,y : LONGINT):U.Text;

  BEGIN
    RETURN NIL;
  END GetText;

  PROCEDURE (t : TableModel) GetObject*(x,y : LONGINT):G.Object;

  BEGIN
    RETURN NIL;
  END GetObject;






  PROCEDURE (t : TestTable) Init*;

  BEGIN
    t.Init^;

    NEW(t.buffer,100);
  END Init;


  PROCEDURE (t : TestTable) GetColumns*():LONGINT;

  BEGIN
    RETURN 10;
  END GetColumns;

  PROCEDURE (t : TestTable) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    RETURN 15*8;
  END GetColumnWidth;

  PROCEDURE (t : TestTable) GetRows*():LONGINT;

  BEGIN
    RETURN 100;
  END GetRows;

  PROCEDURE (t : TestTable) GetText*(x,y : LONGINT):U.Text;

  VAR
    help : ARRAY 32 OF CHAR;

  BEGIN
    COPY("(",t.buffer^);
    co.IntToStr(x,help);
    str.Append(help,t.buffer^);
    str.Append(",",t.buffer^);
    co.IntToStr(y,help);
    str.Append(help,t.buffer^);
    str.Append(")",t.buffer^);

    RETURN t.buffer;
  END GetText;



  (**
    Set the size of the table in rows and columns.
    Currently all previous stored texts are lost. This may change
    in the future.

    NOTE
    We make the internally used array one greater in width and height to store
    possible additional information. Currently the first row is used to store
    the column header texts.
  **)

  PROCEDURE (t : ASTableModel) SetSize*(width,height : LONGINT);

  VAR
    x : LONGINT;

  BEGIN
    NEW(t.rows,height+1);
    FOR x:=0 TO height DO
      NEW(t.rows[x],width+1);
    END;

    t.Notify(NIL);
  END SetSize;

  PROCEDURE (t : ASTableModel) Init*;


  VAR
    h : HM.AHeaderModel;

  BEGIN
    t.Init^;

    NEW(h);
    h.Init;

    t.header:=h;

    t.SetSize(0,0);
  END Init;

  PROCEDURE (t : ASTableModel) SetColumnText*(index : LONGINT; text : U.Text);

  BEGIN
    t.header(HM.AHeaderModel).SetEntryText(index-1,text);
  END SetColumnText;

  PROCEDURE (t : ASTableModel) SetColumnString*(index : LONGINT; text : ARRAY OF CHAR);

  BEGIN
    t.header(HM.AHeaderModel).SetEntryString(index-1,text);
  END SetColumnString;

  PROCEDURE (t : ASTableModel) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    IF t.header.GetEntrySize(index)=0 THEN
      t.header.SetEntrySize(index,15*8);
    END;

    RETURN t.header.GetEntrySize(index);
  END GetColumnWidth;

  PROCEDURE (t : ASTableModel) GetColumns*():LONGINT;

  BEGIN
    IF t.rows=NIL THEN
      RETURN 0;
    ELSE
      RETURN LEN(t.rows[0]^)-1;
    END;
  END GetColumns;

  PROCEDURE (t : ASTableModel) GetRows*():LONGINT;

  BEGIN
    IF t.rows=NIL THEN
      RETURN 0;
    ELSE
      RETURN LEN(t.rows^)-1;
    END;
  END GetRows;

  PROCEDURE (t : ASTableModel) SetText*(x,y : LONGINT; text : U.Text);

  BEGIN
    t.rows[y][x]:=text;

    t.RedrawCell(x,y);
  END SetText;

  PROCEDURE (t : ASTableModel) SetString*(x,y : LONGINT; string : ARRAY OF CHAR);

  VAR
    text : U.Text;

  BEGIN
    NEW(text,str.Length(string)+1);
    COPY(string,text^);

    t.SetText(x,y,text);
  END SetString;

  PROCEDURE (t : ASTableModel) GetText*(x,y : LONGINT):U.Text;

  BEGIN
    RETURN t.rows[y][x];
  END GetText;


  PROCEDURE (t : LTableModel) Init*;

  VAR
    h : HM.AHeaderModel;

  BEGIN
    t.Init^;

    NEW(h);
    h.Init;

    t.header:=h;

    t.first:=NIL;
    t.last:=NIL;
    t.rows:=0;

    t.current:=NIL;
    t.currentPos:=0;
  END Init;

  PROCEDURE (t : LTableModel) SetColumnText*(index : LONGINT; text : U.Text);

  BEGIN
    t.header(HM.AHeaderModel).SetEntryText(index-1,text);
  END SetColumnText;

  PROCEDURE (t : LTableModel) SetColumnString*(index : LONGINT; text : ARRAY OF CHAR);

  BEGIN
    t.header(HM.AHeaderModel).SetEntryString(index-1,text);
  END SetColumnString;

  PROCEDURE (t : LTableModel) SetColumnWidth*(index,width : LONGINT);

  BEGIN
    t.header.SetEntrySize(index-1,width);
  END SetColumnWidth;

  PROCEDURE (t : LTableModel) GetColumns*():LONGINT;

  VAR
    count : LONGINT;

  BEGIN
    count:=t.header.GetEntries();

    IF count=0 THEN
      RETURN 1;
    ELSE
      RETURN count;
    END;
  END GetColumns;

  PROCEDURE (t : LTableModel) GetColumnWidth*(index : LONGINT):LONGINT;

  BEGIN
    IF t.GetColumns()<=1 THEN
      RETURN MAX(INTEGER);
    ELSE
      IF t.header.GetEntrySize(index)=0 THEN
        t.header.SetEntrySize(index,15*8);
      END;

      RETURN t.header.GetEntrySize(index);
    END;
  END GetColumnWidth;

  PROCEDURE (t : LTableModel) GetRows*():LONGINT;

  BEGIN
    RETURN t.rows;
  END GetRows;

  PROCEDURE (t : LTableModel) GetEntry(y : LONGINT):ListEntry;

  BEGIN
    IF t.current=NIL THEN
      t.current:=t.first;
      t.currentPos:=1;
      WHILE y>1 DO
        t.current:=t.current.next;
        INC(t.currentPos);
        DEC(y);
      END;

      RETURN t.current;
    ELSE
      IF y>t.currentPos THEN
        y:=y-t.currentPos;
        WHILE y>0 DO
          t.current:=t.current.next;
          INC(t.currentPos);
          DEC(y);
        END;
      ELSIF y<t.currentPos THEN
        y:=t.currentPos-y;
        WHILE y>0 DO
          t.current:=t.current.last;
          DEC(t.currentPos);
          DEC(y);
        END;
      END;

      RETURN t.current;
    END;
  END GetEntry;

  PROCEDURE (t : LTableModel) InsertEntry*(pos : LONGINT);

  VAR
    insert,
    entry : ListEntry;
    x     : LONGINT;

  BEGIN
   (* we can insert before the first and after the last entry *)
   ASSERT((pos>=0) & (pos<=t.rows));

    NEW(entry);
    entry.next:=NIL;
    entry.last:=NIL;
    NEW(entry.row,t.GetColumns());

    FOR x:=0 TO LEN(entry.row^)-1 DO
      entry.row[x].object:=NIL;
      entry.row[x].text:=NIL;
    END;

    IF pos=0 THEN (* insert before all others *)
      IF t.first=NIL THEN
        t.first:=entry;
        t.last:=entry;
      ELSE
        t.first.last:=entry;
        entry.next:=t.first;
        t.first:=entry;
      END;
    ELSIF pos=t.rows THEN
      t.last.next:=entry;
      entry.last:=t.last;
      t.last:=entry;
    ELSE
      insert:=t.GetEntry(pos);
      entry.next:=insert.next;
      IF insert.next#NIL THEN
        insert.next.last:=entry;
      END;
      entry.last:=insert;
      insert.next:=entry;
    END;

    INC(t.rows);

    t.current:=entry;
    t.currentPos:=pos+1;

    t.NotifyInsert(pos+1,1);
  END InsertEntry;

  PROCEDURE (t : LTableModel) PrependEntry*;

  BEGIN
    t.InsertEntry(0);
  END PrependEntry;

  PROCEDURE (t : LTableModel) AppendEntry*;

  BEGIN
    t.InsertEntry(t.rows);
  END AppendEntry;

  PROCEDURE (t : LTableModel) DeleteEntry*(pos : LONGINT);

  VAR
    entry : ListEntry;
    x     : LONGINT;

  BEGIN
   (* we can insert before the first and after the last entry *)
   ASSERT((pos>=1) & (pos<=t.rows));

    entry:=t.GetEntry(pos);

    IF entry=t.first THEN
      t.first:=t.first.next;
    END;

    IF entry=t.last THEN
      t.last:=t.last.last;
    END;

    IF entry.last#NIL THEN
      entry.last.next:=entry.next;
    END;

    IF entry.next#NIL THEN
      entry.next.last:=entry.last;
    END;

    FOR x:=0 TO LEN(entry.row^)-1 DO
      IF entry.row[x].object#NIL THEN
        entry.row[x].object.Free;
      END;
    END;

    DEC(t.rows);

    t.current:=NIL; (* TODO: We could do better *)

    IF t.sy=pos THEN
      t.sy:=-1;
      (* TODO: Send selectionChange notify *)
    END;

    t.NotifyDelete(pos,1);
  END DeleteEntry;

  PROCEDURE (t : LTableModel) SetObject*(x,y : LONGINT; object: G.Object);

  VAR
    entry : ListEntry;

  BEGIN
    entry:=t.GetEntry(y);

    entry.row[x-1].object:=object;
    entry.row[x-1].text:=NIL;

    t.RedrawCell(x,y);
  END SetObject;

  PROCEDURE (t : LTableModel) SetText*(x,y : LONGINT; text: U.Text);

  VAR
    entry : ListEntry;

  BEGIN
    entry:=t.GetEntry(y);

    entry.row[x-1].object:=NIL;
    entry.row[x-1].text:=text;

    t.RedrawCell(x,y);
  END SetText;

  PROCEDURE (t : LTableModel) SetString*(x,y : LONGINT; string : ARRAY OF CHAR);

  VAR
    text : U.Text;

  BEGIN
    NEW(text,str.Length(string)+1);
    COPY(string,text^);

    t.SetText(x,y,text);
  END SetString;

  PROCEDURE (t : LTableModel) GetText*(x,y : LONGINT):U.Text;

  VAR
    entry : ListEntry;

  BEGIN
    ASSERT((x>=1) & (x<=t.GetColumns()) & (y>=1) & (y<=t.rows));

    entry:=t.GetEntry(y);

    RETURN entry.row[x-1].text;
  END GetText;

  PROCEDURE (t : LTableModel) GetObject*(x,y : LONGINT):G.Object;

  VAR
    entry : ListEntry;

  BEGIN
    ASSERT((x>=1) & (x<=t.GetColumns()) & (y>=1) & (y<=t.rows));

    entry:=t.GetEntry(y);

    RETURN entry.row[x-1].object;
  END GetObject;

BEGIN
  NEW(refreshCell);
  NEW(refreshRow);
  NEW(insertRow);
  NEW(deleteRow);
END VO:Model:Table.