(**
   A cycle gadget as nown from the amiga.
**)

MODULE VO:Cycle;

(*
    Implements a cycle gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VO:Base:Display,
       E  := VO:Base:Event,
       F  := VO:Base:Frame,
       O  := VO:Base:Object,
       U  := VO:Base:Util,

       VM := VO:Model:Value,

       G  := VO:Object,
       V  := VO:VecImage;

CONST
  selectedMsg* = 0;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the cycle is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 image*      : LONGINT;
                 imageRight* : BOOLEAN;
                 highlight*  : BOOLEAN;
               END;


  Cycle*     = POINTER TO CycleDesc;
  CycleDesc* = RECORD (G.GadgetDesc)
                 image    : V.VecImage;

                 current  : G.Object;
                 model    : VM.ValueModel;

                 selected : BOOLEAN;

                 (* copied from Group: *)
                 list*,               (* A linked list of all members *)
                 last*  : G.Object;
                 count* : LONGINT;    (* The number of members *)
               END;

  SelectedMsg*      = POINTER TO SelectedMsgDesc;
  SelectedMsgDesc*  = RECORD (O.MessageDesc)
                      END;


VAR
  prefs* : Prefs;


  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.frame:=F.double3DOut;
    p.image:=V.cycle3D;
    p.imageRight:=TRUE;
    p.highlight:=TRUE;
  END Init;

  PROCEDURE (c : Cycle) Init*;

  BEGIN
    c.Init^;

    c.SetBackground(D.buttonBackgroundColor);

    c.SetFlags({G.canFocus});
    c.RemoveFlags({G.stdFocus});

    c.SetPrefs(prefs);

    NEW(c.image);
    c.image.Init;
    c.image.SetParent(c);
    c.image.SetFlags({G.horizontalFlex,G.verticalFlex});

    c.model:=NIL;
    c.current:=NIL;
    c.selected:=FALSE;
    (* copied from Group: *)
    c.count:=0;
    c.list:=NIL;
  END Init;


  (* copied from Group *)
  (**
    Disables all elements of the group object.
    Note, that inherited group objects may interpret beeing disabled
    different. VOTab f.e. disables itself.
  **)

  PROCEDURE (g : Cycle) Disable*(disabled : BOOLEAN);

  VAR
    help : G.Object;

  BEGIN
    help:=g.list;
    WHILE help#NIL DO
      help.Disable(disabled);
      help:=help.next;
    END;
  END Disable;

  (* copied from Group *)
  (**
    Add a new Object to the group
    Removing objects is currently not supported

    Note that some group-objects have a more special functions for adding
    members. However Add should always be supported
  **)

  PROCEDURE (g : Cycle) Add*(object : G.Object);

  BEGIN
    IF g.list=NIL THEN
      g.list:=object;
    ELSE
      g.last.next:=object;
    END;
    g.last:=object;
    object.SetParent(g);
    INC(g.count);
  END Add;

  (* copied from Group *)
  (**
    The defaulthandler ask all members of the Group

    You have to overload this method if you do not use
    Group.Add and Group.list.
  **)

  PROCEDURE (g : Cycle) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    object,
    return  : G.Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetPosObject(x,y,type);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetPosObject^(x,y,type);
  END GetPosObject;

  (* copied from Group *)
  (**
    Returns the object that coveres the given point and that supports
    drag and drop of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (g : Cycle) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    object,
    return  : G.Object;

  BEGIN
    object:=g.list;
    WHILE object#NIL DO
      return:=object.GetDnDObject(x,y,drag);
      IF return#NIL THEN
        RETURN return;
      END;
      object:=object.next;
    END;
    RETURN g.GetDnDObject^(x,y,drag);
  END GetDnDObject;


  (**
    Assign the model for the cycle object. The cycle object will show the nth entry
    given by the value of the integer model.
  **)

  PROCEDURE (c : Cycle) SetModel*(model : O.Model);

  BEGIN
    IF c.model#NIL THEN
      c.UnattachModel(c.model);
    END;
    IF (model#NIL) & (model IS VM.ValueModel) THEN
      c.model:=model(VM.ValueModel);
      c.model.SetLongint(1);
      c.AttachModel(c.model);
    ELSE
      c.model:=NIL;
    END;
  END SetModel;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (c : Cycle) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=c.model
  END ModelAccepted;


  PROCEDURE (c : Cycle) CalcSize*;

  VAR
    w,mw,h,mh : LONGINT;
    object    : G.Object;

  BEGIN
    c.image.Set(c.prefs(Prefs).image);
    c.image.CalcSize;


    c.height:=0; (* TODO!*)
    c.minHeight:=c.height;

    w :=0; h :=0;
    mw:=0; mh:=0;

    object:=c.list;
    WHILE object#NIL DO
      IF object.StdFocus() THEN
        c.SetFlags({G.stdFocus});
      END;

      object.SetFlags({G.horizontalFlex,G.verticalFlex});
      IF ~c.prefs(Prefs).highlight THEN
        object.SetFlags({G.noHighlight});
      ELSE
        object.RemoveFlags({G.noHighlight});
      END;

      object.CalcSize;
      w:=U.MaxLong(w,object.width);
      mw:=U.MaxLong(mw,object.minWidth);
      h:=U.MaxLong(h,object.height);
      mh:=U.MaxLong(mh,object.minHeight);

      object:=object.next;
    END;

    IF ~c.StdFocus() & c.MayFocus() THEN
      object:=c.list;
      WHILE object#NIL DO
        (* tell the object to reserve space for focus displaying *)
        object.SetFlags({G.mayFocus});
        object:=object.next;
      END;
    END;

    h:=U.MaxLong(h+D.display.spaceHeight,c.image.height);
    mh:=U.MaxLong(mh+D.display.spaceHeight,c.image.height);
    INC(c.height,h);
    INC(c.minHeight,mh);

    (* A little trick to make cycle-Button a little bit nicer *)
    c.image.Resize(c.height,c.height);
    c.width:=c.image.width+D.display.spaceWidth;
    c.minWidth:=c.width;

    INC(c.width,w+D.display.spaceWidth);
    INC(c.minWidth,mw+D.display.spaceWidth);

    c.CalcSize^;
  END CalcSize;

  PROCEDURE (c : Cycle) HandleMouseEvent*(event : E.MouseEvent;
                                          VAR grab : G.Object):BOOLEAN;

  VAR
    selected : SelectedMsg;

  BEGIN
    IF ~c.visible OR c.disabled OR (c.model=NIL) OR c.model.IsNull() THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown) & c.PointIsIn(event.x,event.y)
      & (event.button=E.button1) THEN
        c.selected:=TRUE;
        c.Redraw;

        grab:=c;
        RETURN TRUE;
      ELSIF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        IF c.PointIsIn(event.x,event.y) THEN
          IF c.model.GetLongint()>=c.count THEN
            c.model.SetLongint(1);
          ELSE
            c.model.Inc;
          END;
          NEW(selected);
          c.Send(selected,selectedMsg);
          (* Action: Next entry selected *)
        END;
        c.selected:=FALSE;
        c.Redraw;

        grab:=NIL;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF c.PointIsIn(event.x,event.y) THEN
        IF ~c.selected THEN
          c.selected:=TRUE;
          c.Redraw;

          grab:=c;
          RETURN TRUE;
        END;
      ELSE
        IF c.selected THEN
          c.selected:=FALSE;
          c.Redraw;
        END;

        grab:=NIL;
        RETURN FALSE;
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (c : Cycle) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym   : LONGINT;
    selected : SelectedMsg;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      IF (keysym=E.down) OR (keysym=E.space) THEN
        IF c.model.GetLongint()>=c.count THEN
          c.model.SetLongint(1);
        ELSE
          c.model.Inc;
        END;
        NEW(selected);
        c.Send(selected,selectedMsg);
        RETURN TRUE;
      ELSIF keysym=E.up THEN
        IF c.model.GetLongint()<=1 THEN
          c.model.SetLongint(c.count);
        ELSE
          c.model.Dec;
        END;
        NEW(selected);
        c.Send(selected,selectedMsg);
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (c : Cycle) Draw*(x,y,width,height : LONGINT);

  VAR
    i,w  : LONGINT;
    draw : D.DrawInfo;

  BEGIN
    draw:=c.GetDrawInfo();

    IF c.selected THEN
      draw.mode:={D.selected};
    END;

    c.Draw^(x,y,width,height);

    i:=1;
    c.current:=c.list;
    IF (c.model#NIL) & ~c.model.IsNull() THEN
      WHILE (c.current#NIL) & (i<c.model.GetLongint()) DO
        INC(i);
        c.current:=c.current.next;
      END;
    END;

    IF c.current#NIL THEN
      w:=c.width-c.image.width;
      c.current.Resize(w,c.height);
      c.CopyBackground(c.current);

      IF c.HasFocus() THEN
        (* tell the object to display focus *)
        c.current.SetFlags({G.showFocus});
      ELSE
        (* tell the object to not display focus *)
        c.current.RemoveFlags({G.showFocus});
      END;

      IF c.prefs(Prefs).imageRight THEN
        c.current.Move(c.x+(w-c.current.width) DIV 2,
                       c.y+(c.height-c.current.height) DIV 2);
      ELSE
        c.current.Move(c.x+c.image.width+(w-c.current.width) DIV 2,
                       c.y+(c.height-c.current.height) DIV 2);
      END;
      c.current.Draw(x,y,width,height);

    ELSE
      (* TODO: Clear display *)
    END;

    IF c.prefs(Prefs).imageRight THEN
      c.image.Move(c.x+c.width-c.image.width,c.y);
    ELSE
      c.image.Move(c.x,c.y);
    END;
    c.image.Resize(-1,c.height);
    c.image.Draw(x,y,width,height);

    draw.mode:={};

    IF c.disabled THEN
      c.DrawDisabled;
    END;
  END Draw;

  PROCEDURE (c : Cycle) DrawFocus*;

  VAR
    draw : D.DrawInfo;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (c.current#NIL) & ~c.current.StdFocus() THEN
      draw:=c.GetDrawInfo();

      IF c.selected THEN
        draw.mode:={D.selected};
      ELSE
        draw.mode:={};
      END;
      c.current.DrawFocus;
      draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
      c.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (c : Cycle) HideFocus*;

  VAR
    draw : D.DrawInfo;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (c.current#NIL) & ~c.current.StdFocus() THEN
      draw:=c.GetDrawInfo();

      IF c.selected THEN
        draw.mode:={D.selected};
      ELSE
        draw.mode:={};
      END;
      c.current.HideFocus;
      draw.mode:={};
    ELSE
      (* Delegate drawing to the baseclass *)
     c.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (c : Cycle) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    c.Redraw;
  END Resync;

  PROCEDURE (c : Cycle) Hide*;

  BEGIN
    IF c.visible THEN
      c.image.Hide;
      IF c.current#NIL THEN
        c.current.Hide;
      END;
      c.Hide^;
    END;
  END Hide;

  PROCEDURE CreateCycle*():Cycle;

  VAR
    cycle : Cycle;

  BEGIN
    NEW(cycle);
    cycle.Init;

    RETURN cycle;
  END CreateCycle;

BEGIN
  NEW(prefs);
  prefs.Init;

END VO:Cycle.