#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "libnjb.h"
#include "njb_error.h"
#include "defs.h"
#include "base.h"
#include "unicode.h"
#include "byteorder.h"

extern int njb_unicode_flag;
extern int __sub_depth;


playlist_t *playlist_new (void)
{
	__dsub= "playlist_new";
	playlist_t *pl;

	__enter;

	pl= (playlist_t *) malloc(sizeof(playlist_t));
	if ( pl == NULL ) {
		njb_error_add(__sub, EO_NOMEM);
		__leave;
		return NULL;
	}

	memset(pl, 0, sizeof(playlist_t));
	pl->_state = NJB_PL_NEW;

	__leave;
	return pl;
}

playlist_t *playlist_unpack (void *data, size_t nbytes)
{
	__dsub= "playlist_unpack";
	unsigned char *dp= (unsigned char *) data;
	size_t index;
	playlist_t *pl;
	u_int32_t ntracks, i;
	u_int16_t lname;

	__enter;

	pl= playlist_new();
	if ( pl == NULL ) {
		__leave;
		return NULL;
	}

	pl->plid = njb1_bytes_to_32bit(&dp[0]);
	lname = njb1_bytes_to_16bit(&dp[4]);

	if (njb_unicode_flag == NJB_UC_UTF8) {
	  char *utf8str = NULL;
	  
	  utf8str= strtoutf8(&dp[6]);
	  if (utf8str == NULL) {
	    playlist_destroy(pl);
	    njb_error_add(__sub, EO_NOMEM);
	    return NULL;
	  }
	  pl->name = utf8str;
	} else {
	  pl->name = strdup(&dp[6]);
	  if ( pl->name == NULL ) {
	    playlist_destroy(pl);
	    njb_error_add(__sub, EO_NOMEM);
	    __leave;
	    return NULL;
	  }
	}
	
	index = lname+12;
	ntracks = njb1_bytes_to_32bit(&dp[index]);
	index += 4;

	for (i= 0; i<ntracks; i++) {
		u_int32_t trackid;
		playlist_track_t *track;

		index += 4;
		trackid = njb1_bytes_to_32bit(&dp[index]);
		index += 4;
		if ( index > nbytes ) {
			playlist_destroy(pl);
			njb_error_add(__sub, EO_BADDATA);
			__leave;
			return NULL;
		}

		track= playlist_track_new(trackid);
		if ( track == NULL ) {
			playlist_destroy(pl);
			njb_error_add(__sub, EO_NOMEM);
			__leave;
			return NULL;
		}

		playlist_addtrack(pl, track, NJB_PL_END);
	}

	pl->_state= NJB_PL_UNCHANGED;
	__leave;
	return pl;
}

void playlist_addtrack (playlist_t *pl, playlist_track_t *track, 
	unsigned int pos)
{
	__dsub= "playlist_addtrack";

	__enter;

	if ( pl->_state != NJB_PL_NEW ) pl->_state= NJB_PL_CHTRACKS;

	if ( pos > pl->ntracks ) pos= NJB_PL_END;

	if ( pos == NJB_PL_END ) {
		if ( pl->first == NULL ) {
			pl->first= pl->cur= pl->last= track;
			track->next= track->prev= NULL;
		} else {
			track->prev= pl->last;
			track->next= NULL;
			pl->last->next= track;
			pl->last= track;
		}
	} else if ( pos == NJB_PL_START ) {
		if ( pl->first == NULL ) {
			pl->first= pl->cur= pl->last= track;
			track->next= track->prev= NULL;
		} else {
			track->prev= NULL;
			track->next= pl->first;
			pl->first->prev= track;
			pl->first= track;
		}
	} else {
		int i= 1;
		playlist_track_t *cur;

		playlist_reset_gettrack(pl);
		while ( (cur= playlist_gettrack(pl)) ) {
			if ( i == pos ) {
				cur->prev->next= track;
				track->prev= cur->prev;
				track->next= cur;
				cur->prev= track;

				pl->ntracks++;

				__leave;
				return;
			}

			i++;
		}
	}

	pl->ntracks++;
	__leave;
}

void playlist_deltrack (playlist_t *pl, unsigned int pos)
{
	__dsub= "playlist_deltrack";
	playlist_track_t *track;

	__enter;

	if ( pos > pl->ntracks ) pos= NJB_PL_END;

	pl->_state= NJB_PL_CHTRACKS;

	if ( pos == NJB_PL_START ) {
		track = pl->first;
		pl->first = pl->first->next;
		if (pl->first != NULL) {
		  pl->first->prev = NULL;
		}
	} else if ( pos == NJB_PL_END ) {
		track = pl->last;
		pl->last = pl->last->prev;
		if (pl->last != NULL) {
		  pl->last->next = NULL;
		}
	} else {
		int i= 1;

		playlist_reset_gettrack(pl);
		while ( (track= playlist_gettrack(pl)) ) {
			if ( i == pos ) {
				if ( track->prev ) {
					track->prev->next= track->next;
				}
				if ( track->next ) {
					track->next->prev= track->prev;
				}

				playlist_track_destroy(track);
				pl->ntracks--;

				__leave;
				return;
			}
			i++;
		}
	}

	playlist_track_destroy(track);
	pl->ntracks--;

	__leave;
}

void playlist_destroy (playlist_t *pl)
{
	__dsub= "playlist_destroy";
	playlist_track_t *pltrack;

	__enter;

	pl->cur= pl->first;
	while ( pl->cur != NULL ) {
		pltrack= pl->cur;
		pl->cur= pl->cur->next;
		playlist_track_destroy(pltrack);
	}

	if ( pl->name != NULL ) free(pl->name);
	free(pl);

	__leave;
}

void playlist_reset_gettrack (playlist_t *pl)
{
	__dsub= "playlist_reset_gettrack";

	__enter;

	pl->cur= pl->first;

	__leave;
}

playlist_track_t *playlist_gettrack (playlist_t *pl)
{
	__dsub= "playlist_gettrack";
	playlist_track_t *track;

	__enter;

	if ( pl->cur == NULL ) {
		__leave;
		return NULL;
	}

	track= pl->cur;
	pl->cur= pl->cur->next;

	__leave;
	return track;
}

void playlist_dump (playlist_t *pl, FILE *fp)
{
	__dsub= "playlist_dump";
	playlist_track_t *track;
	unsigned int i= 1;

	__enter;

	fprintf(fp, "Playlist: %s\n", pl->name);
	fprintf(fp, "ID:       %u\n", pl->plid);
	fprintf(fp, "Tracks:   %u\n", pl->ntracks);
	fprintf(fp, "State:    %d\n", pl->_state);

	playlist_reset_gettrack(pl);
	while ( (track= playlist_gettrack(pl)) ) {
		fprintf(fp, "%5u - track ID %u\n", i, track->trackid);
		i++;
	}

	__leave;
}

int playlist_set_name (playlist_t *pl, const char *name)
{
	__dsub= "playlist_set_name";
	char *newname= strdup(name);

	__enter;

	if (newname == NULL ) {
		njb_error_add(__sub, EO_NOMEM);
		__leave;
		return -1;
	}

	if ( pl->name != NULL ) free(pl->name);
	pl->name= newname;

	if ( pl->_state == NJB_PL_UNCHANGED ) pl->_state= NJB_PL_CHNAME;

	__leave;
	return 0;
}

/* Track methods */

playlist_track_t *playlist_track_new (u_int32_t trackid)
{
	__dsub= "playlist_track_new";
	playlist_track_t *track;

	__enter;

	track= (playlist_track_t *) malloc(sizeof(playlist_track_t));
	if ( track == NULL ) {
		njb_error_add(__sub, EO_NOMEM);
		__leave;
		return NULL;
	}

	memset(track, 0, sizeof(playlist_track_t));
	track->trackid= trackid;

	__leave;
	return track;
}

void playlist_track_destroy (playlist_track_t *track)
{
	__dsub= "playlist_track_destroy";

	__enter;
	free(track);
	__leave;
}

