/* libmsv: MonkeySphere Validation library
 *   Copyright (C) 2011  Clint Adams

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#include <stdlib.h>
#include <string.h>
#include <curl/curl.h>

#include <jansson.h>

#include "msv/msv.h"

CURLcode cc;

extern const char *
msv_strerror (int error_code)
{
  switch (error_code)
    {
    case LIBMSV_ERROR_SUCCESS:
      return "No error";
    case LIBMSV_ERROR_INVALID:
      return "Agent reports invalid";
    case LIBMSV_ERROR_NOENVVAR:
      return "Environment variable not set";
    case LIBMSV_ERROR_CURLINIT_FAILED:
      return "curl_easy_init() failed!";
    case LIBMSV_ERROR_INCOMPATIBLE_AGENT:
      return "Validation agent is incompatible";
    case LIBMSV_ERROR_BADARG:
      return "Bad argument(s)";
    case LIBMSV_ERROR_CURLCODE:
      return curl_easy_strerror (cc);
    default:
      return "Unknown error";
    }
}

static size_t
write_function (void *ptr, size_t size, size_t nmemb, void *userdata)
{
  char *ud;

  ud = malloc (size * nmemb + 1);
  memcpy (ud, ptr, size * nmemb);

  ud[size * nmemb] = '\0';

  *((char **) userdata) = (void *) ud;

  return size * nmemb;
}

extern int
msv_check_msva (char *url)
{
  CURL *c;
  char *buf, *socket_location;
  json_t *j, *k;
  json_error_t je;
  void *iter;
  int protoversion, available;

  if (url)
    socket_location = url;
  else
    {
      socket_location = getenv ("MONKEYSPHERE_VALIDATION_AGENT_SOCKET");

      if (socket_location == NULL)
	return LIBMSV_ERROR_NOENVVAR;
    }

  c = curl_easy_init ();
  if (c == NULL)
    return LIBMSV_ERROR_CURLINIT_FAILED;

  cc = curl_easy_setopt (c, CURLOPT_URL, socket_location);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;
  cc = curl_easy_setopt (c, CURLOPT_WRITEFUNCTION, write_function);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;
  cc = curl_easy_setopt (c, CURLOPT_WRITEDATA, &buf);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;

  cc = curl_easy_perform (c);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;

  j = json_loads (buf, 0, &je);
  free (buf);

  iter = json_object_iter_at (j, "protoversion");
  k = json_object_iter_value (iter);

  json_unpack (k, "i", &protoversion);

  iter = json_object_iter_at (j, "available");
  k = json_object_iter_value (iter);

  json_unpack (k, "b", &available);

  if (protoversion == 1 && available == 1)
    return LIBMSV_ERROR_SUCCESS;
  else
    return LIBMSV_ERROR_INCOMPATIBLE_AGENT;
  curl_easy_cleanup (c);
}

extern int
msv_query_agent (char *url, char *context, char *peertype, char *peername,
		 char *pkctype, char *pkcdata, struct msv_response *response)
{

  int oldurllen, validity;
  CURL *c;
  CURLcode cc;
  char *buf, *socket_location, *req;
  json_t *j, *k;
  json_error_t je;
  void *iter;
  struct curl_slist *slist = NULL;


  if (url)
    socket_location = url;
  else
    {
      buf = getenv ("MONKEYSPHERE_VALIDATION_AGENT_SOCKET");

      if (buf == NULL)
	return LIBMSV_ERROR_NOENVVAR;

      oldurllen = strlen (buf);
      socket_location = malloc (oldurllen + 12);
      sprintf (socket_location, "%s%s", buf, "/reviewcert");
    }

  if (context && peertype && peername && pkctype && pkcdata)
    req =
      malloc (strlen (context) + strlen (peertype) + strlen (peername) +
	      strlen (pkctype) + strlen (pkcdata) + 100);
  else
    return LIBMSV_ERROR_BADARG;

  sprintf (req,
	   "{\"pkc\":{\"data\": \"%s\", \"type\": \"%s\"}, \"context\":\"%s\", \"peer\":{\"name\": \"%s\", \"type\": \"%s\"}}",
	   pkcdata, pkctype, context, peername, peertype);

  c = curl_easy_init ();
  if (c == NULL)
    return LIBMSV_ERROR_CURLINIT_FAILED;

  cc = curl_easy_setopt (c, CURLOPT_URL, socket_location);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;
  cc = curl_easy_setopt (c, CURLOPT_WRITEFUNCTION, write_function);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;
  cc = curl_easy_setopt (c, CURLOPT_WRITEDATA, &buf);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;
  cc = curl_easy_setopt (c, CURLOPT_POST, 1);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;

  slist = curl_slist_append (slist, "Content-Type: application/json");
  slist = curl_slist_append (slist, "Connection: close");
  slist = curl_slist_append (slist, "Accept: application/json");
  slist = curl_slist_append (slist, "Expect:");
  cc = curl_easy_setopt (c, CURLOPT_HTTPHEADER, slist);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;

  cc = curl_easy_setopt (c, CURLOPT_POSTFIELDS, (void *) req);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;

  cc = curl_easy_perform (c);
  if (cc)
    return LIBMSV_ERROR_CURLCODE;

  j = json_loads (buf, 0, &je);
  free (buf);

  iter = json_object_iter_at (j, "valid");
  k = json_object_iter_value (iter);

  json_unpack (k, "b", &response->valid);

  iter = json_object_iter_at (j, "message");
  k = json_object_iter_value (iter);

  json_unpack (k, "s", &response->message);

  curl_slist_free_all (slist);

  curl_easy_cleanup (c);

  return !(response->valid);
}
