/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  treeview.cc - GtkTreeView C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by416
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "treeview.h"
#include "private/treeview_p.h"
#include "treemodel.h"
#include "../gdk/dnd.h"
#include "../gdk/pixmap.h"
#include "../gdk/window.h"

using namespace Inti;

/*  Gtk::TreeeView
 */

Gtk::TreeView::TreeView(GtkTreeView *tree_view, bool reference)
: Container((GtkContainer*)tree_view, reference)
{
}

Gtk::TreeView::TreeView()
: Container((GtkContainer*)TreeViewClass::create())
{
}

Gtk::TreeView::TreeView(TreeModel& model)
: Container((GtkContainer*)TreeViewClass::create())
{
	set_model(&model);
}

Gtk::TreeView::~TreeView() 
{
}

GtkTreeViewClass*
Gtk::TreeView::gtk_tree_view_class() const
{
	return get_class<GtkTreeViewClass>();
}

Gtk::TreeView::operator GtkTreeView* () const
{
	return this ? gtk_tree_view() : 0;
}

Gtk::TreeModel*
Gtk::TreeView::get_model() const
{
	GtkTreeModel *model = gtk_tree_view_get_model(gtk_tree_view());
	Gtk::TreeModel *tmp_model = 0;
	if (model)
	{
		G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(model));
		tmp_model = dynamic_cast<Gtk::TreeModel*>(object);
	}
	return tmp_model;
}

Gtk::TreeSelection*
Gtk::TreeView::get_selection() const
{
	return G::Object::wrap<TreeSelection>(gtk_tree_view_get_selection(gtk_tree_view()));
}

Gtk::Adjustment*
Gtk::TreeView::get_hadjustment() const
{
	GtkAdjustment *hadj = gtk_tree_view_get_hadjustment(gtk_tree_view());
	return hadj ? G::Object::wrap<Adjustment>(hadj) : 0;
}

Gtk::Adjustment*
Gtk::TreeView::get_vadjustment() const
{
	GtkAdjustment *vadj = gtk_tree_view_get_vadjustment(gtk_tree_view());
	return vadj ? G::Object::wrap<Adjustment>(vadj) : 0;
}

bool 
Gtk::TreeView::get_headers_visible() const
{
	return gtk_tree_view_get_headers_visible(gtk_tree_view());
}

Gtk::TreeViewColumn*
Gtk::TreeView::get_column(int position) const
{
	GtkTreeViewColumn *column = gtk_tree_view_get_column(gtk_tree_view(), position);
	return column ? G::Object::wrap<TreeViewColumn>(column) : 0;
}

bool 
Gtk::TreeView::get_columns(std::vector<TreeViewColumn*>& columns) const
{
	g_return_val_if_fail(columns.empty(), false);
	GList *first = gtk_tree_view_get_columns(gtk_tree_view());
	GList *next = first;

	while (next != 0)
	{
		columns.push_back(G::Object::wrap<TreeViewColumn>((GtkTreeViewColumn*)next->data));
		next = g_list_next(next);
	}

	g_list_free(first);
	return !columns.empty();
}

Gtk::TreeViewColumn*
Gtk::TreeView::get_expander_column() const
{
	return G::Object::wrap<TreeViewColumn>(gtk_tree_view_get_expander_column(gtk_tree_view()));
}

bool 
Gtk::TreeView::row_expanded(const TreePath& path) const
{
	return gtk_tree_view_row_expanded(gtk_tree_view(), path.gtk_tree_path());
}

bool 
Gtk::TreeView::get_reorderable() const
{
	return gtk_tree_view_get_reorderable(gtk_tree_view());
}

void
Gtk::TreeView::get_cursor(Pointer<TreePath> *path, TreeViewColumn **focus_column) const
{
	GtkTreePath *tmp_path = 0;
	GtkTreeViewColumn *tmp_focus_column = 0;
	gtk_tree_view_get_cursor(gtk_tree_view(), path ? &tmp_path : 0, &tmp_focus_column);

	if (path && tmp_path)
		*path = G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, tmp_path, false);

	if (focus_column && tmp_focus_column)
		*focus_column = G::Object::wrap<TreeViewColumn>(tmp_focus_column);
}

Gdk::Window*
Gtk::TreeView::get_bin_window() const
{
	return G::Object::wrap<Gdk::Window>(gtk_tree_view_get_bin_window(gtk_tree_view()));
}

bool
Gtk::TreeView::get_path_at_pos(int x, int y, Pointer<TreePath> *path, TreeViewColumn **column, int *cell_x, int *cell_y) const
{
	GtkTreePath *tmp_path = 0;
	GtkTreeViewColumn *tmp_column = 0;	
	bool result = gtk_tree_view_get_path_at_pos(gtk_tree_view(), x, y, path ? &tmp_path : 0, column ? &tmp_column : 0, cell_x, cell_y);

	if (path && tmp_path)
		*path = G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, tmp_path, false);
		
	if (column && tmp_column)
		*column = G::Object::wrap<TreeViewColumn>(tmp_column);
	return result;
}	

void
Gtk::TreeView::get_cell_area(const TreePath *path, const TreeViewColumn *column, Gdk::Rectangle& rectangle) const
{
	gtk_tree_view_get_cell_area(gtk_tree_view(), *path, *column, rectangle.gdk_rectangle());
}

Gdk::Rectangle
Gtk::TreeView::get_cell_area(const TreePath *path, const TreeViewColumn *column) const 
{
	GdkRectangle rectangle;
	gtk_tree_view_get_cell_area(gtk_tree_view(), *path, *column, &rectangle);
	return rectangle;
}

void 
Gtk::TreeView::get_background_area(const TreePath *path, const TreeViewColumn *column, Gdk::Rectangle& rectangle) const
{
	gtk_tree_view_get_background_area(gtk_tree_view(), *path, *column, rectangle.gdk_rectangle());
}

Gdk::Rectangle
Gtk::TreeView::get_background_area(const TreePath *path, const TreeViewColumn *column) const 
{
	GdkRectangle rectangle;
	gtk_tree_view_get_background_area(gtk_tree_view(), *path, *column, &rectangle);
	return rectangle;
}

void 
Gtk::TreeView::get_visible_rect(Gdk::Rectangle& visible_rect) const
{
	gtk_tree_view_get_visible_rect(gtk_tree_view(), visible_rect.gdk_rectangle());
}

Gdk::Rectangle
Gtk::TreeView::get_visible_rect() const
{
	GdkRectangle visible_rect;
	gtk_tree_view_get_visible_rect(gtk_tree_view(), &visible_rect);
	return visible_rect;
}

bool 
Gtk::TreeView::get_rules_hint() const
{
	return gtk_tree_view_get_rules_hint(gtk_tree_view());
}

void
Gtk::TreeView::get_drag_dest_row(Pointer<TreePath> *path, TreeViewDropPosition *pos) const
{
	GtkTreePath *tmp_path = 0;
	gtk_tree_view_get_drag_dest_row(gtk_tree_view(), path ? &tmp_path : 0, (GtkTreeViewDropPosition*)pos);
	if (path && tmp_path)
		*path = G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, tmp_path, false);
}

bool 
Gtk::TreeView::get_dest_row_at_pos(int drag_x, int drag_y, Pointer<TreePath> *path, TreeViewDropPosition *pos) const
{
	GtkTreePath *tmp_path = 0;
	bool result = gtk_tree_view_get_dest_row_at_pos(gtk_tree_view(), drag_x, drag_y, path ? &tmp_path : 0, (GtkTreeViewDropPosition*)pos);
	if (path && tmp_path)
		*path = G::Boxed::wrap<TreePath>(GTK_TYPE_TREE_PATH, tmp_path, false);
	return result;
}
	
bool 
Gtk::TreeView::get_enable_search() const
{
	return gtk_tree_view_get_enable_search(gtk_tree_view());
}

int
Gtk::TreeView::get_search_column() const
{
	return gtk_tree_view_get_search_column(gtk_tree_view());
}

void 
Gtk::TreeView::set_model(TreeModel *model)
{
	gtk_tree_view_set_model(gtk_tree_view(), *model);
}

void 
Gtk::TreeView::set_hadjustment(Adjustment *adjustment)
{
	gtk_tree_view_set_hadjustment(gtk_tree_view(), *adjustment);
}

void 
Gtk::TreeView::set_vadjustment(Adjustment *adjustment)
{
	gtk_tree_view_set_vadjustment(gtk_tree_view(), *adjustment);
}

void
Gtk::TreeView::set_headers_visible(bool headers_visible)
{
	gtk_tree_view_set_headers_visible(gtk_tree_view(), headers_visible);
}

void 
Gtk::TreeView::columns_autosize()
{
	gtk_tree_view_columns_autosize(gtk_tree_view());
}

void
Gtk::TreeView::set_headers_clickable(bool setting)
{
	gtk_tree_view_set_headers_clickable(gtk_tree_view(), setting);
}

int
Gtk::TreeView::append_column(TreeViewColumn& column)
{
	return gtk_tree_view_append_column(gtk_tree_view(), column.gtk_tree_view_column());
}

int
Gtk::TreeView::append_column(const String& title, CellRenderer *cell, ...)
{
	GtkTreeViewColumn *column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title(column, title.c_str());
	gtk_tree_view_column_pack_start(column, *cell, TRUE);

	va_list args;
	va_start(args, cell);

	char *attribute = va_arg(args, char*);
	while (attribute)
	{
		int column_id = va_arg(args, int);
		gtk_tree_view_column_add_attribute(column, *cell, attribute, column_id);
		attribute = va_arg(args, char*);
	}

	va_end(args);
	return gtk_tree_view_insert_column(gtk_tree_view(), column, -1);
}

int
Gtk::TreeView::append_column(const String& title, CellRenderer& cell, const TreeViewColumn::CellDataSlot *data)
{
	return insert_column(-1, title, cell, data);
}

int
Gtk::TreeView::remove_column(TreeViewColumn& column)
{
	return gtk_tree_view_remove_column(gtk_tree_view(), column.gtk_tree_view_column());
}

int
Gtk::TreeView::insert_column(TreeViewColumn& column, int position)
{
	return gtk_tree_view_insert_column(gtk_tree_view(), column.gtk_tree_view_column(), position);
}

int
Gtk::TreeView::insert_column(int position, const String& title, CellRenderer *cell, ...)
{
	GtkTreeViewColumn *column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title(column, title.c_str());
	gtk_tree_view_column_pack_start(column, *cell, TRUE);

	va_list args;
	va_start(args, cell);

	char *attribute = va_arg(args, char*);
	while (attribute)
	{
		int column_id = va_arg(args, int);
		gtk_tree_view_column_add_attribute(column, *cell, attribute, column_id);
		attribute = va_arg(args, char*);
	}

	va_end(args);
	return gtk_tree_view_insert_column(gtk_tree_view(), column, position);
}

namespace { // cell_data_slot_callback

void cell_data_slot_callback(GtkTreeViewColumn*, GtkCellRenderer *cell, GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
	Gtk::TreeViewColumn::CellDataSlot *slot = (Gtk::TreeViewColumn::CellDataSlot*)data;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(model));
	Gtk::TreeModel *tmp_model = dynamic_cast<Gtk::TreeModel*>(object);
	Gtk::TreeIter tmp_iter(iter);
	slot->call(*G::Object::wrap<Gtk::CellRenderer>(cell), tmp_model, tmp_iter);
}

} // cell_data_slot_callback

int
Gtk::TreeView::insert_column(int position, const String& title, CellRenderer& cell, const TreeViewColumn::CellDataSlot *data)
{
	return gtk_tree_view_insert_column_with_data_func(gtk_tree_view(), position, title.c_str(),
	                                                  cell.gtk_cell_renderer(), 
							  &cell_data_slot_callback, (void*)data, 0);
}

void
Gtk::TreeView::move_column_after(TreeViewColumn& column, TreeViewColumn *base_column)
{
	gtk_tree_view_move_column_after(gtk_tree_view(), column.gtk_tree_view_column(), *base_column);
}

void
Gtk::TreeView::set_expander_column(TreeViewColumn *column)
{
	gtk_tree_view_set_expander_column(gtk_tree_view(), *column);
}

namespace { // column_drop_slot_callback

gboolean column_drop_slot_callback(GtkTreeView* tree_view, GtkTreeViewColumn *column, GtkTreeViewColumn *prev_column,
                                   GtkTreeViewColumn *next_column, gpointer data)
{
	Gtk::TreeView::ColumnDropSlot *slot = (Gtk::TreeView::ColumnDropSlot*)data;
	return slot->call(*G::Object::wrap<Gtk::TreeView>(tree_view),
	                  *G::Object::wrap<Gtk::TreeViewColumn>(column),
	                   G::Object::wrap<Gtk::TreeViewColumn>(prev_column),
			   G::Object::wrap<Gtk::TreeViewColumn>(next_column));
}

} // column_drop_slot_callback

void
Gtk::TreeView::set_column_drag_function(const ColumnDropSlot *drop)
{
	gtk_tree_view_set_column_drag_function(gtk_tree_view(), drop ? &column_drop_slot_callback : 0, (void*)drop, 0);
}

void
Gtk::TreeView::scroll_to_point(int tree_x, int tree_y)
{
	gtk_tree_view_scroll_to_point(gtk_tree_view(), tree_x, tree_y);
}

void
Gtk::TreeView::scroll_to_cell(const TreePath *path, const TreeViewColumn *column)
{
	gtk_tree_view_scroll_to_cell(gtk_tree_view(), *path, *column, false, 0.0, 0.0);
}

void
Gtk::TreeView::scroll_to_cell(const TreePath *path, const TreeViewColumn *column, float row_align, float col_align)
{
	gtk_tree_view_scroll_to_cell(gtk_tree_view(), *path, *column, true, row_align, col_align);
}

void
Gtk::TreeView::row_activated(TreePath& path, TreeViewColumn& column)
{
	gtk_tree_view_row_activated(gtk_tree_view(), path.gtk_tree_path(), column.gtk_tree_view_column());
}

void
Gtk::TreeView::expand_all()
{
	gtk_tree_view_expand_all(gtk_tree_view());
}

void
Gtk::TreeView::collapse_all()
{
	gtk_tree_view_collapse_all(gtk_tree_view());
}

void 
Gtk::TreeView::expand_to_path(const TreePath& path)
{
	gtk_tree_view_expand_to_path(gtk_tree_view(), path.gtk_tree_path());
}

bool
Gtk::TreeView::expand_row(const TreePath& path, bool open_all)
{
	return gtk_tree_view_expand_row(gtk_tree_view(), path.gtk_tree_path(), open_all);
}

bool
Gtk::TreeView::collapse_row(const TreePath& path)
{
	return gtk_tree_view_collapse_row(gtk_tree_view(), path.gtk_tree_path());
}

namespace { // mapping_slot_callback

void mapping_slot_callback(GtkTreeView *tree_view, GtkTreePath *path, gpointer data)
{
	Gtk::TreeView::MappingSlot *slot = (Gtk::TreeView::MappingSlot*)data;
	Gtk::TreePath tmp_path(path);
	slot->call(*G::Object::wrap<Gtk::TreeView>(tree_view), tmp_path);
}

} // mapping_slot_callback

void
Gtk::TreeView::map_expanded_rows(const MappingSlot *map)
{
	gtk_tree_view_map_expanded_rows(gtk_tree_view(), &mapping_slot_callback, (void*)map);
}

void
Gtk::TreeView::set_reorderable(bool reorderable)
{
	gtk_tree_view_set_reorderable(gtk_tree_view(), reorderable);
}

void
Gtk::TreeView::set_cursor(const TreePath& path, TreeViewColumn *focus_column, bool start_editing)
{
	gtk_tree_view_set_cursor(gtk_tree_view(), path.gtk_tree_path(), *focus_column, start_editing);
}

void 
Gtk::TreeView::set_cursor_on_cell(const TreePath& path, TreeViewColumn *focus_column, CellRenderer *focus_cell, bool start_editing)
{
	gtk_tree_view_set_cursor_on_cell(gtk_tree_view(), path.gtk_tree_path(), *focus_column, *focus_cell, start_editing);
}

void
Gtk::TreeView::widget_to_tree_coords(int wx, int wy, int *tx, int *ty) const
{
	gtk_tree_view_widget_to_tree_coords(gtk_tree_view(), wx, wy, tx, ty);
}

void
Gtk::TreeView::tree_to_widget_coords(int tx, int ty, int *wx, int *wy) const
{
	gtk_tree_view_tree_to_widget_coords(gtk_tree_view(), tx, ty, wx, wy);
}

void
Gtk::TreeView::set_rules_hint(bool setting)
{
	gtk_tree_view_set_rules_hint(gtk_tree_view(), setting);
}

void
Gtk::TreeView::enable_model_drag_source(Gdk::ModifierTypeField start_button_mask, const std::vector<TargetEntry>& targets, Gdk::DragActionField actions)
{
	g_return_if_fail(!targets.empty());
	int count = targets.size();
	GtkTargetEntry *tmp_targets = new GtkTargetEntry[count];

	int i = 0;
	while (i < count)
	{
		tmp_targets[i] = *(targets[i].gtk_target_entry());
		++i;
	}

	gtk_tree_view_enable_model_drag_source(gtk_tree_view(), (GdkModifierType)start_button_mask, tmp_targets, count, (GdkDragAction)actions);
	delete [] tmp_targets;
}

void
Gtk::TreeView::enable_model_drag_dest(const std::vector<TargetEntry>& targets, Gdk::DragActionField actions)
{
	g_return_if_fail(!targets.empty());
	int count = targets.size();
	GtkTargetEntry *tmp_targets = new GtkTargetEntry[count];

	int i = 0;
	while (i < count)
	{
		tmp_targets[i] = *(targets[i].gtk_target_entry());
		++i;
	}

	gtk_tree_view_enable_model_drag_dest(gtk_tree_view(), tmp_targets, count, (GdkDragAction)actions);
	delete [] tmp_targets;
}

void
Gtk::TreeView::unset_rows_drag_source()
{
	gtk_tree_view_unset_rows_drag_source(gtk_tree_view());
}

void
Gtk::TreeView::unset_rows_drag_dest()
{
	gtk_tree_view_unset_rows_drag_dest(gtk_tree_view());
}

void
Gtk::TreeView::set_drag_dest_row(const TreePath& path, TreeViewDropPosition pos)
{
	gtk_tree_view_set_drag_dest_row(gtk_tree_view(), path.gtk_tree_path(), (GtkTreeViewDropPosition)pos);
}

Pointer<Gdk::Pixmap>
Gtk::TreeView::create_row_drag_icon(const TreePath& path)
{
	return G::Object::wrap_new<Gdk::Pixmap>(gtk_tree_view_create_row_drag_icon(gtk_tree_view(), path.gtk_tree_path()), true);
}

void
Gtk::TreeView::set_enable_search(bool enable_search)
{
	gtk_tree_view_set_enable_search(gtk_tree_view(), enable_search);
}

void
Gtk::TreeView::set_search_column(int column)
{
	gtk_tree_view_set_search_column(gtk_tree_view(), column);
}

namespace { // search_equal_slot_callback

gboolean search_equal_slot_callback(GtkTreeModel *model, gint column, const gchar *key, GtkTreeIter *iter, gpointer data)
{
	Gtk::TreeView::SearchEqualSlot *slot = (Gtk::TreeView::SearchEqualSlot*)data;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(model));
	Gtk::TreeModel *tmp_model = dynamic_cast<Gtk::TreeModel*>(object);
	Gtk::TreeIter tmp_iter(iter);
	return slot->call(tmp_model, column, key, tmp_iter);
}

} // search_equal_slot_callback

void
Gtk::TreeView::set_search_equal_func(const SearchEqualSlot *search_equal)
{
	gtk_tree_view_set_search_equal_func(gtk_tree_view(), &search_equal_slot_callback, (void*)search_equal, 0);
}

/*  Gtk::TreeViewClass
 */

void
Gtk::TreeViewClass::init(GtkTreeViewClass *g_class)
{
	ContainerClass::init((GtkContainerClass*)g_class);
	g_class->set_scroll_adjustments = &set_scroll_adjustments_proxy;
	g_class->row_activated = &row_activated_proxy;
	g_class->test_expand_row = &test_expand_row_proxy;
	g_class->test_collapse_row = &test_collapse_row_proxy;
	g_class->row_expanded = &row_expanded_proxy;
	g_class->row_collapsed = &row_collapsed_proxy;
	g_class->columns_changed = &columns_changed_proxy;
	g_class->cursor_changed = &cursor_changed_proxy;
}

GType
Gtk::TreeViewClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TREE_VIEW, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TreeViewClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::TreeViewClass::set_scroll_adjustments_proxy(GtkTreeView *tree_view, GtkAdjustment *hadjustment, GtkAdjustment *vadjustment)
{
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
		tmp_tree_view->on_set_scroll_adjustments(G::Object::wrap<Adjustment>(hadjustment),
		                                         G::Object::wrap<Adjustment>(vadjustment));
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->set_scroll_adjustments)
			g_class->set_scroll_adjustments(tree_view, hadjustment, vadjustment);
	}
}

void
Gtk::TreeViewClass::row_activated_proxy(GtkTreeView *tree_view, GtkTreePath *path, GtkTreeViewColumn *column)
{
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
	{
		TreePath tmp_path(path);
		tmp_tree_view->on_row_activated(tmp_path, *G::Object::wrap<TreeViewColumn>(column));
	}
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->row_activated)
			g_class->row_activated(tree_view, path, column);
	}
}

gboolean
Gtk::TreeViewClass::test_expand_row_proxy(GtkTreeView *tree_view, GtkTreeIter *iter, GtkTreePath *path)
{
	gboolean result = FALSE;
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
	{
		TreeIter tmp_iter(iter);
		TreePath tmp_path(path);
		result = tmp_tree_view->on_test_expand_row(tmp_iter, tmp_path);
	}
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->test_expand_row)
			result = g_class->test_expand_row(tree_view, iter, path);
	}
	return result;
}

gboolean
Gtk::TreeViewClass::test_collapse_row_proxy(GtkTreeView *tree_view, GtkTreeIter *iter, GtkTreePath *path)
{
	gboolean result = FALSE;
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)

	{
		TreeIter tmp_iter(iter);
		TreePath tmp_path(path);
		result = tmp_tree_view->on_test_collapse_row(tmp_iter, tmp_path);
	}
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->test_collapse_row)
			result = g_class->test_collapse_row(tree_view, iter, path);
	}
	return result;
}

void
Gtk::TreeViewClass::row_expanded_proxy(GtkTreeView *tree_view, GtkTreeIter *iter, GtkTreePath *path)
{
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
	{
		TreeIter tmp_iter(iter);
		TreePath tmp_path(path);
		tmp_tree_view->on_row_expanded(tmp_iter, tmp_path);
	}
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->row_expanded)
			g_class->row_expanded(tree_view, iter, path);
	}
}

void
Gtk::TreeViewClass::row_collapsed_proxy(GtkTreeView *tree_view, GtkTreeIter *iter, GtkTreePath *path)
{
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
	{
		TreeIter tmp_iter(iter);
		TreePath tmp_path(path);
		tmp_tree_view->on_row_collapsed(tmp_iter, tmp_path);
	}
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->row_collapsed)
			g_class->row_collapsed(tree_view, iter, path);
	}
}

void
Gtk::TreeViewClass::columns_changed_proxy(GtkTreeView *tree_view)
{
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
		tmp_tree_view->on_columns_changed();
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->columns_changed)
			g_class->columns_changed(tree_view);
	}
}

void
Gtk::TreeViewClass::cursor_changed_proxy(GtkTreeView *tree_view)
{
	TreeView *tmp_tree_view = G::Object::pointer<TreeView>(tree_view);
	if (tmp_tree_view)
		tmp_tree_view->on_cursor_changed();
	else
	{
		GtkTreeViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTreeViewClass>(GTK_TREE_VIEW_GET_CLASS(tree_view));
		if (g_class->cursor_changed)
			g_class->cursor_changed(tree_view);
	}
}

/*  Signal handlers
 */

void
Gtk::TreeView::on_set_scroll_adjustments(Adjustment *hadjustment, Adjustment *vadjustment)
{
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->set_scroll_adjustments)
		g_class->set_scroll_adjustments(gtk_tree_view(), *hadjustment, *vadjustment);
}

void
Gtk::TreeView::on_row_activated(const TreePath& path, TreeViewColumn& column)
{
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->row_activated)
		g_class->row_activated(gtk_tree_view(), path.gtk_tree_path(), column.gtk_tree_view_column());
}

bool
Gtk::TreeView::on_test_expand_row(const TreeIter& iter, const TreePath& path)
{
	bool result = false;
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->test_expand_row)
		result = g_class->test_expand_row(gtk_tree_view(), iter.gtk_tree_iter(), path.gtk_tree_path());
	return result;
}

bool
Gtk::TreeView::on_test_collapse_row(const TreeIter& iter, const TreePath& path)
{
	bool result = false;
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->test_collapse_row)
		result = g_class->test_collapse_row(gtk_tree_view(), iter.gtk_tree_iter(), path.gtk_tree_path());
	return result;
}

void
Gtk::TreeView::on_row_expanded(const TreeIter& iter, const TreePath& path)
{
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->row_expanded)
		g_class->row_expanded(gtk_tree_view(), iter.gtk_tree_iter(), path.gtk_tree_path());
}

void
Gtk::TreeView::on_row_collapsed(const TreeIter& iter, const TreePath& path)
{
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->row_collapsed)
		g_class->row_collapsed(gtk_tree_view(), iter.gtk_tree_iter(), path.gtk_tree_path());
}

void
Gtk::TreeView::on_columns_changed()
{
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->columns_changed)
		g_class->columns_changed(gtk_tree_view());
}

void
Gtk::TreeView::on_cursor_changed()
{
	GtkTreeViewClass *g_class = class_peek_parent<GtkTreeViewClass>(gtk_tree_view_class());
	if (g_class->cursor_changed)
		g_class->cursor_changed(gtk_tree_view());
}

/*  Properties
 */

const Gtk::TreeView::ModelPropertyType Gtk::TreeView::model_property("model");

const Gtk::TreeView::HAdjustmentPropertyType Gtk::TreeView::hadjustment_property("hadjustment");

const Gtk::TreeView::VAdjustmentPropertyType Gtk::TreeView::vadjustment_property("vadjustment");

const Gtk::TreeView::HeadersVisiblePropertyType Gtk::TreeView::headers_visible_property("headers_visible");

const Gtk::TreeView::HeadersClickablePropertyType Gtk::TreeView::headers_clickable_property("headers_clickable");

const Gtk::TreeView::ExpanderColumnPropertyType Gtk::TreeView::expander_column_property("expander_column");

const Gtk::TreeView::ReorderablePropertyType Gtk::TreeView::reorderable_property("reorderable");

const Gtk::TreeView::RulesHintPropertyType Gtk::TreeView::rules_hint_property("rules_hint");

const Gtk::TreeView::EnableSearchPropertyType Gtk::TreeView::enable_search_property("enable_search");

const Gtk::TreeView::SearchColumnPropertyType Gtk::TreeView::search_column_property("search_column");

/*  Signals
 */

const Gtk::TreeView::SetScrollAdjustmentsSignalType Gtk::TreeView::set_scroll_adjustments_signal("set_scroll_adjustments");

const Gtk::TreeView::RowActivatedSignalType Gtk::TreeView::row_activated_signal("row_activated");

const Gtk::TreeView::TestExpandRowSignalType Gtk::TreeView::test_expand_row_signal("test_expand_row");

const Gtk::TreeView::TestCollapseRowSignalType Gtk::TreeView::test_collapse_row_signal("test_collapse_row");

const Gtk::TreeView::RowExpandSignalType Gtk::TreeView::row_expanded_signal("row_expanded");

const Gtk::TreeView::RowCollapsedSignalType Gtk::TreeView::row_collapsed_signal("row_collapsed");

const Gtk::TreeView::ColumnsChangedSignalType Gtk::TreeView::columns_changed_signal("columns_changed");

const Gtk::TreeView::CursorChangedSignalType Gtk::TreeView::cursor_changed_signal("cursor_changed");

