/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  textview.cc - GtkTextView C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "textview.h"
#include "private/textview_p.h"
#include "menu.h"
#include "../gdk/window.h"
#include "../pango/tabs.h"

using namespace Inti;

/*  Gtk::TextView
 */

Gtk::TextView::TextView(GtkTextView *text_view, bool reference)
: Container((GtkContainer*)text_view, reference)
{
}

Gtk::TextView::TextView()
: Container((GtkContainer*)TextViewClass::create())
{
}

Gtk::TextView::TextView(TextBuffer& buffer)
: Container((GtkContainer*)TextViewClass::create())
{
	set_buffer(&buffer);
}

Gtk::TextView::~TextView() 
{
}

GtkTextViewClass*
Gtk::TextView::gtk_text_view_class() const
{
	return get_class<GtkTextViewClass>();
}

Gtk::TextView::operator GtkTextView* () const
{
	return this ? gtk_text_view() : 0;
}

Gtk::TextBuffer*
Gtk::TextView::get_buffer() const
{
	return G::Object::wrap<TextBuffer>(gtk_text_view_get_buffer(gtk_text_view()));
}

void
Gtk::TextView::get_visible_rect(Gdk::Rectangle& visible_rect)
{
	gtk_text_view_get_visible_rect(gtk_text_view(), visible_rect.gdk_rectangle());
}

Gdk::Rectangle
Gtk::TextView::get_visible_rect() const
{
	GdkRectangle rectangle;
	gtk_text_view_get_visible_rect(gtk_text_view(), &rectangle);
	return rectangle;
}

bool 
Gtk::TextView::get_cursor_visible() const
{
	return gtk_text_view_get_cursor_visible(gtk_text_view());
}

void 
Gtk::TextView::get_iter_location(const TextIter& iter, Gdk::Rectangle& location)
{
	gtk_text_view_get_iter_location(gtk_text_view(), iter.gtk_text_iter(), location.gdk_rectangle());
}

Gdk::Rectangle
Gtk::TextView::get_iter_location(const TextIter& iter) const
{
	GdkRectangle rectangle;
	gtk_text_view_get_iter_location(gtk_text_view(), iter.gtk_text_iter(), &rectangle);
	return rectangle;
}

Gtk::TextIter
Gtk::TextView::get_iter_at_location(int x, int y) const
{
	GtkTextIter iter;
	gtk_text_view_get_iter_at_location(gtk_text_view(), &iter, x, y);
	return TextIter(&iter, true);
}

void
Gtk::TextView::get_line_yrange(const TextIter& iter, int *y, int *height) const
{
	gtk_text_view_get_line_yrange(gtk_text_view(), iter.gtk_text_iter(), y, height);
}

int
Gtk::TextView::get_line_y(const TextIter& iter) const
{
	int y;
	gtk_text_view_get_line_yrange(gtk_text_view(), iter.gtk_text_iter(), &y, 0);
	return y;
}

int
Gtk::TextView::get_line_height(const TextIter& iter) const
{
	int height;
	gtk_text_view_get_line_yrange(gtk_text_view(), iter.gtk_text_iter(), 0, &height);
	return height;
}

void
Gtk::TextView::get_line_at_y(TextIter& target_iter, int y, int *line_top) const
{
	gtk_text_view_get_line_at_y(gtk_text_view(), target_iter.gtk_text_iter(), y, line_top);
}

Gtk::TextIter
Gtk::TextView::get_line_at_y(int y) const
{
	GtkTextIter iter;
	gtk_text_view_get_line_at_y(gtk_text_view(), &iter, y, 0);
	return TextIter(&iter, true);
}

Gdk::Window*
Gtk::TextView::get_window(TextWindowType window_type) const
{
	return G::Object::wrap<Gdk::Window>(gtk_text_view_get_window(gtk_text_view(), (GtkTextWindowType)window_type));
}

Gtk::TextWindowType
Gtk::TextView::get_window_type(const Gdk::Window& window) const
{
	return (TextWindowType)gtk_text_view_get_window_type(gtk_text_view(), window.gdk_window());
}

int
Gtk::TextView::get_border_window_size(TextWindowType type) const
{
	return gtk_text_view_get_border_window_size(gtk_text_view(), (GtkTextWindowType)type);
}

Gtk::WrapMode
Gtk::TextView::get_wrap_mode() const
{
	return (WrapMode)gtk_text_view_get_wrap_mode(gtk_text_view());
}

bool
Gtk::TextView::get_editable() const
{
	return gtk_text_view_get_editable(gtk_text_view());
}

int
Gtk::TextView::get_pixels_above_lines() const
{
	return gtk_text_view_get_pixels_above_lines(gtk_text_view());
}

int
Gtk::TextView::get_pixels_below_lines() const
{
	return gtk_text_view_get_pixels_below_lines(gtk_text_view());
}

int
Gtk::TextView::get_pixels_inside_wrap() const
{
	return gtk_text_view_get_pixels_inside_wrap(gtk_text_view());
}

Gtk::Justification
Gtk::TextView::get_justification() const
{
	return (Justification)gtk_text_view_get_justification(gtk_text_view());
}

int
Gtk::TextView::get_left_margin() const
{
	return gtk_text_view_get_left_margin(gtk_text_view());
}

int
Gtk::TextView::get_right_margin() const
{
	return gtk_text_view_get_right_margin(gtk_text_view());
}

int
Gtk::TextView::get_indent() const
{
	return gtk_text_view_get_indent(gtk_text_view());
}

Pointer<Pango::TabArray>
Gtk::TextView::get_tabs() const
{
	return G::Boxed::wrap<Pango::TabArray>(PANGO_TYPE_TAB_ARRAY, gtk_text_view_get_tabs(gtk_text_view()), false);
}

Pointer<Gtk::TextAttributes>
Gtk::TextView::get_default_attributes() const
{
	return G::Boxed::wrap<TextAttributes>(GTK_TYPE_TEXT_ATTRIBUTES, gtk_text_view_get_default_attributes(gtk_text_view()), false);
}

void
Gtk::TextView::set_buffer(TextBuffer *buffer)
{
	gtk_text_view_set_buffer(gtk_text_view(), *buffer);
}

bool
Gtk::TextView::scroll_to_iter(const TextIter& iter, double within_margin, bool use_align, double xalign, double yalign)
{
	return gtk_text_view_scroll_to_iter(gtk_text_view(), iter.gtk_text_iter(), within_margin, use_align, xalign, yalign);
}

void
Gtk::TextView::scroll_to_mark(const TextMark& mark, double within_margin, bool use_align, double xalign, double yalign)
{
	gtk_text_view_scroll_to_mark(gtk_text_view(), mark.gtk_text_mark(), within_margin, use_align, xalign, yalign);
}

void
Gtk::TextView::scroll_mark_onscreen(const TextMark& mark)
{
	gtk_text_view_scroll_mark_onscreen(gtk_text_view(), mark.gtk_text_mark());
}

bool
Gtk::TextView::move_mark_onscreen(const TextMark& mark)
{
	return gtk_text_view_move_mark_onscreen(gtk_text_view(), mark.gtk_text_mark());
}

bool
Gtk::TextView::place_cursor_onscreen()
{
	return gtk_text_view_place_cursor_onscreen(gtk_text_view());
}

void
Gtk::TextView::set_cursor_visible(bool setting)
{
	gtk_text_view_set_cursor_visible(gtk_text_view(), setting);
}

void
Gtk::TextView::buffer_to_window_coords(TextWindowType window_type, int buffer_x, int buffer_y, int *window_x, int *window_y)
{
	gtk_text_view_buffer_to_window_coords(gtk_text_view(), (GtkTextWindowType)window_type, buffer_x, buffer_y, window_x, window_y);
}

void
Gtk::TextView::window_to_buffer_coords(TextWindowType window_type, int window_x, int window_y, int *buffer_x, int *buffer_y)
{
	gtk_text_view_window_to_buffer_coords(gtk_text_view(), (GtkTextWindowType)window_type, window_x, window_y, buffer_x, buffer_y);
}

void
Gtk::TextView::set_border_window_size(TextWindowType window_type, int size)
{
	gtk_text_view_set_border_window_size(gtk_text_view(), (GtkTextWindowType)window_type, size);
}

bool
Gtk::TextView::forward_display_line(TextIter& iter)
{
	return gtk_text_view_forward_display_line(gtk_text_view(), iter.gtk_text_iter());
}

bool
Gtk::TextView::backward_display_line(TextIter& iter)
{
	return gtk_text_view_backward_display_line(gtk_text_view(), iter.gtk_text_iter());
}

bool
Gtk::TextView::forward_display_line_end(TextIter& iter)
{
	return gtk_text_view_forward_display_line_end(gtk_text_view(), iter.gtk_text_iter());
}

bool
Gtk::TextView::backward_display_line_start(TextIter& iter)
{
	return gtk_text_view_backward_display_line_start(gtk_text_view(), iter.gtk_text_iter());
}

bool
Gtk::TextView::starts_display_line(const TextIter& iter)
{
	return gtk_text_view_starts_display_line(gtk_text_view(), iter.gtk_text_iter());
}

bool
Gtk::TextView::move_visually(TextIter& iter, int count)
{
	return gtk_text_view_move_visually(gtk_text_view(), iter.gtk_text_iter(), count);
}

void 
Gtk::TextView::add_child_at_anchor(Widget& child, TextChildAnchor& anchor)
{
	gtk_text_view_add_child_at_anchor(gtk_text_view(), child.gtk_widget(), anchor.gtk_text_child_anchor());
}

void 
Gtk::TextView::add_child_in_window(Widget& child, TextWindowType which_window, int xpos, int ypos)
{
	gtk_text_view_add_child_in_window(gtk_text_view(), child, (GtkTextWindowType)which_window, xpos, ypos);
}

void 
Gtk::TextView::move_child(const Widget& child, int xpos, int ypos)
{
	gtk_text_view_move_child(gtk_text_view(), child.gtk_widget(), xpos, ypos);
}

void 
Gtk::TextView::set_wrap_mode(WrapMode wrap_mode)
{
	gtk_text_view_set_wrap_mode(gtk_text_view(), (GtkWrapMode)wrap_mode);
}

void 
Gtk::TextView::set_editable(bool setting)
{
	gtk_text_view_set_editable(gtk_text_view(), setting);
}

void 
Gtk::TextView::set_pixels_above_lines(int pixels_above_lines)
{
	gtk_text_view_set_pixels_above_lines(gtk_text_view(), pixels_above_lines);
}

void
Gtk::TextView::set_pixels_below_lines(int pixels_below_lines)
{
	gtk_text_view_set_pixels_below_lines(gtk_text_view(), pixels_below_lines);
}

void 
Gtk::TextView::set_pixels_inside_wrap(int pixels_inside_wrap)
{
	gtk_text_view_set_pixels_inside_wrap(gtk_text_view(), pixels_inside_wrap);
}

void 
Gtk::TextView::set_justification(Justification justification)
{
	gtk_text_view_set_justification(gtk_text_view(), (GtkJustification)justification);
}

void
Gtk::TextView::set_left_margin(int left_margin)
{
	gtk_text_view_set_left_margin(gtk_text_view(), left_margin);
}

void
Gtk::TextView::set_right_margin(int right_margin)
{
	gtk_text_view_set_right_margin(gtk_text_view(), right_margin);
}

void 
Gtk::TextView::set_indent(int indent)
{
	gtk_text_view_set_indent(gtk_text_view(), indent);
}

void
Gtk::TextView::set_tabs(const Pango::TabArray& tabs)
{
	gtk_text_view_set_tabs(gtk_text_view(), tabs.pango_tab_array());
}

/*  Gtk::TextViewClass
 */

void
Gtk::TextViewClass::init(GtkTextViewClass *g_class)
{
	ContainerClass::init((GtkContainerClass*)g_class);
	g_class->set_scroll_adjustments = &set_scroll_adjustments_proxy;
	g_class->populate_popup = &populate_popup_proxy;
	g_class->set_anchor = &set_anchor_proxy;
	g_class->insert_at_cursor = &insert_at_cursor_proxy;
}

GType
Gtk::TextViewClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_TEXT_VIEW, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::TextViewClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::TextViewClass::set_scroll_adjustments_proxy(GtkTextView *text_view, GtkAdjustment *hadjustment, GtkAdjustment *vadjustment)
{
	TextView *tmp_text_view = G::Object::pointer<TextView>(text_view);
	if (tmp_text_view)
		tmp_text_view->on_set_scroll_adjustments(G::Object::wrap<Adjustment>(hadjustment),
		                                         G::Object::wrap<Adjustment>(vadjustment));
	else
	{
		GtkTextViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTextViewClass>(GTK_TEXT_VIEW_GET_CLASS(text_view));
		if (g_class->set_scroll_adjustments)
			g_class->set_scroll_adjustments(text_view, hadjustment, vadjustment);
	}
}

void
Gtk::TextViewClass::populate_popup_proxy(GtkTextView *text_view, GtkMenu *menu)
{
	TextView *tmp_text_view = G::Object::pointer<TextView>(text_view);
	if (tmp_text_view)
		tmp_text_view->on_populate_popup(*G::Object::wrap<Menu>(menu));
	else
	{
		GtkTextViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTextViewClass>(GTK_TEXT_VIEW_GET_CLASS(text_view));
		if (g_class->populate_popup)
			g_class->populate_popup(text_view, menu);
	}
}

void
Gtk::TextViewClass::set_anchor_proxy(GtkTextView *text_view)
{
	TextView *tmp_text_view = G::Object::pointer<TextView>(text_view);
	if (tmp_text_view)
		tmp_text_view->on_set_anchor();
	else
	{
		GtkTextViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTextViewClass>(GTK_TEXT_VIEW_GET_CLASS(text_view));
		if (g_class->set_anchor)
			g_class->set_anchor(text_view);
	}
}

void
Gtk::TextViewClass::insert_at_cursor_proxy(GtkTextView *text_view, const gchar *str)
{
	TextView *tmp_text_view = G::Object::pointer<TextView>(text_view);
	if (tmp_text_view)
	{
		std::string tmp_str(str);
		tmp_text_view->on_insert_at_cursor(tmp_str);
	}
	else
	{
		GtkTextViewClass *g_class = G::TypeInstance::class_peek_parent<GtkTextViewClass>(GTK_TEXT_VIEW_GET_CLASS(text_view));
		if (g_class->insert_at_cursor)
			g_class->insert_at_cursor(text_view, str);
	}
}

/*  Signal handlers
 */

void
Gtk::TextView::on_set_scroll_adjustments(Adjustment *hadjustment, Adjustment *vadjustment)
{
	GtkTextViewClass *g_class = class_peek_parent<GtkTextViewClass>(gtk_text_view_class());
	if (g_class->set_scroll_adjustments)
		g_class->set_scroll_adjustments(gtk_text_view(), *hadjustment, *vadjustment);
}

void
Gtk::TextView::on_populate_popup(Menu& menu)
{
	GtkTextViewClass *g_class = class_peek_parent<GtkTextViewClass>(gtk_text_view_class());
	if (g_class->populate_popup)
		g_class->populate_popup(gtk_text_view(), menu.gtk_menu());
}

void
Gtk::TextView::on_set_anchor()
{
	GtkTextViewClass *g_class = class_peek_parent<GtkTextViewClass>(gtk_text_view_class());
	if (g_class->set_anchor)
		g_class->set_anchor(gtk_text_view());
}

void
Gtk::TextView::on_insert_at_cursor(const String& str)
{
	GtkTextViewClass *g_class = class_peek_parent<GtkTextViewClass>(gtk_text_view_class());
	if (g_class->insert_at_cursor)
		g_class->insert_at_cursor(gtk_text_view(), str.c_str());
}

/*  Properties
 */

const Gtk::TextView::PixelsAboveLinesPropertyType Gtk::TextView::pixels_above_lines_property("pixels_above_lines");

const Gtk::TextView::PixelsBelowLinesPropertyType Gtk::TextView::pixels_below_lines_property("pixels_below_lines");

const Gtk::TextView::PixelsInsideWrapPropertyType Gtk::TextView::pixels_inside_wrap_property("pixels_inside_wrap");

const Gtk::TextView::EditablePropertyType Gtk::TextView::editable_property("editable");

const Gtk::TextView::WrapModePropertyType Gtk::TextView::wrap_mode_property("wrap_mode");

const Gtk::TextView::JustificationPropertyType Gtk::TextView::justification_property("justification");

const Gtk::TextView::LeftMarginPropertyType Gtk::TextView::left_margin_property("left_margin");

const Gtk::TextView::RightMarginPropertyType Gtk::TextView::right_margin_property("right_margin");

const Gtk::TextView::IndentPropertyType Gtk::TextView::indent_property("indent");

const Gtk::TextView::TabsPropertyType Gtk::TextView::tabs_property("tabs");

const Gtk::TextView::CursorVisiblePropertyType Gtk::TextView::cursor_visible_property("cursor_visible");

/*  Signals
 */

const Gtk::TextView::SetScrollAdjustmentsSignalType Gtk::TextView::set_scroll_adjustments_signal("set_scroll_adjustments");

const Gtk::TextView::PopulatePopupSignalType Gtk::TextView::populate_popup_signal("populate_popup");

const Gtk::TextView::SetAnchorSignalType Gtk::TextView::set_anchor_signal("set_anchor");

const Gtk::TextView::InsertAtCursorSignalType Gtk::TextView::insert_at_cursor_signal("insert_at_cursor");

