/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  dialog.cc - GtkDialog C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "dialog.h"
#include "private/dialog_p.h"
#include "box.h"
#include "buttonbox.h"
#include "button.h"

using namespace Inti;

/*  Gtk::Dialog
 */

Gtk::Dialog::Dialog(GtkDialog *dialog, bool reference)
: Window((GtkWindow*)dialog, reference)
{
}

Gtk::Dialog::Dialog()
: Window((GtkWindow*)DialogClass::create())
{
}

namespace {

void initialize_dialog(Gtk::Dialog *dialog, const String& title, Gtk::Window *parent, Gtk::DialogFlagsField flags)
{
	dialog->set_title(title);
	if (parent)
		dialog->set_transient_for(parent);
	if (flags & Gtk::DIALOG_MODAL)
		dialog->set_modal(true);
	if (flags & Gtk::DIALOG_DESTROY_WITH_PARENT)
		dialog->set_destroy_with_parent(true);
	if (flags & Gtk::DIALOG_NO_SEPARATOR)
		dialog->set_has_separator(false);
}

} // namespace

Gtk::Dialog::Dialog(const String& title, Window *parent, DialogFlagsField flags)
: Window((GtkWindow*)DialogClass::create())
{
	initialize_dialog(this, title, parent, flags);
}

namespace { // add_buttons_valist

void add_buttons_valist(GtkDialog *dialog, const gchar *first_button_text, va_list args)
{
	if (!first_button_text)
		return;

	const gchar *text = first_button_text;
	int response_id = va_arg(args, int);
	while (text)
	{
		gtk_dialog_add_button(dialog, text, response_id);
		text = va_arg(args, gchar*);
		if (!text)
			break;
		response_id = va_arg (args, int);
	}
}

} // add_buttons_valist

Gtk::Dialog::Dialog(const String& title, Window *parent, DialogFlagsField flags, const char *first_button_text, ...)
: Window((GtkWindow*)DialogClass::create())
{
	initialize_dialog(this, title, parent, flags);
	va_list args;
	va_start(args, first_button_text);
	add_buttons_valist(gtk_dialog(), first_button_text, args);
	va_end (args);
}

Gtk::Dialog::~Dialog()
{
}

GtkDialogClass*
Gtk::Dialog::gtk_dialog_class() const
{
	return get_class<GtkDialogClass>();
}

Gtk::Dialog::operator GtkDialog* () const
{
	return this ? gtk_dialog() : 0;
}

Gtk::VBox*
Gtk::Dialog::client_area() const
{
	return G::Object::wrap<VBox>((GtkVBox*)gtk_dialog()->vbox);
}

Gtk::HButtonBox*
Gtk::Dialog::action_area() const
{
	return G::Object::wrap<HButtonBox>((GtkHButtonBox*)gtk_dialog()->action_area);
}

bool
Gtk::Dialog::get_has_separator() const
{
	return gtk_dialog_get_has_separator(gtk_dialog());
}

void
Gtk::Dialog::add_action_widget(Widget& child, int response_id)
{
	gtk_dialog_add_action_widget(gtk_dialog(), child.gtk_widget(), response_id);
}

Gtk::Button*
Gtk::Dialog::add_button(const char *button_text, int response_id)
{
	return G::Object::wrap<Button>((GtkButton*)gtk_dialog_add_button(gtk_dialog(), button_text, response_id));
}

void
Gtk::Dialog::add_buttons(const char *first_button_text, ...)
{
	va_list args;
	va_start(args, first_button_text);
	add_buttons_valist(gtk_dialog(), first_button_text, args);
	va_end (args);
}

void 
Gtk::Dialog::set_response_sensitive(int response_id, bool setting)
{
	gtk_dialog_set_response_sensitive(gtk_dialog(), response_id, setting);
}

void 
Gtk::Dialog::set_default_response(int response_id)
{
	gtk_dialog_set_default_response(gtk_dialog(), response_id);
}

void
Gtk::Dialog::response(int response_id)
{
	gtk_dialog_response(gtk_dialog(), response_id);
}

int
Gtk::Dialog::run()
{
	return gtk_dialog_run(gtk_dialog());
}

void 
Gtk::Dialog::set_has_separator(bool setting)
{
	gtk_dialog_set_has_separator(gtk_dialog(), setting);
}

/*  Gtk::DialogClass
 */

void
Gtk::DialogClass::init(GtkDialogClass *g_class)
{
	WindowClass::init((GtkWindowClass*)g_class);
	g_class->response = &response_proxy;
	g_class->close = &close_proxy;
}

GType
Gtk::DialogClass::get_type()
{
	static GType type = 0;
	if (!type)
	{
		type = G::TypeInstance::register_type(GTK_TYPE_DIALOG, (GClassInitFunc)init);
	}
	return type;
}

void*
Gtk::DialogClass::create()
{
	return g_object_new(get_type(), 0);
}

void
Gtk::DialogClass::response_proxy(GtkDialog *dialog, gint response_id)
{
	Dialog *tmp_dialog = G::Object::pointer<Dialog>(dialog);
	if (tmp_dialog)
		tmp_dialog->on_response(response_id);
	else
	{
		GtkDialogClass *g_class = G::TypeInstance::class_peek_parent<GtkDialogClass>(GTK_DIALOG_GET_CLASS(dialog));
		if (g_class->response)
			g_class->response(dialog, response_id);
	}
}

void
Gtk::DialogClass::close_proxy(GtkDialog *dialog)
{
	Dialog *tmp_dialog = G::Object::pointer<Dialog>(dialog);
	if (tmp_dialog)
		tmp_dialog->on_close();
	else
	{
		GtkDialogClass *g_class = G::TypeInstance::class_peek_parent<GtkDialogClass>(GTK_DIALOG_GET_CLASS(dialog));
		if (g_class->close)
			g_class->close(dialog);
	}
}

/*  Signal handlers
 */

void
Gtk::Dialog::on_response(int response_id)
{
	GtkDialogClass *g_class = class_peek_parent<GtkDialogClass>(gtk_dialog_class());
	if (g_class->response)
		g_class->response(gtk_dialog(), response_id);
}

void
Gtk::Dialog::on_close()
{
	GtkDialogClass *g_class = class_peek_parent<GtkDialogClass>(gtk_dialog_class());
	if (g_class->close)
		g_class->close(gtk_dialog());
}

/*  Properties
 */

const Gtk::Dialog::HasSeparatorPropertyType Gtk::Dialog::has_separator_property("has_separator");

/*  Signals
 */

const Gtk::Dialog::ResponseSignalType Gtk::Dialog::response_signal("response");

const Gtk::Dialog::CloseSignalType Gtk::Dialog::close_signal("close");

