/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  thread.cc - A GThread C++ wrapper interface.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "thread.h"
#include "error.h"
#include "timeval.h"
#include <gdk/gdk.h>

using namespace Inti;

/*  G::Thread
 */

namespace { // thread_slot_callback

void* thread_slot_callback(void *data)
{
	Slot0<void> *slot = static_cast<Slot0<void>*>(data);
	if (slot)
	{
		slot->call();
		slot->unref();
	}
	return 0;
}

} // namespace

G::Thread*
G::Thread::create(const ThreadSlot *slot, bool joinable, G::Error *error)
{
	return reinterpret_cast<Thread*>(g_thread_create(&thread_slot_callback, (void*)slot, joinable, *error));
}

G::Thread*
G::Thread::create(const ThreadSlot *slot, unsigned long stack_size, bool joinable, bool bound, G::Error *error)
{
	return reinterpret_cast<Thread*>(g_thread_create_full(&thread_slot_callback, (void*)slot, stack_size,
	                                 joinable, bound, G_THREAD_PRIORITY_NORMAL, *error));
}

bool
G::Thread::is_joinable() const
{
	return g_thread()->joinable;
}

G::ThreadPriority
G::Thread::get_priority() const
{
	return (ThreadPriority)g_thread()->priority;
}

void
G::Thread::init(GThreadFunctions *vtable)
{
	g_thread_init(vtable);
	gdk_threads_init();
}

bool
G::Thread::supported()
{
	return g_thread_supported();
}

G::Thread*
G::Thread::self()
{
	return reinterpret_cast<Thread*>(g_thread_self());
}

void
G::Thread::yield()
{
	if (supported())
		g_thread_yield();
}

void
G::Thread::exit()
{
	if (supported())
		g_thread_exit(0);
}

void
G::Thread::join()
{
	g_thread_join(g_thread());
}

void
G::Thread::set_priority(ThreadPriority priority)
{
	g_thread_set_priority(g_thread(), (GThreadPriority)priority);
}

/*  G::Condition
 */

G::Condition::Condition()
: cond_(g_cond_new())
{
}

G::Condition::~Condition()
{
	if (cond_)
	{
		g_cond_free(cond_);
		cond_ = 0;
	}
}

void
G::Condition::signal()
{
	g_cond_signal(cond_);
}

void
G::Condition::broadcast()
{
	g_cond_broadcast(cond_);
}

bool
G::Condition::wait(Mutex& mutex, const TimeVal *abs_time)
{
	return g_cond_timed_wait(cond_, mutex.g_mutex(), *abs_time);
}

