/* 

                          Firewall Builder

                 Copyright (C) 2000 Vadim Kurland

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: XMLTools.cc,v 1.9 2001/12/28 05:32:31 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <fwbuilder/libfwbuilder-config.h>

#include <fwbuilder/XMLTools.hh>

#include <string.h>
#include <unistd.h>

#ifdef HAVE_LIBXSLT_XSLTCONFIG_H
# include <libxslt/xsltconfig.h>
#endif 

#include <libxslt/xslt.h>
#include <libxslt/xsltInternals.h>
#include <libxslt/transform.h>
#include <libxslt/xsltutils.h>

#include <glib.h>

#undef FW_XMLTOOLS_VERBOSE
//#define FW_XMLTOOLS_VERBOSE

#define DTD_LOAD_BITS (XML_DETECT_IDS|XML_COMPLETE_ATTRS)

using namespace std;
using namespace libfwbuilder;

extern int xmlDoValidityCheckingDefaultValue ;
extern int xmlLoadExtDtdDefaultValue         ;

/*
 * This mutex protects access to XML parser.
 * since we change DTD validation flags and error
 * handling function pointers, access should be
 * synchronized.
 */
static GMutex *xml_parser_mutex     = NULL;

/*
 * This mutex protects access to XSLT processor.
 * since we error handling function pointers, access should be
 * synchronized.
 */
static GMutex *xslt_processor_mutex = NULL;

static void xslt_error_handler(void *ctx, const char *msg, ...)
{
    char buf[4096];
    va_list args;

    assert(ctx!=NULL);
    va_start(args, msg);
    vsnprintf(buf, sizeof(buf)-1, msg, args);
    va_end(args);
    
#ifdef FW_XMLTOOLS_VERBOSE
    cerr << "XSLT ERR: " << buf << endl;
#endif

    *((string*)ctx)+=buf;
}

xmlNodePtr XMLTools::getXmlChildNode(xmlNodePtr r,const char *child_name)
{
    xmlNodePtr  cur;

    for(cur=r->xmlChildrenNode; cur; cur=cur->next) {
	if ( xmlIsBlankNode(cur) ) continue;
	if (strcmp(child_name,FROMXMLCAST(cur->name))==SAME)
	    return cur;
    }
    return NULL;
}


xmlNodePtr XMLTools::getXmlNodeByPath(xmlNodePtr r, const string &path)
{
    return getXmlNodeByPath(r, path.c_str());
}

xmlNodePtr XMLTools::getXmlNodeByPath(xmlNodePtr r, const char *path)
{
    char *s1, *cptr;
    char *path_copy;
    xmlNodePtr  cur, res;

    res=NULL;
    
    path_copy= cxx_strdup( path );

    s1=path_copy+strlen(path_copy)-1;
    while (*s1=='/') { *s1='\0'; s1--; }

    s1=path_copy;
    if (*s1=='/') {
	res=getXmlNodeByPath(r,s1+1);
	delete path_copy;
	return(res);
    }

    cptr=strchr(s1,'/');
    if (cptr!=NULL) {
	*cptr='\0';
	cptr++;
    }
    if (strcmp(FROMXMLCAST(r->name), s1)==0) {
	if (cptr) {
	    for(cur=r->xmlChildrenNode; cur; cur=cur->next) {
		if ( xmlIsBlankNode(cur) ) continue;
		res=getXmlNodeByPath(cur,cptr);
		if (res) {
		    delete path_copy;
		    return(res);
		}
	    }
	} else
	    res=r;
    }
    delete path_copy;
    return(res);
}


xmlExternalEntityLoader XMLTools::defaultLoader = NULL;

/** 
 * This is global variable used in 'fwbExternalEntityLoader'
 * parser callback. It is protected by 'xml_parser_mutex'.
 */
static string current_template_dir;

xmlParserInputPtr fwbExternalEntityLoader(const char *URL, 
                                          const char *ID,
                                          xmlParserCtxtPtr ctxt) 
{
    xmlParserInputPtr ret;

#ifdef FW_XMLTOOLS_VERBOSE
    cerr << "ENTITY: " << URL << " " << ID << endl;
#endif

    // Try to load it as file from template directory
    string fname=string(current_template_dir) + "/";
    string url=URL;
    string::size_type pos=url.rfind('/');
    fname+=(pos==string::npos)?url:url.substr(pos+1);

#ifdef FW_XMLTOOLS_VERBOSE
    cerr << "ENTITY FNAME: " << fname << endl;
#endif    
    ret = xmlNewInputFromFile(ctxt, fname.c_str());
    if(ret)
        return(ret);
    else if(XMLTools::defaultLoader)
        return XMLTools::defaultLoader(URL, ID, ctxt);
    else
        return NULL;
}

void XMLTools::initXMLTools()
{
    xml_parser_mutex     = g_mutex_new();
    xslt_processor_mutex = g_mutex_new();
    defaultLoader = xmlGetExternalEntityLoader();
    xmlSetExternalEntityLoader(fwbExternalEntityLoader);
}

xmlDocPtr XMLTools::parseFile(const string &file_name, 
                              bool use_dtd, const string &template_dir) throw(FWException)
{
    g_mutex_lock(xml_parser_mutex);    

    current_template_dir=template_dir;
    
    xmlDoValidityCheckingDefaultValue = use_dtd?1:0;
    xmlLoadExtDtdDefaultValue         = use_dtd?DTD_LOAD_BITS:0;
    
    string errors;
    xmlSetGenericErrorFunc (&errors, xslt_error_handler);
    xmlDocPtr doc = xmlParseFile(file_name.c_str()); 
    xmlSetGenericErrorFunc (NULL, NULL);

    g_mutex_unlock(xml_parser_mutex);    
    if(!doc || errors.length())
    {
        throw FWException("Error parsing XML file: "+file_name+
                          (errors.length()?(string("\nXML Parser reported:\n")+errors):string(""))
        );
    }

    return doc;
}

xmlDocPtr XMLTools::loadFile(const string &data_file , 
                             const string &type      ,
                             const string &dtd_file  ,
                             const UpgradePredicate *upgrade,
                             const string &template_dir,
                             const string &current_version
                             ) throw(FWException)
{
#ifdef FW_XMLTOOLS_VERBOSE
    cerr << "Loading file: " << data_file << endl;
#endif
    
    if(access(data_file.c_str() , R_OK )!=0)
        throw FWException("Could not access data file: "+data_file);
    
    // First load without using DTD to check version
    xmlDocPtr doc = parseFile(data_file, false, template_dir); 

#ifdef FW_XMLTOOLS_VERBOSE
    cerr << "Parsed file: " << data_file << endl;
#endif
    
    xmlDocPtr newdoc=convert(doc, data_file, type, template_dir, current_version);
    if(newdoc)
    {
        const string upgrade_msg="Data file '"+data_file+"' was converted from format used in older versions of the product.\n"
"Do you want to save file in new format? If you choose YES file on disk will be updated and\n"
"backup copy in old format will be made in the same directory with .bak extension. If you select\n"
"NO file loading operation will be cancelled.";

        if(!(*upgrade)(upgrade_msg))
        {
            xmlFreeDoc(newdoc);
            throw FWException("Load operation cancelled for file: '"+data_file);
        }
     
#ifdef FW_XMLTOOLS_VERBOSE
        cerr << "Saving updated file: " << data_file << endl;
#endif
        // file was changed save it
        doc=newdoc;
        string backup_file = data_file+".bak";
        if(rename(data_file.c_str(), backup_file.c_str()))
        {
            xmlFreeDoc(doc);
            throw FWException("Error making backup copy of file: '"+data_file+"' as '"+backup_file+"'");
        }

        try
        {
            saveFile(doc, data_file, type, dtd_file);
        } catch(FWException &ex)
        {
            // Saving converted copy failed
            // let's restore backup
            if(rename(backup_file.c_str(), data_file.c_str()))
            {
                throw FWException(ex.toString()+"\nRestoring backup copy failed "+
                                  "your old data could be found in file: '"+backup_file+"'");
            } else
                throw;
        }
    } 
    assert(doc!=NULL);
    xmlFreeDoc(doc);
    
    // Now we know the version is OK,
    // let us load for real, checking DTD.
    doc = parseFile(data_file, true, template_dir); 
    
    return doc;
}

void XMLTools::setDTD(xmlDocPtr doc, 
                      const string &type_name, 
                      const string &dtd_file) throw(FWException)
{
    xmlCreateIntSubset(doc, STRTOXMLCAST(type_name), 
                       NULL, 
                       STRTOXMLCAST(dtd_file)
    );
    

    g_mutex_lock(xml_parser_mutex);    

    xmlDoValidityCheckingDefaultValue = 1;
    xmlLoadExtDtdDefaultValue         = DTD_LOAD_BITS;

    string errors;
    xmlSetGenericErrorFunc (&errors, xslt_error_handler);

    try
    {
        xmlValidCtxt vctxt;
        if(xmlValidateDocument(&vctxt, doc)!=1)
            throw FWException("DTD validation stage 2 failed");

        xmlSetGenericErrorFunc (NULL, NULL);
        g_mutex_unlock(xml_parser_mutex);    

    } catch(...)
    {
        xmlSetGenericErrorFunc (NULL, NULL);
        g_mutex_unlock(xml_parser_mutex);    
        throw;
    }
}

void XMLTools::saveFile(xmlDocPtr doc, 
                        const string &file_name, 
                        const string &type_name,
                        const string &dtd_file) throw(FWException)
{
    
#ifdef FW_XMLTOOLS_VERBOSE
    cerr << "SAVE: " << file_name << " " <<dtd_file << endl;
#endif
    
    setDTD(doc, type_name, dtd_file);

#ifdef HAVE_XMLSAVEFORMATFILE
    if(xmlSaveFormatFile(file_name.c_str(), doc, 1)==-1)
#else
    if(xmlSaveFile(file_name.c_str(), doc)==-1)
#endif
        throw FWException("Error saving XML file: "+file_name);
}


void XMLTools::transformDocumentToFile(xmlDocPtr doc, 
                                       const string &stylesheet_file,
                                       const char **params,
                                       const string &dst_file
) throw(FWException)
{
    string xslt_errors;

    g_mutex_lock(xslt_processor_mutex);    
    g_mutex_lock(xml_parser_mutex);    
    
    xsltSetGenericErrorFunc(&xslt_errors, xslt_error_handler);
    xmlSetGenericErrorFunc (&xslt_errors, xslt_error_handler);
    
    xmlDoValidityCheckingDefaultValue = 0;
    xmlLoadExtDtdDefaultValue         = 0;
    xsltStylesheetPtr ss = xsltParseStylesheetFile(STRTOXMLCAST(stylesheet_file));
    xmlDoValidityCheckingDefaultValue = 1;
    xmlLoadExtDtdDefaultValue         = DTD_LOAD_BITS;

    if(!ss)
    {
        xsltSetGenericErrorFunc(NULL, NULL);
        xmlSetGenericErrorFunc (NULL, NULL);
        g_mutex_unlock(xml_parser_mutex);    
        g_mutex_unlock(xslt_processor_mutex);    
        throw FWException("File conversion error: Error loading stylesheet: "+stylesheet_file+
                          (xslt_errors.length()?(string("\nXSLT reports: \n")+xslt_errors):string(""))
        );
    }
    
    xmlDocPtr res = xsltApplyStylesheet(ss, doc, params);

    xsltSetGenericErrorFunc(NULL, NULL);
    xmlSetGenericErrorFunc (NULL, NULL);
    g_mutex_unlock(xml_parser_mutex);    
    g_mutex_unlock(xslt_processor_mutex);    
    
    if(!res)
    {
        xsltFreeStylesheet(ss);
        throw FWException("File conversion Error: Error during conversion: "+stylesheet_file+
                          (xslt_errors.length()?(string("XSLT reports: \n")+xslt_errors):string(""))
        );
    }

    if(dst_file=="-")
        xsltSaveResultToFile(stdout, res, ss);
    else
        xsltSaveResultToFilename(dst_file.c_str(), res, ss, 0 /* compression */ );

    xmlFreeDoc(res);
    xsltFreeStylesheet(ss);
}

xmlDocPtr XMLTools::transformDocument(xmlDocPtr doc, 
                                      const string &stylesheet_file,
                                      const char **params
) throw(FWException)
{
    string xslt_errors;

    g_mutex_lock(xslt_processor_mutex);   
    g_mutex_lock(xml_parser_mutex);    
        
    xsltSetGenericErrorFunc(&xslt_errors, xslt_error_handler);
    xmlSetGenericErrorFunc (&xslt_errors, xslt_error_handler);
    
    xmlDoValidityCheckingDefaultValue = 0;
    xmlLoadExtDtdDefaultValue         = 0;
    xsltStylesheetPtr ss = xsltParseStylesheetFile(STRTOXMLCAST(stylesheet_file));
    xmlDoValidityCheckingDefaultValue = 1;
    xmlLoadExtDtdDefaultValue         = DTD_LOAD_BITS;
    if(!ss)
    {
        xsltSetGenericErrorFunc(NULL, NULL);
        xmlSetGenericErrorFunc (NULL, NULL);
        g_mutex_unlock(xml_parser_mutex);    
        g_mutex_unlock(xslt_processor_mutex);    
        throw FWException("File conversion error: Error loading stylesheet: "+stylesheet_file+
                          (xslt_errors.length()?(string("\nXSLT reports: \n")+xslt_errors):string(""))
        );
    }
    
    xmlDocPtr res = xsltApplyStylesheet(ss, doc, params);

    xsltFreeStylesheet(ss);
    xsltSetGenericErrorFunc(NULL, NULL);
    xmlSetGenericErrorFunc (NULL, NULL);
    g_mutex_unlock(xml_parser_mutex);    
    g_mutex_unlock(xslt_processor_mutex);    
    
    if(!res)
    {
        throw FWException("File conversion Error: Error during conversion: "+stylesheet_file+
                          (xslt_errors.length()?(string("XSLT reports: \n")+xslt_errors):string(""))
        );
    }
    
    return res;
}

xmlDocPtr XMLTools::convert(xmlDocPtr doc, 
                            const string &file_name, 
                            const string &type_name, 
                            const string &template_dir,
                            const string &current_version) throw(FWException)
{
    xmlDocPtr  res = NULL;
    
    xmlNodePtr root=xmlDocGetRootElement(doc);
    if(!root || !root->name || type_name!=FROMXMLCAST(root->name))
    {
	xmlFreeDoc(doc);
        throw FWException("XML file '"+file_name+ "' have invalid structure.");
    }
    
    const char *v=FROMXMLCAST(xmlGetProp(root,TOXMLCAST("version")));
    if(!v)
    {
        // no version.
        v="0.8.7"; // at this version attribute was introducted
        xmlNewProp(root, 
                   TOXMLCAST("version") , 
                   TOXMLCAST(v));
        res=doc; // changed
    }

    int c;
    while(v && (c=version_compare(current_version,v))!=0)
    {
        if(c<0)
            throw FWException(string("Data file '"+file_name+ "' is in format, \nof version newer that current product version."));
            
        string oldversion=v;
        
#ifdef FW_XMLTOOLS_VERBOSE
        cerr << "Converting from version: " << oldversion << endl;
#endif

        string fname=template_dir+"/migration"+"/"+v+"/"+type_name+".xslt";

        if(access(fname.c_str() , R_OK )!=0) 
        {
            xmlFreeDoc(doc);
            throw FWException(string("File '"+file_name+ "' conversion error: no converter found for version: ")+oldversion);
        }
        
        try
        {
            res=transformDocument(doc, fname, NULL);
        } catch(FWException &ex)
        {
            xmlFreeDoc(doc);
            throw;
        }
        xmlFreeDoc(doc);
        doc = res;
        
        root=xmlDocGetRootElement(doc);
        if(!root || !root->name || type_name!=FROMXMLCAST(root->name))
        {
            xmlFreeDoc(doc);
            throw FWException("File '"+file_name+ "' conversion Error: conversion produced file with invalid structure.");
        }

        v=FROMXMLCAST(xmlGetProp(root, TOXMLCAST("version")));
        if(!v)
        {
            xmlFreeDoc(doc);
            throw FWException("File '"+file_name+ "' conversion error: converted to unknown version.");
        }
        
        if(version_compare(v, oldversion) <= 0)
        {
            xmlFreeDoc(doc);
            throw FWException("File '"+file_name+ "' conversion error: conversion did not advanced version number!.");
        }
    }

    return res;
}

int XMLTools::major_number(const string &v, string &rest)
{
    string a;
    string::size_type pos=v.find('.');
    if(pos==string::npos)
    {
        a    = v;
        rest = "";
    } else
    {
        a    = v.substr(0,pos);
        rest = v.substr(pos+1);
    }
    //TODO: handle conversion errors, by using 'strtol'
    return atoi(v.c_str());
}

int XMLTools::version_compare(const string &v1, const string &v2)
{
    string rest1, rest2;
    int x1=major_number(v1, rest1);
    int x2=major_number(v2, rest2);
    if(x1!=x2 || rest1.length()==0 || rest2.length()==0)
        return x1-x2;
    else 
        return version_compare(rest1, rest2);
}

string XMLTools::quote_linefeeds(const string &s)
{
    string res;

    for(string::size_type i=0;i<s.size();i++)
        if(s[i]=='\n')
            res.append("\\n");
        else
            res.append(1, s[i]);
    
    return res;
}

string XMLTools::unquote_linefeeds(const string &s)
{
    string res;
    
    for(string::size_type i=0;i<s.size();i++)
    {
        char c=s[i];
        if(c=='\\')
            if(i<(s.size()-1))
                if(s[i+1]=='n')
                {
                    c='\n';
                    i++;
                }
        res.append(1, c);
    }
    
    return res;
}

#undef DTD_LOAD_BITS

#undef FW_XMLTOOLS_VERBOSE
