/* 

                          Firewall Builder

                 Copyright (C) 2001 Vadim Zaliva, Vadim Kurland

  Author:  Vadim Zaliva lord@crocodile.org

  $Id: ThreadTools.cc,v 1.4 2001/12/28 04:50:22 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <fwbuilder/ThreadTools.hh>

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#ifndef _WIN32
#include <sys/poll.h>
#else
#include <windows.h>
#endif

# ifndef DST_NONE
#  define DST_NONE 0
# endif

using namespace std;
using namespace libfwbuilder;

Mutex::Mutex()
{
    mutex = g_mutex_new();
}

Mutex::~Mutex()
{
    g_mutex_free(mutex);
}

void Mutex::lock() const
{
    g_mutex_lock(mutex);
}

void Mutex::unlock() const
{
    g_mutex_unlock(mutex);
}

Cond::Cond()
{
    cond = g_cond_new();
}

Cond::~Cond()
{
    g_cond_free(cond);
}

bool Cond::wait(const Mutex &m, long timeout_ms) const
{
    if(timeout_ms==-1)
    {
        g_cond_wait(cond, m.mutex);
        return true;
    } else
    {
        GTimeVal finale;
	GTimeVal current_time;
	g_get_current_time (&current_time);

        finale.tv_sec  = current_time.tv_sec + timeout_ms/1000;
        finale.tv_usec = current_time.tv_usec + (timeout_ms%1000)*1000;	
        return g_cond_timed_wait(cond, m.mutex, &finale);
    }
}

void Cond::signal() const
{
    g_cond_signal(cond);
}

void Cond::broadcast() const
{
    g_cond_broadcast(cond);
}

SyncFlag::SyncFlag(bool v=false)
{
    value = v;
}

bool SyncFlag::peek() const
{
    return value;
}

bool SyncFlag::get() const
{
    bool v;
    lock();
    v = value; 
    unlock();
    return v;
}

void SyncFlag::modify(bool v)
{
    value = v;
}

void SyncFlag::set(bool v)
{
    lock();
    value = v;
    unlock();
}
    
SyncFlag::operator bool() const
{
    return get();
}
    
SyncFlag& SyncFlag::operator=(const SyncFlag &o)
{
    set(o.get());
    return *this;
}
    
SyncFlag& SyncFlag::operator=(bool v)
{
    set(v);
    return *this;
}

TimeoutCounter::TimeoutCounter(unsigned int _timeout, const string &_name):timeout(_timeout),name(_name)
{
    start();
}

void TimeoutCounter::start()
{
    time_t tres;
    finish  = time(&tres) + timeout ;
}

unsigned int TimeoutCounter::timeLeft() const
{
    time_t tres;
    int res = finish-time(&tres);
    return res<0?0:res;
}

bool TimeoutCounter::isExpired() const
{
    time_t tres;
    return time(&tres) > finish ;
}

void TimeoutCounter::check() const throw(FWException) 
{
    if(isExpired())
    {
        //cerr << "Expired Timeout Counter." << endl;
        throw FWException(name+" timeout");
    }
}

#ifdef __MINGW32__
ssize_t TimeoutCounter::read(int fd, void *buf, size_t n) const throw(FWException)
{
   GIOChannel* pIOchannel = g_io_channel_win32_new_fd (fd);

   GPollFD pollFD;
   g_io_channel_win32_make_pollfd(pIOchannel, G_IO_IN ,&pollFD);
   pollFD.events = G_IO_IN | G_IO_PRI;

   int retval = g_io_channel_win32_poll(&pollFD,1,1000*timeLeft());

   if(retval==0)
        throw FWException("Timeout "+name);
    else if(retval>0)
    {
	guint nRead = 0;
	g_io_channel_read(pIOchannel,(gchar*)buf,n,&nRead);
        return nRead;
    }
    else 
        return -1; //error
}
#else
ssize_t TimeoutCounter::read(int fd, void *buf, size_t n) const throw(FWException)
{
    struct pollfd ufds[1];

    ufds[0].fd=fd;
    ufds[0].events=POLLIN|POLLPRI;
    
    int retval=poll(ufds, 1, 1000*timeLeft());

    if(retval==0)
        throw FWException("Timeout "+name);
    else if(retval>0)
        return ::read(fd, buf, n);
    else 
        return -1; //error
}
#endif
