/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * $Id: FormulaContext.java 3314 2007-09-10 11:35:45Z tmorgner $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */
package org.jfree.formula;

import org.jfree.formula.function.FunctionRegistry;
import org.jfree.formula.operators.OperatorFactory;
import org.jfree.formula.typing.Type;
import org.jfree.formula.typing.TypeRegistry;
import org.jfree.util.Configuration;

/**
 * The formula-context connects the formula functions with the outside world. The context can be used to resolve
 * external references, to query the configuration or to retrieve information about the formula-evaluation system.
 *
 * @author Thomas Morgner
 */
public interface FormulaContext
{
  /**
   * Checks, whether the external object referenced by <code>name</code> has changed.
   *
   * @param name the name that identifies the reference.
   * @return true, if the reference has changed, false otherwise.
   * @throws ContextEvaluationException if an error occurs.
   */
  public boolean isReferenceDirty(Object name) throws ContextEvaluationException;

  /**
   * Resolves the given reference. How the name is interpreted by the outside system is an implementation detail.
   *
   * @param name the name that identifies the reference.
   * @return the resolved object.
   * @throws ContextEvaluationException if an error occurs.
   */
  public Object resolveReference(Object name) throws ContextEvaluationException;

  /**
   * Queries the type of the given reference. How the name is interpreted by the outside system is an implementation
   * detail. This return a LibFormula type object matching the type of the object that would be returned by
   * resolveReference.
   *
   * @param name the name that identifies the reference.
   * @return the type of the resolved object.
   * @throws ContextEvaluationException if an error occurs.
   */
  public Type resolveReferenceType(Object name) throws ContextEvaluationException;

  /**
   * Returns the localization context of this formula. The localization context can be used to query locale specific
   * configuration settings.
   *
   * @return the localization context.
   */
  public LocalizationContext getLocalizationContext();

  /**
   * Returns the local configuration of the formula.
   *
   * @return the local configuration.
   */
  public Configuration getConfiguration();

  /**
   * Returns the function registry. The function registry grants access to all formula-function implementations.
   *
   * @return the function registry.
   */
  public FunctionRegistry getFunctionRegistry();

  /**
   * Returns the type registry. The type registry contains all type information and allows to convert values between
   * different types.
   *
   * @return the function registry.
   */
  public TypeRegistry getTypeRegistry();

  /**
   * Returns the operator registry. The Operator-registry contains all operator-implementations.
   *
   * @return the operator registry.
   */
  public OperatorFactory getOperatorFactory();
}
